use core::ops::Range;
use core::time::Duration;
use safina_async_test::async_test;
use safina_sync::Promise;
use std::time::Instant;

/// # Panics
/// Panics if the time elapsed since `before` is not in `range_ms`.
pub fn expect_elapsed(before: Instant, range_ms: Range<u64>) {
    assert!(!range_ms.is_empty(), "invalid range {:?}", range_ms);
    let elapsed = before.elapsed();
    let duration_range = Duration::from_millis(range_ms.start)..Duration::from_millis(range_ms.end);
    assert!(
        duration_range.contains(&elapsed),
        "{:?} elapsed, out of range {:?}",
        elapsed,
        duration_range
    );
}

#[async_test]
async fn is_set() {
    let promise = Promise::new();
    let clone1 = promise.clone();
    let clone2 = promise.clone();
    assert!(!promise.is_set());
    assert!(!clone1.is_set());
    assert!(!clone2.is_set());
    promise.set(true);
    assert!(clone1.is_set());
    assert!(clone2.is_set());
    assert!(clone1.await);
    assert!(clone2.is_set());
}

#[test]
fn test_set() {
    let promise = Promise::new();
    let promise_clone = promise.clone();
    promise.set(true);
    std::panic::catch_unwind(|| promise_clone.set(true)).unwrap_err();
}

#[test]
#[allow(clippy::let_unit_value)]
fn test_try_set() {
    let promise = Promise::new();
    let _: () = promise.try_set(true).unwrap();
    assert!(promise.try_set(true).unwrap_err());
}

#[async_test]
async fn already_set() {
    let promise = Promise::new();
    let promise_clone = promise.clone();
    promise.set(true);
    let before = Instant::now();
    assert!(promise_clone.await);
    expect_elapsed(before, 0..1);
}

#[async_test]
async fn set_should_wake() {
    let before = Instant::now();
    let promise = Promise::new();
    let promise_clone = promise.clone();
    std::thread::spawn(move || {
        std::thread::sleep(Duration::from_millis(100));
        promise_clone.set(true);
    });
    assert!(promise.await);
    expect_elapsed(before, 100..200);
}

#[async_test]
async fn one_one_await_returns() {
    let before = Instant::now();
    let promise = Promise::new();
    let receiver = {
        let (sender, receiver) = std::sync::mpsc::channel::<usize>();
        for _ in 0..5 {
            let sender_clone = sender.clone();
            let promise_clone = promise.clone();
            safina_executor::spawn(async move {
                safina_timer::sleep_for(Duration::from_millis(100)).await;
                match safina_timer::with_timeout(promise_clone, Duration::from_millis(100)).await {
                    Ok(true) => sender_clone.send(1_usize).unwrap(),
                    Ok(false) => unreachable!(),
                    Err(safina_timer::DeadlineExceeded) => sender_clone.send(0_usize).unwrap(),
                }
            });
        }
        receiver
    };
    promise.set(true);
    assert_eq!(1_usize, receiver.iter().sum());
    expect_elapsed(before, 0..300);
}

#[async_test]
async fn repoll_awaits() {
    let promise = Promise::new();
    let promise_clone1 = promise.clone();
    let promise_clone2 = promise.clone();
    promise.set(true);
    assert!(promise_clone1.await);
    let before = Instant::now();
    safina_timer::with_timeout(promise_clone2, Duration::from_millis(100))
        .await
        .unwrap_err();
    expect_elapsed(before, 100..200);
}
