#![forbid(unsafe_code)]

use core::future::Future;
use core::pin::Pin;
use core::task::{Context, Poll};
use std::sync::{Arc, Mutex};
use std::task::Waker;

enum State<T> {
    Unset,
    Set(Option<T>),
}

struct Inner<T> {
    value: State<T>,
    waker: Option<Waker>,
}

/// A Future that resolves when you call [`set`](#method.set) on its clone.
///
/// Uses [`Arc`](https://doc.rust-lang.org/stable/std/sync/struct.Arc.html)
/// internally.
/// All clones of this struct use the same inner struct on the heap.
///
/// You can call `set` on any clone of this struct.
///
/// You can call `set` once and `await` once, obtaining the value.
///
/// Subsequent or concurrent awaits of the struct will wait forever.
pub struct Promise<T: Send + 'static> {
    inner: Arc<Mutex<Inner<T>>>,
}

impl<T: Send + 'static> Promise<T> {
    #[must_use]
    pub fn new() -> Self {
        Self {
            inner: Arc::new(Mutex::new(Inner {
                value: State::Unset,
                waker: None,
            })),
        }
    }

    /// Returns true if [`set`](#method.set) has been called.
    ///
    /// # Panics
    /// Panics if the inner mutex is
    /// [poisoned](https://doc.rust-lang.org/stable/std/sync/struct.Mutex.html#poisoning).
    #[must_use]
    pub fn is_set(&self) -> bool {
        !matches!(self.inner.lock().unwrap().value, State::Unset)
    }

    /// Sets the value of the struct and wakes up any awaiting task.
    ///
    /// Consumes the struct.
    ///
    /// # Panics
    /// Panics if the value was already set.
    pub fn set(self, value: T) {
        let result = self.try_set(value);
        assert!(result.is_ok(), "value already set");
    }

    /// Sets the value of the struct and wakes up any awaiting task.
    ///
    /// # Errors
    /// Returns `Err(value)` if the value was already set.
    /// This returns the value just passed in.
    #[allow(clippy::missing_panics_doc)]
    pub fn try_set(&self, value: T) -> Result<(), T> {
        let mut guard = self.inner.lock().unwrap();
        if let State::Unset = guard.value {
            guard.value = State::Set(Some(value));
            if let Some(waker) = guard.waker.take() {
                waker.wake();
            }
            Ok(())
        } else {
            Err(value)
        }
    }
}

impl<T: Send + 'static> Default for Promise<T> {
    fn default() -> Self {
        Self::new()
    }
}

impl<T: Send + 'static> Clone for Promise<T> {
    fn clone(&self) -> Self {
        Self {
            inner: self.inner.clone(),
        }
    }
}

impl<T: Send + 'static> Future for Promise<T> {
    type Output = T;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        let mut guard = self.inner.lock().unwrap();
        #[allow(clippy::option_if_let_else)]
        match &mut guard.value {
            State::Set(None) => {
                // This future previously completed and is being polled again.
                Poll::Pending
            }
            State::Set(ref mut opt_value) => Poll::Ready(opt_value.take().unwrap()),
            State::Unset => {
                guard.waker = Some(cx.waker().clone());
                Poll::Pending
            }
        }
    }
}
