#![forbid(unsafe_code)]

use core::fmt::{Debug, Display, Formatter};

pub enum OptionAbcde<A, B, C, D, E> {
    A(A),
    B(B),
    C(C),
    D(D),
    E(E),
}

impl<T> OptionAbcde<T, T, T, T, T> {
    pub fn take(self) -> T {
        match self {
            OptionAbcde::A(value)
            | OptionAbcde::B(value)
            | OptionAbcde::C(value)
            | OptionAbcde::D(value)
            | OptionAbcde::E(value) => value,
        }
    }
}

impl<A, B, C, D, E> OptionAbcde<A, B, C, D, E> {
    pub fn as_ref(&self) -> OptionAbcde<&A, &B, &C, &D, &E> {
        match self {
            OptionAbcde::A(value) => OptionAbcde::A(value),
            OptionAbcde::B(value) => OptionAbcde::B(value),
            OptionAbcde::C(value) => OptionAbcde::C(value),
            OptionAbcde::D(value) => OptionAbcde::D(value),
            OptionAbcde::E(value) => OptionAbcde::E(value),
        }
    }
    pub fn a(&self) -> Option<&A> {
        match self {
            OptionAbcde::A(value) => Some(value),
            _ => None,
        }
    }
    pub fn b(&self) -> Option<&B> {
        match self {
            OptionAbcde::B(value) => Some(value),
            _ => None,
        }
    }
    pub fn c(&self) -> Option<&C> {
        match self {
            OptionAbcde::C(value) => Some(value),
            _ => None,
        }
    }
    pub fn d(&self) -> Option<&D> {
        match self {
            OptionAbcde::D(value) => Some(value),
            _ => None,
        }
    }
    pub fn e(&self) -> Option<&E> {
        match self {
            OptionAbcde::E(value) => Some(value),
            _ => None,
        }
    }
}
impl<A: Debug, B: Debug, C: Debug, D: Debug, E: Debug> OptionAbcde<A, B, C, D, E> {
    /// # Panics
    /// Panics if `self` is not an `OptionABCDE::A`.
    pub fn unwrap_a(self) -> A {
        match self {
            OptionAbcde::A(value) => value,
            _ => panic!("expected OptionABCDE::A(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABCDE::B`.
    pub fn unwrap_b(self) -> B {
        match self {
            OptionAbcde::B(value) => value,
            _ => panic!("expected OptionABCDE::B(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABCDE::C`.
    pub fn unwrap_c(self) -> C {
        match self {
            OptionAbcde::C(value) => value,
            _ => panic!("expected OptionABCDE::C(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABCDE::D`.
    pub fn unwrap_d(self) -> D {
        match self {
            OptionAbcde::D(value) => value,
            _ => panic!("expected OptionABCDE::D(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABCDE::E`.
    pub fn unwrap_e(self) -> E {
        match self {
            OptionAbcde::E(value) => value,
            _ => panic!("expected OptionABCDE::E(_) but found {:?}", self),
        }
    }
}

impl<A: Clone, B: Clone, C: Clone, D: Clone, E: Clone> Clone for OptionAbcde<A, B, C, D, E> {
    fn clone(&self) -> Self {
        match self {
            OptionAbcde::A(value) => OptionAbcde::A(value.clone()),
            OptionAbcde::B(value) => OptionAbcde::B(value.clone()),
            OptionAbcde::C(value) => OptionAbcde::C(value.clone()),
            OptionAbcde::D(value) => OptionAbcde::D(value.clone()),
            OptionAbcde::E(value) => OptionAbcde::E(value.clone()),
        }
    }
}

impl<A: Debug, B: Debug, C: Debug, D: Debug, E: Debug> Debug for OptionAbcde<A, B, C, D, E> {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            OptionAbcde::A(value) => write!(f, "OptionABCDE::A({:?})", value),
            OptionAbcde::B(value) => write!(f, "OptionABCDE::B({:?})", value),
            OptionAbcde::C(value) => write!(f, "OptionABCDE::C({:?})", value),
            OptionAbcde::D(value) => write!(f, "OptionABCDE::D({:?})", value),
            OptionAbcde::E(value) => write!(f, "OptionABCDE::E({:?})", value),
        }
    }
}

impl<A: Display, B: Display, C: Display, D: Display, E: Display> Display
    for OptionAbcde<A, B, C, D, E>
{
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            OptionAbcde::A(value) => write!(f, "{}", value),
            OptionAbcde::B(value) => write!(f, "{}", value),
            OptionAbcde::C(value) => write!(f, "{}", value),
            OptionAbcde::D(value) => write!(f, "{}", value),
            OptionAbcde::E(value) => write!(f, "{}", value),
        }
    }
}

impl<A: PartialEq, B: PartialEq, C: PartialEq, D: PartialEq, E: PartialEq> PartialEq
    for OptionAbcde<A, B, C, D, E>
{
    fn eq(&self, other: &Self) -> bool {
        match (self, other) {
            (OptionAbcde::A(value), OptionAbcde::A(other)) if value == other => true,
            (OptionAbcde::B(value), OptionAbcde::B(other)) if value == other => true,
            (OptionAbcde::C(value), OptionAbcde::C(other)) if value == other => true,
            (OptionAbcde::D(value), OptionAbcde::D(other)) if value == other => true,
            (OptionAbcde::E(value), OptionAbcde::E(other)) if value == other => true,
            _ => false,
        }
    }
}
impl<A: PartialEq, B: PartialEq, C: PartialEq, D: PartialEq, E: PartialEq> Eq
    for OptionAbcde<A, B, C, D, E>
{
}

#[cfg(test)]
#[allow(clippy::float_cmp)]
mod tests {
    use super::*;

    #[allow(clippy::type_complexity)]
    fn test_values() -> (
        OptionAbcde<bool, u8, &'static str, usize, f32>,
        OptionAbcde<bool, u8, &'static str, usize, f32>,
        OptionAbcde<bool, u8, &'static str, usize, f32>,
        OptionAbcde<bool, u8, &'static str, usize, f32>,
        OptionAbcde<bool, u8, &'static str, usize, f32>,
    ) {
        (
            OptionAbcde::A(true),
            OptionAbcde::B(42),
            OptionAbcde::C("s1"),
            OptionAbcde::D(7_usize),
            OptionAbcde::E(0.99_f32),
        )
    }

    #[test]
    fn test_as_ref() {
        let (a, b, c, d, e) = test_values();
        let _: OptionAbcde<&bool, &u8, &&'static str, &usize, &f32> = a.as_ref();
        let _: &u8 = b.as_ref().unwrap_b();
        let _: &&'static str = c.as_ref().unwrap_c();
        let _: &usize = d.as_ref().unwrap_d();
        let _: &f32 = e.as_ref().unwrap_e();
        assert!(*(a.as_ref().unwrap_a()));
        assert_eq!(42_u8, *(b.as_ref().unwrap_b()));
        assert_eq!("s1", *(c.as_ref().unwrap_c()));
        assert_eq!(7_usize, *(d.as_ref().unwrap_d()));
        assert_eq!(0.99_f32, *(e.as_ref().unwrap_e()));
    }

    #[test]
    fn test_accessors() {
        let (a, b, c, d, e) = test_values();
        // Test accessors
        assert!(*(a.a().unwrap()));
        assert_eq!(None, a.b());
        assert_eq!(None, a.c());
        assert_eq!(None, a.d());
        assert_eq!(None, a.e());

        assert_eq!(None, b.a());
        assert_eq!(42_u8, *(b.b().unwrap()));
        assert_eq!(None, b.c());
        assert_eq!(None, b.d());
        assert_eq!(None, b.e());

        assert_eq!(None, c.a());
        assert_eq!(None, c.b());
        assert_eq!("s1", *(c.c().unwrap()));
        assert_eq!(None, c.d());
        assert_eq!(None, c.e());

        assert_eq!(None, d.a());
        assert_eq!(None, d.b());
        assert_eq!(None, d.c());
        assert_eq!(7_usize, *(d.d().unwrap()));
        assert_eq!(None, d.e());

        assert_eq!(None, e.a());
        assert_eq!(None, e.a());
        assert_eq!(None, e.a());
        assert_eq!(None, e.a());
        assert_eq!(0.99_f32, *(e.e().unwrap()));
    }

    #[test]
    fn test_debug() {
        let (a, b, c, d, e) = test_values();
        assert_eq!("OptionABCDE::A(true)", format!("{:?}", a));
        assert_eq!("OptionABCDE::B(42)", format!("{:?}", b));
        assert_eq!("OptionABCDE::C(\"s1\")", format!("{:?}", c));
        assert_eq!("OptionABCDE::D(7)", format!("{:?}", d));
        assert_eq!("OptionABCDE::E(0.99)", format!("{:?}", e));
    }

    #[test]
    fn test_display() {
        let (a, b, c, d, e) = test_values();
        assert_eq!("true", format!("{}", a));
        assert_eq!("42", format!("{}", b));
        assert_eq!("s1", format!("{}", c));
        assert_eq!("7", format!("{}", d));
        assert_eq!("0.99", format!("{}", e));
    }

    #[test]
    fn test_eq() {
        let (a, b, c, d, e) = test_values();
        assert_eq!(OptionAbcde::A(true), a);
        assert_ne!(OptionAbcde::A(false), a);
        assert_eq!(OptionAbcde::B(42_u8), b);
        assert_ne!(OptionAbcde::B(2_u8), b);
        assert_eq!(OptionAbcde::C("s1"), c);
        assert_ne!(OptionAbcde::C("other"), c);
        assert_eq!(OptionAbcde::D(7_usize), d);
        assert_ne!(OptionAbcde::D(2_usize), d);
        assert_eq!(OptionAbcde::E(0.99_f32), e);
        assert_ne!(OptionAbcde::E(0.1_f32), e);
        assert_ne!(a, b);
        assert_ne!(a, c);
        assert_ne!(a, d);
        assert_ne!(a, e);
        assert_ne!(b, c);
        assert_ne!(b, d);
        assert_ne!(b, e);
        assert_ne!(c, d);
        assert_ne!(c, e);
        assert_ne!(d, e);
    }

    #[test]
    fn test_unwrap() {
        let (a, b, c, d, e) = test_values();
        let a_clone = a.clone();
        assert!(a_clone.unwrap_a());
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABCDE::B(_) but found OptionABCDE::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_b())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABCDE::C(_) but found OptionABCDE::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_c())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABCDE::D(_) but found OptionABCDE::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_d())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABCDE::E(_) but found OptionABCDE::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_e())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );

        let b_clone = b.clone();
        assert_eq!(
            "expected OptionABCDE::A(_) but found OptionABCDE::B(42)",
            std::panic::catch_unwind(|| b_clone.unwrap_a())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let b_clone = b.clone();
        assert_eq!(42_u8, b_clone.unwrap_b());
        let b_clone = b.clone();
        std::panic::catch_unwind(|| b_clone.unwrap_c()).unwrap_err();
        let b_clone = b.clone();
        std::panic::catch_unwind(|| b_clone.unwrap_d()).unwrap_err();
        let b_clone = b.clone();
        std::panic::catch_unwind(|| b_clone.unwrap_e()).unwrap_err();

        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_a()).unwrap_err();
        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_b()).unwrap_err();
        let c_clone = c.clone();
        assert_eq!("s1", c_clone.unwrap_c());
        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_d()).unwrap_err();
        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_e()).unwrap_err();

        let d_clone = d.clone();
        std::panic::catch_unwind(|| d_clone.unwrap_a()).unwrap_err();
        let d_clone = d.clone();
        std::panic::catch_unwind(|| d_clone.unwrap_b()).unwrap_err();
        let d_clone = d.clone();
        std::panic::catch_unwind(|| d_clone.unwrap_c()).unwrap_err();
        let d_clone = d.clone();
        assert_eq!(7_usize, d_clone.unwrap_d());
        let d_clone = d.clone();
        std::panic::catch_unwind(|| d_clone.unwrap_e()).unwrap_err();

        let e_clone = e.clone();
        std::panic::catch_unwind(|| e_clone.unwrap_a()).unwrap_err();
        let e_clone = e.clone();
        std::panic::catch_unwind(|| e_clone.unwrap_b()).unwrap_err();
        let e_clone = e.clone();
        std::panic::catch_unwind(|| e_clone.unwrap_c()).unwrap_err();
        let e_clone = e.clone();
        std::panic::catch_unwind(|| e_clone.unwrap_d()).unwrap_err();
        let e_clone = e.clone();
        assert_eq!(0.99_f32, e_clone.unwrap_e());
    }

    #[test]
    fn test_take() {
        let same_a: OptionAbcde<u8, u8, u8, u8, u8> = OptionAbcde::A(42_u8);
        let same_b: OptionAbcde<u8, u8, u8, u8, u8> = OptionAbcde::B(42_u8);
        let same_c: OptionAbcde<u8, u8, u8, u8, u8> = OptionAbcde::C(42_u8);
        let same_d: OptionAbcde<u8, u8, u8, u8, u8> = OptionAbcde::C(42_u8);
        let same_e: OptionAbcde<u8, u8, u8, u8, u8> = OptionAbcde::C(42_u8);
        assert_eq!(42_u8, same_a.take());
        assert_eq!(42_u8, same_b.take());
        assert_eq!(42_u8, same_c.take());
        assert_eq!(42_u8, same_d.take());
        assert_eq!(42_u8, same_e.take());
    }
}
