#![forbid(unsafe_code)]

use core::fmt::{Debug, Display, Formatter};

pub enum OptionAbc<A, B, C> {
    A(A),
    B(B),
    C(C),
}

impl<T> OptionAbc<T, T, T> {
    pub fn take(self) -> T {
        match self {
            OptionAbc::A(value) | OptionAbc::B(value) | OptionAbc::C(value) => value,
        }
    }
}

impl<A, B, C> OptionAbc<A, B, C> {
    pub fn as_ref(&self) -> OptionAbc<&A, &B, &C> {
        match self {
            OptionAbc::A(value) => OptionAbc::A(value),
            OptionAbc::B(value) => OptionAbc::B(value),
            OptionAbc::C(value) => OptionAbc::C(value),
        }
    }
    pub fn a(&self) -> Option<&A> {
        match self {
            OptionAbc::A(value) => Some(value),
            _ => None,
        }
    }
    pub fn b(&self) -> Option<&B> {
        match self {
            OptionAbc::B(value) => Some(value),
            _ => None,
        }
    }
    pub fn c(&self) -> Option<&C> {
        match self {
            OptionAbc::C(value) => Some(value),
            _ => None,
        }
    }
}
impl<A: Debug, B: Debug, C: Debug> OptionAbc<A, B, C> {
    /// # Panics
    /// Panics if `self` is not an `OptionABC::A`.
    pub fn unwrap_a(self) -> A {
        match self {
            OptionAbc::A(value) => value,
            _ => panic!("expected OptionABC::A(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABC::B`.
    pub fn unwrap_b(self) -> B {
        match self {
            OptionAbc::B(value) => value,
            _ => panic!("expected OptionABC::B(_) but found {:?}", self),
        }
    }
    /// # Panics
    /// Panics if `self` is not an `OptionABC::C`.
    pub fn unwrap_c(self) -> C {
        match self {
            OptionAbc::C(value) => value,
            _ => panic!("expected OptionABC::C(_) but found {:?}", self),
        }
    }
}

impl<A: Clone, B: Clone, C: Clone> Clone for OptionAbc<A, B, C> {
    fn clone(&self) -> Self {
        match self {
            OptionAbc::A(value) => OptionAbc::A(value.clone()),
            OptionAbc::B(value) => OptionAbc::B(value.clone()),
            OptionAbc::C(value) => OptionAbc::C(value.clone()),
        }
    }
}

impl<A: Debug, B: Debug, C: Debug> Debug for OptionAbc<A, B, C> {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            OptionAbc::A(value) => write!(f, "OptionABC::A({:?})", value),
            OptionAbc::B(value) => write!(f, "OptionABC::B({:?})", value),
            OptionAbc::C(value) => write!(f, "OptionABC::C({:?})", value),
        }
    }
}

impl<A: Display, B: Display, C: Display> Display for OptionAbc<A, B, C> {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            OptionAbc::A(value) => write!(f, "{}", value),
            OptionAbc::B(value) => write!(f, "{}", value),
            OptionAbc::C(value) => write!(f, "{}", value),
        }
    }
}

impl<A: PartialEq, B: PartialEq, C: PartialEq> PartialEq for OptionAbc<A, B, C> {
    fn eq(&self, other: &Self) -> bool {
        match (self, other) {
            (OptionAbc::A(value), OptionAbc::A(other)) if value == other => true,
            (OptionAbc::B(value), OptionAbc::B(other)) if value == other => true,
            (OptionAbc::C(value), OptionAbc::C(other)) if value == other => true,
            _ => false,
        }
    }
}
impl<A: PartialEq, B: PartialEq, C: PartialEq> Eq for OptionAbc<A, B, C> {}

#[cfg(test)]
mod tests {
    use super::*;

    #[allow(clippy::type_complexity)]
    fn test_values() -> (
        OptionAbc<bool, u8, &'static str>,
        OptionAbc<bool, u8, &'static str>,
        OptionAbc<bool, u8, &'static str>,
    ) {
        (OptionAbc::A(true), OptionAbc::B(42), OptionAbc::C("s1"))
    }

    #[test]
    fn test_as_ref() {
        let (a, b, c) = test_values();
        let _: OptionAbc<&bool, &u8, &&'static str> = a.as_ref();
        let _: &u8 = b.as_ref().unwrap_b();
        let _: &&'static str = c.as_ref().unwrap_c();
        assert!(*(a.as_ref().unwrap_a()));
        assert_eq!(42_u8, *(b.as_ref().unwrap_b()));
        assert_eq!("s1", *(c.as_ref().unwrap_c()));
    }

    #[test]
    fn test_accessors() {
        let (a, b, c) = test_values();
        assert!(*(a.a().unwrap()));
        assert_eq!(None, a.b());
        assert_eq!(None, a.c());

        assert_eq!(None, b.a());
        assert_eq!(42_u8, *(b.b().unwrap()));
        assert_eq!(None, b.c());

        assert_eq!(None, c.a());
        assert_eq!(None, c.b());
        assert_eq!("s1", *(c.c().unwrap()));
    }

    #[test]
    fn test_debug() {
        let (a, b, c) = test_values();
        assert_eq!("OptionABC::A(true)", format!("{:?}", a));
        assert_eq!("OptionABC::B(42)", format!("{:?}", b));
        assert_eq!("OptionABC::C(\"s1\")", format!("{:?}", c));
    }

    #[test]
    fn test_display() {
        let (a, b, c) = test_values();
        assert_eq!("true", format!("{}", a));
        assert_eq!("42", format!("{}", b));
        assert_eq!("s1", format!("{}", c));
    }

    #[test]
    fn test_eq() {
        let (a, b, c) = test_values();
        assert_eq!(OptionAbc::A(true), a);
        assert_ne!(OptionAbc::A(false), a);
        assert_eq!(OptionAbc::B(42_u8), b);
        assert_ne!(OptionAbc::B(2_u8), b);
        assert_eq!(OptionAbc::C("s1"), c);
        assert_ne!(OptionAbc::C("other"), c);
        assert_ne!(a, b);
        assert_ne!(a, c);
        assert_ne!(b, c);
    }

    #[test]
    fn test_unwrap() {
        let (a, b, c) = test_values();
        let a_clone = a.clone();
        assert!(a_clone.unwrap_a());
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABC::B(_) but found OptionABC::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_b())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let a_clone = a.clone();
        assert_eq!(
            "expected OptionABC::C(_) but found OptionABC::A(true)",
            std::panic::catch_unwind(|| a_clone.unwrap_c())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );

        let b_clone = b.clone();
        assert_eq!(
            "expected OptionABC::A(_) but found OptionABC::B(42)",
            std::panic::catch_unwind(|| b_clone.unwrap_a())
                .unwrap_err()
                .downcast::<String>()
                .unwrap()
                .as_str()
        );
        let b_clone = b.clone();
        assert_eq!(42_u8, b_clone.unwrap_b());
        let b_clone = b.clone();
        std::panic::catch_unwind(|| b_clone.unwrap_c()).unwrap_err();

        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_a()).unwrap_err();
        let c_clone = c.clone();
        std::panic::catch_unwind(|| c_clone.unwrap_b()).unwrap_err();
        let c_clone = c.clone();
        assert_eq!("s1", c_clone.unwrap_c());
    }

    #[test]
    fn test_take() {
        let same_a: OptionAbc<u8, u8, u8> = OptionAbc::A(42_u8);
        let same_b: OptionAbc<u8, u8, u8> = OptionAbc::B(42_u8);
        let same_c: OptionAbc<u8, u8, u8> = OptionAbc::C(42_u8);
        assert_eq!(42_u8, same_a.take());
        assert_eq!(42_u8, same_b.take());
        assert_eq!(42_u8, same_c.take());
    }
}
