//! [![crates.io version](https://img.shields.io/crates/v/safina-net.svg)](https://crates.io/crates/safina-net)
//! [![license: Apache 2.0](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/license-apache-2.0.svg)](http://www.apache.org/licenses/LICENSE-2.0)
//! [![unsafe forbidden](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/unsafe-forbidden-success.svg)](https://github.com/rust-secure-code/safety-dance/)
//! [![pipeline status](https://gitlab.com/leonhard-llc/safina-rs/badges/main/pipeline.svg)](https://gitlab.com/leonhard-llc/safina-rs/-/pipelines)
//!
//! This is a safe Rust library for network communication.
//!
//! It is part of [`safina`](https://crates.io/crates/safina), a safe async runtime.
//!
//! # Features
//! - `forbid(unsafe_code)`
//! - Depends only on `std`
//! - Good test coverage (91%)
//! - Works with [`safina-executor`](https://crates.io/crates/safina-executor)
//!   or any async executor
//!
//! # Limitations
//! - Allocates
//! - Unoptimized
//!
//! # Documentation
//! <https://docs.rs/safina-net>
//!
//! # Examples
//! ```rust
//! # use safina_async_test::async_test;
//! # #[async_test]
//! # async fn test1() {
//! let bind_addr =
//!     std::net::SocketAddr::from(([127, 0, 0, 1], 0));
//! let mut listener =
//!     safina_net::TcpListener::bind(&bind_addr)
//!     .unwrap();
//! let addr = listener.inner().local_addr().unwrap();
//! println!("{}", &addr);
//!
//! let executor = safina_executor::Executor::default();
//! executor.spawn(async move {
//!     let mut tcp_stream =
//!         safina_net::TcpStream::connect(addr)
//!         .await
//!         .unwrap();
//!     let mut buf = String::new();
//!     tcp_stream.read_to_string(&mut buf).await.unwrap();
//!     println!("read {:?}", buf);
//! });
//!
//! let (mut tcp_stream, _addr)
//!     = listener.accept().await.unwrap();
//! tcp_stream.write_all(b"response").await.unwrap();
//! println!("wrote response");
//! # }
//! ```
//!
//! For complete examples, see the integration tests in
//! [`tests/`](https://gitlab.com/leonhard-llc/safina-rs/-/tree/main/safina-net/tests).
//!
//! # Alternatives
//! - [`async-net`](https://crates.io/crates/async-net)
//!   - Dependencies are full of `unsafe`
//! - [`async-io`](https://crates.io/crates/async-io)
//!   - Full of `unsafe`
//! - [`tokio`](https://crates.io/crates/tokio)
//!   - Very popular
//!   - Fast
//!   - Internally incredibly complicated
//!   - Full of `unsafe`
//! - [`tcp-stream`](https://crates.io/crates/tcp-stream)
//!   - Blocks async executor threads
//!   - Contains a little `unsafe` code
//!
//! # Changelog
//! - v0.1.7 - Use `safina-executor` v0.3.1.
//! - v0.1.6 - Use `safina-executor` v0.3.
//! - V0.1.5 - Update docs.
//! - v0.1.4 - Use `safina-executor` v0.2.
//! - v0.1.3 - Increase test coverage
//! - v0.1.2
//!   - Increase test coverage
//!   - Handle spurious
//!     [`EPROTOTYPE`](http://erickt.github.io/blog/2014/11/19/adventures-in-debugging-a-potential-osx-kernel-bug/)
//!     errors on macOS.
//! - v0.1.1
//!   - Add methods to `TcpStream`: `peek`, `read_vectored`, `flush`, and `write_vectored`.
//!   - Support Windows
//! - v0.1.0 - First published version
//!
//! # TO DO
//! - Add additional crates with adapters to popular async io traits, like safina-net-tokio,
//!   safina-net-futures, etc.
//!
//! # Release Process
//! 1. Edit `Cargo.toml` and bump version number.
//! 1. Run `./release.sh`
#![forbid(unsafe_code)]

use core::time::Duration;

mod tcp_stream;
pub use tcp_stream::*;

mod tcp_listener;
pub use tcp_listener::*;

async fn sleep() {
    safina_timer::sleep_for(Duration::from_millis(25)).await;
}
