# safina-executor

[![crates.io version](https://img.shields.io/crates/v/safina-executor.svg)](https://crates.io/crates/safina-executor)
[![license: Apache 2.0](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/license-apache-2.0.svg)](http://www.apache.org/licenses/LICENSE-2.0)
[![unsafe forbidden](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/unsafe-forbidden-success.svg)](https://github.com/rust-secure-code/safety-dance/)
[![pipeline status](https://gitlab.com/leonhard-llc/safina-rs/badges/main/pipeline.svg)](https://gitlab.com/leonhard-llc/safina-rs/-/pipelines)

An async executor.

It is part of [`safina`](https://crates.io/crates/safina), a safe async runtime.

## Features
- Spawn async tasks to execute on a threadpool.
- Schedule closures or `FnOnce` to run on a separate thread pool
  for blocking jobs.
- Supports multiple executors.
- `forbid(unsafe_code)`
- Depends only on `std`
- Good test coverage (100%)

## Limitations
- Allocates memory
- Not optimized

## Documentation
<https://docs.rs/safina-executor>

## Examples
```rust
let executor = safina_executor::Executor::default();
let (sender, receiver) = std::sync::mpsc::channel();
executor.spawn(async move {
    sender.send(()).unwrap();
});
receiver.recv().unwrap();
```

```rust
let result = safina_executor::block_on(async {
    prepare_request().await?;
    execute_request().await
})?;
```

```rust
let result = safina_executor::schedule_blocking(|| {
    read_file1()?;
    read_file2()
}).await?;
```

## Alternatives
- [`async-executor`](https://crates.io/crates/async-executor)
  - Popular
  - Dependencies have some `unsafe` code
- [`futures-executor`](https://crates.io/crates/futures-executor)
  - Very popular
  - Full of `unsafe`
- [`tokio-executor`](https://crates.io/crates/tokio-executor)
  - Very popular
  - Fast
  - Internally complicated
  - Full of `unsafe`
- [`executors`](https://crates.io/crates/executors)
  - Dependencies have lots of `unsafe` code
- [`bastion-executor`](https://crates.io/crates/bastion-executor)
  - Lots of `unsafe` code
- [`rayon_core`](https://crates.io/crates/rayon-core)
  - Generous amounts of `unsafe` code
- [`pollster`](https://crates.io/crates/pollster)
  - Minimal
  - Has a little `unsafe` code
- [`lelet`](https://crates.io/crates/lelet)
  - Automatically scales worker thread pool
  - Lots of `unsafe` code
- [`fibers`](https://crates.io/crates/fibers)
  - Dependencies are full of unsafe
- [`nostd_async`](https://crates.io/crates/nostd_async)
  - Has some `unsafe` code
- [`embedded-executor`](https://crates.io/crates/embedded-executor)
  - Generous amounts of `unsafe` code
- [`spin_on`](https://crates.io/crates/spin_on)
  - Minimal
- [`pasts`](https://crates.io/crates/pasts)
- [`switchyard`](https://crates.io/crates/switchyard)
  - Lots of `unsafe` code
- [`sealrs`](https://crates.io/crates/sealrs)
- [`rusty_pool`](https://crates.io/crates/rusty_pool)
  - Automatically adjusts thread pool
  - Dependencies are full of unsafe

## Changelog
- v0.2.0
  - `Executor::new` and `Executor::with_name` to return `Result`.
  - Upgrade to `safina-threadpool` v0.2.0.
- v0.1.7 - `block_on` functions to take futures that are not `Send`.
- v0.1.6 - Fix deadlock in `block_on` and `block_on_unpin` when task is awakened a second time.
- v0.1.5 - Support stable Rust!  Needs 1.51+.
- v0.1.4 - Add
  [`schedule_blocking`](https://docs.rs/safina-executor/latest/safina_executor/fn.schedule_blocking.html)
  and
  [`Executor::schedule_blocking`](https://docs.rs/safina-executor/latest/safina_executor/struct.Executor.html#method.schedule_blocking)
- v0.1.3
  - Removed global executor.  Users must explicitly create executor.
  - Removed dependency on unstable
    [`OnceCell`](https://doc.rust-lang.org/std/lazy/struct.OnceCell.html).
  - Uses [`safina_threadpool`](https://crates.io/crates/safina-threadpool)
    internally.
- v0.1.2 - Let callers pass futures to `spawn` and `block_on` without
  using `Box::pin`.
  Add `spawn_unpin` and `block_on_unpin` for folks who need to avoid allocating.
  so callers don't have to.
- v0.1.1 - Fix badge and update readme
- v0.1.0 - Renamed from `safina`

## TO DO
- DONE - Implement `spawn`
- DONE - Implement `block_on`
- DONE - Implement `increase_threads_to`
- DONE - Drop finished futures
- DONE - Handle task panic
- DONE - Add `stop_threads`, `allow_threads`, and `increase_threads_to`.
- DONE - Add tests
- DONE - Add docs
- DONE - Publish on crates.io
- DONE - Add an `#[async_test]` macro
- DONE - Build with Rust `stable`.
- Add a stress test
- Add a benchmark.  See benchmarks in <https://crates.io/crates/executors>
- Add an `#[async_main]` macro
- Look into using [`flume`](https://crates.io/crates/flume)
  to eliminate the receiver mutex and reduce contention.

## Release Process
1. Edit `Cargo.toml` and bump version number.
1. Run `./release.sh`

License: Apache-2.0
