//! [![crates.io version](https://img.shields.io/crates/v/safina-async-test.svg)](https://crates.io/crates/safina-async-test)
//! [![license: Apache 2.0](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/license-apache-2.0.svg)](http://www.apache.org/licenses/LICENSE-2.0)
//! [![unsafe forbidden](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/unsafe-forbidden-success.svg)](https://github.com/rust-secure-code/safety-dance/)
//! [![pipeline status](https://gitlab.com/leonhard-llc/safina-rs/badges/main/pipeline.svg)](https://gitlab.com/leonhard-llc/safina-rs/-/pipelines)
//!
//! An `async_test` macro for running `async fn` tests.
//!
//! It is part of [`safina`](https://crates.io/crates/safina), a safe async runtime.
//!
//! # Features
//! - Runs tests with [`safina_executor`](https://docs.rs/safina-executor)
//! - Each test gets its own executor with 2 threads for async tasks and 1 thread for blocking tasks.
//! - Also calls
//!   [`safina_timer::start_timer_thread`](https://docs.rs/safina-timer/latest/safina_timer/fn.start_timer_thread.html)
//!   before running the test
//! - `forbid(unsafe_code)`
//! - Lightweight dependencies
//! - Straightforward implementation
//!
//! # Limitations
//! - Building on `stable` requires [`once_cell`](https://crates.io/crates/once_cell)
//!   crate which contains some unsafe code.
//!   This is necessary until
//!   [`std::lazy::OnceCell`](https://doc.rust-lang.org/std/lazy/struct.OnceCell.html)
//!   is stable.
//!
//! # Examples
//! ```rust
//! use safina_async_test::async_test;
//! # async fn async_work() -> Result<(), std::io::Error> { Ok(()) }
//!
//! #[async_test]
//! async fn test1() {
//!     async_work().await.unwrap();
//! }
//! ```
//!
//! ```rust
//! use safina;
//! use safina_async_test::async_test;
//! # use core::time::Duration;
//! # fn blocking_work() -> Result<u8, std::io::Error> { Ok(3) }
//! # async fn background_task() {}
//! # async fn async_work() -> Result<u8, std::io::Error> { Ok(42) }
//!
//! // Make your test an `async fn`.
//! #[async_test]
//! async fn test2() {
//!     // You can `await`.
//!     async_work().await.unwrap();
//!
//!     // You can spawn tasks which will run on
//!     // the executor.
//!     // These tasks stop when the test
//!     // function returns and drops the
//!     // executor.
//!     safina::spawn(background_task());
//!
//!     // You can run blocking code without
//!     // stalling other async tasks.
//!     let result = safina::schedule_blocking(
//!         || blocking_work()
//!     );
//!     assert_eq!(3, result.await.unwrap());
//!
//!     // You can use timer functions.
//!     safina::sleep_for(
//!         Duration::from_millis(10)).await;
//!     safina::with_timeout(
//!         async_work(),
//!         Duration::from_millis(100)
//!     ).await.unwrap();
//! }
//! ```
//!
//! # Documentation
//! <https://docs.rs/safina-async-test>
//!
//! # Alternatives
//! - [`async_std::test`](https://docs.rs/async-std/latest/async_std/attr.test.html)
//! - [`futures_await_test::async_test`](https://docs.rs/futures-await-test/0.3.0/futures_await_test/)
//! - [`tokio::test`](https://docs.rs/tokio/latest/tokio/attr.test.html)
//!
//! # Changelog
//! - V0.1.11 - Update docs.
//! - v0.1.10 - Use `safina-executor` v0.2.0.
//! - v0.1.9 - Don't require users to also depend on `safina-executor` and
//!   `safina-timer` crates.
//! - v0.1.8
//!   - Support stable with rust 1.51 and `once_cell`.
//!   - Start an Executor for each test
//! - v0.1.7 - Update to safina-executor v0.1.4
//! - v0.1.6 - Start [`safina-timer`](https://crates.io/crates/safina-timer)
//!   thread
//! - v0.1.5 - Use [`safina-executor`](https://crates.io/crates/safina-executor)
//!   v0.1.3 API
//! - v0.1.4 - Upgrade to new
//!   [`safina-executor`](https://crates.io/crates/safina-executor)
//!   version which removes need for `Box::pin`.
//! - v0.1.3 - Rename [`safina`](https://crates.io/crates/safina) package to
//!   [`safina-executor`](https://crates.io/crates/safina-executor).
//! - v0.1.2 - Update docs
//! - v0.1.1 - First published version
//!
//! # TO DO
//!
//! # Release Process
//! 1. Edit `Cargo.toml` and bump version number.
//! 1. Run `./release.sh`
#![forbid(unsafe_code)]

pub use safina_async_test_core::async_test;
pub use safina_executor::Executor;
pub use safina_timer::start_timer_thread;
