mod error;

use self::error::ParseUriErr;
use safe_uri::*;

pub use self::error::ParseUriError;

pub fn parse_uri(input: &str) -> Result<Uri, ParseUriError> {
    parse_uri_internal(input).map_err(ParseUriError)
}

pub fn parse_static_uri(input: &'static str) -> Result<Uri<'static>, ParseUriError> {
    parse_uri(input).map(Uri::into_ensured_static)
}

pub fn parse_uri_ref(input: &str) -> Result<UriRef, ParseUriError> {
    parse_uri_ref_internal(input).map_err(ParseUriError)
}

pub fn parse_static_uri_ref(input: &'static str) -> Result<UriRef<'static>, ParseUriError> {
    parse_uri_ref(input).map(UriRef::into_ensured_static)
}

type InputResultOption<'s, T, E> = Result<Option<(T, &'s str)>, E>;

type InputResult<'s, T, E> = Result<(T, &'s str), E>;

fn parse_uri_internal(input: &str) -> Result<Uri, ParseUriErr> {
    parse_uri_ref_internal(input)?
        .try_into()
        .map_err(|_| ParseUriErr::MissingScheme)
}

fn parse_uri_ref_internal(input: &str) -> Result<UriRef, ParseUriErr> {
    let (scheme, remainder) = scheme(input)
        .map_err(ParseUriErr::Scheme)?
        .or_remainder(input);
    let (authority, remainder) = authority(remainder)?.or_remainder(remainder);
    let (path, remainder) = if !remainder.is_empty()
        && (remainder.starts_with('/') || !remainder.starts_with(|c| matches!(c, '?' | '#')))
    {
        path(remainder).map_err(ParseUriErr::Path)?
    } else {
        (Path::new(), remainder)
    };
    let (query, remainder) = query(remainder)
        .map_err(ParseUriErr::Query)?
        .or_remainder(remainder);
    let fragment = fragment(remainder).map_err(ParseUriErr::Fragment)?;
    let mut resource = Resource::new();
    resource.path = path;
    resource.query = query;
    resource.fragment = fragment;
    let mut uri = UriRef::new();
    uri.scheme = scheme;
    uri.authority = authority;
    uri.resource = resource;
    Ok(uri)
}

fn scheme(input: &str) -> InputResultOption<Scheme, InvalidScheme> {
    for (index, byte) in input.bytes().enumerate() {
        match byte {
            b':' => {
                let (scheme, remainder) = split_away(input, index);
                return Ok(Some((scheme.try_into()?, remainder)));
            }
            b'/' | b'?' | b'#' => return Ok(None),
            _ => continue,
        }
    }
    Ok(None)
}

fn authority(input: &str) -> InputResultOption<Authority, ParseUriErr> {
    let remainder = match input.strip_prefix("//") {
        Some(r) => r,
        None => return Ok(None),
    };
    let (user_info, remainder) = user_info(remainder)
        .map_err(ParseUriErr::UserInfo)?
        .or_remainder(remainder);
    let (host, remainder) = host(remainder)?;
    let (port, remainder) = match remainder.strip_prefix(':') {
        Some(remainder) => {
            let (p, r) = port(remainder)?;
            (Some(p), r)
        }
        None => (None, remainder),
    };
    let mut authority = Authority::new();
    authority.user_info = user_info;
    authority.host = host;
    authority.port = port;
    Ok(Some((authority, remainder)))
}

fn user_info(input: &str) -> InputResultOption<UserInfo, InvalidUserInfo> {
    for (index, byte) in input.bytes().enumerate() {
        match byte {
            b'@' => {
                let (user_info, remainder) = split_away(input, index);
                return Ok(Some((user_info.try_into()?, remainder)));
            }
            b':' | b'/' | b'?' | b'#' => return Ok(None),
            _ => continue,
        }
    }
    Ok(None)
}

fn host(input: &str) -> InputResult<Host, ParseUriErr> {
    let (host_str, remainder) = host_str(input);
    match host_from_str(host_str) {
        Ok(host) => Ok((host, remainder)),
        Err(e) => Err(e),
    }
}

fn host_str(input: &str) -> (&str, &str) {
    if input.starts_with('[') {
        for (index, byte) in input.bytes().enumerate() {
            match byte {
                b'/' | b'?' | b'#' => {
                    return split_before(input, index);
                }
                b']' => {
                    return split_before(input, index + 1);
                }
                _ => continue,
            }
        }
    } else {
        for (index, byte) in input.bytes().enumerate() {
            match byte {
                b':' | b'/' | b'?' | b'#' => {
                    return split_before(input, index);
                }
                _ => continue,
            }
        }
    }
    (input, "")
}

fn host_from_str(s: &str) -> Result<Host, ParseUriErr> {
    if let Some(ipv6_str) = s.strip_prefix('[').and_then(|s| s.strip_suffix(']')) {
        return match ipv6_str.parse() {
            Ok(ipv6) => Ok(Host::Ipv6Addr(ipv6)),
            Err(e) => Err(ParseUriErr::Ipv6(e)),
        };
    }
    let host = if let Ok(x) = s.parse() {
        Host::Ipv4Addr(x)
    } else {
        Host::Name(s.try_into().map_err(ParseUriErr::HostName)?)
    };
    Ok(host)
}

fn port(input: &str) -> InputResult<u16, ParseUriErr> {
    let (port, remainder) = (|| {
        for (index, byte) in input.bytes().enumerate() {
            match byte {
                b'/' | b'?' | b'#' => return Ok(split_before(input, index)),
                _ if !byte.is_ascii_digit() => return Err(ParseUriErr::NonDigitPortByte(byte)),
                _ => continue,
            }
        }
        Ok((input, ""))
    })()?;
    Ok((port.parse().map_err(ParseUriErr::Port)?, remainder))
}

fn path(input: &str) -> InputResult<Path, InvalidPath> {
    let split_path_from_remainder = || {
        for (index, byte) in input.bytes().enumerate() {
            match byte {
                b'?' | b'#' => return split_before(input, index),
                _ => continue,
            }
        }
        (input, "")
    };
    let (path, remainder) = split_path_from_remainder();
    Ok((path.try_into()?, remainder))
}

fn query(input: &str) -> InputResultOption<Query, InvalidQuery> {
    let remainder = match input.strip_prefix('?') {
        None => return Ok(None),
        Some(r) => r,
    };
    let split_query_from_remainder = || {
        for (index, byte) in remainder.bytes().enumerate() {
            match byte {
                b'#' => return split_before(remainder, index),
                _ => continue,
            }
        }
        (remainder, "")
    };
    let (query, remainder) = split_query_from_remainder();
    Ok(Some((query.try_into()?, remainder)))
}

fn fragment(input: &str) -> Result<Option<Fragment>, InvalidFragment> {
    match input.strip_prefix('#') {
        None => Ok(None),
        Some(f) => Ok(Some(f.try_into()?)),
    }
}

fn split_before(s: &str, index: usize) -> (&str, &str) {
    s.split_at(index)
}

fn split_away(s: &str, index: usize) -> (&str, &str) {
    let (before_s, remainder) = s.split_at(index);
    let (_, after_s) = remainder.split_at(1);
    (before_s, after_s)
}

trait OrRemainder<'s> {
    type T;
    fn or_remainder(self, remainder: &'s str) -> (Option<Self::T>, &'s str);
}

impl<'s, T> OrRemainder<'s> for Option<(T, &'s str)> {
    type T = T;

    fn or_remainder(self, remainder: &'s str) -> (Option<T>, &'s str) {
        match self {
            Some((t, r)) => (Some(t), r),
            None => (None, remainder),
        }
    }
}
