use safe_uri::*;
use safe_uri_ext::*;
use std::net::{Ipv4Addr, Ipv6Addr};
use tap::Tap;

#[test]
fn wikipedia_article() {
    assert_eq!(
        parse_uri_ref("https://en.wikipedia.org/wiki/Backslash").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("https").unwrap());
            u.authority = Some(Host::Name(HostName::try_from("en.wikipedia.org").unwrap()).into());
            u.resource = Path::try_from("/wiki/Backslash").unwrap().into();
        }),
    );
}

#[test]
fn ipv4_localhost() {
    assert_eq!(
        parse_uri_ref("http://127.0.0.1/api/v2/user/35/alias/1").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("http").unwrap());
            u.authority = Some(Host::Ipv4Addr(Ipv4Addr::LOCALHOST).into());
            u.resource = Path::try_from("/api/v2/user/35/alias/1").unwrap().into();
        }),
    );
}

#[test]
fn ipv4_port_query_fragment() {
    assert_eq!(
        parse_uri_ref("ftp://127.3.2.1:8080?hello=world#africa").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("ftp").unwrap());
            u.authority = Some(Authority::new().tap_mut(|a| {
                a.host = Host::Ipv4Addr([127, 3, 2, 1].into());
                a.port = Some(8080);
            }));
            u.resource = Resource::new().tap_mut(|r| {
                r.query = Some(Query::try_from("hello=world").unwrap());
                r.fragment = Some(Fragment::try_from("africa").unwrap());
            });
        }),
    );
}

#[test]
fn ipv6_localhost() {
    assert_eq!(
        parse_uri_ref("http://[::1]/").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("http").unwrap());
            u.authority = Some(Host::Ipv6Addr(Ipv6Addr::LOCALHOST).into());
            u.resource = Path::try_from("/").unwrap().into();
        }),
    );
}

#[test]
fn ipv6_3_port_80() {
    assert_eq!(
        parse_uri_ref("http://[::3]:80").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("http").unwrap());
            u.authority = Some(Authority::new().tap_mut(|a| {
                a.host = Host::Ipv6Addr(3.into());
                a.port = Some(80);
            }));
        }),
    );
}

#[test]
fn parse_urn() {
    assert_eq!(
        parse_uri_ref("urn:animal:dog:nose").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("urn").unwrap());
            u.resource.path = Path::try_from("animal:dog:nose").unwrap();
        }),
    );
}

#[test]
fn file_absolute_unix() {
    assert_eq!(
        parse_uri_ref("file:///home/test").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("file").unwrap());
            u.authority = Some(Host::Name(HostName::new()).into());
            u.resource.path = Path::try_from("/home/test").unwrap();
        }),
    );
}

#[test]
fn file_absolute_windows_backslashs() {
    parse_uri_ref(r"file://C:\Users\Test").unwrap_err();
}

#[test]
fn file_absolute_windows_forward_slashs() {
    assert_eq!(
        parse_uri_ref("file:///C:/Users/Test").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("file").unwrap());
            u.authority = Some(Host::Name(HostName::new()).into());
            u.resource.path = Path::try_from("/C:/Users/Test").unwrap();
        }),
    );
}

#[test]
fn scheme_with_double_slash() {
    assert_eq!(
        parse_uri_ref("http://").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.scheme = Some(Scheme::try_from("http").unwrap());
            u.authority = Some(Host::Name(HostName::new()).into());
        }),
    );
}

#[test]
fn utf_8_host_and_path() {
    parse_uri_ref("http://über.de/hää").unwrap_err();
}

#[test]
fn authority_only() {
    assert_eq!(
        parse_uri_ref("//example.com").unwrap(),
        UriRef::new().tap_mut(|u| {
            u.authority = Some(Host::Name(HostName::try_from("example.com").unwrap()).into());
        }),
    );
}

#[test]
fn path_only() {
    let path = "/my/resource";
    assert_eq!(
        parse_uri_ref(path).unwrap(),
        UriRef::new().tap_mut(|u| {
            u.resource.path = Path::try_from(path).unwrap();
        }),
    );
}

#[test]
fn invalid_percent_encoding() {
    parse_uri_ref("-//!!%=#").unwrap_err();
}

#[test]
fn port_with_plus() {
    parse_uri_ref("//host:+1").unwrap_err();
}

#[test]
fn port_with_minus() {
    parse_uri_ref("//host:-1").unwrap_err();
}

#[test]
fn port_with_minus_minus() {
    parse_uri_ref("//host:--1").unwrap_err();
}
