// Copyright 2021 MaidSafe.net limited.
//
// This SAFE Network Software is licensed to you under the MIT license <LICENSE-MIT
// https://opensource.org/licenses/MIT> or the Modified BSD license <LICENSE-BSD
// https://opensource.org/licenses/BSD-3-Clause>, at your option. This file may not be copied,
// modified, or distributed except according to those terms. Please review the Licences for the
// specific language governing permissions and limitations relating to use of the SAFE Network
// Software.

//! Module providing keys, keypairs, and signatures.
//!
//! The easiest way to get a `PublicKey` is to create a random `Keypair` first through one of the
//! `new` functions. A `PublicKey` can't be generated by itself; it must always be derived from a
//! secret key.

use super::super::utils;

use hex_fmt::HexFmt;
use serde::{Deserialize, Serialize};
use std::{
    fmt,
    hash::{Hash, Hasher},
};

/// A signature share, with its index in the combined collection.
#[derive(Clone, Hash, Eq, PartialEq, PartialOrd, Serialize, Deserialize, Debug)]
pub struct SignatureShare {
    /// Index in the combined collection.
    pub index: usize,
    /// Signature over some data.
    pub share: bls::SignatureShare,
}

/// Wrapper for different signature types.
#[derive(Clone, Eq, PartialEq, Serialize, Deserialize, custom_debug::Debug)]
#[allow(clippy::large_enum_variant)]
pub enum Signature {
    /// Ed25519 signature.
    #[debug(with = "Self::fmt_ed25519")]
    Ed25519(ed25519_dalek::Signature),
    /// BLS signature.
    Bls(bls::Signature),
    /// BLS signature share.
    BlsShare(SignatureShare),
}

impl Signature {
    /// Returns bls::Signature if Self is a BLS variant.
    pub fn into_bls(self) -> Option<bls::Signature> {
        match self {
            Self::Bls(sig) => Some(sig),
            _ => None,
        }
    }

    /// Returns ed25519_dalek::Signature if Self is a Ed25519 variant.
    pub fn into_ed(self) -> Option<ed25519_dalek::Signature> {
        match self {
            Self::Ed25519(sig) => Some(sig),
            _ => None,
        }
    }

    // ed25519_dalek::Signature has overly verbose debug output, so we provide our own
    pub(crate) fn fmt_ed25519(
        sig: &ed25519_dalek::Signature,
        f: &mut fmt::Formatter,
    ) -> fmt::Result {
        write!(f, "Signature({:0.10})", HexFmt(sig))
    }
}

impl From<bls::Signature> for Signature {
    fn from(sig: bls::Signature) -> Self {
        Self::Bls(sig)
    }
}

impl From<ed25519_dalek::Signature> for Signature {
    fn from(sig: ed25519_dalek::Signature) -> Self {
        Self::Ed25519(sig)
    }
}

impl From<SignatureShare> for Signature {
    fn from(sig: SignatureShare) -> Self {
        Self::BlsShare(sig)
    }
}

impl From<(usize, bls::SignatureShare)> for Signature {
    fn from(sig: (usize, bls::SignatureShare)) -> Self {
        let (index, share) = sig;
        Self::BlsShare(SignatureShare { index, share })
    }
}

#[allow(clippy::derive_hash_xor_eq)]
impl Hash for Signature {
    fn hash<H: Hasher>(&self, state: &mut H) {
        utils::serialise(&self).hash(state)
    }
}
