use std::net::SocketAddr;
#[cfg(not(feature = "unstable-no-connection-pooling"))]
use std::{collections::HashMap, net::IpAddr, sync::Arc};

#[cfg(not(feature = "unstable-no-connection-pooling"))]
use tokio::sync::RwLock;
use xor_name::XorName;

/// A bidirectional map of `XorName` <-> `SocketAddr`, representing peers connected to a node.
///
/// This map is maintained in order to correlate node messages with client requests. For example,
/// when a client send a request for a chunk to an elder, the elder will then send requests for the
/// data to relevant adults. When the adults reply, the elder needs some way to correlate the
/// adult's response to the original client request. Since we can't distinguish client connections
/// from other node connections a-priori, we actually store all peer connections in this map. The
/// semantics are only useful when handling 'client'-oriented messages, however.
///
/// At present, correlation is performed at connection granularity – e.g. each peer connection gets
/// its own correlation ID, based on hashing the `SocketAddr` to a `XorName`. Since some peers
/// cannot be connected to directly, we store a handle to the sending end of the connection for each
/// ID.
#[derive(Clone, Default)]
pub(super) struct ConnectedPeers {
    #[cfg(not(feature = "unstable-no-connection-pooling"))]
    peers: Arc<RwLock<HashMap<XorName, qp2p::Connection>>>,
}

#[cfg(not(feature = "unstable-no-connection-pooling"))]
impl ConnectedPeers {
    /// Get the connected peer with the given `id`.
    ///
    /// Returns `Some(peer)` if `id` corresponds to a connected peer, and `None` otherwise.
    pub(super) async fn get_by_id(&self, id: &XorName) -> Option<ConnectedPeer> {
        let connection = self.peers.read().await.get(id).cloned()?;
        Some(ConnectedPeer {
            id: *id,
            connection,
        })
    }

    /// Get the connected peer with the given `address`.
    ///
    /// Note that connection IDs can computed deterministically from `address`, but we keep the
    /// actual computation as an implementation detail of this type.
    ///
    /// Returns `Some(peer)` if `address` corresponds to a connected peer, and `None` otherwise.
    pub(super) async fn get_by_address(&self, address: &SocketAddr) -> Option<ConnectedPeer> {
        let id = Self::address_to_id(address);
        self.get_by_id(&id).await
    }

    /// Store the given connection.
    ///
    /// The connection will be stored under an ID, computed from the connection's remote address.
    /// If there's an existing connection to the peer, it will be replaced. If an existing
    /// connection is replaced, it won't be explicitly closed, so the connection may continue to
    /// live if there are other handles to it.
    pub(super) async fn insert(&self, connection: qp2p::Connection) {
        let id = Self::address_to_id(&connection.remote_address());

        // we don't do anything with the existing connection (if any), just let it drop
        let _existing = self.peers.write().await.insert(id, connection);
    }

    /// Remove the connection with the given `address`.
    ///
    /// Connections have to be removed manually if they close, or need to be closed. The removed
    /// connection, if any, is simply dropped.
    pub(super) async fn remove_by_address(&self, address: &SocketAddr) {
        let id = Self::address_to_id(address);

        // we don't do anything with the existing connection (if any), just let it drop
        let _existing = self.peers.write().await.remove(&id);
    }

    pub(crate) fn address_to_id(address: &SocketAddr) -> XorName {
        match address.ip() {
            IpAddr::V4(ip) => {
                XorName::from_content_parts(&[&ip.octets(), &address.port().to_be_bytes()])
            }
            IpAddr::V6(ip) => {
                XorName::from_content_parts(&[&ip.octets(), &address.port().to_be_bytes()])
            }
        }
    }
}

#[cfg(feature = "unstable-no-connection-pooling")]
impl ConnectedPeers {
    pub(super) async fn get_by_id(&self, _id: &XorName) -> Option<ConnectedPeer> {
        None
    }
    pub(super) async fn get_by_address(&self, _address: &SocketAddr) -> Option<ConnectedPeer> {
        None
    }
    pub(super) async fn insert(&self, _connection: qp2p::Connection) {}
    pub(super) async fn remove_by_address(&self, _address: &SocketAddr) {}
}

/// A peer connected to a node, as stored by [`ConnectedPeers`].
///
/// Connected peers have an associated connected ID, which is generated by [`ConnectedPeers`]
/// when a connection is registered.
#[derive(Clone)]
pub(super) struct ConnectedPeer {
    id: XorName,
    connection: qp2p::Connection,
}

impl ConnectedPeer {
    pub(super) fn address(&self) -> SocketAddr {
        self.connection.remote_address()
    }

    pub(super) fn connection(&self) -> &qp2p::Connection {
        &self.connection
    }
}
