// Copyright 2021 MaidSafe.net limited.
//
// This SAFE Network Software is licensed to you under The General Public License (GPL), version 3.
// Unless required by applicable law or agreed to in writing, the SAFE Network Software distributed
// under the GPL Licence is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. Please review the Licences for the specific language governing
// permissions and limitations relating to use of the SAFE Network Software.

use super::{Mapping, MsgContext};
use crate::messaging::{
    data::ServiceMsg, AuthorityProof, DstLocation, EndUser, MessageId, ServiceAuth, SrcLocation,
};
use crate::node::node_ops::{MsgType, NodeDuty, OutgoingMsg};

pub(super) fn map_client_msg(
    msg_id: MessageId,
    msg: ServiceMsg,
    auth: AuthorityProof<ServiceAuth>,
    user: EndUser,
) -> Mapping {
    // Signature has already been validated by the routing layer
    let op = map_client_service_msg(msg_id, msg.clone(), user, auth);

    let ctx = Some(MsgContext::Client {
        msg,
        src: SrcLocation::EndUser(user),
    });

    Mapping { op, ctx }
}

fn map_client_service_msg(
    msg_id: MessageId,
    service_msg: ServiceMsg,
    origin: EndUser,
    auth: AuthorityProof<ServiceAuth>,
) -> NodeDuty {
    match service_msg {
        ServiceMsg::Query(query) => NodeDuty::ProcessRead {
            query,
            msg_id,
            auth,
            origin,
        },
        ServiceMsg::Cmd(cmd) => NodeDuty::ProcessWrite {
            cmd,
            msg_id,
            auth,
            origin,
        },
        ServiceMsg::QueryResponse {
            response,
            correlation_id,
        } => {
            let outgoing_msg = OutgoingMsg {
                id: MessageId::in_response_to(&correlation_id),
                msg: MsgType::Client(ServiceMsg::QueryResponse {
                    response,
                    correlation_id,
                }),
                dst: DstLocation::EndUser(origin),
                aggregation: false,
            };
            NodeDuty::Send(outgoing_msg)
        }
        _ => {
            warn!(
                "Dropping unexpected received user msg ({}), origin {:?}, service msg: {:?}",
                msg_id, origin, service_msg
            );
            NodeDuty::NoOp
        }
    }
}
