mod iter;

pub use iter::{IntoIter, Iter, IterMut};

use crate::HeaderValue;

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct HeaderValues<'a> {
    first: HeaderValue<'a>,
    remaining: Vec<HeaderValue<'a>>,
}

impl<'a> HeaderValues<'a> {
    #[inline]
    pub fn new(first: HeaderValue<'a>) -> Self {
        Self {
            first,
            remaining: Vec::new(),
        }
    }

    pub fn has_only_one(&self) -> bool {
        self.remaining.is_empty()
    }

    pub fn push(&mut self, v: HeaderValue<'a>) {
        self.remaining.push(v);
    }

    pub fn first(&self) -> &HeaderValue<'a> {
        &self.first
    }

    pub fn first_mut(&mut self) -> &mut HeaderValue<'a> {
        &mut self.first
    }

    pub fn into_first(self) -> HeaderValue<'a> {
        self.first
    }

    pub fn iter(&self) -> Iter<'_, 'a> {
        Iter {
            first: Some(&self.first),
            remaining: self.remaining.iter(),
        }
    }

    pub fn iter_mut(&mut self) -> IterMut<'_, 'a> {
        IterMut {
            first: Some(&mut self.first),
            remaining: self.remaining.iter_mut(),
        }
    }
}

impl<'a> From<HeaderValue<'a>> for HeaderValues<'a> {
    #[inline]
    fn from(v: HeaderValue<'a>) -> Self {
        Self::new(v)
    }
}

impl<'a> FromIterator<HeaderValue<'a>> for Option<HeaderValues<'a>> {
    fn from_iter<T: IntoIterator<Item = HeaderValue<'a>>>(iter: T) -> Self {
        let mut iter = iter.into_iter();
        let first = iter.next()?;
        let remaining = iter.collect();
        Some(HeaderValues { first, remaining })
    }
}

impl<'a> Extend<HeaderValue<'a>> for HeaderValues<'a> {
    fn extend<T: IntoIterator<Item = HeaderValue<'a>>>(&mut self, iter: T) {
        self.remaining.extend(iter)
    }
}

impl<'a> IntoIterator for HeaderValues<'a> {
    type Item = HeaderValue<'a>;

    type IntoIter = IntoIter<'a>;

    fn into_iter(self) -> Self::IntoIter {
        IntoIter {
            first: Some(self.first),
            remaining: self.remaining.into_iter(),
        }
    }
}

impl<'r, 'a> IntoIterator for &'r HeaderValues<'a> {
    type Item = &'r HeaderValue<'a>;

    type IntoIter = Iter<'r, 'a>;

    fn into_iter(self) -> Self::IntoIter {
        self.iter()
    }
}
