mod constants;
mod invalid;

pub use invalid::InvalidHeaderName;

use boar_::{Boas, BoasStr};
use std::{
    convert::TryFrom,
    fmt,
    hash::{Hash, Hasher},
    sync::Arc,
};

#[derive(Debug, Clone)]
pub struct HeaderName<'a>(BoasStr<'a>);

impl<'a> HeaderName<'a> {
    fn internal_try_from(string: BoasStr<'a>) -> Result<Self, InvalidHeaderName> {
        validate(string.as_bytes())?;
        Ok(Self(string))
    }

    pub fn as_str(&self) -> &str {
        &self.0
    }

    pub(super) const fn internal_from_const(string: &'static str) -> Self {
        if validate(string.as_bytes()).is_err() {
            #[allow(unconditional_panic, clippy::no_effect)]
            ([] as [u8; 0])[0]; // hack for panic in const fn
        }
        Self(Boas::Static(string))
    }

    pub fn into_static(self) -> HeaderName<'static> {
        HeaderName(self.0.into_static())
    }

    pub fn to_borrowed(&self) -> HeaderName {
        HeaderName(self.0.to_borrowed())
    }
}

impl HeaderName<'static> {
    #[inline]
    pub fn ensure_static(&mut self) {
        self.0.ensure_static();
    }

    #[inline]
    pub fn into_ensured_static(mut self) -> Self {
        self.ensure_static();
        self
    }

    pub fn try_from_static(string: &'static str) -> Result<Self, InvalidHeaderName> {
        Self::internal_try_from(Boas::Static(string))
    }

    #[track_caller]
    pub const fn from_static(string: &'static str) -> Self {
        match validate(string.as_bytes()) {
            Ok(()) => Self(BoasStr::Static(string)),
            Err(_e) => panic!("invalid static HeaderName"),
        }
    }
}

impl<'a> TryFrom<&'a str> for HeaderName<'a> {
    type Error = InvalidHeaderName;

    fn try_from(string: &'a str) -> Result<Self, InvalidHeaderName> {
        Self::internal_try_from(Boas::Borrowed(string))
    }
}

impl<'a> TryFrom<String> for HeaderName<'a> {
    type Error = InvalidHeaderName;

    fn try_from(string: String) -> Result<Self, InvalidHeaderName> {
        Self::internal_try_from(Boas::Owned(string))
    }
}

impl<'a> TryFrom<Arc<String>> for HeaderName<'a> {
    type Error = InvalidHeaderName;

    fn try_from(string: Arc<String>) -> Result<Self, InvalidHeaderName> {
        Self::internal_try_from(Boas::Arc(string))
    }
}

#[cfg(feature = "boar")]
impl<'a> TryFrom<BoasStr<'a>> for HeaderName<'a> {
    type Error = InvalidHeaderName;

    fn try_from(string: BoasStr<'a>) -> Result<Self, InvalidHeaderName> {
        Self::internal_try_from(string)
    }
}

impl<'a> Eq for HeaderName<'a> {}

impl<'a> Hash for HeaderName<'a> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        for byte in self.0.bytes() {
            state.write_u8(byte.to_ascii_lowercase())
        }
    }
}

impl<'a> PartialEq for HeaderName<'a> {
    fn eq(&self, other: &Self) -> bool {
        self.0.eq_ignore_ascii_case(&other.0)
    }
}

impl<'a> PartialEq<str> for HeaderName<'a> {
    fn eq(&self, other: &str) -> bool {
        self.0.eq_ignore_ascii_case(other)
    }
}

impl<'a> PartialEq<&'_ str> for HeaderName<'a> {
    fn eq(&self, other: &&str) -> bool {
        self.0.eq_ignore_ascii_case(other)
    }
}

impl fmt::Display for HeaderName<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Display::fmt(&self.0, f)
    }
}

const fn validate(bytes: &[u8]) -> Result<(), InvalidHeaderName> {
    match crate::rfc::token::validate(bytes) {
        Ok(()) => Ok(()),
        Err(e) => Err(InvalidHeaderName {
            byte: e.byte,
            index: e.index,
        }),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use assert_matches::assert_matches;
    use std::collections::hash_map::DefaultHasher;

    #[test]
    fn equals_ignores_case() {
        let upper_case = HeaderName::internal_from_const("Content-Type");
        let lower_case = HeaderName::try_from(upper_case.0.to_ascii_lowercase()).unwrap();
        assert_eq!(lower_case, upper_case);
    }

    #[test]
    fn hash_ignores_case() {
        let upper_case = HeaderName::internal_from_const("Content-Type");
        let lower_case = HeaderName::try_from(upper_case.0.to_ascii_lowercase()).unwrap();
        assert_eq!(hash(&lower_case), hash(&upper_case));
    }

    fn hash<T: Hash>(t: &T) -> u64 {
        let mut hasher = DefaultHasher::new();
        t.hash(&mut hasher);
        hasher.finish()
    }

    #[test]
    fn generated_constants_are_valid() {
        for constant in HeaderName::GENERATED_CONSTANTS.iter().map(|c| c.as_str()) {
            // Invalid constants are a compile error if those constants are used.
            // So these asserts are just there to double check.
            assert_matches!(HeaderName::try_from_static(constant), Ok(name) => assert_eq!(constant, name.as_str()));
        }
    }
}
