use crate::{HeaderName, HeaderValues};
use std::collections::hash_map;

#[derive(Debug)]
pub enum Entry<'map, 'a> {
    Occupied(OccupiedEntry<'map, 'a>),
    Vacant(VacantEntry<'map, 'a>),
}

impl<'map, 'a> Entry<'map, 'a> {
    pub fn and_modify<F>(self, f: F) -> Self
    where
        F: FnOnce(&mut HeaderValues<'a>),
    {
        match self {
            Entry::Occupied(mut x) => {
                f(x.get_mut());
                Entry::Occupied(x)
            }
            Entry::Vacant(x) => Entry::Vacant(x),
        }
    }

    pub fn key(&self) -> &HeaderName<'a> {
        match self {
            Entry::Occupied(x) => x.key(),
            Entry::Vacant(x) => x.key(),
        }
    }

    pub fn or_insert(self, default: HeaderValues<'a>) -> &'map mut HeaderValues<'a> {
        self.or_insert_with(|| default)
    }

    pub fn or_insert_with<F: FnOnce() -> HeaderValues<'a>>(
        self,
        default: F,
    ) -> &'map mut HeaderValues<'a> {
        self.or_insert_with_key(|_| default())
    }

    pub fn or_insert_with_key<F: FnOnce(&HeaderName<'a>) -> HeaderValues<'a>>(
        self,
        default: F,
    ) -> &'map mut HeaderValues<'a> {
        match self {
            Entry::Occupied(x) => x.into_mut(),
            Entry::Vacant(entry) => {
                let value = default(entry.key());
                entry.insert(value)
            }
        }
    }
}

type OccupiedInner<'map, 'a> = hash_map::OccupiedEntry<'map, HeaderName<'a>, HeaderValues<'a>>;

#[derive(Debug)]
pub struct OccupiedEntry<'map, 'a>(pub(super) OccupiedInner<'map, 'a>);

impl<'map, 'a> OccupiedEntry<'map, 'a> {
    pub fn key(&self) -> &HeaderName<'a> {
        self.0.key()
    }

    pub fn get(&self) -> &HeaderValues<'a> {
        self.0.get()
    }

    pub fn get_mut(&mut self) -> &mut HeaderValues<'a> {
        self.0.get_mut()
    }

    pub fn insert(&mut self, values: HeaderValues<'a>) -> HeaderValues<'a> {
        self.0.insert(values)
    }

    pub fn into_mut(self) -> &'map mut HeaderValues<'a> {
        self.0.into_mut()
    }

    pub fn remove(self) -> HeaderValues<'a> {
        self.0.remove()
    }

    pub fn remove_entry(self) -> (HeaderName<'a>, HeaderValues<'a>) {
        self.0.remove_entry()
    }
}

type VacantInner<'map, 'a> = hash_map::VacantEntry<'map, HeaderName<'a>, HeaderValues<'a>>;

#[derive(Debug)]
pub struct VacantEntry<'map, 'a>(pub(super) VacantInner<'map, 'a>);

impl<'map, 'a> VacantEntry<'map, 'a> {
    pub fn key(&self) -> &HeaderName<'a> {
        self.0.key()
    }

    pub fn into_key(self) -> HeaderName<'a> {
        self.0.into_key()
    }

    pub fn insert(self, values: HeaderValues<'a>) -> &'map mut HeaderValues<'a> {
        self.0.insert(values)
    }
}
