mod entry;
mod iter;

pub use self::entry::{Entry, OccupiedEntry, VacantEntry};
pub use self::iter::{
    Drain, IntoIter, IntoKeys, IntoValues, Iter, IterMut, Keys, Values, ValuesMut,
};

use crate::{HeaderName, HeaderValue, HeaderValues};
use std::{
    borrow::Borrow,
    collections::{hash_map, HashMap, TryReserveError},
    hash::Hash,
};

#[derive(Debug, PartialEq, Eq, Clone, Default)]
pub struct HeaderMap<'a>(HashMap<HeaderName<'a>, HeaderValues<'a>>);

impl<'a> HeaderMap<'a> {
    /// Creates a new empty `HeaderMap`.
    pub fn new() -> Self {
        Self(HashMap::new())
    }

    /// Appends the value if the key is already present or inserts the key-value pair into the map.
    ///
    /// Returns `true` if the map had the key present already.
    // We should add a note about not updating the key, if we add a is_sensitive_value marker to HeaderName.
    pub fn append_one(&mut self, key: HeaderName<'a>, value: HeaderValue<'a>) -> bool {
        self.append_values(key, value.into())
    }

    /// Appends the values if the key is already present or inserts the key-values pair into the map.
    ///
    /// Returns `true` if the map had the key present already.
    // We should add a note about not updating the key, if we add a is_sensitive_value marker to HeaderName.
    pub fn append_values(&mut self, key: HeaderName<'a>, values: HeaderValues<'a>) -> bool {
        match self.0.entry(key) {
            hash_map::Entry::Occupied(mut occupied) => {
                occupied.get_mut().extend(values);
                true
            }
            hash_map::Entry::Vacant(vacant) => {
                vacant.insert(values);
                false
            }
        }
    }

    pub fn clear(&mut self) {
        self.0.clear()
    }

    pub fn contains_key<Q: ?Sized>(&self, k: &Q) -> bool
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.contains_key(k)
    }

    pub fn drain(&mut self) -> Drain<'_, 'a> {
        Drain(self.0.drain())
    }

    pub fn entry(&mut self, key: HeaderName<'a>) -> Entry<'_, 'a> {
        match self.0.entry(key) {
            hash_map::Entry::Occupied(x) => Entry::Occupied(OccupiedEntry(x)),
            hash_map::Entry::Vacant(x) => Entry::Vacant(VacantEntry(x)),
        }
    }

    pub fn get<Q: ?Sized>(&self, k: &Q) -> Option<&HeaderValues<'a>>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get(k)
    }

    pub fn get_first<Q: ?Sized>(&self, k: &Q) -> Option<&HeaderValue<'a>>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get(k).map(HeaderValues::first)
    }

    pub fn get_key_values<Q: ?Sized>(&self, k: &Q) -> Option<(&HeaderName<'a>, &HeaderValues<'a>)>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get_key_value(k)
    }

    pub fn get_first_key_value<Q: ?Sized>(
        &self,
        k: &Q,
    ) -> Option<(&HeaderName<'a>, &HeaderValue<'a>)>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get_key_value(k).map(|(n, v)| (n, v.first()))
    }

    pub fn get_mut<Q: ?Sized>(&mut self, k: &Q) -> Option<&mut HeaderValues<'a>>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get_mut(k)
    }

    pub fn get_first_mut<Q: ?Sized>(&mut self, k: &Q) -> Option<&mut HeaderValue<'a>>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.get_mut(k).map(HeaderValues::first_mut)
    }

    /// Inserts a key-value pair into the map.
    ///
    /// If the map did not have this key present, None is returned.
    ///
    /// If the map did have this key present, the values are replaced, and one of the old values is returned.
    // We should add a note about not updating the key, if we add a is_sensitive_value marker to HeaderName.
    pub fn insert_one(
        &mut self,
        key: HeaderName<'a>,
        value: HeaderValue<'a>,
    ) -> Option<HeaderValue<'a>> {
        self.0
            .insert(key, value.into())
            .map(HeaderValues::into_first)
    }

    /// Inserts a key-values pair into the map.
    ///
    /// If the map did not have this key present, None is returned.
    ///
    /// If the map did have this key present, the values are replaced, and the old values are returned.
    // We should add a note about not updating the key, if we add a is_sensitive_value marker to HeaderName.
    pub fn insert(
        &mut self,
        key: HeaderName<'a>,
        values: HeaderValues<'a>,
    ) -> Option<HeaderValues<'a>> {
        self.0.insert(key, values)
    }

    pub fn into_keys(self) -> IntoKeys<'a> {
        IntoKeys(self.0.into_keys())
    }

    pub fn into_values(self) -> IntoValues<'a> {
        IntoValues(self.0.into_values())
    }

    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    pub fn iter(&self) -> Iter<'_, 'a> {
        Iter(self.0.iter())
    }

    pub fn iter_mut(&mut self) -> IterMut<'_, 'a> {
        IterMut(self.0.iter_mut())
    }

    pub fn keys(&self) -> Keys<'_, 'a> {
        Keys(self.0.keys())
    }

    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn remove<Q: ?Sized>(&mut self, k: &Q) -> Option<HeaderValues<'a>>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.remove(k)
    }

    pub fn remove_entry<Q: ?Sized>(&mut self, k: &Q) -> Option<(HeaderName<'a>, HeaderValues<'a>)>
    where
        HeaderName<'a>: Borrow<Q>,
        Q: Hash + Eq,
    {
        self.0.remove_entry(k)
    }

    /// Reserves capacity for at least `additional` more elements to be inserted in the `HeaderMap`.
    /// The collection may reserve more space to avoid frequent reallocations.
    pub fn reserve(&mut self, additional: usize) {
        self.0.reserve(additional)
    }

    pub fn retain<F>(&mut self, f: F)
    where
        F: FnMut(&HeaderName<'a>, &mut HeaderValues<'a>) -> bool,
    {
        self.0.retain(f)
    }

    pub fn shrink_to(&mut self, min_capacity: usize) {
        self.0.shrink_to(min_capacity)
    }

    pub fn shrink_to_fit(&mut self) {
        self.0.shrink_to_fit()
    }

    /// Tries to reserve capacity for at least `additional` more elements to be inserted in the `HeaderMap`.
    /// The collection may reserve more space to avoid frequent reallocations.
    pub fn try_reserve(&mut self, additional: usize) -> Result<(), TryReserveError> {
        self.0.try_reserve(additional)
    }

    pub fn values(&self) -> Values<'_, 'a> {
        Values(self.0.values())
    }

    pub fn values_mut(&mut self) -> ValuesMut<'_, 'a> {
        ValuesMut(self.0.values_mut())
    }

    pub fn with_capacity(capacity: usize) -> Self {
        Self(HashMap::with_capacity(capacity))
    }
}

impl<'a> Extend<(HeaderName<'a>, HeaderValue<'a>)> for HeaderMap<'a> {
    fn extend<T: IntoIterator<Item = (HeaderName<'a>, HeaderValue<'a>)>>(&mut self, iter: T) {
        let iter = iter.into_iter();
        self.reserve(iter.size_hint().0);
        for (key, value) in iter {
            self.append_one(key, value);
        }
    }
}

impl<'a> Extend<(HeaderName<'a>, HeaderValues<'a>)> for HeaderMap<'a> {
    fn extend<T: IntoIterator<Item = (HeaderName<'a>, HeaderValues<'a>)>>(&mut self, iter: T) {
        let iter = iter.into_iter();
        self.reserve(iter.size_hint().0);
        for (key, values) in iter {
            self.append_values(key, values);
        }
    }
}

impl<'a> FromIterator<(HeaderName<'a>, HeaderValue<'a>)> for HeaderMap<'a> {
    #[inline]
    fn from_iter<T: IntoIterator<Item = (HeaderName<'a>, HeaderValue<'a>)>>(iter: T) -> Self {
        let mut this = Self::new();
        this.extend(iter);
        this
    }
}

impl<'a> FromIterator<(HeaderName<'a>, HeaderValues<'a>)> for HeaderMap<'a> {
    #[inline]
    fn from_iter<T: IntoIterator<Item = (HeaderName<'a>, HeaderValues<'a>)>>(iter: T) -> Self {
        let mut this = Self::new();
        this.extend(iter);
        this
    }
}

impl<'a, const N: usize> From<[(HeaderName<'a>, HeaderValue<'a>); N]> for HeaderMap<'a> {
    #[inline]
    fn from(array: [(HeaderName<'a>, HeaderValue<'a>); N]) -> Self {
        Self::from_iter(array)
    }
}

impl<'a, const N: usize> From<[(HeaderName<'a>, HeaderValues<'a>); N]> for HeaderMap<'a> {
    #[inline]
    fn from(array: [(HeaderName<'a>, HeaderValues<'a>); N]) -> Self {
        Self::from_iter(array)
    }
}

impl<'a> IntoIterator for HeaderMap<'a> {
    type Item = (HeaderName<'a>, HeaderValues<'a>);

    type IntoIter = IntoIter<'a>;

    fn into_iter(self) -> Self::IntoIter {
        IntoIter(self.0.into_iter())
    }
}

impl<'map, 'a> IntoIterator for &'map HeaderMap<'a> {
    type Item = (&'map HeaderName<'a>, &'map HeaderValues<'a>);

    type IntoIter = Iter<'map, 'a>;

    fn into_iter(self) -> Self::IntoIter {
        self.iter()
    }
}

impl<'map, 'a> IntoIterator for &'map mut HeaderMap<'a> {
    type Item = (&'map HeaderName<'a>, &'map mut HeaderValues<'a>);

    type IntoIter = IterMut<'map, 'a>;

    fn into_iter(self) -> Self::IntoIter {
        self.iter_mut()
    }
}
