use super::HeaderValue;
use std::{iter::FusedIterator, slice};

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct HeaderValues<'a> {
    first: HeaderValue<'a>,
    remaining: Vec<HeaderValue<'a>>,
}

impl<'a> HeaderValues<'a> {
    #[inline]
    pub fn new(first: HeaderValue<'a>) -> Self {
        Self {
            first,
            remaining: Vec::new(),
        }
    }

    pub fn has_only_one(&self) -> bool {
        self.remaining.is_empty()
    }

    pub fn push(&mut self, v: HeaderValue<'a>) {
        self.remaining.push(v);
    }

    pub fn first(&self) -> &HeaderValue<'a> {
        &self.first
    }

    pub fn first_mut(&mut self) -> &mut HeaderValue<'a> {
        &mut self.first
    }

    pub fn into_first(self) -> HeaderValue<'a> {
        self.first
    }

    pub fn iter(&self) -> HeaderValuesIter<'_, 'a> {
        HeaderValuesIter {
            first: Some(&self.first),
            remaining: self.remaining.iter(),
        }
    }
}

impl<'a> From<HeaderValue<'a>> for HeaderValues<'a> {
    #[inline]
    fn from(v: HeaderValue<'a>) -> Self {
        Self::new(v)
    }
}

impl<'a> Extend<HeaderValue<'a>> for HeaderValues<'a> {
    fn extend<T: IntoIterator<Item = HeaderValue<'a>>>(&mut self, iter: T) {
        self.remaining.extend(iter)
    }
}

#[derive(Debug)]
pub struct HeaderValuesIter<'r, 'a> {
    first: Option<&'r HeaderValue<'a>>,
    remaining: slice::Iter<'r, HeaderValue<'a>>,
}

impl<'r, 'a> Iterator for HeaderValuesIter<'r, 'a> {
    type Item = &'r HeaderValue<'a>;

    fn next(&mut self) -> Option<&'r HeaderValue<'a>> {
        self.first.take().or_else(|| self.remaining.next())
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        if self.first.is_some() {
            let length = self.remaining.len() + 1;
            (length, Some(length))
        } else {
            self.remaining.size_hint()
        }
    }
}

impl DoubleEndedIterator for HeaderValuesIter<'_, '_> {
    fn next_back(&mut self) -> Option<Self::Item> {
        self.remaining.next_back().or_else(|| self.first.take())
    }
}

impl ExactSizeIterator for HeaderValuesIter<'_, '_> {}

impl FusedIterator for HeaderValuesIter<'_, '_> {}
