use super::InvalidHeaderValue;
use boar::{Boas, BoasVec};
use std::{convert::TryFrom, fmt, sync::Arc};

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct HeaderValue<'a>(pub(super) BoasVec<'a, u8>);

impl<'a> HeaderValue<'a> {
    fn internal_try_from(bytes: BoasVec<'a, u8>) -> Result<Self, InvalidHeaderValue> {
        validate(&bytes)?;
        Ok(Self(bytes))
    }

    pub fn as_bytes(&self) -> &[u8] {
        &self.0
    }

    pub fn into_static(self) -> HeaderValue<'static> {
        HeaderValue(self.0.into_static())
    }

    pub fn to_borrowed(&self) -> HeaderValue {
        HeaderValue(self.0.to_borrowed())
    }
}

impl HeaderValue<'static> {
    #[inline]
    pub fn ensure_static(&mut self) {
        self.0.ensure_static();
    }

    #[inline]
    pub fn into_ensured_static(mut self) -> Self {
        self.ensure_static();
        self
    }

    pub fn try_from_static(bytes: &'static [u8]) -> Result<Self, InvalidHeaderValue> {
        Self::internal_try_from(Boas::Static(bytes))
    }
}

impl<'a> TryFrom<&'a [u8]> for HeaderValue<'a> {
    type Error = InvalidHeaderValue;

    fn try_from(bytes: &'a [u8]) -> Result<Self, InvalidHeaderValue> {
        Self::internal_try_from(Boas::Borrowed(bytes))
    }
}

impl<'a> TryFrom<Vec<u8>> for HeaderValue<'a> {
    type Error = InvalidHeaderValue;

    fn try_from(bytes: Vec<u8>) -> Result<Self, InvalidHeaderValue> {
        Self::internal_try_from(Boas::Owned(bytes))
    }
}

impl<'a> TryFrom<Arc<Vec<u8>>> for HeaderValue<'a> {
    type Error = InvalidHeaderValue;

    fn try_from(bytes: Arc<Vec<u8>>) -> Result<Self, InvalidHeaderValue> {
        Self::internal_try_from(Boas::Arc(bytes))
    }
}

impl<'a> TryFrom<BoasVec<'a, u8>> for HeaderValue<'a> {
    type Error = InvalidHeaderValue;

    fn try_from(bytes: BoasVec<'a, u8>) -> Result<Self, InvalidHeaderValue> {
        Self::internal_try_from(bytes)
    }
}

impl fmt::Display for HeaderValue<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        for &byte in &*self.0 {
            char::from(byte).fmt(f)?;
        }
        Ok(())
    }
}

const fn validate(bytes: &[u8]) -> Result<(), InvalidHeaderValue> {
    let mut index = 0;
    while index < bytes.len() {
        let byte = bytes[index];
        let invalid = Err(InvalidHeaderValue { index, byte });
        if index == 0 || index == bytes.len() - 1 {
            if !is_visible_or_obsolete_text(byte) {
                return invalid;
            }
        } else if !is_rfc_content(byte) {
            return invalid;
        }
        index += 1;
    }
    Ok(())
}

/// roughly equivalent to "field-content" in RFC 7230
const fn is_rfc_content(byte: u8) -> bool {
    is_visible_or_obsolete_text(byte) || is_rfc_whitespace(byte)
}

const fn is_visible_or_obsolete_text(b: u8) -> bool {
    is_rfc_visible_char(b) || is_rfc_obsolete_text(b)
}

/// "VCHAR" in RFC 5234
const fn is_rfc_visible_char(b: u8) -> bool {
    matches!(b, 33..=126)
}

/// "obs-text" in RFC 7230
const fn is_rfc_obsolete_text(b: u8) -> bool {
    matches!(b, 128..=255)
}

/// "OWS", "RWS" or "BWS" in RFC 7230
const fn is_rfc_whitespace(b: u8) -> bool {
    matches!(b, b' ' | b'\t')
}
