mod constants;
mod invalid;
mod kind;

use self::{invalid::InvalidMethod, kind::MethodKind};
use boar::BoasStr;
use std::fmt;

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct Method<'a>(MethodKind<'a>);

impl<'a> Method<'a> {
    fn internal_try_from(string: BoasStr<'a>) -> Result<Self, InvalidMethod> {
        validate(string.as_bytes())?;
        Ok(Self(
            MethodKind::standard(&string).unwrap_or(MethodKind::Extension(string)),
        ))
    }

    pub fn as_str(&self) -> &str {
        self.0.as_str()
    }

    pub fn into_static(self) -> Method<'static> {
        Method(self.0.into_static())
    }

    pub fn to_borrowed(&self) -> Method {
        Method(self.0.to_borrowed())
    }
}

impl Method<'static> {
    #[inline]
    pub fn ensure_static(&mut self) {
        if let MethodKind::Extension(x) = &mut self.0 {
            x.ensure_static();
        }
    }

    #[inline]
    pub fn into_ensured_static(mut self) -> Self {
        self.ensure_static();
        self
    }

    pub fn try_from_static(string: &'static str) -> Result<Self, InvalidMethod> {
        Self::internal_try_from(BoasStr::Static(string))
    }
}

impl<'a> TryFrom<&'a str> for Method<'a> {
    type Error = InvalidMethod;

    fn try_from(string: &'a str) -> Result<Self, InvalidMethod> {
        Self::internal_try_from(BoasStr::Borrowed(string))
    }
}

impl TryFrom<String> for Method<'_> {
    type Error = InvalidMethod;

    fn try_from(string: String) -> Result<Self, InvalidMethod> {
        Self::internal_try_from(BoasStr::Owned(string))
    }
}

impl<'a> TryFrom<BoasStr<'a>> for Method<'a> {
    type Error = InvalidMethod;

    fn try_from(string: BoasStr<'a>) -> Result<Self, InvalidMethod> {
        Self::internal_try_from(string)
    }
}

impl Default for Method<'_> {
    fn default() -> Self {
        Method::GET
    }
}

impl fmt::Display for Method<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        self.as_str().fmt(f)
    }
}

impl AsRef<str> for Method<'_> {
    #[inline]
    fn as_ref(&self) -> &str {
        self.as_str()
    }
}

fn validate(bytes: &[u8]) -> Result<(), InvalidMethod> {
    match crate::rfc::token::validate(bytes) {
        Ok(()) => Ok(()),
        Err(e) => Err(InvalidMethod {
            byte: e.byte,
            index: e.index,
        }),
    }
}
