use cotton::prelude::*;
use s3_sync::{S3, Settings, Region, Bucket, ObjectBodyMeta, Object, TransferStatus};
use s3_sync::either::Either;
use s3_sync::ensure::{Present, Absent};
use chrono::Utc;
use running_average::RealTimeRunningAverage;
use byte_unit::Byte;
use std::cell::RefCell;

/// S3 Stream Backup
///
/// Backup standard input data to timestamped S3 objects.
/// Keep constant number of backups.
/// List and restore backups to standard output.
#[derive(Debug, StructOpt)]
struct Cli {
    #[structopt(flatten)]
    logging: LoggingOpt,

    #[structopt(flatten)]
    dry_run: DryRunOpt,

    /// Operation timeout in seconds
    #[structopt(short = "t", long, default_value = "10", parse(try_from_str = Duration::from_secs_str))]
    timeout: Duration,

    /// Data transfer timeout in seconds
    #[structopt(short = "T", long, default_value = "300", parse(try_from_str = Duration::from_secs_str))]
    timeout_data: Duration,

    /// Buffer size in MiB (S3 upload part size) - limits maximum size of single backup object
    #[structopt(short = "b", long, default_value = "50")]
    buffer_size_mib: usize,

    /// Do not report upload progress
    #[structopt(short = "n", long)]
    no_porgress: bool,

    /// S3 bucket for backup storage
    bucket: String,

    /// Prefix to use for backup object key, e.g. 'backups/database'
    prefix: String,

    #[structopt(subcommand)]
    action: Action,
}

#[derive(Debug, StructOpt)]
enum Action {
    /// Make new backup
    Make {
        /// Number of recent backups to keep
        keep: usize,
        /// Postfix to use for backup object key, e.g. 'sql.gz' (for information only)
        postfix: Option<String>,
    },
    /// Remove old backups
    Trim {
        /// Number of recent backups to keep
        keep: usize,
    },
    /// List available backups and their restore index
    List,
    /// Restore backup of given index
    Restore {
        /// Index number of the backup to restore; '0' is the latest backup
        #[structopt(default_value = "0")]
        index: usize
    },
}

struct S3Backups {
    s3: S3,
    bucket: Present<Bucket>,
}

impl S3Backups {
    pub fn new(bucket: String, timeout: Duration, data_timeout: Duration, buffer_size_mib: usize) -> PResult<S3Backups> {
        let s3 = S3::new_with_settings(Region::default(), Settings {
                part_size: buffer_size_mib * 1024 * 1024,
                timeout,
                data_timeout,
                .. Settings::default()});

        info!("S3 maximum upload size limit is {} ({} bytes)", bytes(s3.max_upload_size() as u128), s3.max_upload_size());
        info!("Checking bucket: {}", &bucket);
        let bucket = s3.check_bucket_exists(Bucket::from_name(bucket.clone()))
            .problem_while_with(|| format!("checking if bucket {:?} exists", bucket))?;

        let bucket = match bucket {
            Either::Right(Absent(bucket)) => return problem!("bucket {:?} not found", bucket.name()),
            Either::Left(bucket) => bucket,
        };

        Ok(S3Backups {
            s3,
            bucket,
        })
    }

    pub fn make(&mut self, prefix: &str, postfix: Option<&str>, dry_run: bool, no_porgress: bool) -> PResult<()> {
        let date = Utc::now().format("%Y%m%d%H%M%S");

        let key = if let Some(postfix) = postfix {
            format!("{}-{}.{}", prefix, date, postfix)
        } else {
            format!("{}-{}", prefix, date)
        };

        info!("Backup key: {}", &key);

        let object = Absent(Object::from_key(&self.bucket, key.clone())); // assume absent
        let mut data = stdin();

        self.s3.on_upload_progress({
            let key = key.to_owned();
            let rate = RefCell::new(RealTimeRunningAverage::new(Duration::from_secs(16)));
            move |status| {
                match status {
                    TransferStatus::Init => info!("{}: upload started", key),
                    TransferStatus::Progress(progress) => {
                        if !no_porgress {
                            rate.borrow_mut().insert(progress.bytes as f32);
                            info!("{}: uploaded parts: {} bytes: {} @ {}/s", key, progress.buffers, bytes(progress.bytes_total), bytes(rate.borrow_mut().measurement().rate() as u128))
                        }
                    },
                    TransferStatus::Done(progress) => info!("{}: upload done, object size: {} ({} bytes)", key, bytes(progress.bytes_total), progress.bytes_total),
                    TransferStatus::Failed(err) => info!("{}: upload failed: {}", key, err),
                };
            }});

        if !dry_run {
            self.s3.put_object(object, data, ObjectBodyMeta::default())
                .problem_while_with(|| format!("putting object to S3 bucket {:?} under key {:?}", &self.bucket.name(), key))?;
        } else {
            info!("Would put object (dry run)");
            std::io::copy(&mut data, &mut std::io::sink()).problem_while("reading data")?;
        }

        Ok(())
    }

    fn list_objects(&self, prefix: String) -> PResult<Vec<Present<Object>>> {
        info!("Listing objects in bucket {:?} with prefix {:?}", self.bucket.name(), &prefix);
        let list = self.s3.list_objects(&self.bucket, prefix.clone());

        let backups = list.filter(|obj| match obj {
            Ok(obj) => if let Some(rest) = obj.key()
                .strip_prefix(&prefix)
                .and_then(|rest| rest.strip_prefix("-"))
                .and_then(|rest| rest.splitn(2, ".").next()) {
                rest.chars().all(char::is_numeric)
            } else {
                false
            },
            Err(_) => true,
        }).collect::<Result<Vec<_>, _>>()
            .problem_while("fetchig backup list")?;

        Ok(backups)
    }

    pub fn trim(&self, prefix: String, keep: usize, dry_run: bool) -> PResult<()> {
        let backups = self.list_objects(prefix)?;

        let delete_count = backups.len() - std::cmp::min(keep, backups.len());
        let to_delete = backups.into_iter().take(delete_count);

        if !dry_run {
            info!("Deleting {} backups", delete_count);

            for deleted_batch in self.s3.delete_objects(to_delete) {
                for deleted in deleted_batch.problem_while("deleting batch of backups")? {
                    match deleted {
                        Ok(Absent(obj)) => info!("Deleted {:?} backup", obj.key()),
                        Err((obj, err)) => Err(err).problem_while(format!("deleting object {:?}", obj.key()))?,
                    }
                }
            }
        } else {
            info!("Would delete {} backups (dry run)", delete_count);
            for obj in to_delete {
                info!("Would delete {:?} backup (dry run)", obj.key());
            }
        }

        Ok(())
    }

    pub fn list(&self, prefix: String) -> PResult<()> {
        let backups = self.list_objects(prefix)?;
        let count = backups.len();
        for (i, backup) in backups.into_iter().enumerate() {
            println!("{:2}: {}", count - i - 1, backup.key());
        }
        Ok(())
    }

    pub fn restore(&self, prefix: String, index: usize) -> PResult<()> {
        let backups = self.list_objects(prefix)?;

        if let Some(obj) = backups.into_iter().rev().skip(index).next() {
            info!("Restoring {:?}", obj.key());
            let mut body = self.s3.get_body(&obj).problem_while_with(|| format!("getting object {:?} body", obj.key()))?;
            std::io::copy(&mut body, &mut stdout()).problem_while("fetching data")?;
        } else {
            problem!("No such bacup at index {}", index)?;
        }

        Ok(())
    }
}

fn bytes(bytes: impl Into<u128>) -> impl std::fmt::Display {
    Byte::from_bytes(bytes.into()).get_appropriate_unit(false)
}

fn main() -> FinalResult {
    let args = Cli::from_args();
    init_logger(&args.logging, vec![module_path!(), "s3_sync"]);

    let mut s3 = S3Backups::new(args.bucket, args.timeout, args.timeout_data, args.buffer_size_mib)?;

    match args.action {
        Action::Make { keep, postfix } => {
            s3.make(&args.prefix, postfix.as_deref(), args.dry_run.enabled, args.no_porgress).problem_while("making backup")?;
            s3.trim(args.prefix, keep, args.dry_run.enabled).problem_while("triming backups")?;
        }
        Action::Trim { keep } => s3.trim(args.prefix, keep, args.dry_run.enabled).problem_while("triming backups")?,
        Action::List => s3.list(args.prefix).problem_while("listing backups")?,
        Action::Restore { index } => s3.restore(args.prefix, index).problem_while("restoring backup")?,
    }

    Ok(())
}
