use crate::api::*;
use async_trait::async_trait;
use hyper::{body::to_bytes, Body};

#[derive(Debug, Clone)]
pub struct MockLayer;

impl ApiLayer for MockLayer {}

#[async_trait]
impl BucketApiLayer for MockLayer {
    async fn list_buckets(&self, _params: ListBucketsParams) -> ApiResult<ListBucketsReply> {
        let buckets = Vec::<BucketInfo>::new();
        Ok(ListBucketsReply {
            buckets,
            next_marker: String::new(),
            is_truncated: false,
            owner: UserInfo {
                id: String::from("222"),
                display_name: String::from("user222"),
            },
        })
    }

    async fn get_bucket(&self, params: GetBucketParams) -> ApiResult<GetBucketReply> {
        let info = self.make_bucket_info(params.bucket.as_str());
        Ok(GetBucketReply { info })
    }

    async fn put_bucket(&self, params: PutBucketParams) -> ApiResult<PutBucketReply> {
        let info = self.make_bucket_info(params.bucket.as_str());
        Ok(PutBucketReply { info })
    }

    async fn delete_bucket(&self, params: DeleteBucketParams) -> ApiResult<DeleteBucketReply> {
        let info = self.make_bucket_info(params.bucket.as_str());
        Ok(DeleteBucketReply { info })
    }
}

#[async_trait]
impl ObjectApiLayer for MockLayer {
    async fn list_objects(&self, params: ListObjectsParams) -> ApiResult<ListObjectsReply> {
        let mut objects = Vec::<ObjectInfo>::new();
        for i in 1..4 {
            let object =
                self.make_object_info(params.bucket.as_str(), format!("object_{}", i).as_str());
            if object.key.starts_with(&params.prefix) {
                objects.push(object.clone());
            }
        }
        let common_prefixes = Vec::<String>::new();
        Ok(ListObjectsReply {
            objects,
            common_prefixes,
            next_marker: String::new(),
            is_truncated: false,
            bucket: params.bucket.to_owned(),
            prefix: params.prefix.to_owned(),
            delimiter: params.delimiter.to_owned(),
            marker: params.marker.to_owned(),
            max_keys: params.max_keys,
            encoding_type: params.encoding_type.to_owned(),
        })
    }

    async fn get_object(&self, params: GetObjectParams) -> ApiResult<GetObjectReply> {
        let object = self.make_object_info(params.bucket.as_str(), params.key.as_str());
        Ok(GetObjectReply {
            object,
            body: if params.head_only {
                None
            } else {
                Some(Body::from(""))
            },
        })
    }

    async fn put_object(&self, params: PutObjectParams) -> ApiResult<PutObjectReply> {
        let buf = to_bytes(params.body.unwrap()).await.unwrap();
        let mut object = self.make_object_info(params.bucket.as_str(), params.key.as_str());
        object.size = buf.len() as u64;
        Ok(PutObjectReply { object })
    }

    async fn delete_object(&self, params: DeleteObjectParams) -> ApiResult<DeleteObjectReply> {
        let object = self.make_object_info(params.bucket.as_str(), params.key.as_str());
        Ok(DeleteObjectReply { object })
    }
}

impl MockLayer {
    pub fn new() -> Self {
        MockLayer {}
    }

    fn make_bucket_info(&self, bucket: &str) -> BucketInfo {
        BucketInfo {
            name: bucket.to_owned(),
            class: format!("class_{}", bucket),
            region: format!("region_{}", bucket),
            owner: UserInfo {
                id: format!("user_id_{}", bucket),
                display_name: format!("user_name_{}", bucket),
            },
        }
    }

    fn make_object_info(&self, bucket: &str, key: &str) -> ObjectInfo {
        ObjectInfo {
            bucket: bucket.to_owned(),
            key: key.to_owned(),
            version_id: format!("version_id_{}_{}", bucket, key),
            last_modified: format!("last_modified_{}_{}", bucket, key),
            etag: format!("etag_{}_{}", bucket, key),
            storage_class: format!("storage_class_{}_{}", bucket, key),
            size: 0,
            owner: UserInfo {
                id: format!("user_id_{}_{}", bucket, key),
                display_name: format!("user_name_{}_{}", bucket, key),
            },
        }
    }
}
