use super::*;
use async_trait::async_trait;
use hyper::Body;
use std::fmt::Debug;

/// ObjectApiLayer is an abstract API to an S3-like object store.
/// implementing this trait allows to extend the composable API's.
#[async_trait]
pub trait ObjectApiLayer: Debug + Send + Sync {
    async fn list_objects(&self, params: ListObjectsParams) -> ApiResult<ListObjectsReply>;
    async fn get_object(&self, params: GetObjectParams) -> ApiResult<GetObjectReply>;
    async fn put_object(&self, params: PutObjectParams) -> ApiResult<PutObjectReply>;
    async fn delete_object(&self, params: DeleteObjectParams) -> ApiResult<DeleteObjectReply>;

    // TODO: multipart upload
    // async fn initiate_multipart_upload(&self, params: InitiateMultipartUploadParams) -> ApiResult<InitiateMultipartUploadReply>;
    // async fn complete_multipart_upload(&self, params: CompleteMultipartUploadParams) -> ApiResult<CompleteMultipartUploadReply>;
    // async fn put_upload_part(&self, params: PutUploadPartParams) -> ApiResult<PutUploadPartReply>;
    // async fn list_upload_parts(&self, params: ListUploadPartsParams) -> ApiResult<ListUploadPartsReply>;

    // TODO: batch object operations (see http://docs.aws.amazon.com/AmazonS3/latest/API/batch-ops-deleting-objects.html)
    // async fn get_objects(&self, params: GetObjectsParams) -> ApiResult<GetObjectsReply>;
    // async fn put_objects(&self, params: PutObjectsParams) -> ApiResult<PutObjectsReply>;
    // async fn delete_objects(&self, params: DeleteObjectsParams) -> ApiResult<DeleteObjectsReply>;

    // TODO: versioning
    // async fn list_object_versions(&self, params: ListObjectsParams) -> ApiResult<ListObjectsReply>;
}

#[derive(Debug, Clone)]
pub struct ObjectInfo {
    pub bucket: String,
    pub key: String,
    pub version_id: String,
    pub size: u64,
    pub last_modified: String,
    pub etag: String,
    pub storage_class: String,
    pub owner: UserInfo,
}

#[derive(Debug, Clone)]
pub struct ObjectRange {
    pub start: Option<u64>,
    pub end: Option<u64>,
}

#[derive(Debug, Clone)]
pub struct AuthInfo {
    pub access_key: String,
    pub secret_key: String,
    pub owner: UserInfo,
}

#[derive(Debug, Clone)]
pub struct MetaParams {
    pub bucket: String, // can be empty
    pub key: String,    // can be empty
    pub auth: Option<AuthInfo>,
}

#[derive(Debug, Clone)]
pub struct ListObjectsParams {
    pub meta: MetaParams,
    pub bucket: String,
    pub prefix: String,
    pub delimiter: String,
    pub marker: String,
    pub max_keys: i32,
    pub encoding_type: String,
    // TODO: list objects v2
    // pub start_after: String,
}

#[derive(Debug, Clone)]
pub struct ListObjectsReply {
    pub bucket: String,
    pub prefix: String,
    pub delimiter: String,
    pub marker: String,
    pub max_keys: i32,
    pub encoding_type: String,

    // TODO: list objects v2
    // pub start_after: String,
    pub is_truncated: bool,
    pub next_marker: String,

    pub objects: Vec<ObjectInfo>,
    pub common_prefixes: Vec<String>,
}

#[derive(Debug, Clone)]
pub struct GetObjectParams {
    pub bucket: String,
    pub key: String,
    pub version_id: String,

    // partial reads
    pub head_only: bool, // = HTTP HEAD method - no body should be returned
    pub range: ObjectRange,
    // TODO: conditional reads (e.g. if-match, if-none-match) ?
    // pub if_modified_since: Option<Time>,
    // pub if_unmodified_since: Option<Time>,
    // pub if_match: Option<String>,
    // pub if_none_match: Option<String>,
}

#[derive(Debug)]
pub struct GetObjectReply {
    pub object: ObjectInfo,
    pub body: Option<Body>,
}

#[derive(Debug)]
pub struct PutObjectParams {
    pub bucket: String,
    pub key: String,
    pub body: Option<Body>,
    // TODO partial updates
    // pub head_only: bool, // put only headers but keep content
    // pub range: ObjectRange, // pub only the selected range of the object

    // TODO: conditional uploads (e.g. if-match, if-none-match) ?
    // pub if_modified_since: Option<Time>,
    // pub if_unmodified_since: Option<Time>,
    // pub if_match: Option<String>,
    // pub if_none_match: Option<String>,
}

#[derive(Debug, Clone)]
pub struct PutObjectReply {
    pub object: ObjectInfo,
}

#[derive(Debug, Clone)]
pub struct DeleteObjectParams {
    pub bucket: String,
    pub key: String,
    pub version_id: String,
}

#[derive(Debug, Clone)]
pub struct DeleteObjectReply {
    pub object: ObjectInfo,
}
