use crate::api::*;
use async_trait::async_trait;
use std::fmt::Debug;

pub trait ApiLayer: BucketApiLayer + DynamicObjectApiResolver {}

#[async_trait]
pub trait DynamicObjectApiResolver {
    async fn resolve_object_api<'a>(&'a self, bucket: &str) -> &'a dyn ObjectApiLayer;
}

#[async_trait]
impl<T: ObjectApiLayer> DynamicObjectApiResolver for T {
    async fn resolve_object_api<'a>(&'a self, _bucket: &str) -> &'a dyn ObjectApiLayer {
        self
    }
}

/// BucketApiLayer is an abstract API to an S3-like object store.
/// implementing this trait can be used to add new functionality
/// (i.e list buckets from Azure, or adding caching layer for remote deployments)
/// while still reusing the other implementations in other server configurations
#[async_trait]
pub trait BucketApiLayer: Debug + Send + Sync {
    async fn list_buckets(&self, params: ListBucketsParams) -> ApiResult<ListBucketsReply>;
    async fn get_bucket(&self, params: GetBucketParams) -> ApiResult<GetBucketReply>;
    async fn put_bucket(&self, params: PutBucketParams) -> ApiResult<PutBucketReply>;
    async fn delete_bucket(&self, params: DeleteBucketParams) -> ApiResult<DeleteBucketReply>;
}

#[derive(Debug, Clone)]
pub struct BucketInfo {
    pub name: String,
    pub class: String,
    pub region: String,
    pub owner: UserInfo,
}

#[derive(Debug, Clone)]
pub struct UserInfo {
    pub id: String,
    pub display_name: String,
}

#[derive(Debug, Clone)]
pub struct ListBucketsParams {
    // empty for now
}

#[derive(Debug, Clone)]
pub struct ListBucketsReply {
    pub buckets: Vec<BucketInfo>,
    pub is_truncated: bool,
    pub next_marker: String,
    pub owner: UserInfo,
}

#[derive(Debug, Clone)]
pub struct GetBucketParams {
    pub bucket: String,
}

#[derive(Debug, Clone)]
pub struct GetBucketReply {
    pub info: BucketInfo,
}

#[derive(Debug, Clone)]
pub struct PutBucketParams {
    pub bucket: String,
    pub class: String,
}

#[derive(Debug, Clone)]
pub struct PutBucketReply {
    pub info: BucketInfo,
}

#[derive(Debug, Clone)]
pub struct DeleteBucketParams {
    pub bucket: String,
}

#[derive(Debug, Clone)]
pub struct DeleteBucketReply {
    pub info: BucketInfo,
}
