use super::*;
use hyper::{Body, HeaderMap, Method};
use std::{collections::HashMap, error::Error};

pub type AnyError = Box<dyn Error + Send + Sync>;
pub type ResultOrAnyErr<T> = Result<T, AnyError>;

pub type HttpRequest = hyper::Request<Body>;
pub type HttpResponse = hyper::Response<Body>;
pub type HttpResult = ResultOrAnyErr<HttpResponse>;
pub type HttpResultOrErr<E: Error> = Result<HttpResponse, E>;

pub struct ApiRequest {
    pub op: ApiOps,
    pub bucket: String,
    pub key: String,
    pub query: HashMap<String, String>,
    pub headers: HashMap<String, String>,
    pub body: Body,
}

impl From<HttpRequest> for ApiRequest {
    fn from(req: HttpRequest) -> Self {
        let mut query = HashMap::new();
        for (key, value) in req.uri().query().unwrap_or("").split('&').map(|s| s.split('=')) {
            query.insert(key.to_string(), value.to_string());
        }
        Self {
            op: ApiOps::from_str(req.method().as_str()).unwrap(),
            bucket: req.uri().path().trim_start_matches('/').split('/').nth(0).unwrap_or("").to_string(),
            key: req.uri().path().trim_start_matches('/').split('/').nth(1).unwrap_or("").to_string(),
            query,
            headers: req.headers().clone().into_iter().map(|(k, v)| (k.as_str().to_string(), v.to_str().unwrap().to_string())).collect(),
            body: req.into_body(),
        }
    }
}

pub type ApiResult<T> = Result<T, ApiError>;

/// See https://docs.aws.amazon.com/AmazonS3/latest/API/API_Operations_Amazon_Simple_Storage_Service.html
pub enum ApiOps {
    AbortMultipartUpload,
    CompleteMultipartUpload,
    CopyObject,
    CreateBucket,
    CreateMultipartUpload,
    DeleteBucket,
    DeleteBucketAnalyticsConfiguration,
    DeleteBucketCors,
    DeleteBucketEncryption,
    DeleteBucketIntelligentTieringConfiguration,
    DeleteBucketInventoryConfiguration,
    DeleteBucketLifecycle,
    DeleteBucketMetricsConfiguration,
    DeleteBucketOwnershipControls,
    DeleteBucketPolicy,
    DeleteBucketReplication,
    DeleteBucketTagging,
    DeleteBucketWebsite,
    DeleteObject,
    DeleteObjects,
    DeleteObjectTagging,
    DeletePublicAccessBlock,
    GetBucketAccelerateConfiguration,
    GetBucketAcl,
    GetBucketAnalyticsConfiguration,
    GetBucketCors,
    GetBucketEncryption,
    GetBucketIntelligentTieringConfiguration,
    GetBucketInventoryConfiguration,
    GetBucketLifecycle,
    GetBucketLifecycleConfiguration,
    GetBucketLocation,
    GetBucketLogging,
    GetBucketMetricsConfiguration,
    GetBucketNotification,
    GetBucketNotificationConfiguration,
    GetBucketOwnershipControls,
    GetBucketPolicy,
    GetBucketPolicyStatus,
    GetBucketReplication,
    GetBucketRequestPayment,
    GetBucketTagging,
    GetBucketVersioning,
    GetBucketWebsite,
    GetObject,
    GetObjectAcl,
    GetObjectLegalHold,
    GetObjectLockConfiguration,
    GetObjectRetention,
    GetObjectTagging,
    GetObjectTorrent,
    GetPublicAccessBlock,
    HeadBucket,
    HeadObject,
    ListBucketAnalyticsConfigurations,
    ListBucketIntelligentTieringConfigurations,
    ListBucketInventoryConfigurations,
    ListBucketMetricsConfigurations,
    ListBuckets,
    ListMultipartUploads,
    ListObjects,
    ListObjectsV2,
    ListObjectVersions,
    ListParts,
    PutBucketAccelerateConfiguration,
    PutBucketAcl,
    PutBucketAnalyticsConfiguration,
    PutBucketCors,
    PutBucketEncryption,
    PutBucketIntelligentTieringConfiguration,
    PutBucketInventoryConfiguration,
    PutBucketLifecycle,
    PutBucketLifecycleConfiguration,
    PutBucketLogging,
    PutBucketMetricsConfiguration,
    PutBucketNotification,
    PutBucketNotificationConfiguration,
    PutBucketOwnershipControls,
    PutBucketPolicy,
    PutBucketReplication,
    PutBucketRequestPayment,
    PutBucketTagging,
    PutBucketVersioning,
    PutBucketWebsite,
    PutObject,
    PutObjectAcl,
    PutObjectLegalHold,
    PutObjectLockConfiguration,
    PutObjectRetention,
    PutObjectTagging,
    PutPublicAccessBlock,
    RestoreObject,
    SelectObjectContent,
    UploadPart,
    UploadPartCopy,
    WriteGetObjectResponse,
}
