use std::borrow::BorrowMut;
use std::collections::HashMap;
use std::error::Error;
use std::fs::File;
use std::io::Read;
use std::ops::Mul;
use std::thread;
use std::time::Duration;
use std::process::exit;

/// [所有权](https://rustwiki.org/zh-CN/book/ch04-01-what-is-ownership.html)
pub fn str_own() {
    let s1 = String::from("ownership");
    let mut s2 = s1;
    // print!("{s1}")
    println!("{}", calculate_str_len(&mut s2));

    add_suffix(&mut s2);
    println!("{}", s2);

    // cannot borrow `s2` as mutable more than once at a time
    // 在特定作用域中的特定数据只能有一个可变引用
    let mut _s3 = &mut s2;
    // let mut _s4 = &mut s2;
    // println!("{}, {}", _s3, _s4)
}

/// [引用和借](https://rustwiki.org/zh-CN/book/ch04-02-references-and-borrowing.html)
/// 引用中有一个指针，该指针指向了变量
fn calculate_str_len(s: &String) -> usize {
    s.len()
}

fn add_suffix(s: &mut String) {
    s.push_str("_cn");
}

/// 不允许创建野指针
/// ```
/// fn dangle()-> &String {
///    let s = "dangle";
///     return &s
/// }
/// ```
fn dangle() -> String {
    String::from("Dangle")
}

/// [slice](https://rustwiki.org/zh-CN/book/ch04-03-slices.html)
pub fn first_word(s: &str) -> &str {
    let b = s.as_bytes();
    let mut end = s.len();
    for (index, &item) in b.iter().enumerate() {
        if b' ' == item {
            end = index;
        }
    }
    &s[..end]
}

#[test]
pub fn other_slice() {
    let a = [0, 1, 2, 3];
    let slice = &a[1..4];
    let b: String = slice.iter().map(|&x| format!("{},", &x.mul(2))).collect();
    print!("{}", b)
}

/// [结构体](https://rustwiki.org/zh-CN/book/ch05-00-structs.html)
struct Panda {
    height: f64,
    weight: f64,
}

impl Panda {
    pub fn new(height: f64, weight: f64) -> Self {
        Panda { height, weight }
    }

    pub fn eat() {
        print!("eat bamboo")
    }

    pub fn detail(&self) -> String {
        let f = format!("height = {}, weight = {}", self.height, self.weight);
        f
    }
}

/// 元组结构体
struct Color(i32, i32, i32);

struct Point(f64, f64, f64);

/// [enum 枚举](https://rustwiki.org/zh-CN/book/ch06-01-defining-an-enum.html)
enum IpAddressKind {
    IPv4,
    Ipv6,
}

fn get_add_type() -> IpAddressKind {
    IpAddressKind::IPv4
}

fn is_v4(kind: IpAddressKind) -> bool {
    match kind {
        IpAddressKind::IPv4 => true,
        IpAddressKind::Ipv6 => false,
    }
}

/// [vector](https://rustwiki.org/zh-CN/book/ch08-01-vectors.html)
#[test]
fn vector_sample() {
    let v = vec![1, 2, 3, 4];
    let c: String = v.iter().map(|&x| format!("{},", x)).collect();
    println!("v content = {}, size = {}", c, v.len());

    let mut v2 = v;
    for item in &mut v2 {
        // * 解饮用
        *item += *item
    }

    let c1: String = v2.iter().map(|&x| format!("{},", x)).collect();
    println!("v2 content = {}, size = {}", c1, v2.len());

    enum SpreadsheetCell {
        Int(i32),
        Float(f64),
        Text(String),
    }

    let row = vec![
        SpreadsheetCell::Int(3),
        SpreadsheetCell::Text(String::from("blue")),
        SpreadsheetCell::Float(10.12),
    ];
}

/// [字符串](https://rustwiki.org/zh-CN/book/ch08-02-strings.html)
#[test]
fn string_sample() {
    let mut s = String::new();
    s.push('\n');
    s.push_str("wtf");

    let s1 = s + "1";
    // println!("{}", &s1);

    let s2 = format!("{}, {}, {}", "1", "2", s1);
    // println!("{}", &s2);

    // 字节   Bytes
    // 标量值 scalar values
    // 字形簇 grapheme clusters

    let india = "नमस्ते";
    for i in india.chars() {
        println!("{}", i);
    }
    println!("bytes len = {}", india.bytes().len());
    for j in india.bytes() {
        println!("{}", j);
    }

    // will panic
    // 必须按照字符边界来切割
    // let s3 = &india[0..4];
    let s3 = &india[0..3];
    println!("slice {}", s3);
}

/// [HashMap](https://rustwiki.org/zh-CN/book/ch08-03-hash-maps.html)
#[test]
fn hash_map_sample() {
    let mut map = HashMap::new();
    map.insert("lakers", 11);
    map.insert("sun", 10);

    let teams = vec![String::from("lakers"), String::from("sun")];
    let mut scores = vec![11, 10];
    let mut map2: HashMap<_, _> = teams.iter().zip(scores.iter()).collect();

    for (key, value) in &map2 {
        println!("key = {}, value = {}", key, value)
    }
    println!("{:?}", &map2);

    let &mut update = map2.entry(&"lakers".to_string()).or_insert(&mut 1);
    // todo *update += 1;

    let update = map.entry("sun").or_insert(1);
    *update += 1;
    println!("{:?}", map)
}

/// [错误处理](https://rustwiki.org/zh-CN/book/ch09-00-error-handling.html)
///
/// - 可恢复的错误
/// - 不可恢复的错误
/// `Box<dyn Error>` 是 `trait` 对象 理解为任何可能的错误
///
#[test]
fn error_handling() -> Result<(), Box<dyn Error>> {
    // panic!("crash")
    let mut s1: String = String::new();
    let f = File::open("odk.txt");
    let mut r = match f {
        Ok(t) => t,
        Err(e) => panic!("{}", e),
    };
    File::open("odk.txt")
        .unwrap()
        .read_to_string(&mut s1)
        .unwrap();
    File::open("hello.txt")?;
    Ok(())
}

/// [泛型](https://rustwiki.org/zh-CN/book/ch10-00-generics.html)
/// Rust 通过在编译时进行泛型代码的 单态化（monomorphization）来保证效率。单态化是一个通过填充编译时使用的具体类型，将通用代码转换为特定代码的过程。
#[test]
fn generic_sample() {
    assert_eq!(4, largest(&vec![1, 2, 3, 4]));
    assert_eq!("d", largest(&vec!["a", "b", "c", "d"]))
}

fn largest<T>(list: &[T]) -> T
    where
        T: std::cmp::PartialOrd,
        T: std::marker::Copy,
{
    let mut largest = list[0];
    for &item in list.iter() {
        if item > largest {
            largest = item
        }
    }
    largest
}

struct Offset<T> {
    x: T,
    y: T,
}

enum Option<T> {
    Some(T),
    None,
}

/// [traits](https://rustwiki.org/zh-CN/book/ch10-02-traits.html) sample
pub trait Summary {
    fn summarize(&self) -> String;
}

pub trait Author {
    fn name(&self) -> String {
        "unknown".to_string()
    }
}

pub struct Tweet {
    pub content: String,
    pub name: String,
    pub icon: String,
}

impl Summary for Tweet {
    fn summarize(&self) -> String {
        format!("{}: {}", self.name, self.content)
    }
}

impl Author for Tweet {
    fn name(&self) -> String {
        self.name.clone()
    }
}

pub fn print_title(item: impl Summary) {
    println!("{}", item.summarize())
}

/// [lifetime](https://rustwiki.org/zh-CN/book/ch10-03-lifetime-syntax.html) sample
/// ```
/// &i32        // 引用
/// &'a i32     // 带有显式生命周期的引用
/// &'a mut i32 // 带有显式生命周期的可变引用
/// ```
fn longest<'a>(x: &'a str, y: &'a str) -> &'a str {
    if x.len() > y.len() {
        x
    } else {
        y
    }
}

struct ImportantExcerpt<'a> {
    part: &'a str,
}

fn global_str() {
    let a: &'static str = "aaa";
    print!("{}", a)
}

/// https://rustwiki.org/zh-CN/book/ch11-01-writing-tests.html
#[cfg(test)]
mod test {
    #[test]
    fn exploration() {
        assert_eq!(2 + 2, 4);
    }

    #[test]
    #[should_panic(expected = "Make this test")]
    fn another() {
        panic!("Make this test fail");
    }

    #[test]
    fn it_works() -> Result<(), String> {
        if 2 + 2 == 4 {
            Ok(())
        } else {
            Err(String::from("two plus two does not equal four"))
        }
    }

    #[test]
    #[ignore]
    fn expensive_test() {
        // 需要运行一个小时的代码
    }
}

/// [闭包](https://rustwiki.org/zh-CN/book/ch13-01-closures.html)
/// ```
/// fn  add_one_v1   (x: u32) -> u32 { x + 1 }
/// let add_one_v2 = |x: u32| -> u32 { x + 1 };
/// let add_one_v3 = |x|             { x + 1 };
/// let add_one_v4 = |x|               x + 1  ;
/// ```

#[test]
fn closures_sample() {
    let expensive_closure = |x| {
        thread::sleep(Duration::from_secs(1));
        println!("{}", "closure exe");
        x
    };
    let x = 1;
    expensive_closure(x);

    // 使用结构体中的闭包来简化代码
    let mut c = Cacher::new(|x| {
        thread::sleep(Duration::from_secs(1));
        x
    });
    let i2 = c.value(2);
    assert_eq!(2, i2);
    let i3 = c.value(3);
    assert_eq!(2, i3);
}

/// 结构体中的闭包
struct Cacher<T : Fn(i32) -> i32> {
    calculation: T,
    value: core::option::Option<i32>,
}

impl <T> Cacher<T> where T : Fn(i32) -> i32 {

    pub fn new(calculation: T) -> Cacher<T> {
        Cacher {
            calculation,
            value: None,
        }
    }

    fn value(&mut self, arg: i32) -> i32 {
        match self.value {
            Some(v) => v,
            None => {
                let v = (self.calculation)(arg);
                self.value = Some(v);
                v
            }
        }
    }
}

/// 闭包获取环境变量
#[test]
fn closure_context() {
    let x = 4;
    let y = 4;
    let equal_to_x = |n| n == x;
    assert!(equal_to_x(y));

    let v = String::from("move");
    let move_equal_to_v = move |z| z == v;
    let w = String::from("not move");

    move_equal_to_v(w);
    // error[E0382]: borrow of moved value: `v`
    // print!("cant use {:?}", v)
}

/// [迭代器](https://rustwiki.org/zh-CN/book/ch13-02-iterators.html)
struct Counter {
    count: u32,
}

impl Counter {
    fn new() -> Counter {
        Counter { count: 0 }
    }
}

impl Iterator for Counter {

    type Item = u32;

    fn next(&mut self) -> std::option::Option<Self::Item> {
        self.count += 1;

        if self.count < 6 {
            Some(self.count)
        } else {
            None
        }
    }
}

#[test]
fn using_other_iterator_trait_methods() {
    let sum: u32 = Counter::new()
        .zip(Counter::new().skip(1))
        .map(|(a, b)|  a * b)
        .filter(|x| x % 3 == 0)
        .sum();

    assert_eq!(18, sum)
}

