/*!
# Restricted XML 1.0 parsing facilities

This module contains parsing facilities for processing a token stream
generated by the [`Lexer`]. To satisfy different styles of tree building, the
parsing step is again separated in two stages: the logical stage and the
namespace/attribute resolution stage.

## Logical stage

In the logical stage, the logical elements of the XML document are emitted as
[`RawEvent`] structs. These *may* be used by end-users to build XML document
models, but they are not completely validated even to the XML 1.0
specification. The caveats are documented at the [`Parser`] struct.

## Namespace/attribute resolution stage

This stage resolves namespace declarations found in a stream of [`RawEvent`]
structs and emits [`ResolvedEvent`] structs. It is implemented by the
[`NamespaceResolver`] struct.

Together with the validation of the logical stage, this provides full
conformity checks according to XML 1.0 and Namespaces for XML 1.0.

The downside of using this stage is added processing cost, because
considerable dynamic allocations need to be performed per-element (for
attribute hash maps). In addition, information about the prefixes used to
declare namespaces is lost (but nothing should rely on those anyway).

   [`Lexer`]: crate::Lexer
*/

mod common;
mod namespaces;
mod raw;

use crate::context;
use crate::error::Result;

#[doc(inline)]
pub use common::*;
#[doc(inline)]
pub use namespaces::{NamespaceName, NamespaceResolver, ResolvedEvent, ResolvedQName};
#[doc(inline)]
pub use raw::{RawEvent, RawParser, RawQName};

/**
# Low-level restricted XML 1.0 parser

The [`Parser`] converts [`crate::lexer::Token`]s into [`ResolvedEvent`]s.

It is a low-level interface which expects to be driven from a [`TokenRead`]
source.

The high-level interfaces are [`FeedParser`], [`PullParser`] and
[`AsyncParser`], depending on the use case. See the [`rxml`] top-level
documentation for details.

In contrast to a [`RawParser`], the [`Parser`] enforces well-formedness and
namespace-well-formedness.

   [`FeedParser`]: crate::FeedParser
   [`PullParser`]: crate::PullParser
   [`AsyncParser`]: crate::AsyncParser
   [`rxml`]: crate
*/
pub struct Parser {
	inner: RawParser,
	resolver: NamespaceResolver,
}

impl Default for Parser {
	fn default() -> Self {
		Self::with_context(RcPtr::new(context::Context::new()))
	}
}

impl WithContext for Parser {
	fn with_context(ctx: RcPtr<context::Context>) -> Self {
		Self {
			inner: RawParser::new(),
			resolver: NamespaceResolver::with_context(ctx),
		}
	}
}

impl Parse for Parser {
	type Output = ResolvedEvent;

	fn parse<R: TokenRead>(&mut self, r: &mut R) -> Result<Option<Self::Output>> {
		let inner = &mut self.inner;
		self.resolver.next(|| inner.parse(r))
	}

	fn release_temporaries(&mut self) {
		self.inner.release_temporaries();
		self.resolver.context().release_temporaries();
	}
}
