pub struct FileType {
    name: String,
    hl_opts: HighlightingOptions,
}

#[derive(Default)]
pub struct HighlightingOptions {
    numbers: bool,
    strings: bool,
    characters: bool,
    comments: bool,
    multiline_comments: bool,
    primary_keywords: Vec<String>,
    secondary_keywords: Vec<String>,
}

impl Default for FileType {
    fn default() -> Self {
        Self {
            name: String::from("No filetype"),
            hl_opts: HighlightingOptions::default(),
        }
    }
}

macro_rules! str_vec {
    ($($x:expr),*) => (vec![$($x.to_string()),*]);
}

impl FileType {
    pub fn name(&self) -> String {
        self.name.clone()
    }
    pub fn highlighting_options(&self) -> &HighlightingOptions {
        &self.hl_opts
    }
    pub fn from(file_name: &str) -> Self {
        if file_name.ends_with(".toml") {
            return Self {
                name: String::from("TOML"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: false,
                    primary_keywords: str_vec!["true", "false"],
                    secondary_keywords: str_vec!["[", "]"],
                },
            };
        } else if file_name.ends_with(".gitignore") {
            return Self {
                name: String::from("Gitignore"),
                hl_opts: HighlightingOptions {
                    numbers: false,
                    strings: false,
                    characters: true,
                    comments: true,
                    multiline_comments: false,
                    primary_keywords: str_vec![],
                    secondary_keywords: str_vec![],
                },
            };
        } else if file_name.ends_with(".ini")
            || file_name.ends_with(".cfg")
            || file_name.ends_with(".conf")
        {
            return Self {
                name: String::from("INI"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: false,
                    primary_keywords: str_vec![";", " ", "#"],
                    secondary_keywords: str_vec![],
                },
            };
        } else if file_name.ends_with(".hs") {
            return Self {
                name: String::from("Haskell"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "case",
                        "class",
                        "data",
                        "default",
                        "deriving",
                        "do",
                        "else",
                        "forall",
                        "if",
                        "import",
                        "in",
                        "infix",
                        "infixl",
                        "infixr",
                        "instance",
                        "let",
                        "module",
                        "newtype",
                        "of",
                        "qualified",
                        "then",
                        "type",
                        "where",
                        "_",
                        "foreign",
                        "ccall",
                        "as",
                        "safe",
                        "unsafe"
                    ],
                    secondary_keywords: str_vec![
                        "..", "::", "=", "\\", "|", "<-", "->", "@", "~", "=>", "[", "]", "$", "!",
                        "."
                    ],
                },
            };
        } else if file_name.ends_with(".sh") {
            return Self {
                name: String::from("Bash"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "case", "do", "done", "elif", "else", "esac", "fi", "for", "function",
                        "if", "in", "select", "then", "time", "until", "while"
                    ],
                    secondary_keywords: str_vec![
                        "alias",
                        "bind",
                        "builtin",
                        "caller",
                        "command",
                        "declare",
                        "echo",
                        "enable",
                        "help",
                        "let",
                        "local",
                        "logout",
                        "mapfile",
                        "printf",
                        "read",
                        "readarray",
                        "source",
                        "type",
                        "typeset",
                        "ulimit",
                        "unalias",
                        "pwd",
                        "ls",
                        "cd",
                        "mkdir",
                        "echo",
                        "cat",
                        "cp",
                        "mv",
                        "rm",
                        "man",
                        "head",
                        "tail",
                        "less",
                        "more",
                        "sort",
                        "grep",
                        "which",
                        "chmod",
                        "history",
                        "clear"
                    ],
                },
            };
        } else if file_name.ends_with(".r") {
            return Self {
                name: String::from("R"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "for",
                        "in",
                        "repeat",
                        "while",
                        "function",
                        "if",
                        "else",
                        "next",
                        "break",
                        "TRUE",
                        "FALSE",
                        "NULL",
                        "Inf",
                        "NaN",
                        "NA",
                        "NA_integer_",
                        "NA_real_",
                        "NA_complex_",
                        "NA_character_",
                        "..."
                    ],
                    secondary_keywords: str_vec![],
                },
            };
        } else if file_name.ends_with(".json") {
            return Self {
                name: String::from("JSON"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: false,
                    multiline_comments: false,
                    primary_keywords: str_vec!["true", "false", "null"],
                    secondary_keywords: str_vec!["[", "]", "{", "}"],
                },
            };
        } else if file_name.ends_with(".cs") {
            return Self {
                name: String::from("C#"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "abstract",
                        "as",
                        "base",
                        "bool",
                        "break",
                        "byte",
                        "case",
                        "catch",
                        "char",
                        "checked",
                        "class",
                        "const",
                        "continue",
                        "decimal",
                        "default",
                        "delegate",
                        "do",
                        "double",
                        "else",
                        "enum",
                        "event",
                        "explicit",
                        "extern",
                        "false",
                        "finally",
                        "fixed",
                        "float",
                        "for",
                        "foreach",
                        "goto",
                        "if",
                        "implicit",
                        "in",
                        "int",
                        "interface",
                        "internal",
                        "is",
                        "lock",
                        "long",
                        "namespace",
                        "new",
                        "null",
                        "object",
                        "operator",
                        "out",
                        "override",
                        "params",
                        "private",
                        "protected",
                        "public",
                        "readonly",
                        "ref",
                        "return",
                        "sbyte",
                        "sealed",
                        "short",
                        "sizeof",
                        "stackalloc",
                        "static",
                        "string",
                        "struct",
                        "switch",
                        "this",
                        "throw",
                        "true",
                        "try",
                        "typeof",
                        "uint",
                        "ulong",
                        "unchecked",
                        "unsafe",
                        "ushort",
                        "using",
                        "virtual",
                        "void",
                        "volatile",
                        "while",
                        "add",
                        "and",
                        "alias",
                        "ascending",
                        "async",
                        "await",
                        "by",
                        "descending",
                        "dynamic",
                        "equals",
                        "from",
                        "get",
                        "global",
                        "group",
                        "init",
                        "into",
                        "join",
                        "let",
                        "managed",
                        "nameof",
                        "nint",
                        "not",
                        "notnull",
                        "nuint",
                        "on",
                        "or",
                        "orderby",
                        "partial",
                        "partial",
                        "record",
                        "remove",
                        "select",
                        "set",
                        "unmanaged",
                        "unmanaged",
                        "value",
                        "var",
                        "when",
                        "where",
                        "where",
                        "with",
                        "yield"
                    ],
                    secondary_keywords: str_vec![
                        "bool",
                        "System.Boolean",
                        "byte",
                        "System.Byte",
                        "sbyte",
                        "System.SByte",
                        "char",
                        "System.Char",
                        "decimal",
                        "System.Decimal",
                        "double",
                        "System.Double",
                        "float",
                        "System.Single",
                        "int",
                        "System.Int32",
                        "uint",
                        "System.UInt32",
                        "nint",
                        "System.IntPtr",
                        "nuint",
                        "System.UIntPtr",
                        "long",
                        "System.Int64",
                        "ulong",
                        "System.UInt64",
                        "short",
                        "System.Int16",
                        "ushort",
                        "System.UInt16",
                        "object",
                        "System.Object",
                        "string",
                        "System.String",
                        "dynamic",
                        "System.Object"
                    ],
                },
            };
        } else if file_name.ends_with(".c") {
            return Self {
                name: String::from("C"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "auto", "break", "case", "const", "continue", "default", "do", "enum",
                        "extern", "for", "goto", "if", "register", "return", "sizeof", "static",
                        "struct", "switch", "typedef", "union", "void", "volatile", "while",
                        "#include", "#ifndef", "#if", "#endif", "#define", "#undef"
                    ],
                    secondary_keywords: str_vec![
                        "char",
                        "int",
                        "long",
                        "unsigned",
                        "float",
                        "double",
                        "size_t",
                        "signed",
                        "short",
                        "wchar_t",
                        "__int128_t",
                        "bool"
                    ],
                },
            };
        } else if file_name.ends_with(".cc")
            || file_name.ends_with(".cpp")
            || file_name.ends_with(".C")
            || file_name.ends_with(".h")
            || file_name.ends_with(".hh")
            || file_name.ends_with(".hpp")
        {
            return Self {
                name: String::from("C++"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "alignas",
                        "alignof",
                        "and",
                        "and_eq",
                        "asm",
                        "atomic_cancel",
                        "atomic_commit",
                        "atomic_noexcept",
                        "auto",
                        "bitand",
                        "bitor",
                        "bool",
                        "break",
                        "case",
                        "catch",
                        "char",
                        "char8_t",
                        "char16_t",
                        "char32_t",
                        "class",
                        "compl",
                        "concept",
                        "const",
                        "consteval",
                        "constexpr",
                        "constinit",
                        "const_cast",
                        "continue",
                        "co_await",
                        "co_return",
                        "co_yield",
                        "decltype",
                        "default",
                        "delete",
                        "do",
                        "double",
                        "dynamic_cast",
                        "else",
                        "enum",
                        "explicit",
                        "export",
                        "extern",
                        "false",
                        "float",
                        "for",
                        "friend",
                        "goto",
                        "if",
                        "inline",
                        "int",
                        "long",
                        "mutable",
                        "namespace",
                        "new",
                        "noexcept",
                        "not",
                        "not_eq",
                        "nullptr",
                        "operator",
                        "or",
                        "or_eq",
                        "private",
                        "protected",
                        "public",
                        "reflexpr",
                        "register",
                        "reinterpret_cast",
                        "requires",
                        "return",
                        "short",
                        "signed",
                        "sizeof",
                        "static",
                        "static_assert",
                        "static_cast",
                        "struct",
                        "switch",
                        "synchronized",
                        "template",
                        "this",
                        "thread_local",
                        "throw",
                        "true",
                        "try",
                        "typedef",
                        "typeid",
                        "typename",
                        "union",
                        "unsigned",
                        "using",
                        "virtual",
                        "void",
                        "volatile",
                        "wchar_t",
                        "while",
                        "xor",
                        "xor_eq"
                    ],
                    secondary_keywords: str_vec![
                        "char",
                        "int",
                        "long",
                        "unsigned",
                        "float",
                        "double",
                        "size_t",
                        "signed",
                        "short",
                        "wchar_t",
                        "__int128_t",
                        "bool"
                    ],
                },
            };
        } else if file_name.ends_with(".rs") {
            return Self {
                name: String::from("Rust"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "as", "break", "const", "continue", "crate", "else", "enum", "extern",
                        "false", "fn", "for", "if", "impl", "in", "let", "loop", "match", "mut",
                        "pub", "ref", "return", "self", "Self", "static", "struct", "super",
                        "trait", "true", "type", "unsafe", "use", "where", "while", "dyn", "box",
                        "do", "final", "macro", "typeof", "unsized", "yield", "async", "await",
                        "try"
                    ],
                    secondary_keywords: str_vec![
                        "bool", "char", "i8", "i16", "i32", "i64", "i128", "isize", "u8", "u16",
                        "u32", "u64", "u128", "usize", "f32", "f64", "str"
                    ],
                },
            };
        } else if file_name.ends_with(".js") {
            return Self {
                name: String::from("Javascript"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "async",
                        "await",
                        "break",
                        "case",
                        "catch",
                        "class",
                        "const",
                        "continue",
                        "debugger",
                        "default",
                        "delete",
                        "do",
                        "else",
                        "export",
                        "extends",
                        "finally",
                        "for",
                        "function",
                        "if",
                        "import",
                        "in",
                        "instanceof",
                        "let",
                        "new",
                        "return",
                        "super",
                        "switch",
                        "this",
                        "throw",
                        "try",
                        "typeof",
                        "var",
                        "void",
                        "while",
                        "with",
                        "yield"
                    ],
                    secondary_keywords: str_vec!["get", "set"],
                },
            };
        } else if file_name.ends_with(".rb") {
            return Self {
                name: String::from("Ruby"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "__ENCODING__",
                        "__LINE",
                        "__FILE",
                        "BEGIN",
                        "END",
                        "alias",
                        "and",
                        "begin",
                        "break",
                        "case",
                        "class",
                        "def",
                        "defined?",
                        "do?",
                        "else",
                        "elsif",
                        "end",
                        "ensure",
                        "false",
                        "for",
                        "if",
                        "in",
                        "module",
                        "next",
                        "nil",
                        "not",
                        "or",
                        "redo",
                        "retry",
                        "return",
                        "self",
                        "then",
                        "true",
                        "undef",
                        "unless",
                        "until",
                        "when",
                        "while",
                        "yield"
                    ],
                    secondary_keywords: str_vec![],
                },
            };
        } else if file_name.ends_with(".py") {
            return Self {
                name: String::from("Python"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "and", "as", "assert", "break", "class", "continue", "def", "del", "elif",
                        "else", "except", "False", "finally", "for", "from", "global", "if",
                        "import", "in", "is", "lambda", "None", "nonlocal", "not", "or", "pass",
                        "raise", "return", "True", "try", "while"
                    ],
                    secondary_keywords: str_vec![],
                },
            };
        } else if file_name.ends_with(".java") {
            return Self {
                name: String::from("Java"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "abstract",
                        "continue",
                        "for",
                        "new",
                        "switch",
                        "assert",
                        "default",
                        "package",
                        "synchronized",
                        "boolean",
                        "do",
                        "if",
                        "private",
                        "this",
                        "break",
                        "double",
                        "implements",
                        "protected",
                        "throw",
                        "byte",
                        "else",
                        "import",
                        "public",
                        "throws",
                        "case",
                        "enum",
                        "instanceof",
                        "return",
                        "transient",
                        "catch",
                        "extends",
                        "int",
                        "short",
                        "try",
                        "char",
                        "final",
                        "interface",
                        "static",
                        "void",
                        "class",
                        "finally",
                        "long",
                        "strictfp",
                        "volatile",
                        "float",
                        "native",
                        "super",
                        "while"
                    ],
                    secondary_keywords: str_vec!["true", "false", "null"],
                },
            };
        } else if file_name.ends_with(".go") {
            return Self {
                name: String::from("Golang"),
                hl_opts: HighlightingOptions {
                    numbers: true,
                    strings: true,
                    characters: true,
                    comments: true,
                    multiline_comments: true,
                    primary_keywords: str_vec![
                        "break",
                        "default",
                        "func",
                        "interface",
                        "select",
                        "case",
                        "defer",
                        "go",
                        "map",
                        "struct",
                        "chan",
                        "else",
                        "goto",
                        "package",
                        "switch",
                        "const",
                        "fallthrough",
                        "if",
                        "range",
                        "type",
                        "continue",
                        "for",
                        "import",
                        "return",
                        "var"
                    ],
                    secondary_keywords: str_vec![
                        "+",
                        "&",
                        " +=",
                        "&=",
                        " &&",
                        "==",
                        "!=",
                        "(",
                        ")",
                        "-",
                        "|",
                        " -=",
                        "|=",
                        "||",
                        "<",
                        "<=",
                        "[",
                        "]",
                        "*",
                        "^",
                        "*=",
                        "^=",
                        "<-",
                        ">",
                        ">=",
                        "{",
                        "}",
                        "/",
                        "<<",
                        "/=",
                        "<<=",
                        "++",
                        "=",
                        ":=",
                        ",",
                        ";",
                        "%",
                        ">>",
                        "%=",
                        ">>=",
                        "--",
                        "!",
                        "...",
                        ".",
                        ":",
                        "&^",
                        "&^=",
                        "uint8",
                        "uint16",
                        "uint32",
                        "uint64",
                        "int8",
                        "int16",
                        "int32",
                        "int64",
                        "float32",
                        "float64",
                        "complex64",
                        "complex128",
                        "byte",
                        "rune",
                        "uint",
                        "int",
                        "uintptr"
                    ],
                },
            };
        }
        Self::default()
    }
}

impl HighlightingOptions {
    pub fn numbers(&self) -> bool {
        self.numbers
    }
    pub fn strings(&self) -> bool {
        self.strings
    }
    pub fn characters(&self) -> bool {
        self.characters
    }
    pub fn comments(&self) -> bool {
        self.comments
    }
    pub fn primary_keywords(&self) -> &Vec<String> {
        &self.primary_keywords
    }
    pub fn secondary_keywords(&self) -> &Vec<String> {
        &self.secondary_keywords
    }
    pub fn multiline_comments(&self) -> bool {
        self.multiline_comments
    }
}
