#[non_exhaustive]
struct SlipCode;

/// SLIP codes as defined in [RFC1055](https://datatracker.ietf.org/doc/html/rfc1055)
impl SlipCode {
    pub const END: u8 = 192;
    pub const ESC: u8 = 219;
    pub const ESC_END: u8 = 220;
    pub const ESC_ESC: u8 = 221;
}


/// SLIP-encodes a byte-slice into the given buffer as described in
/// [RFC1055](https://datatracker.ietf.org/doc/html/rfc1055). Returns size of encoded data in bytes.
pub fn encode(encoded_data_buf: &mut[u8], data: &[u8]) -> usize {
    let mut idx = 0;

    for b in data {
        match *b {
            SlipCode::END => {
                // END -> ESC ESC_END
                encoded_data_buf[idx] = SlipCode::ESC;
                encoded_data_buf[idx + 1] = SlipCode::ESC_END;
                idx += 2;
            }
            SlipCode::ESC => {
                // ESC -> ESC ESC_ESC
                encoded_data_buf[idx] = SlipCode::ESC;
                encoded_data_buf[idx + 1] = SlipCode::ESC_ESC;
                idx += 2;
            }
            _ => {
                // no ESC needed, copy to buffer
                encoded_data_buf[idx] = *b;
                idx += 1;
            }
        }
    }

    // signal END of SLIP packet
    encoded_data_buf[idx] = SlipCode::END;
    idx+1
}


/// A SLIP decoder which supports split up packets. This occurs when for example the first half of a
/// SLIP packet is transmitted at the end of a UART DMA read and the other half in the next
/// UART DMA read. Also works over multiple calls of [`Self::decode`]. Decoded
/// packets are available via [`Self::are_decoded_packets_available`] and [`Self::get_packets`].
/// # Example
///
/// ```
/// use rusty_slip::slip::{encode, SlipDecoder};
/// use std::str;
///
/// // encode a slice
/// let data = "hello world!";
/// // make sure tx_buf is large enough
/// let mut tx_buf: [u8; 256] = [0; 256];
/// encode(&mut tx_buf, data.as_bytes());///
///
/// // decode packet
/// let mut decoder = SlipDecoder::new();
/// decoder.decode(&tx_buf);
/// let packets = decoder.get_packets();
/// let first_packet = packets.first().unwrap();
///
/// // decode Vec<u8> into a string
/// let packet_message = str::from_utf8(first_packet).expect("Could not convert decoded SLIP packet into a string!");
/// println!("{}", packet_message);
///
/// assert_eq!(data, packet_message);
/// ```
pub struct SlipDecoder {
    decoded_packets: Vec<Vec<u8>>,
    decode_buf: Vec<u8>,
    last_byte_was_esc: bool
}


impl SlipDecoder {
    /// Instantiates a SlipDecoder.
    pub fn new() -> SlipDecoder {
        SlipDecoder {
            decoded_packets: Vec::new(),
            decode_buf: Vec::new(),
            last_byte_was_esc: false
        }
    }

    /// This method indicates whether new decoded SLIP packets are available.
    pub fn are_decoded_packets_available(&self) -> bool {
        !self.decoded_packets.is_empty()
    }

    fn build_packet(&mut self) {
        let msg = self.decode_buf.clone();

        self.decoded_packets.push(msg);
        self.decode_buf.clear();
    }

    /// This method returns all decoded packets and removes them from the decoders memory.
    pub fn get_packets(&mut self) -> Vec<Vec<u8>> {
        let res = self.decoded_packets.clone();
        self.decoded_packets.clear();
        res
    }

    /// This method tries to decode the given SLIP data into one or more packets. Packets which
    /// are not terminated at the end of the given slice are stored in the internal buffer
    /// of the decoder. The decoder continues decoding these packets on the next call(s) of this method
    /// with the new data supplied. Decoded packets are signaled via [`Self::are_decoded_packets_available`] and
    /// can be fetched via [`Self::get_packets`]
    pub fn decode(&mut self, encoded_data: &[u8]) {
        for b in encoded_data {
            if self.last_byte_was_esc {
                match *b {
                    SlipCode::ESC_END => {
                        self.decode_buf.push(SlipCode::END);
                    }
                    SlipCode::ESC_ESC => {
                        self.decode_buf.push(SlipCode::ESC);
                    }
                    _ => {
                        self.decode_buf.push(*b);
                    }
                }
                // reset ESC flag
                self.last_byte_was_esc = false;
            } else {
                match *b {
                    SlipCode::ESC => {
                        self.last_byte_was_esc = true;
                    },
                    SlipCode::END => {
                        // end of this SLIP packet
                        self.build_packet();
                    }
                    _ => {
                        self.decode_buf.push(*b);
                    }
                }
            }
        }
    }
}



#[cfg(test)]
mod tests {
    use crate::slip::{encode, SlipDecoder};
    use rand::Rng;
    use quickcheck::quickcheck;

    quickcheck! {
        fn sending_split_up_packet_over_2_bufs(data: Vec<u8>) -> bool {

            // splitting a single element would not make sense
            if data.len() < 2 {
                return true;
            }

            // generate random buffer split
            let random_split = rand::thread_rng().gen_range(0..data.len());


            // SLIP encode given data
            let mut tx_buf: [u8; 256] = [0; 256];
            let num_of_bytes_to_send = encode(&mut tx_buf, data.as_slice());

            let mut decoder = SlipDecoder::new();

            // decode twice with both buffer parts
            decoder.decode(&tx_buf[0..random_split]);
            decoder.decode(&tx_buf[random_split..num_of_bytes_to_send]);

            let packets = decoder.get_packets();
            let msg = packets.first().unwrap();

            *msg == data
        }
    }

    quickcheck! {
        fn encode_decode_should_yield_input_data(data: Vec<u8>) -> bool {

            // SLIP encode given data
            let mut tx_buf: [u8; 256] = [0; 256];
            encode(&mut tx_buf, data.as_slice());

            let mut decoder = SlipDecoder::new();
            decoder.decode(&tx_buf);

            let packets = decoder.get_packets();
            let msg = packets.first().unwrap();

            *msg == data
        }
    }


}
