#[cfg(test)]
mod tests {
    use crate::logger::*;

    #[test]
    fn log() {
        let mut logger = Logger::new("main", std::io::stdout());
        logger.options.time_unit = TimeUnit::Milliseconds;
        let timer_name = "Small code block";
        logger.timer_start(timer_name);
        logger.info("Hello !");
        logger.warn("Oups !");
        logger.timer_log_and_reset(timer_name);
        logger.error("Oh no !");
        logger.timer_log_and_stop(timer_name);

        logger.timer_start("Pause test");
        for _i in 0..1000 {
            logger.timer_resume("Pause test");
            std::thread::sleep(std::time::Duration::from_millis(1));
            logger.timer_pause("Pause test");
            std::thread::sleep(std::time::Duration::from_micros(1));
        }

        logger.timer_log_and_stop("Pause test");
    }
}

pub use logger::*;

mod logger {
    use std::{
        collections::HashMap,
        fmt::Display,
        io::stderr,
        time::{SystemTime, Duration},
    };
    use colored::*;
    use chrono::Utc;

    pub struct Logger<T: std::io::Write> {
        name: String,
        pub options: Options<T>,
        timer: Timer,
    }

    pub struct Colors {
        pub name: Color,
        pub info: Color,
        pub warn: Color,
        pub fail: Color,
        pub time: Color,
        pub timer: Color,
    }

    pub struct Options<T: std::io::Write> {
        pub time: bool,
        pub name: bool,
        pub time_unit: TimeUnit,
        pub colors: Option<Colors>,
        output: T,
    }

    pub struct Timer {
        timers: HashMap<String, (SystemTime, SystemTime, bool)>,
    }

    pub enum LogType {
        Info,
        Warning,
        Error,
        Time,
    }

    #[derive(Copy, Clone)]
    pub enum TimeUnit {
        Nanoseconds,
        Microseconds,
        Milliseconds,
        Seconds
    }

    impl Colors {
        pub fn new() -> Colors {
            Colors {
                name: Color::Blue,
                info: Color::Green,
                warn: Color::Yellow,
                fail: Color::Red,
                time: Color::Magenta,
                timer: Color::Cyan,
            }
        }
    }

    impl Default for Colors {
        fn default() -> Self {
            Colors::new()
        }
    }

    impl<T: std::io::Write> Options<T> {
        pub fn new(stream: T) -> Options<T> {
            Options {
                time: false,
                name: true,
                time_unit: TimeUnit::Milliseconds,
                colors: Some(Colors::new()),
                output: stream,
            }
        }
    }

    impl Default for Timer {
        fn default() -> Self {
            Timer::new()
        }
    }

    impl Timer {
        pub fn new() -> Timer {
            Timer {
                timers: HashMap::new(),
            }
        }

        fn start(&mut self, msg: &str) {
            self.timers.insert(msg.to_string(), (SystemTime::now(), SystemTime::now(), false));
        }

        fn get(&self, msg: &str) -> Option<Duration> {
            if let Some(time) = self.timers.get(msg) {
                Some(time.0.elapsed().unwrap_or_else(|_| Duration::from_micros(0)))
            } else {
                None
            }
        }

        fn stop(&mut self, msg: &str) -> Option<Duration> {
            if let Some(time) = self.timers.remove(msg) {
                Some(time.0.elapsed().unwrap_or_else(|_| Duration::from_micros(0)))
            } else {
                None
            }
        }

        fn reset(&mut self, msg: &str) -> Option<Duration> {
            if let Some(time) =
                self.timers.insert(msg.to_string(), (SystemTime::now(), SystemTime::now(), false))
            {
                Some(time.0.elapsed().unwrap_or_else(|_| Duration::from_micros(0)))
            } else {
                None
            }
        }

        fn pause(&mut self, msg: &str) -> Option<Duration> {
            if self.timers.contains_key(msg) {
                let timer = self.timers.entry(msg.to_string()).or_insert((
                    SystemTime::now(),
                    SystemTime::now(),
                    false,
                ));
                if timer.2 {
                    return None;
                }
                timer.1 = SystemTime::now();
                timer.2 = true;
                Some(timer.0.elapsed().unwrap_or_else(|_| Duration::from_micros(0)))
            } else {
                None
            }
        }

        fn resume(&mut self, msg: &str) {
            if self.timers.contains_key(msg) {
                let timer = self.timers.entry(msg.to_string()).or_insert((
                    SystemTime::now(),
                    SystemTime::now(),
                    false,
                ));
                if timer.2 {
                    timer.2 = false;
                    timer.0 = timer
                        .0
                        .checked_add(timer.1.elapsed().unwrap_or_else(|_| Duration::from_micros(0)))
                        .unwrap_or_else(SystemTime::now);
                }
            }
        }
    }

    impl<T: std::io::Write> Logger<T> {
        /// Creates a new Logger with default values and the given name
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::Logger;
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        /// logger.info("This is a new logger named 'name' !");
        /// ```
        pub fn new(s: &str, output: T) -> Self {
            Logger {
                name: s.to_string(),
                options: Options::new(output),
                timer: Timer::new(),
            }
        }

        /// Creates a new Logger with default values, custom options and the given name
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, Options};
        ///
        /// let options = Options::new(std::io::stdout());
        /// let mut logger = Logger::with_options("name", options);
        /// logger.info("This is a new logger named 'name' which has 'options' as its options !");
        /// ```
        pub fn with_options(s: &str, options: Options<T>) -> Logger<T> {
            Logger {
                name: s.to_string(),
                options,
                timer: Timer::new(),
            }
        }

        /// Formats the message with the current logger options
        /// and then prints it. It is a private function and is
        /// only used by the other functions (info, warn and
        /// error) to print the message.
        ///
        /// It goes through the options of the logger it is
        /// called on and formats the message accordingly.
        ///
        /// It is also responsible to print everything in
        /// the correct colors
        fn log<D: Display>(&mut self, msg: D, msg_type: LogType) {
            let mut output;

            if let Some(colors) = &self.options.colors {
                match msg_type {
                    LogType::Error => {
                        output = format!("[{}]", "FAIL".color(colors.fail));
                    }
                    LogType::Warning => {
                        output = format!("[{}]", "WARN".color(colors.warn));
                    }
                    LogType::Info => {
                        output = format!("[{}]", "INFO".color(colors.info));
                    }
                    LogType::Time => {
                        output = format!("[{}]", "TIME".color(colors.timer));
                    }
                };

                if self.options.time {
                    output = format!(
                        "{}[{}]",
                        output,
                        Utc::now().format("%T").to_string().color(colors.time)
                    );
                }

                if self.options.name {
                    output = format!("[{}]{}", self.name.color(colors.name), output);
                }
            } else {
                match msg_type {
                    LogType::Error => {
                        output = format!("[{}]", "FAIL");
                    }
                    LogType::Warning => {
                        output = format!("[{}]", "WARN");
                    }
                    LogType::Info => {
                        output = format!("[{}]", "INFO");
                    }
                    LogType::Time => {
                        output = format!("[{}]", "TIME");
                    }
                };

                if self.options.time {
                    output = format!("{}[{}]", output, Utc::now().format("%T").to_string());
                }

                if self.options.name {
                    output = format!("[{}]{}", self.name, output);
                }
            }

            match writeln!(self.options.output, "{}: {}", output, msg) {
                std::result::Result::Err(err) => Logger::static_log(err, LogType::Error, stderr()),
                std::result::Result::Ok(_) => {}
            }
        }

        /// Static implementation for the log function
        ///
        /// Mainly used to print internal error messages
        /// but can also be used by an end user
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// Logger::static_log("Printing on the fly", LogType::Info, std::io::stdout());
        /// ```
        pub fn static_log<D: Display>(msg: D, msg_type: LogType, output: T) {
            let mut logger = Logger::new("STATIC", output);
            logger.log(msg, msg_type);
        }

        /// Logs the given message to the output
        /// as an information
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        /// logger.info("Here is some information I want to log");
        /// ```
        pub fn info<D: Display>(&mut self, msg: D) {
            self.log(msg, LogType::Info);
        }

        /// Logs the given message to the output
        /// as a warning
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        /// logger.warn("Here is a warning");
        /// ```
        pub fn warn<D: Display>(&mut self, msg: D) {
            self.log(msg, LogType::Warning);
        }

        /// Logs the given message to the output
        /// as an error
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        /// logger.info("Some error happened");
        /// ```
        pub fn error<D: Display>(&mut self, msg: D) {
            self.log(msg, LogType::Error);
        }

        /// Starts a new timer with the given name
        /// and the actual time
        pub fn timer_start(&mut self, msg: &str) {
            self.timer.start(msg);
        }

        /// Gets the duration from when the timer
        /// with the given name was started
        pub fn timer_get(&self, msg: &str) -> Option<Duration> {
            self.timer.get(msg)
        }

        /// Stops the timer with the given name
        /// and returns it as a duration
        pub fn timer_stop(&mut self, msg: &str) -> Option<Duration> {
            self.timer.stop(msg)
        }

        /// Resets the timer with the given name
        /// and returns the duration of the timer
        /// before being reset as a duration
        pub fn timer_reset(&mut self, msg: &str) -> Option<Duration> {
            self.timer.reset(msg)
        }

        /// Pauses the timer with the given name
        /// and returns the duration of the timer
        /// before pausing it as a duration
        pub fn timer_pause(&mut self, msg: &str) -> Option<Duration> {
            self.timer.pause(msg)
        }

        /// Resumes the timer with the given name
        pub fn timer_resume(&mut self, msg: &str) {
            self.timer.resume(msg)
        }

        /// Logs the time elapsed between the
        /// start/reset of the timer and now
        /// **without** reseting it
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        ///
        /// logger.timer_start("new_timer");
        ///
        /// std::thread::sleep(std::time::Duration::from_millis(1));
        ///
        /// logger.timer_log("new_timer");
        /// ```
        pub fn timer_log(&mut self, msg: &str) {
            if let Some(time) = self.timer.get(msg) {
                match self.options.time_unit {
                    TimeUnit::Nanoseconds => {
                        self.log(format!("{} - {}ns", msg, time.as_nanos()), LogType::Time);
                    }
                    TimeUnit::Microseconds => {
                        self.log(format!("{} - {}μs", msg, time.as_micros()), LogType::Time);
                    }
                    TimeUnit::Milliseconds => {
                        self.log(format!("{} - {}ms", msg, time.as_millis()), LogType::Time);
                    }
                    TimeUnit::Seconds => {
                        self.log(format!("{} - {}ms", msg, time.as_secs()), LogType::Time);
                    }
                }
            } else {
                Logger::static_log(msg, LogType::Error, stderr());
            }
        }

        /// Logs the time elapsed between the
        /// start/reset of the timer and now
        /// and then stops it
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        ///
        /// logger.timer_start("new_timer");
        ///
        /// std::thread::sleep(std::time::Duration::from_millis(1));
        ///
        /// logger.timer_log_and_stop("new_timer");
        /// ```
        pub fn timer_log_and_stop(&mut self, msg: &str) {
            self.timer_log(msg);
            self.timer_stop(msg);
        }

        /// Logs the time elapsed between the
        /// start/reset of the timer and now
        /// and then resets it
        ///
        /// # Example
        ///
        /// ```
        /// use rusty_logger::logger::{Logger, LogType};
        ///
        /// let mut logger = Logger::new("name", std::io::stdout());
        ///
        /// logger.timer_start("new_timer");
        ///
        /// std::thread::sleep(std::time::Duration::from_millis(1));
        ///
        /// logger.timer_log_and_reset("new_timer");
        ///
        /// std::thread::sleep(std::time::Duration::from_millis(1));
        ///
        /// logger.timer_log_and_stop("new_timer");
        /// ```
        pub fn timer_log_and_reset(&mut self, msg: &str) {
            self.timer_log(msg);
            self.timer_reset(msg);
        }
    }
}
