#![allow(non_upper_case_globals)]
#![allow(non_camel_case_types)]
#![allow(non_snake_case)]

use crate::const_wrappers::*;
use crate::raw_bindings::d3d12::*;
use crate::struct_wrappers::*;
use crate::utils::*;

use bitflags::bitflags;

// ToDo: variant naming style is not uniform by now

bitflags! {
    pub struct FenceFlags: i32 {
        const None = D3D12_FENCE_FLAGS_D3D12_FENCE_FLAG_NONE;
        const Shared =
        D3D12_FENCE_FLAGS_D3D12_FENCE_FLAG_SHARED;
        const CrossAdapter =
        D3D12_FENCE_FLAGS_D3D12_FENCE_FLAG_SHARED_CROSS_ADAPTER;
        const NonMonitored =
        D3D12_FENCE_FLAGS_D3D12_FENCE_FLAG_NON_MONITORED;
    }
}

// ToDo: implement Debug for all types
#[derive(Debug, Copy, Clone)]
#[repr(i32)]
pub enum DescriptorHeapType {
    CbvSrvUav =
        D3D12_DESCRIPTOR_HEAP_TYPE_D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV,
    Sampler = D3D12_DESCRIPTOR_HEAP_TYPE_D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER,
    Rtv = D3D12_DESCRIPTOR_HEAP_TYPE_D3D12_DESCRIPTOR_HEAP_TYPE_RTV,
    Dsv = D3D12_DESCRIPTOR_HEAP_TYPE_D3D12_DESCRIPTOR_HEAP_TYPE_DSV,
    NumTypes = D3D12_DESCRIPTOR_HEAP_TYPE_D3D12_DESCRIPTOR_HEAP_TYPE_NUM_TYPES,
}

bitflags! {
    pub struct ResourceStates: i32 {
        const Common = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COMMON;
        const VertexAndConstantBuffer = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VERTEX_AND_CONSTANT_BUFFER;
        const IndexBuffer = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_INDEX_BUFFER;
        const RenderTarget = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RENDER_TARGET;
        const UnorderedAccess = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_UNORDERED_ACCESS;
        const DepthWrite = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_DEPTH_WRITE;
        const DepthRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_DEPTH_READ;
        const NonPixelShaderResource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE;
        const PixelShaderResource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PIXEL_SHADER_RESOURCE;
        const StreamOut = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_STREAM_OUT;
        const IndirectArgument = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_INDIRECT_ARGUMENT;
        const CopyDest = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COPY_DEST;
        const CopySource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COPY_SOURCE;
        const ResolveDest = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RESOLVE_DEST;
        const ResolveSource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RESOLVE_SOURCE;
        const RaytracingAccelerationStructure = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RAYTRACING_ACCELERATION_STRUCTURE;
        const ShadingRateSource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_SHADING_RATE_SOURCE;
        const GenericRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_GENERIC_READ;
        const AllShaderResource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_ALL_SHADER_RESOURCE;
        const Present = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PRESENT;
        const Predication = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PREDICATION;
        const VideoDecodeRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_DECODE_READ;
        const VideoDecodeWrite = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_DECODE_WRITE;
        const VideoProcessRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_PROCESS_READ;
        const VideoProcessWrite = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_PROCESS_WRITE;
        const VideoEncodeRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_ENCODE_READ;
        const VideoEncodeWrite = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_ENCODE_WRITE;
    }
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum ResourceStatess {
    Common = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COMMON,
    VertexAndConstantBuffer =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VERTEX_AND_CONSTANT_BUFFER,
    IndexBuffer = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_INDEX_BUFFER,
    RenderTarget = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RENDER_TARGET,
    UnorderedAccess =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_UNORDERED_ACCESS,
    DepthWrite = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_DEPTH_WRITE,
    DepthRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_DEPTH_READ,
    NonPixelShaderResource =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE,
    PixelShaderResource =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PIXEL_SHADER_RESOURCE,
    StreamOut = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_STREAM_OUT,
    IndirectArgumentOrPredication =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_INDIRECT_ARGUMENT,
    CopyDest = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COPY_DEST,
    CopySource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_COPY_SOURCE,
    ResolveDest = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RESOLVE_DEST,
    ResolveSource = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RESOLVE_SOURCE,
    RaytracingAccelerationStructure = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_RAYTRACING_ACCELERATION_STRUCTURE,
    ShadingRateSource =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_SHADING_RATE_SOURCE,
    GenericRead = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_GENERIC_READ,
    // Present = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PRESENT,
    // Predication = D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_PREDICATION,
    VideoDecodeRead =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_DECODE_READ,
    VideoDecodeWrite =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_DECODE_WRITE,
    VideoProcessRead =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_PROCESS_READ,
    VideoProcessWrite =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_PROCESS_WRITE,
    VideoEncodeRead =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_ENCODE_READ,
    VideoEncodeWrite =
        D3D12_RESOURCE_STATES_D3D12_RESOURCE_STATE_VIDEO_ENCODE_WRITE,
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum ResourceDimension {
    Unknown = D3D12_RESOURCE_DIMENSION_D3D12_RESOURCE_DIMENSION_UNKNOWN,
    Buffer = D3D12_RESOURCE_DIMENSION_D3D12_RESOURCE_DIMENSION_BUFFER,
    Texture1D = D3D12_RESOURCE_DIMENSION_D3D12_RESOURCE_DIMENSION_TEXTURE1D,
    Texture2D = D3D12_RESOURCE_DIMENSION_D3D12_RESOURCE_DIMENSION_TEXTURE2D,
    Texture3D = D3D12_RESOURCE_DIMENSION_D3D12_RESOURCE_DIMENSION_TEXTURE3D,
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]
pub enum Format {
    Unknown = DXGI_FORMAT_DXGI_FORMAT_UNKNOWN,
    R32G32B32A32_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32G32B32A32_TYPELESS,
    R32G32B32A32_Float = DXGI_FORMAT_DXGI_FORMAT_R32G32B32A32_FLOAT,
    R32G32B32A32_UInt = DXGI_FORMAT_DXGI_FORMAT_R32G32B32A32_UINT,
    R32G32B32A32_SInt = DXGI_FORMAT_DXGI_FORMAT_R32G32B32A32_SINT,
    R32G32B32_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32G32B32_TYPELESS,
    R32G32B32_Float = DXGI_FORMAT_DXGI_FORMAT_R32G32B32_FLOAT,
    R32G32B32_UInt = DXGI_FORMAT_DXGI_FORMAT_R32G32B32_UINT,
    R32G32B32_SInt = DXGI_FORMAT_DXGI_FORMAT_R32G32B32_SINT,
    R16G16B16A16_Typeless = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_TYPELESS,
    R16G16B16A16_Float = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_FLOAT,
    R16G16B16A16_UNorm = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_UNORM,
    R16G16B16A16_UInt = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_UINT,
    R16G16B16A16_SNorm = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_SNORM,
    R16G16B16A16_SInt = DXGI_FORMAT_DXGI_FORMAT_R16G16B16A16_SINT,
    R32G32_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32G32_TYPELESS,
    R32G32_Float = DXGI_FORMAT_DXGI_FORMAT_R32G32_FLOAT,
    R32G32_UInt = DXGI_FORMAT_DXGI_FORMAT_R32G32_UINT,
    R32G32_SInt = DXGI_FORMAT_DXGI_FORMAT_R32G32_SINT,
    R32G8X24_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32G8X24_TYPELESS,
    D32_Float_S8X24_UInt = DXGI_FORMAT_DXGI_FORMAT_D32_FLOAT_S8X24_UINT,
    R32_Float_X8X24_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS,
    X32_Typeless_G8X24_UInt = DXGI_FORMAT_DXGI_FORMAT_X32_TYPELESS_G8X24_UINT,
    R10G10B10A2_Typeless = DXGI_FORMAT_DXGI_FORMAT_R10G10B10A2_TYPELESS,
    R10G10B10A2_UNorm = DXGI_FORMAT_DXGI_FORMAT_R10G10B10A2_UNORM,
    R10G10B10A2_UInt = DXGI_FORMAT_DXGI_FORMAT_R10G10B10A2_UINT,
    R11G11B10_Float = DXGI_FORMAT_DXGI_FORMAT_R11G11B10_FLOAT,
    R8G8B8A8_Typeless = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_TYPELESS,
    R8G8B8A8_UNorm = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_UNORM,
    R8G8B8A8_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_UNORM_SRGB,
    R8G8B8A8_UInt = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_UINT,
    R8G8B8A8_SNorm = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_SNORM,
    R8G8B8A8_SInt = DXGI_FORMAT_DXGI_FORMAT_R8G8B8A8_SINT,
    R16G16_Typeless = DXGI_FORMAT_DXGI_FORMAT_R16G16_TYPELESS,
    R16G16_Float = DXGI_FORMAT_DXGI_FORMAT_R16G16_FLOAT,
    R16G16_UNorm = DXGI_FORMAT_DXGI_FORMAT_R16G16_UNORM,
    R16G16_UInt = DXGI_FORMAT_DXGI_FORMAT_R16G16_UINT,
    R16G16_SNorm = DXGI_FORMAT_DXGI_FORMAT_R16G16_SNORM,
    R16G16_SInt = DXGI_FORMAT_DXGI_FORMAT_R16G16_SINT,
    R32_Typeless = DXGI_FORMAT_DXGI_FORMAT_R32_TYPELESS,
    D32_Float = DXGI_FORMAT_DXGI_FORMAT_D32_FLOAT,
    R32_Float = DXGI_FORMAT_DXGI_FORMAT_R32_FLOAT,
    R32_UInt = DXGI_FORMAT_DXGI_FORMAT_R32_UINT,
    R32_SInt = DXGI_FORMAT_DXGI_FORMAT_R32_SINT,
    R24G8_Typeless = DXGI_FORMAT_DXGI_FORMAT_R24G8_TYPELESS,
    D24_UNorm_S8_UInt = DXGI_FORMAT_DXGI_FORMAT_D24_UNORM_S8_UINT,
    R24_UNorm_X8_Typeless = DXGI_FORMAT_DXGI_FORMAT_R24_UNORM_X8_TYPELESS,
    X24_Typeless_G8_UInt = DXGI_FORMAT_DXGI_FORMAT_X24_TYPELESS_G8_UINT,
    R8G8_Typeless = DXGI_FORMAT_DXGI_FORMAT_R8G8_TYPELESS,
    R8G8_UNorm = DXGI_FORMAT_DXGI_FORMAT_R8G8_UNORM,
    R8G8_UInt = DXGI_FORMAT_DXGI_FORMAT_R8G8_UINT,
    R8G8_SNorm = DXGI_FORMAT_DXGI_FORMAT_R8G8_SNORM,
    R8G8_SInt = DXGI_FORMAT_DXGI_FORMAT_R8G8_SINT,
    R16_Typeless = DXGI_FORMAT_DXGI_FORMAT_R16_TYPELESS,
    R16_Float = DXGI_FORMAT_DXGI_FORMAT_R16_FLOAT,
    D16_UNorm = DXGI_FORMAT_DXGI_FORMAT_D16_UNORM,
    R16_UNorm = DXGI_FORMAT_DXGI_FORMAT_R16_UNORM,
    R16_UInt = DXGI_FORMAT_DXGI_FORMAT_R16_UINT,
    R16_SNorm = DXGI_FORMAT_DXGI_FORMAT_R16_SNORM,
    R16_SInt = DXGI_FORMAT_DXGI_FORMAT_R16_SINT,
    R8_Typeless = DXGI_FORMAT_DXGI_FORMAT_R8_TYPELESS,
    R8_UNorm = DXGI_FORMAT_DXGI_FORMAT_R8_UNORM,
    R8_UInt = DXGI_FORMAT_DXGI_FORMAT_R8_UINT,
    R8_SNorm = DXGI_FORMAT_DXGI_FORMAT_R8_SNORM,
    R8_SInt = DXGI_FORMAT_DXGI_FORMAT_R8_SINT,
    A8_UNorm = DXGI_FORMAT_DXGI_FORMAT_A8_UNORM,
    R1_UNorm = DXGI_FORMAT_DXGI_FORMAT_R1_UNORM,
    R9G9B9E5_SharedExp = DXGI_FORMAT_DXGI_FORMAT_R9G9B9E5_SHAREDEXP,
    G8R8_G8B8_UNorm = DXGI_FORMAT_DXGI_FORMAT_G8R8_G8B8_UNORM,
    BC1_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC1_TYPELESS,
    BC1_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC1_UNORM,
    BC1_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_BC1_UNORM_SRGB,
    BC2_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC2_TYPELESS,
    BC2_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC2_UNORM,
    BC2_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_BC2_UNORM_SRGB,
    BC3_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC3_TYPELESS,
    BC3_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC3_UNORM,
    BC3_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_BC3_UNORM_SRGB,
    BC4_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC4_TYPELESS,
    BC4_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC4_UNORM,
    BC4_SNorm = DXGI_FORMAT_DXGI_FORMAT_BC4_SNORM,
    BC5_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC5_TYPELESS,
    BC5_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC5_UNORM,
    BC5_SNorm = DXGI_FORMAT_DXGI_FORMAT_BC5_SNORM,
    B5G6R5_UNorm = DXGI_FORMAT_DXGI_FORMAT_B5G6R5_UNORM,
    B5G5R5A1_UNorm = DXGI_FORMAT_DXGI_FORMAT_B5G5R5A1_UNORM,
    B8G8R8A8_UNorm = DXGI_FORMAT_DXGI_FORMAT_B8G8R8A8_UNORM,
    B8G8R8X8_UNorm = DXGI_FORMAT_DXGI_FORMAT_B8G8R8X8_UNORM,
    R10G10B10_XRBiasA2UNorm =
        DXGI_FORMAT_DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM,
    B8G8R8A8_Typeless = DXGI_FORMAT_DXGI_FORMAT_B8G8R8A8_TYPELESS,
    B8G8R8A8_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_B8G8R8A8_UNORM_SRGB,
    B8G8R8X8_Typeless = DXGI_FORMAT_DXGI_FORMAT_B8G8R8X8_TYPELESS,
    B8G8R8X8_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_B8G8R8X8_UNORM_SRGB,
    BC6H_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC6H_TYPELESS,
    BC6H_UF16 = DXGI_FORMAT_DXGI_FORMAT_BC6H_UF16,
    BC6H_SF16 = DXGI_FORMAT_DXGI_FORMAT_BC6H_SF16,
    BC7_Typeless = DXGI_FORMAT_DXGI_FORMAT_BC7_TYPELESS,
    BC7_UNorm = DXGI_FORMAT_DXGI_FORMAT_BC7_UNORM,
    BC7_UNormSRGB = DXGI_FORMAT_DXGI_FORMAT_BC7_UNORM_SRGB,
    AYUV = DXGI_FORMAT_DXGI_FORMAT_AYUV,
    Y410 = DXGI_FORMAT_DXGI_FORMAT_Y410,
    Y416 = DXGI_FORMAT_DXGI_FORMAT_Y416,
    NV12 = DXGI_FORMAT_DXGI_FORMAT_NV12,
    P010 = DXGI_FORMAT_DXGI_FORMAT_P010,
    P016 = DXGI_FORMAT_DXGI_FORMAT_P016,
    F420_Opaque = DXGI_FORMAT_DXGI_FORMAT_420_OPAQUE,
    YUY2 = DXGI_FORMAT_DXGI_FORMAT_YUY2,
    Y210 = DXGI_FORMAT_DXGI_FORMAT_Y210,
    Y216 = DXGI_FORMAT_DXGI_FORMAT_Y216,
    NV11 = DXGI_FORMAT_DXGI_FORMAT_NV11,
    AI44 = DXGI_FORMAT_DXGI_FORMAT_AI44,
    IA44 = DXGI_FORMAT_DXGI_FORMAT_IA44,
    P8 = DXGI_FORMAT_DXGI_FORMAT_P8,
    A8P8 = DXGI_FORMAT_DXGI_FORMAT_A8P8,
    B4G4R4A4_UNorm = DXGI_FORMAT_DXGI_FORMAT_B4G4R4A4_UNORM,
    P208 = DXGI_FORMAT_DXGI_FORMAT_P208,
    V208 = DXGI_FORMAT_DXGI_FORMAT_V208,
    V408 = DXGI_FORMAT_DXGI_FORMAT_V408,
    ForceUInt = DXGI_FORMAT_DXGI_FORMAT_FORCE_UINT,
}

impl Format {
    pub fn get_size(self) -> Bytes {
        match self {
            Self::R16_UInt => Bytes(2),
            Self::R32_UInt => Bytes(4),
            _ => unimplemented!(),
        }
    }
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum TextureLayout {
    Unknown = D3D12_TEXTURE_LAYOUT_D3D12_TEXTURE_LAYOUT_UNKNOWN,
    RowMajor = D3D12_TEXTURE_LAYOUT_D3D12_TEXTURE_LAYOUT_ROW_MAJOR,
    L64KbUndefinedSwizzle =
        D3D12_TEXTURE_LAYOUT_D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE,
    L64KbStandardSwizzle =
        D3D12_TEXTURE_LAYOUT_D3D12_TEXTURE_LAYOUT_64KB_STANDARD_SWIZZLE,
}

bitflags! {
    pub struct ResourceFlags: i32 {
        const None = D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_NONE;
        const AllowRenderTarget =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_ALLOW_RENDER_TARGET;
        const AllowDepthStencil =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_ALLOW_DEPTH_STENCIL;
        const AllowUnorderedAccess =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;
        const DenyShaderResource =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_DENY_SHADER_RESOURCE;
        const AllowCrossAdapter =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_ALLOW_CROSS_ADAPTER;
        const AllowSimultaneousAccess =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_ALLOW_SIMULTANEOUS_ACCESS;
        const VideoDecodeReferenceOnly =
            D3D12_RESOURCE_FLAGS_D3D12_RESOURCE_FLAG_VIDEO_DECODE_REFERENCE_ONLY;
    }
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum HeapType {
    Default = D3D12_HEAP_TYPE_D3D12_HEAP_TYPE_DEFAULT,
    Upload = D3D12_HEAP_TYPE_D3D12_HEAP_TYPE_UPLOAD,
    Readback = D3D12_HEAP_TYPE_D3D12_HEAP_TYPE_READBACK,
    Custom = D3D12_HEAP_TYPE_D3D12_HEAP_TYPE_CUSTOM,
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum CPUPageProperty {
    Unknown = D3D12_CPU_PAGE_PROPERTY_D3D12_CPU_PAGE_PROPERTY_UNKNOWN,
    NotAvailable =
        D3D12_CPU_PAGE_PROPERTY_D3D12_CPU_PAGE_PROPERTY_NOT_AVAILABLE,
    WriteCombine =
        D3D12_CPU_PAGE_PROPERTY_D3D12_CPU_PAGE_PROPERTY_WRITE_COMBINE,
    WriteBack = D3D12_CPU_PAGE_PROPERTY_D3D12_CPU_PAGE_PROPERTY_WRITE_BACK,
}

#[derive(Debug, Clone, Copy)]
#[repr(i32)]

pub enum MemoryPool {
    Unknown = D3D12_MEMORY_POOL_D3D12_MEMORY_POOL_UNKNOWN,
    L0 = D3D12_MEMORY_POOL_D3D12_MEMORY_POOL_L0,
    L1 = D3D12_MEMORY_POOL_D3D12_MEMORY_POOL_L1,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum InputClassification {
    PerVertex =
        D3D12_INPUT_CLASSIFICATION_D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA,
    PerInstance =
        D3D12_INPUT_CLASSIFICATION_D3D12_INPUT_CLASSIFICATION_PER_INSTANCE_DATA,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum PrimitiveTopology {
    Undefined = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_UNDEFINED,
    PointList = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_POINTLIST,
    LineList = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_LINELIST,
    LineStrip = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_LINESTRIP,
    TriangleList = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_TRIANGLELIST,
    TriangleStrip = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP,
    LineListAdj = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_LINELIST_ADJ,
    LineStripAdj = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_LINESTRIP_ADJ,
    TriangleListAdj =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_TRIANGLELIST_ADJ,
    TriangleStripAdj =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP_ADJ,
    t_1_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_1_CONTROL_POINT_PATCHLIST,
    t_2_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_2_CONTROL_POINT_PATCHLIST,
    t_3_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_3_CONTROL_POINT_PATCHLIST,
    t_4_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_4_CONTROL_POINT_PATCHLIST,
    t_5_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_5_CONTROL_POINT_PATCHLIST,
    t_6_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_6_CONTROL_POINT_PATCHLIST,
    t_7_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_7_CONTROL_POINT_PATCHLIST,
    t_8_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_8_CONTROL_POINT_PATCHLIST,
    t_9_ControlPointPatchlist =
        D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_9_CONTROL_POINT_PATCHLIST,
    t_10_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_10_CONTROL_POINT_PATCHLIST,
    t_11_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_11_CONTROL_POINT_PATCHLIST,
    t_12_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_12_CONTROL_POINT_PATCHLIST,
    t_13_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_13_CONTROL_POINT_PATCHLIST,
    t_14_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_14_CONTROL_POINT_PATCHLIST,
    t_15_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_15_CONTROL_POINT_PATCHLIST,
    t_16_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_16_CONTROL_POINT_PATCHLIST,
    t_17_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_17_CONTROL_POINT_PATCHLIST,
    t_18_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_18_CONTROL_POINT_PATCHLIST,
    t_19_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_19_CONTROL_POINT_PATCHLIST,
    t_20_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_20_CONTROL_POINT_PATCHLIST,
    t_21_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_21_CONTROL_POINT_PATCHLIST,
    t_22_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_22_CONTROL_POINT_PATCHLIST,
    t_23_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_23_CONTROL_POINT_PATCHLIST,
    t_24_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_24_CONTROL_POINT_PATCHLIST,
    t_25_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_25_CONTROL_POINT_PATCHLIST,
    t_26_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_26_CONTROL_POINT_PATCHLIST,
    t_27_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_27_CONTROL_POINT_PATCHLIST,
    t_28_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_28_CONTROL_POINT_PATCHLIST,
    t_29_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_29_CONTROL_POINT_PATCHLIST,
    t_30_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_30_CONTROL_POINT_PATCHLIST,
    t_31_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_31_CONTROL_POINT_PATCHLIST,
    t_32_ControlPointPatchlist = D3D_PRIMITIVE_TOPOLOGY_D3D_PRIMITIVE_TOPOLOGY_32_CONTROL_POINT_PATCHLIST,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum IndexBufferStripCut {
    Disabled = D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_DISABLED,
    c16Bits = D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_0xFFFF,
    c32Bits = D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_0xFFFFFFFF,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum PrimitiveTopologyType {
    Undefined =
        D3D12_PRIMITIVE_TOPOLOGY_TYPE_D3D12_PRIMITIVE_TOPOLOGY_TYPE_UNDEFINED,
    Point = D3D12_PRIMITIVE_TOPOLOGY_TYPE_D3D12_PRIMITIVE_TOPOLOGY_TYPE_POINT,
    Line = D3D12_PRIMITIVE_TOPOLOGY_TYPE_D3D12_PRIMITIVE_TOPOLOGY_TYPE_LINE,
    Triangle =
        D3D12_PRIMITIVE_TOPOLOGY_TYPE_D3D12_PRIMITIVE_TOPOLOGY_TYPE_TRIANGLE,
    Patch = D3D12_PRIMITIVE_TOPOLOGY_TYPE_D3D12_PRIMITIVE_TOPOLOGY_TYPE_PATCH,
}

bitflags! {
    pub struct PipelineStateFlags: i32 {
        const None = D3D12_PIPELINE_STATE_FLAGS_D3D12_PIPELINE_STATE_FLAG_NONE;
        const ToolDebug = D3D12_PIPELINE_STATE_FLAGS_D3D12_PIPELINE_STATE_FLAG_TOOL_DEBUG;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum Blend {
    Zero = D3D12_BLEND_D3D12_BLEND_ZERO,
    One = D3D12_BLEND_D3D12_BLEND_ONE,
    Color = D3D12_BLEND_D3D12_BLEND_SRC_COLOR,
    InvSrcColor = D3D12_BLEND_D3D12_BLEND_INV_SRC_COLOR,
    SrcAlpha = D3D12_BLEND_D3D12_BLEND_SRC_ALPHA,
    InvSrcAlpha = D3D12_BLEND_D3D12_BLEND_INV_SRC_ALPHA,
    DestAlpha = D3D12_BLEND_D3D12_BLEND_DEST_ALPHA,
    InvDestAlpha = D3D12_BLEND_D3D12_BLEND_INV_DEST_ALPHA,
    BlendDestColor = D3D12_BLEND_D3D12_BLEND_DEST_COLOR,
    BlendInvDestColor = D3D12_BLEND_D3D12_BLEND_INV_DEST_COLOR,
    SrvAlphaSat = D3D12_BLEND_D3D12_BLEND_SRC_ALPHA_SAT,
    BlendFactor = D3D12_BLEND_D3D12_BLEND_BLEND_FACTOR,
    InvBlendFactor = D3D12_BLEND_D3D12_BLEND_INV_BLEND_FACTOR,
    Src1Color = D3D12_BLEND_D3D12_BLEND_SRC1_COLOR,
    InvSrc1Color = D3D12_BLEND_D3D12_BLEND_INV_SRC1_COLOR,
    Src1Alpha = D3D12_BLEND_D3D12_BLEND_SRC1_ALPHA,
    InvSrc1Alpha = D3D12_BLEND_D3D12_BLEND_INV_SRC1_ALPHA,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum BlendOp {
    Add = D3D12_BLEND_OP_D3D12_BLEND_OP_ADD,
    Subtract = D3D12_BLEND_OP_D3D12_BLEND_OP_SUBTRACT,
    RevSubtract = D3D12_BLEND_OP_D3D12_BLEND_OP_REV_SUBTRACT,
    Min = D3D12_BLEND_OP_D3D12_BLEND_OP_MIN,
    Max = D3D12_BLEND_OP_D3D12_BLEND_OP_MAX,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum LogicOp {
    Clear = D3D12_LOGIC_OP_D3D12_LOGIC_OP_CLEAR,
    Set = D3D12_LOGIC_OP_D3D12_LOGIC_OP_SET,
    Copy = D3D12_LOGIC_OP_D3D12_LOGIC_OP_COPY,
    CopyInverted = D3D12_LOGIC_OP_D3D12_LOGIC_OP_COPY_INVERTED,
    NoOp = D3D12_LOGIC_OP_D3D12_LOGIC_OP_NOOP,
    Invert = D3D12_LOGIC_OP_D3D12_LOGIC_OP_INVERT,
    And = D3D12_LOGIC_OP_D3D12_LOGIC_OP_AND,
    NAnd = D3D12_LOGIC_OP_D3D12_LOGIC_OP_NAND,
    Or = D3D12_LOGIC_OP_D3D12_LOGIC_OP_OR,
    NOr = D3D12_LOGIC_OP_D3D12_LOGIC_OP_NOR,
    XOr = D3D12_LOGIC_OP_D3D12_LOGIC_OP_XOR,
    Equiv = D3D12_LOGIC_OP_D3D12_LOGIC_OP_EQUIV,
    AndReverse = D3D12_LOGIC_OP_D3D12_LOGIC_OP_AND_REVERSE,
    AndInverted = D3D12_LOGIC_OP_D3D12_LOGIC_OP_AND_INVERTED,
    OrReverse = D3D12_LOGIC_OP_D3D12_LOGIC_OP_OR_REVERSE,
    OrInverted = D3D12_LOGIC_OP_D3D12_LOGIC_OP_OR_INVERTED,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum FillMode {
    Wireframe = D3D12_FILL_MODE_D3D12_FILL_MODE_WIREFRAME,
    Solid = D3D12_FILL_MODE_D3D12_FILL_MODE_SOLID,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum CullMode {
    None = D3D12_CULL_MODE_D3D12_CULL_MODE_NONE,
    Front = D3D12_CULL_MODE_D3D12_CULL_MODE_FRONT,
    Back = D3D12_CULL_MODE_D3D12_CULL_MODE_BACK,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum ConservativeRasterizationMode {
    Off = D3D12_CONSERVATIVE_RASTERIZATION_MODE_D3D12_CONSERVATIVE_RASTERIZATION_MODE_OFF,
    On = D3D12_CONSERVATIVE_RASTERIZATION_MODE_D3D12_CONSERVATIVE_RASTERIZATION_MODE_ON,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum DepthWriteMask {
    Zero = D3D12_DEPTH_WRITE_MASK_D3D12_DEPTH_WRITE_MASK_ZERO,
    All = D3D12_DEPTH_WRITE_MASK_D3D12_DEPTH_WRITE_MASK_ALL,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum ComparisonFunc {
    Never = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_NEVER,
    Less = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_LESS,
    Equal = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_EQUAL,
    LessEqual = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_LESS_EQUAL,
    Greater = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_GREATER,
    NotEqual = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_NOT_EQUAL,
    GreaterEqual = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_GREATER_EQUAL,
    Always = D3D12_COMPARISON_FUNC_D3D12_COMPARISON_FUNC_ALWAYS,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum StencilOp {
    Keep = D3D12_STENCIL_OP_D3D12_STENCIL_OP_KEEP,
    Zero = D3D12_STENCIL_OP_D3D12_STENCIL_OP_ZERO,
    Replace = D3D12_STENCIL_OP_D3D12_STENCIL_OP_REPLACE,
    IncrSat = D3D12_STENCIL_OP_D3D12_STENCIL_OP_INCR_SAT,
    DecrSat = D3D12_STENCIL_OP_D3D12_STENCIL_OP_DECR_SAT,
    Invert = D3D12_STENCIL_OP_D3D12_STENCIL_OP_INVERT,
    Incr = D3D12_STENCIL_OP_D3D12_STENCIL_OP_INCR,
    Dec = D3D12_STENCIL_OP_D3D12_STENCIL_OP_DECR,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]
pub enum TextureCopyType {
    SubresourceIndex =
        D3D12_TEXTURE_COPY_TYPE_D3D12_TEXTURE_COPY_TYPE_SUBRESOURCE_INDEX,
    PlacedFootprint =
        D3D12_TEXTURE_COPY_TYPE_D3D12_TEXTURE_COPY_TYPE_PLACED_FOOTPRINT,
}

bitflags! {
    pub struct DescriptorHeapFlags: i32 {
        const None = D3D12_DESCRIPTOR_HEAP_FLAGS_D3D12_DESCRIPTOR_HEAP_FLAG_NONE;
        const ShaderVisible = D3D12_DESCRIPTOR_HEAP_FLAGS_D3D12_DESCRIPTOR_HEAP_FLAG_SHADER_VISIBLE;
    }
}

bitflags! {
    pub struct CreateFactoryFlags: u32 {
        const None = 0; // Is there any constant for it?
        const Debug = DXGI_CREATE_FACTORY_DEBUG;
    }
}

bitflags! {
    pub struct CommandQueueFlags: i32 {
        const None = D3D12_COMMAND_QUEUE_FLAGS_D3D12_COMMAND_QUEUE_FLAG_NONE;
        const DisableGpuTimeout = D3D12_COMMAND_QUEUE_FLAGS_D3D12_COMMAND_QUEUE_FLAG_DISABLE_GPU_TIMEOUT;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum CommandListType {
    Direct = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_DIRECT,
    Bundle = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_BUNDLE,
    Compute = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_COMPUTE,
    Copy = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_COPY,
    VideoDecode = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_VIDEO_DECODE,
    VideoProcess =
        D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_VIDEO_PROCESS,
    VideoEncode = D3D12_COMMAND_LIST_TYPE_D3D12_COMMAND_LIST_TYPE_VIDEO_ENCODE,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum CommandQueuePriority {
    Normal = D3D12_COMMAND_QUEUE_PRIORITY_D3D12_COMMAND_QUEUE_PRIORITY_NORMAL,
    High = D3D12_COMMAND_QUEUE_PRIORITY_D3D12_COMMAND_QUEUE_PRIORITY_HIGH,
    GlobalRealTime = D3D12_COMMAND_QUEUE_PRIORITY_D3D12_COMMAND_QUEUE_PRIORITY_GLOBAL_REALTIME,
}

bitflags! {
    pub struct MakeWindowAssociationFlags: u32 {
        const NoWindowChanges = DXGI_MWA_NO_WINDOW_CHANGES;
        const NoAltEnter = DXGI_MWA_NO_ALT_ENTER;
        const NoPrintScreen = DXGI_MWA_NO_PRINT_SCREEN;
        const Valid = DXGI_MWA_VALID;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum RootSignatureVersion {
    // V1 = D3D_ROOT_SIGNATURE_VERSION_D3D_ROOT_SIGNATURE_VERSION_1,
    V1_0 = D3D_ROOT_SIGNATURE_VERSION_D3D_ROOT_SIGNATURE_VERSION_1,
    V1_1 = D3D_ROOT_SIGNATURE_VERSION_D3D_ROOT_SIGNATURE_VERSION_1_1,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum Feature {
    D3D12Options = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS,
    Architecture = D3D12_FEATURE_D3D12_FEATURE_ARCHITECTURE,
    FeatureLevels = D3D12_FEATURE_D3D12_FEATURE_FEATURE_LEVELS,
    FormatSupport = D3D12_FEATURE_D3D12_FEATURE_FORMAT_SUPPORT,
    MultisampleQualityLevels =
        D3D12_FEATURE_D3D12_FEATURE_MULTISAMPLE_QUALITY_LEVELS,
    FormatInfo = D3D12_FEATURE_D3D12_FEATURE_FORMAT_INFO,
    GPUVirtualAddressSupport =
        D3D12_FEATURE_D3D12_FEATURE_GPU_VIRTUAL_ADDRESS_SUPPORT,
    ShaderModel = D3D12_FEATURE_D3D12_FEATURE_SHADER_MODEL,
    D3D12Options1 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS1,
    ProtectedResourceSessionSupport =
        D3D12_FEATURE_D3D12_FEATURE_PROTECTED_RESOURCE_SESSION_SUPPORT,
    RootSignature = D3D12_FEATURE_D3D12_FEATURE_ROOT_SIGNATURE,
    Architecture1 = D3D12_FEATURE_D3D12_FEATURE_ARCHITECTURE1,
    D3D12Options2 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS2,
    ShaderCache = D3D12_FEATURE_D3D12_FEATURE_SHADER_CACHE,
    CommandQueuePriority = D3D12_FEATURE_D3D12_FEATURE_COMMAND_QUEUE_PRIORITY,
    D3D12Options3 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS3,
    ExistingHeaps = D3D12_FEATURE_D3D12_FEATURE_EXISTING_HEAPS,
    D3D12Options4 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS4,
    Serialization = D3D12_FEATURE_D3D12_FEATURE_SERIALIZATION,
    CrossNode = D3D12_FEATURE_D3D12_FEATURE_CROSS_NODE,
    D3D12Options5 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS5,
    D3D12Options6 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS6,
    QueryMetaCommand = D3D12_FEATURE_D3D12_FEATURE_QUERY_META_COMMAND,
    D3D12Options7 = D3D12_FEATURE_D3D12_FEATURE_D3D12_OPTIONS7,
    ProtectedResourceSessionTypeCount =
        D3D12_FEATURE_D3D12_FEATURE_PROTECTED_RESOURCE_SESSION_TYPE_COUNT,
    ProtectedResourceSessionTypes =
        D3D12_FEATURE_D3D12_FEATURE_PROTECTED_RESOURCE_SESSION_TYPES,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum DescriptorRangeType {
    Srv = D3D12_DESCRIPTOR_RANGE_TYPE_D3D12_DESCRIPTOR_RANGE_TYPE_SRV,
    Uav = D3D12_DESCRIPTOR_RANGE_TYPE_D3D12_DESCRIPTOR_RANGE_TYPE_UAV,
    Cbv = D3D12_DESCRIPTOR_RANGE_TYPE_D3D12_DESCRIPTOR_RANGE_TYPE_CBV,
    Sampler = D3D12_DESCRIPTOR_RANGE_TYPE_D3D12_DESCRIPTOR_RANGE_TYPE_SAMPLER,
}

bitflags! {
    pub struct DescriptorRangeFlags: i32 {
        const DescriptorsVolatile = D3D12_DESCRIPTOR_RANGE_FLAGS_D3D12_DESCRIPTOR_RANGE_FLAG_DESCRIPTORS_VOLATILE;
        const DataVolatile = D3D12_DESCRIPTOR_RANGE_FLAGS_D3D12_DESCRIPTOR_RANGE_FLAG_DATA_VOLATILE;
        const DataStaticWhileSetAtExecute = D3D12_DESCRIPTOR_RANGE_FLAGS_D3D12_DESCRIPTOR_RANGE_FLAG_DATA_STATIC_WHILE_SET_AT_EXECUTE;
        const DataStatic = D3D12_DESCRIPTOR_RANGE_FLAGS_D3D12_DESCRIPTOR_RANGE_FLAG_DATA_STATIC;
        const DescriptorsStaticKeepingBufferBoundsChecks = D3D12_DESCRIPTOR_RANGE_FLAGS_D3D12_DESCRIPTOR_RANGE_FLAG_DESCRIPTORS_STATIC_KEEPING_BUFFER_BOUNDS_CHECKS;
    }
}

#[derive(PartialEq)]
#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum RootParameterType {
    DescriptorTable =
        D3D12_ROOT_PARAMETER_TYPE_D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE,
    T32BitConstants =
        D3D12_ROOT_PARAMETER_TYPE_D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS,
    Cbv = D3D12_ROOT_PARAMETER_TYPE_D3D12_ROOT_PARAMETER_TYPE_CBV,
    Srv = D3D12_ROOT_PARAMETER_TYPE_D3D12_ROOT_PARAMETER_TYPE_SRV,
    Uav = D3D12_ROOT_PARAMETER_TYPE_D3D12_ROOT_PARAMETER_TYPE_UAV,
}

bitflags! {
    pub struct RootDescriptorFlags: i32 {
        const DataVolatile = D3D12_ROOT_DESCRIPTOR_FLAGS_D3D12_ROOT_DESCRIPTOR_FLAG_DATA_VOLATILE;
        const DataStaticWhileSetAtExecute = D3D12_ROOT_DESCRIPTOR_FLAGS_D3D12_ROOT_DESCRIPTOR_FLAG_DATA_STATIC_WHILE_SET_AT_EXECUTE;
        const DataStatic = D3D12_ROOT_DESCRIPTOR_FLAGS_D3D12_ROOT_DESCRIPTOR_FLAG_DATA_STATIC;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum ShaderVisibility {
    All = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_ALL,
    Vertex = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_VERTEX,
    Hull = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_HULL,
    Domain = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_DOMAIN,
    Geometry = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_GEOMETRY,
    Pixel = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_PIXEL,
    Amplification =
        D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_AMPLIFICATION,
    Mesh = D3D12_SHADER_VISIBILITY_D3D12_SHADER_VISIBILITY_MESH,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum Filter {
    MinMagMipPoint = D3D12_FILTER_D3D12_FILTER_MIN_MAG_MIP_POINT,
    MinMagPointMipLinear = D3D12_FILTER_D3D12_FILTER_MIN_MAG_POINT_MIP_LINEAR,
    MinPointMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_MIN_POINT_MAG_LINEAR_MIP_POINT,
    MinPointMagMipLinear = D3D12_FILTER_D3D12_FILTER_MIN_POINT_MAG_MIP_LINEAR,
    MinLinearMagMipPoint = D3D12_FILTER_D3D12_FILTER_MIN_LINEAR_MAG_MIP_POINT,
    MinLinearMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_MIN_LINEAR_MAG_POINT_MIP_LINEAR,
    MinMagLinearMipPoint = D3D12_FILTER_D3D12_FILTER_MIN_MAG_LINEAR_MIP_POINT,
    MinMagMipLinear = D3D12_FILTER_D3D12_FILTER_MIN_MAG_MIP_LINEAR,
    Anisotropic = D3D12_FILTER_D3D12_FILTER_ANISOTROPIC,
    ComparisonMinMagMipPoint =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_MAG_MIP_POINT,
    ComparisonMinMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_MAG_POINT_MIP_LINEAR,
    ComparisonMinPointMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_POINT_MAG_LINEAR_MIP_POINT,
    ComparisonMinPointMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_POINT_MAG_MIP_LINEAR,
    ComparisonMinLinearMagMipPoint =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_LINEAR_MAG_MIP_POINT,
    ComparisonMinLinearMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_LINEAR_MAG_POINT_MIP_LINEAR,
    ComparisonMinMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_MAG_LINEAR_MIP_POINT,
    ComparisonMinMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_COMPARISON_MIN_MAG_MIP_LINEAR,
    ComparisonAnisotropic = D3D12_FILTER_D3D12_FILTER_COMPARISON_ANISOTROPIC,
    MinimumMinMagMipPoint = D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_MAG_MIP_POINT,
    MinimumMinMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_MAG_POINT_MIP_LINEAR,
    MinimumMinPointMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_POINT_MAG_LINEAR_MIP_POINT,
    MinimumMinPointMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_POINT_MAG_MIP_LINEAR,
    MinimumMinLinearMagMipPoint =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_LINEAR_MAG_MIP_POINT,
    MinimumMinLinearMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_LINEAR_MAG_POINT_MIP_LINEAR,
    MinimumMinMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_MAG_LINEAR_MIP_POINT,
    MinimumMinMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_MINIMUM_MIN_MAG_MIP_LINEAR,
    MinimumAnisotropic = D3D12_FILTER_D3D12_FILTER_MINIMUM_ANISOTROPIC,
    MaximumMinMagMipPoint = D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_MAG_MIP_POINT,
    MaximumMinMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_MAG_POINT_MIP_LINEAR,
    MaximumMinPointMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_POINT_MAG_LINEAR_MIP_POINT,
    MaximumMinPointMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_POINT_MAG_MIP_LINEAR,
    MaximumMinLinearMagMipPoint =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_LINEAR_MAG_MIP_POINT,
    MaximumMinLinearMagPointMipLinear =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_LINEAR_MAG_POINT_MIP_LINEAR,
    MaximumMinMagLinearMipPoint =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_MAG_LINEAR_MIP_POINT,
    MaximumMinMagMipLinear =
        D3D12_FILTER_D3D12_FILTER_MAXIMUM_MIN_MAG_MIP_LINEAR,
    MaximumAnisotropic = D3D12_FILTER_D3D12_FILTER_MAXIMUM_ANISOTROPIC,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum TextureAddressMode {
    Wrap = D3D12_TEXTURE_ADDRESS_MODE_D3D12_TEXTURE_ADDRESS_MODE_WRAP,
    Mirror = D3D12_TEXTURE_ADDRESS_MODE_D3D12_TEXTURE_ADDRESS_MODE_MIRROR,
    Clamp = D3D12_TEXTURE_ADDRESS_MODE_D3D12_TEXTURE_ADDRESS_MODE_CLAMP,
    Border = D3D12_TEXTURE_ADDRESS_MODE_D3D12_TEXTURE_ADDRESS_MODE_BORDER,
    MirrorOnce =
        D3D12_TEXTURE_ADDRESS_MODE_D3D12_TEXTURE_ADDRESS_MODE_MIRROR_ONCE,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum StaticBorderColor {
    TransparentBlack =
        D3D12_STATIC_BORDER_COLOR_D3D12_STATIC_BORDER_COLOR_TRANSPARENT_BLACK,
    OpaqueBlack =
        D3D12_STATIC_BORDER_COLOR_D3D12_STATIC_BORDER_COLOR_OPAQUE_BLACK,
    OpaqueWhite =
        D3D12_STATIC_BORDER_COLOR_D3D12_STATIC_BORDER_COLOR_OPAQUE_WHITE,
}

bitflags! {
    pub struct RootSignatureFlags: i32 {
        const None = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_NONE;
        const AllowInputAssemblerInputLayout = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT;
        const DenyVertexShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_VERTEX_SHADER_ROOT_ACCESS;
        const DenyHullShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_HULL_SHADER_ROOT_ACCESS;
        const DenyDomainShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_DOMAIN_SHADER_ROOT_ACCESS;
        const DenyGeometryShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_GEOMETRY_SHADER_ROOT_ACCESS;
        const DenyPixelShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_PIXEL_SHADER_ROOT_ACCESS;
        const AllowStreamOutput = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_ALLOW_STREAM_OUTPUT;
        const LocalRootSignature = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_LOCAL_ROOT_SIGNATURE;
        const DenyAmplificationShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_AMPLIFICATION_SHADER_ROOT_ACCESS;
        const DenyMeshShaderRootAccess = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_DENY_MESH_SHADER_ROOT_ACCESS;
        const CbvSrvUavHeapDirectlyIndexed = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_CBV_SRV_UAV_HEAP_DIRECTLY_INDEXED;
        const SamplerHeapDirectlyIndexed = D3D12_ROOT_SIGNATURE_FLAGS_D3D12_ROOT_SIGNATURE_FLAG_SAMPLER_HEAP_DIRECTLY_INDEXED;
    }
}

bitflags! {
    pub struct HeapFlags: i32 {
        const None = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_NONE;
        const Shared = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_SHARED;
        const DenyBuffers = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_DENY_BUFFERS;
        const AllowDisplay = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_DISPLAY;
        const SharedCrossAdapter = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_SHARED_CROSS_ADAPTER;
        const DenyRtDsTextures = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_DENY_RT_DS_TEXTURES;
        const DenyNonRtDsTextures = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_DENY_NON_RT_DS_TEXTURES;
        const HardwareProtected = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_HARDWARE_PROTECTED;
        const AllowWriteWatch = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_WRITE_WATCH;
        const AllowShaderAtomics = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_SHADER_ATOMICS;
        const CreateNotResident = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_CREATE_NOT_RESIDENT;
        const CreateNotZeroed = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_CREATE_NOT_ZEROED;
        const AllowAllBuffersAndTextures = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_ALL_BUFFERS_AND_TEXTURES;
        const AllowOnlyBuffers = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_ONLY_BUFFERS;
        const AllowOnlyNonRtDsTextures = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_ONLY_NON_RT_DS_TEXTURES;
        const AllowOnlyRtDsTextures = D3D12_HEAP_FLAGS_D3D12_HEAP_FLAG_ALLOW_ONLY_RT_DS_TEXTURES;
    }
}

bitflags! {
    pub struct BufferSrvFlags: i32 {
        const None = D3D12_BUFFER_SRV_FLAGS_D3D12_BUFFER_SRV_FLAG_NONE;
        const Raw = D3D12_BUFFER_SRV_FLAGS_D3D12_BUFFER_SRV_FLAG_RAW;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum SrvDimension {
    Unknown = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_UNKNOWN,
    Buffer = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_BUFFER,
    Texture1D = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE1D,
    Texture1DArray = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE1DARRAY,
    Texture2D = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE2D,
    Texture2DArray = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE2DARRAY,
    Texture2DMs = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE2DMS,
    Texture2DMsArray = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE2DMSARRAY,
    Texture3D = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURE3D,
    TextureCube = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURECUBE,
    TextureCubeArray = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_TEXTURECUBEARRAY,
    RaytracingAccelerationStructure = D3D12_SRV_DIMENSION_D3D12_SRV_DIMENSION_RAYTRACING_ACCELERATION_STRUCTURE,
}

#[repr(u32)]
#[derive(Debug, Clone, Copy)]

pub enum ShaderComponentMappingOptions {
    FromMemoryComponent0 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FROM_MEMORY_COMPONENT_0 as u32,
    FromMemoryComponent1 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FROM_MEMORY_COMPONENT_1 as u32,
    FromMemoryComponent2 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FROM_MEMORY_COMPONENT_2 as u32,
    FromMemoryComponent3 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FROM_MEMORY_COMPONENT_3 as u32,
    ForceValue0 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FORCE_VALUE_0 as u32,
    ForceValue1 = D3D12_SHADER_COMPONENT_MAPPING_D3D12_SHADER_COMPONENT_MAPPING_FORCE_VALUE_1 as u32,
}

const ShaderComponentMappingMask: u32 = 0x7;
const ShaderComponentMappingShift: u32 = 3;

pub struct ShaderComponentMapping(u32);

impl Into<u32> for ShaderComponentMapping {
    fn into(self) -> u32 {
        self.0
    }
}

impl From<u32> for ShaderComponentMapping {
    fn from(value: u32) -> Self {
        Self(value)
    }
}
impl ShaderComponentMapping {
    pub fn default() -> Self {
        Self::encode(
            ShaderComponentMappingOptions::FromMemoryComponent0,
            ShaderComponentMappingOptions::FromMemoryComponent1,
            ShaderComponentMappingOptions::FromMemoryComponent2,
            ShaderComponentMappingOptions::FromMemoryComponent3,
        )
    }

    pub fn encode(
        src0: ShaderComponentMappingOptions,
        src1: ShaderComponentMappingOptions,
        src2: ShaderComponentMappingOptions,
        src3: ShaderComponentMappingOptions,
    ) -> Self {
        Self(
            (src0 as u32 & ShaderComponentMappingMask)
                | ((src1 as u32 & ShaderComponentMappingMask)
                    << ShaderComponentMappingShift)
                | ((src2 as u32 & ShaderComponentMappingMask)
                    << (ShaderComponentMappingShift * 2))
                | ((src3 as u32 & ShaderComponentMappingMask)
                    << (ShaderComponentMappingShift * 3))
                | (1u32 << (ShaderComponentMappingShift * 4u32)),
        )
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum ResourceBarrierType {
    Transition =
        D3D12_RESOURCE_BARRIER_TYPE_D3D12_RESOURCE_BARRIER_TYPE_TRANSITION,
    Aliasing = D3D12_RESOURCE_BARRIER_TYPE_D3D12_RESOURCE_BARRIER_TYPE_ALIASING,
    Uav = D3D12_RESOURCE_BARRIER_TYPE_D3D12_RESOURCE_BARRIER_TYPE_UAV,
}

bitflags! {
    pub struct ResourceBarrierFlags: i32 {
        const None = D3D12_RESOURCE_BARRIER_FLAGS_D3D12_RESOURCE_BARRIER_FLAG_NONE;
        const BeginOnly = D3D12_RESOURCE_BARRIER_FLAGS_D3D12_RESOURCE_BARRIER_FLAG_BEGIN_ONLY;
        const EndOnly = D3D12_RESOURCE_BARRIER_FLAGS_D3D12_RESOURCE_BARRIER_FLAG_END_ONLY;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum DsvDimension {
    Unknown = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_UNKNOWN,
    Texture1D = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE1D,
    Texture1DArray = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE1DARRAY,
    Texture2D = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE2D,
    Texture2DArray = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE2DARRAY,
    Texture2DMs = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE2DMS,
    Texture2DMsArray = D3D12_DSV_DIMENSION_D3D12_DSV_DIMENSION_TEXTURE2DMSARRAY,
}

bitflags! {
    pub struct DsvFlags: i32 {
        const None = D3D12_DSV_FLAGS_D3D12_DSV_FLAG_NONE;
        const ReadOnlyDepth = D3D12_DSV_FLAGS_D3D12_DSV_FLAG_READ_ONLY_DEPTH;
        const ReadOnlyStencil = D3D12_DSV_FLAGS_D3D12_DSV_FLAG_READ_ONLY_STENCIL;
    }
}

bitflags! {
    pub struct ClearFlags: i32 {
        const Depth = D3D12_CLEAR_FLAGS_D3D12_CLEAR_FLAG_DEPTH;
        const Stencil = D3D12_CLEAR_FLAGS_D3D12_CLEAR_FLAG_STENCIL;
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum ShaderModel {
    SM_5_1 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_5_1,
    SM_6_0 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_0,
    SM_6_1 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_1,
    SM_6_2 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_2,
    SM_6_3 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_3,
    SM_6_4 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_4,
    SM_6_5 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_5,
    SM_6_6 = D3D_SHADER_MODEL_D3D_SHADER_MODEL_6_6,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]

pub enum PipelineStateSubobjectType {
    RootSignature = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_ROOT_SIGNATURE,
    VS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_VS,
    PS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_PS,
    DS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_DS,
    HS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_HS,
    GS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_GS,
    CS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_CS,
    StreamOutput = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_STREAM_OUTPUT,
    Blend = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_BLEND,
    SampleMask = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_SAMPLE_MASK,
    Rasterizer = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_RASTERIZER,
    DepthStencil = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_DEPTH_STENCIL,
    InputLayout = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_INPUT_LAYOUT,
    IbStripCutValue = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_IB_STRIP_CUT_VALUE,
    PrimitiveTopology = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_PRIMITIVE_TOPOLOGY,
    RenderTargetFormats = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_RENDER_TARGET_FORMATS,
    DepthStencilFormat = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_DEPTH_STENCIL_FORMAT,
    SampleDesc = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_SAMPLE_DESC,
    NodeMask = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_NODE_MASK,
    CachedPso = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_CACHED_PSO,
    Flags = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_FLAGS,
    DepthStencil1 = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_DEPTH_STENCIL1,
    ViewInstancing = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_VIEW_INSTANCING,
    AS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_AS,
    MS = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_MS,
    MaxValid = D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_D3D12_PIPELINE_STATE_SUBOBJECT_TYPE_MAX_VALID,
}

impl Default for PipelineStateSubobjectType {
    fn default() -> Self {
        Self::MaxValid
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]
pub enum GpuPreference {
    Unspecified = DXGI_GPU_PREFERENCE_DXGI_GPU_PREFERENCE_UNSPECIFIED,
    MinimumPower = DXGI_GPU_PREFERENCE_DXGI_GPU_PREFERENCE_MINIMUM_POWER,
    HighPerformance = DXGI_GPU_PREFERENCE_DXGI_GPU_PREFERENCE_HIGH_PERFORMANCE,
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]
pub enum MessageCategory {
    ApplicationDefined =
        D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_APPLICATION_DEFINED,
    Miscellaneous = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_MISCELLANEOUS,
    Initialization =
        D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_INITIALIZATION,
    Cleanup = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_CLEANUP,
    Compilation = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_COMPILATION,
    StateCreation =
        D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_STATE_CREATION,
    StateSetting = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_STATE_SETTING,
    StateGetting = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_STATE_GETTING,
    ResourceManipulation =
        D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_RESOURCE_MANIPULATION,
    Execution = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_EXECUTION,
    Shader = D3D12_MESSAGE_CATEGORY_D3D12_MESSAGE_CATEGORY_SHADER,
}

// ToDo: macro for enum -> string
impl std::fmt::Display for MessageCategory {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Self::ApplicationDefined => write!(f, "ApplicationDefined"),
            Self::Miscellaneous => write!(f, "Miscellaneous"),
            Self::Initialization => write!(f, "Initialization"),
            Self::Cleanup => write!(f, "Cleanup"),
            Self::Compilation => write!(f, "Compilation"),
            Self::StateCreation => write!(f, "StateCreation"),
            Self::StateSetting => write!(f, "StateSetting"),
            Self::StateGetting => write!(f, "StateGetting"),
            Self::ResourceManipulation => write!(f, "ResourceManipulation"),
            Self::Execution => write!(f, "Execution"),
            Self::Shader => write!(f, "Shader"),
        }
    }
}

#[repr(i32)]
#[derive(Debug, Clone, Copy)]
pub enum MessageSeverity {
    Corruption = D3D12_MESSAGE_SEVERITY_D3D12_MESSAGE_SEVERITY_CORRUPTION,
    Error = D3D12_MESSAGE_SEVERITY_D3D12_MESSAGE_SEVERITY_ERROR,
    Warning = D3D12_MESSAGE_SEVERITY_D3D12_MESSAGE_SEVERITY_WARNING,
    Info = D3D12_MESSAGE_SEVERITY_D3D12_MESSAGE_SEVERITY_INFO,
    Message = D3D12_MESSAGE_SEVERITY_D3D12_MESSAGE_SEVERITY_MESSAGE,
}

impl std::fmt::Display for MessageSeverity {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Self::Corruption => write!(f, "Corruption"),
            Self::Error => write!(f, "Error"),
            Self::Warning => write!(f, "Warning"),
            Self::Info => write!(f, "Info"),
            Self::Message => write!(f, "Message"),
        }
    }
}

// ToDo: fix capital letters
#[repr(i32)]
#[derive(Debug, Clone, Copy)]
pub enum MessageId {
    Unknown = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNKNOWN,
    StringFromApplication =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_STRING_FROM_APPLICATION,
    CorruptedThis = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_THIS,
    CorruptedParameter1 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER1,
    CorruptedParameter2 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER2,
    CorruptedParameter3 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER3,
    CorruptedParameter4 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER4,
    CorruptedParameter5 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER5,
    CorruptedParameter6 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER6,
    CorruptedParameter7 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER7,
    CorruptedParameter8 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER8,
    CorruptedParameter9 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER9,
    CorruptedParameter10 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER10,
    CorruptedParameter11 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER11,
    CorruptedParameter12 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER12,
    CorruptedParameter13 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER13,
    CorruptedParameter14 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER14,
    CorruptedParameter15 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_PARAMETER15,
    CorruptedMultithreading =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CORRUPTED_MULTITHREADING,
    MessageReportingOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MESSAGE_REPORTING_OUTOFMEMORY,
    GetPrivateDataMoredata =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETPRIVATEDATA_MOREDATA,
    SetPrivateDataInvalidfreedata =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETPRIVATEDATA_INVALIDFREEDATA,
    SetPrivateDataChangingparams =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETPRIVATEDATA_CHANGINGPARAMS,
    SetPrivateDataOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETPRIVATEDATA_OUTOFMEMORY,
    CreateShaderResourceViewUnrecognizedformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_UNRECOGNIZEDFORMAT,
    CreateShaderResourceViewInvaliddesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDDESC,
    CreateShaderResourceViewInvalidformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDFORMAT,
    CreateShaderResourceViewInvalidvideoplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDVIDEOPLANESLICE,
    CreateShaderResourceViewInvalidplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDPLANESLICE,
    CreateShaderResourceViewInvaliddimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDDIMENSIONS,
    CreateShaderResourceViewInvalidresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERRESOURCEVIEW_INVALIDRESOURCE,
    CreateRenderTargetViewUnrecognizedformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_UNRECOGNIZEDFORMAT,
    CreateRenderTargetViewUnsupportedformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_UNSUPPORTEDFORMAT,
    CreateRenderTargetViewInvaliddesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDDESC,
    CreateRenderTargetViewInvalidformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDFORMAT,
    CreateRenderTargetViewInvalidvideoplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDVIDEOPLANESLICE,
    CreateRenderTargetViewInvalidplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDPLANESLICE,
    CreateRenderTargetViewInvaliddimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDDIMENSIONS,
    CreateRenderTargetViewInvalidresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERENDERTARGETVIEW_INVALIDRESOURCE,
    CreateDepthStencilViewUnrecognizedformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_UNRECOGNIZEDFORMAT,
    CreateDepthStencilViewInvaliddesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_INVALIDDESC,
    CreateDepthStencilViewInvalidformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_INVALIDFORMAT,
    CreateDepthStencilViewInvaliddimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_INVALIDDIMENSIONS,
    CreateDepthStencilViewInvalidresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_INVALIDRESOURCE,
    CreateInputLayoutOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_OUTOFMEMORY,
    CreateInputLayoutToomanyelements =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_TOOMANYELEMENTS,
    CreateInputLayoutInvalidformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDFORMAT,
    CreateInputLayoutIncompatibleformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INCOMPATIBLEFORMAT,
    CreateInputLayoutInvalidslot =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDSLOT,
    CreateInputLayoutInvalidinputslotclass = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDINPUTSLOTCLASS,
    CreateInputLayoutSteprateslotclassmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_STEPRATESLOTCLASSMISMATCH,
    CreateInputLayoutInvalidslotclasschange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDSLOTCLASSCHANGE,
    CreateInputLayoutInvalidstepratechange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDSTEPRATECHANGE,
    CreateInputLayoutInvalidalignment =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_INVALIDALIGNMENT,
    CreateInputLayoutDuplicatesemantic =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_DUPLICATESEMANTIC,
    CreateInputLayoutUnparseableinputsignature = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_UNPARSEABLEINPUTSIGNATURE,
    CreateInputLayoutNullsemantic =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_NULLSEMANTIC,
    CreateInputLayoutMissingelement =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_MISSINGELEMENT,
    CreateVertexShaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEVERTEXSHADER_OUTOFMEMORY,
    CreateVertexShaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEVERTEXSHADER_INVALIDSHADERBYTECODE,
    CreateVertexShaderInvalidshadertype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEVERTEXSHADER_INVALIDSHADERTYPE,
    CreateGeometryShaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADER_OUTOFMEMORY,
    CreateGeometryShaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADER_INVALIDSHADERBYTECODE,
    CreateGeometryShaderInvalidshadertype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADER_INVALIDSHADERTYPE,
    CreateGeometryShaderWithStreamOutputOutOfMemory = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_OUTOFMEMORY,
    CreateGeometryShaderWithStreamOutputInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDSHADERBYTECODE,
    CreateGeometryShaderWithStreamOutputInvalidshadertype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDSHADERTYPE,
    CreateGeometryShaderWithStreamOutputInvalidnumentries = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDNUMENTRIES,
    CreateGeometryShaderWithStreamOutputOutputstreamstrideunused = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_OUTPUTSTREAMSTRIDEUNUSED,
    CreateGeometryShaderWithStreamOutputOutputslot0Expected = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_OUTPUTSLOT0EXPECTED,
    CreateGeometryShaderWithStreamOutputInvalidoutputslot = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDOUTPUTSLOT,
    CreateGeometryShaderWithStreamOutputOnlyoneelementperslot = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_ONLYONEELEMENTPERSLOT,
    CreateGeometryShaderWithStreamOutputInvalidcomponentcount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDCOMPONENTCOUNT,
    CreateGeometryShaderWithStreamOutputInvalidstartcomponentandcomponentcount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDSTARTCOMPONENTANDCOMPONENTCOUNT,
    CreateGeometryShaderWithStreamOutputInvalidgapdefinition = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDGAPDEFINITION,
    CreateGeometryShaderWithStreamOutputRepeatedOutput = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_REPEATEDOUTPUT,
    CreateGeometryShaderWithStreamOutputInvalidOutputStreamStride = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDOUTPUTSTREAMSTRIDE,
    CreateGeometryShaderWithStreamOutputMissingSemantic = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_MISSINGSEMANTIC,
    CreateGeometryShaderWithStreamOutputMaskMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_MASKMISMATCH,
    CreateGeometryShaderWithStreamOutputCantHaveOnlyGaps = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_CANTHAVEONLYGAPS,
    CreateGeometryShaderWithStreamOutputDeclTooComplex = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_DECLTOOCOMPLEX,
    CreateGeometryShaderWithStreamOutputMissingOutputSignature = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_MISSINGOUTPUTSIGNATURE,
    CreatePixelShaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIXELSHADER_OUTOFMEMORY,
    CreatePixelShaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIXELSHADER_INVALIDSHADERBYTECODE,
    CreatePixelShaderInvalidshadertype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIXELSHADER_INVALIDSHADERTYPE,
    CreateRasterizerStateInvalidfillmode =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALIDFILLMODE,
    CreateRasterizerStateInvalidcullmode =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALIDCULLMODE,
    CreateRasterizerStateInvaliddepthbiasclamp = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALIDDEPTHBIASCLAMP,
    CreateRasterizerStateInvalidslopescaleddepthbias = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALIDSLOPESCALEDDEPTHBIAS,
    CreatedepthstencilstateInvaliddepthwritemask = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDDEPTHWRITEMASK,
    CreatedepthstencilstateInvaliddepthfunc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDDEPTHFUNC,
    CreatedepthstencilstateInvalidfrontfacestencilfailop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDFRONTFACESTENCILFAILOP,
    CreatedepthstencilstateInvalidfrontfacestencilzfailop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDFRONTFACESTENCILZFAILOP,
    CreatedepthstencilstateInvalidfrontfacestencilpassop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDFRONTFACESTENCILPASSOP,
    CreatedepthstencilstateInvalidfrontfacestencilfunc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDFRONTFACESTENCILFUNC,
    CreatedepthstencilstateInvalidbackfacestencilfailop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDBACKFACESTENCILFAILOP,
    CreatedepthstencilstateInvalidbackfacestencilzfailop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDBACKFACESTENCILZFAILOP,
    CreatedepthstencilstateInvalidbackfacestencilpassop = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDBACKFACESTENCILPASSOP,
    CreatedepthstencilstateInvalidbackfacestencilfunc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_INVALIDBACKFACESTENCILFUNC,
    CreateblendstateInvalidsrcblend =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDSRCBLEND,
    CreateblendstateInvaliddestblend =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDDESTBLEND,
    CreateblendstateInvalidblendop =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDBLENDOP,
    CreateblendstateInvalidsrcblendalpha =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDSRCBLENDALPHA,
    CreateblendstateInvaliddestblendalpha = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDDESTBLENDALPHA,
    CreateblendstateInvalidblendopalpha =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDBLENDOPALPHA,
    CreateblendstateInvalidrendertargetwritemask = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDRENDERTARGETWRITEMASK,
    CleardepthstencilviewInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEARDEPTHSTENCILVIEW_INVALID,
    CommandListDrawRootSignatureNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_ROOT_SIGNATURE_NOT_SET,
    CommandListDrawRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_ROOT_SIGNATURE_MISMATCH,
    CommandListDrawVertexBufferNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_VERTEX_BUFFER_NOT_SET,
    CommandListDrawVertexBufferStrideTooSmall = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_VERTEX_BUFFER_STRIDE_TOO_SMALL,
    CommandListDrawVertexBufferTooSmall = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_VERTEX_BUFFER_TOO_SMALL,
    CommandListDrawIndexBufferNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_INDEX_BUFFER_NOT_SET,
    CommandListDrawIndexBufferFormatInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_INDEX_BUFFER_FORMAT_INVALID,
    CommandListDrawIndexBufferTooSmall = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_INDEX_BUFFER_TOO_SMALL,
    CommandListDrawInvalidPrimitivetopology = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_INVALID_PRIMITIVETOPOLOGY,
    CommandListDrawVertexStrideUnaligned = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_VERTEX_STRIDE_UNALIGNED,
    CommandListDrawIndexOffsetUnaligned = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_INDEX_OFFSET_UNALIGNED,
    DeviceRemovalProcessAtFault =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_REMOVAL_PROCESS_AT_FAULT,
    DeviceRemovalProcessPossiblyAtFault = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_REMOVAL_PROCESS_POSSIBLY_AT_FAULT,
    DeviceRemovalProcessNotAtFault =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_REMOVAL_PROCESS_NOT_AT_FAULT,
    CreateInputLayoutTrailingDigitInSemantic = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_TRAILING_DIGIT_IN_SEMANTIC,
    CreateGeometryShaderWithStreamOutputTrailingDigitInSemantic = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_TRAILING_DIGIT_IN_SEMANTIC,
    CreateInputLayoutTypeMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_TYPE_MISMATCH,
    CreateInputLayoutEmptyLayout =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEINPUTLAYOUT_EMPTY_LAYOUT,
    LiveObjectSummary = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_OBJECT_SUMMARY,
    LiveDevice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_DEVICE,
    LiveSwapchain = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_SWAPCHAIN,
    CreateDepthStencilViewInvalidflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILVIEW_INVALIDFLAGS,
    CreateVertexShaderInvalidclasslinkage = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEVERTEXSHADER_INVALIDCLASSLINKAGE,
    CreateGeometryShaderInvalidclasslinkage = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADER_INVALIDCLASSLINKAGE,
    CreateGeometryShaderWithStreamOutputInvalidstreamtorasterizer = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDSTREAMTORASTERIZER,
    CreatePixelShaderInvalidclasslinkage =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIXELSHADER_INVALIDCLASSLINKAGE,
    CreateGeometryShaderWithStreamOutputInvalidstream = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDSTREAM,
    CreateGeometryShaderWithStreamOutputUnexpectedentries = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_UNEXPECTEDENTRIES,
    CreateGeometryShaderWithStreamOutputUnexpectedstrides = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_UNEXPECTEDSTRIDES,
    CreateGeometryShaderWithStreamOutputInvalidnumstrides = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_INVALIDNUMSTRIDES,
    CreatehullshaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHULLSHADER_OUTOFMEMORY,
    CreatehullshaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHULLSHADER_INVALIDSHADERBYTECODE,
    CreatehullshaderInvalidshadertype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHULLSHADER_INVALIDSHADERTYPE,
    CreatehullshaderInvalidclasslinkage =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHULLSHADER_INVALIDCLASSLINKAGE,
    CreatedomainshaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDOMAINSHADER_OUTOFMEMORY,
    CreatedomainshaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDOMAINSHADER_INVALIDSHADERBYTECODE,
    CreatedomainshaderInvalidshadertype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDOMAINSHADER_INVALIDSHADERTYPE,
    CreatedomainshaderInvalidclasslinkage = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDOMAINSHADER_INVALIDCLASSLINKAGE,
    ResourceUnmapNotmapped =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_UNMAP_NOTMAPPED,
    DeviceCheckfeaturesupportMismatchedDataSize = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CHECKFEATURESUPPORT_MISMATCHED_DATA_SIZE,
    CreateComputeShaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTESHADER_OUTOFMEMORY,
    CreateComputeShaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTESHADER_INVALIDSHADERBYTECODE,
    CreateComputeShaderInvalidclasslinkage = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTESHADER_INVALIDCLASSLINKAGE,
    DeviceCreateVertexShaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEVERTEXSHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreatehullshaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEHULLSHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreatedomainshaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEDOMAINSHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreateGeometryShaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreateGeometryShaderWithStreamOutputDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreatePixelShaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEPIXELSHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    DeviceCreateComputeShaderDoublefloatopsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATECOMPUTESHADER_DOUBLEFLOATOPSNOTSUPPORTED,
    CreateunorderedaccessviewInvalidresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDRESOURCE,
    CreateunorderedaccessviewInvaliddesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDDESC,
    CreateunorderedaccessviewInvalidformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDFORMAT,
    CreateunorderedaccessviewInvalidvideoplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDVIDEOPLANESLICE,
    CreateunorderedaccessviewInvalidplaneslice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDPLANESLICE,
    CreateunorderedaccessviewInvaliddimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDDIMENSIONS,
    CreateunorderedaccessviewUnrecognizedformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_UNRECOGNIZEDFORMAT,
    CreateunorderedaccessviewInvalidflags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEUNORDEREDACCESSVIEW_INVALIDFLAGS,
    CreateRasterizerStateInvalidforcedsamplecount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALIDFORCEDSAMPLECOUNT,
    CreateblendstateInvalidlogicops =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_INVALIDLOGICOPS,
    DeviceCreateVertexShaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEVERTEXSHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreatehullshaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEHULLSHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreatedomainshaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEDOMAINSHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreateGeometryShaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreateGeometryShaderWithStreamOutputDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreatePixelShaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEPIXELSHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreateComputeShaderDoubleextensionsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATECOMPUTESHADER_DOUBLEEXTENSIONSNOTSUPPORTED,
    DeviceCreateVertexShaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEVERTEXSHADER_UAVSNOTSUPPORTED,
    DeviceCreatehullshaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEHULLSHADER_UAVSNOTSUPPORTED,
    DeviceCreatedomainshaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEDOMAINSHADER_UAVSNOTSUPPORTED,
    DeviceCreateGeometryShaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADER_UAVSNOTSUPPORTED,
    DeviceCreateGeometryShaderWithStreamOutputUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEGEOMETRYSHADERWITHSTREAMOUTPUT_UAVSNOTSUPPORTED,
    DeviceCreatePixelShaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATEPIXELSHADER_UAVSNOTSUPPORTED,
    DeviceCreateComputeShaderUavsnotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATECOMPUTESHADER_UAVSNOTSUPPORTED,
    DeviceClearviewInvalidsourcerect =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CLEARVIEW_INVALIDSOURCERECT,
    DeviceClearviewEmptyrect =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CLEARVIEW_EMPTYRECT,
    UpdatetilemappingsInvalidParameter =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UPDATETILEMAPPINGS_INVALID_PARAMETER,
    CopytilemappingsInvalidParameter =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTILEMAPPINGS_INVALID_PARAMETER,
    CreatedeviceInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEVICE_INVALIDARGS,
    CreatedeviceWarning =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEVICE_WARNING,
    ResourceBarrierInvalidType =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_TYPE,
    ResourceBarrierNullPointer =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_NULL_POINTER,
    ResourceBarrierInvalidSubresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_SUBRESOURCE,
    ResourceBarrierReservedBits =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_RESERVED_BITS,
    ResourceBarrierMissingBindFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_MISSING_BIND_FLAGS,
    ResourceBarrierMismatchingMiscFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_MISMATCHING_MISC_FLAGS,
    ResourceBarrierMatchingStates =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_MATCHING_STATES,
    ResourceBarrierInvalidCombination =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_COMBINATION,
    ResourceBarrierBeforeAfterMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_BEFORE_AFTER_MISMATCH,
    ResourceBarrierInvalidResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_RESOURCE,
    ResourceBarrierSampleCount =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_SAMPLE_COUNT,
    ResourceBarrierInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_FLAGS,
    ResourceBarrierInvalidCombinedFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_COMBINED_FLAGS,
    ResourceBarrierInvalidFlagsForFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_FLAGS_FOR_FORMAT,
    ResourceBarrierInvalidSplitBarrier = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_SPLIT_BARRIER,
    ResourceBarrierUnmatchedEnd =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_UNMATCHED_END,
    ResourceBarrierUnmatchedBegin =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_UNMATCHED_BEGIN,
    ResourceBarrierInvalidFlag =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_FLAG,
    ResourceBarrierInvalidCommandListType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_COMMAND_LIST_TYPE,
    InvalidSubresourceState =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_SUBRESOURCE_STATE,
    CommandAllocatorContention =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_CONTENTION,
    CommandAllocatorReset =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_RESET,
    CommandAllocatorResetBundle =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_RESET_BUNDLE,
    CommandAllocatorCannotReset =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_CANNOT_RESET,
    CommandListOpen = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_OPEN,
    InvalidBundleApi = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_BUNDLE_API,
    CommandListClosed = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_CLOSED,
    WrongCommandAllocatorType =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRONG_COMMAND_ALLOCATOR_TYPE,
    CommandAllocatorSync =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_SYNC,
    CommandListSync = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_SYNC,
    SetDescriptorHeapInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_DESCRIPTOR_HEAP_INVALID,
    CreateCommandqueue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDQUEUE,
    CreateCommandallocator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDALLOCATOR,
    CreatePipelinestate =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_PIPELINESTATE,
    CreateCommandlist12 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDLIST12,
    CreateResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_RESOURCE,
    CreateDescriptorheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_DESCRIPTORHEAP,
    CreateRootsignature =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOTSIGNATURE,
    CreateLibrary = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_LIBRARY,
    CreateHeap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_HEAP,
    CreateMonitoredfence =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_MONITOREDFENCE,
    CreateQueryheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_QUERYHEAP,
    CreateCommandsignature =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDSIGNATURE,
    LiveCommandqueue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDQUEUE,
    LiveCommandallocator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDALLOCATOR,
    LivePipelinestate = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_PIPELINESTATE,
    LiveCommandlist12 = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDLIST12,
    LiveResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_RESOURCE,
    LiveDescriptorheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_DESCRIPTORHEAP,
    LiveRootsignature = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_ROOTSIGNATURE,
    LiveLibrary = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_LIBRARY,
    LiveHeap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_HEAP,
    LiveMonitoredfence = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_MONITOREDFENCE,
    LiveQueryheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_QUERYHEAP,
    LiveCommandsignature =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDSIGNATURE,
    DestroyCommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDQUEUE,
    DestroyCommandallocator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDALLOCATOR,
    DestroyPipelinestate =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_PIPELINESTATE,
    DestroyCommandlist12 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDLIST12,
    DestroyResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_RESOURCE,
    DestroyDescriptorheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_DESCRIPTORHEAP,
    DestroyRootsignature =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_ROOTSIGNATURE,
    DestroyLibrary = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_LIBRARY,
    DestroyHeap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_HEAP,
    DestroyMonitoredfence =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_MONITOREDFENCE,
    DestroyQueryheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_QUERYHEAP,
    DestroyCommandsignature =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDSIGNATURE,
    CreateResourceInvalidDimensions =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDDIMENSIONS,
    CreateResourceInvalidMiscFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDMISCFLAGS,
    CreateResourceInvalidArgReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDARG_RETURN,
    CreateResourceOutOfMemoryReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_OUTOFMEMORY_RETURN,
    CreateResourceInvalidDesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDDESC,
    PossiblyInvalidSubresourceState =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_POSSIBLY_INVALID_SUBRESOURCE_STATE,
    InvalidUseOfNonResidentResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_USE_OF_NON_RESIDENT_RESOURCE,
    PossibleInvalidUseOfNonResidentResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_POSSIBLE_INVALID_USE_OF_NON_RESIDENT_RESOURCE,
    BundlePipelineStateMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_BUNDLE_PIPELINE_STATE_MISMATCH,
    PrimitiveTopologyMismatchPipelineState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PRIMITIVE_TOPOLOGY_MISMATCH_PIPELINE_STATE,
    RenderTargetFormatMismatchPipelineState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_TARGET_FORMAT_MISMATCH_PIPELINE_STATE,
    RenderTargetSampleDescMismatchPipelineState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_TARGET_SAMPLE_DESC_MISMATCH_PIPELINE_STATE,
    DepthStencilFormatMismatchPipelineState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEPTH_STENCIL_FORMAT_MISMATCH_PIPELINE_STATE,
    DepthStencilSampleDescMismatchPipelineState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEPTH_STENCIL_SAMPLE_DESC_MISMATCH_PIPELINE_STATE,
    CreateshaderInvalidbytecode =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADER_INVALIDBYTECODE,
    CreateHeapNulldesc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_NULLDESC,
    CreateHeapInvalidsize =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_INVALIDSIZE,
    CreateHeapUnrecognizedheaptype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_UNRECOGNIZEDHEAPTYPE,
    CreateHeapUnrecognizedcpupageproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_UNRECOGNIZEDCPUPAGEPROPERTIES,
    CreateHeapUnrecognizedmemorypool =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_UNRECOGNIZEDMEMORYPOOL,
    CreateHeapInvalidproperties =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_INVALIDPROPERTIES,
    CreateHeapInvalidalignment =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_INVALIDALIGNMENT,
    CreateHeapUnrecognizedmiscflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_UNRECOGNIZEDMISCFLAGS,
    CreateHeapInvalidmiscflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_INVALIDMISCFLAGS,
    CreateHeapInvalidargReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_INVALIDARG_RETURN,
    CreateHeapOutOfMemoryReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEHEAP_OUTOFMEMORY_RETURN,
    CreateResourceAndHeapNullheapproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_NULLHEAPPROPERTIES,
    CreateResourceAndHeapUnrecognizedheaptype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_UNRECOGNIZEDHEAPTYPE,
    CreateResourceAndHeapUnrecognizedcpupageproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_UNRECOGNIZEDCPUPAGEPROPERTIES,
    CreateResourceAndHeapUnrecognizedmemorypool = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_UNRECOGNIZEDMEMORYPOOL,
    CreateResourceAndHeapInvalidheapproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_INVALIDHEAPPROPERTIES,
    CreateResourceAndHeapUnrecognizedheapmiscflags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_UNRECOGNIZEDHEAPMISCFLAGS,
    CreateResourceAndHeapInvalidheapmiscflags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_INVALIDHEAPMISCFLAGS,
    CreateResourceAndHeapInvalidargReturn = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_INVALIDARG_RETURN,
    CreateResourceAndHeapOutOfMemoryReturn = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_OUTOFMEMORY_RETURN,
    GetCustomHeapPropertiesUnrecognizedheaptype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETCUSTOMHEAPPROPERTIES_UNRECOGNIZEDHEAPTYPE,
    GetCustomHeapPropertiesInvalidheaptype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETCUSTOMHEAPPROPERTIES_INVALIDHEAPTYPE,
    CreateDescriptorHeapInvalidDesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_DESCRIPTOR_HEAP_INVALID_DESC,
    InvalidDescriptorHandle =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_DESCRIPTOR_HANDLE,
    CreateRasterizerStateInvalidConservativerastermode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERASTERIZERSTATE_INVALID_CONSERVATIVERASTERMODE,
    CreateConstantBufferViewInvalidResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_CONSTANT_BUFFER_VIEW_INVALID_RESOURCE,
    CreateConstantBufferViewInvalidDesc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_CONSTANT_BUFFER_VIEW_INVALID_DESC,
    CreateUnorderedaccessViewInvalidCounterUsage = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_UNORDEREDACCESS_VIEW_INVALID_COUNTER_USAGE,
    CopyDescriptorsInvalidRanges =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPY_DESCRIPTORS_INVALID_RANGES,
    CopyDescriptorsWriteOnlyDescriptor = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPY_DESCRIPTORS_WRITE_ONLY_DESCRIPTOR,
    CreateGraphicsPipelineStateRtvFormatNotUnknown = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_RTV_FORMAT_NOT_UNKNOWN,
    CreateGraphicsPipelineStateInvalidRenderTargetCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_RENDER_TARGET_COUNT,
    CreateGraphicsPipelineStateVertexShaderNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_VERTEX_SHADER_NOT_SET,
    CreateGraphicsPipelineStateInputlayoutNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INPUTLAYOUT_NOT_SET,
    CreateGraphicsPipelineStateShaderLinkageHsDsSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_HS_DS_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateShaderLinkageRegisterindex = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_REGISTERINDEX,
    CreateGraphicsPipelineStateShaderLinkageComponenttype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_COMPONENTTYPE,
    CreateGraphicsPipelineStateShaderLinkageRegistermask = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_REGISTERMASK,
    CreateGraphicsPipelineStateShaderLinkageSystemvalue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_SYSTEMVALUE,
    CreateGraphicsPipelineStateShaderLinkageNeverwrittenAlwaysreads = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_NEVERWRITTEN_ALWAYSREADS,
    CreateGraphicsPipelineStateShaderLinkageMinprecision = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_MINPRECISION,
    CreateGraphicsPipelineStateShaderLinkageSemanticnameNotFound = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_LINKAGE_SEMANTICNAME_NOT_FOUND,
    CreateGraphicsPipelineStateHsXorDsMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_HS_XOR_DS_MISMATCH,
    CreateGraphicsPipelineStateHullShaderInputTopologyMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_HULL_SHADER_INPUT_TOPOLOGY_MISMATCH,
    CreateGraphicsPipelineStateHsDsControlPointCountMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_HS_DS_CONTROL_POINT_COUNT_MISMATCH,
    CreateGraphicsPipelineStateHsDsTessellatorDomainMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_HS_DS_TESSELLATOR_DOMAIN_MISMATCH,
    CreateGraphicsPipelineStateInvalidUseOfCenterMultisamplePattern = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_USE_OF_CENTER_MULTISAMPLE_PATTERN,
    CreateGraphicsPipelineStateInvalidUseOfForcedSampleCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_USE_OF_FORCED_SAMPLE_COUNT,
    CreateGraphicsPipelineStateInvalidPrimitivetopology = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_PRIMITIVETOPOLOGY,
    CreateGraphicsPipelineStateInvalidSystemvalue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_SYSTEMVALUE,
    CreateGraphicsPipelineStateOmDualSourceBlendingCanOnlyHaveRenderTarget0 = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_OM_DUAL_SOURCE_BLENDING_CAN_ONLY_HAVE_RENDER_TARGET_0,
    CreateGraphicsPipelineStateOmRenderTargetDoesNotSupportBlending = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_OM_RENDER_TARGET_DOES_NOT_SUPPORT_BLENDING,
    CreateGraphicsPipelineStatePsOutputTypeMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_PS_OUTPUT_TYPE_MISMATCH,
    CreateGraphicsPipelineStateOmRenderTargetDoesNotSupportLogicOps = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_OM_RENDER_TARGET_DOES_NOT_SUPPORT_LOGIC_OPS,
    CreateGraphicsPipelineStateRendertargetviewNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_RENDERTARGETVIEW_NOT_SET,
    CreateGraphicsPipelineStateDepthstencilviewNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_DEPTHSTENCILVIEW_NOT_SET,
    CreateGraphicsPipelineStateGsInputPrimitiveMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_GS_INPUT_PRIMITIVE_MISMATCH,
    CreateGraphicsPipelineStatePositionNotPresent = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_POSITION_NOT_PRESENT,
    CreateGraphicsPipelineStateMissingRootSignatureFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_MISSING_ROOT_SIGNATURE_FLAGS,
    CreateGraphicsPipelineStateInvalidIndexBufferProperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_INDEX_BUFFER_PROPERTIES,
    CreateGraphicsPipelineStateInvalidSampleDesc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INVALID_SAMPLE_DESC,
    CreateGraphicsPipelineStateHsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_HS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateDsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_DS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateVsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_VS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateGsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_GS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStatePsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_PS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateMissingRootSignature = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_MISSING_ROOT_SIGNATURE,
    ExecuteBundleOpenBundle =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTE_BUNDLE_OPEN_BUNDLE,
    ExecuteBundleDescriptorHeapMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTE_BUNDLE_DESCRIPTOR_HEAP_MISMATCH,
    ExecuteBundleType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTE_BUNDLE_TYPE,
    DrawEmptyScissorRectangle =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DRAW_EMPTY_SCISSOR_RECTANGLE,
    CreateRootSignatureBlobNotFound =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_BLOB_NOT_FOUND,
    CreateRootSignatureDeserializeFailed = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_DESERIALIZE_FAILED,
    CreateRootSignatureInvalidConfiguration = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_INVALID_CONFIGURATION,
    CreateRootSignatureNotSupportedOnDevice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_NOT_SUPPORTED_ON_DEVICE,
    CreateResourceAndHeapNullresourceproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_NULLRESOURCEPROPERTIES,
    CreateResourceAndHeapNullheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCEANDHEAP_NULLHEAP,
    GetresourceallocationinfoInvalidrdescs = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETRESOURCEALLOCATIONINFO_INVALIDRDESCS,
    MakeresidentNullobjectarray =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAKERESIDENT_NULLOBJECTARRAY,
    EvictNullobjectarray =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EVICT_NULLOBJECTARRAY,
    SetDescriptorTableInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_DESCRIPTOR_TABLE_INVALID,
    SetRootConstantInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_ROOT_CONSTANT_INVALID,
    SetRootConstantBufferViewInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_ROOT_CONSTANT_BUFFER_VIEW_INVALID,
    SetRootShaderResourceViewInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_ROOT_SHADER_RESOURCE_VIEW_INVALID,
    SetRootUnorderedAccessViewInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_ROOT_UNORDERED_ACCESS_VIEW_INVALID,
    SetVertexBuffersInvalidDesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_VERTEX_BUFFERS_INVALID_DESC,
    SetIndexBufferInvalidDesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_INDEX_BUFFER_INVALID_DESC,
    SetStreamOutputBuffersInvalidDesc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_STREAM_OUTPUT_BUFFERS_INVALID_DESC,
    CreateResourceUnrecognizeddimensionality = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_UNRECOGNIZEDDIMENSIONALITY,
    CreateResourceUnrecognizedlayout =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_UNRECOGNIZEDLAYOUT,
    CreateResourceInvaliddimensionality =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDDIMENSIONALITY,
    CreateResourceInvalidalignment =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDALIGNMENT,
    CreateResourceInvalidmiplevels =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDMIPLEVELS,
    CreateResourceInvalidsampledesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDSAMPLEDESC,
    CreateResourceInvalidlayout =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDLAYOUT,
    SetIndexBufferInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_INDEX_BUFFER_INVALID,
    SetVertexBuffersInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_VERTEX_BUFFERS_INVALID,
    SetStreamOutputBuffersInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_STREAM_OUTPUT_BUFFERS_INVALID,
    SetRenderTargetsInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_RENDER_TARGETS_INVALID,
    CreatequeryHeapInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEQUERY_HEAP_INVALID_PARAMETERS,
    BeginEndQueryInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_BEGIN_END_QUERY_INVALID_PARAMETERS,
    CloseCommandListOpenQuery =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLOSE_COMMAND_LIST_OPEN_QUERY,
    ResolveQueryDataInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVE_QUERY_DATA_INVALID_PARAMETERS,
    SetPredicationInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_PREDICATION_INVALID_PARAMETERS,
    TimestampsNotSupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_TIMESTAMPS_NOT_SUPPORTED,
    CreateResourceUnrecognizedformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_UNRECOGNIZEDFORMAT,
    CreateResourceInvalidformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDFORMAT,
    GetCopyableFootprintsOrCopyableLayoutInvalidSubresourcerange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETCOPYABLEFOOTPRINTS_INVALIDSUBRESOURCERANGE,
    GetCopyableFootprintsOrCopyableLayoutInvalidbaseoffset = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETCOPYABLEFOOTPRINTS_INVALIDBASEOFFSET,
    ResourceBarrierInvalidHeap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_INVALID_HEAP,
    CreateSamplerInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_SAMPLER_INVALID,
    CreatecommandsignatureInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMMANDSIGNATURE_INVALID,
    ExecuteIndirectInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTE_INDIRECT_INVALID_PARAMETERS,
    GetgpuvirtualaddressInvalidResourceDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETGPUVIRTUALADDRESS_INVALID_RESOURCE_DIMENSION,
    CreateResourceInvalidclearvalue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDCLEARVALUE,
    CreateResourceUnrecognizedclearvalueformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_UNRECOGNIZEDCLEARVALUEFORMAT,
    CreateResourceInvalidclearvalueformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_INVALIDCLEARVALUEFORMAT,
    CreateResourceClearvaluedenormflush =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATERESOURCE_CLEARVALUEDENORMFLUSH,
    ClearrendertargetviewMismatchingclearvalue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEARRENDERTARGETVIEW_MISMATCHINGCLEARVALUE,
    CleardepthstencilviewMismatchingclearvalue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEARDEPTHSTENCILVIEW_MISMATCHINGCLEARVALUE,
    MapInvalidheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDHEAP,
    UnmapInvalidheap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_INVALIDHEAP,
    MapInvalidresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDRESOURCE,
    UnmapInvalidresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_INVALIDRESOURCE,
    MapInvalidSubresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDSUBRESOURCE,
    UnmapInvalidSubresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_INVALIDSUBRESOURCE,
    MapInvalidrange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDRANGE,
    UnmapInvalidrange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_INVALIDRANGE,
    MapInvaliddatapointer =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDDATAPOINTER,
    MapInvalidargReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALIDARG_RETURN,
    MapOutOfMemoryReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_OUTOFMEMORY_RETURN,
    ExecuteCommandListsBundlenotsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_BUNDLENOTSUPPORTED,
    ExecuteCommandListsCommandlistmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_COMMANDLISTMISMATCH,
    ExecuteCommandListsOpenCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_OPENCOMMANDLIST,
    ExecuteCommandListsFailedCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_FAILEDCOMMANDLIST,
    CopyBufferRegionNulldst =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_NULLDST,
    CopyBufferRegionInvaliddstresourcedimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_INVALIDDSTRESOURCEDIMENSION,
    CopyBufferRegionDstrangeoutofbounds =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_DSTRANGEOUTOFBOUNDS,
    CopyBufferRegionNullsrc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_NULLSRC,
    CopyBufferRegionInvalidsrcresourcedimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_INVALIDSRCRESOURCEDIMENSION,
    CopyBufferRegionSrcrangeoutofbounds =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_SRCRANGEOUTOFBOUNDS,
    CopyBufferRegionInvalidcopyflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_INVALIDCOPYFLAGS,
    CopyTextureRegionNulldst =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_NULLDST,
    CopyTextureRegionUnrecognizeddsttype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_UNRECOGNIZEDDSTTYPE,
    CopyTextureRegionInvaliddstresourcedimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTRESOURCEDIMENSION,
    CopyTextureRegionInvaliddstresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTRESOURCE,
    CopyTextureRegionInvaliddstSubresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTSUBRESOURCE,
    CopyTextureRegionInvaliddstoffset =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTOFFSET,
    CopyTextureRegionUnrecognizeddstformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_UNRECOGNIZEDDSTFORMAT,
    CopyTextureRegionInvaliddstformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTFORMAT,
    CopyTextureRegionInvaliddstdimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTDIMENSIONS,
    CopyTextureRegionInvaliddstrowpitch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTROWPITCH,
    CopyTextureRegionInvaliddstplacement =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTPLACEMENT,
    CopyTextureRegionInvaliddstdsplacedfootprintformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTDSPLACEDFOOTPRINTFORMAT,
    CopyTextureRegionDstregionoutofbounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_DSTREGIONOUTOFBOUNDS,
    CopyTextureRegionNullsrc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_NULLSRC,
    CopyTextureRegionUnrecognizedsrctype =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_UNRECOGNIZEDSRCTYPE,
    CopyTextureRegionInvalidsrcresourcedimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCRESOURCEDIMENSION,
    CopyTextureRegionInvalidsrcresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCRESOURCE,
    CopyTextureRegionInvalidsrcSubresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCSUBRESOURCE,
    CopyTextureRegionInvalidsrcoffset =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCOFFSET,
    CopyTextureRegionUnrecognizedsrcformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_UNRECOGNIZEDSRCFORMAT,
    CopyTextureRegionInvalidsrcformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCFORMAT,
    CopyTextureRegionInvalidsrcdimensions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCDIMENSIONS,
    CopyTextureRegionInvalidsrcrowpitch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCROWPITCH,
    CopyTextureRegionInvalidsrcplacement =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCPLACEMENT,
    CopyTextureRegionInvalidsrcdsplacedfootprintformat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCDSPLACEDFOOTPRINTFORMAT,
    CopyTextureRegionSrcregionoutofbounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_SRCREGIONOUTOFBOUNDS,
    CopyTextureRegionInvaliddstcoordinates = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDDSTCOORDINATES,
    CopyTextureRegionInvalidsrcbox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDSRCBOX,
    CopyTextureRegionFormatmismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_FORMATMISMATCH,
    CopyTextureRegionEmptybox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_EMPTYBOX,
    CopyTextureRegionInvalidcopyflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_INVALIDCOPYFLAGS,
    ResolveSubresourceInvalidSubresourceIndex = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_INVALID_SUBRESOURCE_INDEX,
    ResolveSubresourceInvalidFormat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_INVALID_FORMAT,
    ResolveSubresourceResourceMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_RESOURCE_MISMATCH,
    ResolveSubresourceInvalidSampleCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_INVALID_SAMPLE_COUNT,
    CreateComputePipelineStateInvalidShader = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTEPIPELINESTATE_INVALID_SHADER,
    CreateComputePipelineStateCsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTEPIPELINESTATE_CS_ROOT_SIGNATURE_MISMATCH,
    CreateComputePipelineStateMissingRootSignature = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMPUTEPIPELINESTATE_MISSING_ROOT_SIGNATURE,
    CreatePipelineStateInvalidcachedblob =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_INVALIDCACHEDBLOB,
    CreatePipelineStateCachedblobadaptermismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_CACHEDBLOBADAPTERMISMATCH,
    CreatePipelineStateCachedblobdriverversionmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_CACHEDBLOBDRIVERVERSIONMISMATCH,
    CreatePipelineStateCachedblobdescmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_CACHEDBLOBDESCMISMATCH,
    CreatePipelineStateCachedblobignored =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_CACHEDBLOBIGNORED,
    WriteToSubresourceInvalidheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITETOSUBRESOURCE_INVALIDHEAP,
    WriteToSubresourceInvalidresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITETOSUBRESOURCE_INVALIDRESOURCE,
    WriteToSubresourceInvalidbox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITETOSUBRESOURCE_INVALIDBOX,
    WriteToSubresourceInvalidSubresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITETOSUBRESOURCE_INVALIDSUBRESOURCE,
    WriteToSubresourceEmptybox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITETOSUBRESOURCE_EMPTYBOX,
    ReadFromSubresourceInvalidheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_READFROMSUBRESOURCE_INVALIDHEAP,
    ReadFromSubresourceInvalidresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_READFROMSUBRESOURCE_INVALIDRESOURCE,
    ReadFromSubresourceInvalidbox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_READFROMSUBRESOURCE_INVALIDBOX,
    ReadFromSubresourceInvalidSubresource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_READFROMSUBRESOURCE_INVALIDSUBRESOURCE,
    ReadFromSubresourceEmptybox =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_READFROMSUBRESOURCE_EMPTYBOX,
    TooManyNodesSpecified =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_TOO_MANY_NODES_SPECIFIED,
    InvalidNodeIndex = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_NODE_INDEX,
    GetheappropertiesInvalidresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETHEAPPROPERTIES_INVALIDRESOURCE,
    NodeMaskMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NODE_MASK_MISMATCH,
    CommandListOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_OUTOFMEMORY,
    CommandListMultipleSwapchainBufferReferences = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_MULTIPLE_SWAPCHAIN_BUFFER_REFERENCES,
    CommandListTooManySwapchainReferences = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_TOO_MANY_SWAPCHAIN_REFERENCES,
    CommandQueueTooManySwapchainReferences = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_QUEUE_TOO_MANY_SWAPCHAIN_REFERENCES,
    ExecuteCommandListsWrongswapchainbufferreference = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_WRONGSWAPCHAINBUFFERREFERENCE,
    CommandListSetrendertargetsInvalidnumrendertargets = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_SETRENDERTARGETS_INVALIDNUMRENDERTARGETS,
    CreateQueueInvalidType =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_QUEUE_INVALID_TYPE,
    CreateQueueInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_QUEUE_INVALID_FLAGS,
    CreateSharedResourceInvalidflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHAREDRESOURCE_INVALIDFLAGS,
    CreateSharedResourceInvalidformat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHAREDRESOURCE_INVALIDFORMAT,
    CreateSharedHeapInvalidflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHAREDHEAP_INVALIDFLAGS,
    ReflectsharedpropertiesUnrecognizedproperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_REFLECTSHAREDPROPERTIES_UNRECOGNIZEDPROPERTIES,
    ReflectsharedpropertiesInvalidsize =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_REFLECTSHAREDPROPERTIES_INVALIDSIZE,
    ReflectsharedpropertiesInvalidobject =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_REFLECTSHAREDPROPERTIES_INVALIDOBJECT,
    KeyedmutexInvalidobject =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_KEYEDMUTEX_INVALIDOBJECT,
    KeyedmutexInvalidkey =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_KEYEDMUTEX_INVALIDKEY,
    KeyedmutexWrongstate =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_KEYEDMUTEX_WRONGSTATE,
    CreateQueueInvalidPriority =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_QUEUE_INVALID_PRIORITY,
    ObjectDeletedWhileStillInUse =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OBJECT_DELETED_WHILE_STILL_IN_USE,
    CreatePipelineStateInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_INVALID_FLAGS,
    HeapAddressRangeHasNoResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_HEAP_ADDRESS_RANGE_HAS_NO_RESOURCE,
    CommandListDrawRenderTargetDeleted = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DRAW_RENDER_TARGET_DELETED,
    CreateGraphicsPipelineStateAllRenderTargetsHaveUnknownFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_ALL_RENDER_TARGETS_HAVE_UNKNOWN_FORMAT,
    HeapAddressRangeIntersectsMultipleBuffers = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_HEAP_ADDRESS_RANGE_INTERSECTS_MULTIPLE_BUFFERS,
    ExecuteCommandListsGpuWrittenReadbackResourceMapped = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTECOMMANDLISTS_GPU_WRITTEN_READBACK_RESOURCE_MAPPED,
    UnmapRangeNotEmpty =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_RANGE_NOT_EMPTY,
    MapInvalidNullrange =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MAP_INVALID_NULLRANGE,
    UnmapInvalidNullrange =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNMAP_INVALID_NULLRANGE,
    NoGraphicsApiSupport =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NO_GRAPHICS_API_SUPPORT,
    NoComputeApiSupport =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NO_COMPUTE_API_SUPPORT,
    ResolveSubresourceResourceFlagsNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_RESOURCE_FLAGS_NOT_SUPPORTED,
    GpuBasedValidationRootArgumentUninitialized = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_ROOT_ARGUMENT_UNINITIALIZED,
    GpuBasedValidationDescriptorHeapIndexOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_DESCRIPTOR_HEAP_INDEX_OUT_OF_BOUNDS,
    GpuBasedValidationDescriptorTableRegisterIndexOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_DESCRIPTOR_TABLE_REGISTER_INDEX_OUT_OF_BOUNDS,
    GpuBasedValidationDescriptorUninitialized = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_DESCRIPTOR_UNINITIALIZED,
    GpuBasedValidationDescriptorTypeMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_DESCRIPTOR_TYPE_MISMATCH,
    GpuBasedValidationSrvResourceDimensionMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_SRV_RESOURCE_DIMENSION_MISMATCH,
    GpuBasedValidationUavResourceDimensionMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_UAV_RESOURCE_DIMENSION_MISMATCH,
    GpuBasedValidationIncompatibleResourceState = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_INCOMPATIBLE_RESOURCE_STATE,
    CopyresourceNulldst =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_NULLDST,
    CopyresourceInvaliddstresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_INVALIDDSTRESOURCE,
    CopyresourceNullsrc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_NULLSRC,
    CopyresourceInvalidsrcresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_INVALIDSRCRESOURCE,
    ResolveSubresourceNulldst =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_NULLDST,
    ResolveSubresourceInvaliddstresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_INVALIDDSTRESOURCE,
    ResolveSubresourceNullsrc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_NULLSRC,
    ResolveSubresourceInvalidsrcresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_INVALIDSRCRESOURCE,
    PipelineStateTypeMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PIPELINE_STATE_TYPE_MISMATCH,
    CommandListDispatchRootSignatureNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DISPATCH_ROOT_SIGNATURE_NOT_SET,
    CommandListDispatchRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DISPATCH_ROOT_SIGNATURE_MISMATCH,
    ResourceBarrierZeroBarriers =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_ZERO_BARRIERS,
    BeginEndEventMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_BEGIN_END_EVENT_MISMATCH,
    ResourceBarrierPossibleBeforeAfterMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_POSSIBLE_BEFORE_AFTER_MISMATCH,
    ResourceBarrierMismatchingBeginEnd = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_MISMATCHING_BEGIN_END,
    GpuBasedValidationInvalidResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_INVALID_RESOURCE,
    UseOfZeroRefcountObject =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_USE_OF_ZERO_REFCOUNT_OBJECT,
    ObjectEvictedWhileStillInUse =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OBJECT_EVICTED_WHILE_STILL_IN_USE,
    GpuBasedValidationRootDescriptorAccessOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_ROOT_DESCRIPTOR_ACCESS_OUT_OF_BOUNDS,
    CreatepipelinelibraryInvalidlibraryblob = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINELIBRARY_INVALIDLIBRARYBLOB,
    CreatepipelinelibraryDriverversionmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINELIBRARY_DRIVERVERSIONMISMATCH,
    CreatepipelinelibraryAdapterversionmismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINELIBRARY_ADAPTERVERSIONMISMATCH,
    CreatepipelinelibraryUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINELIBRARY_UNSUPPORTED,
    CreatePipelinelibrary =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_PIPELINELIBRARY,
    LivePipelinelibrary =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_PIPELINELIBRARY,
    DestroyPipelinelibrary =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_PIPELINELIBRARY,
    StorepipelineNoname =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_STOREPIPELINE_NONAME,
    StorepipelineDuplicatename =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_STOREPIPELINE_DUPLICATENAME,
    LoadpipelineNamenotfound =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LOADPIPELINE_NAMENOTFOUND,
    LoadpipelineInvaliddesc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LOADPIPELINE_INVALIDDESC,
    PipelinelibrarySerializeNotenoughmemory = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PIPELINELIBRARY_SERIALIZE_NOTENOUGHMEMORY,
    CreateGraphicsPipelineStatePsOutputRtOutputMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_PS_OUTPUT_RT_OUTPUT_MISMATCH,
    SeteventonmultiplefencecompletionInvalidflags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETEVENTONMULTIPLEFENCECOMPLETION_INVALIDFLAGS,
    CreateQueueVideoNotSupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_QUEUE_VIDEO_NOT_SUPPORTED,
    CreateCommandAllocatorVideoNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_ALLOCATOR_VIDEO_NOT_SUPPORTED,
    CreatequeryHeapVideoDecodeStatisticsNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEQUERY_HEAP_VIDEO_DECODE_STATISTICS_NOT_SUPPORTED,
    CreateVideodecodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEODECODECOMMANDLIST,
    CreateVideodecoder = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEODECODER,
    CreateVideodecodestream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEODECODESTREAM,
    LiveVideodecodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEODECODECOMMANDLIST,
    LiveVideodecoder = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEODECODER,
    LiveVideodecodestream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEODECODESTREAM,
    DestroyVideodecodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEODECODECOMMANDLIST,
    DestroyVideodecoder =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEODECODER,
    DestroyVideodecodestream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEODECODESTREAM,
    DecodeFrameInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DECODE_FRAME_INVALID_PARAMETERS,
    DeprecatedApi = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEPRECATED_API,
    ResourceBarrierMismatchingCommandListType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_MISMATCHING_COMMAND_LIST_TYPE,
    CommandListDescriptorTableNotSet =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_DESCRIPTOR_TABLE_NOT_SET,
    CommandListRootConstantBufferViewNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_ROOT_CONSTANT_BUFFER_VIEW_NOT_SET,
    CommandListRootShaderResourceViewNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_ROOT_SHADER_RESOURCE_VIEW_NOT_SET,
    CommandListRootUnorderedAccessViewNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_ROOT_UNORDERED_ACCESS_VIEW_NOT_SET,
    DiscardInvalidSubresourceRange =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DISCARD_INVALID_SUBRESOURCE_RANGE,
    DiscardOneSubresourceForMipsWithRects = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DISCARD_ONE_SUBRESOURCE_FOR_MIPS_WITH_RECTS,
    DiscardNoRectsForNonTexture2D =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DISCARD_NO_RECTS_FOR_NON_TEXTURE2D,
    CopyOnSameSubresource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPY_ON_SAME_SUBRESOURCE,
    SetresidencypriorityInvalidPageable =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETRESIDENCYPRIORITY_INVALID_PAGEABLE,
    GpuBasedValidationUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_UNSUPPORTED,
    StaticDescriptorInvalidDescriptorChange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_STATIC_DESCRIPTOR_INVALID_DESCRIPTOR_CHANGE,
    DataStaticDescriptorInvalidDataChange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DATA_STATIC_DESCRIPTOR_INVALID_DATA_CHANGE,
    DataStaticWhileSetAtExecuteDescriptorInvalidDataChange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DATA_STATIC_WHILE_SET_AT_EXECUTE_DESCRIPTOR_INVALID_DATA_CHANGE,
    ExecuteBundleStaticDescriptorDataStaticNotSet = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EXECUTE_BUNDLE_STATIC_DESCRIPTOR_DATA_STATIC_NOT_SET,
    GpuBasedValidationResourceAccessOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_RESOURCE_ACCESS_OUT_OF_BOUNDS,
    GpuBasedValidationSamplerModeMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_SAMPLER_MODE_MISMATCH,
    CreateFenceInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_FENCE_INVALID_FLAGS,
    ResourceBarrierDuplicateSubresourceTransitions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_BARRIER_DUPLICATE_SUBRESOURCE_TRANSITIONS,
    SetresidencypriorityInvalidPriority =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETRESIDENCYPRIORITY_INVALID_PRIORITY,
    CreateDescriptorHeapLargeNumDescriptors = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_DESCRIPTOR_HEAP_LARGE_NUM_DESCRIPTORS,
    BeginEvent = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_BEGIN_EVENT,
    EndEvent = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_END_EVENT,
    CreatedeviceDebugLayerStartupOptions = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEVICE_DEBUG_LAYER_STARTUP_OPTIONS,
    CreatedepthstencilstateDepthboundstestUnsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEDEPTHSTENCILSTATE_DEPTHBOUNDSTEST_UNSUPPORTED,
    CreatePipelineStateDuplicateSubobject = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_DUPLICATE_SUBOBJECT,
    CreatePipelineStateUnknownSubobject =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_UNKNOWN_SUBOBJECT,
    CreatePipelineStateZeroSizeStream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_ZERO_SIZE_STREAM,
    CreatePipelineStateInvalidStream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_INVALID_STREAM,
    CreatePipelineStateCannotDeduceType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_CANNOT_DEDUCE_TYPE,
    CommandListStaticDescriptorResourceDimensionMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_STATIC_DESCRIPTOR_RESOURCE_DIMENSION_MISMATCH,
    CreateCommandQueueInsufficientPrivilegeForGlobalRealtime = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_QUEUE_INSUFFICIENT_PRIVILEGE_FOR_GLOBAL_REALTIME,
    CreateCommandQueueInsufficientHardwareSupportForGlobalRealtime = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_QUEUE_INSUFFICIENT_HARDWARE_SUPPORT_FOR_GLOBAL_REALTIME,
    AtomiccopybufferInvalidArchitecture =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_ARCHITECTURE,
    AtomiccopybufferNullDst =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_NULL_DST,
    AtomiccopybufferInvalidDstResourceDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_DST_RESOURCE_DIMENSION,
    AtomiccopybufferDstRangeOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_DST_RANGE_OUT_OF_BOUNDS,
    AtomiccopybufferNullSrc =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_NULL_SRC,
    AtomiccopybufferInvalidSrcResourceDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_SRC_RESOURCE_DIMENSION,
    AtomiccopybufferSrcRangeOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_SRC_RANGE_OUT_OF_BOUNDS,
    AtomiccopybufferInvalidOffsetAlignment = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_OFFSET_ALIGNMENT,
    AtomiccopybufferNullDependentResources = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_NULL_DEPENDENT_RESOURCES,
    AtomiccopybufferNullDependentSubresourceRanges = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_NULL_DEPENDENT_SUBRESOURCE_RANGES,
    AtomiccopybufferInvalidDependentResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_DEPENDENT_RESOURCE,
    AtomiccopybufferInvalidDependentSubresourceRange = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_DEPENDENT_SUBRESOURCE_RANGE,
    AtomiccopybufferDependentSubresourceOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_DEPENDENT_SUBRESOURCE_OUT_OF_BOUNDS,
    AtomiccopybufferDependentRangeOutOfBounds = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_DEPENDENT_RANGE_OUT_OF_BOUNDS,
    AtomiccopybufferZeroDependencies =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_ZERO_DEPENDENCIES,
    DeviceCreateSharedHandleInvalidarg = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DEVICE_CREATE_SHARED_HANDLE_INVALIDARG,
    DescriptorHandleWithInvalidResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESCRIPTOR_HANDLE_WITH_INVALID_RESOURCE,
    SetdepthboundsInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETDEPTHBOUNDS_INVALIDARGS,
    GpuBasedValidationResourceStateImprecise = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GPU_BASED_VALIDATION_RESOURCE_STATE_IMPRECISE,
    CommandListPipelineStateNotSet =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_LIST_PIPELINE_STATE_NOT_SET,
    CreateGraphicsPipelineStateShaderModelMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_SHADER_MODEL_MISMATCH,
    ObjectAccessedWhileStillInUse =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OBJECT_ACCESSED_WHILE_STILL_IN_USE,
    ProgrammableMsaaUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PROGRAMMABLE_MSAA_UNSUPPORTED,
    SetsamplepositionsInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETSAMPLEPOSITIONS_INVALIDARGS,
    ResolveSubresourceregionInvalidRect =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCEREGION_INVALID_RECT,
    CreateVideodecodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEODECODECOMMANDQUEUE,
    CreateVideoprocessCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOPROCESSCOMMANDLIST,
    CreateVideoprocesscommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOPROCESSCOMMANDQUEUE,
    LiveVideodecodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEODECODECOMMANDQUEUE,
    LiveVideoprocessCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOPROCESSCOMMANDLIST,
    LiveVideoprocesscommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOPROCESSCOMMANDQUEUE,
    DestroyVideodecodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEODECODECOMMANDQUEUE,
    DestroyVideoprocessCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOPROCESSCOMMANDLIST,
    DestroyVideoprocesscommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOPROCESSCOMMANDQUEUE,
    CreateVideoprocessor =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOPROCESSOR,
    CreateVideoprocessstream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOPROCESSSTREAM,
    LiveVideoprocessor = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOPROCESSOR,
    LiveVideoprocessstream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOPROCESSSTREAM,
    DestroyVideoprocessor =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOPROCESSOR,
    DestroyVideoprocessstream =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOPROCESSSTREAM,
    ProcessFrameInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PROCESS_FRAME_INVALID_PARAMETERS,
    CopyInvalidlayout = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPY_INVALIDLAYOUT,
    CreateCryptoSession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_CRYPTO_SESSION,
    CreateCryptoSessionPolicy =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_CRYPTO_SESSION_POLICY,
    CreateProtectedResourceSession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_PROTECTED_RESOURCE_SESSION,
    LiveCryptoSession = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_CRYPTO_SESSION,
    LiveCryptoSessionPolicy =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_CRYPTO_SESSION_POLICY,
    LiveProtectedResourceSession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_PROTECTED_RESOURCE_SESSION,
    DestroyCryptoSession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_CRYPTO_SESSION,
    DestroyCryptoSessionPolicy =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_CRYPTO_SESSION_POLICY,
    DestroyProtectedResourceSession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_PROTECTED_RESOURCE_SESSION,
    ProtectedResourceSessionUnsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_PROTECTED_RESOURCE_SESSION_UNSUPPORTED,
    FenceInvalidoperation =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_FENCE_INVALIDOPERATION,
    CreatequeryHeapCopyQueueTimestampsNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEQUERY_HEAP_COPY_QUEUE_TIMESTAMPS_NOT_SUPPORTED,
    SamplepositionsMismatchDeferred =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLEPOSITIONS_MISMATCH_DEFERRED,
    SamplepositionsMismatchRecordtimeAssumedfromfirstuse = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLEPOSITIONS_MISMATCH_RECORDTIME_ASSUMEDFROMFIRSTUSE,
    SamplepositionsMismatchRecordtimeAssumedfromclear = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLEPOSITIONS_MISMATCH_RECORDTIME_ASSUMEDFROMCLEAR,
    CreateVideodecoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEODECODERHEAP,
    LiveVideodecoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEODECODERHEAP,
    DestroyVideodecoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEODECODERHEAP,
    OpenexistingheapInvalidargReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OPENEXISTINGHEAP_INVALIDARG_RETURN,
    OpenexistingheapOutOfMemoryReturn =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OPENEXISTINGHEAP_OUTOFMEMORY_RETURN,
    OpenexistingheapInvalidaddress =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OPENEXISTINGHEAP_INVALIDADDRESS,
    OpenexistingheapInvalidhandle =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OPENEXISTINGHEAP_INVALIDHANDLE,
    WritebufferimmediateInvalidDest =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITEBUFFERIMMEDIATE_INVALID_DEST,
    WritebufferimmediateInvalidMode =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITEBUFFERIMMEDIATE_INVALID_MODE,
    WritebufferimmediateInvalidAlignment = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITEBUFFERIMMEDIATE_INVALID_ALIGNMENT,
    WritebufferimmediateNotSupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITEBUFFERIMMEDIATE_NOT_SUPPORTED,
    SetviewinstancemaskInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETVIEWINSTANCEMASK_INVALIDARGS,
    ViewInstancingUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIEW_INSTANCING_UNSUPPORTED,
    ViewInstancingInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIEW_INSTANCING_INVALIDARGS,
    CopyTextureRegionMismatchDecodeReferenceOnlyFlag = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_MISMATCH_DECODE_REFERENCE_ONLY_FLAG,
    CopyresourceMismatchDecodeReferenceOnlyFlag = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_MISMATCH_DECODE_REFERENCE_ONLY_FLAG,
    CreateVideoDecodeHeapCapsFailure =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_DECODE_HEAP_CAPS_FAILURE,
    CreateVideoDecodeHeapCapsUnsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_DECODE_HEAP_CAPS_UNSUPPORTED,
    VideoDecodeSupportInvalidInput =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_DECODE_SUPPORT_INVALID_INPUT,
    CreateVideoDecoderUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_DECODER_UNSUPPORTED,
    CreateGraphicsPipelineStateMetadataError = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_METADATA_ERROR,
    CreateGraphicsPipelineStateViewInstancingVertexSizeExceeded = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_VIEW_INSTANCING_VERTEX_SIZE_EXCEEDED,
    CreateGraphicsPipelineStateRuntimeInternalError = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_RUNTIME_INTERNAL_ERROR,
    NoVideoApiSupport = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NO_VIDEO_API_SUPPORT,
    VideoProcessSupportInvalidInput =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_PROCESS_SUPPORT_INVALID_INPUT,
    CreateVideoProcessorCapsFailure =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_PROCESSOR_CAPS_FAILURE,
    VideoProcessSupportUnsupportedFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_PROCESS_SUPPORT_UNSUPPORTED_FORMAT,
    VideoDecodeFrameInvalidArgument =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_DECODE_FRAME_INVALID_ARGUMENT,
    EnqueueMakeResidentInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ENQUEUE_MAKE_RESIDENT_INVALID_FLAGS,
    OpenexistingheapUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OPENEXISTINGHEAP_UNSUPPORTED,
    VideoProcessFramesInvalidArgument =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_PROCESS_FRAMES_INVALID_ARGUMENT,
    VideoDecodeSupportUnsupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_DECODE_SUPPORT_UNSUPPORTED,
    CreateCommandrecorder =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDRECORDER,
    LiveCommandrecorder =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDRECORDER,
    DestroyCommandrecorder =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDRECORDER,
    CreateCommandRecorderVideoNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_RECORDER_VIDEO_NOT_SUPPORTED,
    CreateCommandRecorderInvalidSupportFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_RECORDER_INVALID_SUPPORT_FLAGS,
    CreateCommandRecorderInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_RECORDER_INVALID_FLAGS,
    CreateCommandRecorderMoreRecordersThanLogicalProcessors = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_RECORDER_MORE_RECORDERS_THAN_LOGICAL_PROCESSORS,
    CreateCommandpool = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMANDPOOL,
    LiveCommandpool = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_COMMANDPOOL,
    DestroyCommandpool = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_COMMANDPOOL,
    CreateCommandPoolInvalidFlags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_POOL_INVALID_FLAGS,
    CreateCommandListVideoNotSupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_LIST_VIDEO_NOT_SUPPORTED,
    CommandRecorderSupportFlagsMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_RECORDER_SUPPORT_FLAGS_MISMATCH,
    CommandRecorderContention =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_RECORDER_CONTENTION,
    CommandRecorderUsageWithCreateCommandListCommandList = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_RECORDER_USAGE_WITH_CREATECOMMANDLIST_COMMAND_LIST,
    CommandAllocatorUsageWithCreateCommandList1CommandList = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_ALLOCATOR_USAGE_WITH_CREATECOMMANDLIST1_COMMAND_LIST,
    CannotExecuteEmptyCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_EXECUTE_EMPTY_COMMAND_LIST,
    CannotResetCommandPoolWithOpenCommandLists = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_RESET_COMMAND_POOL_WITH_OPEN_COMMAND_LISTS,
    CannotUseCommandRecorderWithoutCurrentTarget = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_USE_COMMAND_RECORDER_WITHOUT_CURRENT_TARGET,
    CannotChangeCommandRecorderTargetWhileRecording = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_CHANGE_COMMAND_RECORDER_TARGET_WHILE_RECORDING,
    CommandPoolSync = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMMAND_POOL_SYNC,
    EvictUnderflow = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EVICT_UNDERFLOW,
    CreateMetaCommand = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_META_COMMAND,
    LiveMetaCommand = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_META_COMMAND,
    DestroyMetaCommand = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_META_COMMAND,
    CopyBufferRegionInvalidDstResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_INVALID_DST_RESOURCE,
    CopyBufferRegionInvalidSrcResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYBUFFERREGION_INVALID_SRC_RESOURCE,
    AtomiccopybufferInvalidDstResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_DST_RESOURCE,
    AtomiccopybufferInvalidSrcResource =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ATOMICCOPYBUFFER_INVALID_SRC_RESOURCE,
    CreateplacedresourceonbufferNullBuffer = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_NULL_BUFFER,
    CreateplacedresourceonbufferNullResourceDesc = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_NULL_RESOURCE_DESC,
    CreateplacedresourceonbufferUnsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_UNSUPPORTED,
    CreateplacedresourceonbufferInvalidBufferDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_INVALID_BUFFER_DIMENSION,
    CreateplacedresourceonbufferInvalidBufferFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_INVALID_BUFFER_FLAGS,
    CreateplacedresourceonbufferInvalidBufferOffset = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_INVALID_BUFFER_OFFSET,
    CreateplacedresourceonbufferInvalidResourceDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_INVALID_RESOURCE_DIMENSION,
    CreateplacedresourceonbufferInvalidResourceFlags = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_INVALID_RESOURCE_FLAGS,
    CreateplacedresourceonbufferOutOfMemoryReturn = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPLACEDRESOURCEONBUFFER_OUTOFMEMORY_RETURN,
    CannotCreateGraphicsAndVideoCommandRecorder = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_CREATE_GRAPHICS_AND_VIDEO_COMMAND_RECORDER,
    UpdatetilemappingsPossiblyMismatchingProperties = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UPDATETILEMAPPINGS_POSSIBLY_MISMATCHING_PROPERTIES,
    CreateCommandListInvalidCommandListType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_LIST_INVALID_COMMAND_LIST_TYPE,
    ClearunorderedaccessviewIncompatibleWithStructuredBuffers = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEARUNORDEREDACCESSVIEW_INCOMPATIBLE_WITH_STRUCTURED_BUFFERS,
    ComputeOnlyDeviceOperationUnsupported = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COMPUTE_ONLY_DEVICE_OPERATION_UNSUPPORTED,
    BuildRaytracingAccelerationStructureInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_BUILD_RAYTRACING_ACCELERATION_STRUCTURE_INVALID,
    EmitRaytracingAccelerationStructurePostbuildInfoInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EMIT_RAYTRACING_ACCELERATION_STRUCTURE_POSTBUILD_INFO_INVALID,
    CopyRaytracingAccelerationStructureInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPY_RAYTRACING_ACCELERATION_STRUCTURE_INVALID,
    DispatchRaysInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DISPATCH_RAYS_INVALID,
    GetRaytracingAccelerationStructurePrebuildInfoInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GET_RAYTRACING_ACCELERATION_STRUCTURE_PREBUILD_INFO_INVALID,
    CreateLifetimetracker =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_LIFETIMETRACKER,
    LiveLifetimetracker =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_LIFETIMETRACKER,
    DestroyLifetimetracker =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_LIFETIMETRACKER,
    DestroyownedobjectObjectnotowned =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROYOWNEDOBJECT_OBJECTNOTOWNED,
    CreateTrackedworkload =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_TRACKEDWORKLOAD,
    LiveTrackedworkload =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_TRACKEDWORKLOAD,
    DestroyTrackedworkload =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_TRACKEDWORKLOAD,
    RenderPassError = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_ERROR,
    MetaCommandIdInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_META_COMMAND_ID_INVALID,
    MetaCommandUnsupportedParams =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_META_COMMAND_UNSUPPORTED_PARAMS,
    MetaCommandFailedEnumeration =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_META_COMMAND_FAILED_ENUMERATION,
    MetaCommandParameterSizeMismatch =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_META_COMMAND_PARAMETER_SIZE_MISMATCH,
    UninitializedMetaCommand =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_UNINITIALIZED_META_COMMAND,
    MetaCommandInvalidGpuVirtualAddress = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_META_COMMAND_INVALID_GPU_VIRTUAL_ADDRESS,
    CreateVideoencodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOENCODECOMMANDLIST,
    LiveVideoencodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOENCODECOMMANDLIST,
    DestroyVideoencodeCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOENCODECOMMANDLIST,
    CreateVideoencodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOENCODECOMMANDQUEUE,
    LiveVideoencodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOENCODECOMMANDQUEUE,
    DestroyVideoencodecommandqueue =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOENCODECOMMANDQUEUE,
    CreateVideomotionestimator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOMOTIONESTIMATOR,
    LiveVideomotionestimator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOMOTIONESTIMATOR,
    DestroyVideomotionestimator =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOMOTIONESTIMATOR,
    CreateVideomotionvectorheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOMOTIONVECTORHEAP,
    LiveVideomotionvectorheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOMOTIONVECTORHEAP,
    DestroyVideomotionvectorheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOMOTIONVECTORHEAP,
    MultipleTrackedWorkloads =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MULTIPLE_TRACKED_WORKLOADS,
    MultipleTrackedWorkloadPairs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MULTIPLE_TRACKED_WORKLOAD_PAIRS,
    OutOfOrderTrackedWorkloadPair =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OUT_OF_ORDER_TRACKED_WORKLOAD_PAIR,
    CannotAddTrackedWorkload =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CANNOT_ADD_TRACKED_WORKLOAD,
    IncompleteTrackedWorkloadPair =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INCOMPLETE_TRACKED_WORKLOAD_PAIR,
    CreateStateObjectError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_STATE_OBJECT_ERROR,
    GetShaderIdentifierError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GET_SHADER_IDENTIFIER_ERROR,
    GetShaderStackSizeError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GET_SHADER_STACK_SIZE_ERROR,
    GetPipelineStackSizeError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GET_PIPELINE_STACK_SIZE_ERROR,
    SetPipelineStackSizeError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_PIPELINE_STACK_SIZE_ERROR,
    GetShaderIdentifierSizeInvalid =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GET_SHADER_IDENTIFIER_SIZE_INVALID,
    CheckDriverMatchingIdentifierInvalid = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CHECK_DRIVER_MATCHING_IDENTIFIER_INVALID,
    CheckDriverMatchingIdentifierDriverReportedIssue = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CHECK_DRIVER_MATCHING_IDENTIFIER_DRIVER_REPORTED_ISSUE,
    RenderPassInvalidResourceBarrier =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_INVALID_RESOURCE_BARRIER,
    RenderPassDisallowedApiCalled =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_DISALLOWED_API_CALLED,
    RenderPassCannotNestRenderPasses =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_CANNOT_NEST_RENDER_PASSES,
    RenderPassCannotEndWithoutBegin =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_CANNOT_END_WITHOUT_BEGIN,
    RenderPassCannotCloseCommandList =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_CANNOT_CLOSE_COMMAND_LIST,
    RenderPassGpuWorkWhileSuspended =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_GPU_WORK_WHILE_SUSPENDED,
    RenderPassMismatchingSuspendResume = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_MISMATCHING_SUSPEND_RESUME,
    RenderPassNoPriorSuspendWithinExecuteCommandLists = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_NO_PRIOR_SUSPEND_WITHIN_EXECUTECOMMANDLISTS,
    RenderPassNoSubsequentResumeWithinExecuteCommandLists = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_NO_SUBSEQUENT_RESUME_WITHIN_EXECUTECOMMANDLISTS,
    TrackedWorkloadCommandQueueMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_TRACKED_WORKLOAD_COMMAND_QUEUE_MISMATCH,
    TrackedWorkloadNotSupported =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_TRACKED_WORKLOAD_NOT_SUPPORTED,
    RenderPassMismatchingNoAccess =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_MISMATCHING_NO_ACCESS,
    RenderPassUnsupportedResolve =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RENDER_PASS_UNSUPPORTED_RESOLVE,
    ClearunorderedaccessviewInvalidResourcePtr = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEARUNORDEREDACCESSVIEW_INVALID_RESOURCE_PTR,
    Windows7FenceOutoforderSignal =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WINDOWS7_FENCE_OUTOFORDER_SIGNAL,
    Windows7FenceOutoforderWait =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WINDOWS7_FENCE_OUTOFORDER_WAIT,
    VideoCreateMotionEstimatorInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_CREATE_MOTION_ESTIMATOR_INVALID_ARGUMENT,
    VideoCreateMotionVectorHeapInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_CREATE_MOTION_VECTOR_HEAP_INVALID_ARGUMENT,
    EstimateMotionInvalidArgument =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ESTIMATE_MOTION_INVALID_ARGUMENT,
    ResolveMotionVectorHeapInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVE_MOTION_VECTOR_HEAP_INVALID_ARGUMENT,
    GetgpuvirtualaddressInvalidHeapType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GETGPUVIRTUALADDRESS_INVALID_HEAP_TYPE,
    SetBackgroundProcessingModeInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SET_BACKGROUND_PROCESSING_MODE_INVALID_ARGUMENT,
    CreateCommandListInvalidCommandListTypeForFeatureLevel = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_COMMAND_LIST_INVALID_COMMAND_LIST_TYPE_FOR_FEATURE_LEVEL,
    CreateVideoextensioncommand =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOEXTENSIONCOMMAND,
    LiveVideoextensioncommand =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOEXTENSIONCOMMAND,
    DestroyVideoextensioncommand =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOEXTENSIONCOMMAND,
    InvalidVideoExtensionCommandId =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_INVALID_VIDEO_EXTENSION_COMMAND_ID,
    VideoExtensionCommandInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VIDEO_EXTENSION_COMMAND_INVALID_ARGUMENT,
    CreateRootSignatureNotUniqueInDxilLibrary = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_NOT_UNIQUE_IN_DXIL_LIBRARY,
    VariableShadingRateNotAllowedWithTir = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VARIABLE_SHADING_RATE_NOT_ALLOWED_WITH_TIR,
    GeometryShaderOutputtingBothViewportArrayIndexAndShadingRateNotSupportedOnDevice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_GEOMETRY_SHADER_OUTPUTTING_BOTH_VIEWPORT_ARRAY_INDEX_AND_SHADING_RATE_NOT_SUPPORTED_ON_DEVICE,
    RssetshadingRateInvalidShadingRate = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RSSETSHADING_RATE_INVALID_SHADING_RATE,
    RssetshadingRateShadingRateNotPermittedByCap = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RSSETSHADING_RATE_SHADING_RATE_NOT_PERMITTED_BY_CAP,
    RssetshadingRateInvalidCombiner =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RSSETSHADING_RATE_INVALID_COMBINER,
    RssetshadingrateimageRequiresTier2 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RSSETSHADINGRATEIMAGE_REQUIRES_TIER_2,
    RssetshadingrateRequiresTier1 =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RSSETSHADINGRATE_REQUIRES_TIER_1,
    ShadingRateImageIncorrectFormat =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADING_RATE_IMAGE_INCORRECT_FORMAT,
    ShadingRateImageIncorrectArraySize = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADING_RATE_IMAGE_INCORRECT_ARRAY_SIZE,
    ShadingRateImageIncorrectMipLevel = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADING_RATE_IMAGE_INCORRECT_MIP_LEVEL,
    ShadingRateImageIncorrectSampleCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADING_RATE_IMAGE_INCORRECT_SAMPLE_COUNT,
    ShadingRateImageIncorrectSampleQuality = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADING_RATE_IMAGE_INCORRECT_SAMPLE_QUALITY,
    NonRetailShaderModelWontValidate =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NON_RETAIL_SHADER_MODEL_WONT_VALIDATE,
    CreateGraphicsPipelineStateAsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_AS_ROOT_SIGNATURE_MISMATCH,
    CreateGraphicsPipelineStateMsRootSignatureMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_MS_ROOT_SIGNATURE_MISMATCH,
    AddToStateObjectError =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ADD_TO_STATE_OBJECT_ERROR,
    CreateProtectedResourceSessionInvalidArgument = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_PROTECTED_RESOURCE_SESSION_INVALID_ARGUMENT,
    CreateGraphicsPipelineStateMsPsoDescMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_MS_PSO_DESC_MISMATCH,
    CreatePipelineStateMsIncompleteType = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEPIPELINESTATE_MS_INCOMPLETE_TYPE,
    CreateGraphicsPipelineStateAsNotMsMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_AS_NOT_MS_MISMATCH,
    CreateGraphicsPipelineStateMsNotPsMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_MS_NOT_PS_MISMATCH,
    NonzeroSamplerFeedbackMipRegionWithIncompatibleFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_NONZERO_SAMPLER_FEEDBACK_MIP_REGION_WITH_INCOMPATIBLE_FORMAT,
    CreateGraphicsPipelineStateInputlayoutShaderMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEGRAPHICSPIPELINESTATE_INPUTLAYOUT_SHADER_MISMATCH,
    EmptyDispatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_EMPTY_DISPATCH,
    ResourceFormatRequiresSamplerFeedbackCapability = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOURCE_FORMAT_REQUIRES_SAMPLER_FEEDBACK_CAPABILITY,
    SamplerFeedbackMapInvalidMipRegion = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_INVALID_MIP_REGION,
    SamplerFeedbackMapInvalidDimension = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_INVALID_DIMENSION,
    SamplerFeedbackMapInvalidSampleCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_INVALID_SAMPLE_COUNT,
    SamplerFeedbackMapInvalidSampleQuality = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_INVALID_SAMPLE_QUALITY,
    SamplerFeedbackMapInvalidLayout =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_INVALID_LAYOUT,
    SamplerFeedbackMapRequiresUnorderedAccessFlag = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_MAP_REQUIRES_UNORDERED_ACCESS_FLAG,
    SamplerFeedbackCreateUavNullArguments = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_CREATE_UAV_NULL_ARGUMENTS,
    SamplerFeedbackUavRequiresSamplerFeedbackCapability = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_UAV_REQUIRES_SAMPLER_FEEDBACK_CAPABILITY,
    SamplerFeedbackCreateUavRequiresFeedbackMapFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_CREATE_UAV_REQUIRES_FEEDBACK_MAP_FORMAT,
    CreateMeshShaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADER_INVALIDSHADERBYTECODE,
    CreateMeshShaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADER_OUTOFMEMORY,
    CreateMeshShaderWithStreamOutputInvalidshadertype = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADERWITHSTREAMOUTPUT_INVALIDSHADERTYPE,
    ResolveSubresourceSamplerFeedbackTranscodeInvalidFormat = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_SAMPLER_FEEDBACK_TRANSCODE_INVALID_FORMAT,
    ResolveSubresourceSamplerFeedbackInvalidMipLevelCount = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_SAMPLER_FEEDBACK_INVALID_MIP_LEVEL_COUNT,
    ResolveSubresourceSamplerFeedbackTranscodeArraySizeMismatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVESUBRESOURCE_SAMPLER_FEEDBACK_TRANSCODE_ARRAY_SIZE_MISMATCH,
    SamplerFeedbackCreateUavMismatchingTargetedResource = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SAMPLER_FEEDBACK_CREATE_UAV_MISMATCHING_TARGETED_RESOURCE,
    CreateMeshShaderOutputexceedsmaxsize =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADER_OUTPUTEXCEEDSMAXSIZE,
    CreateMeshShaderGroupsharedexceedsmaxsize = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADER_GROUPSHAREDEXCEEDSMAXSIZE,
    VertexShaderOutputtingBothViewportArrayIndexAndShadingRateNotSupportedOnDevice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_VERTEX_SHADER_OUTPUTTING_BOTH_VIEWPORT_ARRAY_INDEX_AND_SHADING_RATE_NOT_SUPPORTED_ON_DEVICE,
    MeshShaderOutputtingBothViewportArrayIndexAndShadingRateNotSupportedOnDevice = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_MESH_SHADER_OUTPUTTING_BOTH_VIEWPORT_ARRAY_INDEX_AND_SHADING_RATE_NOT_SUPPORTED_ON_DEVICE,
    CreateMeshShaderMismatchedasmspayloadsize = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEMESHSHADER_MISMATCHEDASMSPAYLOADSIZE,
    CreateRootSignatureUnboundedStaticDescriptors = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_ROOT_SIGNATURE_UNBOUNDED_STATIC_DESCRIPTORS,
    CreateamplificationshaderInvalidShaderBytecode = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEAMPLIFICATIONSHADER_INVALIDSHADERBYTECODE,
    CreateamplificationshaderOutOfMemory =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEAMPLIFICATIONSHADER_OUTOFMEMORY,
    CreateShadercachesession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_SHADERCACHESESSION,
    LiveShadercachesession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_SHADERCACHESESSION,
    DestroyShadercachesession =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_SHADERCACHESESSION,
    CreateshadercachesessionInvalidargs =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERCACHESESSION_INVALIDARGS,
    CreateshadercachesessionDisabled =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERCACHESESSION_DISABLED,
    CreateshadercachesessionAlreadyopen =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATESHADERCACHESESSION_ALREADYOPEN,
    ShadercachecontrolDevelopermode =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHECONTROL_DEVELOPERMODE,
    ShadercachecontrolInvalidflags =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHECONTROL_INVALIDFLAGS,
    ShadercachecontrolStatealreadyset =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHECONTROL_STATEALREADYSET,
    ShadercachecontrolIgnoredflag =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHECONTROL_IGNOREDFLAG,
    ShadercachesessionStorevalueAlreadypresent = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_STOREVALUE_ALREADYPRESENT,
    ShadercachesessionStorevalueHashcollision = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_STOREVALUE_HASHCOLLISION,
    ShadercachesessionStorevalueCachefull = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_STOREVALUE_CACHEFULL,
    ShadercachesessionFindvalueNotfound =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_FINDVALUE_NOTFOUND,
    ShadercachesessionCorrupt =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_CORRUPT,
    ShadercachesessionDisabled =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SHADERCACHESESSION_DISABLED,
    OversizedDispatch = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_OVERSIZED_DISPATCH,
    CreateVideoEncoder = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOENCODER,
    LiveVideoEncoder = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOENCODER,
    DestroyVideoEncoder =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOENCODER,
    CreateVideoEncoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEOENCODERHEAP,
    LiveVideoEncoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_LIVE_VIDEOENCODERHEAP,
    DestroyVideoEncoderheap =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESTROY_VIDEOENCODERHEAP,
    CopyTextureRegionMismatchEncodeReferenceOnlyFlag = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYTEXTUREREGION_MISMATCH_ENCODE_REFERENCE_ONLY_FLAG,
    CopyresourceMismatchEncodeReferenceOnlyFlag = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_COPYRESOURCE_MISMATCH_ENCODE_REFERENCE_ONLY_FLAG,
    EncodeFrameInvalidParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ENCODE_FRAME_INVALID_PARAMETERS,
    EncodeFrameUnsupportedParameters =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_ENCODE_FRAME_UNSUPPORTED_PARAMETERS,
    ResolveEncoderOutputMetadataInvalidParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVE_ENCODER_OUTPUT_METADATA_INVALID_PARAMETERS,
    ResolveEncoderOutputMetadataUnsupportedParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVE_ENCODER_OUTPUT_METADATA_UNSUPPORTED_PARAMETERS,
    CreateVideoEncoderInvalidParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_ENCODER_INVALID_PARAMETERS,
    CreateVideoEncoderUnsupportedParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_ENCODER_UNSUPPORTED_PARAMETERS,
    CreateVideoEncoderHeapInvalidParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_ENCODER_HEAP_INVALID_PARAMETERS,
    CreateVideoEncoderHeapUnsupportedParameters = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATE_VIDEO_ENCODER_HEAP_UNSUPPORTED_PARAMETERS,
    CreateCommandListNullCommandallocator = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATECOMMANDLIST_NULL_COMMANDALLOCATOR,
    ClearUnorderedAccessViewInvalidDescriptorHandle = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CLEAR_UNORDERED_ACCESS_VIEW_INVALID_DESCRIPTOR_HANDLE,
    DescriptorHeapNotShaderVisible =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_DESCRIPTOR_HEAP_NOT_SHADER_VISIBLE,
    CreateblendstateBlendopWarning =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_BLENDOP_WARNING,
    CreateblendstateBlendopalphaWarning =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_CREATEBLENDSTATE_BLENDOPALPHA_WARNING,
    WriteCombinePerformanceWarning =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_WRITE_COMBINE_PERFORMANCE_WARNING,
    ResolveQueryInvalidQueryState =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_RESOLVE_QUERY_INVALID_QUERY_STATE,
    SetPrivateDataNoAccess =
        D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_SETPRIVATEDATA_NO_ACCESS,
    D3D12MessagesEnd = D3D12_MESSAGE_ID_D3D12_MESSAGE_ID_D3D12_MESSAGES_END,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum MessageCallbackFlags {
    FlagNone = D3D12_MESSAGE_CALLBACK_FLAGS_D3D12_MESSAGE_CALLBACK_FLAG_NONE,
    IgnoreFilters =
        D3D12_MESSAGE_CALLBACK_FLAGS_D3D12_MESSAGE_CALLBACK_IGNORE_FILTERS,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum QueryHeapType {
    Occlusion = D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_OCCLUSION,
    Timestamp = D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_TIMESTAMP,
    PipelineStatistics =
        D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_PIPELINE_STATISTICS,
    SoStatistics = D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_SO_STATISTICS,
    VideoDecodeStatistics =
        D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_VIDEO_DECODE_STATISTICS,
    CopyQueueTimestamp =
        D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_COPY_QUEUE_TIMESTAMP,
    PipelineStatistics1 =
        D3D12_QUERY_HEAP_TYPE_D3D12_QUERY_HEAP_TYPE_PIPELINE_STATISTICS1,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum ShaderMinPrecisionSupport {
    None = D3D12_SHADER_MIN_PRECISION_SUPPORT_D3D12_SHADER_MIN_PRECISION_SUPPORT_NONE,
    P10Bit = D3D12_SHADER_MIN_PRECISION_SUPPORT_D3D12_SHADER_MIN_PRECISION_SUPPORT_10_BIT,
    P16Bit = D3D12_SHADER_MIN_PRECISION_SUPPORT_D3D12_SHADER_MIN_PRECISION_SUPPORT_16_BIT,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum TiledResourcesTier {
    NotSupported =
        D3D12_TILED_RESOURCES_TIER_D3D12_TILED_RESOURCES_TIER_NOT_SUPPORTED,
    Tier1 = D3D12_TILED_RESOURCES_TIER_D3D12_TILED_RESOURCES_TIER_1,
    Tier2 = D3D12_TILED_RESOURCES_TIER_D3D12_TILED_RESOURCES_TIER_2,
    Tier3 = D3D12_TILED_RESOURCES_TIER_D3D12_TILED_RESOURCES_TIER_3,
    Tier4 = D3D12_TILED_RESOURCES_TIER_D3D12_TILED_RESOURCES_TIER_4,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum ResourceBindingTier {
    Tier1 = D3D12_RESOURCE_BINDING_TIER_D3D12_RESOURCE_BINDING_TIER_1,
    Tier2 = D3D12_RESOURCE_BINDING_TIER_D3D12_RESOURCE_BINDING_TIER_2,
    Tier3 = D3D12_RESOURCE_BINDING_TIER_D3D12_RESOURCE_BINDING_TIER_3,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum ConservativeRasterizationTier {
    NotSupported = D3D12_CONSERVATIVE_RASTERIZATION_TIER_D3D12_CONSERVATIVE_RASTERIZATION_TIER_NOT_SUPPORTED,
    Tier1 = D3D12_CONSERVATIVE_RASTERIZATION_TIER_D3D12_CONSERVATIVE_RASTERIZATION_TIER_1,
    Tier2 = D3D12_CONSERVATIVE_RASTERIZATION_TIER_D3D12_CONSERVATIVE_RASTERIZATION_TIER_2,
    Tier3 = D3D12_CONSERVATIVE_RASTERIZATION_TIER_D3D12_CONSERVATIVE_RASTERIZATION_TIER_3,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum CrossNodeSharingTier {
    NotSupported = D3D12_CROSS_NODE_SHARING_TIER_D3D12_CROSS_NODE_SHARING_TIER_NOT_SUPPORTED,
    Tier1Emulated =
        D3D12_CROSS_NODE_SHARING_TIER_D3D12_CROSS_NODE_SHARING_TIER_1_EMULATED,
    Tier1 = D3D12_CROSS_NODE_SHARING_TIER_D3D12_CROSS_NODE_SHARING_TIER_1,
    Tier2 = D3D12_CROSS_NODE_SHARING_TIER_D3D12_CROSS_NODE_SHARING_TIER_2,
    Tier3 = D3D12_CROSS_NODE_SHARING_TIER_D3D12_CROSS_NODE_SHARING_TIER_3,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum ResourceHeapTier {
    Tier1 = D3D12_RESOURCE_HEAP_TIER_D3D12_RESOURCE_HEAP_TIER_1,
    Tier2 = D3D12_RESOURCE_HEAP_TIER_D3D12_RESOURCE_HEAP_TIER_2,
}

bitflags! {
    pub struct Usage: u32 {
        const ShaderInput = DXGI_USAGE_SHADER_INPUT;
        const RenderTargetOutput =
        DXGI_USAGE_RENDER_TARGET_OUTPUT;
        const BackBuffer =
        DXGI_USAGE_BACK_BUFFER;
        const Shared =
        DXGI_USAGE_SHARED;
        const ReadOnly = DXGI_USAGE_READ_ONLY;
        const DiscardOnPresent = DXGI_USAGE_DISCARD_ON_PRESENT;
        const UnorderedAccess = DXGI_USAGE_UNORDERED_ACCESS;
    }
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum Scaling {
    Stretch = DXGI_SCALING_DXGI_SCALING_STRETCH,
    None = DXGI_SCALING_DXGI_SCALING_NONE,
    AspectRatioStretch = DXGI_SCALING_DXGI_SCALING_ASPECT_RATIO_STRETCH,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum SwapEffect {
    Discard = DXGI_SWAP_EFFECT_DXGI_SWAP_EFFECT_DISCARD,
    Sequential = DXGI_SWAP_EFFECT_DXGI_SWAP_EFFECT_SEQUENTIAL,
    FlipSequential = DXGI_SWAP_EFFECT_DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL,
    FlipDiscard = DXGI_SWAP_EFFECT_DXGI_SWAP_EFFECT_FLIP_DISCARD,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum AlphaMode {
    Unspecified = DXGI_ALPHA_MODE_DXGI_ALPHA_MODE_UNSPECIFIED,
    Premultiplied = DXGI_ALPHA_MODE_DXGI_ALPHA_MODE_PREMULTIPLIED,
    Straight = DXGI_ALPHA_MODE_DXGI_ALPHA_MODE_STRAIGHT,
    Ignore = DXGI_ALPHA_MODE_DXGI_ALPHA_MODE_IGNORE,
    ForceDword = DXGI_ALPHA_MODE_DXGI_ALPHA_MODE_FORCE_DWORD,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum AdapterFlag {
    None = DXGI_ADAPTER_FLAG_DXGI_ADAPTER_FLAG_NONE,
    Remote = DXGI_ADAPTER_FLAG_DXGI_ADAPTER_FLAG_REMOTE,
    Software = DXGI_ADAPTER_FLAG_DXGI_ADAPTER_FLAG_SOFTWARE,
    ForceDword = DXGI_ADAPTER_FLAG_DXGI_ADAPTER_FLAG_FORCE_DWORD,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum QueryType {
    Occlusion = D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_OCCLUSION,
    BinaryOcclusion = D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_BINARY_OCCLUSION,
    Timestamp = D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_TIMESTAMP,
    PipelineStatistics = D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_PIPELINE_STATISTICS,
    SoStatisticsStream0 =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_SO_STATISTICS_STREAM0,
    SoStatisticsStream1 =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_SO_STATISTICS_STREAM1,
    SoStatisticsStream2 =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_SO_STATISTICS_STREAM2,
    SoStatisticsStream3 =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_SO_STATISTICS_STREAM3,
    VideoDecodeStatistics =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_VIDEO_DECODE_STATISTICS,
    PipelineStatistics1 =
        D3D12_QUERY_TYPE_D3D12_QUERY_TYPE_PIPELINE_STATISTICS1,
}

bitflags! {
    pub struct SwapChainFlags: i32 {
        const NonPrerotated = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_NONPREROTATED;
        const AllowModeSwitch = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_ALLOW_MODE_SWITCH;
        const GdiCompatible = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_GDI_COMPATIBLE;
        const RestrictedContent = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_RESTRICTED_CONTENT;
        const RestrictSharedResourceDriver = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_RESTRICT_SHARED_RESOURCE_DRIVER;
        const DisplayOnly = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_DISPLAY_ONLY;
        const FrameLatencyWaitableObject = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_FRAME_LATENCY_WAITABLE_OBJECT;
        const ForegroundLayer = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_FOREGROUND_LAYER;
        const FullscreenVideo = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_FULLSCREEN_VIDEO;
        const YuvVideo = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_YUV_VIDEO;
        const HwProtected = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_HW_PROTECTED;
        const AllowTearing = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_ALLOW_TEARING;
        const RestrictedToAllHolographicDisplays = DXGI_SWAP_CHAIN_FLAG_DXGI_SWAP_CHAIN_FLAG_RESTRICTED_TO_ALL_HOLOGRAPHIC_DISPLAYS;
    }
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum BufferUavFlags {
    None = D3D12_BUFFER_UAV_FLAGS_D3D12_BUFFER_UAV_FLAG_NONE,
    Raw = D3D12_BUFFER_UAV_FLAGS_D3D12_BUFFER_UAV_FLAG_RAW,
}

#[derive(Copy, Clone)]
#[repr(i32)]
#[derive(Debug)]
pub enum UavDimension {
    Unknown = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_UNKNOWN,
    Buffer = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_BUFFER,
    Texture1D = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_TEXTURE1D,
    Texture1DArray = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_TEXTURE1DARRAY,
    Texture2D = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_TEXTURE2D,
    Texture2DArray = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_TEXTURE2DARRAY,
    Texture3D = D3D12_UAV_DIMENSION_D3D12_UAV_DIMENSION_TEXTURE3D,
}

bitflags! {
    pub struct PresentFlags: u32 {
        const None = 0;
        const Test = DXGI_PRESENT_TEST;
        const DoNotSequence = DXGI_PRESENT_DO_NOT_SEQUENCE;
        const Restart = DXGI_PRESENT_RESTART;
        const DoNotWait = DXGI_PRESENT_DO_NOT_WAIT;
        const StereoPreferRight = DXGI_PRESENT_STEREO_PREFER_RIGHT;
        const StereoTemporaryMono = DXGI_PRESENT_STEREO_TEMPORARY_MONO;
        const RestrictToOutput = DXGI_PRESENT_RESTRICT_TO_OUTPUT;
        const UseDuration = DXGI_PRESENT_USE_DURATION;
        const AllowTearing = DXGI_PRESENT_ALLOW_TEARING;
    }
}

bitflags! {
    pub struct ColorWriteEnable: i32 {
        const EnableRed = D3D12_COLOR_WRITE_ENABLE_D3D12_COLOR_WRITE_ENABLE_RED;
        const EnableGreen = D3D12_COLOR_WRITE_ENABLE_D3D12_COLOR_WRITE_ENABLE_GREEN;
        const EnableBlue = D3D12_COLOR_WRITE_ENABLE_D3D12_COLOR_WRITE_ENABLE_BLUE;
        const EnableAlpha = D3D12_COLOR_WRITE_ENABLE_D3D12_COLOR_WRITE_ENABLE_ALPHA;
        const EnableAll = D3D12_COLOR_WRITE_ENABLE_D3D12_COLOR_WRITE_ENABLE_ALL;
    }
}
