mod listener;
mod shutdown;
mod tcp;

use serde::Deserialize;
use simple_error::bail;
use std::error::Error;
use std::fs::File;
use std::path::Path;
use tokio::sync::broadcast;

#[derive(Debug, Deserialize)]
struct Target {
    source: String,
    target: String,
}

#[derive(Debug, Deserialize)]
struct Config {
    tcp: Vec<Target>,
    // udp: Vec<Target>,
}

fn load_yaml(path: &Path) -> Result<Config, Box<dyn Error>> {
    let file = File::open(path)?;
    let config: Config = serde_yaml::from_reader(file).expect("Failed to parse!"); //TODO: Print path

    return Ok(config);
}

fn load_json(path: &Path) -> Result<Config, Box<dyn Error>> {
    let file = File::open(path)?;
    let config: Config = serde_json::from_reader(file).expect("Failed to parse!"); //TODO: Print path

    return Ok(config);
}

fn load_config() -> Result<Config, Box<dyn Error>> {
    for path in [
        "config.yaml",
        "config.json",
        "/etc/rustocat.yaml",
        "/etc/rustocat.json",
    ]
    .iter()
    {
        // if(p)
        let config = if path.ends_with(".yaml") {
            load_yaml(Path::new(path))
        } else {
            load_json(Path::new(path))
        };
        if config.is_ok() {
            return config;
        }
    }
    bail!("No config file found");
}

#[tokio::main]
async fn main() {
    let config = load_config().expect("config not found");

    let (notify_shutdown, _) = broadcast::channel(1);

    for target in config.tcp {
        let listener = listener::Listener {
            shutdown: shutdown::Shutdown::new(notify_shutdown.subscribe()),
            source: target.source,
            target: target.target,
        };
        tokio::spawn(async move {
            if let Err(err) = tcp::start_tcp_listener(listener).await {
                println!("listener error: {}", err);
            }
        });
    }

    // _ = notify_shutdown.send(());
    loop {
        tokio::time::sleep(tokio::time::Duration::from_millis(1000 * 1000)).await;
    }
}
