use reqwest;
use tl;

/// Translates the text. The difference between translate is you don't need to pass a from argument in translate_auto.
///
/// # Arguments
///
/// * `text` - The string slice text will be translated.
/// * `to` - The string slice language will be converted to.
///
/// # Examples
///
/// ```
/// fn main() {
///     println!("{:?}", rustlate::translate_auto("hello", "tr"));
/// }
/// ```
pub fn translate_auto(text: &str, to: &str) -> Result<String, String> {
    parse_result(fetch_page(text, "auto", to))
}

/// Translates the text.
///
/// # Arguments
///
/// * `text` - The string slice text will be translated.
/// * `from` - The string slice text's language.
/// * `to` - The string slice language will be converted to.
///
/// # Examples
///
/// ```
/// fn main() {
///     println!("{:?}", rustlate::translate("hello", "en", "tr"));
///     println!("{:?}", rustlate::translate("cat", "en", "tr"));
/// }
/// ```
pub fn translate(text: &str, from: &str, to: &str) -> Result<String, String> {
    parse_result(fetch_page(text, from, to))
}

fn fetch_page(text: &str, from: &str, to: &str) -> Result<String, String> {
    let formatted_url = format!("https://translate.google.com/m?tl={}&sl={}&q={}", to, from, text);

    match reqwest::blocking::get(formatted_url) {
        Ok(response) => {
            match response.text() {
                Ok(body) => {
                    return Ok(body);
                },
                Err(err) => {
                    return Err(err.to_string());
                }
            }
        },
        Err(err) => {
            return Err(err.to_string());
        }
    }
}

fn parse_result(result: Result<String, String>) -> Result<String, String> {
    match result {
        Ok(body) => {
            match tl::parse(&body.to_owned()).get_elements_by_class_name("result-container") {
                Some(element) => {
                    return Ok(element[0].inner_text().into_owned());
                },
                None => {
                    return Err(String::from("unexcepted error."))
                }
            }
        },
        Err(err) => {
            return Err(err);
        }
    }
}