//! # rustify
//!
//! > A Rust library for interacting with HTTP API endpoints
//!
//! Rustify is a small library written in Rust which eases the burden of
//! scaffolding HTTP APIs. It provides an [Endpoint][crate::endpoint::Endpoint]
//! trait along with a macro helper which allows templating various remote
//! endpoints. Both asynchronous and synchrounous clients are offered for
//! executing requests against endpoints with the option of implementing custom
//! clients using the [Client][crate::client::Client] trait.
//!
//! Rustify provides support for serializing requests and deserializing responses in
//! JSON. Raw requests and responses in the form of bytes are also supported. The
//! library also contains many helpers for dealing with requests like support for
//! middleware and wrapping API responses.
//!
//! ## Usage
//!
//! The below example creates a `Test` endpoint that, when executed, will send a
//! GET request to `http://api.com/test/path` and expect an empty response:
//!
//! ```should_panic
//! use rustify::{Client, Endpoint};
//! use rustify_derive::Endpoint;
//! use serde::Serialize;
//!
//! #[derive(Debug, Endpoint, Serialize)]
//! #[endpoint(path = "test/path")]
//! struct Test {}
//!
//! # tokio_test::block_on(async {
//! let endpoint = Test {};
//! let client = Client::default("http://api.com");
//! let result = endpoint.exec(&client).await;
//!
//! assert!(result.is_ok());
//! # });
//! ```
//!
//! ## Examples
//!
//! You can find example usage in the examples directory of the repo. They can
//! be run with cargo:
//!
//! ```ignore
//! cargo run --package rustify --example reqres1
//! cargo run --package rustify --example reqres2
//! ```
//!
//! Additionally, the [vaultrs](https://github.com/jmgilman/vaultrs) is a great
//! reference for a crate which relies heavily on rustify.
//!
//! ## Features
//! The following features are available for this crate:
//!
//! * `blocking`: Enables the blocking variants of `Client`s as well as the blocking
//!    `exec()` functions in `Endpoint`s.
//!
//! ## Error Handling
//!
//! All errors generated by this crate are wrapped in the `ClientError` enum
//! provided by the crate.

#[cfg(feature = "blocking")]
pub mod blocking;
pub mod client;
pub mod clients;
pub mod endpoint;
pub mod enums;
pub mod errors;
pub mod http;

pub use crate::{
    clients::reqwest::Client,
    endpoint::{Endpoint, MiddleWare, Wrapper},
};
