use std::fs::File;
use std::io::BufReader;
use std::path::PathBuf;
use std::time::Duration;

use anyhow::{bail, Result};
use indicatif::{HumanDuration, ProgressBar, ProgressStyle};
use rpassword::{prompt_password_stderr, read_password_with_reader};
use vlog::*;

use crate::backend::ReadBackend;
use crate::crypto::Key;
use crate::repo::find_key_in_backend;

const MAX_PASSWORD_RETRIES: usize = 5;

pub fn get_key(be: &impl ReadBackend, password_file: Option<PathBuf>) -> Result<Key> {
    let key = match password_file {
        None => (0..MAX_PASSWORD_RETRIES)
            .map(|_| {
                let pass = prompt_password_stderr("enter repository password: ")?;
                find_key_in_backend(be, &pass, None)
            })
            .find(Result::is_ok)
            .unwrap_or_else(|| bail!("tried too often...aborting!"))?,
        Some(file) => {
            let mut file = BufReader::new(File::open(file)?);
            let pass = read_password_with_reader(Some(&mut file))?;
            find_key_in_backend(be, &pass, None)?
        }
    };
    ve1!("password is correct");
    Ok(key)
}

pub fn progress_counter() -> ProgressBar {
    if get_verbosity_level() == 1 {
        ProgressBar::new(0).with_style(
            ProgressStyle::default_bar()
                .template("[{elapsed_precise}] {bar:40.cyan/blue} {pos:>10}/{len:10}")
        )
    } else {
        ProgressBar::hidden()
    }
}

pub fn progress_bytes() -> ProgressBar {
    if get_verbosity_level() == 1 {
        ProgressBar::new(0).with_style(
            ProgressStyle::default_bar()
            /* with indicatif 0.17
            .with_key("my_eta", |s| 
                 match (s.pos(), s.len()){
                    (0, _) => "-".to_string(),
                    (pos,len) => format!("{:#}", HumanDuration(Duration::from_secs(s.elapsed().as_secs() * (len-pos)/pos))),
                })
            .template("[{elapsed_precise}] {bar:40.cyan/blue} {bytes:>10}/{total_bytes:10} {bytes_per_sec:12} (ETA {my_eta})")
            .unwrap(),
            */
            .template("[{elapsed_precise}] {bar:40.cyan/blue} {bytes:>10}/{total_bytes:10} {bytes_per_sec:12}")
        )
    } else {
        ProgressBar::hidden()
    }
}
