use instruction::BasicBlock;
use crate::program::instruction::{BasicBlockRef, Instruction, InstructionRef};
use std::io;
use crate::frontend::parser::{BfParsingError, BfParser};
use std::path::Path;
use std::io::{BufReader, Write, Read};
use crate::frontend::lexer::BfLexer;
use crate::program::exec::{EofMode, ExecError, ExecContext};
use std::fmt::{Display, Formatter};
use std::error::Error;

pub mod instruction;
pub mod exec;

pub struct BfProgram {
    blocks: Vec<BasicBlock>,
}
impl BfProgram {
    pub fn new() -> BfProgram {
        let start_block = BasicBlock::new();

        BfProgram {
            blocks: vec!(start_block),
        }
    }
    pub fn start_block(&self) -> BasicBlockRef {
        BasicBlockRef(0)
    }

    pub fn parse_from_file<P>(p: P) -> Result<Self, ParseFromFileError>
        where P: AsRef<Path> {
        let file = std::fs::File::open(p)?;
        let buff_read = BufReader::new(file);
        let lexer = BfLexer::new(buff_read);
        let parser = BfParser::new(lexer);

        let program = parser.parse()?.build_program();
        Ok(program)
    }
    pub fn parse_from_src<S>(s: S) -> Result<Self, BfParsingError>
        where S: AsRef<[u8]> {
        let s = s.as_ref();
        let lexer = BfLexer::new(s);
        let parser = BfParser::new(lexer);
        let program = parser.parse()?;
        let program = program.build_program();
        Ok(program)
    }


    pub fn execute<O, I>(&self, mut out: O, mut src: I, eof: EofMode) -> Result<(), ExecError>
        where O: Write,
              I: Read {
        let ctx = ExecContext::new(self, &mut out, &mut src, eof);
        ctx.execute()
    }


    pub fn add_block(&mut self) -> BasicBlockRef {
        let i = self.blocks.len();
        self.blocks.push(BasicBlock::new());
        BasicBlockRef(i)
    }
    pub fn block(&self, b: BasicBlockRef) -> &BasicBlock {
        let i = b.index();
        &self.blocks[i]
    }
    pub fn block_mut(&mut self, b: BasicBlockRef) -> &mut BasicBlock {
        let i = b.index();
        &mut self.blocks[i]
    }
    pub fn end_of(&self, b: BasicBlockRef) -> InstructionRef {
        let len = self.block(b).instructions.len();
        let i = len - 1;
        InstructionRef(b, i)
    }

    pub fn add_instruction(&mut self, b: BasicBlockRef, i: Instruction) {
        let b = self.block_mut(b);
        b.instructions.push(i);
    }
    pub fn instruction(&self, i: InstructionRef) -> &Instruction {
        let b = self.block(i.block());
        &b.instructions[i.index()]
    }
}

#[derive(Debug)]
pub enum ParseFromFileError {
    Io(io::Error),
    Parse(BfParsingError),
}
impl Display for ParseFromFileError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::Io(err) => write!(f, "{}", err),
            Self::Parse(err) => write!(f, "{}", err),
        }
    }
}
impl Error for ParseFromFileError { }
impl From<io::Error> for ParseFromFileError {
    fn from(err: io::Error) -> Self {
        Self::Io(err)
    }
}
impl From<BfParsingError> for ParseFromFileError {
    fn from(err: BfParsingError) -> Self {
        Self::Parse(err)
    }
}




pub struct BfProgramBuilder {
    program: BfProgram,
    curr_block: BasicBlockRef,
}
impl BfProgramBuilder {
    pub fn new() -> BfProgramBuilder {
        let program = BfProgram::new();
        let curr_block = program.start_block();

        BfProgramBuilder {
            program,
            curr_block
        }
    }
    pub fn finish(self) -> BfProgram {
        self.program
    }

    fn append_instruction(&mut self, i: Instruction) {
        let b = self.program.block_mut(self.curr_block);
        b.instructions.push(i);
    }
    fn select_block(&mut self, b: BasicBlockRef) {
        self.curr_block = b;
    }

    pub fn build_loop<F>(&mut self, f: F)
        where F: FnOnce(&mut Self) {
        let origin = self.curr_block;
        let body = self.program.add_block();
        let end = self.program.add_block();

        self.append_instruction(Instruction::BeginLoop { body, end });
        self.select_block(body);

        f(self);
        self.append_instruction(Instruction::EndLoop { origin });
        self.select_block(end);
    }

    pub fn increment(&mut self, amount: u8) {
        self.append_instruction(Instruction::Increment(amount));
    }
    pub fn decrement(&mut self, amount: u8) {
        self.append_instruction(Instruction::Decrement(amount));
    }
    pub fn set(&mut self, value: u8) {
        self.append_instruction(Instruction::Set(value));
    }
    pub fn next(&mut self, amount: usize) {
        self.append_instruction(Instruction::Next(amount));
    }
    pub fn previous(&mut self, amount: usize) {
        self.append_instruction(Instruction::Previous(amount));
    }
    pub fn output(&mut self) {
        self.append_instruction(Instruction::Output);
    }
    pub fn input(&mut self) {
        self.append_instruction(Instruction::Input);
    }
}
