use std::error::Error;
use std::fmt::{Display, Formatter};
use std::str::FromStr;
use clap::Clap;
use std::io::{Write, Read};
use crate::program::BfProgram;
use crate::program::instruction::{BasicBlockRef, InstructionRef, Instruction};
use std::io;


pub struct ExecContext<'a, 'b, 'c> {
    program: &'c BfProgram,
    out: &'a mut dyn Write,
    src: &'b mut dyn Read,
    cell_ptr: usize,
    cells: Vec<u8>,
    eof: EofMode,
}
impl<'a, 'b, 'c> ExecContext<'a, 'b, 'c> {
    pub fn new(program: &'c BfProgram, out: &'a mut dyn Write, src: &'b mut dyn Read, eof: EofMode) -> Self {
        Self {
            program,
            out,
            src,
            cell_ptr: 0,
            cells: Vec::new(),
            eof,
        }
    }


    pub fn execute(mut self) -> Result<(), ExecError> {
        let mut block = self.program.start_block();

        while let Some(next) = self.exec_block(block)? {
            block = next;
        }

        Ok(())
    }


    fn exec_block(&mut self, b: BasicBlockRef) -> Result<Option<BasicBlockRef>, ExecError> {
        let block = self.program.block(b);
        let len = block.instructions.len();

        for i in 0..len {
            let i_ref = InstructionRef(b, i);
            let over = self.exec_instruction(i_ref)?;

            if let BlockOver::Yes(next) = over {
                return Ok(next);
            }
        }

        Ok(None)
    }
    fn exec_instruction(&mut self, i: InstructionRef) -> Result<BlockOver, ExecError> {
        let instruction = *self.program.instruction(i);

        match instruction {
            Instruction::BeginLoop { body, end } => return Ok(self.begin_loop(body, end)),
            Instruction::EndLoop { origin } => return Ok(self.end_loop(origin)),
            Instruction::Increment(a) => self.increment(a),
            Instruction::Decrement(a) => self.decrement(a),
            Instruction::Set(a) => self.set(a),
            Instruction::Next(a) => self.next(a)?,
            Instruction::Previous(a) => self.previous(a)?,
            Instruction::Output => self.output()?,
            Instruction::Input => self.input()?,
        }

        Ok(BlockOver::No)
    }


    fn pad_cells(&mut self) {
        let new_len = self.cell_ptr + 1;

        if new_len > self.cells.len() {
            let diff = new_len - self.cells.len();
            let padding = std::iter::once(0).cycle().take(diff);
            self.cells.extend(padding);
        }
    }
    fn write_byte(&mut self, b: u8) -> io::Result<()> {
        self.out.write(&[b])?;
        self.out.flush()
    }
    fn read_byte(&mut self) -> io::Result<Option<u8>> {
        let mut buff = [0];
        let read = self.src.read(&mut buff)?;

        if read == 0 {
            Ok(None)
        }
        else {
            Ok(Some(buff[0]))
        }
    }

    fn begin_loop(&mut self, body: BasicBlockRef, end: BasicBlockRef) -> BlockOver {
        self.pad_cells();

        if self.cells[self.cell_ptr] != 0 {
            BlockOver::Yes(Some(body))
        }
        else {
            BlockOver::Yes(Some(end))
        }
    }
    fn end_loop(&mut self, origin: BasicBlockRef) -> BlockOver {
        let last_i = self.program.end_of(origin);
        let last = *self.program.instruction(last_i);


        if let Instruction::BeginLoop { body, end } = last {
            self.pad_cells();

            if self.cells[self.cell_ptr] != 0 {
                BlockOver::Yes(Some(body))
            }
            else {
                BlockOver::Yes(Some(end))
            }
        }
        else {
            unreachable!()
        }
    }

    fn increment(&mut self, a: u8) {
        self.pad_cells();
        let cell = &mut self.cells[self.cell_ptr];
        *cell = cell.wrapping_add(a);
    }
    fn decrement(&mut self, a: u8) {
        self.pad_cells();
        let cell = &mut self.cells[self.cell_ptr];
        *cell = cell.wrapping_sub(a);
    }
    fn set(&mut self, v: u8) {
        self.pad_cells();
        self.cells[self.cell_ptr] = v;
    }
    fn next(&mut self, a: usize) -> Result<(), ExecError> {
        let (new, wrapped) = self.cell_ptr.overflowing_add(a);

        if wrapped {
            Err(ExecError::CellPtrOverflow)
        }
        else {
            self.cell_ptr = new;
            Ok(())
        }
    }
    fn previous(&mut self, a: usize) -> Result<(), ExecError> {
        let (new, wrapped) = self.cell_ptr.overflowing_sub(a);

        if wrapped {
            Err(ExecError::CellPtrUnderflow)
        }
        else {
            self.cell_ptr = new;
            Ok(())
        }
    }
    fn output(&mut self) -> Result<(), ExecError> {
        self.pad_cells();
        let byte = self.cells[self.cell_ptr];
        self.write_byte(byte)?;
        Ok(())
    }
    fn input(&mut self) -> Result<(), ExecError> {
        let byte = self.read_byte()?;
        self.pad_cells();

        let new_val = match byte {
            Some(byte) => byte,
            None => match self.eof {
                EofMode::Max => u8::MAX,
                EofMode::Zero => 0,
                EofMode::NoChange => self.cells[self.cell_ptr],
            },
        };
        self.cells[self.cell_ptr] = new_val;

        Ok(())
    }
}

enum BlockOver {
    No,
    Yes(Option<BasicBlockRef>),
}

#[derive(Debug)]
pub enum ExecError {
    CellPtrUnderflow,
    CellPtrOverflow,
    Io(io::Error),
}
impl Display for ExecError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::Io(err) => write!(f, "{}", err),
            Self::CellPtrOverflow => write!(f, "cell pointer overflowed past max value"),
            Self::CellPtrUnderflow => write!(f, "cell pointer underflowwed past zero"),
        }
    }
}
impl Error for ExecError {}
impl From<io::Error> for ExecError {
    fn from(err: io::Error) -> Self {
        Self::Io(err)
    }
}


/// Indicates what should be done when a brainfuck program tries to read from stdin
/// and encountered an EOF;
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash, Clap)]
pub enum EofMode {
    /// Set the cell under the data pointer to 0xFF
    Max,
    /// Set the cell under the data pointer to 0x00
    Zero,
    /// Don't change the cell under the data pointer
    NoChange,
}
impl FromStr for EofMode {
    type Err = InvalidEofError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let s = s.to_lowercase();

        match s.as_str() {
            "max" => Ok(Self::Max),
            "zero" => Ok(Self::Zero),
            "no" => Ok(Self::NoChange),
            _ => Err(InvalidEofError(s)),
        }
    }
}

#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct InvalidEofError(String);
impl Display for InvalidEofError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{} is not a valid EofMode; Possible: 'max', 'zero', 'no'", self.0)
    }
}
impl Error for InvalidEofError {}
