use clap::{AppSettings, Clap};
use rustfuck::program::exec::EofMode;
use rustfuck::program::BfProgram;
use std::io::{stdout, stdin};

fn main() {
    let opts: Config = Config::parse();

    match opts.run() {
        Ok(()) => (),
        Err(err) => println!("{}", err),
    }
}

#[derive(Clap)]
#[clap(version = env!("CARGO_PKG_VERSION"))]
#[clap(setting = AppSettings::ColoredHelp)]
pub struct Config {
    #[clap(subcommand)]
    subcommand: SubCommand,
}
impl Config {
    fn run(&self) -> anyhow::Result<()> {
        self.subcommand.run()
    }
}

#[derive(Clap)]
pub enum SubCommand {
    /// Interpret the given brainfuck program
    Int {
        /// The source path of the program.
        file: String,
        #[clap(short, long, default_value = "no")]
        /// What to do when the interpreter encounters an EOF from stdin.
        eof: EofMode,
    },
}
impl SubCommand {
    fn run(&self) -> anyhow::Result<()> {
        match self {
            Self::Int { file, eof } => interpret(file, *eof),
        }
    }
}


fn interpret(file: &str, eof: EofMode) -> anyhow::Result<()> {
    let program = BfProgram::parse_from_file(file)?;
    program.execute(stdout(), stdin(), eof)?;
    Ok(())
}

