use std::io::Read;


/// Defines the capabilities of something that can lex brainfuck code.
pub trait BfLexedSrc: Iterator<Item = BfToken> {

}


/// Lexes a series of bytes into a list of BfTokens to be processed by a parser.
/// ```
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// use rustfuck::frontend::lexer::{BfLexer,BfToken, BfTokenKind};
/// let src = "[->+<]".as_bytes();
/// let mut lexer = BfLexer::new(src);
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::OpenLoop));
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::Decrement));
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::Next));
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::Increment));
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::Previous));
/// assert_eq!(lexer.next().map(BfToken::kind), Some(BfTokenKind::CloseLoop));
/// assert_eq!(lexer.next(), None);
///
/// # Ok(())
/// # }
/// ```
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct BfLexer<I> {
    src: I,
    line: u32,
    char: u32,
}
impl<I: Read> BfLexer<I> {
    pub fn new(src: I) -> Self {
        let ret = BfLexer {
            src,
            line: 1,
            char: 1,
        };

        ret
    }


    fn next_token(&mut self) -> Option<BfToken> {
        loop {
            let mut buff = [0];
            let len = self.src.read(&mut buff).ok()?;
            if len == 0 {
                return None;
            }

            let line = self.line;
            let char = self.char;
            let kind = self.process_byte(buff[0]);

            if let Some(kind) = kind {
                return Some(BfToken {
                    kind,
                    line,
                    char,
                });
            }
        }
    }
    fn process_byte(&mut self, b: u8) -> Option<BfTokenKind> {
        self.char += 1;

        if b < 128 {
            let c = b as char;
            self.process_char(c)
        }
        else {
            None
        }
    }
    fn process_char(&mut self, c: char) -> Option<BfTokenKind> {
        match c {
            '\n' => {
                self.line += 1;
                self.char = 1;
                None
            },
            '+' => Some(BfTokenKind::Increment),
            '-' => Some(BfTokenKind::Decrement),
            '>' => Some(BfTokenKind::Next),
            '<' => Some(BfTokenKind::Previous),
            '.' => Some(BfTokenKind::Output),
            ',' => Some(BfTokenKind::Input),
            '[' => Some(BfTokenKind::OpenLoop),
            ']' => Some(BfTokenKind::CloseLoop),
            _ => None,
        }

    }
}
impl<I: Read> Iterator for BfLexer<I> {
    type Item = BfToken;

    fn next(&mut self) -> Option<Self::Item> {
        self.next_token()
    }
}
impl<I: Read> BfLexedSrc for BfLexer<I> {

}



#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct BfToken {
    kind: BfTokenKind,
    line: u32,
    char: u32,
}
impl BfToken {
    pub fn kind(self) -> BfTokenKind {
        self.kind
    }
    pub fn line(self) -> u32 {
        self.line
    }
    pub fn char(self) -> u32 {
        self.char
    }
}
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub enum BfTokenKind {
    Increment,
    Decrement,
    Next,
    Previous,
    Output,
    Input,
    OpenLoop,
    CloseLoop,
}
