use crate::program::{BfProgram, BfProgramBuilder};

/// Represents a complete brainfuck program as a series of BfInstructions.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct IntProgram {
    instructions: Vec<BfInstruction>,
}
impl IntProgram {
    pub fn new(instructions: Vec<BfInstruction>) -> Self {
        Self {
            instructions,
        }
    }

    /// Returns the amount of BfInstructions on the top level of the Program,
    /// that is, not contained in any loops whatsoever.
    /// Loops are counted as one instruction.
    pub fn top_level_instruction_count(&self) -> usize {
        self.instructions.len()
    }
    /// Returns a reference to the internal instruction buffer
    /// containing all top level instructions.
    pub fn instructions(&self) -> &[BfInstruction] {
        &self.instructions
    }
    
    
    pub fn build_program(&self) -> BfProgram {
        let mut builder = BfProgramBuilder::new();
        
        for i in &self.instructions {
            Self::build_instruction(&mut builder, i);
        }
        
        builder.finish()
    }
    fn build_instruction(builder: &mut BfProgramBuilder, i: &BfInstruction) {
        match i {
            BfInstruction::Increment(a) => builder.increment(*a),
            BfInstruction::Decrement(a) => builder.decrement(*a),
            BfInstruction::Set(a) => builder.set(*a),
            BfInstruction::Next(a) => builder.next(*a),
            BfInstruction::Previous(a) => builder.previous(*a),
            BfInstruction::Output => builder.output(),
            BfInstruction::Input => builder.input(),
            BfInstruction::Loop(nodes) => builder.build_loop(|b| {
                for node in nodes {
                    Self::build_instruction(b, node);
                }
            }),
        }
    }
}

/// Represents a brainfuck instruction in a compressed manner.
/// This is not the same as the actual textual brainfuck commands.
/// While they perform the same functions, these instructions can have
/// additional 'amount' parameters, combining potentially many hundreds of bf-commands
/// into a single instruction.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub enum BfInstruction {
    /// Increment the cell under the data pointer by a certain amount.
    Increment(u8),
    /// Decrement the cell under the data pointer by a certain amount.
    Decrement(u8),
    /// Move the data pointer to the right by a certain amount.
    Next(usize),
    /// Move the data pointer to the right by a certain amount.
    Previous(usize),
    /// Write the value of the cell under the data pointer to stdout.
    Output,
    /// Read a byte from stdin and save it in the cell under the data pointer.
    Input,
    /// Repeat contained instructions while the cell under the data pointer is zero.
    Loop(Vec<BfInstruction>),
    /// Set the cell under the data pointer to a constant value.
    Set(u8),
}
