use std::io::{Write, Read, BufReader};
use std::io;
use std::path::Path;
use crate::frontend::parser::{BfParsingError, BfParser};
use crate::frontend::lexer::BfLexer;
use std::fmt::{Display, Formatter};
use std::error::Error;
use clap::Clap;
use std::str::FromStr;

/// Represents a complete brainfuck program as a series of BfInstructions.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct BfProgram {
    instructions: Vec<BfInstruction>,
}
impl BfProgram {
    pub fn new(instructions: Vec<BfInstruction>) -> Self {
        Self {
            instructions,
        }
    }
    pub fn parse_from_src<P>(file: P) -> Result<Self, BfParsingError>
        where P: AsRef<str> {
        let src = file.as_ref().as_bytes();
        let lexer = BfLexer::new(src);
        let parser = BfParser::new(lexer);
        parser.parse()
    }
    pub fn parse_from_file<P>(file: P) -> Result<Self, ParseFromFileError>
        where P: AsRef<Path> {
        let src = std::fs::File::open(file)?;
        let buf_read = BufReader::new(src);
        let lexer = BfLexer::new(buf_read);
        let parser = BfParser::new(lexer);

        let program = parser.parse()?;
        Ok(program)
    }

    /// Returns the amount of BfInstructions on the top level of the Program,
    /// that is, not contained in any loops whatsoever.
    /// Loops are counted as one instruction.
    pub fn top_level_instruction_count(&self) -> usize {
        self.instructions.len()
    }
    /// Returns a reference to the internal instruction buffer
    /// containing all top level instructions.
    pub fn instructions(&self) -> &[BfInstruction] {
        &self.instructions
    }

    /// Executes a BfProgram
    ///
    /// ```
    /// # use std::error::Error;
    /// # use crate::rustfuck::program::BfProgram;
    /// # fn get_hello_world_program() -> Result<BfProgram, Box<dyn Error>> {
    /// # use rustfuck::frontend::lexer::BfLexer;
    /// # use rustfuck::frontend::parser::BfParser;
    /// # let src ="\
    /// # ++++++++[>++++[>++>+++>+++>+<<<<-]>+>+>->>+[<]<-]>>.>---.+++++++..+++.>>.<-.<.+++.------.--------.>>+.>++.
    /// # ".as_bytes();
    /// # let lexer = BfLexer::new(src);
    /// # Ok(BfParser::new(lexer).parse()?)
    /// # }
    ///
    /// # fn main() -> Result<(), Box<dyn Error>> {
    /// use std::io::empty;
    /// use rustfuck::program::EofMode;
    /// let program = get_hello_world_program()?;
    /// let mut out = Vec::new();
    /// program.execute(&mut out, empty(), EofMode::Max)?;
    /// assert_eq!(out, "Hello World!\n".as_bytes());
    /// # Ok(())
    /// # }
    ///
    ///
    /// ```
    pub fn execute<O, I>(&self, mut out: O, mut src: I, eof: EofMode) -> ExecResult<()>
        where O: Write,
              I: Read {
        let ctx = ExecContext::new(&mut out, &mut src, eof);
        ctx.exec(self)
    }
}

#[derive(Debug)]
pub enum ParseFromFileError {
    ParsingError(BfParsingError),
    IoError(io::Error),
}
impl Display for ParseFromFileError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "Failed to parse input bf file: ")?;

        match self {
            Self::ParsingError(err) => write!(f, "{}", err),
            Self::IoError(err) => write!(f, "{}", err),
        }
    }
}
impl From<io::Error> for ParseFromFileError {
    fn from(err: io::Error) -> Self {
        Self::IoError(err)
    }
}
impl From<BfParsingError> for ParseFromFileError {
    fn from(err: BfParsingError) -> Self {
        Self::ParsingError(err)
    }
}


/// Indicates what should be done when a brainfuck program tries to read from stdin
/// and encountered an EOF;
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash, Clap)]
pub enum EofMode {
    /// Set the cell under the data pointer to 0xFF
    Max,
    /// Set the cell under the data pointer to 0x00
    Zero,
    /// Don't change the cell under the data pointer
    NoChange,
}
impl FromStr for EofMode {
    type Err = InvalidEofError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let s = s.to_lowercase();

        match s.as_str() {
            "max" => Ok(Self::Max),
            "zero" => Ok(Self::Zero),
            "no" => Ok(Self::NoChange),
            _ => Err(InvalidEofError(s)),
        }
    }
}

#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct InvalidEofError(String);
impl Display for InvalidEofError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{} is not a valid EofMode; Possible: 'max', 'zero', 'no'", self.0)
    }
}
impl Error for InvalidEofError {}


/// Represents a brainfuck instruction in a compressed manner.
/// This is not the same as the actual textual brainfuck commands.
/// While they perform the same functions, these instructions can have
/// additional 'amount' parameters, combining potentially many hundreds of bf-commands
/// into a single instruction.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub enum BfInstruction {
    /// Increment the cell under the data pointer by a certain amount.
    Increment(u8),
    /// Decrement the cell under the data pointer by a certain amount.
    Decrement(u8),
    /// Move the data pointer to the right by a certain amount.
    Next(usize),
    /// Move the data pointer to the right by a certain amount.
    Previous(usize),
    /// Write the value of the cell under the data pointer to stdout.
    Output,
    /// Read a byte from stdin and save it in the cell under the data pointer.
    Input,
    /// Repeat contained instructions while the cell under the data pointer is zero.
    Loop(Vec<BfInstruction>),
    /// Set the cell under the data pointer to a constant value.
    Set(u8),
}


struct ExecContext<'a, 'b> {
    out: &'a mut dyn Write,
    src: &'b mut dyn Read,
    eof: EofMode,
    data_ptr: usize,
    cells: Vec<u8>,
}
impl<'a, 'b> ExecContext<'a, 'b> {
    pub fn new(out: &'a mut dyn Write, src: &'b mut dyn Read, eof: EofMode) -> Self {
        Self {
            out,
            src,
            eof,
            data_ptr: 0,
            cells: Vec::with_capacity(30_000),
        }
    }

    pub fn exec(mut self, program: &BfProgram) -> ExecResult<()> {
        for i in program.instructions() {
            self.exec_instruction(i)?;
        }

        Ok(())
    }
    fn exec_instruction(&mut self, i: &BfInstruction) -> ExecResult<()> {
        match i {
            BfInstruction::Increment(a) => self.increment(*a),
            BfInstruction::Decrement(a) => self.decrement(*a),
            BfInstruction::Next(a) => self.next(*a),
            BfInstruction::Previous(a) => self.previous(*a),
            BfInstruction::Output => self.output()?,
            BfInstruction::Input => self.input()?,
            BfInstruction::Set(val) => self.set(*val),
            BfInstruction::Loop(nodes) => self.exec_loop(nodes)?,
        }

        Ok(())
    }

    fn pad_cells(&mut self) {
        if self.data_ptr >= self.cells.len() {
            let diff = self.data_ptr + 1 - self.cells.len();

            self.cells.extend(std::iter::once(0).cycle().take(diff));
        }
    }
    fn increment(&mut self, amount: u8) {
        self.pad_cells();
        let cell = &mut self.cells[self.data_ptr];
        *cell = cell.wrapping_add(amount);
    }
    fn decrement(&mut self, amount: u8) {
        self.pad_cells();
        let cell = &mut self.cells[self.data_ptr];
        *cell = cell.wrapping_sub(amount);
    }
    fn next(&mut self, amount: usize) {
        self.data_ptr = self.data_ptr.wrapping_add(amount);
    }
    fn previous(&mut self, amount: usize) {
        self.data_ptr = self.data_ptr.wrapping_sub(amount);
    }
    fn output(&mut self) -> ExecResult<()> {
        self.pad_cells();
        let byte = self.cells[self.data_ptr];
        self.write(byte)
    }
    fn input(&mut self) -> ExecResult<()> {
        self.pad_cells();
        let b = self.read()?;

        if let Some(b) = b {
            self.cells[self.data_ptr] = b;
        }
        else {
            self.cells[self.data_ptr] = match self.eof {
                EofMode::Max => 0xFF,
                EofMode::Zero => 0x00,
                EofMode::NoChange => return Ok(()),
            };
        }

        Ok(())
    }
    fn set(&mut self, val: u8) {
        self.pad_cells();
        self.cells[self.data_ptr] = val;
    }
    fn exec_loop(&mut self, nodes: &[BfInstruction]) -> ExecResult<()> {
        self.pad_cells();
        while self.cells[self.data_ptr] != 0 {
            for i in nodes {
                self.exec_instruction(i)?;
            }
            self.pad_cells();
        }

        Ok(())
    }

    fn write(&mut self, b: u8) -> ExecResult<()> {
        self.out.write(&[b])?;
        self.out.flush()?;
        Ok(())
    }
    fn read(&mut self) -> ExecResult<Option<u8>> {
        let mut buff = [0];
        let read = self.src.read(&mut buff);

        let read = match read {
            Ok(read) => read,
            Err(err) => return match err.kind() {
                io::ErrorKind::UnexpectedEof => Ok(None),
                _ => Err(err.into()),
            }
        };
        if read == 0 {
            return Ok(None);
        }

        Ok(Some(buff[0]))
    }
}

#[derive(Debug)]
pub enum BfExecError {
    IoError(io::Error),
    DataPtrUnderflow,
    DataPtrOverflow,
}
impl Display for BfExecError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::IoError(err) => write!(f, "{}", err)?,
            Self::DataPtrUnderflow => write!(f, "data pointer underflowed below zero")?,
            Self::DataPtrOverflow => write!(f, "data pointer overflowed beyond max value")?,
        }

        Ok(())
    }
}
impl Error for BfExecError {}
impl From<io::Error> for BfExecError {
    fn from(err: io::Error) -> Self {
        Self::IoError(err)
    }
}

pub type ExecResult<T> = Result<T, BfExecError>;
