use clap::{Clap, AppSettings};
use rustfuck::program::{ParseFromFileError, BfProgram, BfExecError, EofMode};
use std::io::{stdout, stdin};
use std::fmt::{Display, Formatter};
use std::error::Error;

fn main() {
    let opts: Config = Config::parse();

    match opts.run() {
        Ok(()) => (),
        Err(err) => eprintln!("{}", err),
    }
}

#[derive(Clap)]
#[clap(version = env!("CARGO_PKG_VERSION"))]
#[clap(setting = AppSettings::ColoredHelp)]
pub struct Config {
    #[clap(subcommand)]
    subcommand: SubCommand,
}
impl Config {
    fn run(&self) -> Result<(), RunSubCommandError> {
        self.subcommand.run()
    }
}

#[derive(Clap)]
pub enum SubCommand {
    /// Interpret the given brainfuck program
    Int {
        /// The source path of the program.
        file: String,
        #[clap(short, long, default_value = "max")]
        /// What to do when the interpreter encounters an EOF from stdin.
        eof: EofMode,
    },
}
impl SubCommand {
    fn run(&self) -> Result<(), RunSubCommandError> {
        match self {
            Self::Int { file, eof } => interpret(file, *eof)?,
        }

        Ok(())
    }
}

#[derive(Debug)]
enum RunSubCommandError {
    Interpret(InterpretError),
}
impl Display for RunSubCommandError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::Interpret(err) => write!(f, "{}", err),
        }
    }
}
impl From<InterpretError> for RunSubCommandError {
    fn from(err: InterpretError) -> Self {
        Self::Interpret(err)
    }
}


fn interpret(file: &String, eof_mode: EofMode) -> Result<(), InterpretError> {
    let program = BfProgram::parse_from_file(file)?;
    program.execute(stdout(), stdin(), eof_mode)?;
    Ok(())
}

#[derive(Debug)]
enum InterpretError {
    ParseError(ParseFromFileError),
    Execute(BfExecError)
}
impl Display for InterpretError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::ParseError(err) => write!(f, "{}", err),
            Self::Execute(err) => write!(f, "{}", err),
        }
    }
}
impl Error for InterpretError {}
impl From<ParseFromFileError> for InterpretError {
    fn from(err: ParseFromFileError) -> Self {
        Self::ParseError(err)
    }
}
impl From<BfExecError> for InterpretError {
    fn from(err: BfExecError) -> Self {
        Self::Execute(err)
    }
}
