use crate::frontend::lexer::{BfLexedSrc, BfTokenKind};
use std::fmt::{Display, Formatter};
use std::error::Error;
use crate::program::{BfProgram, BfInstruction};
use crate::frontend::simple_opts::{merge_instructions, collapse_loop};


/// Parses a series of BfTokens into a BfProgram
///
/// ```
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// use rustfuck::frontend::lexer::BfLexer;
/// use rustfuck::frontend::parser::BfParser;
/// use rustfuck::program::BfInstruction;
/// let src = "[->+<]".as_bytes();
/// let lexer = BfLexer::new(src);
/// let parsed = BfParser::new(lexer).parse()?;
///
/// assert_eq!(parsed.top_level_instruction_count(), 1);
/// let expected_loop = vec!(
///     BfInstruction::Decrement(1),
///     BfInstruction::Next(1),
///     BfInstruction::Increment(1),
///     BfInstruction::Previous(1),
/// );
///
/// assert_eq!(parsed.instructions()[0], BfInstruction::Loop(expected_loop));
///
/// # Ok(())
/// # }
/// ```
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct BfParser<I> {
    src: I,
    at_end: bool,
}
impl<I: BfLexedSrc> BfParser<I> {
    pub fn new(src: I) -> Self {
        Self {
            src,
            at_end: false,
        }
    }

    pub fn parse(mut self) -> BfParsingResult<BfProgram> {
        let instructions = self.parse_series(false)?;
        Ok(BfProgram::new(instructions))
    }

    fn parse_series(&mut self, in_loop: bool) -> BfParsingResult<Vec<BfInstruction>> {
        let mut instructions = Vec::new();
        let mut last_instruction = None;

        while let Some(new) = self.loop_buffer(in_loop) {
            let new = new?;

            match &last_instruction {
                Some(last) => {
                    if let Some(merged) = merge_instructions(last, &new) {
                        last_instruction = Some(merged);
                    }
                    else {
                        instructions.push(last_instruction.take().unwrap());
                        last_instruction = Some(new);
                    }
                }
                None => last_instruction = Some(new),
            }
        }
        if let Some(last) = last_instruction {
            instructions.push(last);
        }

        Ok(instructions)
    }
    fn loop_buffer(&mut self, in_loop: bool) -> Option<BfParsingResult<BfInstruction>> {
        match self.parse_token()? {
            Ok(new) => Some(Ok(new)),
            Err(err) => match err.kind() {
                BfParsingErrorKind::UnexpectedCloseLoop if in_loop => None,
                _ => Some(Err(err)),
            }
        }
    }
    fn parse_loop(&mut self, line: u32, char: u32) -> BfParsingResult<BfInstruction> {
        let instructions = self.parse_series(true)?;

        if self.at_end {
            Err(BfParsingError {
                line,
                char,
                kind: BfParsingErrorKind::LoopNotClosed,
            })
        }
        else {
            if let Some(better) = collapse_loop(&instructions) {
                Ok(better)
            }
            else {
                Ok(BfInstruction::Loop(instructions))
            }
        }
    }
    fn parse_token(&mut self) -> Option<BfParsingResult<BfInstruction>> {
        if let Some(token) = self.src.next() {
            let line = token.line();
            let char = token.char();

            match token.kind() {
                BfTokenKind::Increment => Some(Ok(BfInstruction::Increment(1))),
                BfTokenKind::Decrement => Some(Ok(BfInstruction::Decrement(1))),
                BfTokenKind::Next => Some(Ok(BfInstruction::Next(1))),
                BfTokenKind::Previous => Some(Ok(BfInstruction::Previous(1))),
                BfTokenKind::Output => Some(Ok(BfInstruction::Output)),
                BfTokenKind::Input => Some(Ok(BfInstruction::Input)),
                BfTokenKind::OpenLoop => Some(self.parse_loop(line, char)),
                BfTokenKind::CloseLoop => Some(Err(BfParsingError {
                    line,
                    char,
                    kind: BfParsingErrorKind::UnexpectedCloseLoop,
                })),
            }
        }
        else {
            self.at_end = true;
            None
        }
    }
}

pub type BfParsingResult<T> = Result<T, BfParsingError>;

#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct BfParsingError {
    line: u32,
    char: u32,
    kind: BfParsingErrorKind,
}
impl BfParsingError {
    pub fn kind(self) -> BfParsingErrorKind {
        self.kind
    }
    pub fn line(self) -> u32 {
        self.line
    }
    pub fn char(self) -> u32 {
        self.char
    }
}
impl Display for BfParsingError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "({}|{}): {}", self.line, self.char, self.kind)
    }
}
impl Error for BfParsingError {}

#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub enum BfParsingErrorKind {
    UnexpectedCloseLoop,
    LoopNotClosed,
}
impl Display for BfParsingErrorKind {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::UnexpectedCloseLoop => write!(f, "unexpected ] token")?,
            Self::LoopNotClosed => write!(f, "loop never closed")?,
        }
        
        Ok(())
    }
}
