use anyhow::Result;
use argh::FromArgs;

/// 例子：指定两个目录
/// `rustdx day sh/lday sz/lday`，会根据文件名自动指定为 `-e sz,0,3,sh,6` 格式。
#[derive(FromArgs, PartialEq, Debug)]
#[argh(subcommand, name = "day")]
pub struct DayCmd {
    /// 可选。指定交易所或者代码开头的文件。使用 `rustdx day -h e`
    /// 查看详细使用说明。
    #[argh(option, short = 'e')]
    pub exchange: Option<String>,

    /// 可选。匹配 6 位代码的前几位：
    /// `-e sz -c 0000` == `-l sz0000开头的股票` == `sz0000*.day`
    #[argh(option, short = 'c')]
    pub code: Option<String>,

    /// 可选。指定解析文件的数量。如果指定多个路径，则为每个路径下待解析的文件数量。
    #[argh(option, short = 'n')]
    pub amount: Option<usize>,

    /// 可选。指定 6 位代码来解析股票。此参数会覆盖掉 `--exchange` 和 `--amount`。使用
    /// `rustdx day -h l` 查看详细使用说明。
    #[argh(option, short = 'l')]
    pub stocklist: Option<String>,

    /// 可选。指定 6 位代码来解析股票。此参数会覆盖掉 `--exchange` 和 `--amount`。使用
    /// `rustdx day -h l` 查看详细使用说明。
    #[argh(option, short = 'o', default = "String::from(\"stocks.csv\")")]
    pub output: String,

    /// 可选。指定时，表示保存 csv 文件。只针对非 csv output 有效。
    #[argh(switch, short = 'k', long = "keep-csv")]
    pub keep_csv: bool,

    /// 可选。`day -e sh -l xlsx_path.xlsx -x 0`
    #[argh(option, short = 'x', long = "xlsx-col")]
    pub xlsx_col: Option<usize>,

    /// 指定一个或多个路径。使用空格分隔每个路径。
    #[argh(positional)]
    pub path: Vec<std::path::PathBuf>,

    /// 复权。
    #[argh(option, short = 'g')]
    pub gbbq: Option<std::path::PathBuf>,

    /// 复权。
    #[argh(option, short = 'p')]
    pub previous: Option<std::path::PathBuf>,

    /// 可选。显示详细的使用说明。
    #[argh(option, short = 'h')]
    description: Vec<String>,
}

pub type Stocklist = std::collections::HashSet<String>;

impl DayCmd {
    pub fn run(&self) -> Result<()> {
        match self.output.as_str() {
            "clickhouse" => crate::io::run_clickhouse(self),
            x if x.ends_with("csv") && self.gbbq.is_some() => {
                if self.previous.is_some() {
                    crate::io::run_csv_fq_previous(self)
                } else {
                    crate::io::run_csv_fq(self)
                }
            }
            x if x.ends_with("csv") => crate::io::run_csv(self),
            "mongodb" => crate::io::run_mongodb(self),
            _ => todo!(),
        }
    }

    pub fn help_info(&self) -> &Self {
        self.path.iter().map(|p| p.read_dir()).for_each(|p| println!("{:?}", p));
        for arg in &self.description {
            match arg.as_str() {
                "exchange" | "e" => println!("{}", DAYCMD_E),
                "stocklist" | "l" => println!("{}", DAYCMD_L),
                _ => println!("请查询以下参数之一：exchange e stocklist l\n使用 `-h e -h l` \
                               的形式查询多个参数的使用方法"),
            }
        }
        self
    }

    /// 匹配 `.day` 之前的内容：比如 `sz000001`
    pub fn stocklist(&self) -> Option<Stocklist> {
        use crate::io::{get_offical_stocks, read_xlsx};
        match (self.stocklist.as_deref(), self.exchange.as_deref(), self.xlsx_col) {
            (Some("official"), _, _) => get_offical_stocks("official"),
            (Some("sse"), _, _) => get_offical_stocks("sse"),
            (Some("szse"), _, _) => get_offical_stocks("szse"),
            (Some(ex), Some(prefix), _) if ex.len() == 6 || ex.contains(',') => {
                self.parse_list(prefix)
            }
            (Some(ex), Some("sz"), _) => read_xlsx(ex, 4, "sz"),
            (Some(ex), Some("sh"), _) => read_xlsx(ex, 0, "sh"),
            (Some(ex), None, Some(n)) => read_xlsx(ex, n, ""),
            (Some(ex), Some(prefix), Some(n)) => read_xlsx(ex, n, prefix),
            _ => self.parse_list(""),
        }
    }

    /// 筛选 sz/sh 交易所和股票代码的开头，并把代码转换为 u32
    /// 当 -e 为 auto 时，自动匹配 6 开头的股票为 sh，否则为 sz
    /// TODO: 移除转换成 u32 的代码
    pub fn filter_ec(&self, fname: &str) -> (bool, u32) {
        let len = fname.len();
        let code = &fname[len - 10..len - 4];
        let ex_f = &fname[len - 12..len - 10];
        let match_ex = |ex: &str| ex == ex_f || ex == "auto";
        let c = code.parse();
        (c.is_ok()
         && self.exchange.as_deref().map(match_ex).unwrap_or(true)
         && self.code.as_ref().map(|s| code.starts_with(s)).unwrap_or(true),
         c.unwrap_or(0))
    }

    fn parse_list(&self, p: &str) -> Option<Stocklist> {
        let prefix = |x: &str| format!("{}{}", auto_prefix(p, x), x);
        self.stocklist.as_ref().map(|s| s.split(',').map(prefix).collect())
    }
}

#[inline]
pub fn auto_prefix<'a>(prefix: &'a str, code: &'a str) -> &'a str {
    if prefix == "auto" && &code[0..1] == "6" {
        "sh"
    } else if prefix == "auto" {
        "sz"
    } else {
        prefix
    }
}

#[rustfmt::skip]
const DAYCMD_E: &'static str = "--exchange 或 -e ：
1. 如果提供 `--exchange` 或 `-e` 则表示指定交易所或者代码开头的文件：
 * `sz`：sz 开头的文件，而且只取 0 和 3 代码开头的 A 股文件，即 `sz[0|3]\\d{5}\\.day`。
 * `sh`：sh 开头的文件，而且只取 6 开头的 A 股文件，如 `sh6\\d{5}\\.day`。
 * 不超过 6 位的数字：可以指定多个数字，如 `6` => `*6\\d{5}\\.day`、`68` => `*68\\d{4}\\.day`。
 * 可以同时指定以上三种，但必须先指定 `sz/sh` 再指定代码开头，用逗号分隔：
   比如 `-e sz,00,sh,68` => `sz00\\d{4}\\.day` + `sh68\\d{4}\\.day`、
        `-e sz,00,sh,68,60` =>`sz00\\d{4}\\.day` + `sh6[8|0]\\d{4}\\.day`；
        不支持 `00,sh,68,60`这样格式，因为这会造成歧义：sh `00` 开头的 day 文件可能是指标数据。
 * `full`：解析所有 day 文件。
2. 不提供这个参数：会根据文件名自动指定为 `-e sz,0,3,sh,6` 格式。
3. 如果提供了 `--stocklist` ，这个参数所指定的内容直接被忽略。
";

#[rustfmt::skip]
const DAYCMD_L: &'static str = "--stocklist 或 -l ：
匹配 `.day` 之前的内容：比如 `sz000001`

`-l official` 从上交所和深交所官网获取 A 股、科创板、创业板股票代码列表
`-l sse` 从上交所官网获取 A 股、科创板股票代码列表
`-l szse` 从深交所官网获取 A 股、创业板股票代码列表

`-l excel_path.xls[x] -e sz` 从本地路径获取深交所官网下载的代码列表
  （或者第 4 (E) 列 6 为股票代码的 excel，代码开头会自动添 `sz`）
`-l excel_path.xls[x] -e sh` 从本地路径获取上交所官网下载的代码列表
  （或者第 0 (A) 列 6 为股票代码的 excel，代码开头会自动添 `sh` ）
* 或者更一般地：`-l excel_path.xls[x] -c n [-e prefix]` 表示
  识别 excel_path.xls[x] 文件第 n 列股票代码，如果需要添加前缀，则指定 -e，
  如果不需要添加前缀，则不需要 -e

`-l sz000001,sh688001` 逗号分隔的带 sh/sz 标识的代码字符串
`-l 000001,000002 -e sz` 等价于 `-l sz000001,sz000002`
`-l 688001,688002 -e sh` 等价于 `-l sh688001,sh688002`
* 或者更一般地： `-l 688001,688002 -e xx` 等价于 `-l xx688001,xx688002`

* 【设想】如果提供 txt 文件路径，则读取里面的六位代码数据。使用 `\\n` 分隔。
* 【设想】如果提供数据库路径，则使用数据库的股票代码。
";
