//Copyright 2010 Charles Lohr - under the NewBSD license.
//Generator for FontCharMap and FontCharData included in CNFGFunctions.c
#include <stdio.h>

unsigned CodePage437PlusUnicode[256][33] = {
	{ 0 },
	{ 5, 0, 0, 0, 1,  2, 0, 2, 1,  0, 3, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3 },
	{ 5, 0, 0, 0, 1,  2, 0, 2, 1,  0, 4, 2, 4,  2, 4, 1, 3,  1, 3, 0, 4 },
	{ 6, 0, 1, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1,  1, 1, 1, 2 },
	{ 6, 1, 1, 2, 2,  2, 2, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 2,  0, 2, 1, 1 },
	{ 6, 0, 1, 2, 1,  2, 1, 2, 3,  2, 3, 0, 1,  0, 3, 2, 1,  0, 3, 0, 1,  1, 2, 1, 4 },
	{ 6, 0, 3, 2, 3,  1, 3, 1, 4,  2, 3, 2, 2,  2, 2, 1, 1,  1, 1, 0, 2,  0, 2, 0, 3 },  //spades
	{ 1, 1, 2, 1, 2 },
	{ 5, 1, 2, 1, 2,  0, 1, 2, 1,  2, 1, 2, 3,  2, 3, 0, 3,  0, 3, 0, 1 },
	{ 0 }, //[tab]
	{ 0 }, //[\r]
	{ 6, 0, 3, 2, 1,  2, 1, 2, 2,  2, 1, 1, 1,  0, 3, 1, 4,  1, 4, 2, 3,  2, 3, 1, 2 },  //male 
	{ 6, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 0, 1,  1, 2, 1, 4,  0, 3, 2, 3 },  //female
	{ 0 }, //[\n]
	{ 5, 0, 2, 0, 3,  0, 3, 1, 3,  0, 2, 1, 2,  1, 3, 1, 0,  1, 0, 2, 1 },  //note
	{ 6, 0, 1, 2, 3,  0, 3, 2, 1,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 1, 3,  1, 3, 0, 2 },
	{ 3, 0, 0, 2, 2,  2, 2, 0, 4,  0, 4, 0, 0 },
	{ 3, 2, 0, 0, 2,  0, 2, 2, 4,  2, 4, 2, 0 }, //backwards sign (17)
	{ 5, 0, 1, 1, 0,  1, 0, 2, 1,  1, 0, 1, 4,  1, 4, 0, 3,  1, 4, 2, 3 },
	{ 4, 0, 0, 0, 3,  0, 4, 0, 4,  2, 0, 2, 3,  2, 4, 2, 4 },
	{ 5, 0, 0, 2, 0,  0, 0, 0, 2,  0, 2, 2, 2,  1, 0, 1, 4,  2, 0, 2, 4 },
	{ 7, 0, 1, 2, 1,  2, 1, 2, 3,  2, 3, 0, 3,  0, 3, 0, 1,  2, 0, 1, 0,  1, 0, 1, 4,  1, 4, 0, 4 },
	{ 4, 0, 3, 2, 3,  2, 3, 2, 4,  2, 4, 0, 4,  0, 4, 0, 3 }, // lowblock
	{ 6, 0, 1, 1, 0,  1, 0, 2, 1,  1, 0, 1, 4,  1, 4, 0, 3,  1, 4, 2, 3,  0, 4, 2, 4 },
	{ 3, 0, 1, 1, 0,  2, 1, 1, 0,  1, 0, 1, 4 },
	{ 3, 0, 3, 1, 4,  2, 3, 1, 4,  1, 0, 1, 4 },
	{ 3, 0, 2, 2, 2,  2, 2, 1, 1,  2, 2, 1, 3 },
	{ 3, 0, 2, 2, 2,  0, 2, 1, 1,  0, 2, 1, 3 },
	{ 2, 0, 1, 0, 2,  0, 2, 2, 2 }, //micro-l
	{ 5, 0, 2, 2, 2,  2, 2, 1, 1,  1, 1, 0, 2,  0, 2, 1, 3,  1, 3, 2, 2 },
	{ 3, 1, 1, 2, 2,  2, 2, 0, 2,  0, 2, 1, 1 },
	{ 3, 0, 2, 1, 3,  1, 3, 2, 2,  2, 2, 0, 2 },
	{ 0 }, //[space]  (32)
	{ 2, 1, 0, 1, 3,  1, 4, 1, 4 },
	{ 2, 1, 0, 0, 1,  2, 0, 1, 1 },
	{ 4, 1, 0, 1, 4,  2, 0, 2, 4,  0, 1, 2, 1,  0, 3, 2, 3 },
	{ 6, 2, 1, 1, 0,  1, 0, 0, 1,  0, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  1, 0, 1, 4 },
	{ 5, 0, 0, 0, 1,  2, 3, 2, 4,  0, 4, 0, 3,  0, 3, 2, 1,  2, 1, 2, 0 },
	{ 7, 2, 1, 1, 0,  1, 0, 0, 1,  0, 1, 1, 2,  1, 2, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3,  0, 2, 2, 4 },
	{ 1, 1, 0, 1, 1 },
	{ 3, 1, 0, 0, 1,  0, 1, 0, 3,  0, 3, 1, 4 },
	{ 3, 1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4 },  // )
	{ 4, 0, 1, 2, 3,  1, 1, 1, 3,  2, 1, 0, 3,  0, 2, 2, 2 },
	{ 2, 0, 2, 2, 2,  1, 1, 1, 3 },
	{ 1, 0, 4, 1, 3 },  // ,
	{ 1, 0, 2, 2, 2 },
	{ 1, 1, 4, 1, 4 },
	{ 1, 0, 3, 2, 1 },
	{ 5, 0, 0, 2, 0,  2, 0, 2, 4,  2, 4, 0, 4,  0, 4, 0, 0,  1, 2, 1, 2 },
	{ 3, 0, 1, 1, 0,  1, 0, 1, 4,  0, 4, 2, 4 },
	{ 5, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 2,  2, 2, 0, 4,  0, 4, 2, 4 },
	{ 4, 0, 0, 2, 0,  2, 0, 2, 4,  2, 4, 0, 4,  1, 2, 2, 2 },
	{ 3, 0, 0, 0, 2,  0, 2, 2, 2,  2, 0, 2, 4 },
	{ 5, 2, 0, 0, 0,  0, 0, 0, 2,  0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4 },
	{ 5, 2, 0, 0, 2,  0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2 },  //6
	{ 4, 0, 0, 2, 0,  2, 0, 2, 1,  2, 1, 1, 2,  1, 2, 1, 4 },
	{ 5, 0, 0, 0, 4,  0, 0, 2, 0,  2, 0, 2, 4,  0, 4, 2, 4,  0, 2, 2, 2 },
	{ 5, 0, 0, 0, 2,  0, 2, 2, 2,  2, 2, 2, 0,  2, 0, 0, 0,  2, 2, 0, 4 },
	{ 2, 1, 1, 1, 1,  1, 3, 1, 3 },
	{ 2, 1, 1, 1, 1,  1, 3, 0, 4 },
	{ 2, 2, 0, 0, 2,  0, 2, 2, 4 },
	{ 2, 0, 1, 2, 1,  0, 3, 2, 3 },
	{ 2, 0, 0, 2, 2,  2, 2, 0, 4 },
	{ 5, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 1, 3,  1, 4, 1, 4 }, //?
	{ 7, 2, 1, 0, 1,  0, 1, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 1, 2,  1, 2, 1, 3,  1, 3, 2, 3 },
	{ 5, 0, 4, 0, 1,  0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 4,  0, 2, 2, 2 },  //A
	{ 8, 0, 0, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 3, 1, 2,  1, 2, 0, 2,  1, 2, 2, 1,  2, 1, 1, 0,  1, 0, 0, 0 },  //B
	{ 5, 2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1,  0, 1, 1, 0,  1, 0, 2, 1 }, //C
	{ 6, 0, 0, 1, 0,  1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 4,  0, 4, 0, 0 }, //D
	{ 4, 0, 0, 0, 4,  0, 4, 2, 4,  0, 0, 2, 0,  0, 2, 1, 2 },
	{ 3, 0, 0, 0, 4,  0, 0, 2, 0,  0, 2, 1, 2 },
	{ 7, 2, 1, 1, 0,  1, 0, 0, 1,  0, 1, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3,  2, 3, 2, 2,  2, 2, 1, 2 },
	{ 3, 0, 0, 0, 4,  2, 0, 2, 4,  0, 2, 2, 2 },
	{ 3, 0, 0, 2, 0,  1, 0, 1, 4,  0, 4, 2, 4 },
	{ 4, 0, 0, 2, 0,  2, 0, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3 },
	{ 6, 0, 0, 0, 4,  0, 2, 1, 2,  1, 2, 2, 1,  2, 1, 2, 0,  1, 2, 2, 3,  2, 3, 2, 4 },  //K
	{ 2, 0, 0, 0, 4,  0, 4, 2, 4 },
	{ 4, 0, 4, 0, 0,  0, 0, 1, 1,  1, 1, 2, 0,  2, 0, 2, 4 },
	{ 3, 0, 4, 0, 0,  0, 0, 2, 2,  2, 0, 2, 4 },
	{ 6, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1 }, //O
	{ 5, 0, 0, 0, 4,  0, 0, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 0, 2 },
	{ 7, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1,  0, 4, 1, 3 },
	{ 6, 0, 0, 0, 4,  0, 0, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 0, 2,  0, 2, 2, 4 },
	{ 5, 2, 1, 1, 0,  1, 0, 0, 1,  0, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3 },
	{ 2, 0, 0, 2, 0,  1, 0, 1, 4 },
	{ 3, 0, 0, 0, 4,  0, 4, 2, 4,  2, 4, 2, 0 },
	{ 4, 0, 0, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3,  2, 3, 2, 0 },
	{ 4, 0, 0, 0, 4,  0, 4, 2, 4,  1, 4, 1, 3,  2, 4, 2, 0 },
	{ 6, 0, 0, 0, 1,  0, 1, 2, 3,  2, 3, 2, 4,  0, 4, 0, 3,  0, 3, 2, 1,  2, 1, 2, 0 },
	{ 5, 0, 0, 0, 1,  0, 1, 1, 2,  1, 2, 1, 4,  1, 2, 2, 1,  2, 1, 2, 0 },
	{ 4, 0, 0, 2, 0,  2, 0, 0, 2,  0, 2, 0, 4,  0, 4, 2, 4 },
	{ 3, 1, 0, 0, 0,  0, 0, 0, 4,  0, 4, 1, 4 },
	{ 1, 0, 1, 2, 3 },
	{ 3, 1, 0, 2, 0,  2, 0, 2, 4,  2, 4, 1, 4 }, //]
	{ 2, 0, 1, 1, 0,  1, 0, 2, 1 },
	{ 1, 0, 4, 2, 4 },
	{ 1, 0, 0, 1, 1 },
	{ 5, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2 }, //a
	{ 4, 0, 0, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2 },
	{ 4, 2, 4, 0, 4,  0, 4, 0, 3,  0, 3, 1, 2,  1, 2, 2, 2 }, //c
	{ 4, 2, 0, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2 },
	{ 5, 2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2,  2, 2, 2, 3,  2, 3, 1, 3 }, //e
	{ 4, 0, 4, 0, 1,  0, 2, 1, 2,  0, 1, 1, 0,  1, 0, 2, 1 },
	{ 7, 2, 3, 1, 2,  1, 2, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3,  2, 3, 2, 4,  2, 4, 1, 5,  1, 5, 0, 4 },
	{ 4, 0, 0, 0, 4,  0, 3, 1, 2,  1, 2, 2, 3,  2, 3, 2, 4 },
	{ 2, 1, 1, 1, 1,  1, 2, 1, 4 },  //i
	{ 4, 2, 2, 2, 2,  2, 3, 2, 4,  2, 4, 1, 5,  1, 5, 0, 4 },  //j
	{ 4, 0, 0, 0, 4,  0, 3, 1, 3,  1, 3, 2, 2,  1, 3, 2, 4 },
	{ 1, 1, 0, 1, 4 },
	{ 5, 0, 2, 0, 4,  0, 2, 1, 3,  1, 2, 1, 4,  1, 2, 2, 3,  2, 3, 2, 4 },
	{ 4, 0, 2, 0, 4,  0, 3, 1, 2,  1, 2, 2, 3,  2, 3, 2, 4 }, //n
	{ 4, 0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2 }, //o
	{ 4, 0, 2, 0, 5,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2 }, // p
	{ 4, 0, 2, 0, 4,  0, 4, 2, 4,  2, 5, 2, 2,  2, 2, 0, 2 }, // q
	{ 3, 0, 2, 0, 4,  0, 3, 1, 2,  1, 2, 2, 2 }, //r
	{ 5, 2, 2, 0, 2,  0, 2, 0, 3,  0, 3, 2, 3,  2, 3, 2, 4,  2, 4, 0, 4 }, //s
	{ 2, 1, 1, 1, 4,  0, 2, 2, 2 }, //t < really is t>
	{ 4, 0, 2, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 4, 2, 2 }, //u
	{ 4, 0, 2, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3,  2, 3, 2, 2 }, //v
	{ 5, 0, 2, 0, 3,  0, 3, 1, 4,  1, 4, 1, 2,  1, 3, 2, 4,  2, 4, 2, 2 }, //w
	{ 2, 0, 2, 2, 4,  0, 4, 2, 2 }, //x
	{ 4, 0, 2, 0, 3,  0, 3, 1, 4,  2, 2, 2, 3,  2, 3, 0, 5 }, //y
	{ 3, 0, 2, 2, 2,  2, 2, 0, 4,  0, 4, 2, 4 }, //z
	{ 4, 2, 0, 1, 0,  1, 0, 1, 4,  1, 4, 2, 4,  1, 2, 0, 2 },
	{ 2, 1, 0, 1, 1,  1, 3, 1, 4 },
	{ 4, 0, 0, 1, 0,  1, 0, 1, 4,  1, 4, 0, 4,  1, 2, 2, 2 }, //}
	{ 3, 0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 },
	{ 5, 0, 3, 0, 4,  0, 4, 2, 4,  2, 4, 2, 3,  2, 3, 1, 2,  1, 2, 0, 3 },  //house (end of base charset)

	{ 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 },
	{ 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 },
	{ 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 },
	{ 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 },

	//Now, at 161
	{ 2, 1, 0, 1, 0,  1, 1, 1, 4 }, //! upside down
	{ 6, 2, 1, 1, 0,  1, 0, 0, 1,  0, 1, 0, 2,  0, 2, 1, 3,  1, 3, 2, 2,  1, 0, 1, 3 },  //cents
	{ 5, 1, 1, 0, 0,  0, 0, 0, 4,  0, 4, 2, 4,  2, 4, 2, 3,  0, 2, 1, 2 }, //pound
	{ 8, 0, 1, 0, 2,  1, 1, 2, 1,  2, 2, 2, 3,  0, 3, 1, 3,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 1, 3,  1, 3, 0, 2 }, //spindle
	{ 5, 0, 0, 1, 1,  1, 1, 2, 0,  1, 1, 1, 4,  0, 2, 2, 2,  0, 3, 2, 3 }, //Yen
	{ 2, 1, 0, 1, 2,  1, 3, 1, 5 },	//big bars
	{ 8, 2, 0, 0, 0,  0, 0, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  0, 2, 1, 3,  1, 3, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4 }, //Subsection
	{ 2, 0, 0, 0, 0,  2, 0, 2, 0 }, //Diaeresis
	//{ 8, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 2,  2, 2, 1, 3,  1, 3, 0, 2,  0, 2, 0, 1,  0, 2, 1, 1,  0, 2, 1, 2 }, //Copyright
	{ 3, 2, 0, 1, 0,  1, 0, 1, 1,  1, 1, 2, 1 }, //copyright
	{ 4, 1, 0, 2, 1,  2, 0, 2, 1,  2, 1, 1, 1,  1, 1, 1, 0 }, //small a w/ bar underneath
	{ 4, 1, 1, 0, 2,  0, 2, 1, 3,  2, 1, 1, 2,  1, 2, 2, 3 }, //Brittish quote left
	{ 2, 0, 1, 2, 1,  2, 1, 2, 2 }, // not sign
	{ 1, 1, 2, 2, 2 }, //tiny hyphen
	{ 5, 1, 0, 2, 0,  2, 0, 2, 1,  2, 1, 1, 1,  1, 2, 1, 0, 1, 1, 2, 2 }, //Registered trademark
	{ 1, 0, 0, 2, 0 }, //Macron
	{ 4, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 0, 1 }, //Degree sign
	{ 3, 0, 2, 2, 2,  1, 1, 1, 3,  0, 3, 2, 3 }, //Plus/minus
	{ 4, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 0, 3,  0, 3, 2, 3 }, //tiny two
	{ 6, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  1, 2, 0, 2,  1, 2, 2, 3,  2, 3, 0, 3 }, //tiny three
	{ 1, 1, 1, 2, 0 }, //Accent

	//Now at 181
	{ 4, 0, 2, 0, 5,  0, 4, 1, 4,  1, 4, 2, 3,  2, 2, 2, 4 }, //myuu
	{ 5, 1, 4, 1, 0,  1, 0, 0, 1,  0, 1, 1, 2,  1, 0, 2, 0,  2, 0, 2, 4 }, //Paragraph
	{ 1, 1, 2, 1, 2 }, //middle dot
	{ 2, 1, 4, 1, 5,  1, 5, 0, 5 }, //cedilla
	{ 1, 2, 0, 2, 1 },  //superscript 1
	{ 4, 1, 0, 2, 0,  2, 0, 2, 1,  2, 1, 1, 1,  1, 1, 1, 0 }, //superscript 0
	{ 4, 0, 1, 1, 2,  1, 2, 0, 3,  1, 1, 2, 2,  2, 2, 1, 3 },  //Brittish right side thingie
	{ 5, 0, 0, 0, 1,  0, 2, 2, 0,  1, 2, 1, 3,  1, 3, 2, 3,  2, 2, 2, 4 }, //1/4
	{ 6, 0, 0, 0, 1,  0, 2, 2, 0,  1, 2, 2, 2,  2, 2, 1, 3,  1, 3, 1, 4,  1, 4, 2, 4}, //1/2
	{ 8, 0, 0, 1, 0,  1, 0, 1, 1,  1, 1, 0, 1,  1, 1, 0, 2,  0, 2, 2, 0,  1, 2, 1, 3,  1, 3, 2, 3,  2, 2, 2, 4 }, //3/4
	{ 5, 1, 0, 1, 0,  1, 1, 1, 2,  1, 2, 0, 3,  0, 3, 1, 4,  1, 4, 2, 3 }, //? upside down
	{ 6, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  1, 0, 2, 1 }, //A, unaccent
	{ 6, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  0, 1, 1, 0 }, //A, accent
	{ 7, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  0, 1, 1, 0,  1, 0, 2, 1 }, //A, hat
	{ 8, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 }, //A, tilde
	{ 7, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  0, 0, 0, 0,  2, 0, 2, 0 }, //A, dots
	{ 8, 0, 4, 0, 2,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 2, 4,  0, 3, 2, 3,  0, 0, 2, 0,  2, 0, 1, 1,  1, 1, 0, 0 }, //A, circle
	{ 6, 0, 0, 0, 4,  0, 2, 2, 2,  0, 0, 1, 1,  1, 0, 1, 4,  1, 0, 2, 0,  1, 4, 2, 4 }, //AE
	{ 7, 2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1,  0, 1, 1, 0,  1, 0, 2, 1,  1, 4, 1, 5,  1, 5, 0, 5 },  //C with hook
	{ 5, 0, 2, 2, 2,  0, 2, 0, 4,  0, 4, 2, 4,  0, 3, 1, 3,  0, 0, 1, 1 },  //E with unaccent

	//Now, at 201
	{ 5, 0, 2, 2, 2,  0, 2, 0, 4,  0, 4, 2, 4,  0, 3, 1, 3,  1, 1, 2, 0 },  //E with accent
	{ 6, 0, 2, 2, 2,  0, 2, 0, 4,  0, 4, 2, 4,  0, 3, 1, 3,  0, 1, 1, 0,  1, 0, 2, 1 }, //E with hat
	{ 6, 0, 2, 2, 2,  0, 2, 0, 4,  0, 4, 2, 4,  0, 3, 1, 3,  0, 0, 0, 0,  2, 0, 2, 0 }, //E with dots
	{ 4, 0, 2, 2, 2,  1, 2, 1, 4,  0, 4, 2, 4,  0, 0, 1, 1 }, //I with unaccent
	{ 4, 0, 2, 2, 2,  1, 2, 1, 4,  0, 4, 2, 4,  1, 1, 2, 0 }, //I with accent
	{ 5, 0, 2, 2, 2,  1, 2, 1, 4,  0, 4, 2, 4,  0, 1, 1, 0,  1, 0, 2, 1 }, //I with hat
	{ 5, 0, 2, 2, 2,  1, 2, 1, 4,  0, 4, 2, 4,  2, 0, 2, 0,  0, 0, 0, 0 }, //I with dots
	{ 7, 0, 0, 1, 0,  1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 4,  0, 4, 0, 0,  0, 2, 1, 2 }, //eth
	{ 6, 0, 4, 0, 2,  0, 2, 2, 4,  2, 4, 2, 2,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 }, //N with tilde
	{ 5, 0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2,  0, 0, 1, 1 },  //O with unaccent
	{ 5, 0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2,  1, 1, 2, 0 },  //O with accent
	{ 6, 0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2,  1, 1, 2, 0,  0, 0, 1, 1 },  //O with hat
	{ 7, 0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 },  //O with tilde
	{ 6, 0, 1, 2, 1,  2, 1, 2, 4,  2, 4, 0, 4,  0, 4, 0, 1,  0, 0, 0, 0,  2, 0, 2, 0 },  //O with dots
	{ 2, 0, 1, 2, 3,  0, 3, 2, 1 }, //Multiplication symbol
	{ 7, 0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 2, 3,  2, 3, 1, 4,  1, 4, 0, 3,  0, 3, 0, 1,  0, 3, 2, 1 }, //O with stroke
	{ 4, 0, 1, 0, 4,  0, 4, 2, 4,  2, 4, 2, 1,  1, 1, 2, 0 },  //U with unaccent
	{ 4, 0, 1, 0, 4,  0, 4, 2, 4,  2, 4, 2, 1,  0, 0, 1, 1 },  //U with unaccent
	{ 5, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  0, 1, 1, 0,  1, 0, 2, 1 },  //U with hat
	{ 5, 0, 1, 0, 4,  0, 4, 2, 4,  2, 4, 2, 1,  0, 0, 0, 0,  2, 0, 2, 0 },  //U with dots
	
	//221
	{ 6, 0, 1, 0, 2,  0, 2, 1, 3,  1, 3, 1, 4,  1, 3, 2, 2,  2, 2, 2, 1,  1, 1, 2, 0 }, // Y with accent
	{ 5, 0, 0, 0, 4,  0, 1, 1, 1,  1, 1, 2, 2,  2, 2, 1, 3,  1, 3, 0, 3 }, //Capitol thorn
	{ 7, 0, 0, 0, 5,  0, 0, 1, 0,  1, 0, 2, 1,  2, 1, 1, 2,  0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4 }, //Beta
	{ 6, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  0, 0, 1, 1 }, //a with unaccent
	{ 6, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  1, 1, 2, 0 }, //a with accent
	{ 7, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  0, 1, 1, 0,  1, 0, 2, 1 }, //a with hat
	{ 8, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 }, //a with tilde
	{ 7, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  0, 1, 0, 1,  2, 1, 2, 1 }, //a with dots
	{ 8, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 3, 1, 2,  1, 2, 0, 2,  0, 1, 1, 0,  1, 0, 2, 1,  2, 1, 0, 1 }, //a with circle
	{ 7, 0, 2, 1, 3,  0, 2, 0, 4,  0, 4, 2, 4,  1, 2, 1, 4,  1, 2, 2, 2,  1, 3, 2, 3,  2, 2, 2, 3 }, // ae
	{ 6, 0, 3, 0, 4,  0, 4, 2, 4,  0, 3, 1, 2,  1, 2, 2, 2,  1, 4, 1, 5,  1, 5, 0, 5 }, //c with hook
	{ 6, 2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2,  2, 2, 2, 3,  2, 3, 1, 3,  0, 0, 1, 1 }, //e with unaccent
	{ 6, 2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2,  2, 2, 2, 3,  2, 3, 1, 3,  1, 1, 2, 0 }, //e with accent
	{ 7, 2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2,  2, 2, 2, 3,  2, 3, 1, 3,  0, 1, 1, 0,  1, 0, 2, 1 }, //e with hat
	{ 7, 2, 4, 0, 4,  0, 4, 0, 2,  0, 2, 2, 2,  2, 2, 2, 3,  2, 3, 1, 3,  0, 1, 0, 1,  2, 1, 2, 1 }, //e with dots
	{ 2, 1, 2, 1, 4,  0, 0, 1, 1 }, //i with unaccent
	{ 2, 1, 2, 1, 4,  1, 1, 2, 0 }, //i with accent
	{ 3, 1, 2, 1, 4,  0, 1, 1, 0,  1, 0, 2, 1 }, //i with hat
	{ 3, 1, 2, 1, 4,  0, 1, 0, 1,  2, 1, 2, 1 }, //i with dots
	{ 6, 0, 0, 2, 2,  1, 1, 2, 0,  0, 2, 2, 2,  2, 2, 2, 4,  2, 4, 0, 4,  0, 4, 0, 2  }, //little eth

	//241
	{ 7, 0, 2, 0, 4,  0, 3, 1, 2,  1, 2, 2, 3,  2, 3, 2, 4,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 }, //n with tilde
	{ 5, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  0, 0, 1, 1 }, //o with unaccent
	{ 5, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  1, 1, 2, 0 }, //o with accent
	{ 6, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  0, 1, 1, 0,  1, 0, 2, 1 }, //o with hat
	{ 7, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  0, 1, 1, 0,  1, 0, 1, 1,  1, 1, 2, 0 }, //o with tilde
	{ 6, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  0, 1, 0, 1,  2, 1, 2, 1 }, //o with dots
	{ 3, 1, 1, 1, 1,  0, 2, 2, 2,  1, 3, 1, 3 }, // division
	{ 5, 0, 2, 0, 4,  0, 4, 2, 4,  2, 4, 2, 2,  2, 2, 0, 2,  0, 4, 2, 2 }, //o, crossed out
	{ 5, 0, 2, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 4, 2, 2,  0, 0, 1, 1 }, //u with unaccent
	{ 5, 0, 2, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 4, 2, 2,  1, 1, 2, 0 }, //u with accent
	{ 6, 0, 2, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 4, 2, 2,  0, 1, 1, 0,  1, 0, 2, 1 }, //u with hat
	{ 6, 0, 2, 0, 4,  0, 4, 1, 4,  1, 4, 2, 3,  2, 4, 2, 2,  0, 1, 0, 1,  2, 1, 2, 1 }, //u with dots
	{ 5, 0, 2, 0, 3,  0, 3, 1, 4,  2, 2, 2, 3,  2, 3, 0, 5,  1, 1, 2, 0 }, //y with accent
	{ 5, 0, 0, 0, 4,  0, 2, 1, 1,  1, 1, 2, 2,  2, 2, 1, 3,  1, 3, 0, 2 }, // little thorn
	{ 6, 0, 2, 0, 3,  0, 3, 1, 4,  2, 2, 2, 3,  2, 3, 0, 5,  0, 1, 0, 1,  2, 1, 2, 1 }, //y with dots
};
/*
void DrawText( const char * text, XSegment * buf, int * pos, int maxpos, int size )
{
	float iox = xo;
	float ioy = yo;
	if( *pos == maxpos ) return;

	int place = 0;
	while( text[place] )
	{
		unsigned char c = text[place];

		switch( c )
		{
		case 9:
			iox += 12 * size;
			break;
		case 10:
			iox = xo;
			ioy += 6 * size;
			break;
		default:
			unsigned * lmap = CodePage437PlusUnicode[c];
			for( unsigned i = 0; i < lmap[0]; i++ )
			{
				buf[*pos].x1 = lmap[i*4+1]*size + iox;
				buf[*pos].y1 = lmap[i*4+2]*size + ioy;
				buf[*pos].x2 = lmap[i*4+3]*size + iox;
				buf[*pos].y2 = lmap[i*4+4]*size + ioy;
				(*pos)++;
				if( *pos == maxpos ) return;
			}
			iox += 3 * size;
		}
		place++;
	}
}

*/

int main()
{
	int TMapPos = 0;
	unsigned char TMap[20000];
	unsigned short Offsets[256]; 
	unsigned c;
	for( c = 0; c < 256; c++ )
	{
		//Synthesize bitstream for this character.
		unsigned char BitsToMap[256];
		
		unsigned * ttm = CodePage437PlusUnicode[c];
		if( ttm[0] > 256 )
		{
			printf( "Character %d too complicated.\n", c );
			return -1;
		}

		unsigned int codelen = ttm[0]*2;
		if( codelen == 0 )
		{
			Offsets[c] = 65535;
			continue;
		}

		int i;
		for( i = 0; i < codelen; i++ )
		{
			unsigned x = ttm[i*2+1];
			unsigned y = ttm[i*2+2];
			unsigned char code = ( x << 4 ) | y;

			if( i == codelen - 1 )
				code |= 0x80;

			BitsToMap[i] = code;
		}
//		printf( "%d -- %d\n", c, codelen );

		bool bFound = false;
		//Now, search for code in map.
		fprintf( stderr, "Code for %d: length %d\n", c, codelen );
		for( i = 0; i < (int)(TMapPos - codelen + 1); i++ )
		{
//			printf( "%d / %d - %d + 1 (%d) (%d)\n", i, TMapPos, codelen,TMapPos - codelen + 1,  i < TMapPos - codelen + 1 );
			unsigned j;
			for( j = 0; j < codelen; j++ )
			{
//				printf( "%d\n", j );
				if( TMap[j+i] != BitsToMap[j] ) break;
			}

			//Found?
			if( j == codelen ) 
			{
				Offsets[c] = i;
				bFound = true;
				break;
			}
		}
		if( !bFound )
		{
			fprintf( stderr, "no matching pattern found.  Copying.\n" );
			Offsets[c] = TMapPos;
			for( unsigned j = 0; j < codelen; j++ )
				TMap[TMapPos++] = BitsToMap[j];
		}
		fprintf( stderr, "match found.\n" );
	}

	printf( "const unsigned short FontCharMap[256] = {" );
	for( c = 0; c < 256; c++ )
	{
		if( c % 16 == 0 )
		{
			printf( "\n\t" );
		}
		printf( "%d, ", Offsets[c] );
	}
	printf( "};\n\n" );
	printf( "const unsigned char FontCharData[%d] = {", TMapPos );
	for( unsigned i = 0; i < TMapPos; i++ )
	{
		if( i % 16 == 0 ) 
			printf( "\n\t" );
		printf( "0x%02x, ", TMap[i] );
	}
	printf( "};\n\n" );
}
