use std::time::Duration;

use criterion::{black_box, criterion_group, criterion_main, Criterion};
use lib::{wavefront_alignment::wavefront_align, alignment_lib::Penalties};

fn wavefront_bench_l100_e1(c: &mut Criterion) {
    let query = "ACTCTATTTTACTCAGTGCAGGGTGAGCCGCCTATGCGGAGTGCAGTTACATAGGGAAAGCGGGGCTCAATTGCTACTCGTATGGGGTGTCACAGACGC";
    let text = "ACTCTATTTTACTCAGTGCAGGGTGAGCCGCCTATGCGGAGTGCAGTTACATAGGGTAAAGCGGGGCTCAATTGCTACTCGTATGGGGTGTCACAGACGC";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 100 1% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l100_e10(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCG";
    let text = "TTTTTGCCTCGAATCTGAAGTGCGCTGCCACAGAACTGGAGATTAGCATAGGGGGCAAGTGAACCATCCCCTTGGCGATCCGGAATAAGTTGACAACCGGTCG";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 100 10% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l100_e30(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCG";
    let text = "TTTTTGCCTCGGAATCCGAAGTGCGCCTGCCACAGAACTGCAGATTAGCAATAGGGGGCAAGTGAGCCATCACCTTTCCGGCGATCCGGGAATGTTGACAACCGGTCG";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 100 30% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l1000_e1(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGA";
    let text = "TTTTTGACTCGAATGAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGCGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATAGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTAAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTAAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGCGATGCAAACCAACCAGACTCGGTCGA";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 1000 1% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l1000_e10(c: &mut Criterion) {
    let query = "TTTTGACTCGAATGAAGTGCTGCTGCCGCAGAACCTGGAGATTCAGGGGCTAAGGAACCATCCCCTTGCACGATACGGAATAAGTTGACAAACCGGTGCAGGGTGGAACAACACATATTCTATCTCAAACCTACGGTGGTTGTAGCCTGCAGTTGAACTCGGTCCGGACCTCATGCGCGGTCGGAGAAAAGTGCATTTCCTTCGGGAACTTGGTTATACTGAGTCCTACCGAATTGCAGTATGGGCGGCACTAATCCTATAAAATTGTGCTCAATTGATATTCTTTCGGAGTCTATCAGGACAAGAGCGTTGAGGCGCCTTTAGCCTGAGTGTATCGCCCACAGTTTATTACTCTTAACTGATTTCCCGTCTTTGGTGCGTGGGAAGACTTTTGTAAAGTCGTCACGTAAGGCCGGTGTTGTTGCCGCGGTCTGACTGGGCGGTCAAGTGGCCCAAAGGTACCAGCGTGGCACATAGCACAGTCACGCGTAACAGGTAAGGGATTATGCTTCGTTGCGTTGACTCACACATTAGTCCCGCCATCACGCGCAATACACCGAGAGGAACCCCCTGTGCCCGTCTGAGCTGAGTTACTCGCGATGCGAATCGGACACGTCGGCCCACGTATGGGTCAGTAGTGCCCGTGTGCCAATGCAGACCGATTGTCTCCAAATGGCCCCTTATAGATTGATGACCCATTCTTAGGCTTTTGCGTCAGGTCCTATGGAGTACAACAGTGCTCACTAAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACCGCGGAGTAGCATTCGCTATAGTAAAGGACAGACCACCGACACCGGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGGAGTATAGTAGTCAAACCCTGGCGATACGCCGCCAAGACACCTAGTATCCCAAACTCCCTGTCGACGCCACGCGACGGCGGCGAGTGCAAACCAACCAGACTTCGGTCGA";
    let text = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGA";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 1000 10% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l1000_e30(c: &mut Criterion) {
    let query = "TTTTGACTCGGATGAGATGCTGCTGCCCAGATCCGAGATCAGGGGCCTATAGGAACCATCCCCTTGCCACGATAGATAAGTTGCAAACGCGGTGCAGGGTGGAACAACATGTATTACTACCTCAAACCTACGGTGGTTGTAGCCTGCAGTTGAACTCGGTCCTGACCTCATGCGCGCGTCGGAGTAAAATTCCCATTTCTTCGGGAACTGGGTATTCTAGTCACTACACGAATTCAGTATGGGCGGCACCTAATTTCTATAAAATTTGTCGAATTGCTATTTTTTGGAGGTCTGACCAGGACAGAGCGTTGAGACGCTTCACCTGAGTGTATCGCCCACAGTTATAACTCTTAACTGATTTCCCGTCTTTGGTGCGTGGGAGACTTTGAAAGTCGTCACGGTAAGGCCGAGTGTTGTTCCCCGCGGTTCGACTGGGCGGCAAGTGGACCAAAGGCACAGCGTGGCCACATACCACAGTTACGCGTAACAGGTAAGGGATTATGCTGTGTTGCGTTGATCACACATGAGCACCGCCTATAGCCCAAAAACAGCAGAGGAACCCCCTTGCCCGTCTCAGTTGAGTCTCCTCGCGAGCGAATCGGGGACACGTCGGCCCACGTATGGGTCGTAGTGCCCGTGTGCCACGAGCGCAGACACGATTGGCTACAAATGGCCCCTTATAGATTAATGCCCATTCTTAGGCTTTCGCGTCACGCCTATCGGTAGCAAACAGTGCTATCTAGGGCTCTTAGTAGTGGTTGCTCTCCTATGGTAGTGAGGGTGGGGAACCGCGGAGTAGCATTCGCTATAGTTAAAGGGGCAGCCACGGACACCGGCTACAAAAGAACAATTTTGAAACGTCTAGGACTCCACCCAGACGCATAAGGTATAAGAGTCAAAACCCTGGCGATACGGCCCCAGACACCTAGTTATCCCAAACCCCTGCGACGCCACGCGCCCCGGCGAAGTGCAAACCCACTACACCCGGTCGC";
    let text = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGA";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 1000 30% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l10000_e1(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGATCCGAATATAAAACCCCTGGCTTCAGATTGATAGGAGCAGCAGGGATATGGACGCGAGGAAGACAGCTACGGGAAAAACAGTAACGGGAATCTTGTGTTGCCAAGCCTCGTACTATGTCGACATAGTTGGCTAGAGTCGCTCCCAGATCGGGTGCAAATTTCTTATGAAACACTCGCGGTAACTGGATTTGGCGGCCACGCGGGCACTGTGTTTGGGTGCGCTATCCAGTTTGCGAGATCTCATCTGAGACCCCCATGCCGACAGTACTTGTCGTTGCTAGCCAGTCGCGATTCAGCATCTGCCCAATGAAACGAACCGTAGGAATTGTTTTCAATCCGTCTATCCTGTGCACACGTAAGTACCCGAGATGCGTCTAACATTACTCTTGTGCTAGCGTTGGGAACCTGGCTGCTTCCGAAGTCCTATGTGCCCCCGTCCGTAAAATGGCGGTCACCAATATACAGTTGCAACCACCTATCGGCAGGGGTCTGGCTAAGTGTTTATCGTCGCAAGCGATCACCTTGGAGCTCAGCCGGTGACTACCGAGCGGGTCGCAGACGGCCTGTCGCCTCGCGAGACCGAAATGACTCACTCAGCCCTTAACGGGAGACCCGGACGGAAGCATCTCTAATCACTGTACAGGCCCCGTTGAGACGGATCCTCGTGGGGTTGGGTTTACTAATGCCATGACTTGGTACCGCACTGAGGCACCCCAGGCCACGAACTAGACACCCAAGTGTATCCCCATGCATAAAGTAAAGCCTTGACGTCGGGATTATGGTGTATCATCAGGGGGAGTTCACATCAGCAATATTAGAAGGCACTGTAATTGAAGTGTCATGATGTAATAGCGGGCAGCCCCACGTAAGGAGCCGTAGTATCCCAAGCGATTGTCAGCACTCTAGCATATTTCCAGGTTAATAGGCGCTTGGAGTTATGCTGACTATCATGAGTTACCGAACTCAGCCGAGCGGTGTGAACTATGAAACTAGATAAGGCAGTGAGCTCATCCGGCCGGCACGTGGAGAGACAAACGCAACAGTTACCTATAGCAATATAGTGTAGCGTGTCGCCGGGTGTGGCATCATACGATCACGTCAGGTTAGGTCAGATTCGAGGCTACGGGTCCAGTTGTAAGTGGCGGAAACGTGCTAAGGATACGTAATAACAGACCTGGTAATTGGTGAGTAGAGGCGGGTTGGATTTGGTGGGCCCCAGATGGTCCAATGATCGTCGTTAGGTCAAGCCCCAAAACCCGAAGTCATGTGGTCAGTATGGATTCTACCCGTCGGCTTAGGTAGAAATCAGTACTGCTTGCGGTAAAATCGGATACACCCCTCGAAACAAACGGAGTCTTGGCGCCTCTCTAAATACTGCTGTCCTAAATTGGCCATCGGGCGGGCTTCGCCCAGGATTCTTACAGCATTGGCCCTTATATCCAGATGTTATGAATGGTGCGATTCGGGGTCTCGCCGACCAAACCTATTGCCGCAATTGGGAGAGTCAATGCAGGGGGCGCTTTAATAAGTTGCTCCCGCAAGCTCTCTATAATACAAGAGCTCGGGTTACAATCAAAGACCCCCGTCGCCCGCGCTACTAGTCGACGATTCCCTTTTCCATGTACCACGTGAGAGTGCTGCCAGCTCAAGCTCAAGCAATTTACTGAGAAACGATTAATCTACCCCATCAGAATGAGTGGAGAGTTCAACCTGGATTAGACCCCCACTTACTTCGGCATCCAAAGCAGGTTTACCCCTCCTGCTATGGTCTCCACAGCGAAGACTCTAAAGATTTAGCGCGGGAGAAACTCGGCTGTGTATCAGAAGGGCACTTGGTTGCTATCCTTCGGCCTCAAGTGCAGGTGACCAAGTCGATCAATGCCGTCGCACTAACAGGGAGTTGACGTTGTTCCAAACTGCACGATTTGCTCCGTCACCTACGGGTTGAGAGGCCTCATTGGAACTCGCTCCAAATTTCACGTAAAACCTAAGCATAGCCCTTGATCAGTAACTAGAATCAAGGCGGGAAACTTAATCACACGGAGACTATAACGGTCATCACAAGATTCCAGTTTGTACTTGAAATACAAGCGTCAAAACGAACATTTTCTTTAAATAGTACATACTCGTTAATGAGCAGTTCGTGTTGCGTGTGGCGAACTCCTTGGAGATCGGCACCTGATATCTCTACCAGTTCCCGTCGCAATAGAACCTGAAGCAGTGGCGGTTTCTCTCCACGTCGAGCGTGACTGAGCGATCATTTAGAGTCCGTTGGCCGACTAAAGGGTGATCTGGGTACCCACCTATAGAGGAGTTCGGTGCAGAGAATGCCGTGGTGGGGACTTAGATATCGAAGTCGGTATGAGATCTGTTTGGACTTGTCATCTGTGCGTACTTGTACCAACTTGATCGGTTGCGCACGTATATTGGTTATAGGAATTGGGCCATACTCAGGTTCGCAGCCGCTCCTACCGTTAAGACGCGCTGACCGGTTTGCTTTGAAATAATAGTGACACCGGTAACATCGAACATGAACCAACGTTTCACGTTCTATTTACACTCACCCTTCTAATAAAAGTTGATGAGGAATGAAGCGACCCACGTCGAGAACGAAAATTCAGTAGGGAGTGATTGTCAATAACTCGAGGTAGCTTAGCAAAGAAGTAATCAAGGTTAAGTTGCCCACAGGATCCTAAAAACAATTATTCCGATAGCCTCTGGGTAGAATACCCGAAGAAAGTACATTCGCCCTAATTCAGCCTTAAATCATTAAAATAGGCATACAAACCTGTTGTTGGATCTTCATACCGGCGTTGTTTCTGAGATCTCGGTGCCTACAGATATGTCGTATGTTAATGTACCAATCTATAGTTTTGGACGCAAAACATTAAAGAACCACCAACATATGAACCCCGGGTCGTGGGTTACAGGCGTCGTCCACGCTAACTTGCTCAGATGCAAACTTGGTTATCTAATCCGAAAACAGGACTACACCTAAGAACCTGGTTTGAAGCGTAATCAATACATCCGGCATCGGCGCTTGGAACAACCCCACTCCCGATAATCTCTCTGCCCCGTGGCTATAGGGTTCTTCAGTTTGCGCGATAGGGCAGCTTTCGTCTTACTGCTGAACGTCCTAACCAATTTTCCGTCGTGGGTCCAGGGCAACAAGGCTCCGAGATCCCCTTATACATAGAAGCCTATATACGCAGAGAGAACAAGGACGGGTAAAAATAAAGTCTCGGGAGGGTGACGGAACTCCACGAGGCCCTATCGCGCAAAGCGAATCCTTGTCAATTCACACCGATACTCGGTAGTTAATTTGAATTAAAGAGTTGTTTGCCAAGTTGTACTAATATTCAATCGGAAGTTAACCCTCAGCCTACATGTTAATTCCGCCAACCCGGGGATACAAGGAGCACCAAACGGGTTTCAATTTGTGGATTGGTATTGCATGCCCAAATGAACTAAGTTTGGGCCTCCGGGCCTGGATGACTATTCTGAGGTACAGAGATCGAGGGAGCTGTTTGGAGAACTAGAAACTCAATAGGTGACGTACTTATCCATACCAGCCCGCCGGTACGCGCATTAGTAAACTAGCCATGCCATAGTATCGGTTCGTACTCGACTGGATCGCTGAACACCGTACGGTCGCCAAGGTCATAGGGGCTGGTAACTAGGGGACACCATTAAATCTCTGGGCCCCCTCCGTTCTAGCGGCGGGGGTGATATCCATGGTAACTTTTACCTAGCGAACAGCTGCCACCTACTAGGACGCTTATGCGTGTTACTTGCTATGAAGATCCCTGATTTCACAGAATGTGACGCACCATGAATCAGCTCAAGCCCGCAAATCGCGGGATGGTGGAGACGCGGAATCGCCTCCCTGTGCTCTGCGACACTGTGCTGCCCGTAAAGAGCTACCTAGCCAGTTCGGTGACTCTGGAGCCCTACACCCTTGTGCTTACTGGAGAAATATACCTGTATACGAAGGGGGGTCCGGTGGACTTGACCGACGCTACTTACCGACACTGAGCAAAAGCCTACCGTATAAGCGTTGCATCTTTAATACCCTCCCGTATGCTCGCCGAGGTTGTTGAGGCAAACCGACGATTACGGGAGATGACCTTCGGTGGAATGCACGACGGTCATTCTCGCTCAATTAGCATCTAGAGTCCCAGCACTATCCGGGTGCTCTAAATCTAAGCCTGTCTCATCGCTCTTTAGCATCAGGAGGGAAGGAAGCCCTTGTAGGATCCTAGATCAGATGAGCAGAGACCTACATTAGACGCAGAGTTTGCTATATTTTCCTGGTGCTAGCGAGCCCAAATGTAGATCATTCGGAGAGCCCTACGCGTGGTCCTCTACCAAAGTTGCAATCGGCTCTGACAAGGTACTCGAGTGCTCGTGCAAAGTCCCGGGGCCGCGGTCGCCCTATCTCTAGAGCGATATGTATGGGACTTTACGCATACTCTTACCACGTAGATAGCATCATGCCTCCATACTTGCCTGTGGCGGGCAGCCTATACTGGTCGCCGAGATCTCTCAGTCGGGACTTGATATTGGAAAAGATTACTGGCGTTCGTTCTCGTGCGCTGCATTCTTGAGATTTGTACTTTTACGTGAACTGGGCTGCCTTGGTCAGCGCACACGGACTCTGTCCCATGGGCAAACCTGACTCGAAACCAGAGGGTTGTGCCTCGATTAGACCCTGCATTAGATCGGCCCTGTGTCTTCTCCATTACCACCAACTGTGTAAGTACTGAGTAACATCCCGCTTGTAGGAACCCTACATTCCTCCCGCGTGCACATCCCTGGCTTAACATCGCCTTGCACCTGGCTCTCGTAAATAAAGCTATCCTAACCCGCAAGACCGAAGCAGCCGAGGAGTCCAGGCACGCCCCTGAATGCACTCTTACATCGACATGACCCTGTGCTGAGACTTGTTTCTGGCTTGTGTATATGAGTACAGCCGCACATGACAGCAAAGTAGATACCTCGTAACGCCAGAGTGTCTAGTGGAATAAACACAACCCAGGACTTACTCATAAGGCGCCTCTTTTACCCCTTGCGCCAATCATGGGTATAACTACTAGACAACGCAAGAAGCAGTGAGTATTCGATCACCGCCTATTAGACACGGCATAATTACCAATCGATAGGAGGCAGGATATCCGCCGAGAGGCCTTCTTACAGTCTGTTGATTCGATCCCCTAGAACGTCTACGAGCTGCCGAAAGCTAGACGCTGCCCATACATGTACCCGACTAAATATAGCCGATTCTTGCTCTTCCAGTCAGCCTCGGGATAGTTGCGCTTCCTATACGATGGCAGGGACTACGAAAGTGGTATGAGTGATCTGGTACGCGCCTTGAAATTGGTGCAAAGCTAACGAGATTGACGAAGCACCTAACCCGCTTCAATTACTTCCAGTCGGGTATGAAACCGTCCACCAACCTCGCGCCTATGTTGGAGGCAACCGGTTGACATTTCCCGACGGCGATGTGATTCCCAAATAGTACGCTTATCTGGCTGTTAAGTGTCGAGCCACCGATCAACCTAACTGCGTCCTTCACGGTTTAAGAAAAACGACGCCCGAATGATAAAGTTCTAGCAGCCTTTTCGCTTTATTGAGGCTCCTGATAGATAGTTCGATAAGAAGAGTGCAGGCCAGCCAGCAAAACTACTCAACTATGTCTCTATATCCTTTATTACTCTTGTTTTTTGGACATGTTCACCAACAACCTGAGAACTACGGTGTTGGTCTAGAATGATTGTCTGGTAGTCAGCTGACAATAAGCCCTCCTACATATGACGGGATGTAATATACGGGACAGAATCATCGAGCGCGGTCTTCTTGCCAACGATTCTCCTGAAAGGCCCCGAAAATGCCTGGCTGAAGCGCTTCACAGGGTCGAGTACACAGACCGGGCACGTAAAGCTCTGTCTCATTCATAACGTACAAGTATAAGCTAAGTGGCTCACCACGGAGGCTGATGATTGATGAGGTATGCAATAGGTGACAAAATTAGGTTGCAAATATGGTGGAAATTCCTCCACACATATCGTCGTGGGCGATGCAGACTTTATACCAACACAATGTGACAGGTCTTTACTAAAAACCCGGCAGGACCGGTGCGGTGGTGTGGTGGTATCTGGTCCTTGGCGCTCTGTCCGATAAGTTTTCGCAGTCGGTATGGGTCCTTGCAGGAACAACCTTCGGGAAGCAATTACAAAGCCCACGGCGTGGGAAAGACAGCCCAGTCGTAAACCGTTTGTGAGGAGCCACTCTCGGATCATCCATTATTGGCGAATTCATTACTCACGTGGTATATGAGCGCTACTTCCACCAGAGGGAGAGCCCGCAGACACCCTACAGCGTGAAATGAGTTACCGGCAGTCTCGTGATAGCAAGCCGAGAAGCCTTTCTCCACGTAAAACGGCTCTATTTATCCAAATGACACGGCTTCTATGCGTATGAGGTAAACATTACTGAGTTCATCACTGCAGTACGACGATTATACGGGCCCGCCACTAAAATTCAGGGCGCATAACTGTCACTCGATCAAGCTGTAACTCACGAGGCGCATCCTTATTCATAGTGCAGTTAACGATCGGTGCAAAAAATGATGGGCATAATAAATCACAGAGACTACACCGGACCTGCGGCCTGGTAGAACCTCGTGATTCGCCTGCGATCGCCCAACTCGATCGCAAGATTGCCGTTAATCGAGTATATAGAAAGCCGACGAATCGTCGTAACATACTCCAAGACAGTTGCGGCCATCCGACTGGTTAGAGGCAGGCAGAGTTCAACAGTTGATATGTTAACTCGGATGGACCTTGAAGGAGCCCCATCAAGGAAAGATATAGCACAAGTTGCATGCAGTAGCGGCGAGGCGTATTCATATGGAGGGAGCAGTGCGAATGGCCGCCCTCCAATGAGCTTTATGAGATGAGCTAGCGAGCCTCGTAGGCGCCTCAAAATGAGTTGGCAGACAATGTACGGTAATCATCCGACAACTTTTCTCGTCATCCCACCAATCACGTCATGGGCCGTATCAACTGTAGAGTCCACGGAATATCTATCTAACACTTTGAGGTCTACCTACATCTCCAAGTATCACAACGCTCACCCGGCTTCTCCAGCACCACGCGTCTCTTGAATCTGGGGACAGCTAGACTCAATTGTCGGAGGATCGGAATCACTTCGATCGACCTAACGTCACCAGATGCGAGAAGGGGCTACCGCCTGAGACAGCGGACTTGCTGACAGTATCCGTTCGAAGCAATTTTATGTCGGCATAATTGCCCCTTCTGTTTGGGCTCAGTTAATAAATTGCCTGATTTCGTCACTGACTTAGACGACGACACTATAATACGTTTAGGCACCCACTGGACGGGTGGTGGTGACCTCGGTATACCATATCCATAGTGCGAACGCTACAGATGTAACCTGCATCAAATGGGTTAGATGCACCAACAATGTCTTACTACTTACGTGATGGCGATGACTAACTCCTGGTTAGGAACATCTCACTCAGCCCCGTCTACGGACTTATAGTTTTAAAACCGGTAACTGCATATTGGTGGTACGACTCGTTAGGTGGCGCTGATCACATAGTCAGGCCTACGAAATTTCCAAACAGCGGCGAATAGGGACGGAGGAGCAATACAGCGTACCGGCGTAACCGATGGCGCTTGGTTGGCCCCGTTAAGTCTCGCCCTATGGGCAACTTTAACGTACTCATGTACCGTAGGTAACCCTACGCCATCCCAGGACCCTTTGAGCTTCCATACATTGTGTAAATGCCAGATGGTCTTACACCCTGAGCACAAGCAAGCCCGCACCCCAGGCAATTTTCATTCACGAGCGTTAATCCAGGTCATCTGCTACGTGTCCATTGGGTTCACGGACAAAGATTAAATATGCAGATCTCATTAACTAGACTGGTCGTGCTCTAGGCGTCATATACCTCATTTGCGATGGCCAAGTTACGGGGGCGTGGCTGAGCGCCCAAGGATAATCCCTTAGCTGACATACGCTATCATAGTATTTCCAACCTTACAGCTGGGAGTTTAGTTGAGGTAAACGCACCGCTACGTTGCTGCTAAGTCACACGAGACGCTATGAACAATCTCGGGTACACTAGCCCGGTTGATCCTGCAGGTACGGATACAACTGGAACAATCTCGTTATCTTCACACCCGCCACATGCAGCATCCGCCTAGGAATGCATGTCATGAGCCTATTACCGGGCTCATCTGGACCTTCTGCCTTATTATACGGAACCACGATCCCGAATTGCCGCCCAGACTTTTTCCAGACGCGAATCATTGACTGTAAACATCATGCTCCAATGAACCAACTCTTAGAAGCCGGCCCTATCCTCAACCGGGTGATCACCCATGTGAATTCTCTGGTTACCGACAAAGAGTGTTAGCCTCCCAGCTAAGCCAACCAGGTGCTCAGCTTTTAACCGGATGGAAGACACTGCGAAAAGCAGCATCCAATATGTAGCCAGTAAATAATCTCGGGGCAACAAGTTCAGTCGCCTGCTTGGAGATATGCTGTTGCTCGTTCCGTGGCACAGACAAACTGGTAGGTTCCGCCAGTTCGACTAATCAAGGACATCGGCGATACACAAACGCTCGTCAAGGGTTGTCCCTTCGATTGTTGACATTGTCTTTGCGCACTCTTCCGGACATATAGTAGGTTGGTAACGATCCTCTATCAACACCCTCCCTCCCAACAGTACCGCTGCGTGTCGAGTTTCCTGGCGGTTATAGCGTAACGATCGGCCCCAGTACGCAACTCATACGTCTCAAAGGGATAGATCCCTTCAACCACCGACATTAATGGGTGTCTTGAAGTGGATACAGTAATATAATTTGGCCTCACCCACTGCGGGTCGACGAAGCTCCCAGTTATCCACTAAAGGTGTCGCTTCGCAAGGACCTCCCTGGGGGGACGTCACGTCTGGAGCGGGACTCGGTCCCT";
    let text = "TTTTTGACTCGAATGAAAGTGGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGCTTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGCCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGATCCGAATATAAAACCCCTGGCTTCAGATTGATAGGAGCAGCAGGGATATGGACGCGAGGAAGACAGCTACGGGAAAAACAGTAACGGGGATCTTGTGTTGCCAAGCCTCGTACTATGTCGACATAGTTGGCTAGAGTCGCTCCCAGATCGGGTGCAAATTTCTTATGAAACACTCGCGGTAACTGGATTTGGCGGCCACGCGGGCACTGTGTTTGGGTGCGCTATCCAGTTTGCGAGATCTCATCTGAGACCCCCATGCCGACAGTACTTGTCGTTGCTAGCCAGTCGCGATTCAGCATCTGCCCAATGAAACGAACCGTAGGAATTGTTTTCAATCCGTCTATCCTGTGCACACGTAAGTACCCGAGATGCGTCTAACATTACTCTTGTGCTAGCGTGGGAACCTGGCTGCGTTCCGAAGTCCTATGTGCCCCCGTCCGTAAAATGGCGGTCACCAATATACAGTGCAACCACCTATCGTCAGGGGTCTGGCTAAGTGTTTATCGTCGCAAGCGATCACCTTGGAGCTCAGCCGGTGACTACCGAGCGGGTCGCAGACGGCCTGTCGCCTCGCGAGACCGAAATGACTCACTCAGCCCTTAACGGGAGACCCGGACGGAAGCATCTCTAATCACTGTACAGGCCCCGTTGAGACGGATCCTCGTGGGGTTGGGTTTACTAATGCCATGACTTGGTACCGCACTGAGGCACCCCAGGCCACGAACTAGACACCCAAGTGTATCCCCATGCATAAAGTAAAGCCTTGACGTCGCGATCATGGTGTATCATCAGGGGGAGTTCACATCAGCAATATTAGAAGGCACTGTAATTGAAGTGTCATGATGTAATAGCGGGCAGCCCCACGTAAGGAGCCGTAGTATCCCAAGCGATTGTCAGCACTCTAGCAATTTCCAGGTTAATAGGCGCTTGGAGTTATGCTGACTATCATGAGTTACCGAACTCAGCCGAGCGGTGTGAACTATGAAACTAGATAAGGCAGTGAGCTCATCCGGCCGGCACGTGGAGAGACAAACGCAACAGTTACCTATAGCAATATAGTGTAGCGTGTCGCCGGGTGTGGCATCATACGATCACGTCAGGTTAGGTCAGATTCGAGGCTACGGGTCCAGTTGTAAGTGGCGGAAACGTGCTAAGGATACGTAATAACAGACCTGGTAATTGGTGAGTAGAGGCGGGTTGGATTGGCTGGGCCCCAGATGGTCCAATGATCGTCGTTAGGTCAAGCCCCAAAACCCGAAGTCATGTGGTCAGTATGGATTCTACCCGTCGGCTTAGGTAGAAATCAGTACTGCTTGCGGTAAAATCGGATACACCCCTCGAAACAAACGGAGTCATTGGCGCCTCTCTAAATACTGCTGTCCTAAATTGGCCATCGGGCGGGCTTCGCCCAGGATTCTTACAGCATTGGCCCTTATATCCAGATGTTATGAATGGTGCGATTCGGGGTCTCGCCGACCAAACCTATTGCCGCAATTGGGAGAGTCAATGCAGGGGGCGCTTTAATAAGTTGCTCCCGCAAGCTCTCTATAATACAAGAGCTCGGGTTACAATCAAAGACCCCCGTCGCCCGCGCTACTAGTCGACGATTCCCTTTTCCATGTACCACGTGAGAGTGCTGCCAGCTCAAGCTCAAGCAATTTACTGAGAAACGATTATCTACCCCATCAGAATGAGTGGAGGAGTTCAACCTGGATTAGACCCCCACTTACTTCGGCATCCAAAGCAGGTTTACCCCTCCTGCTATGGTCTCCACAGCGAAGACTCTAAAGATTTAGCGCGGGAGAAACTCGGCTGTGTATCAGACGGGCACTTGGTTGCTATCCTTCGGCCTCAAGTGCAGGTGACCAAGTCGATCAATGCCGTCGCACTAAACAGGGAGTTGACGTTGTTCCAAACTGCACGATTTGCTCCGTCACCTACCGGGTTGAGAGGCCTCATTGGAACTCGCTCCAAATTTCACGTAAAACCTAAGCATAGCCCTTGATCAGTAACTAGAATCAAGGCGGGAAACATTAATCACACGGAGACTATAACGGTCATCACAAGATTCCAGTTTGTACTTGAAATACAAGCGTCAAATCGAACATTTTCCTTAAATAGTACATACTCGTTAATGAGCAGTTCGTGTTGCGTGTGGCGAACTCCTTAGAGATCGGCACCTGATATCTCTACCAGTTCCCGTCGCAATAGAACCTGAAGCAGTGCGGTTTTTCTCCACGTCGAGCGTGACTGAGCGATCATTTAGAGTCCGTTGGCCGACTAAAGGGTGATCTGGGTACCCACCTATAGAGGAGTTCGGTGCAGAGAATGCCGTGGTGGGGACTTAGAAATCGAAGTCGGTATGAGATCTGTTTGGACTTGTCATCTGTGCGTACTTGTTCCAACTTGATCGGTTGCGCACGTATATTGGTTATAGGAATTGGGCCATACTCAGGTTCGCAGCCGCTCCTACCGTTAAGACGCGCTGACCGGTTTGCTTTGAAATAATAGTGACACCGGTAACATCGAACATGAACCAACGTTTCACGTTCTATTTACACTCACCCTTCTAATAAAAGTTGATGAGGAATGAAGCGACCCACGTCGAGAACGAAAATTCAGTAGCGAGTGATTGTCAATAACTCGAGGTAGCTTAGCAAAGAAGTAATCAAGGTTAAGTTGCCCACAGGATCCTAAAAACAATTATTCCGATAGCCTCTGGGTCAGAATACCCGAAGAAAGTACATTCGCCCTAATTCAGCCTTAAATCATTAAAATAGGCATACAAACCTGTTGTTGGATCTTCATACCGGCGTTGTTTCTGAGATCTCGGTGCCTACAGATATGTCACTATGTTAATGTACCAATCTATAGTTTTGGACGCAAAACATTAAAGAACCACCAACATATGAACCCCGGGTCGTGGGTTACAGGCGTCGTCCACGCTAACTTGCTCAGATGCAAACTGGTTATCTAATCCGAAAACAGGACTACACCTAAGAACCTGGTTTGAAGCGTAATCAATACATCCGGCATCGGCGCTTGGAACAACCCCACTCCCGATAATCTCTCTGCCCCGTGGCTATAGGGTTCTTCAGTTTGCGCGATAGGGCAGCTTTCGTCTTACTGCTGAACGTCCTAACCAATTTTCCGTCGTGGGTCCAGGGCAACAAGGCTCCGAGATCCCCTTATACATAGAAGCCTATATACGCAGAGAGAACAAGGACGGGTAAAAATAAAGTCTCGGGAGGGTGACGGAACTCCACGAGGCCCTATCGCGCAAAGCGAATTCTTGTCAATTCACACCGATACTCGGTAGTTAATTTGAATTAAAGAGTTGTTTGCCAAGTTGTACTAATATTCAATCGGAAGTTAACCCTCAGCCTACTGTTAATTCCGCCAACCCGGGGATACAAGGAGCACCAAACGGGTTTCAATTTTGGATTGGTATTGCATGCCCAAATGAACTAAGTTTGGGCCTCCGGGCCTGGATGACTATTCTGAGGTACAGAAGATCGAGGGAGCTGTTTGGAGAACTAGAAACTCAATAGGTGACGTACTTATCCATACCAGCCCGCCGGTACGCGCATTAGTAAACTAGTCCATGCCATAGTATCGGTTCGTACTCGACTGGATCGCTGAACACCGTACGGTCGCCAAGGTCATAGGGGCTGGTAACTAGGGGACACCATTAAATCTCTGGGCCCCCTCCGTTCTAGCGGCGGGGGTGATATCCATGGTACACTTTTACCTAGCGAACAGCTGCCACTACTAGGACGCTTATGCGTGTTACTTGCTATGAAGATCCCTGATTTCACAGAATGTGACGCACCATGAATCAGCTCAAGCCCGCAAATCGCGGGATGGTGGAGACGCGGATCGCCTCCCTGTGCTCTGCGACACTGTGCTGCCCGTAAAGAGCTACCTAGCCAGTTCTGTGACTCTGGAGCCCTACACCCTTGTGCTTACTGGAGAAATATACCTGTGATACGAAGGGGGGTCCGGTGGACTTGACCGACGCTACTTACCGACACTGAGCAAAAGCCTACCGTATAAGCGTTGCATCTTTAATACCCTCCCGTATGCTCGCCGAGGTTGTTGAGGCAAACCGACGATTACGGGAGATGACCTTCGGTGGAATGCACGACGGTCATTCTCGCTCAATTAGCATCTAGAGTCCCAGCACTATCCGGGTGGCTCTAAATCTAAGCCTGTCTCATCGCTCTTTAGCATCAGGAGGGAAGGAAGCCCTTGTAGGATCCTAGATCAGATGAGCAGAGACCTACATTAGACGCAGAGTTTGCTATATTTTCCTGGTGCTAGCGAGCCCAAATGTAGATCATTCGGAGAGCCCTACGCGTGGTCCTCTACCAAAGTTCAATCGGCTCTGACAAGGTACTCGAGTGCTCGTGCAAAGTCCCGGGGCCGCGGTCGCCCTATCTCTAGAGCGATATGTATGGGACTTTACGCATACTCTTACCACGTAGATAGCATCATGCCTCCATACTTGCCTGTGGCGGGCAGCCTATACTGGTCGCCGAGATCTCTCAGTCGGGACTTGATATTGGAAAAGATTACTGGCGTTCGTTCTCGTGCGCTGCATTCTTGAGATTTGTACTTTTACGTGAACTGGGCTGCCTTGGTCAGCGCACACGGACTCTGTCCCATGGGCAAACCTGACTCGAAACCAGAGGGTTGTGCCTCGATTAGACCCTGCATTAGAGTCGGCCCTGTGTCTTCTCCATTACCACCAACTGTGTAAGTACTGAGTAACATCCCCGCTTGTAGGAACCCTACATTCCTCCCGCGTGCACATCCCTGGCTTAACATCGCCTTGCACCTGGCTCTCGTAAATAAAGACTATCCTAACCCGCAAGACCGAAGCAGCCGAGGAGTCCAGGCACGCCCCTGAAGGCACTCTTACATCGACATGACCCTGTGCTGAGACTTGTTTCTGGCTTGTGTATATGAGTACAGCCGCACATGACAGCAAAGTAGATACCTCGTAACGCCAGAGTGTCTAGTGGAATAAACACAACCCAGGACTTACTCATAAGGCGCCTCTTTTACCCCTTGCGCCAATCATGGGTATAACTACTAGACAACGCAAGAAGCAGTGAGTATTCGATCACCGCCTATTAGACACGGCATAATTACCAATCGATAGGAGGCATGGATATCCGCCGAGAGGCCTTCTTACAGTCTGTTGATTCGATCCCCTAGAACGTCTACGAGCTGCGAAAGCTAGACGCTGCCCATACATGTACCCGACTAAATATAGCCGATTCTTGCTCTTCCAGTCAGCCTCGGGATAGTTGCGCTTCCTATACGATGGCAGGGACTACGAAAGTGGTATGAGTGTCTGGTACGCGCCTTGAAATTGGTGCAAAGCTAACGAGATTGACGAAGCACCTAACCCGCTTCAATTACTTCCAGTCGGGTATGAAACCGTCCACCAACCTCGCGCCTATGTTGGAGGCAACCGGTTGACATTTCCCGACGGCGATGTGATTCCCAAATAGTACGCTTATCTGGCTGTTAAGTGTCGAGCCACCGATCAACCTAACTGCTCCTTCACGGTTTAAGAAAAACGACGCCCGAATGATAAAGTTCTAGCAGCCTTTTCGCTTTATTGAGGCTCCTGATAGATAGTTCGATAAGAAGAGTGCAGGCCAGCCAGCAAAACTACTCAACTATGTCTCTATATCCTTTATTACTCTTGTTTTTTGGACATGTTCACCAACAACCTGAGAACTACGGTGTTGGTCTAGAATGATTGTCTGGTAGTCAACTGACAATAAGCCCTCCTACATATGACGGATGTAATATACGGGACAGAATCATCGAGCGCGGTCTTCTTGCCAACGATTCTCCTGAAAGGCCCCGAAAATGCCTGGCTGAAGCCGCTTCACAGGGTCGAGTACACAGACCGGGCACGTAAAGCTCTGTCTCATTCATAACGTACAAGTATAAGCTCAGTGGCTCACCACGGAGGCTGATGATTGATGAGGTATGCAATAGGTGACAAAATTAGCGTTGCAAATATGGTGGAAATTCCTCCACACATATCGTCGTGGGCGATGCAGACTTTATACCAACACAATGTGACAGGTCTTTACTAAAAACCCGGCAGGACCGGTGCGGTGGTGTGGTGGTATCTGGTCCTTGGCGCTCTGTCCGATAAGTTTTCGCAGTCGGTATGGGTCCTTGCAGGAACAACCTTCGGGAAGCAATTACAAAGCCCACGGCCGTGGGAAAGACAGCCCAGTCGTAAACCGTTTGTGAGGAGCCACTCTCGGATCATCCATTATTGGCGAATTCATTACTCACGTGGTATATGAGCGCTACTTCCACCAGAGGGAGAGCGCCGCAGACACCCTACAGCGTGAAATGAGTTACCGGCAGTCTCGTGATAGCAAGCCGAGAAGCCTTTCATCCACGTAAAAACGGCTCTATTTTCCAAATGACACGGCTTCTATGCGTATGAGGTAAACATTACTGAGTTCATCACTGCAGTACGACGATTATACGGGCCCGCCACTAAAATTCAGGGCGCATAACTGTCACTCGATCAAGCTGTAACTCACGAGGCGCATCCTTATTCATAGTGCAGTTAACGATCGGTGCAAAAAATGATGGGCATAATAAATCACAGAGACTACACCGGACCTGCGGCCTGGTAGAACCTCGTGATTCGCCTGCGATCGCCCAACTCGATCGCAAGATTGCCGTTAATCGAGTATATAGAAAGCCGACGAATCGTCGTAACATACTCCAAGACAGTTGCGGCCATCCGACTGGTTAGAGGCAGGCAAGTTCAACAGTTGATATGTTAACTCGGATGGACCTTGAAGGAGCCCCATCAAGGATAGATATAGCACAAGTTGCATGCAGTAGCGGCGAGGCGTATTCATATGGAGGGAGCAGTGCGGAATGGCCGCCCTCCAATGAGCTTTATGAGATGAGCTAGCGAGCCTCGTAGGGCCTCAAAATGAGTTGGCAGACAATGTACGGTAATCATCCGACAACTTTTCTCGTCATCCCACCAATCACGTCATGGGCCGTATCAACTGTAGAGTCCACGGAATATCTATCTAACACTTTGAGGTCTACCTACATCTCCAAGTATCACAACGCTCACCCGGCTTCTCCAGCACCACGCGTCTCTTGAATCTGGGGACAGCTAGACTCAATTGTCGGAGGATCGGAATCACTTCGATCGACCTAACGTCACCAGATGCGAGAAGGGGCTACCGCCTGAAACAGCGGATTGCGACAGTATCCGTTCGAAGCAATTTTATGTCGGCATAATTGCCCCTTCTGTTTGGGCTCAGTTAATAAATTGCCTGATATCGTCACTGACTTAGACGACGACACTATAATACGTTTAGGCACCCACTGGACGGGTGGTGGTGACCTCGGTATACCATATCCATAGTGCGAACGCTACAGATGTAACCTGCATCAAATGGGTTAGATCCACCAACAATGTCTAACTACTTACGTGATGGCGATGACTAACTCCTGGTTAGGAACATCTCACTCAGCCCCGTCTACGGACTTATAGTTTTAAAACCGGTAACTGCATATTGGTGGTACGACTCGTTAGGTGGCGCTGATCACATAGTCAGGCCTACGAAATTTCCAAACAGCGGCGAATAGGGACGGAGGAGCAATACAGCGTACCGGCGTAACCGATGGCGCTTGGTTGGCCCCGTTAAGTCTCGCCCTATGGGCAACTTTAACGTACTCATGTACCGTAGGTAACCCTACGCCATCCCAGGACCCTTGAGCTTCCATACATTGTGTAAATGCCAGATGGTCTTACACCCTGAGCACAAGCAAGCCCGCACCCCAGGCAATTTTCATTCACGAGCGTTAATCCTAGGTCATCTGCTACGTGTCCATTGGGTTCACGGACAAAGATTAAATATGCAGATCTCATTAACTAGACTGGTCGTGCTCTAGGCGTCATATACCTCATTTGCGATGGCCAAGTTACGGGGGCGTGGCTGAGCGCCCAAGGATAATCCCTTAGCTGACATACGCTATCATAGTATTTCCAACCTTACAGCTGGGAGTTTAGTTGAGGTAAACGCCACCGCTACGTTGCTGCTAAGTCAACACGAGACGCTATGAACAATCTCGTGGTACACTAGCCCGGTTGATCCTGCAGGGTACGGATACAACTGGAACAATCTCGTTATCTTCACACCCGCCACATGCAGCATCCGCCTAGGAATGCATGTCATGAGGCCTATTACCGGGCTCATCTGGACCTTCTGCCTTATTATACGGAACCACGATCCCGAATTGCCGCCCAGACTTTTTCCAGACGCGAATCATTGACTGTAAACATCATGCTCCAATGAACCAACTCTTAGAAGCCGGCCCTATCCTCAACCGGGTGATCACCCATGTGAATTTTCTGGTTACCGACAAAGAGTGTTAGCCTCCCAGCTAAGCCAACCAGGTGCTCAGCTTTTAACCGGATGGAAGACACTGCGAAAAGCAGCATCCAATACTGTAGCCAGTAAATAATCTCGGGGCAACAAGTTCAGTCGCCTGCTTGGAGATATGCTGTTGCTCGTTCCGTGGCACAGACAAACTGGTAGGTTCCGCCAGTTCGACTGAATCAAGGACATCGGCGATACACAAACGCTCGTCAAGGGTTGTCCCTTCGATTGTTGACATTGTCTTTGCGCACTCCTCCGGACATATAGTAGGTTGGTAACGATCCTCTATCAACACCCTCCCTCCCAACAGTACCGCTGCGTGTCGAGTTTCCTGGCGGTTATAGCGTAACGATCGGCCCCAGTACGCAACTCATACGTCTCAACGGGATAGATCCCTTCAACCAGCCGACATTAATGGGTTGTCTTGAAGTGGATACAGTAATATAATTTGGCCTCAACCCACTGCGGGTCGACGAAGCTCCCAGTTATCCACTAAAGGTGTCCGCTTCGCAAGGACCTCCCTGGGGGGACGTCACGTCTGGAGCGGGACTCGGTCCCT";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 10000 1% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l10000_e10(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGATCCGAATATAAAACCCCTGGCTTCAGATTGATAGGAGCAGCAGGGATATGGACGCGAGGAAGACAGCTACGGGAAAAACAGTAACGGGAATCTTGTGTTGCCAAGCCTCGTACTATGTCGACATAGTTGGCTAGAGTCGCTCCCAGATCGGGTGCAAATTTCTTATGAAACACTCGCGGTAACTGGATTTGGCGGCCACGCGGGCACTGTGTTTGGGTGCGCTATCCAGTTTGCGAGATCTCATCTGAGACCCCCATGCCGACAGTACTTGTCGTTGCTAGCCAGTCGCGATTCAGCATCTGCCCAATGAAACGAACCGTAGGAATTGTTTTCAATCCGTCTATCCTGTGCACACGTAAGTACCCGAGATGCGTCTAACATTACTCTTGTGCTAGCGTTGGGAACCTGGCTGCTTCCGAAGTCCTATGTGCCCCCGTCCGTAAAATGGCGGTCACCAATATACAGTTGCAACCACCTATCGGCAGGGGTCTGGCTAAGTGTTTATCGTCGCAAGCGATCACCTTGGAGCTCAGCCGGTGACTACCGAGCGGGTCGCAGACGGCCTGTCGCCTCGCGAGACCGAAATGACTCACTCAGCCCTTAACGGGAGACCCGGACGGAAGCATCTCTAATCACTGTACAGGCCCCGTTGAGACGGATCCTCGTGGGGTTGGGTTTACTAATGCCATGACTTGGTACCGCACTGAGGCACCCCAGGCCACGAACTAGACACCCAAGTGTATCCCCATGCATAAAGTAAAGCCTTGACGTCGGGATTATGGTGTATCATCAGGGGGAGTTCACATCAGCAATATTAGAAGGCACTGTAATTGAAGTGTCATGATGTAATAGCGGGCAGCCCCACGTAAGGAGCCGTAGTATCCCAAGCGATTGTCAGCACTCTAGCATATTTCCAGGTTAATAGGCGCTTGGAGTTATGCTGACTATCATGAGTTACCGAACTCAGCCGAGCGGTGTGAACTATGAAACTAGATAAGGCAGTGAGCTCATCCGGCCGGCACGTGGAGAGACAAACGCAACAGTTACCTATAGCAATATAGTGTAGCGTGTCGCCGGGTGTGGCATCATACGATCACGTCAGGTTAGGTCAGATTCGAGGCTACGGGTCCAGTTGTAAGTGGCGGAAACGTGCTAAGGATACGTAATAACAGACCTGGTAATTGGTGAGTAGAGGCGGGTTGGATTTGGTGGGCCCCAGATGGTCCAATGATCGTCGTTAGGTCAAGCCCCAAAACCCGAAGTCATGTGGTCAGTATGGATTCTACCCGTCGGCTTAGGTAGAAATCAGTACTGCTTGCGGTAAAATCGGATACACCCCTCGAAACAAACGGAGTCTTGGCGCCTCTCTAAATACTGCTGTCCTAAATTGGCCATCGGGCGGGCTTCGCCCAGGATTCTTACAGCATTGGCCCTTATATCCAGATGTTATGAATGGTGCGATTCGGGGTCTCGCCGACCAAACCTATTGCCGCAATTGGGAGAGTCAATGCAGGGGGCGCTTTAATAAGTTGCTCCCGCAAGCTCTCTATAATACAAGAGCTCGGGTTACAATCAAAGACCCCCGTCGCCCGCGCTACTAGTCGACGATTCCCTTTTCCATGTACCACGTGAGAGTGCTGCCAGCTCAAGCTCAAGCAATTTACTGAGAAACGATTAATCTACCCCATCAGAATGAGTGGAGAGTTCAACCTGGATTAGACCCCCACTTACTTCGGCATCCAAAGCAGGTTTACCCCTCCTGCTATGGTCTCCACAGCGAAGACTCTAAAGATTTAGCGCGGGAGAAACTCGGCTGTGTATCAGAAGGGCACTTGGTTGCTATCCTTCGGCCTCAAGTGCAGGTGACCAAGTCGATCAATGCCGTCGCACTAACAGGGAGTTGACGTTGTTCCAAACTGCACGATTTGCTCCGTCACCTACGGGTTGAGAGGCCTCATTGGAACTCGCTCCAAATTTCACGTAAAACCTAAGCATAGCCCTTGATCAGTAACTAGAATCAAGGCGGGAAACTTAATCACACGGAGACTATAACGGTCATCACAAGATTCCAGTTTGTACTTGAAATACAAGCGTCAAAACGAACATTTTCTTTAAATAGTACATACTCGTTAATGAGCAGTTCGTGTTGCGTGTGGCGAACTCCTTGGAGATCGGCACCTGATATCTCTACCAGTTCCCGTCGCAATAGAACCTGAAGCAGTGGCGGTTTCTCTCCACGTCGAGCGTGACTGAGCGATCATTTAGAGTCCGTTGGCCGACTAAAGGGTGATCTGGGTACCCACCTATAGAGGAGTTCGGTGCAGAGAATGCCGTGGTGGGGACTTAGATATCGAAGTCGGTATGAGATCTGTTTGGACTTGTCATCTGTGCGTACTTGTACCAACTTGATCGGTTGCGCACGTATATTGGTTATAGGAATTGGGCCATACTCAGGTTCGCAGCCGCTCCTACCGTTAAGACGCGCTGACCGGTTTGCTTTGAAATAATAGTGACACCGGTAACATCGAACATGAACCAACGTTTCACGTTCTATTTACACTCACCCTTCTAATAAAAGTTGATGAGGAATGAAGCGACCCACGTCGAGAACGAAAATTCAGTAGGGAGTGATTGTCAATAACTCGAGGTAGCTTAGCAAAGAAGTAATCAAGGTTAAGTTGCCCACAGGATCCTAAAAACAATTATTCCGATAGCCTCTGGGTAGAATACCCGAAGAAAGTACATTCGCCCTAATTCAGCCTTAAATCATTAAAATAGGCATACAAACCTGTTGTTGGATCTTCATACCGGCGTTGTTTCTGAGATCTCGGTGCCTACAGATATGTCGTATGTTAATGTACCAATCTATAGTTTTGGACGCAAAACATTAAAGAACCACCAACATATGAACCCCGGGTCGTGGGTTACAGGCGTCGTCCACGCTAACTTGCTCAGATGCAAACTTGGTTATCTAATCCGAAAACAGGACTACACCTAAGAACCTGGTTTGAAGCGTAATCAATACATCCGGCATCGGCGCTTGGAACAACCCCACTCCCGATAATCTCTCTGCCCCGTGGCTATAGGGTTCTTCAGTTTGCGCGATAGGGCAGCTTTCGTCTTACTGCTGAACGTCCTAACCAATTTTCCGTCGTGGGTCCAGGGCAACAAGGCTCCGAGATCCCCTTATACATAGAAGCCTATATACGCAGAGAGAACAAGGACGGGTAAAAATAAAGTCTCGGGAGGGTGACGGAACTCCACGAGGCCCTATCGCGCAAAGCGAATCCTTGTCAATTCACACCGATACTCGGTAGTTAATTTGAATTAAAGAGTTGTTTGCCAAGTTGTACTAATATTCAATCGGAAGTTAACCCTCAGCCTACATGTTAATTCCGCCAACCCGGGGATACAAGGAGCACCAAACGGGTTTCAATTTGTGGATTGGTATTGCATGCCCAAATGAACTAAGTTTGGGCCTCCGGGCCTGGATGACTATTCTGAGGTACAGAGATCGAGGGAGCTGTTTGGAGAACTAGAAACTCAATAGGTGACGTACTTATCCATACCAGCCCGCCGGTACGCGCATTAGTAAACTAGCCATGCCATAGTATCGGTTCGTACTCGACTGGATCGCTGAACACCGTACGGTCGCCAAGGTCATAGGGGCTGGTAACTAGGGGACACCATTAAATCTCTGGGCCCCCTCCGTTCTAGCGGCGGGGGTGATATCCATGGTAACTTTTACCTAGCGAACAGCTGCCACCTACTAGGACGCTTATGCGTGTTACTTGCTATGAAGATCCCTGATTTCACAGAATGTGACGCACCATGAATCAGCTCAAGCCCGCAAATCGCGGGATGGTGGAGACGCGGAATCGCCTCCCTGTGCTCTGCGACACTGTGCTGCCCGTAAAGAGCTACCTAGCCAGTTCGGTGACTCTGGAGCCCTACACCCTTGTGCTTACTGGAGAAATATACCTGTATACGAAGGGGGGTCCGGTGGACTTGACCGACGCTACTTACCGACACTGAGCAAAAGCCTACCGTATAAGCGTTGCATCTTTAATACCCTCCCGTATGCTCGCCGAGGTTGTTGAGGCAAACCGACGATTACGGGAGATGACCTTCGGTGGAATGCACGACGGTCATTCTCGCTCAATTAGCATCTAGAGTCCCAGCACTATCCGGGTGCTCTAAATCTAAGCCTGTCTCATCGCTCTTTAGCATCAGGAGGGAAGGAAGCCCTTGTAGGATCCTAGATCAGATGAGCAGAGACCTACATTAGACGCAGAGTTTGCTATATTTTCCTGGTGCTAGCGAGCCCAAATGTAGATCATTCGGAGAGCCCTACGCGTGGTCCTCTACCAAAGTTGCAATCGGCTCTGACAAGGTACTCGAGTGCTCGTGCAAAGTCCCGGGGCCGCGGTCGCCCTATCTCTAGAGCGATATGTATGGGACTTTACGCATACTCTTACCACGTAGATAGCATCATGCCTCCATACTTGCCTGTGGCGGGCAGCCTATACTGGTCGCCGAGATCTCTCAGTCGGGACTTGATATTGGAAAAGATTACTGGCGTTCGTTCTCGTGCGCTGCATTCTTGAGATTTGTACTTTTACGTGAACTGGGCTGCCTTGGTCAGCGCACACGGACTCTGTCCCATGGGCAAACCTGACTCGAAACCAGAGGGTTGTGCCTCGATTAGACCCTGCATTAGATCGGCCCTGTGTCTTCTCCATTACCACCAACTGTGTAAGTACTGAGTAACATCCCGCTTGTAGGAACCCTACATTCCTCCCGCGTGCACATCCCTGGCTTAACATCGCCTTGCACCTGGCTCTCGTAAATAAAGCTATCCTAACCCGCAAGACCGAAGCAGCCGAGGAGTCCAGGCACGCCCCTGAATGCACTCTTACATCGACATGACCCTGTGCTGAGACTTGTTTCTGGCTTGTGTATATGAGTACAGCCGCACATGACAGCAAAGTAGATACCTCGTAACGCCAGAGTGTCTAGTGGAATAAACACAACCCAGGACTTACTCATAAGGCGCCTCTTTTACCCCTTGCGCCAATCATGGGTATAACTACTAGACAACGCAAGAAGCAGTGAGTATTCGATCACCGCCTATTAGACACGGCATAATTACCAATCGATAGGAGGCAGGATATCCGCCGAGAGGCCTTCTTACAGTCTGTTGATTCGATCCCCTAGAACGTCTACGAGCTGCCGAAAGCTAGACGCTGCCCATACATGTACCCGACTAAATATAGCCGATTCTTGCTCTTCCAGTCAGCCTCGGGATAGTTGCGCTTCCTATACGATGGCAGGGACTACGAAAGTGGTATGAGTGATCTGGTACGCGCCTTGAAATTGGTGCAAAGCTAACGAGATTGACGAAGCACCTAACCCGCTTCAATTACTTCCAGTCGGGTATGAAACCGTCCACCAACCTCGCGCCTATGTTGGAGGCAACCGGTTGACATTTCCCGACGGCGATGTGATTCCCAAATAGTACGCTTATCTGGCTGTTAAGTGTCGAGCCACCGATCAACCTAACTGCGTCCTTCACGGTTTAAGAAAAACGACGCCCGAATGATAAAGTTCTAGCAGCCTTTTCGCTTTATTGAGGCTCCTGATAGATAGTTCGATAAGAAGAGTGCAGGCCAGCCAGCAAAACTACTCAACTATGTCTCTATATCCTTTATTACTCTTGTTTTTTGGACATGTTCACCAACAACCTGAGAACTACGGTGTTGGTCTAGAATGATTGTCTGGTAGTCAGCTGACAATAAGCCCTCCTACATATGACGGGATGTAATATACGGGACAGAATCATCGAGCGCGGTCTTCTTGCCAACGATTCTCCTGAAAGGCCCCGAAAATGCCTGGCTGAAGCGCTTCACAGGGTCGAGTACACAGACCGGGCACGTAAAGCTCTGTCTCATTCATAACGTACAAGTATAAGCTAAGTGGCTCACCACGGAGGCTGATGATTGATGAGGTATGCAATAGGTGACAAAATTAGGTTGCAAATATGGTGGAAATTCCTCCACACATATCGTCGTGGGCGATGCAGACTTTATACCAACACAATGTGACAGGTCTTTACTAAAAACCCGGCAGGACCGGTGCGGTGGTGTGGTGGTATCTGGTCCTTGGCGCTCTGTCCGATAAGTTTTCGCAGTCGGTATGGGTCCTTGCAGGAACAACCTTCGGGAAGCAATTACAAAGCCCACGGCGTGGGAAAGACAGCCCAGTCGTAAACCGTTTGTGAGGAGCCACTCTCGGATCATCCATTATTGGCGAATTCATTACTCACGTGGTATATGAGCGCTACTTCCACCAGAGGGAGAGCCCGCAGACACCCTACAGCGTGAAATGAGTTACCGGCAGTCTCGTGATAGCAAGCCGAGAAGCCTTTCTCCACGTAAAACGGCTCTATTTATCCAAATGACACGGCTTCTATGCGTATGAGGTAAACATTACTGAGTTCATCACTGCAGTACGACGATTATACGGGCCCGCCACTAAAATTCAGGGCGCATAACTGTCACTCGATCAAGCTGTAACTCACGAGGCGCATCCTTATTCATAGTGCAGTTAACGATCGGTGCAAAAAATGATGGGCATAATAAATCACAGAGACTACACCGGACCTGCGGCCTGGTAGAACCTCGTGATTCGCCTGCGATCGCCCAACTCGATCGCAAGATTGCCGTTAATCGAGTATATAGAAAGCCGACGAATCGTCGTAACATACTCCAAGACAGTTGCGGCCATCCGACTGGTTAGAGGCAGGCAGAGTTCAACAGTTGATATGTTAACTCGGATGGACCTTGAAGGAGCCCCATCAAGGAAAGATATAGCACAAGTTGCATGCAGTAGCGGCGAGGCGTATTCATATGGAGGGAGCAGTGCGAATGGCCGCCCTCCAATGAGCTTTATGAGATGAGCTAGCGAGCCTCGTAGGCGCCTCAAAATGAGTTGGCAGACAATGTACGGTAATCATCCGACAACTTTTCTCGTCATCCCACCAATCACGTCATGGGCCGTATCAACTGTAGAGTCCACGGAATATCTATCTAACACTTTGAGGTCTACCTACATCTCCAAGTATCACAACGCTCACCCGGCTTCTCCAGCACCACGCGTCTCTTGAATCTGGGGACAGCTAGACTCAATTGTCGGAGGATCGGAATCACTTCGATCGACCTAACGTCACCAGATGCGAGAAGGGGCTACCGCCTGAGACAGCGGACTTGCTGACAGTATCCGTTCGAAGCAATTTTATGTCGGCATAATTGCCCCTTCTGTTTGGGCTCAGTTAATAAATTGCCTGATTTCGTCACTGACTTAGACGACGACACTATAATACGTTTAGGCACCCACTGGACGGGTGGTGGTGACCTCGGTATACCATATCCATAGTGCGAACGCTACAGATGTAACCTGCATCAAATGGGTTAGATGCACCAACAATGTCTTACTACTTACGTGATGGCGATGACTAACTCCTGGTTAGGAACATCTCACTCAGCCCCGTCTACGGACTTATAGTTTTAAAACCGGTAACTGCATATTGGTGGTACGACTCGTTAGGTGGCGCTGATCACATAGTCAGGCCTACGAAATTTCCAAACAGCGGCGAATAGGGACGGAGGAGCAATACAGCGTACCGGCGTAACCGATGGCGCTTGGTTGGCCCCGTTAAGTCTCGCCCTATGGGCAACTTTAACGTACTCATGTACCGTAGGTAACCCTACGCCATCCCAGGACCCTTTGAGCTTCCATACATTGTGTAAATGCCAGATGGTCTTACACCCTGAGCACAAGCAAGCCCGCACCCCAGGCAATTTTCATTCACGAGCGTTAATCCAGGTCATCTGCTACGTGTCCATTGGGTTCACGGACAAAGATTAAATATGCAGATCTCATTAACTAGACTGGTCGTGCTCTAGGCGTCATATACCTCATTTGCGATGGCCAAGTTACGGGGGCGTGGCTGAGCGCCCAAGGATAATCCCTTAGCTGACATACGCTATCATAGTATTTCCAACCTTACAGCTGGGAGTTTAGTTGAGGTAAACGCACCGCTACGTTGCTGCTAAGTCACACGAGACGCTATGAACAATCTCGGGTACACTAGCCCGGTTGATCCTGCAGGTACGGATACAACTGGAACAATCTCGTTATCTTCACACCCGCCACATGCAGCATCCGCCTAGGAATGCATGTCATGAGCCTATTACCGGGCTCATCTGGACCTTCTGCCTTATTATACGGAACCACGATCCCGAATTGCCGCCCAGACTTTTTCCAGACGCGAATCATTGACTGTAAACATCATGCTCCAATGAACCAACTCTTAGAAGCCGGCCCTATCCTCAACCGGGTGATCACCCATGTGAATTCTCTGGTTACCGACAAAGAGTGTTAGCCTCCCAGCTAAGCCAACCAGGTGCTCAGCTTTTAACCGGATGGAAGACACTGCGAAAAGCAGCATCCAATATGTAGCCAGTAAATAATCTCGGGGCAACAAGTTCAGTCGCCTGCTTGGAGATATGCTGTTGCTCGTTCCGTGGCACAGACAAACTGGTAGGTTCCGCCAGTTCGACTAATCAAGGACATCGGCGATACACAAACGCTCGTCAAGGGTTGTCCCTTCGATTGTTGACATTGTCTTTGCGCACTCTTCCGGACATATAGTAGGTTGGTAACGATCCTCTATCAACACCCTCCCTCCCAACAGTACCGCTGCGTGTCGAGTTTCCTGGCGGTTATAGCGTAACGATCGGCCCCAGTACGCAACTCATACGTCTCAAAGGGATAGATCCCTTCAACCACCGACATTAATGGGTGTCTTGAAGTGGATACAGTAATATAATTTGGCCTCACCCACTGCGGGTCGACGAAGCTCCCAGTTATCCACTAAAGGTGTCGCTTCGCAAGGACCTCCCTGGGGGGACGTCACGTCTGGAGCGGGACTCGGTCCCT";
    let text = "TTTTGAGTCGAATGAAAGTGGCTGCCGACAGAACTGGAGATTAGCAGGGGGGCAAGTAACCAATCGCCTATAGGACGATATCGGAATAAGTTGACAACCGGTCGCAGGTGGGAACCAACAATATTCTATCTCAAACCTAAGCGTGGATTGAGTCCTGCACGTTGAACTCGGTCCCCGGACCTCATGTGCGGTGTAATAAGTGGCATTTTCCTTTCGGGAACTTGGTTATATGAGTCCCCCGAAATTGCAGTAGGGGCGGCACTAAGCCTTTACAAATTGTGCTCACTGATATTCTTTCGGAGTCTAAGATAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTGATTAACTCCTTAACTGATTTTCCCGCTTTTGGTGGTGGCGAAGACTTTTTGTTACAACTCGTCACCTAAGGCCGGTATTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCGAAGGTACCAGCGTGAGCACATGGCACAGTCACGCGAACCGCTACGGTTTATGCTTCGTGTGCGTTGACTAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTGAGTTACTCGCGATATGAATACGGTCACGTCGGCCCCGTATGGGTCAGTGGTGCCGTGGCCAATGCAACCATTGTATCCAAATGGCCTCCTTGATGGATTTATGACCCATTCTTAGGGTTTTGCTCAGGTCCTATAGAGTGGCTAAAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTGCGTATAGTCGTGAGGTCGTGGGGAACAGCCGTGAGTAGCTTTTGAGCTACGTGAAGGACAGTATCCGCCGCAACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCACCCGACGCTTAAGAGTATAGCAGTCAAGACCCTGGATACCCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGGGCGGGATGCAAACCAACCAGACTTCGCTCAATCGATAAAAACCCCTGCGTTCAGCTGATAGGAAGCACAGGGATAAATGGACGCGAAGGAAGGACAGCACGGGAAAAACAGTAACGGGGATCTTGTGCTTGCCAAGCCTCGTACTATGTCGACATAGTTGGCTAGAGTCGCTCCCAGATCGGGTGCAAATTTCTTATGAAACACTCGCGGTAACTGGAATTGGCGGCCACGCGGGCAACTGTGTCTTTGGGTAGACGCTATCCAGTTTGCGAGATTCATCTGAGACCCTCCATGCCGACAGTACTTGTCTTTCTAGCCAGTCGCGGATTCAGCGTCTGCCCAATGAAACGAACCTAGGATTTGTTTTCAATCCCGTGCTACACCTGTGCACACGTAAGTACCCGAGATGCGTCTAACATTACTCTTGTGCTAGCGTGGGAACCTGGGTGGTTCTGACGCCTATGTGCCCCCGTCCGTAGATTGGCGGTCACCAATATACAGTGCAACCACCTATCGTCAGGGGTCTGGCTAAGTGTTTACGTCGCAAGCGATCACCCTTGGAGCTCAAGCCGGTGACTACGCGGGCGGGCGCGACGGCCCTGTCGCCTCGCGAGACCGAAATGACTCACTCAGCCCTTAACGGGGAGACCCGGACGGAAGCATCTCTAATCACTGTACAGGGCCCCGTTGAGACGGTCCTCGGGGGTTGGGTTTACTAATGCCATGACTTGGTACCGCACTGAGGCACCCCCGCCACGAACTAGACACCCAAGTGTCCACCATGCATAAGTAAAGCCTTGACGTCGCGATCAAGGTGTATCATTCAGGGGGAGTTCACATCAGCATATTAGAAAGGCACTGTAATTGAAGTGTATTGATGTATAGCGGGCAGCCCCAGTAAAGAGCCGTAGTATCCCACAGCGATTGCAGCACATCTCGCAATTTCCAGGTTAAATAGGCGCTTGATTATGCTGACTATCATGAGTTACCGAACTCAGCCGAGCGGTGTAAACTATGAAATAGATAAGGCAGTGAGCTCATCGGCCGGCACGTGGAGGACAAACGCAACAGTTACCTATAGCAATATTAGGTACTGTGTCGCCGGGTGGGCATTCTATACATCACGTCAGTTTAGGATCAGTTTCGAGGCTACGGGTCCAGTTGTAGTTGCGGATACGTGGTAAGGATACGAATAACAGACCTGGTAATTGGTGAGTAGAGGCGGTTGGATTGGCTGGGCCCCAGGATGGTCCAATAGATCGTCGTTAGGTCAAGCCCCAAAACCCGAAGCATGCTGGTCAGTATGGATATCTACCCGTCGGTTATCGTAGAAATCAGTACTGGTTGCGATAAATGTGATACACCCCTCGACAAACGGAGTCATTAGGCGCCTCTCTAAATACTGCTGTCCTGCAATGGCTATCGGGCAGGCTTCGCCAAGGATTCCTTCAGCATTGGCCCTTATATCCAGATGTTGATGAATGGTGGCGATTCGGGGTCTCCTGCCTGGACCAAACCTATGCCGCATTTGGGAGAGTCAATGCAGGGGGCGCTTTAATTAGTTGCTCCTCGCAAGCTCTCATATATAACAAGAGCTCGGGTACAATAAAGACCCCCGTCGCCCGGCGCTACTAGTCGGACGATTCCCTTTTCCATGTACCACGTGAGGTGCTGCCAGCCTCAAGCTCAAGCAATCTACTGAGAAACGGATTATCTACTCCCCATCAGAATGAGTGGAGGAGTTCAACTTGGATTAGACCCCCACTTACTTCGGCATCCAAAGAGGGTTACCCCTCCTGCTATGGTCTCCAAGCGAAGACTCTAAAGATTAGCGCGGGAGATAACTCGGCTGTGTATCAGACGCGGCACTTGGTTGCTATCCTTTCTGGCCTCAAGTCTAGGTGACCAAGTCGATCAATGCCGTCGCACTAAACAGGCGAGTTGACGTTGTTCCAAACTGCACGATTTGCTCCGTCACTACCGGGTTGAGAGGCCTCATTGGACACTCGCTCCAAATTTCACGCAAAACCTAAGTCATAGCCCTTGATCAGTAACTAGAACTGCAAGGCGGAAACATTAATCACACGGAGACTATAACGGTCATCTCAAATTCCAGTTTGTACTTGAAAATACAAGCGTCAAATCGAACATTTTCCTTAATAGTATCATACTCGTCTAATGAGCAGTCCGTGTTGCGTGTGGCGAACTCTTAGAGATCGGCACCTGATATCCTACCAGTTCCCGTCGCTACAGAACCTGAAGCAGTGCGGTTTTTCTCCACGTCGAGCGTGTCTGAGCGATCATTTAGAGTCCGTGGCCGACGAAAGGGTGATCTGGTACCCACCTATAGAGGAGTTGCTGCAGAGAATCCGTGGTGGGGACTTAGAAATCGTATCGTACGAGATCTGGTTTGGAGTTTCACTGTGCTTACTTGTATCCAACTTGATCGGTTGCGCACGTATATTGGTTATAGGAATTGGCCATTACTTCAGGTTCGCAGCCCGCTCCTAACCGTTAATTCGCGCTGACCGGTGTGCTTTGAAATAATAGTGACACCGTAACTTCCAACATGAACCAACGTTACACGTTCTATTTACACTCACCCTTCTAATAAAAGTTTTGAGGAATGAAGCGTACCCACGTCGGAACGAAAATTCAAGTGAGAGTGATTGTCAATACTCGAGAAGCTTAGCAAAGAAGTAATCAAGGTTAAGTTGACTCCACAGGATCCTAAAAACAATTATTCCGATAGCCTCGGGGTCAGTATACACCGCAGAAAGTACATTCGCCCTAATTAGCCTTAAATATTAAAATAGGCATACAAACCTGTTGTTGGAGTCTTCATACCGGCGTTGTTTCTGAGATCTCGGTGCCTACAGTATATGTACTATGTTAATGTACCAATCTTATGTTTTGGACGCAAAACATAAAGAACCACCAACATATGAACCCCCGGGTCGCGGGTTACAGGCGTCGTCCAACGCTACTTGCTCTAGATCAAACTGGTTATATATCCGAAAACAGGACTACACACTAAGAACCGTGGTTGAAGCGTAATCAATAACATCCGGCACTCGGCGCTTGGAACAACCCCACTCCCGATAGATCTCTCTGCCCCGTGGCTATAGGTTCTTCAGTTTGCGCGATAGGGCAGCTTTCGTCTTACTGCTGAACGTCCTAACCAATTTTCCGCTCGTGGGTCCAGGGCAACAAGGCTCCGAGATCCCCTTATACTAGAACGCCCTATATACGCAGAGAGAACAAGGACGGGTAAAAATAAAGTCCTTCTGGAAGGGTGACCGAACTCCCACCGAGGCCCTATCGCGCAAAGCGAATTCTTGTCAATTCACACCGATAACTCGGTAGTTAATTTGAATTAAAGAGTGTTTGCATTTGTACTAATATTCAATCGAAGTTAACCCTCCAGCCTACTGTAATTCCGCCAACCCGGGGATACAAGGAGGCACCAAACGGGTTTCAATTTGGATTGGTATGCATGCCCAATGAACTAAGTTTGGGCCACCGGGCCTGCATGACTATGTCTGAGAACAAAGATGAGGGAGCTGTTTGGTGAACTAGAAACTGCAATATGTGACGTACTATCCATACCAGCCCGCCGGTACGCGCATAGTAAACTAGTCCATGCCATAGTATCGGTTCGTACTTGACTGTATCGCTGAACACCGTACGGTCGTCAAGGTTCATAGGGGCTGGTAACAAGGGAACCATTAAATACTCTGGGCCCCCTCCGTTCTGACCGGCGGGGGTATATCCATGTGTACACTTTACCTAGCGAACAGCTGCCCTACTAGGACCTTATGCTTGTTACTTGCTATGACAGATCCCTGATTTCACAGAATGATGACGCAGCATGAACCAGCTCAGCCCGCAAATCGGGGATGGTGGAGTCGCGGATTTGCCTCCCTGGGCTTGCGACACTGTTGCCCGTAAAGAGTCTACCTACCAGTTCTCTAACTCTGGAGCCCTACCTCCCTTGTGCTTACTGGAGAATATATACCATGTGATACGAAGGGGGGTCCGGTGGACTTGATCCCGACGCTACTAACGGACAACTGAGCAAGAGGCCTACCGTTAAGCGTTGCATCTTTAATACCCTACCGTATTGCTCGCCGAGGTTGTTGAGGCATACCGACGATTACGGGAGATGACCTTCGGTGGAATGCACGACGGTCATTCTCGTCATTTAGCATCTAGAGAGTCCCAGCACTATCCGGGTGGCTCATAAATCTAAGCCTGTCTCATCGCTCTTTAGCATCGGAGGGAAGGAAGCCCTTGTAGGATCCTAGATCAGATGACGCAGAGACCTCATTGCAATCGCATGAGTTTGCTATATTTCCTGGTGGTAGCGAGCCCAAATGTATATCATTCGGAGAGACCCTACTCGGGTCCTCCTACCAAAGCTCAATCGGCTCTGACCAAGGTACTCGAGGACTCGTGCAAAGTCCCGGGGCCGCGGTCGCCCTATCTCTAGAGCGATATGTATGGGACTTTACGCAACTCTTACACGTAGATAGCAGTCATGCCTCCTATACTTGCCTGGCGAGCGAGGCTAGCCTATATTGGTCGCCGATCTCTCAGTCGGGACTTGATATTGGAAAAGATTACTGCGTTCGTTCTCGTGCGCTGCATTCTTGAGATTTGTACTGTTTACGTGAACTGGGCGTGCTTGGCCAGCGACGAGACTCTGTCCCATGAGGCAAACCTGACTCGAAACCAGAGGGGTTGTGCCTTCGATAGACCCTGCATTAGAATCGGCCCTATGTCTTCTCCATTACCACCAACTGTTTAAGGACTGAGAACATCCACCGCTTGTAGAACCCACATTCCTCCCGCGTGCACGATCCCTGGCTTAACATCGCCTTGCACCTGGCTCTCCGTAAATAAAGACTATCCTTAACCCGCAAGACCGAAGCAGCGTAGGTGTCCAGGCCGCCCCTGAGGCACTCTTCACATGGACATAGACCCTGGTGCTGAGACTTGTTTCTGGCTTGGTATATGAGTACAGCCGCACAGGACAGCAAAGTAGATACCTCGTAACGCCAGAGTGCTAGTGGAAATAACACAACCCAGGATTTACTCATAAGGCGCCTCTTTTACCCCTGCGCCAGATCATGGGTATAACTACTAGACAACGCGAGAAGCAGTGAGTATTCGACCACCGCTATTAGAACGGCATAATTACCAATCGATAGGAGGCATGGATATTCGCCAGAGGCCTTCTTACAGTTGTTGATTCGAACCCCTAGAACGTCTACGAGCTGCGAAAGCTAGACGCTGTCATACATGTACCCGACTAAGATATAGCCGATTCTTGCGCTTCCAGGTCAGCCATCGGGATAGTTGCGCTTCCTTACGATGGCAGGGACACGAAAGTGGTATGAGTGTTGGTACGCGCCTTGAAATTGGTGCAAAGCTAACGAGTTGGACGAAGCACCTAACCCGCTTCAATTACTTTCCAGTCGTGGTATGAAACCGCCACCAACCTCGCGCCTAGTTGGAGGCCACCGGTTGACATTTCCTCGACGGCGATGTGATTCCCAAAAGTACGCTTATCTGGCTGTTAAGCTTCGACCACCGATCAACCTAACTGCTCCTTCACGGTTTCAAGAAAACACGAGGCCCGAATGATAAAGTTCTAGCAGCCTTTTCGCTTTATTGAGGCTCCTGATAGATAGTTCGATAAGAAGAGTGCAGGCCCAGCGAGCAAAACTGCTCACCTATGTCTCTATATCCTTTATTACTACTATGTTTTTTGGACATGTTCACCAACAACCTGAGTACTACGGTGTTGGTCAGAATGATTGTCTGGTAGTCAACTGACAATAAGCCCTCCTACAATAAGGCCGGATGTATATACGGACAGATATCAGCGGAGCGTTCTTCTTGCCAACGCTTCTCCTGAAAGGCCCCCGAATGCCTGGCTGAAGCCGCTTCACGGGCTCGATACACAGACCGGGCACGTAAGCTCTGTCTATTCATAACGAACAAGTATAAGCTCAGTGCTCACCACTGAGTTGATGATTGATGAGGTATGCAATAGGGACAAAAATAGCGTTGCAATGATGGGTGGAAATTCCTCACCACATATTCGTCGTGGGCGATGCGCACTTTATACCAACACAATGTGACAGGTCTTATTAAAAACCGACAGGACCGGTGCGGTGGTGTGGTGGTATCTGGTCCTTGGCGCTCTGTCCGATAAGTTTTGCAGTCGGTTGGGTCCTTGCAGGAACACCTCGGAGCACATTACAAAGCGCCACGGGCCGTGGGAAAGACAGCCCAGTCGTAAACCGTTTGTGAGGAGCGCACTCTGGATCATCCATTATTGGCGAATTCATAGCTCACGTGGTATATGAGCGCTACTTCCAACAAGGGAGAGCGCCGCAGACACCCTTCAGCGTGAAATGAGTTACCGGCAGTCTCGTGATAGCAAGCCGACAAGCCTTCACTCCACGTAAAAACGGCTCTATTTTCCAAATGTCACGGCTTCTATGCGTATGAGTAACATTTACTGAGTTCATCACTGCACTACGACGATTATGACGGGCCAGCCACTAAAAGTTCAGGGCGCACAACTGGTCACTCGATCAAGCTGTAACTTACGAGGCGCATCCTCGATTCATAGTGCAGTTAACGATAGATGCAAAAAATGATGGGCATAATAAATCACAGAGACTACCACCGGACTGCGGCCTGGTAGACCTCGTGATTCGCCTCGATCGCCCAACTCGATCGCAAGATTGCCGTTAATCGAGTTATAGACCGCCGACGAATCTGTCGTAACATACTCCAAGACAAGATGGGTCATCGACTTGTTAAGGCAGGCAAGTTCAACAGTTGATAGTGTTAACTCGGATGGACCTTGAAGGAGCCCCATCAAGGATAGATATAGCACAAGTTGATGCATAGCGGCGAGGCGTATTCATATGGAGGGAGCAGTGCGGAATGTCTGCCCTCCATGAGCTTTATGAGATAGCTAGCGAGCCTCGTAGAGCCTCAAAATGAGTTGGGAGACAATGTACGGTAATCATCCGACATTGTTCTCGTCAATCCCACCAATCACGTCCGATGGGCCGTATCAACTGTAGGTATGTCCACGGAATATCTGATCTAACACTTTGAGTCTCTACATTCCAAGTATCACAACGCTCACCCGGCTTCTCCGCACCACGCGTCTCTGAATCTGGGTGACAGCTATAGCTCAATTGTCGGAGGATCGGAATCACTTCGATCGACCTAACTCACCAGATGCGAGAAGGGGGCTACCGCCTGAAACAGCGGATTGCGACAGTATCCGTTCGAACAATTTTATGTCGGCATAATGCCCCTTCTGTTTGGGCTCATTAATAAGATTGCCCTGATTCGTCACTAACCTAGACGACGACACTAGTAATACGGTTTAGCACCCACTGGACGGGTGGTGGTGCCTCGGTATAACATATCATAGTGGACGTTACAGATGTAACCTGCATCGAAATGGGTAGATCCCCAACAATGTCTAACTATTACGTGATGGCATACTAACTCCTCGGTTAGGAACATCTCACTCAGCCCCTCTACGGCTTATAGTTTTAAAACCGGTAACTGCATATTTGGTGGTACGACTCGTTAGGTGGCCTGATCCATAGTAGGCCTACAAATTTCTCCAAACAGGGCGAATAGGGACTGAGCGCGCAATACAGCGTACCGGCGTAACCGATTGGCGCGTTGGTTGGCCCGGTTAAGTCTCGCCCTATGGCAACGTTTAACGTACTCATGTACCCGTAGGTAACCCTGACGCCATCCCAGGACCTTGTGAGCTTCCATACAATGTGTAAATCCAGATGGTCTTACACCTGAGCACACAGCAAGCCCGCACCAGGCAATTTTCATTCACGAGCGTTAATTACTAGGTCATCTGCTACGTGTCCATTGGGATCACGGATCAAGATTAAATATGCAGATCTCATTAACTATACTGGTCGTGCTCTAGGCGTCATATACCTTCATTGCGACAGGTCAAGTTCCGGGGGCGTGGCTGAGCGCCCCAGGATAATCCCTTAGCTGACATACGCTATATAGTATTTCCAACTTACAGCTGGGAGTTTATTGAGGTAAACGCCACGCTACGTTGCTGCTAAGTCAACACGAGACGCTATGTAACAATCTCGTGGTACACTAGCCCGGTTGATCATGCAGGGTACTGGATACAACTGGAACTATCACGTTATCTTCACACCCGCCACATGCAGCATCCGCCTAGGAATGCATGTCGATGAGGCCTATTATCGGGCTCATCTGGAGCTCCTGCCTCTGATTATACGGAACCACGATGCCGAATTGCCGCCCAGACTTTTTCAACGCGAATCATTGACTGTAAACATCAATGCTCCAGTGAACCAACTCTTAGAAGCCGACCCTTCCTGCAACCGGGTGATCTCCCTGTGAATTTTCTGGTTACCGACAAAGTAGTGTTAGCCTCCCAGCTAAGCCAACCAGGTGGCTCAGCTTTAAACCGGATGGAAGACACTGCTAAAAGCAGATCCATACTGTACCAGTAAATAAGTTCGGGGCAACAAGTTCAGTCGCCTGCTTGGAGATATGCTGTTGCCTGTTCCGTGGCACAGACAAACCGGTAGGTTCCGCCAGTTCGACTGAATCAGGACATCGGCGATACACAAACGCTCGTCAAGGGTTGTCCCTTCGATTGTTGACATTGTCTTTGCGCTCGCCTCCGGACATATAGTAGGTTGGTAAACGATCCCCTATCAAAGACCCTCCCTCCCAACAGTACCGCTGCGTGTCGAGTTTCCTGGCGGGTTATAGCGTAACGACGGCCCCAGTACGCAACCTCACACGTCTGCAACGGGATAGGATCCCCTTCAACAGCCGACTTAATGGGTTGTCTTGAAGTGGATACAGTAATAAATTGGCCAATCCCAGTGCGGGTCGACGAAGCTCCCAGTTATCCACTAAAGGTGGCCACTTCGACAAGGACCTCCCTGGGGGGGAGTCACGTCTGGAGGGGACTCGGTACCCT";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 10000 10% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

fn wavefront_bench_l10000_e30(c: &mut Criterion) {
    let query = "TTTTTGACTCGAATGAAAGTGCGCTGCCGCAGAACTGGAGATTAGCAGGGGCAAGTGAACCATCCCCTTGGACGATACGGAATAAGTTGACAACCGGTCGCAGGGTGGAACAACACATATTCTATCTCAAACCTAAGGTGGATTGTAGTCCTGCACGTTGAAACTCGGTCCGGACCTCATGCGCGGTCGTAGAAAAGTGCATTTTCCTTCGGGAACTTGGTTATACTGAGTCCTCCCGAATTGCAGTAGGGGCGGCACTAATCCTTTACAAATTGTGCTCACTGACTATTCTTTCGGAGTCTAAGGACAAGAGCGTTGAGGCGCCTTTAGTCCTGAGTGGTATCGCCCACAGTTTATTAACTCTTAACTGATTTCCCGTCTTTGGTGGTGGCGAAGACTTTTTGTAAAGTCGTCACCTAAGGCCGGTGTTGTTCCGCGGACTGACTGGGGGCAAGTGGCCCAAAGGTACCAGCGTGGCACATGCACAGTCACGCGAACACGCTAAGGGATTATGCTTCGTTGCGTTGACTCAACACATTAGTCCCGCCATCACGGCGCCAATACACGAAGAGGGAACCCCCTGTGCCCGTCTGGAGTTGAGTTACTCGCGATGTGAATCGGTCACGTCGGCCCCGTATGGGTCAGTCGTGCCGTGTGCCAATGCAACCATTGTCTCCAAATGGCCCCTTGATGGATTGATGACCCATTCTTAGGCTTTTGCTCAGGTCCTATAGAGTACAACAGTGCTCACTATAGGTTTTAGTGTGGTTGTCTCCGTATAGTAGTGAGGTGTGGGGAACAGCCGTGAGTAGCATTTGAGCTAAGTGAAGGACAGATCCGACCGCACCGCTACAAAAGAACTATTTTGAAACGCTAGCCCTCCACCCGACGCATAAGAGTATAGCAGTCAAACCCTGGCGATACGCCCCAAGACACCTAGTATCCCAAACTCCTGTCACGCCACGCGACGGCGGGATGCAAACCAACCAGACTCGGTCGATCCGAATATAAAACCCCTGGCTTCAGATTGATAGGAGCAGCAGGGATATGGACGCGAGGAAGACAGCTACGGGAAAAACAGTAACGGGAATCTTGTGTTGCCAAGCCTCGTACTATGTCGACATAGTTGGCTAGAGTCGCTCCCAGATCGGGTGCAAATTTCTTATGAAACACTCGCGGTAACTGGATTTGGCGGCCACGCGGGCACTGTGTTTGGGTGCGCTATCCAGTTTGCGAGATCTCATCTGAGACCCCCATGCCGACAGTACTTGTCGTTGCTAGCCAGTCGCGATTCAGCATCTGCCCAATGAAACGAACCGTAGGAATTGTTTTCAATCCGTCTATCCTGTGCACACGTAAGTACCCGAGATGCGTCTAACATTACTCTTGTGCTAGCGTTGGGAACCTGGCTGCTTCCGAAGTCCTATGTGCCCCCGTCCGTAAAATGGCGGTCACCAATATACAGTTGCAACCACCTATCGGCAGGGGTCTGGCTAAGTGTTTATCGTCGCAAGCGATCACCTTGGAGCTCAGCCGGTGACTACCGAGCGGGTCGCAGACGGCCTGTCGCCTCGCGAGACCGAAATGACTCACTCAGCCCTTAACGGGAGACCCGGACGGAAGCATCTCTAATCACTGTACAGGCCCCGTTGAGACGGATCCTCGTGGGGTTGGGTTTACTAATGCCATGACTTGGTACCGCACTGAGGCACCCCAGGCCACGAACTAGACACCCAAGTGTATCCCCATGCATAAAGTAAAGCCTTGACGTCGGGATTATGGTGTATCATCAGGGGGAGTTCACATCAGCAATATTAGAAGGCACTGTAATTGAAGTGTCATGATGTAATAGCGGGCAGCCCCACGTAAGGAGCCGTAGTATCCCAAGCGATTGTCAGCACTCTAGCATATTTCCAGGTTAATAGGCGCTTGGAGTTATGCTGACTATCATGAGTTACCGAACTCAGCCGAGCGGTGTGAACTATGAAACTAGATAAGGCAGTGAGCTCATCCGGCCGGCACGTGGAGAGACAAACGCAACAGTTACCTATAGCAATATAGTGTAGCGTGTCGCCGGGTGTGGCATCATACGATCACGTCAGGTTAGGTCAGATTCGAGGCTACGGGTCCAGTTGTAAGTGGCGGAAACGTGCTAAGGATACGTAATAACAGACCTGGTAATTGGTGAGTAGAGGCGGGTTGGATTTGGTGGGCCCCAGATGGTCCAATGATCGTCGTTAGGTCAAGCCCCAAAACCCGAAGTCATGTGGTCAGTATGGATTCTACCCGTCGGCTTAGGTAGAAATCAGTACTGCTTGCGGTAAAATCGGATACACCCCTCGAAACAAACGGAGTCTTGGCGCCTCTCTAAATACTGCTGTCCTAAATTGGCCATCGGGCGGGCTTCGCCCAGGATTCTTACAGCATTGGCCCTTATATCCAGATGTTATGAATGGTGCGATTCGGGGTCTCGCCGACCAAACCTATTGCCGCAATTGGGAGAGTCAATGCAGGGGGCGCTTTAATAAGTTGCTCCCGCAAGCTCTCTATAATACAAGAGCTCGGGTTACAATCAAAGACCCCCGTCGCCCGCGCTACTAGTCGACGATTCCCTTTTCCATGTACCACGTGAGAGTGCTGCCAGCTCAAGCTCAAGCAATTTACTGAGAAACGATTAATCTACCCCATCAGAATGAGTGGAGAGTTCAACCTGGATTAGACCCCCACTTACTTCGGCATCCAAAGCAGGTTTACCCCTCCTGCTATGGTCTCCACAGCGAAGACTCTAAAGATTTAGCGCGGGAGAAACTCGGCTGTGTATCAGAAGGGCACTTGGTTGCTATCCTTCGGCCTCAAGTGCAGGTGACCAAGTCGATCAATGCCGTCGCACTAACAGGGAGTTGACGTTGTTCCAAACTGCACGATTTGCTCCGTCACCTACGGGTTGAGAGGCCTCATTGGAACTCGCTCCAAATTTCACGTAAAACCTAAGCATAGCCCTTGATCAGTAACTAGAATCAAGGCGGGAAACTTAATCACACGGAGACTATAACGGTCATCACAAGATTCCAGTTTGTACTTGAAATACAAGCGTCAAAACGAACATTTTCTTTAAATAGTACATACTCGTTAATGAGCAGTTCGTGTTGCGTGTGGCGAACTCCTTGGAGATCGGCACCTGATATCTCTACCAGTTCCCGTCGCAATAGAACCTGAAGCAGTGGCGGTTTCTCTCCACGTCGAGCGTGACTGAGCGATCATTTAGAGTCCGTTGGCCGACTAAAGGGTGATCTGGGTACCCACCTATAGAGGAGTTCGGTGCAGAGAATGCCGTGGTGGGGACTTAGATATCGAAGTCGGTATGAGATCTGTTTGGACTTGTCATCTGTGCGTACTTGTACCAACTTGATCGGTTGCGCACGTATATTGGTTATAGGAATTGGGCCATACTCAGGTTCGCAGCCGCTCCTACCGTTAAGACGCGCTGACCGGTTTGCTTTGAAATAATAGTGACACCGGTAACATCGAACATGAACCAACGTTTCACGTTCTATTTACACTCACCCTTCTAATAAAAGTTGATGAGGAATGAAGCGACCCACGTCGAGAACGAAAATTCAGTAGGGAGTGATTGTCAATAACTCGAGGTAGCTTAGCAAAGAAGTAATCAAGGTTAAGTTGCCCACAGGATCCTAAAAACAATTATTCCGATAGCCTCTGGGTAGAATACCCGAAGAAAGTACATTCGCCCTAATTCAGCCTTAAATCATTAAAATAGGCATACAAACCTGTTGTTGGATCTTCATACCGGCGTTGTTTCTGAGATCTCGGTGCCTACAGATATGTCGTATGTTAATGTACCAATCTATAGTTTTGGACGCAAAACATTAAAGAACCACCAACATATGAACCCCGGGTCGTGGGTTACAGGCGTCGTCCACGCTAACTTGCTCAGATGCAAACTTGGTTATCTAATCCGAAAACAGGACTACACCTAAGAACCTGGTTTGAAGCGTAATCAATACATCCGGCATCGGCGCTTGGAACAACCCCACTCCCGATAATCTCTCTGCCCCGTGGCTATAGGGTTCTTCAGTTTGCGCGATAGGGCAGCTTTCGTCTTACTGCTGAACGTCCTAACCAATTTTCCGTCGTGGGTCCAGGGCAACAAGGCTCCGAGATCCCCTTATACATAGAAGCCTATATACGCAGAGAGAACAAGGACGGGTAAAAATAAAGTCTCGGGAGGGTGACGGAACTCCACGAGGCCCTATCGCGCAAAGCGAATCCTTGTCAATTCACACCGATACTCGGTAGTTAATTTGAATTAAAGAGTTGTTTGCCAAGTTGTACTAATATTCAATCGGAAGTTAACCCTCAGCCTACATGTTAATTCCGCCAACCCGGGGATACAAGGAGCACCAAACGGGTTTCAATTTGTGGATTGGTATTGCATGCCCAAATGAACTAAGTTTGGGCCTCCGGGCCTGGATGACTATTCTGAGGTACAGAGATCGAGGGAGCTGTTTGGAGAACTAGAAACTCAATAGGTGACGTACTTATCCATACCAGCCCGCCGGTACGCGCATTAGTAAACTAGCCATGCCATAGTATCGGTTCGTACTCGACTGGATCGCTGAACACCGTACGGTCGCCAAGGTCATAGGGGCTGGTAACTAGGGGACACCATTAAATCTCTGGGCCCCCTCCGTTCTAGCGGCGGGGGTGATATCCATGGTAACTTTTACCTAGCGAACAGCTGCCACCTACTAGGACGCTTATGCGTGTTACTTGCTATGAAGATCCCTGATTTCACAGAATGTGACGCACCATGAATCAGCTCAAGCCCGCAAATCGCGGGATGGTGGAGACGCGGAATCGCCTCCCTGTGCTCTGCGACACTGTGCTGCCCGTAAAGAGCTACCTAGCCAGTTCGGTGACTCTGGAGCCCTACACCCTTGTGCTTACTGGAGAAATATACCTGTATACGAAGGGGGGTCCGGTGGACTTGACCGACGCTACTTACCGACACTGAGCAAAAGCCTACCGTATAAGCGTTGCATCTTTAATACCCTCCCGTATGCTCGCCGAGGTTGTTGAGGCAAACCGACGATTACGGGAGATGACCTTCGGTGGAATGCACGACGGTCATTCTCGCTCAATTAGCATCTAGAGTCCCAGCACTATCCGGGTGCTCTAAATCTAAGCCTGTCTCATCGCTCTTTAGCATCAGGAGGGAAGGAAGCCCTTGTAGGATCCTAGATCAGATGAGCAGAGACCTACATTAGACGCAGAGTTTGCTATATTTTCCTGGTGCTAGCGAGCCCAAATGTAGATCATTCGGAGAGCCCTACGCGTGGTCCTCTACCAAAGTTGCAATCGGCTCTGACAAGGTACTCGAGTGCTCGTGCAAAGTCCCGGGGCCGCGGTCGCCCTATCTCTAGAGCGATATGTATGGGACTTTACGCATACTCTTACCACGTAGATAGCATCATGCCTCCATACTTGCCTGTGGCGGGCAGCCTATACTGGTCGCCGAGATCTCTCAGTCGGGACTTGATATTGGAAAAGATTACTGGCGTTCGTTCTCGTGCGCTGCATTCTTGAGATTTGTACTTTTACGTGAACTGGGCTGCCTTGGTCAGCGCACACGGACTCTGTCCCATGGGCAAACCTGACTCGAAACCAGAGGGTTGTGCCTCGATTAGACCCTGCATTAGATCGGCCCTGTGTCTTCTCCATTACCACCAACTGTGTAAGTACTGAGTAACATCCCGCTTGTAGGAACCCTACATTCCTCCCGCGTGCACATCCCTGGCTTAACATCGCCTTGCACCTGGCTCTCGTAAATAAAGCTATCCTAACCCGCAAGACCGAAGCAGCCGAGGAGTCCAGGCACGCCCCTGAATGCACTCTTACATCGACATGACCCTGTGCTGAGACTTGTTTCTGGCTTGTGTATATGAGTACAGCCGCACATGACAGCAAAGTAGATACCTCGTAACGCCAGAGTGTCTAGTGGAATAAACACAACCCAGGACTTACTCATAAGGCGCCTCTTTTACCCCTTGCGCCAATCATGGGTATAACTACTAGACAACGCAAGAAGCAGTGAGTATTCGATCACCGCCTATTAGACACGGCATAATTACCAATCGATAGGAGGCAGGATATCCGCCGAGAGGCCTTCTTACAGTCTGTTGATTCGATCCCCTAGAACGTCTACGAGCTGCCGAAAGCTAGACGCTGCCCATACATGTACCCGACTAAATATAGCCGATTCTTGCTCTTCCAGTCAGCCTCGGGATAGTTGCGCTTCCTATACGATGGCAGGGACTACGAAAGTGGTATGAGTGATCTGGTACGCGCCTTGAAATTGGTGCAAAGCTAACGAGATTGACGAAGCACCTAACCCGCTTCAATTACTTCCAGTCGGGTATGAAACCGTCCACCAACCTCGCGCCTATGTTGGAGGCAACCGGTTGACATTTCCCGACGGCGATGTGATTCCCAAATAGTACGCTTATCTGGCTGTTAAGTGTCGAGCCACCGATCAACCTAACTGCGTCCTTCACGGTTTAAGAAAAACGACGCCCGAATGATAAAGTTCTAGCAGCCTTTTCGCTTTATTGAGGCTCCTGATAGATAGTTCGATAAGAAGAGTGCAGGCCAGCCAGCAAAACTACTCAACTATGTCTCTATATCCTTTATTACTCTTGTTTTTTGGACATGTTCACCAACAACCTGAGAACTACGGTGTTGGTCTAGAATGATTGTCTGGTAGTCAGCTGACAATAAGCCCTCCTACATATGACGGGATGTAATATACGGGACAGAATCATCGAGCGCGGTCTTCTTGCCAACGATTCTCCTGAAAGGCCCCGAAAATGCCTGGCTGAAGCGCTTCACAGGGTCGAGTACACAGACCGGGCACGTAAAGCTCTGTCTCATTCATAACGTACAAGTATAAGCTAAGTGGCTCACCACGGAGGCTGATGATTGATGAGGTATGCAATAGGTGACAAAATTAGGTTGCAAATATGGTGGAAATTCCTCCACACATATCGTCGTGGGCGATGCAGACTTTATACCAACACAATGTGACAGGTCTTTACTAAAAACCCGGCAGGACCGGTGCGGTGGTGTGGTGGTATCTGGTCCTTGGCGCTCTGTCCGATAAGTTTTCGCAGTCGGTATGGGTCCTTGCAGGAACAACCTTCGGGAAGCAATTACAAAGCCCACGGCGTGGGAAAGACAGCCCAGTCGTAAACCGTTTGTGAGGAGCCACTCTCGGATCATCCATTATTGGCGAATTCATTACTCACGTGGTATATGAGCGCTACTTCCACCAGAGGGAGAGCCCGCAGACACCCTACAGCGTGAAATGAGTTACCGGCAGTCTCGTGATAGCAAGCCGAGAAGCCTTTCTCCACGTAAAACGGCTCTATTTATCCAAATGACACGGCTTCTATGCGTATGAGGTAAACATTACTGAGTTCATCACTGCAGTACGACGATTATACGGGCCCGCCACTAAAATTCAGGGCGCATAACTGTCACTCGATCAAGCTGTAACTCACGAGGCGCATCCTTATTCATAGTGCAGTTAACGATCGGTGCAAAAAATGATGGGCATAATAAATCACAGAGACTACACCGGACCTGCGGCCTGGTAGAACCTCGTGATTCGCCTGCGATCGCCCAACTCGATCGCAAGATTGCCGTTAATCGAGTATATAGAAAGCCGACGAATCGTCGTAACATACTCCAAGACAGTTGCGGCCATCCGACTGGTTAGAGGCAGGCAGAGTTCAACAGTTGATATGTTAACTCGGATGGACCTTGAAGGAGCCCCATCAAGGAAAGATATAGCACAAGTTGCATGCAGTAGCGGCGAGGCGTATTCATATGGAGGGAGCAGTGCGAATGGCCGCCCTCCAATGAGCTTTATGAGATGAGCTAGCGAGCCTCGTAGGCGCCTCAAAATGAGTTGGCAGACAATGTACGGTAATCATCCGACAACTTTTCTCGTCATCCCACCAATCACGTCATGGGCCGTATCAACTGTAGAGTCCACGGAATATCTATCTAACACTTTGAGGTCTACCTACATCTCCAAGTATCACAACGCTCACCCGGCTTCTCCAGCACCACGCGTCTCTTGAATCTGGGGACAGCTAGACTCAATTGTCGGAGGATCGGAATCACTTCGATCGACCTAACGTCACCAGATGCGAGAAGGGGCTACCGCCTGAGACAGCGGACTTGCTGACAGTATCCGTTCGAAGCAATTTTATGTCGGCATAATTGCCCCTTCTGTTTGGGCTCAGTTAATAAATTGCCTGATTTCGTCACTGACTTAGACGACGACACTATAATACGTTTAGGCACCCACTGGACGGGTGGTGGTGACCTCGGTATACCATATCCATAGTGCGAACGCTACAGATGTAACCTGCATCAAATGGGTTAGATGCACCAACAATGTCTTACTACTTACGTGATGGCGATGACTAACTCCTGGTTAGGAACATCTCACTCAGCCCCGTCTACGGACTTATAGTTTTAAAACCGGTAACTGCATATTGGTGGTACGACTCGTTAGGTGGCGCTGATCACATAGTCAGGCCTACGAAATTTCCAAACAGCGGCGAATAGGGACGGAGGAGCAATACAGCGTACCGGCGTAACCGATGGCGCTTGGTTGGCCCCGTTAAGTCTCGCCCTATGGGCAACTTTAACGTACTCATGTACCGTAGGTAACCCTACGCCATCCCAGGACCCTTTGAGCTTCCATACATTGTGTAAATGCCAGATGGTCTTACACCCTGAGCACAAGCAAGCCCGCACCCCAGGCAATTTTCATTCACGAGCGTTAATCCAGGTCATCTGCTACGTGTCCATTGGGTTCACGGACAAAGATTAAATATGCAGATCTCATTAACTAGACTGGTCGTGCTCTAGGCGTCATATACCTCATTTGCGATGGCCAAGTTACGGGGGCGTGGCTGAGCGCCCAAGGATAATCCCTTAGCTGACATACGCTATCATAGTATTTCCAACCTTACAGCTGGGAGTTTAGTTGAGGTAAACGCACCGCTACGTTGCTGCTAAGTCACACGAGACGCTATGAACAATCTCGGGTACACTAGCCCGGTTGATCCTGCAGGTACGGATACAACTGGAACAATCTCGTTATCTTCACACCCGCCACATGCAGCATCCGCCTAGGAATGCATGTCATGAGCCTATTACCGGGCTCATCTGGACCTTCTGCCTTATTATACGGAACCACGATCCCGAATTGCCGCCCAGACTTTTTCCAGACGCGAATCATTGACTGTAAACATCATGCTCCAATGAACCAACTCTTAGAAGCCGGCCCTATCCTCAACCGGGTGATCACCCATGTGAATTCTCTGGTTACCGACAAAGAGTGTTAGCCTCCCAGCTAAGCCAACCAGGTGCTCAGCTTTTAACCGGATGGAAGACACTGCGAAAAGCAGCATCCAATATGTAGCCAGTAAATAATCTCGGGGCAACAAGTTCAGTCGCCTGCTTGGAGATATGCTGTTGCTCGTTCCGTGGCACAGACAAACTGGTAGGTTCCGCCAGTTCGACTAATCAAGGACATCGGCGATACACAAACGCTCGTCAAGGGTTGTCCCTTCGATTGTTGACATTGTCTTTGCGCACTCTTCCGGACATATAGTAGGTTGGTAACGATCCTCTATCAACACCCTCCCTCCCAACAGTACCGCTGCGTGTCGAGTTTCCTGGCGGTTATAGCGTAACGATCGGCCCCAGTACGCAACTCATACGTCTCAAAGGGATAGATCCCTTCAACCACCGACATTAATGGGTGTCTTGAAGTGGATACAGTAATATAATTTGGCCTCACCCACTGCGGGTCGACGAAGCTCCCAGTTATCCACTAAAGGTGTCGCTTCGCAAGGACCTCCCTGGGGGGACGTCACGTCTGGAGCGGGACTCGGTCCCT";
    let text = "TTTCGAGTCGAATGAAAGTGGCTGCCGACAGAACTAGGAGATTAAGCAGGGAGGCAAGTAACGCAATCGCAGTCATAGGACGTATCAGCATAAGGTGACAACCGGTGCAGGTGGGACCAACATATATTTATCCAAACTAGCGTGGAATTGAGTCCTTCACGTCGAATGGTCCCCGGACCTCATGTGCGTTTGTATAAAAGCATTTCCTTCGGGGACTTGGTTATCATGAGCCCCGAATTGCAGTAGGGGTCGGCATAAGTCTTACAATTGTGCTCACTGATAATTCATCGGAGTCAGATAAGAGTGTTAACGCCGTTTAGTCCTGAGGTGGTATCGCCACAGTTGACCTTAACTCCTTAAACTGTTTTCCGCGGCTTTTGGGTGGTGGCGAAGACTTATTTGTTACAACCTACGTCCTCTAAGGCCGGTATTGTTCGCGGACTGATAGGGGCAAGTGCCCGAAGGTACCAGCCTGAGCACATGGCACAGTCTACGCGCACCGCAACCGAGTTGATGCTTCGTGTCGTTGACTAACACATTAGTCCCGCCACCAGGGCCAATACACAAGAGGGCGAACTCCCCTGTGCCCGTCTGGTAGTGAGTATACTCGGGATATGAATAGCGGTCACGTCGGCCCCCGTAGGTCAGTGGTGGCCGCGGCCAATGCACCATTGTATCAAATGGCCTCTTGATGTTATATGACTCCATTATGTAGGGTTATGTCAAGTCTCCTAGAGAGTGGCTAAAGTGCTCATCTATAGTTTTAGTGTTGGTTCTGCGTAATAGCGATGAGGCCGTGGGGAAACAGCCGTGAGTAGCTTTGACACTACGTGAGGGACAGTATCCGGCGCAACCGCTACAAAACGAGTATTTTGAAACCTGCCTCACCCGACGCCTTAAGAGATATAGCAGTCAAAGACCCTGGGGATATCCCCCCAAGACACCTAGTATACCAAATCCTGTCACGCCACGCGGGCGGGATGCAAACCTACCAGACTTCCGCTCAATCGATAAAACCCCTGCGCTCAACTGATGGAAGGCAACAGATAAACTGAGTACGCGAAGATGGGACGAGCCGCGAAACAGTAATCGGGGAAGCTATCTGCTTGCCCAAGCCTCGTACTATGTGAGCATAGTTGGCAGAGTCGCATCACAGATCGGGGTGAAATTTTTATGAAACACTCCGGTAACTGGATTGGCGCCACGCGGGCAAATGTGTACTTTGGGCACGACGCAGGCCAGTTTGCGAGATTCATCAGGATCCCTCCATGCCGTCAGAATTTCCTTCTAGCCAGTGCGGATTTCCGCTACTGCCACATGAAACGAGACTGGACTTGTTCTTCATCCGTGCTAACACCTGTCCACACGTAAGCACCGAGTTCTCTAATATTACTCTGTCCTAGCATGCGAAACCTGGGGGTTCTAGCCTATGTGCCCCCGTCCCGAAGATTGGACGGTCGCCAAATACAGGGCAACCAACCTATCGTCAGGGGTCGTGGCTAAGTGTTACGACACAAGCGATCACCTTGGAGCTCACGCCGGTGACTCACGCGGGCGGGCGCGACGGCCCTGTGCCTCGGAGACCGAAATGCTCACTCAGTCCTTAACCGGGGAGACCCGGACGGAAGCATCCCTATCACTGTACAGGGCACCGTTGAGACGTCCTCGCGGGGTTGGGTTATACTAATACCCAGAACTTTTGGTAACCGCACTGAGGCACCCCCGCCAGGAAACTAGACACCCAAGTGTCCAAAAGCATAAGTAAAGCATTGACCTCGCAGATCAAGTGTATCATTCAGGGGGAGTTCACATCAGCATATTAGAAAGGCACTGTAAAGAAGTGTATTTGATGTATAGCGGGACAGCCCGCACGCTAAAGGCCGTAGTACCCACAGCGATTGCAAGCACAGTCTCGCACGATTTCTAGGTTGAAACAGGCGCTTGATATATGCTGACCTAGCATGACAGTTACCGACTCAGCCCAGCGGTGTAAACTTGAAATAGATACGGCAGTAGCGCATTCGCGGCACGAGGGAGGACAAACGCAACAGTTACCTAGAGCATGATTAGGTACTGTGGTGCGCCGGGTGGGGCATTCTATCACATCCGTCCGTTTGGGATCAGTTCTGAGGTCAAGGTCCAGTTGTGTTGCGGATACGCGGAAGGATAGAATAACAGACCTGTAATTGGTGAGGGTAGAGGCGGTTGGTTGGCTGGCCCCAGGAGCGGTGCAAGAGACGTCCTTAGGTCATAGCCCTCAGAATACTCGAAGCATGCTGGTGAGTCTGGATATCACCGTCAGGTTATCCGATAGCAATACAGTACGGTGTGCGATAAATGTGATACACCCCTCGACATAACGGAGTCATTAGCGCCATCTCTAAACCTGCTGTCTGCAATCGGCTATCGAGCAGGCTCACCAAGGATTCCTTCGCATTGGCCCTTAGATCAGATTGTTTGATGAATGGCGGCGATTCGGCTCCCCTGCCGGACAACCTAGGCGCATTTGGGAGAGTCAAGGCAGGGGGTCGCTTTAATAGTTGCTCCTCGAAGGCTCTCATATATACAGAGCTCAGTACAATAAAGAACCCCCGTCGCCCGGGCTATAGTCGGACGATCCCTTTTCCATGTAGCCCACTTAGGTGCTGGCCAGCCTCAGCTCAAGCAATTAACCCTTAGAGACGGTTATACCATTCCCATCAGAATGAGGGAGAGTTCAACTTCGGACTAGACCGCCCCATCTATTTCGGCATACCAAAGAGGTTTACCCCCCTGCTATGGCTGCAAGCGAAGACTCTAAAGATAACGCGGGAGATAACTCGCTGTGTATCAGAGGCGGCACTTGGTTGCTATCCTTTTGGCCTCAGTCTAGGTACCAAGTCGATATCAATGCCGTTCCACTTAAACCACGAGTTGACGTTGAATCGAACTGACAGATTTAGCTCTCGTCACTACCGCGTTGAGGGCCTTCGCGACATCGCTCCAGATTTCACGCAAACCTAAGTCATAGCCTTGATCAGTAACTAGAACTGCAGGGGGTAACATTAATCACACGGAGACTATAACGGTCATGCTCAAATTTCCAGTTTCGTACTGAAAGATACGAAGCTGTCATATATCGAACATTTTGCTTAATAGTATCCATAGCTCGTCTCATGAGCATGTGTTGCGTGTGCGAACATCTAAAGTCGTGCACCTTATATCTAGCAGTTCCCTGGCTAACATGAACTTAAGCAGTGCGGTTTTTCTCCAAGGCGACCTTGTCTGAGCAGATCATTTAGAGTCCGTGTAGGCCGACGAGGGTGACCTCGGTACCCACCTAAAGGAGGAGTTGCTGCAGAGAATGCGTGGTAGGGACTGAGAACCTATCGTACAGAGATCTGGTTTTGGAAGTTTCACTGTGTCTCACTTGTATCAACTTAGATCGGTTGTCTGACGTATATATGGTTATAGGATGGAGCATACTCAGGTTCGCAGCCCGCGCGTAAACCGTTAATTCGCGCTGACCGGGATAGCTTTGAACAGTTAATAGTGACATCGTAACTTTCCAAGATGACCCAAGTTTACACGTTCTTATTTTCACGTCCTCCTTCATATAAAATTTCGAGGAAAGAAGCGTCCCCGACTCGGAACGACAAATTCAAGAGTGAGATATGTCAATAATCGAGAACTTACCAAATAAGTAATTACAGGTTAAGTTGACTCCACAGGTTCCTAAATACAATTATTTTGATAGCTCGGCTCAGTATACACCGCGCAAGTACAGTTCGCCCTAACTAGCCTTAAATATAATATATGTAGTACAATACCTTTCTTGATGTCTTCACTAGCGGGCGTGATTCTTGAGATCTCGGTGCCTACAGTATTGTGCTATAATGTACCAATCTTATGTTTTGGACGCAAAACTTATAAGAAACCACCCAACTATGAAACCCCCCGGGTCCGAGGTTACTAGGGTCTCAACCGCTAGTAGCTCTTGATCAAAGCTGGTTATATATCCCAAAACAGGACTACCACTAAGAACCCGTGGTATGAAGCGTAATCAATAACATCCGGCACTCGGGGCTTGGAAAACCCCACCCTATGACTCTCTGCCTCCGTGGCTAGTAGGTTCTCAGTTGCGCGCCATAGGGCAGCTTCTTCTTATTACTGAACGTCCTCAACACAAATTTTCCGCTCGTGGGTCCAGGGCAACAAGGCCAGATCCCGCTTATATTGAAAACCGCCCTCATATACGCAAGAGACAAGGACGTTAAAAATACAGTCCCTTCTGGTATGGATGACGAACTCCACCGTAGCGCCCTACTCCGCGCAAAGCTAATTCTCGTCAATTCACACCCGATAACTCGGTGTTAATTTGTTAAAGAGTGTTTGATTTGTACTAATATTCACTTCGAGTTACACCCTCGGCCTAACTGTCATTCCGCCAACGGGGTTCAAGGAGGCAACCAAACGGGTTTAATTTGGTGGATGCATGGCCCAATTCAAATAAGTTGGGCACCGGCTCTCATGACTGTGTCCGAGTCAAAGATCGAGGGAGCGTGTTTGTGATCTAGAACTGCACATATGTGACGATACTATCCATCCAGCCGCCGGTACGCGCAATATAAACCTGTCCATGCCATAGGATCGGTTCCTACTTTTTCCGTTTCGCTGAACACCGTACGGTCGTCAGGTTCATAGGGGCTGGGTAACAAGAGGAACCATTAAATTAAGCCTCTCGAGGCCCCCTACGTTCTGACCGGGGGGTATACTCCATTGTGTACACTTTCCTAGCCGAACAGCAGCACCCTACGTAGGACCTCATCTTTTACTTGCTCATGACAGATCCCAGATTCGAATAATGACACAGCACGAACCAGCCAGCCGACAAATCGGGATCGTGAGTCGCGGATTTGCCTCCCATGGGCTTGCGACACTGTTACCGTACAGAGTCTACTCCGCTTCTCATAACTCTGGAGCCCTACCTCCCTGGTCTTCTGGAGCATTATATATCATGTGATAGCGAGGGGGTCCGGTGGCCTGATCCGACTCTACTAACGGAAATGAGGCAAGAGTCCTACGCATTAAGCACTTGCATCTTGTAATACCCTACCGTATTGGCTCGCCCGAGGTTGTTGAGTATACCACGTTACGGGGGACGATCGTCCGGTGTAAGACGACGGTCATCTGTCATTTGCATCAGACAGTCGCAGACTATCGCGGGTGGTCTTAAGCTAAGCCTGTCTGTGGTCTTTAGCCATGAGCGGAAATGGAAGCCCTTTAGGATCCTAGATCCGATGACGCAGAGACCCTTGCAAGTGGCATCGAGTTTTGCGATATATGCCCTGTAGCTTAGCGAACCCAAATGTCTATCATCGAAAACCCTACCGGGTCCTCCTACCAAGCTAATCGGCACTGACCAAGGTACTCGGAGGACTCGTGCAAAGTCCCGGGGCCGGCGTCGCCTATCTCTAGAGCGATATGTATGGGCCTTACAACTCTTACACGTAGATAGGGCAGTCCATGCCCTAATACTTGCCGCGGCGAGCGAGGCTAGACTATATCGGTCGACCGATCTCTCAGTCGGGCTTGTATATTGCTGAAGATTACTGCGTCCCGCTCTCGTGCGATGTTTCTTGAGATTTGTAGTGTTTACGTGTAACTGGGCGTGCTTGGCCAGCCAACGAGATCTGTCCCATGAGGCCAACCTGACTTCGACACCGGGGTTGTGCCTTGATAGAACCCTGCATAGATCGTGCCCCATGTCTCTCCATTACCACCATCTGTTTAAGACTGAGAACATCCAACCGCTTTGAACCCACATTCCTCCCGCTTGCACAGATCCCTGCTGAACAACGCCCTTGCAACCTTGGCTCTCCGTAAACTAAAGACTATACTTAACCCGCAAACCGGAGCAGCGCTAGCTGCCCAGGCCGCCCCTGGGGAGTCTCACATGGACATAGACCCTGGTGCTGAGAATTTGTCCTGACTTGGTATATTGAGTACAGCCCACAGGAGCAAGCAACGGTAGATACCTCGTAACCGGTGGGCTGTTGGAAATAACACAACCCAGGCATTCACTCATAAGCCTGCCTACCGTTTCACCCCTCGGCAGATCCATGGTATAACTACTAGACAACGCGAAGTGAGTGAGTTTCGACCACCGGTATTAGTACGAGCATACATCCAATCGATAGGAGGCATGGATAGTTCGCACAGGAGGCCTCTTACGCGTTGGTTGATTCGAACTCCTGACGCCTACGAGCTGGCGAAAGCTAGACATCGTCAACATGTACCCGACCAAGATACTAGCTCGATTCTCGCGCTTCACCGGTCACCCACGGGATGTAGCTCCTTCCTTACCAATGGCAGGGATACGAAAGGTGTATGAGGTTCGTACGCGCCTTGCAAATTGGGTGCAAATGCTAACGAGTTTTGGACGAAGCTCCTCAGCCGCTTCAATGAGCTCTCACAGGTCGTGGTTGAAACTGGCCACCAACCTCGGCCTAGTTGGTGGCCACCCAGTTGTTATTTCCCAGACGGCGATGTGATGTCCCAATAAGCCGTTACCTGGCTGTGAGCTTCGACCACCGATCACCTAACTGCGCCTTCAGGTTTCCAACAAACACGAGCCACTAATGATAAGTTAAATCAGGCCTTTTGCTGTATTGAGGTCTAGATAGATAGTTGATAAGAAGAGTGTAGCTCAGCGAAGCAAAACCTGTCGTCACCTATGTCTCTATAGTCCTTTTTATAACCGTATGTTTCTCTGGACATGGTTTGACCACAACTCCTGAGTACTACGGTTTGGTCAGAAGATTGTTCTGGAGTCAACTGACAATAGCCCTCCTAGATAAGGCCGGAAGTAATCATGAGGACAAGATGATCAGCCTAGCGTTCTTGCTTGTCAACACCTTCTCCTGAAAGGCCCCCGAAAGCATGGCTTGAAGCGCTTTCACGGGCTCGATACACAGACCGGGCACGTAGGCTCAGTCTATTCATAACGAACAAGTAAAGCTCAATGTCTACCACGGAGATTCATGATTGTGGAGTATGCAATAGGAACAAAAATCAACGTTGCTAATGATAGGGTGGGAATCCTCACCACATTCGTCGTGTGTGATGCGCATTTAGTACCAACACAGATAGACACGTTTATTATAAAACCGAAGACCGGTGCGGTGGTATGGTGAGATCTCGTCCTTGGCGCTTGTCCGATAAGTTTTCAGTCGGTTGGGTCCTGAGGAACACCTCGGGCACATTACAAAGCGCCACGGGCCGGGGAAAAAAGCCCACGTCGTAGTCCGTGTTGAGAGCGCACTCTGGATCATCCATTATTGGCGATTCATAGCTTACGTGGTTATGAGCGCTACCCACATAGGGAGAGCGCCGCATACACCTGGGCGTGAAATGAGTTCCGGCAATTCTCGTTGGTATAGCAATCCGACAGCTTTACTCCAGTAAAAAACGGCTCTATTTCCTAAATGTCACGTGCATGCGTATGCAGTACACTTACTGATGCATCACTGCATACACGATTTGACAGGCCAGGCCACGAAAGTTAGATGGGCGCACAACTGGGTTCACTCACGTAAGCTGTATAACTTACGAGGCGCATCTCCATTACAATAGTTGGAGCTAAACAATAGATGCAAAAAATGGAAGCGCATAATCAAACAGACAGAGACTATCACCTCTGCGGCACTGAGCTAGACCTCGTGATTCGTCTCGATCGACCCAACTCGATCGCAAGATTCCGTTAATCGAGTTATATGACCGCCGACGAAATCTCGTGCTAACAACTCCAAGACAAGAGTGGGTCATCGACTTGTTAAGGCGGGCAAGTTCAACTTAGTTGATTAGTGTTAACTCGTGAGTGGACCTTGAAGGAAGCCCCATCAAGATATAGATATAGCACAAGTTGATGCATAGCGGCGAGGGTATTTGATAATAGAGGGAGCAGCGCGGAATGTCATCCCCTCCATGAAGCTTTATGAGATAGCTAGCGAGCCTCGTAGAGCCTCAAAGATGATAGGGAGCAATGTACGGGTGAATCATCCGACATATGTTTCTCGCAATCCCACAAGTAGTCCGATGGGCGTATCAACGATTAGGTAGTCACGGAATATCTGATCTGACACTTTAGTCCTTACATTCAAATATCACACGCTCACCCGGTCTTCTCCGCACCCGCGCCTCTGAATCTGGGTGACAGATATAGCTCAATTTTCGGACGTCGGAATCACTTTCAATCGACCTAACTCACAGATGCAAGAAGGGGGCTACTGCCTGAATCAGCGGATGGGAAGTTCTTAACAATTTTATGTCCGTATAATGCCCCCTCTAGTTTACGTCATTAATAAGCTTGCACCTCATATCGTGACTAACGTAGACGACGAGACTAGTAATACCGGGATTAGGCACCCCCTGGACGGGTGGTGGGTGCGTGGTATAACAACTTCATACTGAGATCGTTACAGATGTAACCTGCATCGTAATGGATAGGTCCCCAAACAATGTCGTAACTATTGACGTGATGGTAATACTAAATTTTGGGTTAGGGACACACCTCGTCCCTCTCGGCGTATGAGTTGTAAAAACTGTAACTGCATAATGTGGCGTGTACGACTCGTAGGTGGCCTGCACCATGTAGCCTACAATATTCTCCAAGTCAGGGCGAATAGGGATGAGCGCGCAATACAAGCTTACCGGCGTAACGATTGGCGCGTTCGGTGGCCTGGTTAAAGTCTCGCCTATGGCAACGTCTTAACAGTATCATGTCCCGTGGTAACCTGACGCCATCGCAGGACTTGTCACTTCCATGCATGTGTAAATCCCAGATGGTCTTACAACACTTGAAAGACAGCAAGCCCGCACAGGCAATCTTCATTCCGTAGCGTTATATTACTAGGTAGCTCTACATGTCCATTGGGACACGGATCAAGATTAATGTATGCGATCTCATTAACTAATGGTTTCGTGCCTAGGCGTCATTTACCTTCATTGCGAAGGTTCAAGTTCCGCGAGTGCTTGGCTGAGCGCCCCAGGGACTATTCCTTAGCTGAGCATACGCTATATAAATTTCTTAACTACAGCTGGCAGTATTATGAGGTAAACGCCATGCTACGTTGCTGCTATCAACACAGACGGCTATGTAACAATTCTGTTTTACACTAGCCCGGTTGATCATGCAGGCTCTGGATACCGGAAACTATACGTTATCTTACACGCTCCGCCACATGGAGCACCGCCTAGGGAAGCACTGTGCGATGAGGCCTACATCATCGGGGTCGTTGAGATCCTGCCTCTGATATACGGAACCACATCCCGAATCTGCCGCCCGCTTTTCAACGCGATCATTGACTGTAAACATCTATCTCTCAGTGAACCAACTCTTAGAAGCCCTACCCTTCCTGCAACTGGGTGCATCTTCGCCTGTCAATTTCTGTTCCCGACAAAGTAGTGTCTAGCCTCCCAGGTAAGCACCAGGTGCTCAATTTTAAACGATGGAAGACACCTGCTAAAGCAGATCCATACCGTACCAGTAAAGTAAGTTTGTGGGCAGCAAGTTCATGTCGCTCTGCTAGGAGATATCTGTTGCCTTTCCGTGGCAAGACAAACCGGTAGGTTCCACCGATCGACTAGAACTGCAGGAGCAGCGTGCGATACACACACGTTCGTCAAGGTTCCACCTGCGAGTGTTGAATTGTCTTGCATTTCGCCTCCGGACATATAGCAGGTTGTAACGATCCCCTACAAAGATCTCTCCTCCCAACCAGAACCGCGCGTTGTCCGAGTTTGATGCGGGTTATACGTAACGACGGCCCCAGTACGCAAACCTCACACGTCTCCGAACGGATAGATCCCTTCCAACAGCCGACTTAAGGCTTGTCTGACAGCTGGATAAGTAATAAGTTGGGGCCGATCCAGCGGCGCGGTCGACAAGTCTCTCAGTTATCCAACCTAAAGGTGAGCCACTATCAGACAAGGAGCCTTCCCTGTGGACGGGGACGTCAACGTCGGTAGGGGACTCGGATACCCT";
    let pens = Penalties {
        mismatch_pen: 1,
        open_pen: 2,
        extd_pen: 2,
    };

    c.bench_function("wfa length 10000 30% error", |b| b.iter(|| wavefront_align(black_box(query), black_box(text), black_box(&pens))));
}

criterion_group!{
    name = benches_100;
    config = Criterion::default().significance_level(0.05).sample_size(10).measurement_time(Duration::from_secs(1));
    targets = wavefront_bench_l100_e1,
              wavefront_bench_l100_e10,
              wavefront_bench_l100_e30,
}

criterion_group!{
    name = benches_1000;
    config = Criterion::default().significance_level(0.05).sample_size(10).measurement_time(Duration::from_secs(10));
    targets = wavefront_bench_l1000_e1,
              wavefront_bench_l1000_e10,
              wavefront_bench_l1000_e30,
}

criterion_group!{
    name = benches_10000;
    config = Criterion::default().significance_level(0.05).sample_size(10).measurement_time(Duration::from_secs(60));
    targets = wavefront_bench_l10000_e1,
              wavefront_bench_l10000_e10,
              wavefront_bench_l10000_e30
}

criterion_main!(benches_100, benches_1000, benches_10000);
