
use phf::phf_map;
use phf::Map;

/// # Sample code
/// ```
/// let country = rust_iso3166::from_alpha2("GB").unwrap();
/// let subdivisions = country.subdivisions();
/// assert!(subdivisions.unwrap().len() > 0);
/// let country = rust_iso3166::iso3166_2::from_code("GB-EDH");
/// assert_eq!("Edinburgh, City of", country.unwrap().name); 
/// ```

/// Data for each Country Code defined by ISO 3166-2
#[derive(Debug, Copy, Clone)]
pub struct Subdivision {
    ///Name
    pub name: &'static str,
    ///Subdivision Type
    pub subdivision_type: &'static str,
    ///Code
    pub code: &'static str,
    ///Country Name
    pub country_name: &'static str,
    ///Country Code
    pub country_code: &'static str,
    ///Region Code
    pub region_code: &'static str,
}

/// Returns the Subdivision with the given code, if exists.
/// #Sample
/// ```
/// let sub = rust_iso3166::iso3166_2::from_code("SE-O");
/// assert_eq!("Västra Götalands län", sub.unwrap().name);
/// ```
pub fn from_code(code: &str) -> Option<Subdivision> {
    SUBDIVISION_MAP.get(code).cloned()
}


pub const AF_BAL: Subdivision = Subdivision {
    name: "Balkh",
    code: "AF-BAL",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-BAL",
};


pub const AF_BAM: Subdivision = Subdivision {
    name: "Bāmyān",
    code: "AF-BAM",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-BAM",
};


pub const AF_BDG: Subdivision = Subdivision {
    name: "Bādghīs",
    code: "AF-BDG",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-BDG",
};


pub const AF_BDS: Subdivision = Subdivision {
    name: "Badakhshān",
    code: "AF-BDS",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-BDS",
};


pub const AF_BGL: Subdivision = Subdivision {
    name: "Baghlān",
    code: "AF-BGL",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-BGL",
};


pub const AF_DAY: Subdivision = Subdivision {
    name: "Dāykundī",
    code: "AF-DAY",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-DAY",
};


pub const AF_FRA: Subdivision = Subdivision {
    name: "Farāh",
    code: "AF-FRA",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-FRA",
};


pub const AF_FYB: Subdivision = Subdivision {
    name: "Fāryāb",
    code: "AF-FYB",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-FYB",
};


pub const AF_GHA: Subdivision = Subdivision {
    name: "Ghaznī",
    code: "AF-GHA",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-GHA",
};


pub const AF_GHO: Subdivision = Subdivision {
    name: "Ghōr",
    code: "AF-GHO",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-GHO",
};


pub const AF_HEL: Subdivision = Subdivision {
    name: "Helmand",
    code: "AF-HEL",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-HEL",
};


pub const AF_HER: Subdivision = Subdivision {
    name: "Herāt",
    code: "AF-HER",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-HER",
};


pub const AF_JOW: Subdivision = Subdivision {
    name: "Jowzjān",
    code: "AF-JOW",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-JOW",
};


pub const AF_KAB: Subdivision = Subdivision {
    name: "Kābul",
    code: "AF-KAB",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KAB",
};


pub const AF_KAN: Subdivision = Subdivision {
    name: "Kandahār",
    code: "AF-KAN",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KAN",
};


pub const AF_KAP: Subdivision = Subdivision {
    name: "Kāpīsā",
    code: "AF-KAP",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KAP",
};


pub const AF_KDZ: Subdivision = Subdivision {
    name: "Kunduz",
    code: "AF-KDZ",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KDZ",
};


pub const AF_KHO: Subdivision = Subdivision {
    name: "Khōst",
    code: "AF-KHO",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KHO",
};


pub const AF_KNR: Subdivision = Subdivision {
    name: "Kunar",
    code: "AF-KNR",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-KNR",
};


pub const AF_LAG: Subdivision = Subdivision {
    name: "Laghmān",
    code: "AF-LAG",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-LAG",
};


pub const AF_LOG: Subdivision = Subdivision {
    name: "Lōgar",
    code: "AF-LOG",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-LOG",
};


pub const AF_NAN: Subdivision = Subdivision {
    name: "Nangarhār",
    code: "AF-NAN",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-NAN",
};


pub const AF_NIM: Subdivision = Subdivision {
    name: "Nīmrōz",
    code: "AF-NIM",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-NIM",
};


pub const AF_NUR: Subdivision = Subdivision {
    name: "Nūristān",
    code: "AF-NUR",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-NUR",
};


pub const AF_PAN: Subdivision = Subdivision {
    name: "Panjshayr",
    code: "AF-PAN",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-PAN",
};


pub const AF_PAR: Subdivision = Subdivision {
    name: "Parwān",
    code: "AF-PAR",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-PAR",
};


pub const AF_PIA: Subdivision = Subdivision {
    name: "Paktiyā",
    code: "AF-PIA",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-PIA",
};


pub const AF_PKA: Subdivision = Subdivision {
    name: "Paktīkā",
    code: "AF-PKA",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-PKA",
};


pub const AF_SAM: Subdivision = Subdivision {
    name: "Samangān",
    code: "AF-SAM",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-SAM",
};


pub const AF_SAR: Subdivision = Subdivision {
    name: "Sar-e Pul",
    code: "AF-SAR",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-SAR",
};


pub const AF_TAK: Subdivision = Subdivision {
    name: "Takhār",
    code: "AF-TAK",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-TAK",
};


pub const AF_URU: Subdivision = Subdivision {
    name: "Uruzgān",
    code: "AF-URU",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-URU",
};


pub const AF_WAR: Subdivision = Subdivision {
    name: "Wardak",
    code: "AF-WAR",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-WAR",
};


pub const AF_ZAB: Subdivision = Subdivision {
    name: "Zābul",
    code: "AF-ZAB",
    subdivision_type: "Province",
    country_name: "Afghanistan",
    country_code: "AF",
    region_code: "AF-ZAB",
};


pub const AX_AX: Subdivision = Subdivision {
    name: "Åland Islands",
    code: "AX-AX",
    subdivision_type: "Country",
    country_name: "Åland Islands",
    country_code: "AX",
    region_code: "AX-AX",
};


pub const AL_01: Subdivision = Subdivision {
    name: "Berat",
    code: "AL-01",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-01",
};


pub const AL_02: Subdivision = Subdivision {
    name: "Durrës",
    code: "AL-02",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-02",
};


pub const AL_03: Subdivision = Subdivision {
    name: "Elbasan",
    code: "AL-03",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-03",
};


pub const AL_04: Subdivision = Subdivision {
    name: "Fier",
    code: "AL-04",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-04",
};


pub const AL_05: Subdivision = Subdivision {
    name: "Gjirokastër",
    code: "AL-05",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-05",
};


pub const AL_06: Subdivision = Subdivision {
    name: "Korçë",
    code: "AL-06",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-06",
};


pub const AL_07: Subdivision = Subdivision {
    name: "Kukës",
    code: "AL-07",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-07",
};


pub const AL_08: Subdivision = Subdivision {
    name: "Lezhë",
    code: "AL-08",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-08",
};


pub const AL_09: Subdivision = Subdivision {
    name: "Dibër",
    code: "AL-09",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-09",
};


pub const AL_10: Subdivision = Subdivision {
    name: "Shkodër",
    code: "AL-10",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-10",
};


pub const AL_11: Subdivision = Subdivision {
    name: "Tiranë",
    code: "AL-11",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-11",
};


pub const AL_12: Subdivision = Subdivision {
    name: "Vlorë",
    code: "AL-12",
    subdivision_type: "County",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-12",
};


pub const AL_BR: Subdivision = Subdivision {
    name: "Berat",
    code: "AL-BR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-BR",
};


pub const AL_BU: Subdivision = Subdivision {
    name: "Bulqizë",
    code: "AL-BU",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-BU",
};


pub const AL_DI: Subdivision = Subdivision {
    name: "Dibër",
    code: "AL-DI",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-DI",
};


pub const AL_DL: Subdivision = Subdivision {
    name: "Delvinë",
    code: "AL-DL",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-DL",
};


pub const AL_DR: Subdivision = Subdivision {
    name: "Durrës",
    code: "AL-DR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-DR",
};


pub const AL_DV: Subdivision = Subdivision {
    name: "Devoll",
    code: "AL-DV",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-DV",
};


pub const AL_EL: Subdivision = Subdivision {
    name: "Elbasan",
    code: "AL-EL",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-EL",
};


pub const AL_ER: Subdivision = Subdivision {
    name: "Kolonjë",
    code: "AL-ER",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-ER",
};


pub const AL_FR: Subdivision = Subdivision {
    name: "Fier",
    code: "AL-FR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-FR",
};


pub const AL_GJ: Subdivision = Subdivision {
    name: "Gjirokastër",
    code: "AL-GJ",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-GJ",
};


pub const AL_GR: Subdivision = Subdivision {
    name: "Gramsh",
    code: "AL-GR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-GR",
};


pub const AL_HA: Subdivision = Subdivision {
    name: "Has",
    code: "AL-HA",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-HA",
};


pub const AL_KA: Subdivision = Subdivision {
    name: "Kavajë",
    code: "AL-KA",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KA",
};


pub const AL_KB: Subdivision = Subdivision {
    name: "Kurbin",
    code: "AL-KB",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KB",
};


pub const AL_KC: Subdivision = Subdivision {
    name: "Kuçovë",
    code: "AL-KC",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KC",
};


pub const AL_KO: Subdivision = Subdivision {
    name: "Korçë",
    code: "AL-KO",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KO",
};


pub const AL_KR: Subdivision = Subdivision {
    name: "Krujë",
    code: "AL-KR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KR",
};


pub const AL_KU: Subdivision = Subdivision {
    name: "Kukës",
    code: "AL-KU",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-KU",
};


pub const AL_LB: Subdivision = Subdivision {
    name: "Librazhd",
    code: "AL-LB",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-LB",
};


pub const AL_LE: Subdivision = Subdivision {
    name: "Lezhë",
    code: "AL-LE",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-LE",
};


pub const AL_LU: Subdivision = Subdivision {
    name: "Lushnjë",
    code: "AL-LU",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-LU",
};


pub const AL_MK: Subdivision = Subdivision {
    name: "Mallakastër",
    code: "AL-MK",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-MK",
};


pub const AL_MM: Subdivision = Subdivision {
    name: "Malësi e Madhe",
    code: "AL-MM",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-MM",
};


pub const AL_MR: Subdivision = Subdivision {
    name: "Mirditë",
    code: "AL-MR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-MR",
};


pub const AL_MT: Subdivision = Subdivision {
    name: "Mat",
    code: "AL-MT",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-MT",
};


pub const AL_PG: Subdivision = Subdivision {
    name: "Pogradec",
    code: "AL-PG",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-PG",
};


pub const AL_PQ: Subdivision = Subdivision {
    name: "Peqin",
    code: "AL-PQ",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-PQ",
};


pub const AL_PR: Subdivision = Subdivision {
    name: "Përmet",
    code: "AL-PR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-PR",
};


pub const AL_PU: Subdivision = Subdivision {
    name: "Pukë",
    code: "AL-PU",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-PU",
};


pub const AL_SH: Subdivision = Subdivision {
    name: "Shkodër",
    code: "AL-SH",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-SH",
};


pub const AL_SK: Subdivision = Subdivision {
    name: "Skrapar",
    code: "AL-SK",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-SK",
};


pub const AL_SR: Subdivision = Subdivision {
    name: "Sarandë",
    code: "AL-SR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-SR",
};


pub const AL_TE: Subdivision = Subdivision {
    name: "Tepelenë",
    code: "AL-TE",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-TE",
};


pub const AL_TP: Subdivision = Subdivision {
    name: "Tropojë",
    code: "AL-TP",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-TP",
};


pub const AL_TR: Subdivision = Subdivision {
    name: "Tiranë",
    code: "AL-TR",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-TR",
};


pub const AL_VL: Subdivision = Subdivision {
    name: "Vlorë",
    code: "AL-VL",
    subdivision_type: "District",
    country_name: "Albania",
    country_code: "AL",
    region_code: "AL-VL",
};


pub const DZ_01: Subdivision = Subdivision {
    name: "Adrar",
    code: "DZ-01",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-01",
};


pub const DZ_02: Subdivision = Subdivision {
    name: "Chlef",
    code: "DZ-02",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-02",
};


pub const DZ_03: Subdivision = Subdivision {
    name: "Laghouat",
    code: "DZ-03",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-03",
};


pub const DZ_04: Subdivision = Subdivision {
    name: "Oum el Bouaghi",
    code: "DZ-04",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-04",
};


pub const DZ_05: Subdivision = Subdivision {
    name: "Batna",
    code: "DZ-05",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-05",
};


pub const DZ_06: Subdivision = Subdivision {
    name: "Béjaïa",
    code: "DZ-06",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-06",
};


pub const DZ_07: Subdivision = Subdivision {
    name: "Biskra",
    code: "DZ-07",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-07",
};


pub const DZ_08: Subdivision = Subdivision {
    name: "Béchar",
    code: "DZ-08",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-08",
};


pub const DZ_09: Subdivision = Subdivision {
    name: "Blida",
    code: "DZ-09",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-09",
};


pub const DZ_10: Subdivision = Subdivision {
    name: "Bouira",
    code: "DZ-10",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-10",
};


pub const DZ_11: Subdivision = Subdivision {
    name: "Tamanrasset",
    code: "DZ-11",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-11",
};


pub const DZ_12: Subdivision = Subdivision {
    name: "Tébessa",
    code: "DZ-12",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-12",
};


pub const DZ_13: Subdivision = Subdivision {
    name: "Tlemcen",
    code: "DZ-13",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-13",
};


pub const DZ_14: Subdivision = Subdivision {
    name: "Tiaret",
    code: "DZ-14",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-14",
};


pub const DZ_15: Subdivision = Subdivision {
    name: "Tizi Ouzou",
    code: "DZ-15",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-15",
};


pub const DZ_16: Subdivision = Subdivision {
    name: "Alger",
    code: "DZ-16",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-16",
};


pub const DZ_17: Subdivision = Subdivision {
    name: "Djelfa",
    code: "DZ-17",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-17",
};


pub const DZ_18: Subdivision = Subdivision {
    name: "Jijel",
    code: "DZ-18",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-18",
};


pub const DZ_19: Subdivision = Subdivision {
    name: "Sétif",
    code: "DZ-19",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-19",
};


pub const DZ_20: Subdivision = Subdivision {
    name: "Saïda",
    code: "DZ-20",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-20",
};


pub const DZ_21: Subdivision = Subdivision {
    name: "Skikda",
    code: "DZ-21",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-21",
};


pub const DZ_22: Subdivision = Subdivision {
    name: "Sidi Bel Abbès",
    code: "DZ-22",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-22",
};


pub const DZ_23: Subdivision = Subdivision {
    name: "Annaba",
    code: "DZ-23",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-23",
};


pub const DZ_24: Subdivision = Subdivision {
    name: "Guelma",
    code: "DZ-24",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-24",
};


pub const DZ_25: Subdivision = Subdivision {
    name: "Constantine",
    code: "DZ-25",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-25",
};


pub const DZ_26: Subdivision = Subdivision {
    name: "Médéa",
    code: "DZ-26",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-26",
};


pub const DZ_27: Subdivision = Subdivision {
    name: "Mostaganem",
    code: "DZ-27",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-27",
};


pub const DZ_28: Subdivision = Subdivision {
    name: "Msila",
    code: "DZ-28",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-28",
};


pub const DZ_29: Subdivision = Subdivision {
    name: "Mascara",
    code: "DZ-29",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-29",
};


pub const DZ_30: Subdivision = Subdivision {
    name: "Ouargla",
    code: "DZ-30",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-30",
};


pub const DZ_31: Subdivision = Subdivision {
    name: "Oran",
    code: "DZ-31",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-31",
};


pub const DZ_32: Subdivision = Subdivision {
    name: "El Bayadh",
    code: "DZ-32",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-32",
};


pub const DZ_33: Subdivision = Subdivision {
    name: "Illizi",
    code: "DZ-33",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-33",
};


pub const DZ_34: Subdivision = Subdivision {
    name: "Bordj Bou Arréridj",
    code: "DZ-34",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-34",
};


pub const DZ_35: Subdivision = Subdivision {
    name: "Boumerdès",
    code: "DZ-35",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-35",
};


pub const DZ_36: Subdivision = Subdivision {
    name: "El Tarf",
    code: "DZ-36",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-36",
};


pub const DZ_37: Subdivision = Subdivision {
    name: "Tindouf",
    code: "DZ-37",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-37",
};


pub const DZ_38: Subdivision = Subdivision {
    name: "Tissemsilt",
    code: "DZ-38",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-38",
};


pub const DZ_39: Subdivision = Subdivision {
    name: "El Oued",
    code: "DZ-39",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-39",
};


pub const DZ_40: Subdivision = Subdivision {
    name: "Khenchela",
    code: "DZ-40",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-40",
};


pub const DZ_41: Subdivision = Subdivision {
    name: "Souk Ahras",
    code: "DZ-41",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-41",
};


pub const DZ_42: Subdivision = Subdivision {
    name: "Tipaza",
    code: "DZ-42",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-42",
};


pub const DZ_43: Subdivision = Subdivision {
    name: "Mila",
    code: "DZ-43",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-43",
};


pub const DZ_44: Subdivision = Subdivision {
    name: "Aïn Defla",
    code: "DZ-44",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-44",
};


pub const DZ_45: Subdivision = Subdivision {
    name: "Naama",
    code: "DZ-45",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-45",
};


pub const DZ_46: Subdivision = Subdivision {
    name: "Aïn Témouchent",
    code: "DZ-46",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-46",
};


pub const DZ_47: Subdivision = Subdivision {
    name: "Ghardaïa",
    code: "DZ-47",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-47",
};


pub const DZ_48: Subdivision = Subdivision {
    name: "Relizane",
    code: "DZ-48",
    subdivision_type: "Province",
    country_name: "Algeria",
    country_code: "DZ",
    region_code: "DZ-48",
};


pub const AS_AS: Subdivision = Subdivision {
    name: "American Samoa",
    code: "AS-AS",
    subdivision_type: "Country",
    country_name: "American Samoa",
    country_code: "AS",
    region_code: "AS-AS",
};


pub const AD_02: Subdivision = Subdivision {
    name: "Canillo",
    code: "AD-02",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-02",
};


pub const AD_03: Subdivision = Subdivision {
    name: "Encamp",
    code: "AD-03",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-03",
};


pub const AD_04: Subdivision = Subdivision {
    name: "La Massana",
    code: "AD-04",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-04",
};


pub const AD_05: Subdivision = Subdivision {
    name: "Ordino",
    code: "AD-05",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-05",
};


pub const AD_06: Subdivision = Subdivision {
    name: "Sant Julià de Lòria",
    code: "AD-06",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-06",
};


pub const AD_07: Subdivision = Subdivision {
    name: "Andorra la Vella",
    code: "AD-07",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-07",
};


pub const AD_08: Subdivision = Subdivision {
    name: "Escaldes-Engordany",
    code: "AD-08",
    subdivision_type: "Parish",
    country_name: "Andorra",
    country_code: "AD",
    region_code: "AD-08",
};


pub const AO_BGO: Subdivision = Subdivision {
    name: "Bengo",
    code: "AO-BGO",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-BGO",
};


pub const AO_BGU: Subdivision = Subdivision {
    name: "Benguela",
    code: "AO-BGU",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-BGU",
};


pub const AO_BIE: Subdivision = Subdivision {
    name: "Bié",
    code: "AO-BIE",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-BIE",
};


pub const AO_CAB: Subdivision = Subdivision {
    name: "Cabinda",
    code: "AO-CAB",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-CAB",
};


pub const AO_CCU: Subdivision = Subdivision {
    name: "Kuando Kubango",
    code: "AO-CCU",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-CCU",
};


pub const AO_CNN: Subdivision = Subdivision {
    name: "Cunene",
    code: "AO-CNN",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-CNN",
};


pub const AO_CNO: Subdivision = Subdivision {
    name: "Kwanza Norte",
    code: "AO-CNO",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-CNO",
};


pub const AO_CUS: Subdivision = Subdivision {
    name: "Kwanza Sul",
    code: "AO-CUS",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-CUS",
};


pub const AO_HUA: Subdivision = Subdivision {
    name: "Huambo",
    code: "AO-HUA",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-HUA",
};


pub const AO_HUI: Subdivision = Subdivision {
    name: "Huíla",
    code: "AO-HUI",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-HUI",
};


pub const AO_LNO: Subdivision = Subdivision {
    name: "Lunda Norte",
    code: "AO-LNO",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-LNO",
};


pub const AO_LSU: Subdivision = Subdivision {
    name: "Lunda Sul",
    code: "AO-LSU",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-LSU",
};


pub const AO_LUA: Subdivision = Subdivision {
    name: "Luanda",
    code: "AO-LUA",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-LUA",
};


pub const AO_MAL: Subdivision = Subdivision {
    name: "Malange",
    code: "AO-MAL",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-MAL",
};


pub const AO_MOX: Subdivision = Subdivision {
    name: "Moxico",
    code: "AO-MOX",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-MOX",
};


pub const AO_NAM: Subdivision = Subdivision {
    name: "Namibe",
    code: "AO-NAM",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-NAM",
};


pub const AO_UIG: Subdivision = Subdivision {
    name: "Uíge",
    code: "AO-UIG",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-UIG",
};


pub const AO_ZAI: Subdivision = Subdivision {
    name: "Zaire",
    code: "AO-ZAI",
    subdivision_type: "Province",
    country_name: "Angola",
    country_code: "AO",
    region_code: "AO-ZAI",
};


pub const AI_AI: Subdivision = Subdivision {
    name: "Anguilla",
    code: "AI-AI",
    subdivision_type: "Country",
    country_name: "Anguilla",
    country_code: "AI",
    region_code: "AI-AI",
};


pub const AQ_AQ: Subdivision = Subdivision {
    name: "Antarctica",
    code: "AQ-AQ",
    subdivision_type: "Country",
    country_name: "Antarctica",
    country_code: "AQ",
    region_code: "AQ-AQ",
};


pub const AG_03: Subdivision = Subdivision {
    name: "Saint George",
    code: "AG-03",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-03",
};


pub const AG_04: Subdivision = Subdivision {
    name: "Saint John",
    code: "AG-04",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-04",
};


pub const AG_05: Subdivision = Subdivision {
    name: "Saint Mary",
    code: "AG-05",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-05",
};


pub const AG_06: Subdivision = Subdivision {
    name: "Saint Paul",
    code: "AG-06",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-06",
};


pub const AG_07: Subdivision = Subdivision {
    name: "Saint Peter",
    code: "AG-07",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-07",
};


pub const AG_08: Subdivision = Subdivision {
    name: "Saint Philip",
    code: "AG-08",
    subdivision_type: "Parish",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-08",
};


pub const AG_10: Subdivision = Subdivision {
    name: "Barbuda",
    code: "AG-10",
    subdivision_type: "Dependency",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-10",
};


pub const AG_11: Subdivision = Subdivision {
    name: "Redonda",
    code: "AG-11",
    subdivision_type: "Dependency",
    country_name: "Antigua and Barbuda",
    country_code: "AG",
    region_code: "AG-11",
};


pub const AR_A: Subdivision = Subdivision {
    name: "Salta",
    code: "AR-A",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-A",
};


pub const AR_B: Subdivision = Subdivision {
    name: "Buenos Aires",
    code: "AR-B",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-B",
};


pub const AR_C: Subdivision = Subdivision {
    name: "Ciudad Autónoma de Buenos Aires",
    code: "AR-C",
    subdivision_type: "City",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-C",
};


pub const AR_D: Subdivision = Subdivision {
    name: "San Luis",
    code: "AR-D",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-D",
};


pub const AR_E: Subdivision = Subdivision {
    name: "Entre Ríos",
    code: "AR-E",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-E",
};


pub const AR_F: Subdivision = Subdivision {
    name: "La Rioja",
    code: "AR-F",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-F",
};


pub const AR_G: Subdivision = Subdivision {
    name: "Santiago del Estero",
    code: "AR-G",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-G",
};


pub const AR_H: Subdivision = Subdivision {
    name: "Chaco",
    code: "AR-H",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-H",
};


pub const AR_J: Subdivision = Subdivision {
    name: "San Juan",
    code: "AR-J",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-J",
};


pub const AR_K: Subdivision = Subdivision {
    name: "Catamarca",
    code: "AR-K",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-K",
};


pub const AR_L: Subdivision = Subdivision {
    name: "La Pampa",
    code: "AR-L",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-L",
};


pub const AR_M: Subdivision = Subdivision {
    name: "Mendoza",
    code: "AR-M",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-M",
};


pub const AR_N: Subdivision = Subdivision {
    name: "Misiones",
    code: "AR-N",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-N",
};


pub const AR_P: Subdivision = Subdivision {
    name: "Formosa",
    code: "AR-P",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-P",
};


pub const AR_Q: Subdivision = Subdivision {
    name: "Neuquén",
    code: "AR-Q",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-Q",
};


pub const AR_R: Subdivision = Subdivision {
    name: "Río Negro",
    code: "AR-R",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-R",
};


pub const AR_S: Subdivision = Subdivision {
    name: "Santa Fe",
    code: "AR-S",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-S",
};


pub const AR_T: Subdivision = Subdivision {
    name: "Tucumán",
    code: "AR-T",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-T",
};


pub const AR_U: Subdivision = Subdivision {
    name: "Chubut",
    code: "AR-U",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-U",
};


pub const AR_V: Subdivision = Subdivision {
    name: "Tierra del Fuego",
    code: "AR-V",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-V",
};


pub const AR_W: Subdivision = Subdivision {
    name: "Corrientes",
    code: "AR-W",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-W",
};


pub const AR_X: Subdivision = Subdivision {
    name: "Córdoba",
    code: "AR-X",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-X",
};


pub const AR_Y: Subdivision = Subdivision {
    name: "Jujuy",
    code: "AR-Y",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-Y",
};


pub const AR_Z: Subdivision = Subdivision {
    name: "Santa Cruz",
    code: "AR-Z",
    subdivision_type: "Province",
    country_name: "Argentina",
    country_code: "AR",
    region_code: "AR-Z",
};


pub const AM_AG: Subdivision = Subdivision {
    name: "Aragac̣otn",
    code: "AM-AG",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-AG",
};


pub const AM_AR: Subdivision = Subdivision {
    name: "Ararat",
    code: "AM-AR",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-AR",
};


pub const AM_AV: Subdivision = Subdivision {
    name: "Armavir",
    code: "AM-AV",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-AV",
};


pub const AM_ER: Subdivision = Subdivision {
    name: "Erevan",
    code: "AM-ER",
    subdivision_type: "City",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-ER",
};


pub const AM_GR: Subdivision = Subdivision {
    name: "Geġark'unik'",
    code: "AM-GR",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-GR",
};


pub const AM_KT: Subdivision = Subdivision {
    name: "Kotayk'",
    code: "AM-KT",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-KT",
};


pub const AM_LO: Subdivision = Subdivision {
    name: "Loṙi",
    code: "AM-LO",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-LO",
};


pub const AM_SH: Subdivision = Subdivision {
    name: "Širak",
    code: "AM-SH",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-SH",
};


pub const AM_SU: Subdivision = Subdivision {
    name: "Syunik'",
    code: "AM-SU",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-SU",
};


pub const AM_TV: Subdivision = Subdivision {
    name: "Tavuš",
    code: "AM-TV",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-TV",
};


pub const AM_VD: Subdivision = Subdivision {
    name: "Vayoć Jor",
    code: "AM-VD",
    subdivision_type: "Region",
    country_name: "Armenia",
    country_code: "AM",
    region_code: "AM-VD",
};


pub const AW_AW: Subdivision = Subdivision {
    name: "Aruba",
    code: "AW-AW",
    subdivision_type: "Country",
    country_name: "Aruba",
    country_code: "AW",
    region_code: "AW-AW",
};


pub const AU_ACT: Subdivision = Subdivision {
    name: "Australian Capital Territory",
    code: "AU-ACT",
    subdivision_type: "Territory",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-ACT",
};


pub const AU_NSW: Subdivision = Subdivision {
    name: "New South Wales",
    code: "AU-NSW",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-NSW",
};


pub const AU_NT: Subdivision = Subdivision {
    name: "Northern Territory",
    code: "AU-NT",
    subdivision_type: "Territory",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-NT",
};


pub const AU_QLD: Subdivision = Subdivision {
    name: "Queensland",
    code: "AU-QLD",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-QLD",
};


pub const AU_SA: Subdivision = Subdivision {
    name: "South Australia",
    code: "AU-SA",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-SA",
};


pub const AU_TAS: Subdivision = Subdivision {
    name: "Tasmania",
    code: "AU-TAS",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-TAS",
};


pub const AU_VIC: Subdivision = Subdivision {
    name: "Victoria",
    code: "AU-VIC",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-VIC",
};


pub const AU_WA: Subdivision = Subdivision {
    name: "Western Australia",
    code: "AU-WA",
    subdivision_type: "State",
    country_name: "Australia",
    country_code: "AU",
    region_code: "AU-WA",
};


pub const AT_1: Subdivision = Subdivision {
    name: "Burgenland",
    code: "AT-1",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-1",
};


pub const AT_2: Subdivision = Subdivision {
    name: "Kärnten",
    code: "AT-2",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-2",
};


pub const AT_3: Subdivision = Subdivision {
    name: "Niederösterreich",
    code: "AT-3",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-3",
};


pub const AT_4: Subdivision = Subdivision {
    name: "Oberösterreich",
    code: "AT-4",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-4",
};


pub const AT_5: Subdivision = Subdivision {
    name: "Salzburg",
    code: "AT-5",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-5",
};


pub const AT_6: Subdivision = Subdivision {
    name: "Steiermark",
    code: "AT-6",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-6",
};


pub const AT_7: Subdivision = Subdivision {
    name: "Tirol",
    code: "AT-7",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-7",
};


pub const AT_8: Subdivision = Subdivision {
    name: "Vorarlberg",
    code: "AT-8",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-8",
};


pub const AT_9: Subdivision = Subdivision {
    name: "Wien",
    code: "AT-9",
    subdivision_type: "Federal länder",
    country_name: "Austria",
    country_code: "AT",
    region_code: "AT-9",
};


pub const AZ_ABS: Subdivision = Subdivision {
    name: "Abşeron",
    code: "AZ-ABS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ABS",
};


pub const AZ_AGA: Subdivision = Subdivision {
    name: "Ağstafa",
    code: "AZ-AGA",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AGA",
};


pub const AZ_AGC: Subdivision = Subdivision {
    name: "Ağcabәdi",
    code: "AZ-AGC",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AGC",
};


pub const AZ_AGM: Subdivision = Subdivision {
    name: "Ağdam",
    code: "AZ-AGM",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AGM",
};


pub const AZ_AGS: Subdivision = Subdivision {
    name: "Ağdaş",
    code: "AZ-AGS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AGS",
};


pub const AZ_AGU: Subdivision = Subdivision {
    name: "Ağsu",
    code: "AZ-AGU",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AGU",
};


pub const AZ_AST: Subdivision = Subdivision {
    name: "Astara",
    code: "AZ-AST",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-AST",
};


pub const AZ_BA: Subdivision = Subdivision {
    name: "Bakı",
    code: "AZ-BA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BA",
};


pub const AZ_BAB: Subdivision = Subdivision {
    name: "Babək",
    code: "AZ-BAB",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BAB",
};


pub const AZ_BAL: Subdivision = Subdivision {
    name: "Balakən",
    code: "AZ-BAL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BAL",
};


pub const AZ_BAR: Subdivision = Subdivision {
    name: "Bərdə",
    code: "AZ-BAR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BAR",
};


pub const AZ_BEY: Subdivision = Subdivision {
    name: "Beyləqan",
    code: "AZ-BEY",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BEY",
};


pub const AZ_BIL: Subdivision = Subdivision {
    name: "Biləsuvar",
    code: "AZ-BIL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-BIL",
};


pub const AZ_CAB: Subdivision = Subdivision {
    name: "Cəbrayıl",
    code: "AZ-CAB",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-CAB",
};


pub const AZ_CAL: Subdivision = Subdivision {
    name: "Cəlilabad",
    code: "AZ-CAL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-CAL",
};


pub const AZ_CUL: Subdivision = Subdivision {
    name: "Culfa",
    code: "AZ-CUL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-CUL",
};


pub const AZ_DAS: Subdivision = Subdivision {
    name: "Daşkəsən",
    code: "AZ-DAS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-DAS",
};


pub const AZ_FUZ: Subdivision = Subdivision {
    name: "Füzuli",
    code: "AZ-FUZ",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-FUZ",
};


pub const AZ_GAD: Subdivision = Subdivision {
    name: "Gədəbəy",
    code: "AZ-GAD",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-GAD",
};


pub const AZ_GA: Subdivision = Subdivision {
    name: "Gәncә",
    code: "AZ-GA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-GA",
};


pub const AZ_GOR: Subdivision = Subdivision {
    name: "Goranboy",
    code: "AZ-GOR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-GOR",
};


pub const AZ_GOY: Subdivision = Subdivision {
    name: "Göyçay",
    code: "AZ-GOY",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-GOY",
};


pub const AZ_GYG: Subdivision = Subdivision {
    name: "Göygöl",
    code: "AZ-GYG",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-GYG",
};


pub const AZ_HAC: Subdivision = Subdivision {
    name: "Hacıqabul",
    code: "AZ-HAC",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-HAC",
};


pub const AZ_IMI: Subdivision = Subdivision {
    name: "İmişli",
    code: "AZ-IMI",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-IMI",
};


pub const AZ_ISM: Subdivision = Subdivision {
    name: "İsmayıllı",
    code: "AZ-ISM",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ISM",
};


pub const AZ_KAL: Subdivision = Subdivision {
    name: "Kəlbəcər",
    code: "AZ-KAL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-KAL",
};


pub const AZ_KAN: Subdivision = Subdivision {
    name: "Kǝngǝrli",
    code: "AZ-KAN",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-KAN",
};


pub const AZ_KUR: Subdivision = Subdivision {
    name: "Kürdəmir",
    code: "AZ-KUR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-KUR",
};


pub const AZ_LAC: Subdivision = Subdivision {
    name: "Laçın",
    code: "AZ-LAC",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-LAC",
};


pub const AZ_LA: Subdivision = Subdivision {
    name: "Lәnkәran",
    code: "AZ-LA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-LA",
};


pub const AZ_LAN: Subdivision = Subdivision {
    name: "Lənkəran",
    code: "AZ-LAN",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-LAN",
};


pub const AZ_LER: Subdivision = Subdivision {
    name: "Lerik",
    code: "AZ-LER",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-LER",
};


pub const AZ_MAS: Subdivision = Subdivision {
    name: "Masallı",
    code: "AZ-MAS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-MAS",
};


pub const AZ_MI: Subdivision = Subdivision {
    name: "Mingәçevir",
    code: "AZ-MI",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-MI",
};


pub const AZ_NA: Subdivision = Subdivision {
    name: "Naftalan",
    code: "AZ-NA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-NA",
};


pub const AZ_NEF: Subdivision = Subdivision {
    name: "Neftçala",
    code: "AZ-NEF",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-NEF",
};


pub const AZ_NV: Subdivision = Subdivision {
    name: "Naxçıvan",
    code: "AZ-NV",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-NV",
};


pub const AZ_NX: Subdivision = Subdivision {
    name: "Naxçıvan",
    code: "AZ-NX",
    subdivision_type: "Autonomous republic",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-NX",
};


pub const AZ_OGU: Subdivision = Subdivision {
    name: "Oğuz",
    code: "AZ-OGU",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-OGU",
};


pub const AZ_ORD: Subdivision = Subdivision {
    name: "Ordubad",
    code: "AZ-ORD",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ORD",
};


pub const AZ_QAB: Subdivision = Subdivision {
    name: "Qәbәlә",
    code: "AZ-QAB",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QAB",
};


pub const AZ_QAX: Subdivision = Subdivision {
    name: "Qax",
    code: "AZ-QAX",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QAX",
};


pub const AZ_QAZ: Subdivision = Subdivision {
    name: "Qazax",
    code: "AZ-QAZ",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QAZ",
};


pub const AZ_QBA: Subdivision = Subdivision {
    name: "Quba",
    code: "AZ-QBA",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QBA",
};


pub const AZ_QBI: Subdivision = Subdivision {
    name: "Qubadlı",
    code: "AZ-QBI",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QBI",
};


pub const AZ_QOB: Subdivision = Subdivision {
    name: "Qobustan",
    code: "AZ-QOB",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QOB",
};


pub const AZ_QUS: Subdivision = Subdivision {
    name: "Qusar",
    code: "AZ-QUS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-QUS",
};


pub const AZ_SAB: Subdivision = Subdivision {
    name: "Sabirabad",
    code: "AZ-SAB",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAB",
};


pub const AZ_SAD: Subdivision = Subdivision {
    name: "Sәdәrәk",
    code: "AZ-SAD",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAD",
};


pub const AZ_SAH: Subdivision = Subdivision {
    name: "Şahbuz",
    code: "AZ-SAH",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAH",
};


pub const AZ_SAK: Subdivision = Subdivision {
    name: "Şәki",
    code: "AZ-SAK",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAK",
};


pub const AZ_SAL: Subdivision = Subdivision {
    name: "Salyan",
    code: "AZ-SAL",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAL",
};


pub const AZ_SAR: Subdivision = Subdivision {
    name: "Şәrur",
    code: "AZ-SAR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAR",
};


pub const AZ_SA: Subdivision = Subdivision {
    name: "Şәki",
    code: "AZ-SA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SA",
};


pub const AZ_SAT: Subdivision = Subdivision {
    name: "Saatlı",
    code: "AZ-SAT",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SAT",
};


pub const AZ_SBN: Subdivision = Subdivision {
    name: "Şabran",
    code: "AZ-SBN",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SBN",
};


pub const AZ_SIY: Subdivision = Subdivision {
    name: "Siyәzәn",
    code: "AZ-SIY",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SIY",
};


pub const AZ_SKR: Subdivision = Subdivision {
    name: "Şәmkir",
    code: "AZ-SKR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SKR",
};


pub const AZ_SMI: Subdivision = Subdivision {
    name: "Şamaxı",
    code: "AZ-SMI",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SMI",
};


pub const AZ_SM: Subdivision = Subdivision {
    name: "Sumqayıt",
    code: "AZ-SM",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SM",
};


pub const AZ_SMX: Subdivision = Subdivision {
    name: "Samux",
    code: "AZ-SMX",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SMX",
};


pub const AZ_SR: Subdivision = Subdivision {
    name: "Şirvan",
    code: "AZ-SR",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SR",
};


pub const AZ_SUS: Subdivision = Subdivision {
    name: "Şuşa",
    code: "AZ-SUS",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-SUS",
};


pub const AZ_TAR: Subdivision = Subdivision {
    name: "Tәrtәr",
    code: "AZ-TAR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-TAR",
};


pub const AZ_TOV: Subdivision = Subdivision {
    name: "Tovuz",
    code: "AZ-TOV",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-TOV",
};


pub const AZ_UCA: Subdivision = Subdivision {
    name: "Ucar",
    code: "AZ-UCA",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-UCA",
};


pub const AZ_XAC: Subdivision = Subdivision {
    name: "Xaçmaz",
    code: "AZ-XAC",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-XAC",
};


pub const AZ_XA: Subdivision = Subdivision {
    name: "Xankәndi",
    code: "AZ-XA",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-XA",
};


pub const AZ_XCI: Subdivision = Subdivision {
    name: "Xocalı",
    code: "AZ-XCI",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-XCI",
};


pub const AZ_XIZ: Subdivision = Subdivision {
    name: "Xızı",
    code: "AZ-XIZ",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-XIZ",
};


pub const AZ_XVD: Subdivision = Subdivision {
    name: "Xocavәnd",
    code: "AZ-XVD",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-XVD",
};


pub const AZ_YAR: Subdivision = Subdivision {
    name: "Yardımlı",
    code: "AZ-YAR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-YAR",
};


pub const AZ_YEV: Subdivision = Subdivision {
    name: "Yevlax",
    code: "AZ-YEV",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-YEV",
};


pub const AZ_YE: Subdivision = Subdivision {
    name: "Yevlax",
    code: "AZ-YE",
    subdivision_type: "Municipality",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-YE",
};


pub const AZ_ZAN: Subdivision = Subdivision {
    name: "Zәngilan",
    code: "AZ-ZAN",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ZAN",
};


pub const AZ_ZAQ: Subdivision = Subdivision {
    name: "Zaqatala",
    code: "AZ-ZAQ",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ZAQ",
};


pub const AZ_ZAR: Subdivision = Subdivision {
    name: "Zәrdab",
    code: "AZ-ZAR",
    subdivision_type: "Rayon",
    country_name: "Azerbaijan",
    country_code: "AZ",
    region_code: "AZ-ZAR",
};


pub const BS_AK: Subdivision = Subdivision {
    name: "Acklins",
    code: "BS-AK",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-AK",
};


pub const BS_BI: Subdivision = Subdivision {
    name: "Bimini",
    code: "BS-BI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-BI",
};


pub const BS_BP: Subdivision = Subdivision {
    name: "Black Point",
    code: "BS-BP",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-BP",
};


pub const BS_BY: Subdivision = Subdivision {
    name: "Berry Islands",
    code: "BS-BY",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-BY",
};


pub const BS_CE: Subdivision = Subdivision {
    name: "Central Eleuthera",
    code: "BS-CE",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-CE",
};


pub const BS_CI: Subdivision = Subdivision {
    name: "Cat Island",
    code: "BS-CI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-CI",
};


pub const BS_CK: Subdivision = Subdivision {
    name: "Crooked Island and Long Cay",
    code: "BS-CK",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-CK",
};


pub const BS_CO: Subdivision = Subdivision {
    name: "Central Abaco",
    code: "BS-CO",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-CO",
};


pub const BS_CS: Subdivision = Subdivision {
    name: "Central Andros",
    code: "BS-CS",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-CS",
};


pub const BS_EG: Subdivision = Subdivision {
    name: "East Grand Bahama",
    code: "BS-EG",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-EG",
};


pub const BS_EX: Subdivision = Subdivision {
    name: "Exuma",
    code: "BS-EX",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-EX",
};


pub const BS_FP: Subdivision = Subdivision {
    name: "City of Freeport",
    code: "BS-FP",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-FP",
};


pub const BS_GC: Subdivision = Subdivision {
    name: "Grand Cay",
    code: "BS-GC",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-GC",
};


pub const BS_HI: Subdivision = Subdivision {
    name: "Harbour Island",
    code: "BS-HI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-HI",
};


pub const BS_HT: Subdivision = Subdivision {
    name: "Hope Town",
    code: "BS-HT",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-HT",
};


pub const BS_IN: Subdivision = Subdivision {
    name: "Inagua",
    code: "BS-IN",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-IN",
};


pub const BS_LI: Subdivision = Subdivision {
    name: "Long Island",
    code: "BS-LI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-LI",
};


pub const BS_MC: Subdivision = Subdivision {
    name: "Mangrove Cay",
    code: "BS-MC",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-MC",
};


pub const BS_MG: Subdivision = Subdivision {
    name: "Mayaguana",
    code: "BS-MG",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-MG",
};


pub const BS_MI: Subdivision = Subdivision {
    name: "Moore's Island",
    code: "BS-MI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-MI",
};


pub const BS_NE: Subdivision = Subdivision {
    name: "North Eleuthera",
    code: "BS-NE",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-NE",
};


pub const BS_NO: Subdivision = Subdivision {
    name: "North Abaco",
    code: "BS-NO",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-NO",
};


pub const BS_NS: Subdivision = Subdivision {
    name: "North Andros",
    code: "BS-NS",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-NS",
};


pub const BS_RC: Subdivision = Subdivision {
    name: "Rum Cay",
    code: "BS-RC",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-RC",
};


pub const BS_RI: Subdivision = Subdivision {
    name: "Ragged Island",
    code: "BS-RI",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-RI",
};


pub const BS_SA: Subdivision = Subdivision {
    name: "South Andros",
    code: "BS-SA",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-SA",
};


pub const BS_SE: Subdivision = Subdivision {
    name: "South Eleuthera",
    code: "BS-SE",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-SE",
};


pub const BS_SO: Subdivision = Subdivision {
    name: "South Abaco",
    code: "BS-SO",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-SO",
};


pub const BS_SS: Subdivision = Subdivision {
    name: "San Salvador",
    code: "BS-SS",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-SS",
};


pub const BS_SW: Subdivision = Subdivision {
    name: "Spanish Wells",
    code: "BS-SW",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-SW",
};


pub const BS_WG: Subdivision = Subdivision {
    name: "West Grand Bahama",
    code: "BS-WG",
    subdivision_type: "District",
    country_name: "Bahamas",
    country_code: "BS",
    region_code: "BS-WG",
};


pub const BH_13: Subdivision = Subdivision {
    name: "Al Manāmah",
    code: "BH-13",
    subdivision_type: "Governorate",
    country_name: "Bahrain",
    country_code: "BH",
    region_code: "BH-13",
};


pub const BH_14: Subdivision = Subdivision {
    name: "Al Janūbīyah",
    code: "BH-14",
    subdivision_type: "Governorate",
    country_name: "Bahrain",
    country_code: "BH",
    region_code: "BH-14",
};


pub const BH_15: Subdivision = Subdivision {
    name: "Al Muḩarraq",
    code: "BH-15",
    subdivision_type: "Governorate",
    country_name: "Bahrain",
    country_code: "BH",
    region_code: "BH-15",
};


pub const BH_16: Subdivision = Subdivision {
    name: "Al Wusţá",
    code: "BH-16",
    subdivision_type: "Governorate",
    country_name: "Bahrain",
    country_code: "BH",
    region_code: "BH-16",
};


pub const BH_17: Subdivision = Subdivision {
    name: "Ash Shamālīyah",
    code: "BH-17",
    subdivision_type: "Governorate",
    country_name: "Bahrain",
    country_code: "BH",
    region_code: "BH-17",
};


pub const BD_01: Subdivision = Subdivision {
    name: "Bandarban",
    code: "BD-01",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-01",
};


pub const BD_02: Subdivision = Subdivision {
    name: "Barguna",
    code: "BD-02",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-02",
};


pub const BD_03: Subdivision = Subdivision {
    name: "Bogra",
    code: "BD-03",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-03",
};


pub const BD_04: Subdivision = Subdivision {
    name: "Brahmanbaria",
    code: "BD-04",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-04",
};


pub const BD_05: Subdivision = Subdivision {
    name: "Bagerhat",
    code: "BD-05",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-05",
};


pub const BD_06: Subdivision = Subdivision {
    name: "Barisal",
    code: "BD-06",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-06",
};


pub const BD_07: Subdivision = Subdivision {
    name: "Bhola",
    code: "BD-07",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-07",
};


pub const BD_08: Subdivision = Subdivision {
    name: "Comilla",
    code: "BD-08",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-08",
};


pub const BD_09: Subdivision = Subdivision {
    name: "Chandpur",
    code: "BD-09",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-09",
};


pub const BD_10: Subdivision = Subdivision {
    name: "Chittagong",
    code: "BD-10",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-10",
};


pub const BD_11: Subdivision = Subdivision {
    name: "Cox's Bazar",
    code: "BD-11",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-11",
};


pub const BD_12: Subdivision = Subdivision {
    name: "Chuadanga",
    code: "BD-12",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-12",
};


pub const BD_13: Subdivision = Subdivision {
    name: "Dhaka",
    code: "BD-13",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-13",
};


pub const BD_14: Subdivision = Subdivision {
    name: "Dinajpur",
    code: "BD-14",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-14",
};


pub const BD_15: Subdivision = Subdivision {
    name: "Faridpur",
    code: "BD-15",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-15",
};


pub const BD_16: Subdivision = Subdivision {
    name: "Feni",
    code: "BD-16",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-16",
};


pub const BD_17: Subdivision = Subdivision {
    name: "Gopalganj",
    code: "BD-17",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-17",
};


pub const BD_18: Subdivision = Subdivision {
    name: "Gazipur",
    code: "BD-18",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-18",
};


pub const BD_19: Subdivision = Subdivision {
    name: "Gaibandha",
    code: "BD-19",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-19",
};


pub const BD_20: Subdivision = Subdivision {
    name: "Habiganj",
    code: "BD-20",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-20",
};


pub const BD_21: Subdivision = Subdivision {
    name: "Jamalpur",
    code: "BD-21",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-21",
};


pub const BD_22: Subdivision = Subdivision {
    name: "Jessore",
    code: "BD-22",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-22",
};


pub const BD_23: Subdivision = Subdivision {
    name: "Jhenaidah",
    code: "BD-23",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-23",
};


pub const BD_24: Subdivision = Subdivision {
    name: "Jaipurhat",
    code: "BD-24",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-24",
};


pub const BD_25: Subdivision = Subdivision {
    name: "Jhalakati",
    code: "BD-25",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-25",
};


pub const BD_26: Subdivision = Subdivision {
    name: "Kishoreganj",
    code: "BD-26",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-26",
};


pub const BD_27: Subdivision = Subdivision {
    name: "Khulna",
    code: "BD-27",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-27",
};


pub const BD_28: Subdivision = Subdivision {
    name: "Kurigram",
    code: "BD-28",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-28",
};


pub const BD_29: Subdivision = Subdivision {
    name: "Khagrachari",
    code: "BD-29",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-29",
};


pub const BD_30: Subdivision = Subdivision {
    name: "Kushtia",
    code: "BD-30",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-30",
};


pub const BD_31: Subdivision = Subdivision {
    name: "Lakshmipur",
    code: "BD-31",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-31",
};


pub const BD_32: Subdivision = Subdivision {
    name: "Lalmonirhat",
    code: "BD-32",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-32",
};


pub const BD_33: Subdivision = Subdivision {
    name: "Manikganj",
    code: "BD-33",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-33",
};


pub const BD_34: Subdivision = Subdivision {
    name: "Mymensingh",
    code: "BD-34",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-34",
};


pub const BD_35: Subdivision = Subdivision {
    name: "Munshiganj",
    code: "BD-35",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-35",
};


pub const BD_36: Subdivision = Subdivision {
    name: "Madaripur",
    code: "BD-36",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-36",
};


pub const BD_37: Subdivision = Subdivision {
    name: "Magura",
    code: "BD-37",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-37",
};


pub const BD_38: Subdivision = Subdivision {
    name: "Moulvibazar",
    code: "BD-38",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-38",
};


pub const BD_39: Subdivision = Subdivision {
    name: "Meherpur",
    code: "BD-39",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-39",
};


pub const BD_40: Subdivision = Subdivision {
    name: "Narayanganj",
    code: "BD-40",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-40",
};


pub const BD_41: Subdivision = Subdivision {
    name: "Netrakona",
    code: "BD-41",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-41",
};


pub const BD_42: Subdivision = Subdivision {
    name: "Narsingdi",
    code: "BD-42",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-42",
};


pub const BD_43: Subdivision = Subdivision {
    name: "Narail",
    code: "BD-43",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-43",
};


pub const BD_44: Subdivision = Subdivision {
    name: "Natore",
    code: "BD-44",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-44",
};


pub const BD_45: Subdivision = Subdivision {
    name: "Nawabganj",
    code: "BD-45",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-45",
};


pub const BD_46: Subdivision = Subdivision {
    name: "Nilphamari",
    code: "BD-46",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-46",
};


pub const BD_47: Subdivision = Subdivision {
    name: "Noakhali",
    code: "BD-47",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-47",
};


pub const BD_48: Subdivision = Subdivision {
    name: "Naogaon",
    code: "BD-48",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-48",
};


pub const BD_49: Subdivision = Subdivision {
    name: "Pabna",
    code: "BD-49",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-49",
};


pub const BD_50: Subdivision = Subdivision {
    name: "Pirojpur",
    code: "BD-50",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-50",
};


pub const BD_51: Subdivision = Subdivision {
    name: "Patuakhali",
    code: "BD-51",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-51",
};


pub const BD_52: Subdivision = Subdivision {
    name: "Panchagarh",
    code: "BD-52",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-52",
};


pub const BD_53: Subdivision = Subdivision {
    name: "Rajbari",
    code: "BD-53",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-53",
};


pub const BD_54: Subdivision = Subdivision {
    name: "Rajshahi",
    code: "BD-54",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-54",
};


pub const BD_55: Subdivision = Subdivision {
    name: "Rangpur",
    code: "BD-55",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-55",
};


pub const BD_56: Subdivision = Subdivision {
    name: "Rangamati",
    code: "BD-56",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-56",
};


pub const BD_57: Subdivision = Subdivision {
    name: "Sherpur",
    code: "BD-57",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-57",
};


pub const BD_58: Subdivision = Subdivision {
    name: "Satkhira",
    code: "BD-58",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-58",
};


pub const BD_59: Subdivision = Subdivision {
    name: "Sirajganj",
    code: "BD-59",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-59",
};


pub const BD_60: Subdivision = Subdivision {
    name: "Sylhet",
    code: "BD-60",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-60",
};


pub const BD_61: Subdivision = Subdivision {
    name: "Sunamganj",
    code: "BD-61",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-61",
};


pub const BD_62: Subdivision = Subdivision {
    name: "Shariatpur",
    code: "BD-62",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-62",
};


pub const BD_63: Subdivision = Subdivision {
    name: "Tangail",
    code: "BD-63",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-63",
};


pub const BD_64: Subdivision = Subdivision {
    name: "Thakurgaon",
    code: "BD-64",
    subdivision_type: "District",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-64",
};


pub const BD_A: Subdivision = Subdivision {
    name: "Barisal",
    code: "BD-A",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-A",
};


pub const BD_B: Subdivision = Subdivision {
    name: "Chittagong",
    code: "BD-B",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-B",
};


pub const BD_C: Subdivision = Subdivision {
    name: "Dhaka",
    code: "BD-C",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-C",
};


pub const BD_D: Subdivision = Subdivision {
    name: "Khulna",
    code: "BD-D",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-D",
};


pub const BD_E: Subdivision = Subdivision {
    name: "Rajshahi",
    code: "BD-E",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-E",
};


pub const BD_F: Subdivision = Subdivision {
    name: "Rangpur",
    code: "BD-F",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-F",
};


pub const BD_G: Subdivision = Subdivision {
    name: "Sylhet",
    code: "BD-G",
    subdivision_type: "Division",
    country_name: "Bangladesh",
    country_code: "BD",
    region_code: "BD-G",
};


pub const BB_01: Subdivision = Subdivision {
    name: "Christ Church",
    code: "BB-01",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-01",
};


pub const BB_02: Subdivision = Subdivision {
    name: "Saint Andrew",
    code: "BB-02",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-02",
};


pub const BB_03: Subdivision = Subdivision {
    name: "Saint George",
    code: "BB-03",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-03",
};


pub const BB_04: Subdivision = Subdivision {
    name: "Saint James",
    code: "BB-04",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-04",
};


pub const BB_05: Subdivision = Subdivision {
    name: "Saint John",
    code: "BB-05",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-05",
};


pub const BB_06: Subdivision = Subdivision {
    name: "Saint Joseph",
    code: "BB-06",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-06",
};


pub const BB_07: Subdivision = Subdivision {
    name: "Saint Lucy",
    code: "BB-07",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-07",
};


pub const BB_08: Subdivision = Subdivision {
    name: "Saint Michael",
    code: "BB-08",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-08",
};


pub const BB_09: Subdivision = Subdivision {
    name: "Saint Peter",
    code: "BB-09",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-09",
};


pub const BB_10: Subdivision = Subdivision {
    name: "Saint Philip",
    code: "BB-10",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-10",
};


pub const BB_11: Subdivision = Subdivision {
    name: "Saint Thomas",
    code: "BB-11",
    subdivision_type: "Parish",
    country_name: "Barbados",
    country_code: "BB",
    region_code: "BB-11",
};


pub const BY_BR: Subdivision = Subdivision {
    name: "Bresckaja voblasć",
    code: "BY-BR",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-BR",
};


pub const BY_HM: Subdivision = Subdivision {
    name: "Gorod Minsk",
    code: "BY-HM",
    subdivision_type: "City",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-HM",
};


pub const BY_HO: Subdivision = Subdivision {
    name: "Gomel'skaja oblast'",
    code: "BY-HO",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-HO",
};


pub const BY_HR: Subdivision = Subdivision {
    name: "Grodnenskaja oblast'",
    code: "BY-HR",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-HR",
};


pub const BY_MA: Subdivision = Subdivision {
    name: "Mahilioŭskaja voblasć",
    code: "BY-MA",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-MA",
};


pub const BY_MI: Subdivision = Subdivision {
    name: "Minskaja oblast'",
    code: "BY-MI",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-MI",
};


pub const BY_VI: Subdivision = Subdivision {
    name: "Viciebskaja voblasć",
    code: "BY-VI",
    subdivision_type: "Oblast",
    country_name: "Belarus",
    country_code: "BY",
    region_code: "BY-VI",
};


pub const BE_BRU: Subdivision = Subdivision {
    name: "Brussels Hoofdstedelijk Gewest",
    code: "BE-BRU",
    subdivision_type: "Region",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-BRU",
};


pub const BE_VAN: Subdivision = Subdivision {
    name: "Antwerpen",
    code: "BE-VAN",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VAN",
};


pub const BE_VBR: Subdivision = Subdivision {
    name: "Vlaams-Brabant",
    code: "BE-VBR",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VBR",
};


pub const BE_VLG: Subdivision = Subdivision {
    name: "Vlaams Gewest",
    code: "BE-VLG",
    subdivision_type: "Region",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VLG",
};


pub const BE_VLI: Subdivision = Subdivision {
    name: "Limburg",
    code: "BE-VLI",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VLI",
};


pub const BE_VOV: Subdivision = Subdivision {
    name: "Oost-Vlaanderen",
    code: "BE-VOV",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VOV",
};


pub const BE_VWV: Subdivision = Subdivision {
    name: "West-Vlaanderen",
    code: "BE-VWV",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-VWV",
};


pub const BE_WAL: Subdivision = Subdivision {
    name: "wallonne, Région",
    code: "BE-WAL",
    subdivision_type: "Region",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WAL",
};


pub const BE_WBR: Subdivision = Subdivision {
    name: "Brabant wallon",
    code: "BE-WBR",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WBR",
};


pub const BE_WHT: Subdivision = Subdivision {
    name: "Hainaut",
    code: "BE-WHT",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WHT",
};


pub const BE_WLG: Subdivision = Subdivision {
    name: "Liège",
    code: "BE-WLG",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WLG",
};


pub const BE_WLX: Subdivision = Subdivision {
    name: "Luxembourg",
    code: "BE-WLX",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WLX",
};


pub const BE_WNA: Subdivision = Subdivision {
    name: "Namur",
    code: "BE-WNA",
    subdivision_type: "Province",
    country_name: "Belgium",
    country_code: "BE",
    region_code: "BE-WNA",
};


pub const BZ_BZ: Subdivision = Subdivision {
    name: "Belize",
    code: "BZ-BZ",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-BZ",
};


pub const BZ_CY: Subdivision = Subdivision {
    name: "Cayo",
    code: "BZ-CY",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-CY",
};


pub const BZ_CZL: Subdivision = Subdivision {
    name: "Corozal",
    code: "BZ-CZL",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-CZL",
};


pub const BZ_OW: Subdivision = Subdivision {
    name: "Orange Walk",
    code: "BZ-OW",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-OW",
};


pub const BZ_SC: Subdivision = Subdivision {
    name: "Stann Creek",
    code: "BZ-SC",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-SC",
};


pub const BZ_TOL: Subdivision = Subdivision {
    name: "Toledo",
    code: "BZ-TOL",
    subdivision_type: "District",
    country_name: "Belize",
    country_code: "BZ",
    region_code: "BZ-TOL",
};


pub const BJ_AK: Subdivision = Subdivision {
    name: "Atakora",
    code: "BJ-AK",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-AK",
};


pub const BJ_AL: Subdivision = Subdivision {
    name: "Alibori",
    code: "BJ-AL",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-AL",
};


pub const BJ_AQ: Subdivision = Subdivision {
    name: "Atlantique",
    code: "BJ-AQ",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-AQ",
};


pub const BJ_BO: Subdivision = Subdivision {
    name: "Borgou",
    code: "BJ-BO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-BO",
};


pub const BJ_CO: Subdivision = Subdivision {
    name: "Collines",
    code: "BJ-CO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-CO",
};


pub const BJ_DO: Subdivision = Subdivision {
    name: "Donga",
    code: "BJ-DO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-DO",
};


pub const BJ_KO: Subdivision = Subdivision {
    name: "Kouffo",
    code: "BJ-KO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-KO",
};


pub const BJ_LI: Subdivision = Subdivision {
    name: "Littoral",
    code: "BJ-LI",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-LI",
};


pub const BJ_MO: Subdivision = Subdivision {
    name: "Mono",
    code: "BJ-MO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-MO",
};


pub const BJ_OU: Subdivision = Subdivision {
    name: "Ouémé",
    code: "BJ-OU",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-OU",
};


pub const BJ_PL: Subdivision = Subdivision {
    name: "Plateau",
    code: "BJ-PL",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-PL",
};


pub const BJ_ZO: Subdivision = Subdivision {
    name: "Zou",
    code: "BJ-ZO",
    subdivision_type: "Department",
    country_name: "Benin",
    country_code: "BJ",
    region_code: "BJ-ZO",
};


pub const BM_BM: Subdivision = Subdivision {
    name: "Bermuda",
    code: "BM-BM",
    subdivision_type: "Country",
    country_name: "Bermuda",
    country_code: "BM",
    region_code: "BM-BM",
};


pub const BT_11: Subdivision = Subdivision {
    name: "Paro",
    code: "BT-11",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-11",
};


pub const BT_12: Subdivision = Subdivision {
    name: "Chhukha",
    code: "BT-12",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-12",
};


pub const BT_13: Subdivision = Subdivision {
    name: "Ha",
    code: "BT-13",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-13",
};


pub const BT_14: Subdivision = Subdivision {
    name: "Samtse",
    code: "BT-14",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-14",
};


pub const BT_15: Subdivision = Subdivision {
    name: "Thimphu",
    code: "BT-15",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-15",
};


pub const BT_21: Subdivision = Subdivision {
    name: "Tsirang",
    code: "BT-21",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-21",
};


pub const BT_22: Subdivision = Subdivision {
    name: "Dagana",
    code: "BT-22",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-22",
};


pub const BT_23: Subdivision = Subdivision {
    name: "Punakha",
    code: "BT-23",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-23",
};


pub const BT_24: Subdivision = Subdivision {
    name: "Wangdue Phodrang",
    code: "BT-24",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-24",
};


pub const BT_31: Subdivision = Subdivision {
    name: "Sarpang",
    code: "BT-31",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-31",
};


pub const BT_32: Subdivision = Subdivision {
    name: "Trongsa",
    code: "BT-32",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-32",
};


pub const BT_33: Subdivision = Subdivision {
    name: "Bumthang",
    code: "BT-33",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-33",
};


pub const BT_34: Subdivision = Subdivision {
    name: "Zhemgang",
    code: "BT-34",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-34",
};


pub const BT_41: Subdivision = Subdivision {
    name: "Trashigang",
    code: "BT-41",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-41",
};


pub const BT_42: Subdivision = Subdivision {
    name: "Monggar",
    code: "BT-42",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-42",
};


pub const BT_43: Subdivision = Subdivision {
    name: "Pemagatshel",
    code: "BT-43",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-43",
};


pub const BT_44: Subdivision = Subdivision {
    name: "Lhuentse",
    code: "BT-44",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-44",
};


pub const BT_45: Subdivision = Subdivision {
    name: "Samdrup Jongkha",
    code: "BT-45",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-45",
};


pub const BT_GA: Subdivision = Subdivision {
    name: "Gasa",
    code: "BT-GA",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-GA",
};


pub const BT_TY: Subdivision = Subdivision {
    name: "Trashi Yangtse",
    code: "BT-TY",
    subdivision_type: "District",
    country_name: "Bhutan",
    country_code: "BT",
    region_code: "BT-TY",
};


pub const BO_B: Subdivision = Subdivision {
    name: "El Beni",
    code: "BO-B",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-B",
};


pub const BO_C: Subdivision = Subdivision {
    name: "Cochabamba",
    code: "BO-C",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-C",
};


pub const BO_H: Subdivision = Subdivision {
    name: "Chuquisaca",
    code: "BO-H",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-H",
};


pub const BO_L: Subdivision = Subdivision {
    name: "La Paz",
    code: "BO-L",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-L",
};


pub const BO_N: Subdivision = Subdivision {
    name: "Pando",
    code: "BO-N",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-N",
};


pub const BO_O: Subdivision = Subdivision {
    name: "Oruro",
    code: "BO-O",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-O",
};


pub const BO_P: Subdivision = Subdivision {
    name: "Potosí",
    code: "BO-P",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-P",
};


pub const BO_S: Subdivision = Subdivision {
    name: "Santa Cruz",
    code: "BO-S",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-S",
};


pub const BO_T: Subdivision = Subdivision {
    name: "Tarija",
    code: "BO-T",
    subdivision_type: "Department",
    country_name: "Bolivia",
    country_code: "BO",
    region_code: "BO-T",
};


pub const BQ_BO: Subdivision = Subdivision {
    name: "Bonaire",
    code: "BQ-BO",
    subdivision_type: "Special municipality",
    country_name: "Bonaire, Sint Eustatius and Saba",
    country_code: "BQ",
    region_code: "BQ-BO",
};


pub const BQ_SA: Subdivision = Subdivision {
    name: "Saba",
    code: "BQ-SA",
    subdivision_type: "Special municipality",
    country_name: "Bonaire, Sint Eustatius and Saba",
    country_code: "BQ",
    region_code: "BQ-SA",
};


pub const BQ_SE: Subdivision = Subdivision {
    name: "Sint Eustatius",
    code: "BQ-SE",
    subdivision_type: "Special municipality",
    country_name: "Bonaire, Sint Eustatius and Saba",
    country_code: "BQ",
    region_code: "BQ-SE",
};


pub const BA_01: Subdivision = Subdivision {
    name: "Unsko-sanska županija",
    code: "BA-01",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-01",
};


pub const BA_02: Subdivision = Subdivision {
    name: "Posavska županija",
    code: "BA-02",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-02",
};


pub const BA_03: Subdivision = Subdivision {
    name: "Tuzlanska županija",
    code: "BA-03",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-03",
};


pub const BA_04: Subdivision = Subdivision {
    name: "Zeničko-dobojska županija",
    code: "BA-04",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-04",
};


pub const BA_05: Subdivision = Subdivision {
    name: "Bosansko-podrinjska županija",
    code: "BA-05",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-05",
};


pub const BA_06: Subdivision = Subdivision {
    name: "Srednjobosanska županija",
    code: "BA-06",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-06",
};


pub const BA_07: Subdivision = Subdivision {
    name: "Hercegovačko-neretvanska županija",
    code: "BA-07",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-07",
};


pub const BA_08: Subdivision = Subdivision {
    name: "Zapadnohercegovačka županija",
    code: "BA-08",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-08",
};


pub const BA_09: Subdivision = Subdivision {
    name: "Kanton Sarajevo",
    code: "BA-09",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-09",
};


pub const BA_10: Subdivision = Subdivision {
    name: "Kanton br. 10",
    code: "BA-10",
    subdivision_type: "Canton",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-10",
};


pub const BA_BIH: Subdivision = Subdivision {
    name: "Federacija Bosne i Hercegovine",
    code: "BA-BIH",
    subdivision_type: "Entity",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-BIH",
};


pub const BA_BRC: Subdivision = Subdivision {
    name: "Brčko distrikt",
    code: "BA-BRC",
    subdivision_type: "District with special status",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-BRC",
};


pub const BA_SRP: Subdivision = Subdivision {
    name: "Republika Srpska",
    code: "BA-SRP",
    subdivision_type: "Entity",
    country_name: "Bosnia and Herzegovina",
    country_code: "BA",
    region_code: "BA-SRP",
};


pub const BW_CE: Subdivision = Subdivision {
    name: "Central",
    code: "BW-CE",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-CE",
};


pub const BW_CH: Subdivision = Subdivision {
    name: "Chobe",
    code: "BW-CH",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-CH",
};


pub const BW_FR: Subdivision = Subdivision {
    name: "Francistown",
    code: "BW-FR",
    subdivision_type: "City",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-FR",
};


pub const BW_GA: Subdivision = Subdivision {
    name: "Gaborone",
    code: "BW-GA",
    subdivision_type: "City",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-GA",
};


pub const BW_GH: Subdivision = Subdivision {
    name: "Ghanzi",
    code: "BW-GH",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-GH",
};


pub const BW_JW: Subdivision = Subdivision {
    name: "Jwaneng",
    code: "BW-JW",
    subdivision_type: "Town",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-JW",
};


pub const BW_KG: Subdivision = Subdivision {
    name: "Kgalagadi",
    code: "BW-KG",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-KG",
};


pub const BW_KL: Subdivision = Subdivision {
    name: "Kgatleng",
    code: "BW-KL",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-KL",
};


pub const BW_KW: Subdivision = Subdivision {
    name: "Kweneng",
    code: "BW-KW",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-KW",
};


pub const BW_LO: Subdivision = Subdivision {
    name: "Lobatse",
    code: "BW-LO",
    subdivision_type: "Town",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-LO",
};


pub const BW_NE: Subdivision = Subdivision {
    name: "North East",
    code: "BW-NE",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-NE",
};


pub const BW_NW: Subdivision = Subdivision {
    name: "North West",
    code: "BW-NW",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-NW",
};


pub const BW_SE: Subdivision = Subdivision {
    name: "South East",
    code: "BW-SE",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-SE",
};


pub const BW_SO: Subdivision = Subdivision {
    name: "Southern",
    code: "BW-SO",
    subdivision_type: "District",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-SO",
};


pub const BW_SP: Subdivision = Subdivision {
    name: "Selibe Phikwe",
    code: "BW-SP",
    subdivision_type: "Town",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-SP",
};


pub const BW_ST: Subdivision = Subdivision {
    name: "Sowa Town",
    code: "BW-ST",
    subdivision_type: "Town",
    country_name: "Botswana",
    country_code: "BW",
    region_code: "BW-ST",
};


pub const BV_BV: Subdivision = Subdivision {
    name: "Bouvet Island",
    code: "BV-BV",
    subdivision_type: "Country",
    country_name: "Bouvet Island",
    country_code: "BV",
    region_code: "BV-BV",
};


pub const BR_AC: Subdivision = Subdivision {
    name: "Acre",
    code: "BR-AC",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-AC",
};


pub const BR_AL: Subdivision = Subdivision {
    name: "Alagoas",
    code: "BR-AL",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-AL",
};


pub const BR_AM: Subdivision = Subdivision {
    name: "Amazonas",
    code: "BR-AM",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-AM",
};


pub const BR_AP: Subdivision = Subdivision {
    name: "Amapá",
    code: "BR-AP",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-AP",
};


pub const BR_BA: Subdivision = Subdivision {
    name: "Bahia",
    code: "BR-BA",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-BA",
};


pub const BR_CE: Subdivision = Subdivision {
    name: "Ceará",
    code: "BR-CE",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-CE",
};


pub const BR_DF: Subdivision = Subdivision {
    name: "Distrito Federal",
    code: "BR-DF",
    subdivision_type: "Federal district",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-DF",
};


pub const BR_ES: Subdivision = Subdivision {
    name: "Espírito Santo",
    code: "BR-ES",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-ES",
};


pub const BR_GO: Subdivision = Subdivision {
    name: "Goiás",
    code: "BR-GO",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-GO",
};


pub const BR_MA: Subdivision = Subdivision {
    name: "Maranhão",
    code: "BR-MA",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-MA",
};


pub const BR_MG: Subdivision = Subdivision {
    name: "Minas Gerais",
    code: "BR-MG",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-MG",
};


pub const BR_MS: Subdivision = Subdivision {
    name: "Mato Grosso do Sul",
    code: "BR-MS",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-MS",
};


pub const BR_MT: Subdivision = Subdivision {
    name: "Mato Grosso",
    code: "BR-MT",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-MT",
};


pub const BR_PA: Subdivision = Subdivision {
    name: "Pará",
    code: "BR-PA",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-PA",
};


pub const BR_PB: Subdivision = Subdivision {
    name: "Paraíba",
    code: "BR-PB",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-PB",
};


pub const BR_PE: Subdivision = Subdivision {
    name: "Pernambuco",
    code: "BR-PE",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-PE",
};


pub const BR_PI: Subdivision = Subdivision {
    name: "Piauí",
    code: "BR-PI",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-PI",
};


pub const BR_PR: Subdivision = Subdivision {
    name: "Paraná",
    code: "BR-PR",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-PR",
};


pub const BR_RJ: Subdivision = Subdivision {
    name: "Rio de Janeiro",
    code: "BR-RJ",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-RJ",
};


pub const BR_RN: Subdivision = Subdivision {
    name: "Rio Grande do Norte",
    code: "BR-RN",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-RN",
};


pub const BR_RO: Subdivision = Subdivision {
    name: "Rondônia",
    code: "BR-RO",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-RO",
};


pub const BR_RR: Subdivision = Subdivision {
    name: "Roraima",
    code: "BR-RR",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-RR",
};


pub const BR_RS: Subdivision = Subdivision {
    name: "Rio Grande do Sul",
    code: "BR-RS",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-RS",
};


pub const BR_SC: Subdivision = Subdivision {
    name: "Santa Catarina",
    code: "BR-SC",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-SC",
};


pub const BR_SE: Subdivision = Subdivision {
    name: "Sergipe",
    code: "BR-SE",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-SE",
};


pub const BR_SP: Subdivision = Subdivision {
    name: "São Paulo",
    code: "BR-SP",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-SP",
};


pub const BR_TO: Subdivision = Subdivision {
    name: "Tocantins",
    code: "BR-TO",
    subdivision_type: "State",
    country_name: "Brazil",
    country_code: "BR",
    region_code: "BR-TO",
};


pub const IO_IO: Subdivision = Subdivision {
    name: "British Indian Ocean Territory",
    code: "IO-IO",
    subdivision_type: "Country",
    country_name: "British Indian Ocean Territory",
    country_code: "IO",
    region_code: "IO-IO",
};


pub const BN_BE: Subdivision = Subdivision {
    name: "Belait",
    code: "BN-BE",
    subdivision_type: "District",
    country_name: "Brunei Darussalam",
    country_code: "BN",
    region_code: "BN-BE",
};


pub const BN_BM: Subdivision = Subdivision {
    name: "Brunei-Muara",
    code: "BN-BM",
    subdivision_type: "District",
    country_name: "Brunei Darussalam",
    country_code: "BN",
    region_code: "BN-BM",
};


pub const BN_TE: Subdivision = Subdivision {
    name: "Temburong",
    code: "BN-TE",
    subdivision_type: "District",
    country_name: "Brunei Darussalam",
    country_code: "BN",
    region_code: "BN-TE",
};


pub const BN_TU: Subdivision = Subdivision {
    name: "Tutong",
    code: "BN-TU",
    subdivision_type: "District",
    country_name: "Brunei Darussalam",
    country_code: "BN",
    region_code: "BN-TU",
};


pub const BG_01: Subdivision = Subdivision {
    name: "Blagoevgrad",
    code: "BG-01",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-01",
};


pub const BG_02: Subdivision = Subdivision {
    name: "Burgas",
    code: "BG-02",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-02",
};


pub const BG_03: Subdivision = Subdivision {
    name: "Varna",
    code: "BG-03",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-03",
};


pub const BG_04: Subdivision = Subdivision {
    name: "Veliko Tarnovo",
    code: "BG-04",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-04",
};


pub const BG_05: Subdivision = Subdivision {
    name: "Vidin",
    code: "BG-05",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-05",
};


pub const BG_06: Subdivision = Subdivision {
    name: "Vratsa",
    code: "BG-06",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-06",
};


pub const BG_07: Subdivision = Subdivision {
    name: "Gabrovo",
    code: "BG-07",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-07",
};


pub const BG_08: Subdivision = Subdivision {
    name: "Dobrich",
    code: "BG-08",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-08",
};


pub const BG_09: Subdivision = Subdivision {
    name: "Kardzhali",
    code: "BG-09",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-09",
};


pub const BG_10: Subdivision = Subdivision {
    name: "Kyustendil",
    code: "BG-10",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-10",
};


pub const BG_11: Subdivision = Subdivision {
    name: "Lovech",
    code: "BG-11",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-11",
};


pub const BG_12: Subdivision = Subdivision {
    name: "Montana",
    code: "BG-12",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-12",
};


pub const BG_13: Subdivision = Subdivision {
    name: "Pazardzhik",
    code: "BG-13",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-13",
};


pub const BG_14: Subdivision = Subdivision {
    name: "Pernik",
    code: "BG-14",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-14",
};


pub const BG_15: Subdivision = Subdivision {
    name: "Pleven",
    code: "BG-15",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-15",
};


pub const BG_16: Subdivision = Subdivision {
    name: "Plovdiv",
    code: "BG-16",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-16",
};


pub const BG_17: Subdivision = Subdivision {
    name: "Razgrad",
    code: "BG-17",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-17",
};


pub const BG_18: Subdivision = Subdivision {
    name: "Ruse",
    code: "BG-18",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-18",
};


pub const BG_19: Subdivision = Subdivision {
    name: "Silistra",
    code: "BG-19",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-19",
};


pub const BG_20: Subdivision = Subdivision {
    name: "Sliven",
    code: "BG-20",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-20",
};


pub const BG_21: Subdivision = Subdivision {
    name: "Smolyan",
    code: "BG-21",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-21",
};


pub const BG_22: Subdivision = Subdivision {
    name: "Sofia",
    code: "BG-22",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-22",
};


pub const BG_23: Subdivision = Subdivision {
    name: "Sofia",
    code: "BG-23",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-23",
};


pub const BG_24: Subdivision = Subdivision {
    name: "Stara Zagora",
    code: "BG-24",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-24",
};


pub const BG_25: Subdivision = Subdivision {
    name: "Targovishte",
    code: "BG-25",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-25",
};


pub const BG_26: Subdivision = Subdivision {
    name: "Haskovo",
    code: "BG-26",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-26",
};


pub const BG_27: Subdivision = Subdivision {
    name: "Shumen",
    code: "BG-27",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-27",
};


pub const BG_28: Subdivision = Subdivision {
    name: "Yambol",
    code: "BG-28",
    subdivision_type: "Region",
    country_name: "Bulgaria",
    country_code: "BG",
    region_code: "BG-28",
};


pub const BF_01: Subdivision = Subdivision {
    name: "Boucle du Mouhoun",
    code: "BF-01",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-01",
};


pub const BF_02: Subdivision = Subdivision {
    name: "Cascades",
    code: "BF-02",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-02",
};


pub const BF_03: Subdivision = Subdivision {
    name: "Centre",
    code: "BF-03",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-03",
};


pub const BF_04: Subdivision = Subdivision {
    name: "Centre-Est",
    code: "BF-04",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-04",
};


pub const BF_05: Subdivision = Subdivision {
    name: "Centre-Nord",
    code: "BF-05",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-05",
};


pub const BF_06: Subdivision = Subdivision {
    name: "Centre-Ouest",
    code: "BF-06",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-06",
};


pub const BF_07: Subdivision = Subdivision {
    name: "Centre-Sud",
    code: "BF-07",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-07",
};


pub const BF_08: Subdivision = Subdivision {
    name: "Est",
    code: "BF-08",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-08",
};


pub const BF_09: Subdivision = Subdivision {
    name: "Hauts-Bassins",
    code: "BF-09",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-09",
};


pub const BF_10: Subdivision = Subdivision {
    name: "Nord",
    code: "BF-10",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-10",
};


pub const BF_11: Subdivision = Subdivision {
    name: "Plateau-Central",
    code: "BF-11",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-11",
};


pub const BF_12: Subdivision = Subdivision {
    name: "Sahel",
    code: "BF-12",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-12",
};


pub const BF_13: Subdivision = Subdivision {
    name: "Sud-Ouest",
    code: "BF-13",
    subdivision_type: "Region",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-13",
};


pub const BF_BAL: Subdivision = Subdivision {
    name: "Balé",
    code: "BF-BAL",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BAL",
};


pub const BF_BAM: Subdivision = Subdivision {
    name: "Bam",
    code: "BF-BAM",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BAM",
};


pub const BF_BAN: Subdivision = Subdivision {
    name: "Banwa",
    code: "BF-BAN",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BAN",
};


pub const BF_BAZ: Subdivision = Subdivision {
    name: "Bazèga",
    code: "BF-BAZ",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BAZ",
};


pub const BF_BGR: Subdivision = Subdivision {
    name: "Bougouriba",
    code: "BF-BGR",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BGR",
};


pub const BF_BLG: Subdivision = Subdivision {
    name: "Boulgou",
    code: "BF-BLG",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BLG",
};


pub const BF_BLK: Subdivision = Subdivision {
    name: "Boulkiemdé",
    code: "BF-BLK",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-BLK",
};


pub const BF_COM: Subdivision = Subdivision {
    name: "Comoé",
    code: "BF-COM",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-COM",
};


pub const BF_GAN: Subdivision = Subdivision {
    name: "Ganzourgou",
    code: "BF-GAN",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-GAN",
};


pub const BF_GNA: Subdivision = Subdivision {
    name: "Gnagna",
    code: "BF-GNA",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-GNA",
};


pub const BF_GOU: Subdivision = Subdivision {
    name: "Gourma",
    code: "BF-GOU",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-GOU",
};


pub const BF_HOU: Subdivision = Subdivision {
    name: "Houet",
    code: "BF-HOU",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-HOU",
};


pub const BF_IOB: Subdivision = Subdivision {
    name: "Ioba",
    code: "BF-IOB",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-IOB",
};


pub const BF_KAD: Subdivision = Subdivision {
    name: "Kadiogo",
    code: "BF-KAD",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KAD",
};


pub const BF_KEN: Subdivision = Subdivision {
    name: "Kénédougou",
    code: "BF-KEN",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KEN",
};


pub const BF_KMD: Subdivision = Subdivision {
    name: "Komondjari",
    code: "BF-KMD",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KMD",
};


pub const BF_KMP: Subdivision = Subdivision {
    name: "Kompienga",
    code: "BF-KMP",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KMP",
};


pub const BF_KOP: Subdivision = Subdivision {
    name: "Koulpélogo",
    code: "BF-KOP",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KOP",
};


pub const BF_KOS: Subdivision = Subdivision {
    name: "Kossi",
    code: "BF-KOS",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KOS",
};


pub const BF_KOT: Subdivision = Subdivision {
    name: "Kouritenga",
    code: "BF-KOT",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KOT",
};


pub const BF_KOW: Subdivision = Subdivision {
    name: "Kourwéogo",
    code: "BF-KOW",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-KOW",
};


pub const BF_LER: Subdivision = Subdivision {
    name: "Léraba",
    code: "BF-LER",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-LER",
};


pub const BF_LOR: Subdivision = Subdivision {
    name: "Loroum",
    code: "BF-LOR",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-LOR",
};


pub const BF_MOU: Subdivision = Subdivision {
    name: "Mouhoun",
    code: "BF-MOU",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-MOU",
};


pub const BF_NAM: Subdivision = Subdivision {
    name: "Namentenga",
    code: "BF-NAM",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-NAM",
};


pub const BF_NAO: Subdivision = Subdivision {
    name: "Nahouri",
    code: "BF-NAO",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-NAO",
};


pub const BF_NAY: Subdivision = Subdivision {
    name: "Nayala",
    code: "BF-NAY",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-NAY",
};


pub const BF_NOU: Subdivision = Subdivision {
    name: "Noumbiel",
    code: "BF-NOU",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-NOU",
};


pub const BF_OUB: Subdivision = Subdivision {
    name: "Oubritenga",
    code: "BF-OUB",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-OUB",
};


pub const BF_OUD: Subdivision = Subdivision {
    name: "Oudalan",
    code: "BF-OUD",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-OUD",
};


pub const BF_PAS: Subdivision = Subdivision {
    name: "Passoré",
    code: "BF-PAS",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-PAS",
};


pub const BF_PON: Subdivision = Subdivision {
    name: "Poni",
    code: "BF-PON",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-PON",
};


pub const BF_SEN: Subdivision = Subdivision {
    name: "Séno",
    code: "BF-SEN",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SEN",
};


pub const BF_SIS: Subdivision = Subdivision {
    name: "Sissili",
    code: "BF-SIS",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SIS",
};


pub const BF_SMT: Subdivision = Subdivision {
    name: "Sanmatenga",
    code: "BF-SMT",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SMT",
};


pub const BF_SNG: Subdivision = Subdivision {
    name: "Sanguié",
    code: "BF-SNG",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SNG",
};


pub const BF_SOM: Subdivision = Subdivision {
    name: "Soum",
    code: "BF-SOM",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SOM",
};


pub const BF_SOR: Subdivision = Subdivision {
    name: "Sourou",
    code: "BF-SOR",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-SOR",
};


pub const BF_TAP: Subdivision = Subdivision {
    name: "Tapoa",
    code: "BF-TAP",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-TAP",
};


pub const BF_TUI: Subdivision = Subdivision {
    name: "Tui",
    code: "BF-TUI",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-TUI",
};


pub const BF_YAG: Subdivision = Subdivision {
    name: "Yagha",
    code: "BF-YAG",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-YAG",
};


pub const BF_YAT: Subdivision = Subdivision {
    name: "Yatenga",
    code: "BF-YAT",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-YAT",
};


pub const BF_ZIR: Subdivision = Subdivision {
    name: "Ziro",
    code: "BF-ZIR",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-ZIR",
};


pub const BF_ZON: Subdivision = Subdivision {
    name: "Zondoma",
    code: "BF-ZON",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-ZON",
};


pub const BF_ZOU: Subdivision = Subdivision {
    name: "Zoundwéogo",
    code: "BF-ZOU",
    subdivision_type: "Province",
    country_name: "Burkina Faso",
    country_code: "BF",
    region_code: "BF-ZOU",
};


pub const BI_BB: Subdivision = Subdivision {
    name: "Bubanza",
    code: "BI-BB",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-BB",
};


pub const BI_BL: Subdivision = Subdivision {
    name: "Bujumbura Rural",
    code: "BI-BL",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-BL",
};


pub const BI_BM: Subdivision = Subdivision {
    name: "Bujumbura Mairie",
    code: "BI-BM",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-BM",
};


pub const BI_BR: Subdivision = Subdivision {
    name: "Bururi",
    code: "BI-BR",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-BR",
};


pub const BI_CA: Subdivision = Subdivision {
    name: "Cankuzo",
    code: "BI-CA",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-CA",
};


pub const BI_CI: Subdivision = Subdivision {
    name: "Cibitoke",
    code: "BI-CI",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-CI",
};


pub const BI_GI: Subdivision = Subdivision {
    name: "Gitega",
    code: "BI-GI",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-GI",
};


pub const BI_KI: Subdivision = Subdivision {
    name: "Kirundo",
    code: "BI-KI",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-KI",
};


pub const BI_KR: Subdivision = Subdivision {
    name: "Karuzi",
    code: "BI-KR",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-KR",
};


pub const BI_KY: Subdivision = Subdivision {
    name: "Kayanza",
    code: "BI-KY",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-KY",
};


pub const BI_MA: Subdivision = Subdivision {
    name: "Makamba",
    code: "BI-MA",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-MA",
};


pub const BI_MU: Subdivision = Subdivision {
    name: "Muramvya",
    code: "BI-MU",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-MU",
};


pub const BI_MW: Subdivision = Subdivision {
    name: "Mwaro",
    code: "BI-MW",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-MW",
};


pub const BI_MY: Subdivision = Subdivision {
    name: "Muyinga",
    code: "BI-MY",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-MY",
};


pub const BI_NG: Subdivision = Subdivision {
    name: "Ngozi",
    code: "BI-NG",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-NG",
};


pub const BI_RT: Subdivision = Subdivision {
    name: "Rutana",
    code: "BI-RT",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-RT",
};


pub const BI_RY: Subdivision = Subdivision {
    name: "Ruyigi",
    code: "BI-RY",
    subdivision_type: "Province",
    country_name: "Burundi",
    country_code: "BI",
    region_code: "BI-RY",
};


pub const KH_10: Subdivision = Subdivision {
    name: "Krâchéh",
    code: "KH-10",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-10",
};


pub const KH_11: Subdivision = Subdivision {
    name: "Môndól Kiri",
    code: "KH-11",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-11",
};


pub const KH_12: Subdivision = Subdivision {
    name: "Phnom Penh",
    code: "KH-12",
    subdivision_type: "Autonomous municipality",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-12",
};


pub const KH_13: Subdivision = Subdivision {
    name: "Preăh Vihéar",
    code: "KH-13",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-13",
};


pub const KH_14: Subdivision = Subdivision {
    name: "Prey Veaeng",
    code: "KH-14",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-14",
};


pub const KH_15: Subdivision = Subdivision {
    name: "Pousaat",
    code: "KH-15",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-15",
};


pub const KH_16: Subdivision = Subdivision {
    name: "Rotanak Kiri",
    code: "KH-16",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-16",
};


pub const KH_17: Subdivision = Subdivision {
    name: "Siem Reab",
    code: "KH-17",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-17",
};


pub const KH_18: Subdivision = Subdivision {
    name: "Krŏng Preăh Sihanouk",
    code: "KH-18",
    subdivision_type: "Autonomous municipality",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-18",
};


pub const KH_19: Subdivision = Subdivision {
    name: "Stoĕng Trêng",
    code: "KH-19",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-19",
};


pub const KH_1: Subdivision = Subdivision {
    name: "Bântéay Méanchey",
    code: "KH-1",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-1",
};


pub const KH_20: Subdivision = Subdivision {
    name: "Svaay Rieng",
    code: "KH-20",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-20",
};


pub const KH_21: Subdivision = Subdivision {
    name: "Taakaev",
    code: "KH-21",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-21",
};


pub const KH_22: Subdivision = Subdivision {
    name: "Ŏtdâr Méanchey",
    code: "KH-22",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-22",
};


pub const KH_23: Subdivision = Subdivision {
    name: "Krong Kaeb",
    code: "KH-23",
    subdivision_type: "Autonomous municipality",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-23",
};


pub const KH_24: Subdivision = Subdivision {
    name: "Krŏng Pailĭn",
    code: "KH-24",
    subdivision_type: "Autonomous municipality",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-24",
};


pub const KH_2: Subdivision = Subdivision {
    name: "Baat Dambang",
    code: "KH-2",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-2",
};


pub const KH_3: Subdivision = Subdivision {
    name: "Kampong Chaam",
    code: "KH-3",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-3",
};


pub const KH_4: Subdivision = Subdivision {
    name: "Kampong Chhnang",
    code: "KH-4",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-4",
};


pub const KH_5: Subdivision = Subdivision {
    name: "Kâmpóng Spœ",
    code: "KH-5",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-5",
};


pub const KH_6: Subdivision = Subdivision {
    name: "Kâmpóng Thum",
    code: "KH-6",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-6",
};


pub const KH_7: Subdivision = Subdivision {
    name: "Kampot",
    code: "KH-7",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-7",
};


pub const KH_8: Subdivision = Subdivision {
    name: "Kandaal",
    code: "KH-8",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-8",
};


pub const KH_9: Subdivision = Subdivision {
    name: "Kaoh Kong",
    code: "KH-9",
    subdivision_type: "Province",
    country_name: "Cambodia",
    country_code: "KH",
    region_code: "KH-9",
};


pub const CM_AD: Subdivision = Subdivision {
    name: "Adamaoua",
    code: "CM-AD",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-AD",
};


pub const CM_CE: Subdivision = Subdivision {
    name: "Centre",
    code: "CM-CE",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-CE",
};


pub const CM_EN: Subdivision = Subdivision {
    name: "Extrême-Nord",
    code: "CM-EN",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-EN",
};


pub const CM_ES: Subdivision = Subdivision {
    name: "East",
    code: "CM-ES",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-ES",
};


pub const CM_LT: Subdivision = Subdivision {
    name: "Littoral",
    code: "CM-LT",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-LT",
};


pub const CM_NO: Subdivision = Subdivision {
    name: "Nord",
    code: "CM-NO",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-NO",
};


pub const CM_NW: Subdivision = Subdivision {
    name: "Nord-Ouest",
    code: "CM-NW",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-NW",
};


pub const CM_OU: Subdivision = Subdivision {
    name: "Ouest",
    code: "CM-OU",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-OU",
};


pub const CM_SU: Subdivision = Subdivision {
    name: "South",
    code: "CM-SU",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-SU",
};


pub const CM_SW: Subdivision = Subdivision {
    name: "South-West",
    code: "CM-SW",
    subdivision_type: "Region",
    country_name: "Cameroon",
    country_code: "CM",
    region_code: "CM-SW",
};


pub const CA_AB: Subdivision = Subdivision {
    name: "Alberta",
    code: "CA-AB",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-AB",
};


pub const CA_BC: Subdivision = Subdivision {
    name: "British Columbia",
    code: "CA-BC",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-BC",
};


pub const CA_MB: Subdivision = Subdivision {
    name: "Manitoba",
    code: "CA-MB",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-MB",
};


pub const CA_NB: Subdivision = Subdivision {
    name: "New Brunswick",
    code: "CA-NB",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-NB",
};


pub const CA_NL: Subdivision = Subdivision {
    name: "Newfoundland and Labrador",
    code: "CA-NL",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-NL",
};


pub const CA_NS: Subdivision = Subdivision {
    name: "Nova Scotia",
    code: "CA-NS",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-NS",
};


pub const CA_NT: Subdivision = Subdivision {
    name: "Northwest Territories",
    code: "CA-NT",
    subdivision_type: "Territory",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-NT",
};


pub const CA_NU: Subdivision = Subdivision {
    name: "Nunavut",
    code: "CA-NU",
    subdivision_type: "Territory",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-NU",
};


pub const CA_ON: Subdivision = Subdivision {
    name: "Ontario",
    code: "CA-ON",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-ON",
};


pub const CA_PE: Subdivision = Subdivision {
    name: "Prince Edward Island",
    code: "CA-PE",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-PE",
};


pub const CA_QC: Subdivision = Subdivision {
    name: "Quebec",
    code: "CA-QC",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-QC",
};


pub const CA_SK: Subdivision = Subdivision {
    name: "Saskatchewan",
    code: "CA-SK",
    subdivision_type: "Province",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-SK",
};


pub const CA_YT: Subdivision = Subdivision {
    name: "Yukon",
    code: "CA-YT",
    subdivision_type: "Territory",
    country_name: "Canada",
    country_code: "CA",
    region_code: "CA-YT",
};


pub const CV_B: Subdivision = Subdivision {
    name: "Ilhas de Barlavento",
    code: "CV-B",
    subdivision_type: "Geographical region",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-B",
};


pub const CV_BR: Subdivision = Subdivision {
    name: "Brava",
    code: "CV-BR",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-BR",
};


pub const CV_BV: Subdivision = Subdivision {
    name: "Boa Vista",
    code: "CV-BV",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-BV",
};


pub const CV_CA: Subdivision = Subdivision {
    name: "Santa Catarina",
    code: "CV-CA",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-CA",
};


pub const CV_CF: Subdivision = Subdivision {
    name: "Santa Catarina do Fogo",
    code: "CV-CF",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-CF",
};


pub const CV_CR: Subdivision = Subdivision {
    name: "Santa Cruz",
    code: "CV-CR",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-CR",
};


pub const CV_MA: Subdivision = Subdivision {
    name: "Maio",
    code: "CV-MA",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-MA",
};


pub const CV_MO: Subdivision = Subdivision {
    name: "Mosteiros",
    code: "CV-MO",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-MO",
};


pub const CV_PA: Subdivision = Subdivision {
    name: "Paul",
    code: "CV-PA",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-PA",
};


pub const CV_PN: Subdivision = Subdivision {
    name: "Porto Novo",
    code: "CV-PN",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-PN",
};


pub const CV_PR: Subdivision = Subdivision {
    name: "Praia",
    code: "CV-PR",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-PR",
};


pub const CV_RB: Subdivision = Subdivision {
    name: "Ribeira Brava",
    code: "CV-RB",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-RB",
};


pub const CV_RG: Subdivision = Subdivision {
    name: "Ribeira Grande",
    code: "CV-RG",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-RG",
};


pub const CV_RS: Subdivision = Subdivision {
    name: "Ribeira Grande de Santiago",
    code: "CV-RS",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-RS",
};


pub const CV_SD: Subdivision = Subdivision {
    name: "São Domingos",
    code: "CV-SD",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SD",
};


pub const CV_SF: Subdivision = Subdivision {
    name: "São Filipe",
    code: "CV-SF",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SF",
};


pub const CV_S: Subdivision = Subdivision {
    name: "Ilhas de Sotavento",
    code: "CV-S",
    subdivision_type: "Geographical region",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-S",
};


pub const CV_SL: Subdivision = Subdivision {
    name: "Sal",
    code: "CV-SL",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SL",
};


pub const CV_SM: Subdivision = Subdivision {
    name: "São Miguel",
    code: "CV-SM",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SM",
};


pub const CV_SO: Subdivision = Subdivision {
    name: "São Lourenço dos Órgãos",
    code: "CV-SO",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SO",
};


pub const CV_SS: Subdivision = Subdivision {
    name: "São Salvador do Mundo",
    code: "CV-SS",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SS",
};


pub const CV_SV: Subdivision = Subdivision {
    name: "São Vicente",
    code: "CV-SV",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-SV",
};


pub const CV_TA: Subdivision = Subdivision {
    name: "Tarrafal",
    code: "CV-TA",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-TA",
};


pub const CV_TS: Subdivision = Subdivision {
    name: "Tarrafal de São Nicolau",
    code: "CV-TS",
    subdivision_type: "Municipality",
    country_name: "Cape Verde",
    country_code: "CV",
    region_code: "CV-TS",
};


pub const KY_KY: Subdivision = Subdivision {
    name: "Cayman Islands",
    code: "KY-KY",
    subdivision_type: "Country",
    country_name: "Cayman Islands",
    country_code: "KY",
    region_code: "KY-KY",
};


pub const XC_BA: Subdivision = Subdivision {
    name: "Barcelona",
    code: "XC-BA",
    subdivision_type: "Province",
    country_name: "Catalonia",
    country_code: "XC",
    region_code: "XC-BA",
};


pub const XC_GI: Subdivision = Subdivision {
    name: "Girona",
    code: "XC-GI",
    subdivision_type: "Province",
    country_name: "Catalonia",
    country_code: "XC",
    region_code: "XC-GI",
};


pub const XC_LL: Subdivision = Subdivision {
    name: "Lleida",
    code: "XC-LL",
    subdivision_type: "Province",
    country_name: "Catalonia",
    country_code: "XC",
    region_code: "XC-LL",
};


pub const XC_TA: Subdivision = Subdivision {
    name: "Tarragona",
    code: "XC-TA",
    subdivision_type: "Province",
    country_name: "Catalonia",
    country_code: "XC",
    region_code: "XC-TA",
};


pub const CF_AC: Subdivision = Subdivision {
    name: "Ouham",
    code: "CF-AC",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-AC",
};


pub const CF_BB: Subdivision = Subdivision {
    name: "Bamïngï-Bangoran",
    code: "CF-BB",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-BB",
};


pub const CF_BGF: Subdivision = Subdivision {
    name: "Bangî",
    code: "CF-BGF",
    subdivision_type: "Commune",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-BGF",
};


pub const CF_BK: Subdivision = Subdivision {
    name: "Basse-Kotto",
    code: "CF-BK",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-BK",
};


pub const CF_HK: Subdivision = Subdivision {
    name: "Haute-Kotto",
    code: "CF-HK",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-HK",
};


pub const CF_HM: Subdivision = Subdivision {
    name: "Haut-Mbomou",
    code: "CF-HM",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-HM",
};


pub const CF_HS: Subdivision = Subdivision {
    name: "Haute-Sangha / Mambéré-Kadéï",
    code: "CF-HS",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-HS",
};


pub const CF_KB: Subdivision = Subdivision {
    name: "Gïrïbïngï",
    code: "CF-KB",
    subdivision_type: "Economic prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-KB",
};


pub const CF_KG: Subdivision = Subdivision {
    name: "Kemö-Gïrïbïngï",
    code: "CF-KG",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-KG",
};


pub const CF_LB: Subdivision = Subdivision {
    name: "Lobaye",
    code: "CF-LB",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-LB",
};


pub const CF_MB: Subdivision = Subdivision {
    name: "Mbomou",
    code: "CF-MB",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-MB",
};


pub const CF_MP: Subdivision = Subdivision {
    name: "Ömbëlä-Pökö",
    code: "CF-MP",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-MP",
};


pub const CF_NM: Subdivision = Subdivision {
    name: "Nana-Mambéré",
    code: "CF-NM",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-NM",
};


pub const CF_OP: Subdivision = Subdivision {
    name: "Ouham-Pendé",
    code: "CF-OP",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-OP",
};


pub const CF_SE: Subdivision = Subdivision {
    name: "Sangä",
    code: "CF-SE",
    subdivision_type: "Economic prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-SE",
};


pub const CF_UK: Subdivision = Subdivision {
    name: "Ouaka",
    code: "CF-UK",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-UK",
};


pub const CF_VK: Subdivision = Subdivision {
    name: "Vakaga",
    code: "CF-VK",
    subdivision_type: "Prefecture",
    country_name: "Central African Republic",
    country_code: "CF",
    region_code: "CF-VK",
};


pub const TD_BA: Subdivision = Subdivision {
    name: "Al Baţḩah",
    code: "TD-BA",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-BA",
};


pub const TD_BG: Subdivision = Subdivision {
    name: "Baḩr al Ghazāl",
    code: "TD-BG",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-BG",
};


pub const TD_BO: Subdivision = Subdivision {
    name: "Borkou",
    code: "TD-BO",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-BO",
};


pub const TD_CB: Subdivision = Subdivision {
    name: "Chari-Baguirmi",
    code: "TD-CB",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-CB",
};


pub const TD_EE: Subdivision = Subdivision {
    name: "Ennedi-Est",
    code: "TD-EE",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-EE",
};


pub const TD_EO: Subdivision = Subdivision {
    name: "Ennedi-Ouest",
    code: "TD-EO",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-EO",
};


pub const TD_GR: Subdivision = Subdivision {
    name: "Guéra",
    code: "TD-GR",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-GR",
};


pub const TD_HL: Subdivision = Subdivision {
    name: "Hadjer Lamis",
    code: "TD-HL",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-HL",
};


pub const TD_KA: Subdivision = Subdivision {
    name: "Kanem",
    code: "TD-KA",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-KA",
};


pub const TD_LC: Subdivision = Subdivision {
    name: "Al Buḩayrah",
    code: "TD-LC",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-LC",
};


pub const TD_LO: Subdivision = Subdivision {
    name: "Logone-Occidental",
    code: "TD-LO",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-LO",
};


pub const TD_LR: Subdivision = Subdivision {
    name: "Logone-Oriental",
    code: "TD-LR",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-LR",
};


pub const TD_MA: Subdivision = Subdivision {
    name: "Mandoul",
    code: "TD-MA",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-MA",
};


pub const TD_MC: Subdivision = Subdivision {
    name: "Moyen-Chari",
    code: "TD-MC",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-MC",
};


pub const TD_ME: Subdivision = Subdivision {
    name: "Mayo-Kebbi-Est",
    code: "TD-ME",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-ME",
};


pub const TD_MO: Subdivision = Subdivision {
    name: "Mayo-Kebbi-Ouest",
    code: "TD-MO",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-MO",
};


pub const TD_ND: Subdivision = Subdivision {
    name: "Madīnat Injamīnā",
    code: "TD-ND",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-ND",
};


pub const TD_OD: Subdivision = Subdivision {
    name: "Ouaddaï",
    code: "TD-OD",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-OD",
};


pub const TD_SA: Subdivision = Subdivision {
    name: "Salamat",
    code: "TD-SA",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-SA",
};


pub const TD_SI: Subdivision = Subdivision {
    name: "Sila",
    code: "TD-SI",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-SI",
};


pub const TD_TA: Subdivision = Subdivision {
    name: "Tandjilé",
    code: "TD-TA",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-TA",
};


pub const TD_TI: Subdivision = Subdivision {
    name: "Tibastī",
    code: "TD-TI",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-TI",
};


pub const TD_WF: Subdivision = Subdivision {
    name: "Wadi Fira",
    code: "TD-WF",
    subdivision_type: "Region",
    country_name: "Chad",
    country_code: "TD",
    region_code: "TD-WF",
};


pub const CL_AI: Subdivision = Subdivision {
    name: "Aysén",
    code: "CL-AI",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-AI",
};


pub const CL_AN: Subdivision = Subdivision {
    name: "Antofagasta",
    code: "CL-AN",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-AN",
};


pub const CL_AP: Subdivision = Subdivision {
    name: "Arica y Parinacota",
    code: "CL-AP",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-AP",
};


pub const CL_AR: Subdivision = Subdivision {
    name: "Araucanía",
    code: "CL-AR",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-AR",
};


pub const CL_AT: Subdivision = Subdivision {
    name: "Atacama",
    code: "CL-AT",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-AT",
};


pub const CL_BI: Subdivision = Subdivision {
    name: "Biobío",
    code: "CL-BI",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-BI",
};


pub const CL_CO: Subdivision = Subdivision {
    name: "Coquimbo",
    code: "CL-CO",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-CO",
};


pub const CL_LI: Subdivision = Subdivision {
    name: "Libertador General Bernardo O'Higgins",
    code: "CL-LI",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-LI",
};


pub const CL_LL: Subdivision = Subdivision {
    name: "Los Lagos",
    code: "CL-LL",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-LL",
};


pub const CL_LR: Subdivision = Subdivision {
    name: "Los Ríos",
    code: "CL-LR",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-LR",
};


pub const CL_MA: Subdivision = Subdivision {
    name: "Magallanes",
    code: "CL-MA",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-MA",
};


pub const CL_ML: Subdivision = Subdivision {
    name: "Maule",
    code: "CL-ML",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-ML",
};


pub const CL_RM: Subdivision = Subdivision {
    name: "Región Metropolitana de Santiago",
    code: "CL-RM",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-RM",
};


pub const CL_TA: Subdivision = Subdivision {
    name: "Tarapacá",
    code: "CL-TA",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-TA",
};


pub const CL_VS: Subdivision = Subdivision {
    name: "Valparaíso",
    code: "CL-VS",
    subdivision_type: "Region",
    country_name: "Chile",
    country_code: "CL",
    region_code: "CL-VS",
};


pub const CN_11: Subdivision = Subdivision {
    name: "Beijing",
    code: "CN-11",
    subdivision_type: "Municipality",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-11",
};


pub const CN_12: Subdivision = Subdivision {
    name: "Tianjin",
    code: "CN-12",
    subdivision_type: "Municipality",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-12",
};


pub const CN_13: Subdivision = Subdivision {
    name: "Hebei",
    code: "CN-13",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-13",
};


pub const CN_14: Subdivision = Subdivision {
    name: "Shanxi",
    code: "CN-14",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-14",
};


pub const CN_15: Subdivision = Subdivision {
    name: "Nei Mongol",
    code: "CN-15",
    subdivision_type: "Autonomous region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-15",
};


pub const CN_21: Subdivision = Subdivision {
    name: "Liaoning",
    code: "CN-21",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-21",
};


pub const CN_22: Subdivision = Subdivision {
    name: "Jilin",
    code: "CN-22",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-22",
};


pub const CN_23: Subdivision = Subdivision {
    name: "Heilongjiang",
    code: "CN-23",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-23",
};


pub const CN_31: Subdivision = Subdivision {
    name: "Shanghai",
    code: "CN-31",
    subdivision_type: "Municipality",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-31",
};


pub const CN_32: Subdivision = Subdivision {
    name: "Jiangsu",
    code: "CN-32",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-32",
};


pub const CN_33: Subdivision = Subdivision {
    name: "Zhejiang",
    code: "CN-33",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-33",
};


pub const CN_34: Subdivision = Subdivision {
    name: "Anhui",
    code: "CN-34",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-34",
};


pub const CN_35: Subdivision = Subdivision {
    name: "Fujian",
    code: "CN-35",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-35",
};


pub const CN_36: Subdivision = Subdivision {
    name: "Jiangxi",
    code: "CN-36",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-36",
};


pub const CN_37: Subdivision = Subdivision {
    name: "Shandong",
    code: "CN-37",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-37",
};


pub const CN_41: Subdivision = Subdivision {
    name: "Henan",
    code: "CN-41",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-41",
};


pub const CN_42: Subdivision = Subdivision {
    name: "Hubei",
    code: "CN-42",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-42",
};


pub const CN_43: Subdivision = Subdivision {
    name: "Hunan",
    code: "CN-43",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-43",
};


pub const CN_44: Subdivision = Subdivision {
    name: "Guangdong",
    code: "CN-44",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-44",
};


pub const CN_45: Subdivision = Subdivision {
    name: "Guangxi",
    code: "CN-45",
    subdivision_type: "Autonomous region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-45",
};


pub const CN_46: Subdivision = Subdivision {
    name: "Hainan",
    code: "CN-46",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-46",
};


pub const CN_50: Subdivision = Subdivision {
    name: "Chongqing",
    code: "CN-50",
    subdivision_type: "Municipality",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-50",
};


pub const CN_51: Subdivision = Subdivision {
    name: "Sichuan",
    code: "CN-51",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-51",
};


pub const CN_52: Subdivision = Subdivision {
    name: "Guizhou",
    code: "CN-52",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-52",
};


pub const CN_53: Subdivision = Subdivision {
    name: "Yunnan",
    code: "CN-53",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-53",
};


pub const CN_54: Subdivision = Subdivision {
    name: "Xizang",
    code: "CN-54",
    subdivision_type: "Autonomous region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-54",
};


pub const CN_61: Subdivision = Subdivision {
    name: "Shaanxi",
    code: "CN-61",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-61",
};


pub const CN_62: Subdivision = Subdivision {
    name: "Gansu",
    code: "CN-62",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-62",
};


pub const CN_63: Subdivision = Subdivision {
    name: "Qinghai",
    code: "CN-63",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-63",
};


pub const CN_64: Subdivision = Subdivision {
    name: "Ningxia",
    code: "CN-64",
    subdivision_type: "Autonomous region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-64",
};


pub const CN_65: Subdivision = Subdivision {
    name: "Xinjiang",
    code: "CN-65",
    subdivision_type: "Autonomous region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-65",
};


pub const CN_71: Subdivision = Subdivision {
    name: "Taiwan",
    code: "CN-71",
    subdivision_type: "Province",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-71",
};


pub const CN_91: Subdivision = Subdivision {
    name: "Hong Kong",
    code: "CN-91",
    subdivision_type: "Special administrative region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-91",
};


pub const CN_92: Subdivision = Subdivision {
    name: "Aomen",
    code: "CN-92",
    subdivision_type: "Special administrative region",
    country_name: "China",
    country_code: "CN",
    region_code: "CN-92",
};


pub const CX_CX: Subdivision = Subdivision {
    name: "Christmas Island",
    code: "CX-CX",
    subdivision_type: "Country",
    country_name: "Christmas Island",
    country_code: "CX",
    region_code: "CX-CX",
};


pub const CC_CC: Subdivision = Subdivision {
    name: "Cocos  Islands",
    code: "CC-CC",
    subdivision_type: "Country",
    country_name: "Cocos  Islands",
    country_code: "CC",
    region_code: "CC-CC",
};


pub const CO_AMA: Subdivision = Subdivision {
    name: "Amazonas",
    code: "CO-AMA",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-AMA",
};


pub const CO_ANT: Subdivision = Subdivision {
    name: "Antioquia",
    code: "CO-ANT",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-ANT",
};


pub const CO_ARA: Subdivision = Subdivision {
    name: "Arauca",
    code: "CO-ARA",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-ARA",
};


pub const CO_ATL: Subdivision = Subdivision {
    name: "Atlántico",
    code: "CO-ATL",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-ATL",
};


pub const CO_BOL: Subdivision = Subdivision {
    name: "Bolívar",
    code: "CO-BOL",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-BOL",
};


pub const CO_BOY: Subdivision = Subdivision {
    name: "Boyacá",
    code: "CO-BOY",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-BOY",
};


pub const CO_CAL: Subdivision = Subdivision {
    name: "Caldas",
    code: "CO-CAL",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CAL",
};


pub const CO_CAQ: Subdivision = Subdivision {
    name: "Caquetá",
    code: "CO-CAQ",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CAQ",
};


pub const CO_CAS: Subdivision = Subdivision {
    name: "Casanare",
    code: "CO-CAS",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CAS",
};


pub const CO_CAU: Subdivision = Subdivision {
    name: "Cauca",
    code: "CO-CAU",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CAU",
};


pub const CO_CES: Subdivision = Subdivision {
    name: "Cesar",
    code: "CO-CES",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CES",
};


pub const CO_CHO: Subdivision = Subdivision {
    name: "Chocó",
    code: "CO-CHO",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CHO",
};


pub const CO_COR: Subdivision = Subdivision {
    name: "Córdoba",
    code: "CO-COR",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-COR",
};


pub const CO_CUN: Subdivision = Subdivision {
    name: "Cundinamarca",
    code: "CO-CUN",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-CUN",
};


pub const CO_DC: Subdivision = Subdivision {
    name: "Distrito Capital de Bogotá",
    code: "CO-DC",
    subdivision_type: "Capital district",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-DC",
};


pub const CO_GUA: Subdivision = Subdivision {
    name: "Guainía",
    code: "CO-GUA",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-GUA",
};


pub const CO_GUV: Subdivision = Subdivision {
    name: "Guaviare",
    code: "CO-GUV",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-GUV",
};


pub const CO_HUI: Subdivision = Subdivision {
    name: "Huila",
    code: "CO-HUI",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-HUI",
};


pub const CO_LAG: Subdivision = Subdivision {
    name: "La Guajira",
    code: "CO-LAG",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-LAG",
};


pub const CO_MAG: Subdivision = Subdivision {
    name: "Magdalena",
    code: "CO-MAG",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-MAG",
};


pub const CO_MET: Subdivision = Subdivision {
    name: "Meta",
    code: "CO-MET",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-MET",
};


pub const CO_NAR: Subdivision = Subdivision {
    name: "Nariño",
    code: "CO-NAR",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-NAR",
};


pub const CO_NSA: Subdivision = Subdivision {
    name: "Norte de Santander",
    code: "CO-NSA",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-NSA",
};


pub const CO_PUT: Subdivision = Subdivision {
    name: "Putumayo",
    code: "CO-PUT",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-PUT",
};


pub const CO_QUI: Subdivision = Subdivision {
    name: "Quindío",
    code: "CO-QUI",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-QUI",
};


pub const CO_RIS: Subdivision = Subdivision {
    name: "Risaralda",
    code: "CO-RIS",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-RIS",
};


pub const CO_SAN: Subdivision = Subdivision {
    name: "Santander",
    code: "CO-SAN",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-SAN",
};


pub const CO_SAP: Subdivision = Subdivision {
    name: "San Andrés, Providencia y Santa Catalina",
    code: "CO-SAP",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-SAP",
};


pub const CO_SUC: Subdivision = Subdivision {
    name: "Sucre",
    code: "CO-SUC",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-SUC",
};


pub const CO_TOL: Subdivision = Subdivision {
    name: "Tolima",
    code: "CO-TOL",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-TOL",
};


pub const CO_VAC: Subdivision = Subdivision {
    name: "Valle del Cauca",
    code: "CO-VAC",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-VAC",
};


pub const CO_VAU: Subdivision = Subdivision {
    name: "Vaupés",
    code: "CO-VAU",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-VAU",
};


pub const CO_VID: Subdivision = Subdivision {
    name: "Vichada",
    code: "CO-VID",
    subdivision_type: "Department",
    country_name: "Colombia",
    country_code: "CO",
    region_code: "CO-VID",
};


pub const KM_A: Subdivision = Subdivision {
    name: "Andjouân",
    code: "KM-A",
    subdivision_type: "Island",
    country_name: "Comoros",
    country_code: "KM",
    region_code: "KM-A",
};


pub const KM_G: Subdivision = Subdivision {
    name: "Andjazîdja",
    code: "KM-G",
    subdivision_type: "Island",
    country_name: "Comoros",
    country_code: "KM",
    region_code: "KM-G",
};


pub const KM_M: Subdivision = Subdivision {
    name: "Mohéli",
    code: "KM-M",
    subdivision_type: "Island",
    country_name: "Comoros",
    country_code: "KM",
    region_code: "KM-M",
};


pub const CG_11: Subdivision = Subdivision {
    name: "Bouenza",
    code: "CG-11",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-11",
};


pub const CG_12: Subdivision = Subdivision {
    name: "Pool",
    code: "CG-12",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-12",
};


pub const CG_13: Subdivision = Subdivision {
    name: "Sangha",
    code: "CG-13",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-13",
};


pub const CG_14: Subdivision = Subdivision {
    name: "Plateaux",
    code: "CG-14",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-14",
};


pub const CG_15: Subdivision = Subdivision {
    name: "Cuvette-Ouest",
    code: "CG-15",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-15",
};


pub const CG_16: Subdivision = Subdivision {
    name: "Pointe-Noire",
    code: "CG-16",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-16",
};


pub const CG_2: Subdivision = Subdivision {
    name: "Lékoumou",
    code: "CG-2",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-2",
};


pub const CG_5: Subdivision = Subdivision {
    name: "Kouilou",
    code: "CG-5",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-5",
};


pub const CG_7: Subdivision = Subdivision {
    name: "Likouala",
    code: "CG-7",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-7",
};


pub const CG_8: Subdivision = Subdivision {
    name: "Cuvette",
    code: "CG-8",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-8",
};


pub const CG_9: Subdivision = Subdivision {
    name: "Niari",
    code: "CG-9",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-9",
};


pub const CG_BZV: Subdivision = Subdivision {
    name: "Brazzaville",
    code: "CG-BZV",
    subdivision_type: "Department",
    country_name: "Congo",
    country_code: "CG",
    region_code: "CG-BZV",
};


pub const CD_BC: Subdivision = Subdivision {
    name: "Bas-Congo",
    code: "CD-BC",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-BC",
};


pub const CD_BN: Subdivision = Subdivision {
    name: "Bandundu",
    code: "CD-BN",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-BN",
};


pub const CD_EQ: Subdivision = Subdivision {
    name: "Équateur",
    code: "CD-EQ",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-EQ",
};


pub const CD_KA: Subdivision = Subdivision {
    name: "Katanga",
    code: "CD-KA",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-KA",
};


pub const CD_KE: Subdivision = Subdivision {
    name: "Kasai-Oriental",
    code: "CD-KE",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-KE",
};


pub const CD_KN: Subdivision = Subdivision {
    name: "Kinshasa",
    code: "CD-KN",
    subdivision_type: "City",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-KN",
};


pub const CD_KW: Subdivision = Subdivision {
    name: "Kasai-Occidental",
    code: "CD-KW",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-KW",
};


pub const CD_MA: Subdivision = Subdivision {
    name: "Maniema",
    code: "CD-MA",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-MA",
};


pub const CD_NK: Subdivision = Subdivision {
    name: "Nord-Kivu",
    code: "CD-NK",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-NK",
};


pub const CD_OR: Subdivision = Subdivision {
    name: "Orientale",
    code: "CD-OR",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-OR",
};


pub const CD_SK: Subdivision = Subdivision {
    name: "Sud-Kivu",
    code: "CD-SK",
    subdivision_type: "Province",
    country_name: "Congo, The Democratic Republic Of The",
    country_code: "CD",
    region_code: "CD-SK",
};


pub const CK_CK: Subdivision = Subdivision {
    name: "Cook Islands",
    code: "CK-CK",
    subdivision_type: "Country",
    country_name: "Cook Islands",
    country_code: "CK",
    region_code: "CK-CK",
};


pub const CR_A: Subdivision = Subdivision {
    name: "Alajuela",
    code: "CR-A",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-A",
};


pub const CR_C: Subdivision = Subdivision {
    name: "Cartago",
    code: "CR-C",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-C",
};


pub const CR_G: Subdivision = Subdivision {
    name: "Guanacaste",
    code: "CR-G",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-G",
};


pub const CR_H: Subdivision = Subdivision {
    name: "Heredia",
    code: "CR-H",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-H",
};


pub const CR_L: Subdivision = Subdivision {
    name: "Limón",
    code: "CR-L",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-L",
};


pub const CR_P: Subdivision = Subdivision {
    name: "Puntarenas",
    code: "CR-P",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-P",
};


pub const CR_SJ: Subdivision = Subdivision {
    name: "San José",
    code: "CR-SJ",
    subdivision_type: "Province",
    country_name: "Costa Rica",
    country_code: "CR",
    region_code: "CR-SJ",
};


pub const CI_01: Subdivision = Subdivision {
    name: "Lagunes",
    code: "CI-01",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-01",
};


pub const CI_02: Subdivision = Subdivision {
    name: "Haut-Sassandra",
    code: "CI-02",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-02",
};


pub const CI_03: Subdivision = Subdivision {
    name: "Savanes",
    code: "CI-03",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-03",
};


pub const CI_04: Subdivision = Subdivision {
    name: "Vallée du Bandama",
    code: "CI-04",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-04",
};


pub const CI_05: Subdivision = Subdivision {
    name: "Moyen-Comoé",
    code: "CI-05",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-05",
};


pub const CI_06: Subdivision = Subdivision {
    name: "18 Montagnes",
    code: "CI-06",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-06",
};


pub const CI_07: Subdivision = Subdivision {
    name: "Lacs",
    code: "CI-07",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-07",
};


pub const CI_08: Subdivision = Subdivision {
    name: "Zanzan",
    code: "CI-08",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-08",
};


pub const CI_09: Subdivision = Subdivision {
    name: "Bas-Sassandra",
    code: "CI-09",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-09",
};


pub const CI_10: Subdivision = Subdivision {
    name: "Denguélé",
    code: "CI-10",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-10",
};


pub const CI_11: Subdivision = Subdivision {
    name: "Nzi-Comoé",
    code: "CI-11",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-11",
};


pub const CI_12: Subdivision = Subdivision {
    name: "Marahoué",
    code: "CI-12",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-12",
};


pub const CI_13: Subdivision = Subdivision {
    name: "Sud-Comoé",
    code: "CI-13",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-13",
};


pub const CI_14: Subdivision = Subdivision {
    name: "Worodougou",
    code: "CI-14",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-14",
};


pub const CI_15: Subdivision = Subdivision {
    name: "Sud-Bandama",
    code: "CI-15",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-15",
};


pub const CI_16: Subdivision = Subdivision {
    name: "Agnébi",
    code: "CI-16",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-16",
};


pub const CI_17: Subdivision = Subdivision {
    name: "Bafing",
    code: "CI-17",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-17",
};


pub const CI_18: Subdivision = Subdivision {
    name: "Fromager",
    code: "CI-18",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-18",
};


pub const CI_19: Subdivision = Subdivision {
    name: "Moyen-Cavally",
    code: "CI-19",
    subdivision_type: "Region",
    country_name: "Côte D'Ivoire",
    country_code: "CI",
    region_code: "CI-19",
};


pub const HR_01: Subdivision = Subdivision {
    name: "Zagrebačka županija",
    code: "HR-01",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-01",
};


pub const HR_02: Subdivision = Subdivision {
    name: "Krapinsko-zagorska županija",
    code: "HR-02",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-02",
};


pub const HR_03: Subdivision = Subdivision {
    name: "Sisačko-moslavačka županija",
    code: "HR-03",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-03",
};


pub const HR_04: Subdivision = Subdivision {
    name: "Karlovačka županija",
    code: "HR-04",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-04",
};


pub const HR_05: Subdivision = Subdivision {
    name: "Varaždinska županija",
    code: "HR-05",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-05",
};


pub const HR_06: Subdivision = Subdivision {
    name: "Koprivničko-križevačka županija",
    code: "HR-06",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-06",
};


pub const HR_07: Subdivision = Subdivision {
    name: "Bjelovarsko-bilogorska županija",
    code: "HR-07",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-07",
};


pub const HR_08: Subdivision = Subdivision {
    name: "Primorsko-goranska županija",
    code: "HR-08",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-08",
};


pub const HR_09: Subdivision = Subdivision {
    name: "Ličko-senjska županija",
    code: "HR-09",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-09",
};


pub const HR_10: Subdivision = Subdivision {
    name: "Virovitičko-podravska županija",
    code: "HR-10",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-10",
};


pub const HR_11: Subdivision = Subdivision {
    name: "Požeško-slavonska županija",
    code: "HR-11",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-11",
};


pub const HR_12: Subdivision = Subdivision {
    name: "Brodsko-posavska županija",
    code: "HR-12",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-12",
};


pub const HR_13: Subdivision = Subdivision {
    name: "Zadarska županija",
    code: "HR-13",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-13",
};


pub const HR_14: Subdivision = Subdivision {
    name: "Osječko-baranjska županija",
    code: "HR-14",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-14",
};


pub const HR_15: Subdivision = Subdivision {
    name: "Šibensko-kninska županija",
    code: "HR-15",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-15",
};


pub const HR_16: Subdivision = Subdivision {
    name: "Vukovarsko-srijemska županija",
    code: "HR-16",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-16",
};


pub const HR_17: Subdivision = Subdivision {
    name: "Splitsko-dalmatinska županija",
    code: "HR-17",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-17",
};


pub const HR_18: Subdivision = Subdivision {
    name: "Istarska županija",
    code: "HR-18",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-18",
};


pub const HR_19: Subdivision = Subdivision {
    name: "Dubrovačko-neretvanska županija",
    code: "HR-19",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-19",
};


pub const HR_20: Subdivision = Subdivision {
    name: "Međimurska županija",
    code: "HR-20",
    subdivision_type: "County",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-20",
};


pub const HR_21: Subdivision = Subdivision {
    name: "Grad Zagreb",
    code: "HR-21",
    subdivision_type: "City",
    country_name: "Croatia",
    country_code: "HR",
    region_code: "HR-21",
};


pub const CU_01: Subdivision = Subdivision {
    name: "Pinar del Río",
    code: "CU-01",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-01",
};


pub const CU_03: Subdivision = Subdivision {
    name: "La Habana",
    code: "CU-03",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-03",
};


pub const CU_04: Subdivision = Subdivision {
    name: "Matanzas",
    code: "CU-04",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-04",
};


pub const CU_05: Subdivision = Subdivision {
    name: "Villa Clara",
    code: "CU-05",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-05",
};


pub const CU_06: Subdivision = Subdivision {
    name: "Cienfuegos",
    code: "CU-06",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-06",
};


pub const CU_07: Subdivision = Subdivision {
    name: "Sancti Spíritus",
    code: "CU-07",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-07",
};


pub const CU_08: Subdivision = Subdivision {
    name: "Ciego de Ávila",
    code: "CU-08",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-08",
};


pub const CU_09: Subdivision = Subdivision {
    name: "Camagüey",
    code: "CU-09",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-09",
};


pub const CU_10: Subdivision = Subdivision {
    name: "Las Tunas",
    code: "CU-10",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-10",
};


pub const CU_11: Subdivision = Subdivision {
    name: "Holguín",
    code: "CU-11",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-11",
};


pub const CU_12: Subdivision = Subdivision {
    name: "Granma",
    code: "CU-12",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-12",
};


pub const CU_13: Subdivision = Subdivision {
    name: "Santiago de Cuba",
    code: "CU-13",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-13",
};


pub const CU_14: Subdivision = Subdivision {
    name: "Guantánamo",
    code: "CU-14",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-14",
};


pub const CU_15: Subdivision = Subdivision {
    name: "Artemisa",
    code: "CU-15",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-15",
};


pub const CU_16: Subdivision = Subdivision {
    name: "Mayabeque",
    code: "CU-16",
    subdivision_type: "Province",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-16",
};


pub const CU_99: Subdivision = Subdivision {
    name: "Isla de la Juventud",
    code: "CU-99",
    subdivision_type: "Special municipality",
    country_name: "Cuba",
    country_code: "CU",
    region_code: "CU-99",
};


pub const CW_CW: Subdivision = Subdivision {
    name: "Curaçao",
    code: "CW-CW",
    subdivision_type: "Country",
    country_name: "Curaçao",
    country_code: "CW",
    region_code: "CW-CW",
};


pub const CY_01: Subdivision = Subdivision {
    name: "Lefkoşa",
    code: "CY-01",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-01",
};


pub const CY_02: Subdivision = Subdivision {
    name: "Lemesos",
    code: "CY-02",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-02",
};


pub const CY_03: Subdivision = Subdivision {
    name: "Larnaka",
    code: "CY-03",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-03",
};


pub const CY_04: Subdivision = Subdivision {
    name: "Ammochostos",
    code: "CY-04",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-04",
};


pub const CY_05: Subdivision = Subdivision {
    name: "Baf",
    code: "CY-05",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-05",
};


pub const CY_06: Subdivision = Subdivision {
    name: "Girne",
    code: "CY-06",
    subdivision_type: "District",
    country_name: "Cyprus",
    country_code: "CY",
    region_code: "CY-06",
};


pub const CZ_101: Subdivision = Subdivision {
    name: "Praha 1",
    code: "CZ-101",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-101",
};


pub const CZ_102: Subdivision = Subdivision {
    name: "Praha 2",
    code: "CZ-102",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-102",
};


pub const CZ_103: Subdivision = Subdivision {
    name: "Praha 3",
    code: "CZ-103",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-103",
};


pub const CZ_104: Subdivision = Subdivision {
    name: "Praha 4",
    code: "CZ-104",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-104",
};


pub const CZ_105: Subdivision = Subdivision {
    name: "Praha 5",
    code: "CZ-105",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-105",
};


pub const CZ_106: Subdivision = Subdivision {
    name: "Praha 6",
    code: "CZ-106",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-106",
};


pub const CZ_107: Subdivision = Subdivision {
    name: "Praha 7",
    code: "CZ-107",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-107",
};


pub const CZ_108: Subdivision = Subdivision {
    name: "Praha 8",
    code: "CZ-108",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-108",
};


pub const CZ_109: Subdivision = Subdivision {
    name: "Praha 9",
    code: "CZ-109",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-109",
};


pub const CZ_10A: Subdivision = Subdivision {
    name: "Praha 10",
    code: "CZ-10A",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10A",
};


pub const CZ_10B: Subdivision = Subdivision {
    name: "Praha 11",
    code: "CZ-10B",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10B",
};


pub const CZ_10C: Subdivision = Subdivision {
    name: "Praha 12",
    code: "CZ-10C",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10C",
};


pub const CZ_10D: Subdivision = Subdivision {
    name: "Praha 13",
    code: "CZ-10D",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10D",
};


pub const CZ_10E: Subdivision = Subdivision {
    name: "Praha 14",
    code: "CZ-10E",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10E",
};


pub const CZ_10F: Subdivision = Subdivision {
    name: "Praha 15",
    code: "CZ-10F",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-10F",
};


pub const CZ_201: Subdivision = Subdivision {
    name: "Benešov",
    code: "CZ-201",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-201",
};


pub const CZ_202: Subdivision = Subdivision {
    name: "Beroun",
    code: "CZ-202",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-202",
};


pub const CZ_203: Subdivision = Subdivision {
    name: "Kladno",
    code: "CZ-203",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-203",
};


pub const CZ_204: Subdivision = Subdivision {
    name: "Kolín",
    code: "CZ-204",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-204",
};


pub const CZ_205: Subdivision = Subdivision {
    name: "Kutná Hora",
    code: "CZ-205",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-205",
};


pub const CZ_206: Subdivision = Subdivision {
    name: "Mělník",
    code: "CZ-206",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-206",
};


pub const CZ_207: Subdivision = Subdivision {
    name: "Mladá Boleslav",
    code: "CZ-207",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-207",
};


pub const CZ_208: Subdivision = Subdivision {
    name: "Nymburk",
    code: "CZ-208",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-208",
};


pub const CZ_209: Subdivision = Subdivision {
    name: "Praha-východ",
    code: "CZ-209",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-209",
};


pub const CZ_20A: Subdivision = Subdivision {
    name: "Praha-západ",
    code: "CZ-20A",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-20A",
};


pub const CZ_20B: Subdivision = Subdivision {
    name: "Příbram",
    code: "CZ-20B",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-20B",
};


pub const CZ_20C: Subdivision = Subdivision {
    name: "Rakovník",
    code: "CZ-20C",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-20C",
};


pub const CZ_311: Subdivision = Subdivision {
    name: "České Budějovice",
    code: "CZ-311",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-311",
};


pub const CZ_312: Subdivision = Subdivision {
    name: "Český Krumlov",
    code: "CZ-312",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-312",
};


pub const CZ_313: Subdivision = Subdivision {
    name: "Jindřichův Hradec",
    code: "CZ-313",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-313",
};


pub const CZ_314: Subdivision = Subdivision {
    name: "Písek",
    code: "CZ-314",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-314",
};


pub const CZ_315: Subdivision = Subdivision {
    name: "Prachatice",
    code: "CZ-315",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-315",
};


pub const CZ_316: Subdivision = Subdivision {
    name: "Strakonice",
    code: "CZ-316",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-316",
};


pub const CZ_317: Subdivision = Subdivision {
    name: "Tábor",
    code: "CZ-317",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-317",
};


pub const CZ_321: Subdivision = Subdivision {
    name: "Domažlice",
    code: "CZ-321",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-321",
};


pub const CZ_322: Subdivision = Subdivision {
    name: "Klatovy",
    code: "CZ-322",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-322",
};


pub const CZ_323: Subdivision = Subdivision {
    name: "Plzeň-město",
    code: "CZ-323",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-323",
};


pub const CZ_324: Subdivision = Subdivision {
    name: "Plzeň-jih",
    code: "CZ-324",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-324",
};


pub const CZ_325: Subdivision = Subdivision {
    name: "Plzeň-sever",
    code: "CZ-325",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-325",
};


pub const CZ_326: Subdivision = Subdivision {
    name: "Rokycany",
    code: "CZ-326",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-326",
};


pub const CZ_327: Subdivision = Subdivision {
    name: "Tachov",
    code: "CZ-327",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-327",
};


pub const CZ_411: Subdivision = Subdivision {
    name: "Cheb",
    code: "CZ-411",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-411",
};


pub const CZ_412: Subdivision = Subdivision {
    name: "Karlovy Vary",
    code: "CZ-412",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-412",
};


pub const CZ_413: Subdivision = Subdivision {
    name: "Sokolov",
    code: "CZ-413",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-413",
};


pub const CZ_421: Subdivision = Subdivision {
    name: "Děčín",
    code: "CZ-421",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-421",
};


pub const CZ_422: Subdivision = Subdivision {
    name: "Chomutov",
    code: "CZ-422",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-422",
};


pub const CZ_423: Subdivision = Subdivision {
    name: "Litoměřice",
    code: "CZ-423",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-423",
};


pub const CZ_424: Subdivision = Subdivision {
    name: "Louny",
    code: "CZ-424",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-424",
};


pub const CZ_425: Subdivision = Subdivision {
    name: "Most",
    code: "CZ-425",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-425",
};


pub const CZ_426: Subdivision = Subdivision {
    name: "Teplice",
    code: "CZ-426",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-426",
};


pub const CZ_427: Subdivision = Subdivision {
    name: "Ústí nad Labem",
    code: "CZ-427",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-427",
};


pub const CZ_511: Subdivision = Subdivision {
    name: "Česká Lípa",
    code: "CZ-511",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-511",
};


pub const CZ_512: Subdivision = Subdivision {
    name: "Jablonec nad Nisou",
    code: "CZ-512",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-512",
};


pub const CZ_513: Subdivision = Subdivision {
    name: "Liberec",
    code: "CZ-513",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-513",
};


pub const CZ_514: Subdivision = Subdivision {
    name: "Semily",
    code: "CZ-514",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-514",
};


pub const CZ_521: Subdivision = Subdivision {
    name: "Hradec Králové",
    code: "CZ-521",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-521",
};


pub const CZ_522: Subdivision = Subdivision {
    name: "Jičín",
    code: "CZ-522",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-522",
};


pub const CZ_523: Subdivision = Subdivision {
    name: "Náchod",
    code: "CZ-523",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-523",
};


pub const CZ_524: Subdivision = Subdivision {
    name: "Rychnov nad Kněžnou",
    code: "CZ-524",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-524",
};


pub const CZ_525: Subdivision = Subdivision {
    name: "Trutnov",
    code: "CZ-525",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-525",
};


pub const CZ_531: Subdivision = Subdivision {
    name: "Chrudim",
    code: "CZ-531",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-531",
};


pub const CZ_532: Subdivision = Subdivision {
    name: "Pardubice",
    code: "CZ-532",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-532",
};


pub const CZ_533: Subdivision = Subdivision {
    name: "Svitavy",
    code: "CZ-533",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-533",
};


pub const CZ_534: Subdivision = Subdivision {
    name: "Ústí nad Orlicí",
    code: "CZ-534",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-534",
};


pub const CZ_611: Subdivision = Subdivision {
    name: "Havlíčkův Brod",
    code: "CZ-611",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-611",
};


pub const CZ_612: Subdivision = Subdivision {
    name: "Jihlava",
    code: "CZ-612",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-612",
};


pub const CZ_613: Subdivision = Subdivision {
    name: "Pelhřimov",
    code: "CZ-613",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-613",
};


pub const CZ_614: Subdivision = Subdivision {
    name: "Třebíč",
    code: "CZ-614",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-614",
};


pub const CZ_615: Subdivision = Subdivision {
    name: "Žd'ár nad Sázavou",
    code: "CZ-615",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-615",
};


pub const CZ_621: Subdivision = Subdivision {
    name: "Blansko",
    code: "CZ-621",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-621",
};


pub const CZ_622: Subdivision = Subdivision {
    name: "Brno-město",
    code: "CZ-622",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-622",
};


pub const CZ_623: Subdivision = Subdivision {
    name: "Brno-venkov",
    code: "CZ-623",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-623",
};


pub const CZ_624: Subdivision = Subdivision {
    name: "Břeclav",
    code: "CZ-624",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-624",
};


pub const CZ_625: Subdivision = Subdivision {
    name: "Hodonín",
    code: "CZ-625",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-625",
};


pub const CZ_626: Subdivision = Subdivision {
    name: "Vyškov",
    code: "CZ-626",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-626",
};


pub const CZ_627: Subdivision = Subdivision {
    name: "Znojmo",
    code: "CZ-627",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-627",
};


pub const CZ_711: Subdivision = Subdivision {
    name: "Jeseník",
    code: "CZ-711",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-711",
};


pub const CZ_712: Subdivision = Subdivision {
    name: "Olomouc",
    code: "CZ-712",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-712",
};


pub const CZ_713: Subdivision = Subdivision {
    name: "Prostĕjov",
    code: "CZ-713",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-713",
};


pub const CZ_714: Subdivision = Subdivision {
    name: "Přerov",
    code: "CZ-714",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-714",
};


pub const CZ_715: Subdivision = Subdivision {
    name: "Šumperk",
    code: "CZ-715",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-715",
};


pub const CZ_721: Subdivision = Subdivision {
    name: "Kromĕříž",
    code: "CZ-721",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-721",
};


pub const CZ_722: Subdivision = Subdivision {
    name: "Uherské Hradištĕ",
    code: "CZ-722",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-722",
};


pub const CZ_723: Subdivision = Subdivision {
    name: "Vsetín",
    code: "CZ-723",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-723",
};


pub const CZ_724: Subdivision = Subdivision {
    name: "Zlín",
    code: "CZ-724",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-724",
};


pub const CZ_801: Subdivision = Subdivision {
    name: "Bruntál",
    code: "CZ-801",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-801",
};


pub const CZ_802: Subdivision = Subdivision {
    name: "Frýdek Místek",
    code: "CZ-802",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-802",
};


pub const CZ_803: Subdivision = Subdivision {
    name: "Karviná",
    code: "CZ-803",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-803",
};


pub const CZ_804: Subdivision = Subdivision {
    name: "Nový Jičín",
    code: "CZ-804",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-804",
};


pub const CZ_805: Subdivision = Subdivision {
    name: "Opava",
    code: "CZ-805",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-805",
};


pub const CZ_806: Subdivision = Subdivision {
    name: "Ostrava město",
    code: "CZ-806",
    subdivision_type: "District",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-806",
};


pub const CZ_JC: Subdivision = Subdivision {
    name: "Jihočeský kraj",
    code: "CZ-JC",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-JC",
};


pub const CZ_JM: Subdivision = Subdivision {
    name: "Jihomoravský kraj",
    code: "CZ-JM",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-JM",
};


pub const CZ_KA: Subdivision = Subdivision {
    name: "Karlovarský kraj",
    code: "CZ-KA",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-KA",
};


pub const CZ_KR: Subdivision = Subdivision {
    name: "Královéhradecký kraj",
    code: "CZ-KR",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-KR",
};


pub const CZ_LI: Subdivision = Subdivision {
    name: "Liberecký kraj",
    code: "CZ-LI",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-LI",
};


pub const CZ_MO: Subdivision = Subdivision {
    name: "Moravskoslezský kraj",
    code: "CZ-MO",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-MO",
};


pub const CZ_OL: Subdivision = Subdivision {
    name: "Olomoucký kraj",
    code: "CZ-OL",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-OL",
};


pub const CZ_PA: Subdivision = Subdivision {
    name: "Pardubický kraj",
    code: "CZ-PA",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-PA",
};


pub const CZ_PL: Subdivision = Subdivision {
    name: "Plzeňský kraj",
    code: "CZ-PL",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-PL",
};


pub const CZ_PR: Subdivision = Subdivision {
    name: "Praha, hlavní mešto",
    code: "CZ-PR",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-PR",
};


pub const CZ_ST: Subdivision = Subdivision {
    name: "Středočeský kraj",
    code: "CZ-ST",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-ST",
};


pub const CZ_US: Subdivision = Subdivision {
    name: "Ústecký kraj",
    code: "CZ-US",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-US",
};


pub const CZ_VY: Subdivision = Subdivision {
    name: "Vysočina",
    code: "CZ-VY",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-VY",
};


pub const CZ_ZL: Subdivision = Subdivision {
    name: "Zlínský kraj",
    code: "CZ-ZL",
    subdivision_type: "Region",
    country_name: "Czech Republic",
    country_code: "CZ",
    region_code: "CZ-ZL",
};


pub const DK_81: Subdivision = Subdivision {
    name: "Nordjylland",
    code: "DK-81",
    subdivision_type: "Region",
    country_name: "Denmark",
    country_code: "DK",
    region_code: "DK-81",
};


pub const DK_82: Subdivision = Subdivision {
    name: "Midtjylland",
    code: "DK-82",
    subdivision_type: "Region",
    country_name: "Denmark",
    country_code: "DK",
    region_code: "DK-82",
};


pub const DK_83: Subdivision = Subdivision {
    name: "Syddanmark",
    code: "DK-83",
    subdivision_type: "Region",
    country_name: "Denmark",
    country_code: "DK",
    region_code: "DK-83",
};


pub const DK_84: Subdivision = Subdivision {
    name: "Hovedstaden",
    code: "DK-84",
    subdivision_type: "Region",
    country_name: "Denmark",
    country_code: "DK",
    region_code: "DK-84",
};


pub const DK_85: Subdivision = Subdivision {
    name: "Sjælland",
    code: "DK-85",
    subdivision_type: "Region",
    country_name: "Denmark",
    country_code: "DK",
    region_code: "DK-85",
};


pub const DJ_AR: Subdivision = Subdivision {
    name: "Arta",
    code: "DJ-AR",
    subdivision_type: "Region",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-AR",
};


pub const DJ_AS: Subdivision = Subdivision {
    name: "Ali Sabieh",
    code: "DJ-AS",
    subdivision_type: "Region",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-AS",
};


pub const DJ_DI: Subdivision = Subdivision {
    name: "Dikhīl",
    code: "DJ-DI",
    subdivision_type: "Region",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-DI",
};


pub const DJ_DJ: Subdivision = Subdivision {
    name: "Djibouti",
    code: "DJ-DJ",
    subdivision_type: "City",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-DJ",
};


pub const DJ_OB: Subdivision = Subdivision {
    name: "Obock",
    code: "DJ-OB",
    subdivision_type: "Region",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-OB",
};


pub const DJ_TA: Subdivision = Subdivision {
    name: "Tadjourah",
    code: "DJ-TA",
    subdivision_type: "Region",
    country_name: "Djibouti",
    country_code: "DJ",
    region_code: "DJ-TA",
};


pub const DM_02: Subdivision = Subdivision {
    name: "Saint Andrew",
    code: "DM-02",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-02",
};


pub const DM_03: Subdivision = Subdivision {
    name: "Saint David",
    code: "DM-03",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-03",
};


pub const DM_04: Subdivision = Subdivision {
    name: "Saint George",
    code: "DM-04",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-04",
};


pub const DM_05: Subdivision = Subdivision {
    name: "Saint John",
    code: "DM-05",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-05",
};


pub const DM_06: Subdivision = Subdivision {
    name: "Saint Joseph",
    code: "DM-06",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-06",
};


pub const DM_07: Subdivision = Subdivision {
    name: "Saint Luke",
    code: "DM-07",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-07",
};


pub const DM_08: Subdivision = Subdivision {
    name: "Saint Mark",
    code: "DM-08",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-08",
};


pub const DM_09: Subdivision = Subdivision {
    name: "Saint Patrick",
    code: "DM-09",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-09",
};


pub const DM_10: Subdivision = Subdivision {
    name: "Saint Paul",
    code: "DM-10",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-10",
};


pub const DM_11: Subdivision = Subdivision {
    name: "Saint Peter",
    code: "DM-11",
    subdivision_type: "Parish",
    country_name: "Dominica",
    country_code: "DM",
    region_code: "DM-11",
};


pub const DO_01: Subdivision = Subdivision {
    name: "Distrito Nacional",
    code: "DO-01",
    subdivision_type: "District",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-01",
};


pub const DO_02: Subdivision = Subdivision {
    name: "Azua",
    code: "DO-02",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-02",
};


pub const DO_03: Subdivision = Subdivision {
    name: "Baoruco",
    code: "DO-03",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-03",
};


pub const DO_04: Subdivision = Subdivision {
    name: "Barahona",
    code: "DO-04",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-04",
};


pub const DO_05: Subdivision = Subdivision {
    name: "Dajabón",
    code: "DO-05",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-05",
};


pub const DO_06: Subdivision = Subdivision {
    name: "Duarte",
    code: "DO-06",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-06",
};


pub const DO_07: Subdivision = Subdivision {
    name: "La Estrelleta",
    code: "DO-07",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-07",
};


pub const DO_08: Subdivision = Subdivision {
    name: "El Seibo",
    code: "DO-08",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-08",
};


pub const DO_09: Subdivision = Subdivision {
    name: "Espaillat",
    code: "DO-09",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-09",
};


pub const DO_10: Subdivision = Subdivision {
    name: "Independencia",
    code: "DO-10",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-10",
};


pub const DO_11: Subdivision = Subdivision {
    name: "La Altagracia",
    code: "DO-11",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-11",
};


pub const DO_12: Subdivision = Subdivision {
    name: "La Romana",
    code: "DO-12",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-12",
};


pub const DO_13: Subdivision = Subdivision {
    name: "La Vega",
    code: "DO-13",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-13",
};


pub const DO_14: Subdivision = Subdivision {
    name: "María Trinidad Sánchez",
    code: "DO-14",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-14",
};


pub const DO_15: Subdivision = Subdivision {
    name: "Monte Cristi",
    code: "DO-15",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-15",
};


pub const DO_16: Subdivision = Subdivision {
    name: "Pedernales",
    code: "DO-16",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-16",
};


pub const DO_17: Subdivision = Subdivision {
    name: "Peravia",
    code: "DO-17",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-17",
};


pub const DO_18: Subdivision = Subdivision {
    name: "Puerto Plata",
    code: "DO-18",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-18",
};


pub const DO_19: Subdivision = Subdivision {
    name: "Hermanas Mirabal",
    code: "DO-19",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-19",
};


pub const DO_20: Subdivision = Subdivision {
    name: "Samaná",
    code: "DO-20",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-20",
};


pub const DO_21: Subdivision = Subdivision {
    name: "San Cristóbal",
    code: "DO-21",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-21",
};


pub const DO_22: Subdivision = Subdivision {
    name: "San Juan",
    code: "DO-22",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-22",
};


pub const DO_23: Subdivision = Subdivision {
    name: "San Pedro de Macorís",
    code: "DO-23",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-23",
};


pub const DO_24: Subdivision = Subdivision {
    name: "Sánchez Ramírez",
    code: "DO-24",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-24",
};


pub const DO_25: Subdivision = Subdivision {
    name: "Santiago",
    code: "DO-25",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-25",
};


pub const DO_26: Subdivision = Subdivision {
    name: "Santiago Rodríguez",
    code: "DO-26",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-26",
};


pub const DO_27: Subdivision = Subdivision {
    name: "Valverde",
    code: "DO-27",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-27",
};


pub const DO_28: Subdivision = Subdivision {
    name: "Monseñor Nouel",
    code: "DO-28",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-28",
};


pub const DO_29: Subdivision = Subdivision {
    name: "Monte Plata",
    code: "DO-29",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-29",
};


pub const DO_30: Subdivision = Subdivision {
    name: "Hato Mayor",
    code: "DO-30",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-30",
};


pub const DO_31: Subdivision = Subdivision {
    name: "San José de Ocoa",
    code: "DO-31",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-31",
};


pub const DO_32: Subdivision = Subdivision {
    name: "Santo Domingo",
    code: "DO-32",
    subdivision_type: "Province",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-32",
};


pub const DO_33: Subdivision = Subdivision {
    name: "Cibao Nordeste",
    code: "DO-33",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-33",
};


pub const DO_34: Subdivision = Subdivision {
    name: "Cibao Noroeste",
    code: "DO-34",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-34",
};


pub const DO_35: Subdivision = Subdivision {
    name: "Cibao Norte",
    code: "DO-35",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-35",
};


pub const DO_36: Subdivision = Subdivision {
    name: "Cibao Sur",
    code: "DO-36",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-36",
};


pub const DO_37: Subdivision = Subdivision {
    name: "El Valle",
    code: "DO-37",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-37",
};


pub const DO_38: Subdivision = Subdivision {
    name: "Enriquillo",
    code: "DO-38",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-38",
};


pub const DO_39: Subdivision = Subdivision {
    name: "Higuamo",
    code: "DO-39",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-39",
};


pub const DO_40: Subdivision = Subdivision {
    name: "Ozama",
    code: "DO-40",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-40",
};


pub const DO_41: Subdivision = Subdivision {
    name: "Valdesia",
    code: "DO-41",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-41",
};


pub const DO_42: Subdivision = Subdivision {
    name: "Yuma",
    code: "DO-42",
    subdivision_type: "Region",
    country_name: "Dominican Republic",
    country_code: "DO",
    region_code: "DO-42",
};


pub const EC_A: Subdivision = Subdivision {
    name: "Azuay",
    code: "EC-A",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-A",
};


pub const EC_B: Subdivision = Subdivision {
    name: "Bolívar",
    code: "EC-B",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-B",
};


pub const EC_C: Subdivision = Subdivision {
    name: "Carchi",
    code: "EC-C",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-C",
};


pub const EC_D: Subdivision = Subdivision {
    name: "Orellana",
    code: "EC-D",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-D",
};


pub const EC_E: Subdivision = Subdivision {
    name: "Esmeraldas",
    code: "EC-E",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-E",
};


pub const EC_F: Subdivision = Subdivision {
    name: "Cañar",
    code: "EC-F",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-F",
};


pub const EC_G: Subdivision = Subdivision {
    name: "Guayas",
    code: "EC-G",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-G",
};


pub const EC_H: Subdivision = Subdivision {
    name: "Chimborazo",
    code: "EC-H",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-H",
};


pub const EC_I: Subdivision = Subdivision {
    name: "Imbabura",
    code: "EC-I",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-I",
};


pub const EC_L: Subdivision = Subdivision {
    name: "Loja",
    code: "EC-L",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-L",
};


pub const EC_M: Subdivision = Subdivision {
    name: "Manabí",
    code: "EC-M",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-M",
};


pub const EC_N: Subdivision = Subdivision {
    name: "Napo",
    code: "EC-N",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-N",
};


pub const EC_O: Subdivision = Subdivision {
    name: "El Oro",
    code: "EC-O",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-O",
};


pub const EC_P: Subdivision = Subdivision {
    name: "Pichincha",
    code: "EC-P",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-P",
};


pub const EC_R: Subdivision = Subdivision {
    name: "Los Ríos",
    code: "EC-R",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-R",
};


pub const EC_SD: Subdivision = Subdivision {
    name: "Santo Domingo de los Tsáchilas",
    code: "EC-SD",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-SD",
};


pub const EC_SE: Subdivision = Subdivision {
    name: "Santa Elena",
    code: "EC-SE",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-SE",
};


pub const EC_S: Subdivision = Subdivision {
    name: "Morona-Santiago",
    code: "EC-S",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-S",
};


pub const EC_T: Subdivision = Subdivision {
    name: "Tungurahua",
    code: "EC-T",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-T",
};


pub const EC_U: Subdivision = Subdivision {
    name: "Sucumbíos",
    code: "EC-U",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-U",
};


pub const EC_W: Subdivision = Subdivision {
    name: "Galápagos",
    code: "EC-W",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-W",
};


pub const EC_X: Subdivision = Subdivision {
    name: "Cotopaxi",
    code: "EC-X",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-X",
};


pub const EC_Y: Subdivision = Subdivision {
    name: "Pastaza",
    code: "EC-Y",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-Y",
};


pub const EC_Z: Subdivision = Subdivision {
    name: "Zamora-Chinchipe",
    code: "EC-Z",
    subdivision_type: "Province",
    country_name: "Ecuador",
    country_code: "EC",
    region_code: "EC-Z",
};


pub const EG_ALX: Subdivision = Subdivision {
    name: "Al Iskandarīyah",
    code: "EG-ALX",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-ALX",
};


pub const EG_ASN: Subdivision = Subdivision {
    name: "Aswān",
    code: "EG-ASN",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-ASN",
};


pub const EG_AST: Subdivision = Subdivision {
    name: "Asyūţ",
    code: "EG-AST",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-AST",
};


pub const EG_BA: Subdivision = Subdivision {
    name: "Al Baḩr al Aḩmar",
    code: "EG-BA",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-BA",
};


pub const EG_BH: Subdivision = Subdivision {
    name: "Al Buḩayrah",
    code: "EG-BH",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-BH",
};


pub const EG_BNS: Subdivision = Subdivision {
    name: "Banī Suwayf",
    code: "EG-BNS",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-BNS",
};


pub const EG_C: Subdivision = Subdivision {
    name: "Al Qāhirah",
    code: "EG-C",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-C",
};


pub const EG_DK: Subdivision = Subdivision {
    name: "Ad Daqahlīyah",
    code: "EG-DK",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-DK",
};


pub const EG_DT: Subdivision = Subdivision {
    name: "Dumyāţ",
    code: "EG-DT",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-DT",
};


pub const EG_FYM: Subdivision = Subdivision {
    name: "Al Fayyūm",
    code: "EG-FYM",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-FYM",
};


pub const EG_GH: Subdivision = Subdivision {
    name: "Al Gharbīyah",
    code: "EG-GH",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-GH",
};


pub const EG_GZ: Subdivision = Subdivision {
    name: "Al Jīzah",
    code: "EG-GZ",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-GZ",
};


pub const EG_IS: Subdivision = Subdivision {
    name: "Al Ismā'īlīyah",
    code: "EG-IS",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-IS",
};


pub const EG_JS: Subdivision = Subdivision {
    name: "Janūb Sīnā'",
    code: "EG-JS",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-JS",
};


pub const EG_KB: Subdivision = Subdivision {
    name: "Al Qalyūbīyah",
    code: "EG-KB",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-KB",
};


pub const EG_KFS: Subdivision = Subdivision {
    name: "Kafr ash Shaykh",
    code: "EG-KFS",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-KFS",
};


pub const EG_KN: Subdivision = Subdivision {
    name: "Qinā",
    code: "EG-KN",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-KN",
};


pub const EG_LX: Subdivision = Subdivision {
    name: "Al Uqşur",
    code: "EG-LX",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-LX",
};


pub const EG_MN: Subdivision = Subdivision {
    name: "Al Minyā",
    code: "EG-MN",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-MN",
};


pub const EG_MNF: Subdivision = Subdivision {
    name: "Al Minūfīyah",
    code: "EG-MNF",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-MNF",
};


pub const EG_MT: Subdivision = Subdivision {
    name: "Maţrūḩ",
    code: "EG-MT",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-MT",
};


pub const EG_PTS: Subdivision = Subdivision {
    name: "Būr Sa‘īd",
    code: "EG-PTS",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-PTS",
};


pub const EG_SHG: Subdivision = Subdivision {
    name: "Sūhāj",
    code: "EG-SHG",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-SHG",
};


pub const EG_SHR: Subdivision = Subdivision {
    name: "Ash Sharqīyah",
    code: "EG-SHR",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-SHR",
};


pub const EG_SIN: Subdivision = Subdivision {
    name: "Shamāl Sīnā'",
    code: "EG-SIN",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-SIN",
};


pub const EG_SUZ: Subdivision = Subdivision {
    name: "As Suways",
    code: "EG-SUZ",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-SUZ",
};


pub const EG_WAD: Subdivision = Subdivision {
    name: "Al Wādī al Jadīd",
    code: "EG-WAD",
    subdivision_type: "Governorate",
    country_name: "Egypt",
    country_code: "EG",
    region_code: "EG-WAD",
};


pub const SV_AH: Subdivision = Subdivision {
    name: "Ahuachapán",
    code: "SV-AH",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-AH",
};


pub const SV_CA: Subdivision = Subdivision {
    name: "Cabañas",
    code: "SV-CA",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-CA",
};


pub const SV_CH: Subdivision = Subdivision {
    name: "Chalatenango",
    code: "SV-CH",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-CH",
};


pub const SV_CU: Subdivision = Subdivision {
    name: "Cuscatlán",
    code: "SV-CU",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-CU",
};


pub const SV_LI: Subdivision = Subdivision {
    name: "La Libertad",
    code: "SV-LI",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-LI",
};


pub const SV_MO: Subdivision = Subdivision {
    name: "Morazán",
    code: "SV-MO",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-MO",
};


pub const SV_PA: Subdivision = Subdivision {
    name: "La Paz",
    code: "SV-PA",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-PA",
};


pub const SV_SA: Subdivision = Subdivision {
    name: "Santa Ana",
    code: "SV-SA",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-SA",
};


pub const SV_SM: Subdivision = Subdivision {
    name: "San Miguel",
    code: "SV-SM",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-SM",
};


pub const SV_SO: Subdivision = Subdivision {
    name: "Sonsonate",
    code: "SV-SO",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-SO",
};


pub const SV_SS: Subdivision = Subdivision {
    name: "San Salvador",
    code: "SV-SS",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-SS",
};


pub const SV_SV: Subdivision = Subdivision {
    name: "San Vicente",
    code: "SV-SV",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-SV",
};


pub const SV_UN: Subdivision = Subdivision {
    name: "La Unión",
    code: "SV-UN",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-UN",
};


pub const SV_US: Subdivision = Subdivision {
    name: "Usulután",
    code: "SV-US",
    subdivision_type: "Department",
    country_name: "El Salvador",
    country_code: "SV",
    region_code: "SV-US",
};


pub const GQ_AN: Subdivision = Subdivision {
    name: "Annobón",
    code: "GQ-AN",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-AN",
};


pub const GQ_BN: Subdivision = Subdivision {
    name: "Bioko Nord",
    code: "GQ-BN",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-BN",
};


pub const GQ_BS: Subdivision = Subdivision {
    name: "Bioko Sud",
    code: "GQ-BS",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-BS",
};


pub const GQ_C: Subdivision = Subdivision {
    name: "Região Continental",
    code: "GQ-C",
    subdivision_type: "Region",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-C",
};


pub const GQ_CS: Subdivision = Subdivision {
    name: "Centro Sud",
    code: "GQ-CS",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-CS",
};


pub const GQ_I: Subdivision = Subdivision {
    name: "Região Insular",
    code: "GQ-I",
    subdivision_type: "Region",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-I",
};


pub const GQ_KN: Subdivision = Subdivision {
    name: "Kié-Ntem",
    code: "GQ-KN",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-KN",
};


pub const GQ_LI: Subdivision = Subdivision {
    name: "Litoral",
    code: "GQ-LI",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-LI",
};


pub const GQ_WN: Subdivision = Subdivision {
    name: "Wele-Nzas",
    code: "GQ-WN",
    subdivision_type: "Province",
    country_name: "Equatorial Guinea",
    country_code: "GQ",
    region_code: "GQ-WN",
};


pub const ER_AN: Subdivision = Subdivision {
    name: "Ansabā",
    code: "ER-AN",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-AN",
};


pub const ER_DK: Subdivision = Subdivision {
    name: "Debubawi K’eyyĭḥ Baḥri",
    code: "ER-DK",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-DK",
};


pub const ER_DU: Subdivision = Subdivision {
    name: "Al Janūbī",
    code: "ER-DU",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-DU",
};


pub const ER_GB: Subdivision = Subdivision {
    name: "Gash-Barka",
    code: "ER-GB",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-GB",
};


pub const ER_MA: Subdivision = Subdivision {
    name: "Al Awsaţ",
    code: "ER-MA",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-MA",
};


pub const ER_SK: Subdivision = Subdivision {
    name: "Semienawi K’eyyĭḥ Baḥri",
    code: "ER-SK",
    subdivision_type: "Region",
    country_name: "Eritrea",
    country_code: "ER",
    region_code: "ER-SK",
};


pub const EE_37: Subdivision = Subdivision {
    name: "Harjumaa",
    code: "EE-37",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-37",
};


pub const EE_39: Subdivision = Subdivision {
    name: "Hiiumaa",
    code: "EE-39",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-39",
};


pub const EE_44: Subdivision = Subdivision {
    name: "Ida-Virumaa",
    code: "EE-44",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-44",
};


pub const EE_49: Subdivision = Subdivision {
    name: "Jõgevamaa",
    code: "EE-49",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-49",
};


pub const EE_51: Subdivision = Subdivision {
    name: "Järvamaa",
    code: "EE-51",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-51",
};


pub const EE_57: Subdivision = Subdivision {
    name: "Läänemaa",
    code: "EE-57",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-57",
};


pub const EE_59: Subdivision = Subdivision {
    name: "Lääne-Virumaa",
    code: "EE-59",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-59",
};


pub const EE_65: Subdivision = Subdivision {
    name: "Põlvamaa",
    code: "EE-65",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-65",
};


pub const EE_67: Subdivision = Subdivision {
    name: "Pärnumaa",
    code: "EE-67",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-67",
};


pub const EE_70: Subdivision = Subdivision {
    name: "Raplamaa",
    code: "EE-70",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-70",
};


pub const EE_74: Subdivision = Subdivision {
    name: "Saaremaa",
    code: "EE-74",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-74",
};


pub const EE_78: Subdivision = Subdivision {
    name: "Tartumaa",
    code: "EE-78",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-78",
};


pub const EE_82: Subdivision = Subdivision {
    name: "Valgamaa",
    code: "EE-82",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-82",
};


pub const EE_84: Subdivision = Subdivision {
    name: "Viljandimaa",
    code: "EE-84",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-84",
};


pub const EE_86: Subdivision = Subdivision {
    name: "Võrumaa",
    code: "EE-86",
    subdivision_type: "County",
    country_name: "Estonia",
    country_code: "EE",
    region_code: "EE-86",
};


pub const ET_AA: Subdivision = Subdivision {
    name: "Addis Ababa",
    code: "ET-AA",
    subdivision_type: "Administration",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-AA",
};


pub const ET_AF: Subdivision = Subdivision {
    name: "Afar",
    code: "ET-AF",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-AF",
};


pub const ET_AM: Subdivision = Subdivision {
    name: "Amara",
    code: "ET-AM",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-AM",
};


pub const ET_BE: Subdivision = Subdivision {
    name: "Benshangul-Gumaz",
    code: "ET-BE",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-BE",
};


pub const ET_DD: Subdivision = Subdivision {
    name: "Dire Dawa",
    code: "ET-DD",
    subdivision_type: "Administration",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-DD",
};


pub const ET_GA: Subdivision = Subdivision {
    name: "Gambēla Hizboch",
    code: "ET-GA",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-GA",
};


pub const ET_HA: Subdivision = Subdivision {
    name: "Harari People",
    code: "ET-HA",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-HA",
};


pub const ET_OR: Subdivision = Subdivision {
    name: "Oromia",
    code: "ET-OR",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-OR",
};


pub const ET_SN: Subdivision = Subdivision {
    name: "Southern Nations, Nationalities and Peoples",
    code: "ET-SN",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-SN",
};


pub const ET_SO: Subdivision = Subdivision {
    name: "Somali",
    code: "ET-SO",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-SO",
};


pub const ET_TI: Subdivision = Subdivision {
    name: "Tigrai",
    code: "ET-TI",
    subdivision_type: "State",
    country_name: "Ethiopia",
    country_code: "ET",
    region_code: "ET-TI",
};


pub const FK_FK: Subdivision = Subdivision {
    name: "Falkland Islands",
    code: "FK-FK",
    subdivision_type: "Country",
    country_name: "Falkland Islands",
    country_code: "FK",
    region_code: "FK-FK",
};


pub const FO_FO: Subdivision = Subdivision {
    name: "Faroe Islands",
    code: "FO-FO",
    subdivision_type: "Country",
    country_name: "Faroe Islands",
    country_code: "FO",
    region_code: "FO-FO",
};


pub const FJ_01: Subdivision = Subdivision {
    name: "Ba",
    code: "FJ-01",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-01",
};


pub const FJ_02: Subdivision = Subdivision {
    name: "Bua",
    code: "FJ-02",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-02",
};


pub const FJ_03: Subdivision = Subdivision {
    name: "Cakaudrove",
    code: "FJ-03",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-03",
};


pub const FJ_04: Subdivision = Subdivision {
    name: "Kadavu",
    code: "FJ-04",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-04",
};


pub const FJ_05: Subdivision = Subdivision {
    name: "Lau",
    code: "FJ-05",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-05",
};


pub const FJ_06: Subdivision = Subdivision {
    name: "Lomaiviti",
    code: "FJ-06",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-06",
};


pub const FJ_07: Subdivision = Subdivision {
    name: "Macuata",
    code: "FJ-07",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-07",
};


pub const FJ_08: Subdivision = Subdivision {
    name: "Nadroga and Navosa",
    code: "FJ-08",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-08",
};


pub const FJ_09: Subdivision = Subdivision {
    name: "Naitasiri",
    code: "FJ-09",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-09",
};


pub const FJ_10: Subdivision = Subdivision {
    name: "Namosi",
    code: "FJ-10",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-10",
};


pub const FJ_11: Subdivision = Subdivision {
    name: "Ra",
    code: "FJ-11",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-11",
};


pub const FJ_12: Subdivision = Subdivision {
    name: "Rewa",
    code: "FJ-12",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-12",
};


pub const FJ_13: Subdivision = Subdivision {
    name: "Serua",
    code: "FJ-13",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-13",
};


pub const FJ_14: Subdivision = Subdivision {
    name: "Tailevu",
    code: "FJ-14",
    subdivision_type: "Province",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-14",
};


pub const FJ_C: Subdivision = Subdivision {
    name: "Central",
    code: "FJ-C",
    subdivision_type: "Division",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-C",
};


pub const FJ_E: Subdivision = Subdivision {
    name: "Eastern",
    code: "FJ-E",
    subdivision_type: "Division",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-E",
};


pub const FJ_N: Subdivision = Subdivision {
    name: "Northern",
    code: "FJ-N",
    subdivision_type: "Division",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-N",
};


pub const FJ_R: Subdivision = Subdivision {
    name: "Rotuma",
    code: "FJ-R",
    subdivision_type: "Dependency",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-R",
};


pub const FJ_W: Subdivision = Subdivision {
    name: "Western",
    code: "FJ-W",
    subdivision_type: "Division",
    country_name: "Fiji",
    country_code: "FJ",
    region_code: "FJ-W",
};


pub const FI_01: Subdivision = Subdivision {
    name: "Ahvenanmaan maakunta",
    code: "FI-01",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-01",
};


pub const FI_02: Subdivision = Subdivision {
    name: "Etelä-Karjala",
    code: "FI-02",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-02",
};


pub const FI_03: Subdivision = Subdivision {
    name: "Etelä-Pohjanmaa",
    code: "FI-03",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-03",
};


pub const FI_04: Subdivision = Subdivision {
    name: "Etelä-Savo",
    code: "FI-04",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-04",
};


pub const FI_05: Subdivision = Subdivision {
    name: "Kainuu",
    code: "FI-05",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-05",
};


pub const FI_06: Subdivision = Subdivision {
    name: "Egentliga Tavastland",
    code: "FI-06",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-06",
};


pub const FI_07: Subdivision = Subdivision {
    name: "Keski-Pohjanmaa",
    code: "FI-07",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-07",
};


pub const FI_08: Subdivision = Subdivision {
    name: "Keski-Suomi",
    code: "FI-08",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-08",
};


pub const FI_09: Subdivision = Subdivision {
    name: "Kymenlaakso",
    code: "FI-09",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-09",
};


pub const FI_10: Subdivision = Subdivision {
    name: "Lappi",
    code: "FI-10",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-10",
};


pub const FI_11: Subdivision = Subdivision {
    name: "Birkaland",
    code: "FI-11",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-11",
};


pub const FI_12: Subdivision = Subdivision {
    name: "Österbotten",
    code: "FI-12",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-12",
};


pub const FI_13: Subdivision = Subdivision {
    name: "Norra Karelen",
    code: "FI-13",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-13",
};


pub const FI_14: Subdivision = Subdivision {
    name: "Norra Österbotten",
    code: "FI-14",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-14",
};


pub const FI_15: Subdivision = Subdivision {
    name: "Norra Savolax",
    code: "FI-15",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-15",
};


pub const FI_16: Subdivision = Subdivision {
    name: "Päijänne-Tavastland",
    code: "FI-16",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-16",
};


pub const FI_17: Subdivision = Subdivision {
    name: "Satakunda",
    code: "FI-17",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-17",
};


pub const FI_18: Subdivision = Subdivision {
    name: "Nyland",
    code: "FI-18",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-18",
};


pub const FI_19: Subdivision = Subdivision {
    name: "Egentliga Finland",
    code: "FI-19",
    subdivision_type: "Region",
    country_name: "Finland",
    country_code: "FI",
    region_code: "FI-19",
};


pub const FR_01: Subdivision = Subdivision {
    name: "Ain",
    code: "FR-01",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-01",
};


pub const FR_02: Subdivision = Subdivision {
    name: "Aisne",
    code: "FR-02",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-02",
};


pub const FR_03: Subdivision = Subdivision {
    name: "Allier",
    code: "FR-03",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-03",
};


pub const FR_04: Subdivision = Subdivision {
    name: "Alpes-de-Haute-Provence",
    code: "FR-04",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-04",
};


pub const FR_05: Subdivision = Subdivision {
    name: "Hautes-Alpes",
    code: "FR-05",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-05",
};


pub const FR_06: Subdivision = Subdivision {
    name: "Alpes-Maritimes",
    code: "FR-06",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-06",
};


pub const FR_07: Subdivision = Subdivision {
    name: "Ardèche",
    code: "FR-07",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-07",
};


pub const FR_08: Subdivision = Subdivision {
    name: "Ardennes",
    code: "FR-08",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-08",
};


pub const FR_09: Subdivision = Subdivision {
    name: "Ariège",
    code: "FR-09",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-09",
};


pub const FR_10: Subdivision = Subdivision {
    name: "Aube",
    code: "FR-10",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-10",
};


pub const FR_11: Subdivision = Subdivision {
    name: "Aude",
    code: "FR-11",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-11",
};


pub const FR_12: Subdivision = Subdivision {
    name: "Aveyron",
    code: "FR-12",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-12",
};


pub const FR_13: Subdivision = Subdivision {
    name: "Bouches-du-Rhône",
    code: "FR-13",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-13",
};


pub const FR_14: Subdivision = Subdivision {
    name: "Calvados",
    code: "FR-14",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-14",
};


pub const FR_15: Subdivision = Subdivision {
    name: "Cantal",
    code: "FR-15",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-15",
};


pub const FR_16: Subdivision = Subdivision {
    name: "Charente",
    code: "FR-16",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-16",
};


pub const FR_17: Subdivision = Subdivision {
    name: "Charente-Maritime",
    code: "FR-17",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-17",
};


pub const FR_18: Subdivision = Subdivision {
    name: "Cher",
    code: "FR-18",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-18",
};


pub const FR_19: Subdivision = Subdivision {
    name: "Corrèze",
    code: "FR-19",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-19",
};


pub const FR_21: Subdivision = Subdivision {
    name: "Côte-d'Or",
    code: "FR-21",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-21",
};


pub const FR_22: Subdivision = Subdivision {
    name: "Côtes-d'Armor",
    code: "FR-22",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-22",
};


pub const FR_23: Subdivision = Subdivision {
    name: "Creuse",
    code: "FR-23",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-23",
};


pub const FR_24: Subdivision = Subdivision {
    name: "Dordogne",
    code: "FR-24",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-24",
};


pub const FR_25: Subdivision = Subdivision {
    name: "Doubs",
    code: "FR-25",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-25",
};


pub const FR_26: Subdivision = Subdivision {
    name: "Drôme",
    code: "FR-26",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-26",
};


pub const FR_27: Subdivision = Subdivision {
    name: "Eure",
    code: "FR-27",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-27",
};


pub const FR_28: Subdivision = Subdivision {
    name: "Eure-et-Loir",
    code: "FR-28",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-28",
};


pub const FR_29: Subdivision = Subdivision {
    name: "Finistère",
    code: "FR-29",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-29",
};


pub const FR_2A: Subdivision = Subdivision {
    name: "Corse-du-Sud",
    code: "FR-2A",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-2A",
};


pub const FR_2B: Subdivision = Subdivision {
    name: "Haute-Corse",
    code: "FR-2B",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-2B",
};


pub const FR_30: Subdivision = Subdivision {
    name: "Gard",
    code: "FR-30",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-30",
};


pub const FR_31: Subdivision = Subdivision {
    name: "Haute-Garonne",
    code: "FR-31",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-31",
};


pub const FR_32: Subdivision = Subdivision {
    name: "Gers",
    code: "FR-32",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-32",
};


pub const FR_33: Subdivision = Subdivision {
    name: "Gironde",
    code: "FR-33",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-33",
};


pub const FR_34: Subdivision = Subdivision {
    name: "Hérault",
    code: "FR-34",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-34",
};


pub const FR_35: Subdivision = Subdivision {
    name: "Ille-et-Vilaine",
    code: "FR-35",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-35",
};


pub const FR_36: Subdivision = Subdivision {
    name: "Indre",
    code: "FR-36",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-36",
};


pub const FR_37: Subdivision = Subdivision {
    name: "Indre-et-Loire",
    code: "FR-37",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-37",
};


pub const FR_38: Subdivision = Subdivision {
    name: "Isère",
    code: "FR-38",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-38",
};


pub const FR_39: Subdivision = Subdivision {
    name: "Jura",
    code: "FR-39",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-39",
};


pub const FR_40: Subdivision = Subdivision {
    name: "Landes",
    code: "FR-40",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-40",
};


pub const FR_41: Subdivision = Subdivision {
    name: "Loir-et-Cher",
    code: "FR-41",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-41",
};


pub const FR_42: Subdivision = Subdivision {
    name: "Loire",
    code: "FR-42",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-42",
};


pub const FR_43: Subdivision = Subdivision {
    name: "Haute-Loire",
    code: "FR-43",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-43",
};


pub const FR_44: Subdivision = Subdivision {
    name: "Loire-Atlantique",
    code: "FR-44",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-44",
};


pub const FR_45: Subdivision = Subdivision {
    name: "Loiret",
    code: "FR-45",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-45",
};


pub const FR_46: Subdivision = Subdivision {
    name: "Lot",
    code: "FR-46",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-46",
};


pub const FR_47: Subdivision = Subdivision {
    name: "Lot-et-Garonne",
    code: "FR-47",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-47",
};


pub const FR_48: Subdivision = Subdivision {
    name: "Lozère",
    code: "FR-48",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-48",
};


pub const FR_49: Subdivision = Subdivision {
    name: "Maine-et-Loire",
    code: "FR-49",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-49",
};


pub const FR_50: Subdivision = Subdivision {
    name: "Manche",
    code: "FR-50",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-50",
};


pub const FR_51: Subdivision = Subdivision {
    name: "Marne",
    code: "FR-51",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-51",
};


pub const FR_52: Subdivision = Subdivision {
    name: "Haute-Marne",
    code: "FR-52",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-52",
};


pub const FR_53: Subdivision = Subdivision {
    name: "Mayenne",
    code: "FR-53",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-53",
};


pub const FR_54: Subdivision = Subdivision {
    name: "Meurthe-et-Moselle",
    code: "FR-54",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-54",
};


pub const FR_55: Subdivision = Subdivision {
    name: "Meuse",
    code: "FR-55",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-55",
};


pub const FR_56: Subdivision = Subdivision {
    name: "Morbihan",
    code: "FR-56",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-56",
};


pub const FR_57: Subdivision = Subdivision {
    name: "Moselle",
    code: "FR-57",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-57",
};


pub const FR_58: Subdivision = Subdivision {
    name: "Nièvre",
    code: "FR-58",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-58",
};


pub const FR_59: Subdivision = Subdivision {
    name: "Nord",
    code: "FR-59",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-59",
};


pub const FR_60: Subdivision = Subdivision {
    name: "Oise",
    code: "FR-60",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-60",
};


pub const FR_61: Subdivision = Subdivision {
    name: "Orne",
    code: "FR-61",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-61",
};


pub const FR_62: Subdivision = Subdivision {
    name: "Pas-de-Calais",
    code: "FR-62",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-62",
};


pub const FR_63: Subdivision = Subdivision {
    name: "Puy-de-Dôme",
    code: "FR-63",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-63",
};


pub const FR_64: Subdivision = Subdivision {
    name: "Pyrénées-Atlantiques",
    code: "FR-64",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-64",
};


pub const FR_65: Subdivision = Subdivision {
    name: "Hautes-Pyrénées",
    code: "FR-65",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-65",
};


pub const FR_66: Subdivision = Subdivision {
    name: "Pyrénées-Orientales",
    code: "FR-66",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-66",
};


pub const FR_67: Subdivision = Subdivision {
    name: "Bas-Rhin",
    code: "FR-67",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-67",
};


pub const FR_68: Subdivision = Subdivision {
    name: "Haut-Rhin",
    code: "FR-68",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-68",
};


pub const FR_69: Subdivision = Subdivision {
    name: "Rhône",
    code: "FR-69",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-69",
};


pub const FR_70: Subdivision = Subdivision {
    name: "Haute-Saône",
    code: "FR-70",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-70",
};


pub const FR_71: Subdivision = Subdivision {
    name: "Saône-et-Loire",
    code: "FR-71",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-71",
};


pub const FR_72: Subdivision = Subdivision {
    name: "Sarthe",
    code: "FR-72",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-72",
};


pub const FR_73: Subdivision = Subdivision {
    name: "Savoie",
    code: "FR-73",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-73",
};


pub const FR_74: Subdivision = Subdivision {
    name: "Haute-Savoie",
    code: "FR-74",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-74",
};


pub const FR_75: Subdivision = Subdivision {
    name: "Paris",
    code: "FR-75",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-75",
};


pub const FR_76: Subdivision = Subdivision {
    name: "Seine-Maritime",
    code: "FR-76",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-76",
};


pub const FR_77: Subdivision = Subdivision {
    name: "Seine-et-Marne",
    code: "FR-77",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-77",
};


pub const FR_78: Subdivision = Subdivision {
    name: "Yvelines",
    code: "FR-78",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-78",
};


pub const FR_79: Subdivision = Subdivision {
    name: "Deux-Sèvres",
    code: "FR-79",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-79",
};


pub const FR_80: Subdivision = Subdivision {
    name: "Somme",
    code: "FR-80",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-80",
};


pub const FR_81: Subdivision = Subdivision {
    name: "Tarn",
    code: "FR-81",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-81",
};


pub const FR_82: Subdivision = Subdivision {
    name: "Tarn-et-Garonne",
    code: "FR-82",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-82",
};


pub const FR_83: Subdivision = Subdivision {
    name: "Var",
    code: "FR-83",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-83",
};


pub const FR_84: Subdivision = Subdivision {
    name: "Vaucluse",
    code: "FR-84",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-84",
};


pub const FR_85: Subdivision = Subdivision {
    name: "Vendée",
    code: "FR-85",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-85",
};


pub const FR_86: Subdivision = Subdivision {
    name: "Vienne",
    code: "FR-86",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-86",
};


pub const FR_87: Subdivision = Subdivision {
    name: "Haute-Vienne",
    code: "FR-87",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-87",
};


pub const FR_88: Subdivision = Subdivision {
    name: "Vosges",
    code: "FR-88",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-88",
};


pub const FR_89: Subdivision = Subdivision {
    name: "Yonne",
    code: "FR-89",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-89",
};


pub const FR_90: Subdivision = Subdivision {
    name: "Territoire de Belfort",
    code: "FR-90",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-90",
};


pub const FR_91: Subdivision = Subdivision {
    name: "Essonne",
    code: "FR-91",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-91",
};


pub const FR_92: Subdivision = Subdivision {
    name: "Hauts-de-Seine",
    code: "FR-92",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-92",
};


pub const FR_93: Subdivision = Subdivision {
    name: "Seine-Saint-Denis",
    code: "FR-93",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-93",
};


pub const FR_94: Subdivision = Subdivision {
    name: "Val-de-Marne",
    code: "FR-94",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-94",
};


pub const FR_95: Subdivision = Subdivision {
    name: "Val-d'Oise",
    code: "FR-95",
    subdivision_type: "Metropolitan department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-95",
};


pub const FR_A: Subdivision = Subdivision {
    name: "Alsace",
    code: "FR-A",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-A",
};


pub const FR_B: Subdivision = Subdivision {
    name: "Aquitaine",
    code: "FR-B",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-B",
};


pub const FR_BL: Subdivision = Subdivision {
    name: "Saint-Barthélemy",
    code: "FR-BL",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-BL",
};


pub const FR_C: Subdivision = Subdivision {
    name: "Auvergne",
    code: "FR-C",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-C",
};


pub const FR_CP: Subdivision = Subdivision {
    name: "Clipperton",
    code: "FR-CP",
    subdivision_type: "Dependency",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-CP",
};


pub const FR_D: Subdivision = Subdivision {
    name: "Bourgogne",
    code: "FR-D",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-D",
};


pub const FR_E: Subdivision = Subdivision {
    name: "Bretagne",
    code: "FR-E",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-E",
};


pub const FR_F: Subdivision = Subdivision {
    name: "Centre",
    code: "FR-F",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-F",
};


pub const FR_G: Subdivision = Subdivision {
    name: "Champagne-Ardenne",
    code: "FR-G",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-G",
};


pub const FR_GF: Subdivision = Subdivision {
    name: "Guyane",
    code: "FR-GF",
    subdivision_type: "Overseas department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-GF",
};


pub const FR_GP: Subdivision = Subdivision {
    name: "Guadeloupe",
    code: "FR-GP",
    subdivision_type: "Overseas department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-GP",
};


pub const FR_H: Subdivision = Subdivision {
    name: "Corse",
    code: "FR-H",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-H",
};


pub const FR_I: Subdivision = Subdivision {
    name: "Franche-Comté",
    code: "FR-I",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-I",
};


pub const FR_J: Subdivision = Subdivision {
    name: "Île-de-France",
    code: "FR-J",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-J",
};


pub const FR_K: Subdivision = Subdivision {
    name: "Languedoc-Roussillon",
    code: "FR-K",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-K",
};


pub const FR_L: Subdivision = Subdivision {
    name: "Limousin",
    code: "FR-L",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-L",
};


pub const FR_MF: Subdivision = Subdivision {
    name: "Saint-Martin",
    code: "FR-MF",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-MF",
};


pub const FR_M: Subdivision = Subdivision {
    name: "Lorraine",
    code: "FR-M",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-M",
};


pub const FR_MQ: Subdivision = Subdivision {
    name: "Martinique",
    code: "FR-MQ",
    subdivision_type: "Overseas department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-MQ",
};


pub const FR_NC: Subdivision = Subdivision {
    name: "Nouvelle-Calédonie",
    code: "FR-NC",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-NC",
};


pub const FR_N: Subdivision = Subdivision {
    name: "Midi-Pyrénées",
    code: "FR-N",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-N",
};


pub const FR_O: Subdivision = Subdivision {
    name: "Nord-Pas-de-Calais",
    code: "FR-O",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-O",
};


pub const FR_P: Subdivision = Subdivision {
    name: "Basse-Normandie",
    code: "FR-P",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-P",
};


pub const FR_PF: Subdivision = Subdivision {
    name: "Polynésie française",
    code: "FR-PF",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-PF",
};


pub const FR_PM: Subdivision = Subdivision {
    name: "Saint-Pierre-et-Miquelon",
    code: "FR-PM",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-PM",
};


pub const FR_Q: Subdivision = Subdivision {
    name: "Haute-Normandie",
    code: "FR-Q",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-Q",
};


pub const FR_RE: Subdivision = Subdivision {
    name: "La Réunion",
    code: "FR-RE",
    subdivision_type: "Overseas department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-RE",
};


pub const FR_R: Subdivision = Subdivision {
    name: "Pays-de-la-Loire",
    code: "FR-R",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-R",
};


pub const FR_S: Subdivision = Subdivision {
    name: "Picardie",
    code: "FR-S",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-S",
};


pub const FR_TF: Subdivision = Subdivision {
    name: "Terres australes françaises",
    code: "FR-TF",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-TF",
};


pub const FR_T: Subdivision = Subdivision {
    name: "Poitou-Charentes",
    code: "FR-T",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-T",
};


pub const FR_U: Subdivision = Subdivision {
    name: "Provence-Alpes-Côte-d'Azur",
    code: "FR-U",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-U",
};


pub const FR_V: Subdivision = Subdivision {
    name: "Rhône-Alpes",
    code: "FR-V",
    subdivision_type: "Metropolitan region",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-V",
};


pub const FR_WF: Subdivision = Subdivision {
    name: "Wallis-et-Futuna",
    code: "FR-WF",
    subdivision_type: "Overseas territorial collectivity",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-WF",
};


pub const FR_YT: Subdivision = Subdivision {
    name: "Mayotte",
    code: "FR-YT",
    subdivision_type: "Overseas department",
    country_name: "France",
    country_code: "FR",
    region_code: "FR-YT",
};


pub const GF_GF: Subdivision = Subdivision {
    name: "French Guiana",
    code: "GF-GF",
    subdivision_type: "Country",
    country_name: "French Guiana",
    country_code: "GF",
    region_code: "GF-GF",
};


pub const PF_PF: Subdivision = Subdivision {
    name: "French Polynesia",
    code: "PF-PF",
    subdivision_type: "Country",
    country_name: "French Polynesia",
    country_code: "PF",
    region_code: "PF-PF",
};


pub const TF_TF: Subdivision = Subdivision {
    name: "French Southern Territories",
    code: "TF-TF",
    subdivision_type: "Country",
    country_name: "French Southern Territories",
    country_code: "TF",
    region_code: "TF-TF",
};


pub const GA_1: Subdivision = Subdivision {
    name: "Estuaire",
    code: "GA-1",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-1",
};


pub const GA_2: Subdivision = Subdivision {
    name: "Haut-Ogooué",
    code: "GA-2",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-2",
};


pub const GA_3: Subdivision = Subdivision {
    name: "Moyen-Ogooué",
    code: "GA-3",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-3",
};


pub const GA_4: Subdivision = Subdivision {
    name: "Ngounié",
    code: "GA-4",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-4",
};


pub const GA_5: Subdivision = Subdivision {
    name: "Nyanga",
    code: "GA-5",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-5",
};


pub const GA_6: Subdivision = Subdivision {
    name: "Ogooué-Ivindo",
    code: "GA-6",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-6",
};


pub const GA_7: Subdivision = Subdivision {
    name: "Ogooué-Lolo",
    code: "GA-7",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-7",
};


pub const GA_8: Subdivision = Subdivision {
    name: "Ogooué-Maritime",
    code: "GA-8",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-8",
};


pub const GA_9: Subdivision = Subdivision {
    name: "Woleu-Ntem",
    code: "GA-9",
    subdivision_type: "Province",
    country_name: "Gabon",
    country_code: "GA",
    region_code: "GA-9",
};


pub const GM_B: Subdivision = Subdivision {
    name: "Banjul",
    code: "GM-B",
    subdivision_type: "City",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-B",
};


pub const GM_L: Subdivision = Subdivision {
    name: "Lower River",
    code: "GM-L",
    subdivision_type: "Division",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-L",
};


pub const GM_M: Subdivision = Subdivision {
    name: "Central River",
    code: "GM-M",
    subdivision_type: "Division",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-M",
};


pub const GM_N: Subdivision = Subdivision {
    name: "North Bank",
    code: "GM-N",
    subdivision_type: "Division",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-N",
};


pub const GM_U: Subdivision = Subdivision {
    name: "Upper River",
    code: "GM-U",
    subdivision_type: "Division",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-U",
};


pub const GM_W: Subdivision = Subdivision {
    name: "Western",
    code: "GM-W",
    subdivision_type: "Division",
    country_name: "Gambia",
    country_code: "GM",
    region_code: "GM-W",
};


pub const GE_AB: Subdivision = Subdivision {
    name: "Abkhazia",
    code: "GE-AB",
    subdivision_type: "Autonomous republic",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-AB",
};


pub const GE_AJ: Subdivision = Subdivision {
    name: "Ajaria",
    code: "GE-AJ",
    subdivision_type: "Autonomous republic",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-AJ",
};


pub const GE_GU: Subdivision = Subdivision {
    name: "Guria",
    code: "GE-GU",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-GU",
};


pub const GE_IM: Subdivision = Subdivision {
    name: "Imereti",
    code: "GE-IM",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-IM",
};


pub const GE_KA: Subdivision = Subdivision {
    name: "K'akheti",
    code: "GE-KA",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-KA",
};


pub const GE_KK: Subdivision = Subdivision {
    name: "Kvemo Kartli",
    code: "GE-KK",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-KK",
};


pub const GE_MM: Subdivision = Subdivision {
    name: "Mtskheta-Mtianeti",
    code: "GE-MM",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-MM",
};


pub const GE_RL: Subdivision = Subdivision {
    name: "Rach'a-Lechkhumi-Kvemo Svaneti",
    code: "GE-RL",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-RL",
};


pub const GE_SJ: Subdivision = Subdivision {
    name: "Samtskhe-Javakheti",
    code: "GE-SJ",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-SJ",
};


pub const GE_SK: Subdivision = Subdivision {
    name: "Shida Kartli",
    code: "GE-SK",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-SK",
};


pub const GE_SZ: Subdivision = Subdivision {
    name: "Samegrelo-Zemo Svaneti",
    code: "GE-SZ",
    subdivision_type: "Region",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-SZ",
};


pub const GE_TB: Subdivision = Subdivision {
    name: "Tbilisi",
    code: "GE-TB",
    subdivision_type: "City",
    country_name: "Georgia",
    country_code: "GE",
    region_code: "GE-TB",
};


pub const DE_BB: Subdivision = Subdivision {
    name: "Brandenburg",
    code: "DE-BB",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-BB",
};


pub const DE_BE: Subdivision = Subdivision {
    name: "Berlin",
    code: "DE-BE",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-BE",
};


pub const DE_BW: Subdivision = Subdivision {
    name: "Baden-Württemberg",
    code: "DE-BW",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-BW",
};


pub const DE_BY: Subdivision = Subdivision {
    name: "Bayern",
    code: "DE-BY",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-BY",
};


pub const DE_HB: Subdivision = Subdivision {
    name: "Bremen",
    code: "DE-HB",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-HB",
};


pub const DE_HE: Subdivision = Subdivision {
    name: "Hessen",
    code: "DE-HE",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-HE",
};


pub const DE_HH: Subdivision = Subdivision {
    name: "Hamburg",
    code: "DE-HH",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-HH",
};


pub const DE_MV: Subdivision = Subdivision {
    name: "Mecklenburg-Vorpommern",
    code: "DE-MV",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-MV",
};


pub const DE_NI: Subdivision = Subdivision {
    name: "Niedersachsen",
    code: "DE-NI",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-NI",
};


pub const DE_NW: Subdivision = Subdivision {
    name: "Nordrhein-Westfalen",
    code: "DE-NW",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-NW",
};


pub const DE_RP: Subdivision = Subdivision {
    name: "Rheinland-Pfalz",
    code: "DE-RP",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-RP",
};


pub const DE_SH: Subdivision = Subdivision {
    name: "Schleswig-Holstein",
    code: "DE-SH",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-SH",
};


pub const DE_SL: Subdivision = Subdivision {
    name: "Saarland",
    code: "DE-SL",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-SL",
};


pub const DE_SN: Subdivision = Subdivision {
    name: "Sachsen",
    code: "DE-SN",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-SN",
};


pub const DE_ST: Subdivision = Subdivision {
    name: "Sachsen-Anhalt",
    code: "DE-ST",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-ST",
};


pub const DE_TH: Subdivision = Subdivision {
    name: "Thüringen",
    code: "DE-TH",
    subdivision_type: "Länder",
    country_name: "Germany",
    country_code: "DE",
    region_code: "DE-TH",
};


pub const GH_AA: Subdivision = Subdivision {
    name: "Greater Accra",
    code: "GH-AA",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-AA",
};


pub const GH_AH: Subdivision = Subdivision {
    name: "Ashanti",
    code: "GH-AH",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-AH",
};


pub const GH_BA: Subdivision = Subdivision {
    name: "Brong-Ahafo",
    code: "GH-BA",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-BA",
};


pub const GH_CP: Subdivision = Subdivision {
    name: "Central",
    code: "GH-CP",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-CP",
};


pub const GH_EP: Subdivision = Subdivision {
    name: "Eastern",
    code: "GH-EP",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-EP",
};


pub const GH_NP: Subdivision = Subdivision {
    name: "Northern",
    code: "GH-NP",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-NP",
};


pub const GH_TV: Subdivision = Subdivision {
    name: "Volta",
    code: "GH-TV",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-TV",
};


pub const GH_UE: Subdivision = Subdivision {
    name: "Upper East",
    code: "GH-UE",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-UE",
};


pub const GH_UW: Subdivision = Subdivision {
    name: "Upper West",
    code: "GH-UW",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-UW",
};


pub const GH_WP: Subdivision = Subdivision {
    name: "Western",
    code: "GH-WP",
    subdivision_type: "Region",
    country_name: "Ghana",
    country_code: "GH",
    region_code: "GH-WP",
};


pub const GI_GI: Subdivision = Subdivision {
    name: "Gibraltar",
    code: "GI-GI",
    subdivision_type: "Country",
    country_name: "Gibraltar",
    country_code: "GI",
    region_code: "GI-GI",
};


pub const GR_01: Subdivision = Subdivision {
    name: "Aitoloakarnanía",
    code: "GR-01",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-01",
};


pub const GR_03: Subdivision = Subdivision {
    name: "Voiotía",
    code: "GR-03",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-03",
};


pub const GR_04: Subdivision = Subdivision {
    name: "Évvoia",
    code: "GR-04",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-04",
};


pub const GR_05: Subdivision = Subdivision {
    name: "Evrytanía",
    code: "GR-05",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-05",
};


pub const GR_06: Subdivision = Subdivision {
    name: "Fthiótida",
    code: "GR-06",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-06",
};


pub const GR_07: Subdivision = Subdivision {
    name: "Fokída",
    code: "GR-07",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-07",
};


pub const GR_11: Subdivision = Subdivision {
    name: "Argolída",
    code: "GR-11",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-11",
};


pub const GR_12: Subdivision = Subdivision {
    name: "Arkadía",
    code: "GR-12",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-12",
};


pub const GR_13: Subdivision = Subdivision {
    name: "Achaḯa",
    code: "GR-13",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-13",
};


pub const GR_14: Subdivision = Subdivision {
    name: "Ileía",
    code: "GR-14",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-14",
};


pub const GR_15: Subdivision = Subdivision {
    name: "Korinthía",
    code: "GR-15",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-15",
};


pub const GR_16: Subdivision = Subdivision {
    name: "Lakonía",
    code: "GR-16",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-16",
};


pub const GR_17: Subdivision = Subdivision {
    name: "Messinía",
    code: "GR-17",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-17",
};


pub const GR_21: Subdivision = Subdivision {
    name: "Zákynthos",
    code: "GR-21",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-21",
};


pub const GR_22: Subdivision = Subdivision {
    name: "Kérkyra",
    code: "GR-22",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-22",
};


pub const GR_23: Subdivision = Subdivision {
    name: "Kefallinía",
    code: "GR-23",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-23",
};


pub const GR_24: Subdivision = Subdivision {
    name: "Lefkáda",
    code: "GR-24",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-24",
};


pub const GR_31: Subdivision = Subdivision {
    name: "Árta",
    code: "GR-31",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-31",
};


pub const GR_32: Subdivision = Subdivision {
    name: "Thesprotía",
    code: "GR-32",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-32",
};


pub const GR_33: Subdivision = Subdivision {
    name: "Ioánnina",
    code: "GR-33",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-33",
};


pub const GR_34: Subdivision = Subdivision {
    name: "Préveza",
    code: "GR-34",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-34",
};


pub const GR_41: Subdivision = Subdivision {
    name: "Kardítsa",
    code: "GR-41",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-41",
};


pub const GR_42: Subdivision = Subdivision {
    name: "Lárisa",
    code: "GR-42",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-42",
};


pub const GR_43: Subdivision = Subdivision {
    name: "Magnisía",
    code: "GR-43",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-43",
};


pub const GR_44: Subdivision = Subdivision {
    name: "Tríkala",
    code: "GR-44",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-44",
};


pub const GR_51: Subdivision = Subdivision {
    name: "Grevená",
    code: "GR-51",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-51",
};


pub const GR_52: Subdivision = Subdivision {
    name: "Dráma",
    code: "GR-52",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-52",
};


pub const GR_53: Subdivision = Subdivision {
    name: "Imathía",
    code: "GR-53",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-53",
};


pub const GR_54: Subdivision = Subdivision {
    name: "Thessaloníki",
    code: "GR-54",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-54",
};


pub const GR_55: Subdivision = Subdivision {
    name: "Kavála",
    code: "GR-55",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-55",
};


pub const GR_56: Subdivision = Subdivision {
    name: "Kastoriá",
    code: "GR-56",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-56",
};


pub const GR_57: Subdivision = Subdivision {
    name: "Kilkís",
    code: "GR-57",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-57",
};


pub const GR_58: Subdivision = Subdivision {
    name: "Kozáni",
    code: "GR-58",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-58",
};


pub const GR_59: Subdivision = Subdivision {
    name: "Pélla",
    code: "GR-59",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-59",
};


pub const GR_61: Subdivision = Subdivision {
    name: "Piería",
    code: "GR-61",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-61",
};


pub const GR_62: Subdivision = Subdivision {
    name: "Sérres",
    code: "GR-62",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-62",
};


pub const GR_63: Subdivision = Subdivision {
    name: "Flórina",
    code: "GR-63",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-63",
};


pub const GR_64: Subdivision = Subdivision {
    name: "Chalkidikí",
    code: "GR-64",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-64",
};


pub const GR_69: Subdivision = Subdivision {
    name: "Ágion Óros",
    code: "GR-69",
    subdivision_type: "Self-governed part",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-69",
};


pub const GR_71: Subdivision = Subdivision {
    name: "Évros",
    code: "GR-71",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-71",
};


pub const GR_72: Subdivision = Subdivision {
    name: "Xánthi",
    code: "GR-72",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-72",
};


pub const GR_73: Subdivision = Subdivision {
    name: "Rodópi",
    code: "GR-73",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-73",
};


pub const GR_81: Subdivision = Subdivision {
    name: "Dodekánisa",
    code: "GR-81",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-81",
};


pub const GR_82: Subdivision = Subdivision {
    name: "Kykládes",
    code: "GR-82",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-82",
};


pub const GR_83: Subdivision = Subdivision {
    name: "Lésvos",
    code: "GR-83",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-83",
};


pub const GR_84: Subdivision = Subdivision {
    name: "Sámos",
    code: "GR-84",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-84",
};


pub const GR_85: Subdivision = Subdivision {
    name: "Chíos",
    code: "GR-85",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-85",
};


pub const GR_91: Subdivision = Subdivision {
    name: "Irakleío",
    code: "GR-91",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-91",
};


pub const GR_92: Subdivision = Subdivision {
    name: "Lasíthi",
    code: "GR-92",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-92",
};


pub const GR_93: Subdivision = Subdivision {
    name: "Rethýmnis",
    code: "GR-93",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-93",
};


pub const GR_94: Subdivision = Subdivision {
    name: "Chaniá",
    code: "GR-94",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-94",
};


pub const GR_A1: Subdivision = Subdivision {
    name: "Attikí",
    code: "GR-A1",
    subdivision_type: "Department",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-A1",
};


pub const GR_A: Subdivision = Subdivision {
    name: "Anatolikí Makedonía kai Thráki",
    code: "GR-A",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-A",
};


pub const GR_B: Subdivision = Subdivision {
    name: "Kentrikí Makedonía",
    code: "GR-B",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-B",
};


pub const GR_C: Subdivision = Subdivision {
    name: "Dytikí Makedonía",
    code: "GR-C",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-C",
};


pub const GR_D: Subdivision = Subdivision {
    name: "Ípeiros",
    code: "GR-D",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-D",
};


pub const GR_E: Subdivision = Subdivision {
    name: "Thessalía",
    code: "GR-E",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-E",
};


pub const GR_F: Subdivision = Subdivision {
    name: "Ionía Nísia",
    code: "GR-F",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-F",
};


pub const GR_G: Subdivision = Subdivision {
    name: "Dytikí Elláda",
    code: "GR-G",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-G",
};


pub const GR_H: Subdivision = Subdivision {
    name: "Stereá Elláda",
    code: "GR-H",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-H",
};


pub const GR_I: Subdivision = Subdivision {
    name: "Attikí",
    code: "GR-I",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-I",
};


pub const GR_J: Subdivision = Subdivision {
    name: "Peloponnísos",
    code: "GR-J",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-J",
};


pub const GR_K: Subdivision = Subdivision {
    name: "Voreío Aigaío",
    code: "GR-K",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-K",
};


pub const GR_L: Subdivision = Subdivision {
    name: "Notío Aigaío",
    code: "GR-L",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-L",
};


pub const GR_M: Subdivision = Subdivision {
    name: "Krítí",
    code: "GR-M",
    subdivision_type: "Administrative region",
    country_name: "Greece",
    country_code: "GR",
    region_code: "GR-M",
};


pub const GL_KU: Subdivision = Subdivision {
    name: "Kommune Kujalleq",
    code: "GL-KU",
    subdivision_type: "Municipality",
    country_name: "Greenland",
    country_code: "GL",
    region_code: "GL-KU",
};


pub const GL_QA: Subdivision = Subdivision {
    name: "Qaasuitsup Kommunia",
    code: "GL-QA",
    subdivision_type: "Municipality",
    country_name: "Greenland",
    country_code: "GL",
    region_code: "GL-QA",
};


pub const GL_QE: Subdivision = Subdivision {
    name: "Qeqqata Kommunia",
    code: "GL-QE",
    subdivision_type: "Municipality",
    country_name: "Greenland",
    country_code: "GL",
    region_code: "GL-QE",
};


pub const GL_SM: Subdivision = Subdivision {
    name: "Kommuneqarfik Sermersooq",
    code: "GL-SM",
    subdivision_type: "Municipality",
    country_name: "Greenland",
    country_code: "GL",
    region_code: "GL-SM",
};


pub const GD_01: Subdivision = Subdivision {
    name: "Saint Andrew",
    code: "GD-01",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-01",
};


pub const GD_02: Subdivision = Subdivision {
    name: "Saint David",
    code: "GD-02",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-02",
};


pub const GD_03: Subdivision = Subdivision {
    name: "Saint George",
    code: "GD-03",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-03",
};


pub const GD_04: Subdivision = Subdivision {
    name: "Saint John",
    code: "GD-04",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-04",
};


pub const GD_05: Subdivision = Subdivision {
    name: "Saint Mark",
    code: "GD-05",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-05",
};


pub const GD_06: Subdivision = Subdivision {
    name: "Saint Patrick",
    code: "GD-06",
    subdivision_type: "Parish",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-06",
};


pub const GD_10: Subdivision = Subdivision {
    name: "Southern Grenadine Islands",
    code: "GD-10",
    subdivision_type: "Dependency",
    country_name: "Grenada",
    country_code: "GD",
    region_code: "GD-10",
};


pub const GP_GP: Subdivision = Subdivision {
    name: "Guadeloupe",
    code: "GP-GP",
    subdivision_type: "Country",
    country_name: "Guadeloupe",
    country_code: "GP",
    region_code: "GP-GP",
};


pub const GU_GU: Subdivision = Subdivision {
    name: "Guam",
    code: "GU-GU",
    subdivision_type: "Country",
    country_name: "Guam",
    country_code: "GU",
    region_code: "GU-GU",
};


pub const GT_AV: Subdivision = Subdivision {
    name: "Alta Verapaz",
    code: "GT-AV",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-AV",
};


pub const GT_BV: Subdivision = Subdivision {
    name: "Baja Verapaz",
    code: "GT-BV",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-BV",
};


pub const GT_CM: Subdivision = Subdivision {
    name: "Chimaltenango",
    code: "GT-CM",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-CM",
};


pub const GT_CQ: Subdivision = Subdivision {
    name: "Chiquimula",
    code: "GT-CQ",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-CQ",
};


pub const GT_ES: Subdivision = Subdivision {
    name: "Escuintla",
    code: "GT-ES",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-ES",
};


pub const GT_GU: Subdivision = Subdivision {
    name: "Guatemala",
    code: "GT-GU",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-GU",
};


pub const GT_HU: Subdivision = Subdivision {
    name: "Huehuetenango",
    code: "GT-HU",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-HU",
};


pub const GT_IZ: Subdivision = Subdivision {
    name: "Izabal",
    code: "GT-IZ",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-IZ",
};


pub const GT_JA: Subdivision = Subdivision {
    name: "Jalapa",
    code: "GT-JA",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-JA",
};


pub const GT_JU: Subdivision = Subdivision {
    name: "Jutiapa",
    code: "GT-JU",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-JU",
};


pub const GT_PE: Subdivision = Subdivision {
    name: "Petén",
    code: "GT-PE",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-PE",
};


pub const GT_PR: Subdivision = Subdivision {
    name: "El Progreso",
    code: "GT-PR",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-PR",
};


pub const GT_QC: Subdivision = Subdivision {
    name: "Quiché",
    code: "GT-QC",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-QC",
};


pub const GT_QZ: Subdivision = Subdivision {
    name: "Quetzaltenango",
    code: "GT-QZ",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-QZ",
};


pub const GT_RE: Subdivision = Subdivision {
    name: "Retalhuleu",
    code: "GT-RE",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-RE",
};


pub const GT_SA: Subdivision = Subdivision {
    name: "Sacatepéquez",
    code: "GT-SA",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-SA",
};


pub const GT_SM: Subdivision = Subdivision {
    name: "San Marcos",
    code: "GT-SM",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-SM",
};


pub const GT_SO: Subdivision = Subdivision {
    name: "Sololá",
    code: "GT-SO",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-SO",
};


pub const GT_SR: Subdivision = Subdivision {
    name: "Santa Rosa",
    code: "GT-SR",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-SR",
};


pub const GT_SU: Subdivision = Subdivision {
    name: "Suchitepéquez",
    code: "GT-SU",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-SU",
};


pub const GT_TO: Subdivision = Subdivision {
    name: "Totonicapán",
    code: "GT-TO",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-TO",
};


pub const GT_ZA: Subdivision = Subdivision {
    name: "Zacapa",
    code: "GT-ZA",
    subdivision_type: "Department",
    country_name: "Guatemala",
    country_code: "GT",
    region_code: "GT-ZA",
};


pub const GG_GG: Subdivision = Subdivision {
    name: "Guernsey",
    code: "GG-GG",
    subdivision_type: "Country",
    country_name: "Guernsey",
    country_code: "GG",
    region_code: "GG-GG",
};


pub const GW_BA: Subdivision = Subdivision {
    name: "Bafatá",
    code: "GW-BA",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-BA",
};


pub const GW_BL: Subdivision = Subdivision {
    name: "Bolama",
    code: "GW-BL",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-BL",
};


pub const GW_BM: Subdivision = Subdivision {
    name: "Biombo",
    code: "GW-BM",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-BM",
};


pub const GW_BS: Subdivision = Subdivision {
    name: "Bissau",
    code: "GW-BS",
    subdivision_type: "Autonomous sector",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-BS",
};


pub const GW_CA: Subdivision = Subdivision {
    name: "Cacheu",
    code: "GW-CA",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-CA",
};


pub const GW_GA: Subdivision = Subdivision {
    name: "Gabú",
    code: "GW-GA",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-GA",
};


pub const GW_L: Subdivision = Subdivision {
    name: "Leste",
    code: "GW-L",
    subdivision_type: "Province",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-L",
};


pub const GW_N: Subdivision = Subdivision {
    name: "Norte",
    code: "GW-N",
    subdivision_type: "Province",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-N",
};


pub const GW_OI: Subdivision = Subdivision {
    name: "Oio",
    code: "GW-OI",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-OI",
};


pub const GW_QU: Subdivision = Subdivision {
    name: "Quinara",
    code: "GW-QU",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-QU",
};


pub const GW_S: Subdivision = Subdivision {
    name: "Sul",
    code: "GW-S",
    subdivision_type: "Province",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-S",
};


pub const GW_TO: Subdivision = Subdivision {
    name: "Tombali",
    code: "GW-TO",
    subdivision_type: "Region",
    country_name: "Guinea-Bissau",
    country_code: "GW",
    region_code: "GW-TO",
};


pub const GN_B: Subdivision = Subdivision {
    name: "Boké",
    code: "GN-B",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-B",
};


pub const GN_BE: Subdivision = Subdivision {
    name: "Beyla",
    code: "GN-BE",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-BE",
};


pub const GN_BF: Subdivision = Subdivision {
    name: "Boffa",
    code: "GN-BF",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-BF",
};


pub const GN_BK: Subdivision = Subdivision {
    name: "Boké",
    code: "GN-BK",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-BK",
};


pub const GN_C: Subdivision = Subdivision {
    name: "Conakry",
    code: "GN-C",
    subdivision_type: "Governorate",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-C",
};


pub const GN_CO: Subdivision = Subdivision {
    name: "Coyah",
    code: "GN-CO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-CO",
};


pub const GN_DB: Subdivision = Subdivision {
    name: "Dabola",
    code: "GN-DB",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-DB",
};


pub const GN_DI: Subdivision = Subdivision {
    name: "Dinguiraye",
    code: "GN-DI",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-DI",
};


pub const GN_D: Subdivision = Subdivision {
    name: "Kindia",
    code: "GN-D",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-D",
};


pub const GN_DL: Subdivision = Subdivision {
    name: "Dalaba",
    code: "GN-DL",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-DL",
};


pub const GN_DU: Subdivision = Subdivision {
    name: "Dubréka",
    code: "GN-DU",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-DU",
};


pub const GN_FA: Subdivision = Subdivision {
    name: "Faranah",
    code: "GN-FA",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-FA",
};


pub const GN_F: Subdivision = Subdivision {
    name: "Faranah",
    code: "GN-F",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-F",
};


pub const GN_FO: Subdivision = Subdivision {
    name: "Forécariah",
    code: "GN-FO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-FO",
};


pub const GN_FR: Subdivision = Subdivision {
    name: "Fria",
    code: "GN-FR",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-FR",
};


pub const GN_GA: Subdivision = Subdivision {
    name: "Gaoual",
    code: "GN-GA",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-GA",
};


pub const GN_GU: Subdivision = Subdivision {
    name: "Guékédou",
    code: "GN-GU",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-GU",
};


pub const GN_KA: Subdivision = Subdivision {
    name: "Kankan",
    code: "GN-KA",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KA",
};


pub const GN_KB: Subdivision = Subdivision {
    name: "Koubia",
    code: "GN-KB",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KB",
};


pub const GN_KD: Subdivision = Subdivision {
    name: "Kindia",
    code: "GN-KD",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KD",
};


pub const GN_KE: Subdivision = Subdivision {
    name: "Kérouané",
    code: "GN-KE",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KE",
};


pub const GN_K: Subdivision = Subdivision {
    name: "Kankan",
    code: "GN-K",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-K",
};


pub const GN_KN: Subdivision = Subdivision {
    name: "Koundara",
    code: "GN-KN",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KN",
};


pub const GN_KO: Subdivision = Subdivision {
    name: "Kouroussa",
    code: "GN-KO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KO",
};


pub const GN_KS: Subdivision = Subdivision {
    name: "Kissidougou",
    code: "GN-KS",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-KS",
};


pub const GN_LA: Subdivision = Subdivision {
    name: "Labé",
    code: "GN-LA",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-LA",
};


pub const GN_LE: Subdivision = Subdivision {
    name: "Lélouma",
    code: "GN-LE",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-LE",
};


pub const GN_L: Subdivision = Subdivision {
    name: "Labé",
    code: "GN-L",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-L",
};


pub const GN_LO: Subdivision = Subdivision {
    name: "Lola",
    code: "GN-LO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-LO",
};


pub const GN_MC: Subdivision = Subdivision {
    name: "Macenta",
    code: "GN-MC",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-MC",
};


pub const GN_MD: Subdivision = Subdivision {
    name: "Mandiana",
    code: "GN-MD",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-MD",
};


pub const GN_ML: Subdivision = Subdivision {
    name: "Mali",
    code: "GN-ML",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-ML",
};


pub const GN_M: Subdivision = Subdivision {
    name: "Mamou",
    code: "GN-M",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-M",
};


pub const GN_MM: Subdivision = Subdivision {
    name: "Mamou",
    code: "GN-MM",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-MM",
};


pub const GN_N: Subdivision = Subdivision {
    name: "Nzérékoré",
    code: "GN-N",
    subdivision_type: "Administrative region",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-N",
};


pub const GN_NZ: Subdivision = Subdivision {
    name: "Nzérékoré",
    code: "GN-NZ",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-NZ",
};


pub const GN_PI: Subdivision = Subdivision {
    name: "Pita",
    code: "GN-PI",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-PI",
};


pub const GN_SI: Subdivision = Subdivision {
    name: "Siguiri",
    code: "GN-SI",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-SI",
};


pub const GN_TE: Subdivision = Subdivision {
    name: "Télimélé",
    code: "GN-TE",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-TE",
};


pub const GN_TO: Subdivision = Subdivision {
    name: "Tougué",
    code: "GN-TO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-TO",
};


pub const GN_YO: Subdivision = Subdivision {
    name: "Yomou",
    code: "GN-YO",
    subdivision_type: "Prefecture",
    country_name: "Guinea",
    country_code: "GN",
    region_code: "GN-YO",
};


pub const GY_BA: Subdivision = Subdivision {
    name: "Barima-Waini",
    code: "GY-BA",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-BA",
};


pub const GY_CU: Subdivision = Subdivision {
    name: "Cuyuni-Mazaruni",
    code: "GY-CU",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-CU",
};


pub const GY_DE: Subdivision = Subdivision {
    name: "Demerara-Mahaica",
    code: "GY-DE",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-DE",
};


pub const GY_EB: Subdivision = Subdivision {
    name: "East Berbice-Corentyne",
    code: "GY-EB",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-EB",
};


pub const GY_ES: Subdivision = Subdivision {
    name: "Essequibo Islands-West Demerara",
    code: "GY-ES",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-ES",
};


pub const GY_MA: Subdivision = Subdivision {
    name: "Mahaica-Berbice",
    code: "GY-MA",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-MA",
};


pub const GY_PM: Subdivision = Subdivision {
    name: "Pomeroon-Supenaam",
    code: "GY-PM",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-PM",
};


pub const GY_PT: Subdivision = Subdivision {
    name: "Potaro-Siparuni",
    code: "GY-PT",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-PT",
};


pub const GY_UD: Subdivision = Subdivision {
    name: "Upper Demerara-Berbice",
    code: "GY-UD",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-UD",
};


pub const GY_UT: Subdivision = Subdivision {
    name: "Upper Takutu-Upper Essequibo",
    code: "GY-UT",
    subdivision_type: "Region",
    country_name: "Guyana",
    country_code: "GY",
    region_code: "GY-UT",
};


pub const HT_AR: Subdivision = Subdivision {
    name: "Artibonite",
    code: "HT-AR",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-AR",
};


pub const HT_CE: Subdivision = Subdivision {
    name: "Centre",
    code: "HT-CE",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-CE",
};


pub const HT_GA: Subdivision = Subdivision {
    name: "Grandans",
    code: "HT-GA",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-GA",
};


pub const HT_ND: Subdivision = Subdivision {
    name: "Nò",
    code: "HT-ND",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-ND",
};


pub const HT_NE: Subdivision = Subdivision {
    name: "Nòdès",
    code: "HT-NE",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-NE",
};


pub const HT_NI: Subdivision = Subdivision {
    name: "Nip",
    code: "HT-NI",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-NI",
};


pub const HT_NO: Subdivision = Subdivision {
    name: "Nòdwès",
    code: "HT-NO",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-NO",
};


pub const HT_OU: Subdivision = Subdivision {
    name: "Lwès",
    code: "HT-OU",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-OU",
};


pub const HT_SD: Subdivision = Subdivision {
    name: "Sid",
    code: "HT-SD",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-SD",
};


pub const HT_SE: Subdivision = Subdivision {
    name: "Sidès",
    code: "HT-SE",
    subdivision_type: "Department",
    country_name: "Haiti",
    country_code: "HT",
    region_code: "HT-SE",
};


pub const HM_HM: Subdivision = Subdivision {
    name: "Heard & McDonald Islands",
    code: "HM-HM",
    subdivision_type: "Country",
    country_name: "Heard & McDonald Islands",
    country_code: "HM",
    region_code: "HM-HM",
};


pub const HN_AT: Subdivision = Subdivision {
    name: "Atlántida",
    code: "HN-AT",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-AT",
};


pub const HN_CH: Subdivision = Subdivision {
    name: "Choluteca",
    code: "HN-CH",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-CH",
};


pub const HN_CL: Subdivision = Subdivision {
    name: "Colón",
    code: "HN-CL",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-CL",
};


pub const HN_CM: Subdivision = Subdivision {
    name: "Comayagua",
    code: "HN-CM",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-CM",
};


pub const HN_CP: Subdivision = Subdivision {
    name: "Copán",
    code: "HN-CP",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-CP",
};


pub const HN_CR: Subdivision = Subdivision {
    name: "Cortés",
    code: "HN-CR",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-CR",
};


pub const HN_EP: Subdivision = Subdivision {
    name: "El Paraíso",
    code: "HN-EP",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-EP",
};


pub const HN_FM: Subdivision = Subdivision {
    name: "Francisco Morazán",
    code: "HN-FM",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-FM",
};


pub const HN_GD: Subdivision = Subdivision {
    name: "Gracias a Dios",
    code: "HN-GD",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-GD",
};


pub const HN_IB: Subdivision = Subdivision {
    name: "Islas de la Bahía",
    code: "HN-IB",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-IB",
};


pub const HN_IN: Subdivision = Subdivision {
    name: "Intibucá",
    code: "HN-IN",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-IN",
};


pub const HN_LE: Subdivision = Subdivision {
    name: "Lempira",
    code: "HN-LE",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-LE",
};


pub const HN_LP: Subdivision = Subdivision {
    name: "La Paz",
    code: "HN-LP",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-LP",
};


pub const HN_OC: Subdivision = Subdivision {
    name: "Ocotepeque",
    code: "HN-OC",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-OC",
};


pub const HN_OL: Subdivision = Subdivision {
    name: "Olancho",
    code: "HN-OL",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-OL",
};


pub const HN_SB: Subdivision = Subdivision {
    name: "Santa Bárbara",
    code: "HN-SB",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-SB",
};


pub const HN_VA: Subdivision = Subdivision {
    name: "Valle",
    code: "HN-VA",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-VA",
};


pub const HN_YO: Subdivision = Subdivision {
    name: "Yoro",
    code: "HN-YO",
    subdivision_type: "Department",
    country_name: "Honduras",
    country_code: "HN",
    region_code: "HN-YO",
};


pub const HK_HK: Subdivision = Subdivision {
    name: "Hong Kong",
    code: "HK-HK",
    subdivision_type: "Country",
    country_name: "Hong Kong",
    country_code: "HK",
    region_code: "HK-HK",
};


pub const HU_BA: Subdivision = Subdivision {
    name: "Baranya",
    code: "HU-BA",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BA",
};


pub const HU_BC: Subdivision = Subdivision {
    name: "Békéscsaba",
    code: "HU-BC",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BC",
};


pub const HU_BE: Subdivision = Subdivision {
    name: "Békés",
    code: "HU-BE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BE",
};


pub const HU_BK: Subdivision = Subdivision {
    name: "Bács-Kiskun",
    code: "HU-BK",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BK",
};


pub const HU_BU: Subdivision = Subdivision {
    name: "Budapest",
    code: "HU-BU",
    subdivision_type: "Capital city",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BU",
};


pub const HU_BZ: Subdivision = Subdivision {
    name: "Borsod-Abaúj-Zemplén",
    code: "HU-BZ",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-BZ",
};


pub const HU_CS: Subdivision = Subdivision {
    name: "Csongrád",
    code: "HU-CS",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-CS",
};


pub const HU_DE: Subdivision = Subdivision {
    name: "Debrecen",
    code: "HU-DE",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-DE",
};


pub const HU_DU: Subdivision = Subdivision {
    name: "Dunaújváros",
    code: "HU-DU",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-DU",
};


pub const HU_EG: Subdivision = Subdivision {
    name: "Eger",
    code: "HU-EG",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-EG",
};


pub const HU_ER: Subdivision = Subdivision {
    name: "Érd",
    code: "HU-ER",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-ER",
};


pub const HU_FE: Subdivision = Subdivision {
    name: "Fejér",
    code: "HU-FE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-FE",
};


pub const HU_GS: Subdivision = Subdivision {
    name: "Győr-Moson-Sopron",
    code: "HU-GS",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-GS",
};


pub const HU_GY: Subdivision = Subdivision {
    name: "Győr",
    code: "HU-GY",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-GY",
};


pub const HU_HB: Subdivision = Subdivision {
    name: "Hajdú-Bihar",
    code: "HU-HB",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-HB",
};


pub const HU_HE: Subdivision = Subdivision {
    name: "Heves",
    code: "HU-HE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-HE",
};


pub const HU_HV: Subdivision = Subdivision {
    name: "Hódmezővásárhely",
    code: "HU-HV",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-HV",
};


pub const HU_JN: Subdivision = Subdivision {
    name: "Jász-Nagykun-Szolnok",
    code: "HU-JN",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-JN",
};


pub const HU_KE: Subdivision = Subdivision {
    name: "Komárom-Esztergom",
    code: "HU-KE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-KE",
};


pub const HU_KM: Subdivision = Subdivision {
    name: "Kecskemét",
    code: "HU-KM",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-KM",
};


pub const HU_KV: Subdivision = Subdivision {
    name: "Kaposvár",
    code: "HU-KV",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-KV",
};


pub const HU_MI: Subdivision = Subdivision {
    name: "Miskolc",
    code: "HU-MI",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-MI",
};


pub const HU_NK: Subdivision = Subdivision {
    name: "Nagykanizsa",
    code: "HU-NK",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-NK",
};


pub const HU_NO: Subdivision = Subdivision {
    name: "Nógrád",
    code: "HU-NO",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-NO",
};


pub const HU_NY: Subdivision = Subdivision {
    name: "Nyíregyháza",
    code: "HU-NY",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-NY",
};


pub const HU_PE: Subdivision = Subdivision {
    name: "Pest",
    code: "HU-PE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-PE",
};


pub const HU_PS: Subdivision = Subdivision {
    name: "Pécs",
    code: "HU-PS",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-PS",
};


pub const HU_SD: Subdivision = Subdivision {
    name: "Szeged",
    code: "HU-SD",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SD",
};


pub const HU_SF: Subdivision = Subdivision {
    name: "Székesfehérvár",
    code: "HU-SF",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SF",
};


pub const HU_SH: Subdivision = Subdivision {
    name: "Szombathely",
    code: "HU-SH",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SH",
};


pub const HU_SK: Subdivision = Subdivision {
    name: "Szolnok",
    code: "HU-SK",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SK",
};


pub const HU_SN: Subdivision = Subdivision {
    name: "Sopron",
    code: "HU-SN",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SN",
};


pub const HU_SO: Subdivision = Subdivision {
    name: "Somogy",
    code: "HU-SO",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SO",
};


pub const HU_SS: Subdivision = Subdivision {
    name: "Szekszárd",
    code: "HU-SS",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SS",
};


pub const HU_ST: Subdivision = Subdivision {
    name: "Salgótarján",
    code: "HU-ST",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-ST",
};


pub const HU_SZ: Subdivision = Subdivision {
    name: "Szabolcs-Szatmár-Bereg",
    code: "HU-SZ",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-SZ",
};


pub const HU_TB: Subdivision = Subdivision {
    name: "Tatabánya",
    code: "HU-TB",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-TB",
};


pub const HU_TO: Subdivision = Subdivision {
    name: "Tolna",
    code: "HU-TO",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-TO",
};


pub const HU_VA: Subdivision = Subdivision {
    name: "Vas",
    code: "HU-VA",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-VA",
};


pub const HU_VE: Subdivision = Subdivision {
    name: "Veszprém",
    code: "HU-VE",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-VE",
};


pub const HU_VM: Subdivision = Subdivision {
    name: "Veszprém",
    code: "HU-VM",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-VM",
};


pub const HU_ZA: Subdivision = Subdivision {
    name: "Zala",
    code: "HU-ZA",
    subdivision_type: "County",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-ZA",
};


pub const HU_ZE: Subdivision = Subdivision {
    name: "Zalaegerszeg",
    code: "HU-ZE",
    subdivision_type: "City of county right",
    country_name: "Hungary",
    country_code: "HU",
    region_code: "HU-ZE",
};


pub const IS_0: Subdivision = Subdivision {
    name: "Reykjavík",
    code: "IS-0",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-0",
};


pub const IS_1: Subdivision = Subdivision {
    name: "Höfuðborgarsvæði utan Reykjavíkur",
    code: "IS-1",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-1",
};


pub const IS_2: Subdivision = Subdivision {
    name: "Suðurnes",
    code: "IS-2",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-2",
};


pub const IS_3: Subdivision = Subdivision {
    name: "Vesturland",
    code: "IS-3",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-3",
};


pub const IS_4: Subdivision = Subdivision {
    name: "Vestfirðir",
    code: "IS-4",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-4",
};


pub const IS_5: Subdivision = Subdivision {
    name: "Norðurland vestra",
    code: "IS-5",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-5",
};


pub const IS_6: Subdivision = Subdivision {
    name: "Norðurland eystra",
    code: "IS-6",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-6",
};


pub const IS_7: Subdivision = Subdivision {
    name: "Austurland",
    code: "IS-7",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-7",
};


pub const IS_8: Subdivision = Subdivision {
    name: "Suðurland",
    code: "IS-8",
    subdivision_type: "Region",
    country_name: "Iceland",
    country_code: "IS",
    region_code: "IS-8",
};


pub const IN_AN: Subdivision = Subdivision {
    name: "Andaman and Nicobar Islands",
    code: "IN-AN",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-AN",
};


pub const IN_AP: Subdivision = Subdivision {
    name: "Andhra Pradesh",
    code: "IN-AP",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-AP",
};


pub const IN_AR: Subdivision = Subdivision {
    name: "Arunachal Pradesh",
    code: "IN-AR",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-AR",
};


pub const IN_AS: Subdivision = Subdivision {
    name: "Assam",
    code: "IN-AS",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-AS",
};


pub const IN_BR: Subdivision = Subdivision {
    name: "Bihar",
    code: "IN-BR",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-BR",
};


pub const IN_CH: Subdivision = Subdivision {
    name: "Chandigarh",
    code: "IN-CH",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-CH",
};


pub const IN_CT: Subdivision = Subdivision {
    name: "Chhattisgarh",
    code: "IN-CT",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-CT",
};


pub const IN_DD: Subdivision = Subdivision {
    name: "Daman and Diu",
    code: "IN-DD",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-DD",
};


pub const IN_DL: Subdivision = Subdivision {
    name: "Delhi",
    code: "IN-DL",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-DL",
};


pub const IN_DN: Subdivision = Subdivision {
    name: "Dadra and Nagar Haveli",
    code: "IN-DN",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-DN",
};


pub const IN_GA: Subdivision = Subdivision {
    name: "Goa",
    code: "IN-GA",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-GA",
};


pub const IN_GJ: Subdivision = Subdivision {
    name: "Gujarat",
    code: "IN-GJ",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-GJ",
};


pub const IN_HP: Subdivision = Subdivision {
    name: "Himachal Pradesh",
    code: "IN-HP",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-HP",
};


pub const IN_HR: Subdivision = Subdivision {
    name: "Haryana",
    code: "IN-HR",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-HR",
};


pub const IN_JH: Subdivision = Subdivision {
    name: "Jharkhand",
    code: "IN-JH",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-JH",
};


pub const IN_JK: Subdivision = Subdivision {
    name: "Jammu and Kashmir",
    code: "IN-JK",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-JK",
};


pub const IN_KA: Subdivision = Subdivision {
    name: "Karnataka",
    code: "IN-KA",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-KA",
};


pub const IN_KL: Subdivision = Subdivision {
    name: "Kerala",
    code: "IN-KL",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-KL",
};


pub const IN_LD: Subdivision = Subdivision {
    name: "Lakshadweep",
    code: "IN-LD",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-LD",
};


pub const IN_MH: Subdivision = Subdivision {
    name: "Maharashtra",
    code: "IN-MH",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-MH",
};


pub const IN_ML: Subdivision = Subdivision {
    name: "Meghalaya",
    code: "IN-ML",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-ML",
};


pub const IN_MN: Subdivision = Subdivision {
    name: "Manipur",
    code: "IN-MN",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-MN",
};


pub const IN_MP: Subdivision = Subdivision {
    name: "Madhya Pradesh",
    code: "IN-MP",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-MP",
};


pub const IN_MZ: Subdivision = Subdivision {
    name: "Mizoram",
    code: "IN-MZ",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-MZ",
};


pub const IN_NL: Subdivision = Subdivision {
    name: "Nagaland",
    code: "IN-NL",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-NL",
};


pub const IN_OR: Subdivision = Subdivision {
    name: "Odisha",
    code: "IN-OR",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-OR",
};


pub const IN_PB: Subdivision = Subdivision {
    name: "Punjab",
    code: "IN-PB",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-PB",
};


pub const IN_PY: Subdivision = Subdivision {
    name: "Puducherry",
    code: "IN-PY",
    subdivision_type: "Union territory",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-PY",
};


pub const IN_RJ: Subdivision = Subdivision {
    name: "Rajasthan",
    code: "IN-RJ",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-RJ",
};


pub const IN_SK: Subdivision = Subdivision {
    name: "Sikkim",
    code: "IN-SK",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-SK",
};


pub const IN_TG: Subdivision = Subdivision {
    name: "Telangana",
    code: "IN-TG",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-TG",
};


pub const IN_TN: Subdivision = Subdivision {
    name: "Tamil Nadu",
    code: "IN-TN",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-TN",
};


pub const IN_TR: Subdivision = Subdivision {
    name: "Tripura",
    code: "IN-TR",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-TR",
};


pub const IN_UP: Subdivision = Subdivision {
    name: "Uttar Pradesh",
    code: "IN-UP",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-UP",
};


pub const IN_UT: Subdivision = Subdivision {
    name: "Uttarakhand",
    code: "IN-UT",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-UT",
};


pub const IN_WB: Subdivision = Subdivision {
    name: "West Bengal",
    code: "IN-WB",
    subdivision_type: "State",
    country_name: "India",
    country_code: "IN",
    region_code: "IN-WB",
};


pub const ID_AC: Subdivision = Subdivision {
    name: "Aceh",
    code: "ID-AC",
    subdivision_type: "Autonomous province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-AC",
};


pub const ID_BA: Subdivision = Subdivision {
    name: "Bali",
    code: "ID-BA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-BA",
};


pub const ID_BB: Subdivision = Subdivision {
    name: "Bangka Belitung",
    code: "ID-BB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-BB",
};


pub const ID_BE: Subdivision = Subdivision {
    name: "Bengkulu",
    code: "ID-BE",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-BE",
};


pub const ID_BT: Subdivision = Subdivision {
    name: "Banten",
    code: "ID-BT",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-BT",
};


pub const ID_GO: Subdivision = Subdivision {
    name: "Gorontalo",
    code: "ID-GO",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-GO",
};


pub const ID_JA: Subdivision = Subdivision {
    name: "Jambi",
    code: "ID-JA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JA",
};


pub const ID_JB: Subdivision = Subdivision {
    name: "Jawa Barat",
    code: "ID-JB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JB",
};


pub const ID_JI: Subdivision = Subdivision {
    name: "Jawa Timur",
    code: "ID-JI",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JI",
};


pub const ID_JK: Subdivision = Subdivision {
    name: "Jakarta Raya",
    code: "ID-JK",
    subdivision_type: "Special district",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JK",
};


pub const ID_JT: Subdivision = Subdivision {
    name: "Jawa Tengah",
    code: "ID-JT",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JT",
};


pub const ID_JW: Subdivision = Subdivision {
    name: "Jawa",
    code: "ID-JW",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-JW",
};


pub const ID_KA: Subdivision = Subdivision {
    name: "Kalimantan",
    code: "ID-KA",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KA",
};


pub const ID_KB: Subdivision = Subdivision {
    name: "Kalimantan Barat",
    code: "ID-KB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KB",
};


pub const ID_KI: Subdivision = Subdivision {
    name: "Kalimantan Timur",
    code: "ID-KI",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KI",
};


pub const ID_KR: Subdivision = Subdivision {
    name: "Kepulauan Riau",
    code: "ID-KR",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KR",
};


pub const ID_KS: Subdivision = Subdivision {
    name: "Kalimantan Selatan",
    code: "ID-KS",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KS",
};


pub const ID_KT: Subdivision = Subdivision {
    name: "Kalimantan Tengah",
    code: "ID-KT",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KT",
};


pub const ID_KU: Subdivision = Subdivision {
    name: "Kalimantan Utara",
    code: "ID-KU",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-KU",
};


pub const ID_LA: Subdivision = Subdivision {
    name: "Lampung",
    code: "ID-LA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-LA",
};


pub const ID_MA: Subdivision = Subdivision {
    name: "Maluku",
    code: "ID-MA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-MA",
};


pub const ID_ML: Subdivision = Subdivision {
    name: "Maluku",
    code: "ID-ML",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-ML",
};


pub const ID_MU: Subdivision = Subdivision {
    name: "Maluku Utara",
    code: "ID-MU",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-MU",
};


pub const ID_NB: Subdivision = Subdivision {
    name: "Nusa Tenggara Barat",
    code: "ID-NB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-NB",
};


pub const ID_NT: Subdivision = Subdivision {
    name: "Nusa Tenggara Timur",
    code: "ID-NT",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-NT",
};


pub const ID_NU: Subdivision = Subdivision {
    name: "Nusa Tenggara",
    code: "ID-NU",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-NU",
};


pub const ID_PA: Subdivision = Subdivision {
    name: "Papua",
    code: "ID-PA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-PA",
};


pub const ID_PB: Subdivision = Subdivision {
    name: "Papua Barat",
    code: "ID-PB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-PB",
};


pub const ID_PP: Subdivision = Subdivision {
    name: "Papua",
    code: "ID-PP",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-PP",
};


pub const ID_RI: Subdivision = Subdivision {
    name: "Riau",
    code: "ID-RI",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-RI",
};


pub const ID_SA: Subdivision = Subdivision {
    name: "Sulawesi Utara",
    code: "ID-SA",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SA",
};


pub const ID_SB: Subdivision = Subdivision {
    name: "Sumatera Barat",
    code: "ID-SB",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SB",
};


pub const ID_SG: Subdivision = Subdivision {
    name: "Sulawesi Tenggara",
    code: "ID-SG",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SG",
};


pub const ID_SL: Subdivision = Subdivision {
    name: "Sulawesi",
    code: "ID-SL",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SL",
};


pub const ID_SM: Subdivision = Subdivision {
    name: "Sumatera",
    code: "ID-SM",
    subdivision_type: "Geographical unit",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SM",
};


pub const ID_SN: Subdivision = Subdivision {
    name: "Sulawesi Selatan",
    code: "ID-SN",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SN",
};


pub const ID_SR: Subdivision = Subdivision {
    name: "Sulawesi Barat",
    code: "ID-SR",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SR",
};


pub const ID_SS: Subdivision = Subdivision {
    name: "Sumatera Selatan",
    code: "ID-SS",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SS",
};


pub const ID_ST: Subdivision = Subdivision {
    name: "Sulawesi Tengah",
    code: "ID-ST",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-ST",
};


pub const ID_SU: Subdivision = Subdivision {
    name: "Sumatera Utara",
    code: "ID-SU",
    subdivision_type: "Province",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-SU",
};


pub const ID_YO: Subdivision = Subdivision {
    name: "Yogyakarta",
    code: "ID-YO",
    subdivision_type: "Special region",
    country_name: "Indonesia",
    country_code: "ID",
    region_code: "ID-YO",
};


pub const IR_01: Subdivision = Subdivision {
    name: "Āz̄arbāyjān-e Sharqī",
    code: "IR-01",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-01",
};


pub const IR_02: Subdivision = Subdivision {
    name: "Āz̄arbāyjān-e Gharbī",
    code: "IR-02",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-02",
};


pub const IR_03: Subdivision = Subdivision {
    name: "Ardabīl",
    code: "IR-03",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-03",
};


pub const IR_04: Subdivision = Subdivision {
    name: "Eşfahān",
    code: "IR-04",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-04",
};


pub const IR_05: Subdivision = Subdivision {
    name: "Īlām",
    code: "IR-05",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-05",
};


pub const IR_06: Subdivision = Subdivision {
    name: "Būshehr",
    code: "IR-06",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-06",
};


pub const IR_07: Subdivision = Subdivision {
    name: "Tehrān",
    code: "IR-07",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-07",
};


pub const IR_08: Subdivision = Subdivision {
    name: "Chahār Maḩāll va Bakhtīārī",
    code: "IR-08",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-08",
};


pub const IR_10: Subdivision = Subdivision {
    name: "Khūzestān",
    code: "IR-10",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-10",
};


pub const IR_11: Subdivision = Subdivision {
    name: "Zanjān",
    code: "IR-11",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-11",
};


pub const IR_12: Subdivision = Subdivision {
    name: "Semnān",
    code: "IR-12",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-12",
};


pub const IR_13: Subdivision = Subdivision {
    name: "Sīstān va Balūchestān",
    code: "IR-13",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-13",
};


pub const IR_14: Subdivision = Subdivision {
    name: "Fārs",
    code: "IR-14",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-14",
};


pub const IR_15: Subdivision = Subdivision {
    name: "Kermān",
    code: "IR-15",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-15",
};


pub const IR_16: Subdivision = Subdivision {
    name: "Kordestān",
    code: "IR-16",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-16",
};


pub const IR_17: Subdivision = Subdivision {
    name: "Kermānshāh",
    code: "IR-17",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-17",
};


pub const IR_18: Subdivision = Subdivision {
    name: "Kohgīlūyeh va Būyer Aḩmad",
    code: "IR-18",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-18",
};


pub const IR_19: Subdivision = Subdivision {
    name: "Gīlān",
    code: "IR-19",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-19",
};


pub const IR_20: Subdivision = Subdivision {
    name: "Lorestān",
    code: "IR-20",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-20",
};


pub const IR_21: Subdivision = Subdivision {
    name: "Māzandarān",
    code: "IR-21",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-21",
};


pub const IR_22: Subdivision = Subdivision {
    name: "Markazī",
    code: "IR-22",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-22",
};


pub const IR_23: Subdivision = Subdivision {
    name: "Hormozgān",
    code: "IR-23",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-23",
};


pub const IR_24: Subdivision = Subdivision {
    name: "Hamadān",
    code: "IR-24",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-24",
};


pub const IR_25: Subdivision = Subdivision {
    name: "Yazd",
    code: "IR-25",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-25",
};


pub const IR_26: Subdivision = Subdivision {
    name: "Qom",
    code: "IR-26",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-26",
};


pub const IR_27: Subdivision = Subdivision {
    name: "Golestān",
    code: "IR-27",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-27",
};


pub const IR_28: Subdivision = Subdivision {
    name: "Qazvīn",
    code: "IR-28",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-28",
};


pub const IR_29: Subdivision = Subdivision {
    name: "Khorāsān-e Janūbī",
    code: "IR-29",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-29",
};


pub const IR_30: Subdivision = Subdivision {
    name: "Khorāsān-e Razavī",
    code: "IR-30",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-30",
};


pub const IR_31: Subdivision = Subdivision {
    name: "Khorāsān-e Shemālī",
    code: "IR-31",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-31",
};


pub const IR_32: Subdivision = Subdivision {
    name: "Alborz",
    code: "IR-32",
    subdivision_type: "Province",
    country_name: "Iran",
    country_code: "IR",
    region_code: "IR-32",
};


pub const IQ_AN: Subdivision = Subdivision {
    name: "Al Anbār",
    code: "IQ-AN",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-AN",
};


pub const IQ_AR: Subdivision = Subdivision {
    name: "Arbīl",
    code: "IQ-AR",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-AR",
};


pub const IQ_BA: Subdivision = Subdivision {
    name: "Al Başrah",
    code: "IQ-BA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-BA",
};


pub const IQ_BB: Subdivision = Subdivision {
    name: "Bābil",
    code: "IQ-BB",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-BB",
};


pub const IQ_BG: Subdivision = Subdivision {
    name: "Baghdād",
    code: "IQ-BG",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-BG",
};


pub const IQ_DA: Subdivision = Subdivision {
    name: "Dahūk",
    code: "IQ-DA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-DA",
};


pub const IQ_DI: Subdivision = Subdivision {
    name: "Diyālá",
    code: "IQ-DI",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-DI",
};


pub const IQ_DQ: Subdivision = Subdivision {
    name: "Dhī Qār",
    code: "IQ-DQ",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-DQ",
};


pub const IQ_KA: Subdivision = Subdivision {
    name: "Karbalā'",
    code: "IQ-KA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-KA",
};


pub const IQ_KI: Subdivision = Subdivision {
    name: "Kirkūk",
    code: "IQ-KI",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-KI",
};


pub const IQ_MA: Subdivision = Subdivision {
    name: "Maysān",
    code: "IQ-MA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-MA",
};


pub const IQ_MU: Subdivision = Subdivision {
    name: "Al Muthanná",
    code: "IQ-MU",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-MU",
};


pub const IQ_NA: Subdivision = Subdivision {
    name: "An Najaf",
    code: "IQ-NA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-NA",
};


pub const IQ_NI: Subdivision = Subdivision {
    name: "Nīnawá",
    code: "IQ-NI",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-NI",
};


pub const IQ_QA: Subdivision = Subdivision {
    name: "Al Qādisīyah",
    code: "IQ-QA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-QA",
};


pub const IQ_SD: Subdivision = Subdivision {
    name: "Şalāḩ ad Dīn",
    code: "IQ-SD",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-SD",
};


pub const IQ_SU: Subdivision = Subdivision {
    name: "As Sulaymānīyah",
    code: "IQ-SU",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-SU",
};


pub const IQ_WA: Subdivision = Subdivision {
    name: "Wāsiţ",
    code: "IQ-WA",
    subdivision_type: "Governorate",
    country_name: "Iraq",
    country_code: "IQ",
    region_code: "IQ-WA",
};


pub const IE_C: Subdivision = Subdivision {
    name: "Connacht",
    code: "IE-C",
    subdivision_type: "Province",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-C",
};


pub const IE_CE: Subdivision = Subdivision {
    name: "An Clár",
    code: "IE-CE",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-CE",
};


pub const IE_CN: Subdivision = Subdivision {
    name: "An Cabhán",
    code: "IE-CN",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-CN",
};


pub const IE_CO: Subdivision = Subdivision {
    name: "Corcaigh",
    code: "IE-CO",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-CO",
};


pub const IE_CW: Subdivision = Subdivision {
    name: "Carlow",
    code: "IE-CW",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-CW",
};


pub const IE_D: Subdivision = Subdivision {
    name: "Baile Átha Cliath",
    code: "IE-D",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-D",
};


pub const IE_DL: Subdivision = Subdivision {
    name: "Donegal",
    code: "IE-DL",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-DL",
};


pub const IE_G: Subdivision = Subdivision {
    name: "Gaillimh",
    code: "IE-G",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-G",
};


pub const IE_KE: Subdivision = Subdivision {
    name: "Cill Dara",
    code: "IE-KE",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-KE",
};


pub const IE_KK: Subdivision = Subdivision {
    name: "Cill Chainnigh",
    code: "IE-KK",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-KK",
};


pub const IE_KY: Subdivision = Subdivision {
    name: "Ciarraí",
    code: "IE-KY",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-KY",
};


pub const IE_LD: Subdivision = Subdivision {
    name: "An Longfort",
    code: "IE-LD",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-LD",
};


pub const IE_LH: Subdivision = Subdivision {
    name: "Louth",
    code: "IE-LH",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-LH",
};


pub const IE_LK: Subdivision = Subdivision {
    name: "Limerick",
    code: "IE-LK",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-LK",
};


pub const IE_L: Subdivision = Subdivision {
    name: "Laighin",
    code: "IE-L",
    subdivision_type: "Province",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-L",
};


pub const IE_LM: Subdivision = Subdivision {
    name: "Leitrim",
    code: "IE-LM",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-LM",
};


pub const IE_LS: Subdivision = Subdivision {
    name: "Laois",
    code: "IE-LS",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-LS",
};


pub const IE_M: Subdivision = Subdivision {
    name: "An Mhumhain",
    code: "IE-M",
    subdivision_type: "Province",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-M",
};


pub const IE_MH: Subdivision = Subdivision {
    name: "An Mhí",
    code: "IE-MH",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-MH",
};


pub const IE_MN: Subdivision = Subdivision {
    name: "Monaghan",
    code: "IE-MN",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-MN",
};


pub const IE_MO: Subdivision = Subdivision {
    name: "Maigh Eo",
    code: "IE-MO",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-MO",
};


pub const IE_OY: Subdivision = Subdivision {
    name: "Offaly",
    code: "IE-OY",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-OY",
};


pub const IE_RN: Subdivision = Subdivision {
    name: "Ros Comáin",
    code: "IE-RN",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-RN",
};


pub const IE_SO: Subdivision = Subdivision {
    name: "Sligeach",
    code: "IE-SO",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-SO",
};


pub const IE_TA: Subdivision = Subdivision {
    name: "Tiobraid Árann",
    code: "IE-TA",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-TA",
};


pub const IE_U: Subdivision = Subdivision {
    name: "Ulaidh",
    code: "IE-U",
    subdivision_type: "Province",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-U",
};


pub const IE_WD: Subdivision = Subdivision {
    name: "Port Láirge",
    code: "IE-WD",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-WD",
};


pub const IE_WH: Subdivision = Subdivision {
    name: "An Iarmhí",
    code: "IE-WH",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-WH",
};


pub const IE_WW: Subdivision = Subdivision {
    name: "Cill Mhantáin",
    code: "IE-WW",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-WW",
};


pub const IE_WX: Subdivision = Subdivision {
    name: "Loch Garman",
    code: "IE-WX",
    subdivision_type: "County",
    country_name: "Ireland",
    country_code: "IE",
    region_code: "IE-WX",
};


pub const IM_IM: Subdivision = Subdivision {
    name: "Isle of Main",
    code: "IM-IM",
    subdivision_type: "Country",
    country_name: "Isle of Man",
    country_code: "IM",
    region_code: "IM-IM",
};


pub const IL_D: Subdivision = Subdivision {
    name: "Southern District",
    code: "IL-D",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-D",
};


pub const IL_HA: Subdivision = Subdivision {
    name: "Haifa",
    code: "IL-HA",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-HA",
};


pub const IL_JM: Subdivision = Subdivision {
    name: "Jerusalem",
    code: "IL-JM",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-JM",
};


pub const IL_M: Subdivision = Subdivision {
    name: "Central District",
    code: "IL-M",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-M",
};


pub const IL_TA: Subdivision = Subdivision {
    name: "Tel Aviv",
    code: "IL-TA",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-TA",
};


pub const IL_Z: Subdivision = Subdivision {
    name: "Northern District",
    code: "IL-Z",
    subdivision_type: "District",
    country_name: "Israel",
    country_code: "IL",
    region_code: "IL-Z",
};


pub const IT_21: Subdivision = Subdivision {
    name: "Piemonte",
    code: "IT-21",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-21",
};


pub const IT_23: Subdivision = Subdivision {
    name: "Val d'Aoste",
    code: "IT-23",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-23",
};


pub const IT_25: Subdivision = Subdivision {
    name: "Lombardia",
    code: "IT-25",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-25",
};


pub const IT_32: Subdivision = Subdivision {
    name: "Trentino-Alto Adige",
    code: "IT-32",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-32",
};


pub const IT_34: Subdivision = Subdivision {
    name: "Veneto",
    code: "IT-34",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-34",
};


pub const IT_36: Subdivision = Subdivision {
    name: "Friuli-Venezia Giulia",
    code: "IT-36",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-36",
};


pub const IT_42: Subdivision = Subdivision {
    name: "Liguria",
    code: "IT-42",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-42",
};


pub const IT_45: Subdivision = Subdivision {
    name: "Emilia-Romagna",
    code: "IT-45",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-45",
};


pub const IT_52: Subdivision = Subdivision {
    name: "Toscana",
    code: "IT-52",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-52",
};


pub const IT_55: Subdivision = Subdivision {
    name: "Umbria",
    code: "IT-55",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-55",
};


pub const IT_57: Subdivision = Subdivision {
    name: "Marche",
    code: "IT-57",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-57",
};


pub const IT_62: Subdivision = Subdivision {
    name: "Lazio",
    code: "IT-62",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-62",
};


pub const IT_65: Subdivision = Subdivision {
    name: "Abruzzo",
    code: "IT-65",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-65",
};


pub const IT_67: Subdivision = Subdivision {
    name: "Molise",
    code: "IT-67",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-67",
};


pub const IT_72: Subdivision = Subdivision {
    name: "Campania",
    code: "IT-72",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-72",
};


pub const IT_75: Subdivision = Subdivision {
    name: "Puglia",
    code: "IT-75",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-75",
};


pub const IT_77: Subdivision = Subdivision {
    name: "Basilicata",
    code: "IT-77",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-77",
};


pub const IT_78: Subdivision = Subdivision {
    name: "Calabria",
    code: "IT-78",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-78",
};


pub const IT_82: Subdivision = Subdivision {
    name: "Sicilia",
    code: "IT-82",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-82",
};


pub const IT_88: Subdivision = Subdivision {
    name: "Sardegna",
    code: "IT-88",
    subdivision_type: "Region",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-88",
};


pub const IT_AG: Subdivision = Subdivision {
    name: "Agrigento",
    code: "IT-AG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AG",
};


pub const IT_AL: Subdivision = Subdivision {
    name: "Alessandria",
    code: "IT-AL",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AL",
};


pub const IT_AN: Subdivision = Subdivision {
    name: "Ancona",
    code: "IT-AN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AN",
};


pub const IT_AO: Subdivision = Subdivision {
    name: "Aosta",
    code: "IT-AO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AO",
};


pub const IT_AP: Subdivision = Subdivision {
    name: "Ascoli Piceno",
    code: "IT-AP",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AP",
};


pub const IT_AQ: Subdivision = Subdivision {
    name: "L'Aquila",
    code: "IT-AQ",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AQ",
};


pub const IT_AR: Subdivision = Subdivision {
    name: "Arezzo",
    code: "IT-AR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AR",
};


pub const IT_AT: Subdivision = Subdivision {
    name: "Asti",
    code: "IT-AT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AT",
};


pub const IT_AV: Subdivision = Subdivision {
    name: "Avellino",
    code: "IT-AV",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-AV",
};


pub const IT_BA: Subdivision = Subdivision {
    name: "Bari",
    code: "IT-BA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BA",
};


pub const IT_BG: Subdivision = Subdivision {
    name: "Bergamo",
    code: "IT-BG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BG",
};


pub const IT_BI: Subdivision = Subdivision {
    name: "Biella",
    code: "IT-BI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BI",
};


pub const IT_BL: Subdivision = Subdivision {
    name: "Belluno",
    code: "IT-BL",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BL",
};


pub const IT_BN: Subdivision = Subdivision {
    name: "Benevento",
    code: "IT-BN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BN",
};


pub const IT_BO: Subdivision = Subdivision {
    name: "Bologna",
    code: "IT-BO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BO",
};


pub const IT_BR: Subdivision = Subdivision {
    name: "Brindisi",
    code: "IT-BR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BR",
};


pub const IT_BS: Subdivision = Subdivision {
    name: "Brescia",
    code: "IT-BS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BS",
};


pub const IT_BT: Subdivision = Subdivision {
    name: "Barletta-Andria-Trani",
    code: "IT-BT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BT",
};


pub const IT_BZ: Subdivision = Subdivision {
    name: "Bolzano",
    code: "IT-BZ",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-BZ",
};


pub const IT_CA: Subdivision = Subdivision {
    name: "Cagliari",
    code: "IT-CA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CA",
};


pub const IT_CB: Subdivision = Subdivision {
    name: "Campobasso",
    code: "IT-CB",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CB",
};


pub const IT_CE: Subdivision = Subdivision {
    name: "Caserta",
    code: "IT-CE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CE",
};


pub const IT_CH: Subdivision = Subdivision {
    name: "Chieti",
    code: "IT-CH",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CH",
};


pub const IT_CI: Subdivision = Subdivision {
    name: "Carbonia-Iglesias",
    code: "IT-CI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CI",
};


pub const IT_CL: Subdivision = Subdivision {
    name: "Caltanissetta",
    code: "IT-CL",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CL",
};


pub const IT_CN: Subdivision = Subdivision {
    name: "Cuneo",
    code: "IT-CN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CN",
};


pub const IT_CO: Subdivision = Subdivision {
    name: "Como",
    code: "IT-CO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CO",
};


pub const IT_CR: Subdivision = Subdivision {
    name: "Cremona",
    code: "IT-CR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CR",
};


pub const IT_CS: Subdivision = Subdivision {
    name: "Cosenza",
    code: "IT-CS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CS",
};


pub const IT_CT: Subdivision = Subdivision {
    name: "Catania",
    code: "IT-CT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CT",
};


pub const IT_CZ: Subdivision = Subdivision {
    name: "Catanzaro",
    code: "IT-CZ",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-CZ",
};


pub const IT_EN: Subdivision = Subdivision {
    name: "Enna",
    code: "IT-EN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-EN",
};


pub const IT_FC: Subdivision = Subdivision {
    name: "Forlì-Cesena",
    code: "IT-FC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FC",
};


pub const IT_FE: Subdivision = Subdivision {
    name: "Ferrara",
    code: "IT-FE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FE",
};


pub const IT_FG: Subdivision = Subdivision {
    name: "Foggia",
    code: "IT-FG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FG",
};


pub const IT_FI: Subdivision = Subdivision {
    name: "Firenze",
    code: "IT-FI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FI",
};


pub const IT_FM: Subdivision = Subdivision {
    name: "Fermo",
    code: "IT-FM",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FM",
};


pub const IT_FR: Subdivision = Subdivision {
    name: "Frosinone",
    code: "IT-FR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-FR",
};


pub const IT_GE: Subdivision = Subdivision {
    name: "Genova",
    code: "IT-GE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-GE",
};


pub const IT_GO: Subdivision = Subdivision {
    name: "Gorizia",
    code: "IT-GO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-GO",
};


pub const IT_GR: Subdivision = Subdivision {
    name: "Grosseto",
    code: "IT-GR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-GR",
};


pub const IT_IM: Subdivision = Subdivision {
    name: "Imperia",
    code: "IT-IM",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-IM",
};


pub const IT_IS: Subdivision = Subdivision {
    name: "Isernia",
    code: "IT-IS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-IS",
};


pub const IT_KR: Subdivision = Subdivision {
    name: "Crotone",
    code: "IT-KR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-KR",
};


pub const IT_LC: Subdivision = Subdivision {
    name: "Lecco",
    code: "IT-LC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LC",
};


pub const IT_LE: Subdivision = Subdivision {
    name: "Lecce",
    code: "IT-LE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LE",
};


pub const IT_LI: Subdivision = Subdivision {
    name: "Livorno",
    code: "IT-LI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LI",
};


pub const IT_LO: Subdivision = Subdivision {
    name: "Lodi",
    code: "IT-LO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LO",
};


pub const IT_LT: Subdivision = Subdivision {
    name: "Latina",
    code: "IT-LT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LT",
};


pub const IT_LU: Subdivision = Subdivision {
    name: "Lucca",
    code: "IT-LU",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-LU",
};


pub const IT_MB: Subdivision = Subdivision {
    name: "Monza e Brianza",
    code: "IT-MB",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MB",
};


pub const IT_MC: Subdivision = Subdivision {
    name: "Macerata",
    code: "IT-MC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MC",
};


pub const IT_ME: Subdivision = Subdivision {
    name: "Messina",
    code: "IT-ME",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-ME",
};


pub const IT_MI: Subdivision = Subdivision {
    name: "Milano",
    code: "IT-MI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MI",
};


pub const IT_MN: Subdivision = Subdivision {
    name: "Mantova",
    code: "IT-MN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MN",
};


pub const IT_MO: Subdivision = Subdivision {
    name: "Modena",
    code: "IT-MO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MO",
};


pub const IT_MS: Subdivision = Subdivision {
    name: "Massa-Carrara",
    code: "IT-MS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MS",
};


pub const IT_MT: Subdivision = Subdivision {
    name: "Matera",
    code: "IT-MT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-MT",
};


pub const IT_NA: Subdivision = Subdivision {
    name: "Napoli",
    code: "IT-NA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-NA",
};


pub const IT_NO: Subdivision = Subdivision {
    name: "Novara",
    code: "IT-NO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-NO",
};


pub const IT_NU: Subdivision = Subdivision {
    name: "Nuoro",
    code: "IT-NU",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-NU",
};


pub const IT_OG: Subdivision = Subdivision {
    name: "Ogliastra",
    code: "IT-OG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-OG",
};


pub const IT_OR: Subdivision = Subdivision {
    name: "Oristano",
    code: "IT-OR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-OR",
};


pub const IT_OT: Subdivision = Subdivision {
    name: "Olbia-Tempio",
    code: "IT-OT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-OT",
};


pub const IT_PA: Subdivision = Subdivision {
    name: "Palermo",
    code: "IT-PA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PA",
};


pub const IT_PC: Subdivision = Subdivision {
    name: "Piacenza",
    code: "IT-PC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PC",
};


pub const IT_PD: Subdivision = Subdivision {
    name: "Padova",
    code: "IT-PD",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PD",
};


pub const IT_PE: Subdivision = Subdivision {
    name: "Pescara",
    code: "IT-PE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PE",
};


pub const IT_PG: Subdivision = Subdivision {
    name: "Perugia",
    code: "IT-PG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PG",
};


pub const IT_PI: Subdivision = Subdivision {
    name: "Pisa",
    code: "IT-PI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PI",
};


pub const IT_PN: Subdivision = Subdivision {
    name: "Pordenone",
    code: "IT-PN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PN",
};


pub const IT_PO: Subdivision = Subdivision {
    name: "Prato",
    code: "IT-PO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PO",
};


pub const IT_PR: Subdivision = Subdivision {
    name: "Parma",
    code: "IT-PR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PR",
};


pub const IT_PT: Subdivision = Subdivision {
    name: "Pistoia",
    code: "IT-PT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PT",
};


pub const IT_PU: Subdivision = Subdivision {
    name: "Pesaro e Urbino",
    code: "IT-PU",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PU",
};


pub const IT_PV: Subdivision = Subdivision {
    name: "Pavia",
    code: "IT-PV",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PV",
};


pub const IT_PZ: Subdivision = Subdivision {
    name: "Potenza",
    code: "IT-PZ",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-PZ",
};


pub const IT_RA: Subdivision = Subdivision {
    name: "Ravenna",
    code: "IT-RA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RA",
};


pub const IT_RC: Subdivision = Subdivision {
    name: "Reggio Calabria",
    code: "IT-RC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RC",
};


pub const IT_RE: Subdivision = Subdivision {
    name: "Reggio Emilia",
    code: "IT-RE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RE",
};


pub const IT_RG: Subdivision = Subdivision {
    name: "Ragusa",
    code: "IT-RG",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RG",
};


pub const IT_RI: Subdivision = Subdivision {
    name: "Rieti",
    code: "IT-RI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RI",
};


pub const IT_RM: Subdivision = Subdivision {
    name: "Roma",
    code: "IT-RM",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RM",
};


pub const IT_RN: Subdivision = Subdivision {
    name: "Rimini",
    code: "IT-RN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RN",
};


pub const IT_RO: Subdivision = Subdivision {
    name: "Rovigo",
    code: "IT-RO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-RO",
};


pub const IT_SA: Subdivision = Subdivision {
    name: "Salerno",
    code: "IT-SA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SA",
};


pub const IT_SI: Subdivision = Subdivision {
    name: "Siena",
    code: "IT-SI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SI",
};


pub const IT_SO: Subdivision = Subdivision {
    name: "Sondrio",
    code: "IT-SO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SO",
};


pub const IT_SP: Subdivision = Subdivision {
    name: "La Spezia",
    code: "IT-SP",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SP",
};


pub const IT_SR: Subdivision = Subdivision {
    name: "Siracusa",
    code: "IT-SR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SR",
};


pub const IT_SS: Subdivision = Subdivision {
    name: "Sassari",
    code: "IT-SS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SS",
};


pub const IT_SV: Subdivision = Subdivision {
    name: "Savona",
    code: "IT-SV",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-SV",
};


pub const IT_TA: Subdivision = Subdivision {
    name: "Taranto",
    code: "IT-TA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TA",
};


pub const IT_TE: Subdivision = Subdivision {
    name: "Teramo",
    code: "IT-TE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TE",
};


pub const IT_TN: Subdivision = Subdivision {
    name: "Trento",
    code: "IT-TN",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TN",
};


pub const IT_TO: Subdivision = Subdivision {
    name: "Torino",
    code: "IT-TO",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TO",
};


pub const IT_TP: Subdivision = Subdivision {
    name: "Trapani",
    code: "IT-TP",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TP",
};


pub const IT_TR: Subdivision = Subdivision {
    name: "Terni",
    code: "IT-TR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TR",
};


pub const IT_TS: Subdivision = Subdivision {
    name: "Trieste",
    code: "IT-TS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TS",
};


pub const IT_TV: Subdivision = Subdivision {
    name: "Treviso",
    code: "IT-TV",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-TV",
};


pub const IT_UD: Subdivision = Subdivision {
    name: "Udine",
    code: "IT-UD",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-UD",
};


pub const IT_VA: Subdivision = Subdivision {
    name: "Varese",
    code: "IT-VA",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VA",
};


pub const IT_VB: Subdivision = Subdivision {
    name: "Verbano-Cusio-Ossola",
    code: "IT-VB",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VB",
};


pub const IT_VC: Subdivision = Subdivision {
    name: "Vercelli",
    code: "IT-VC",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VC",
};


pub const IT_VE: Subdivision = Subdivision {
    name: "Venezia",
    code: "IT-VE",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VE",
};


pub const IT_VI: Subdivision = Subdivision {
    name: "Vicenza",
    code: "IT-VI",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VI",
};


pub const IT_VR: Subdivision = Subdivision {
    name: "Verona",
    code: "IT-VR",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VR",
};


pub const IT_VS: Subdivision = Subdivision {
    name: "Medio Campidano",
    code: "IT-VS",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VS",
};


pub const IT_VT: Subdivision = Subdivision {
    name: "Viterbo",
    code: "IT-VT",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VT",
};


pub const IT_VV: Subdivision = Subdivision {
    name: "Vibo Valentia",
    code: "IT-VV",
    subdivision_type: "Province",
    country_name: "Italy",
    country_code: "IT",
    region_code: "IT-VV",
};


pub const JM_01: Subdivision = Subdivision {
    name: "Kingston",
    code: "JM-01",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-01",
};


pub const JM_02: Subdivision = Subdivision {
    name: "Saint Andrew",
    code: "JM-02",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-02",
};


pub const JM_03: Subdivision = Subdivision {
    name: "Saint Thomas",
    code: "JM-03",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-03",
};


pub const JM_04: Subdivision = Subdivision {
    name: "Portland",
    code: "JM-04",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-04",
};


pub const JM_05: Subdivision = Subdivision {
    name: "Saint Mary",
    code: "JM-05",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-05",
};


pub const JM_06: Subdivision = Subdivision {
    name: "Saint Ann",
    code: "JM-06",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-06",
};


pub const JM_07: Subdivision = Subdivision {
    name: "Trelawny",
    code: "JM-07",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-07",
};


pub const JM_08: Subdivision = Subdivision {
    name: "Saint James",
    code: "JM-08",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-08",
};


pub const JM_09: Subdivision = Subdivision {
    name: "Hanover",
    code: "JM-09",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-09",
};


pub const JM_10: Subdivision = Subdivision {
    name: "Westmoreland",
    code: "JM-10",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-10",
};


pub const JM_11: Subdivision = Subdivision {
    name: "Saint Elizabeth",
    code: "JM-11",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-11",
};


pub const JM_12: Subdivision = Subdivision {
    name: "Manchester",
    code: "JM-12",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-12",
};


pub const JM_13: Subdivision = Subdivision {
    name: "Clarendon",
    code: "JM-13",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-13",
};


pub const JM_14: Subdivision = Subdivision {
    name: "Saint Catherine",
    code: "JM-14",
    subdivision_type: "Parish",
    country_name: "Jamaica",
    country_code: "JM",
    region_code: "JM-14",
};


pub const JP_01: Subdivision = Subdivision {
    name: "Hokkaido",
    code: "JP-01",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-01",
};


pub const JP_02: Subdivision = Subdivision {
    name: "Aomori",
    code: "JP-02",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-02",
};


pub const JP_03: Subdivision = Subdivision {
    name: "Iwate",
    code: "JP-03",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-03",
};


pub const JP_04: Subdivision = Subdivision {
    name: "Miyagi",
    code: "JP-04",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-04",
};


pub const JP_05: Subdivision = Subdivision {
    name: "Akita",
    code: "JP-05",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-05",
};


pub const JP_06: Subdivision = Subdivision {
    name: "Yamagata",
    code: "JP-06",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-06",
};


pub const JP_07: Subdivision = Subdivision {
    name: "Fukushima",
    code: "JP-07",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-07",
};


pub const JP_08: Subdivision = Subdivision {
    name: "Ibaraki",
    code: "JP-08",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-08",
};


pub const JP_09: Subdivision = Subdivision {
    name: "Tochigi",
    code: "JP-09",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-09",
};


pub const JP_10: Subdivision = Subdivision {
    name: "Gunma",
    code: "JP-10",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-10",
};


pub const JP_11: Subdivision = Subdivision {
    name: "Saitama",
    code: "JP-11",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-11",
};


pub const JP_12: Subdivision = Subdivision {
    name: "Chiba",
    code: "JP-12",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-12",
};


pub const JP_13: Subdivision = Subdivision {
    name: "Tôkyô",
    code: "JP-13",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-13",
};


pub const JP_14: Subdivision = Subdivision {
    name: "Kanagawa",
    code: "JP-14",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-14",
};


pub const JP_15: Subdivision = Subdivision {
    name: "Niigata",
    code: "JP-15",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-15",
};


pub const JP_16: Subdivision = Subdivision {
    name: "Toyama",
    code: "JP-16",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-16",
};


pub const JP_17: Subdivision = Subdivision {
    name: "Ishikawa",
    code: "JP-17",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-17",
};


pub const JP_18: Subdivision = Subdivision {
    name: "Fukui",
    code: "JP-18",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-18",
};


pub const JP_19: Subdivision = Subdivision {
    name: "Yamanashi",
    code: "JP-19",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-19",
};


pub const JP_20: Subdivision = Subdivision {
    name: "Nagano",
    code: "JP-20",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-20",
};


pub const JP_21: Subdivision = Subdivision {
    name: "Gifu",
    code: "JP-21",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-21",
};


pub const JP_22: Subdivision = Subdivision {
    name: "Shizuoka",
    code: "JP-22",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-22",
};


pub const JP_23: Subdivision = Subdivision {
    name: "Aichi",
    code: "JP-23",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-23",
};


pub const JP_24: Subdivision = Subdivision {
    name: "Mie",
    code: "JP-24",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-24",
};


pub const JP_25: Subdivision = Subdivision {
    name: "Shiga",
    code: "JP-25",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-25",
};


pub const JP_26: Subdivision = Subdivision {
    name: "Kyoto",
    code: "JP-26",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-26",
};


pub const JP_27: Subdivision = Subdivision {
    name: "Osaka",
    code: "JP-27",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-27",
};


pub const JP_28: Subdivision = Subdivision {
    name: "Hyogo",
    code: "JP-28",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-28",
};


pub const JP_29: Subdivision = Subdivision {
    name: "Nara",
    code: "JP-29",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-29",
};


pub const JP_30: Subdivision = Subdivision {
    name: "Wakayama",
    code: "JP-30",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-30",
};


pub const JP_31: Subdivision = Subdivision {
    name: "Tottori",
    code: "JP-31",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-31",
};


pub const JP_32: Subdivision = Subdivision {
    name: "Shimane",
    code: "JP-32",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-32",
};


pub const JP_33: Subdivision = Subdivision {
    name: "Okayama",
    code: "JP-33",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-33",
};


pub const JP_34: Subdivision = Subdivision {
    name: "Hiroshima",
    code: "JP-34",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-34",
};


pub const JP_35: Subdivision = Subdivision {
    name: "Yamaguchi",
    code: "JP-35",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-35",
};


pub const JP_36: Subdivision = Subdivision {
    name: "Tokushima",
    code: "JP-36",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-36",
};


pub const JP_37: Subdivision = Subdivision {
    name: "Kagawa",
    code: "JP-37",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-37",
};


pub const JP_38: Subdivision = Subdivision {
    name: "Ehime",
    code: "JP-38",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-38",
};


pub const JP_39: Subdivision = Subdivision {
    name: "Kochi",
    code: "JP-39",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-39",
};


pub const JP_40: Subdivision = Subdivision {
    name: "Fukuoka",
    code: "JP-40",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-40",
};


pub const JP_41: Subdivision = Subdivision {
    name: "Saga",
    code: "JP-41",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-41",
};


pub const JP_42: Subdivision = Subdivision {
    name: "Nagasaki",
    code: "JP-42",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-42",
};


pub const JP_43: Subdivision = Subdivision {
    name: "Kumamoto",
    code: "JP-43",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-43",
};


pub const JP_44: Subdivision = Subdivision {
    name: "Oita",
    code: "JP-44",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-44",
};


pub const JP_45: Subdivision = Subdivision {
    name: "Miyazaki",
    code: "JP-45",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-45",
};


pub const JP_46: Subdivision = Subdivision {
    name: "Kagoshima",
    code: "JP-46",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-46",
};


pub const JP_47: Subdivision = Subdivision {
    name: "Okinawa",
    code: "JP-47",
    subdivision_type: "Prefecture",
    country_name: "Japan",
    country_code: "JP",
    region_code: "JP-47",
};


pub const JE_JE: Subdivision = Subdivision {
    name: "Jersey",
    code: "JE-JE",
    subdivision_type: "Country",
    country_name: "Jersey",
    country_code: "JE",
    region_code: "JE-JE",
};


pub const JO_AJ: Subdivision = Subdivision {
    name: "ʽAjlūn",
    code: "JO-AJ",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-AJ",
};


pub const JO_AM: Subdivision = Subdivision {
    name: "‘Ammān",
    code: "JO-AM",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-AM",
};


pub const JO_AQ: Subdivision = Subdivision {
    name: "Al ʽAqabah",
    code: "JO-AQ",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-AQ",
};


pub const JO_AT: Subdivision = Subdivision {
    name: "Aţ Ţafīlah",
    code: "JO-AT",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-AT",
};


pub const JO_AZ: Subdivision = Subdivision {
    name: "Az Zarqā'",
    code: "JO-AZ",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-AZ",
};


pub const JO_BA: Subdivision = Subdivision {
    name: "Al Balqā'",
    code: "JO-BA",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-BA",
};


pub const JO_IR: Subdivision = Subdivision {
    name: "Irbid",
    code: "JO-IR",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-IR",
};


pub const JO_JA: Subdivision = Subdivision {
    name: "Jarash",
    code: "JO-JA",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-JA",
};


pub const JO_KA: Subdivision = Subdivision {
    name: "Al Karak",
    code: "JO-KA",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-KA",
};


pub const JO_MA: Subdivision = Subdivision {
    name: "Al Mafraq",
    code: "JO-MA",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-MA",
};


pub const JO_MD: Subdivision = Subdivision {
    name: "Mādabā",
    code: "JO-MD",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-MD",
};


pub const JO_MN: Subdivision = Subdivision {
    name: "Ma'ān",
    code: "JO-MN",
    subdivision_type: "Governorate",
    country_name: "Jordan",
    country_code: "JO",
    region_code: "JO-MN",
};


pub const KZ_AKM: Subdivision = Subdivision {
    name: "Akmolinskaja oblast'",
    code: "KZ-AKM",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-AKM",
};


pub const KZ_AKT: Subdivision = Subdivision {
    name: "Aktjubinskaja oblast'",
    code: "KZ-AKT",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-AKT",
};


pub const KZ_ALA: Subdivision = Subdivision {
    name: "Almaty",
    code: "KZ-ALA",
    subdivision_type: "City",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-ALA",
};


pub const KZ_ALM: Subdivision = Subdivision {
    name: "Almatinskaja oblast'",
    code: "KZ-ALM",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-ALM",
};


pub const KZ_AST: Subdivision = Subdivision {
    name: "Astana",
    code: "KZ-AST",
    subdivision_type: "City",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-AST",
};


pub const KZ_ATY: Subdivision = Subdivision {
    name: "Atyraū oblysy",
    code: "KZ-ATY",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-ATY",
};


pub const KZ_KAR: Subdivision = Subdivision {
    name: "Karagandinskaja oblast'",
    code: "KZ-KAR",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-KAR",
};


pub const KZ_KUS: Subdivision = Subdivision {
    name: "Kostanajskaja oblast'",
    code: "KZ-KUS",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-KUS",
};


pub const KZ_KZY: Subdivision = Subdivision {
    name: "Kyzylordinskaja oblast'",
    code: "KZ-KZY",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-KZY",
};


pub const KZ_MAN: Subdivision = Subdivision {
    name: "Mangghystaū oblysy",
    code: "KZ-MAN",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-MAN",
};


pub const KZ_PAV: Subdivision = Subdivision {
    name: "Pavlodar oblysy",
    code: "KZ-PAV",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-PAV",
};


pub const KZ_SEV: Subdivision = Subdivision {
    name: "Severo-Kazahstanskaja oblast'",
    code: "KZ-SEV",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-SEV",
};


pub const KZ_VOS: Subdivision = Subdivision {
    name: "Shyghys Qazaqstan oblysy",
    code: "KZ-VOS",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-VOS",
};


pub const KZ_YUZ: Subdivision = Subdivision {
    name: "Južno-Kazahstanskaja oblast'",
    code: "KZ-YUZ",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-YUZ",
};


pub const KZ_ZAP: Subdivision = Subdivision {
    name: "Batys Qazaqstan oblysy",
    code: "KZ-ZAP",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-ZAP",
};


pub const KZ_ZHA: Subdivision = Subdivision {
    name: "Žambylskaja oblast'",
    code: "KZ-ZHA",
    subdivision_type: "Region",
    country_name: "Kazakhstan",
    country_code: "KZ",
    region_code: "KZ-ZHA",
};


pub const KE_01: Subdivision = Subdivision {
    name: "Baringo",
    code: "KE-01",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-01",
};


pub const KE_02: Subdivision = Subdivision {
    name: "Bomet",
    code: "KE-02",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-02",
};


pub const KE_03: Subdivision = Subdivision {
    name: "Bungoma",
    code: "KE-03",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-03",
};


pub const KE_04: Subdivision = Subdivision {
    name: "Busia",
    code: "KE-04",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-04",
};


pub const KE_05: Subdivision = Subdivision {
    name: "Elgeyo/Marakwet",
    code: "KE-05",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-05",
};


pub const KE_06: Subdivision = Subdivision {
    name: "Embu",
    code: "KE-06",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-06",
};


pub const KE_07: Subdivision = Subdivision {
    name: "Garissa",
    code: "KE-07",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-07",
};


pub const KE_08: Subdivision = Subdivision {
    name: "Homa Bay",
    code: "KE-08",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-08",
};


pub const KE_09: Subdivision = Subdivision {
    name: "Isiolo",
    code: "KE-09",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-09",
};


pub const KE_10: Subdivision = Subdivision {
    name: "Kajiado",
    code: "KE-10",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-10",
};


pub const KE_11: Subdivision = Subdivision {
    name: "Kakamega",
    code: "KE-11",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-11",
};


pub const KE_12: Subdivision = Subdivision {
    name: "Kericho",
    code: "KE-12",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-12",
};


pub const KE_13: Subdivision = Subdivision {
    name: "Kiambu",
    code: "KE-13",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-13",
};


pub const KE_14: Subdivision = Subdivision {
    name: "Kilifi",
    code: "KE-14",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-14",
};


pub const KE_15: Subdivision = Subdivision {
    name: "Kirinyaga",
    code: "KE-15",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-15",
};


pub const KE_16: Subdivision = Subdivision {
    name: "Kisii",
    code: "KE-16",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-16",
};


pub const KE_17: Subdivision = Subdivision {
    name: "Kisumu",
    code: "KE-17",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-17",
};


pub const KE_18: Subdivision = Subdivision {
    name: "Kitui",
    code: "KE-18",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-18",
};


pub const KE_19: Subdivision = Subdivision {
    name: "Kwale",
    code: "KE-19",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-19",
};


pub const KE_20: Subdivision = Subdivision {
    name: "Laikipia",
    code: "KE-20",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-20",
};


pub const KE_21: Subdivision = Subdivision {
    name: "Lamu",
    code: "KE-21",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-21",
};


pub const KE_22: Subdivision = Subdivision {
    name: "Machakos",
    code: "KE-22",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-22",
};


pub const KE_23: Subdivision = Subdivision {
    name: "Makueni",
    code: "KE-23",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-23",
};


pub const KE_24: Subdivision = Subdivision {
    name: "Mandera",
    code: "KE-24",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-24",
};


pub const KE_25: Subdivision = Subdivision {
    name: "Marsabit",
    code: "KE-25",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-25",
};


pub const KE_26: Subdivision = Subdivision {
    name: "Meru",
    code: "KE-26",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-26",
};


pub const KE_27: Subdivision = Subdivision {
    name: "Migori",
    code: "KE-27",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-27",
};


pub const KE_28: Subdivision = Subdivision {
    name: "Mombasa",
    code: "KE-28",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-28",
};


pub const KE_29: Subdivision = Subdivision {
    name: "Murang'a",
    code: "KE-29",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-29",
};


pub const KE_30: Subdivision = Subdivision {
    name: "Nairobi City",
    code: "KE-30",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-30",
};


pub const KE_31: Subdivision = Subdivision {
    name: "Nakuru",
    code: "KE-31",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-31",
};


pub const KE_32: Subdivision = Subdivision {
    name: "Nandi",
    code: "KE-32",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-32",
};


pub const KE_33: Subdivision = Subdivision {
    name: "Narok",
    code: "KE-33",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-33",
};


pub const KE_34: Subdivision = Subdivision {
    name: "Nyamira",
    code: "KE-34",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-34",
};


pub const KE_35: Subdivision = Subdivision {
    name: "Nyandarua",
    code: "KE-35",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-35",
};


pub const KE_36: Subdivision = Subdivision {
    name: "Nyeri",
    code: "KE-36",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-36",
};


pub const KE_37: Subdivision = Subdivision {
    name: "Samburu",
    code: "KE-37",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-37",
};


pub const KE_38: Subdivision = Subdivision {
    name: "Siaya",
    code: "KE-38",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-38",
};


pub const KE_39: Subdivision = Subdivision {
    name: "Taita/Taveta",
    code: "KE-39",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-39",
};


pub const KE_40: Subdivision = Subdivision {
    name: "Tana River",
    code: "KE-40",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-40",
};


pub const KE_41: Subdivision = Subdivision {
    name: "Tharaka-Nithi",
    code: "KE-41",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-41",
};


pub const KE_42: Subdivision = Subdivision {
    name: "Trans Nzoia",
    code: "KE-42",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-42",
};


pub const KE_43: Subdivision = Subdivision {
    name: "Turkana",
    code: "KE-43",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-43",
};


pub const KE_44: Subdivision = Subdivision {
    name: "Uasin Gishu",
    code: "KE-44",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-44",
};


pub const KE_45: Subdivision = Subdivision {
    name: "Vihiga",
    code: "KE-45",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-45",
};


pub const KE_46: Subdivision = Subdivision {
    name: "Wajir",
    code: "KE-46",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-46",
};


pub const KE_47: Subdivision = Subdivision {
    name: "West Pokot",
    code: "KE-47",
    subdivision_type: "County",
    country_name: "Kenya",
    country_code: "KE",
    region_code: "KE-47",
};


pub const KI_G: Subdivision = Subdivision {
    name: "Gilbert Islands",
    code: "KI-G",
    subdivision_type: "Group of islands",
    country_name: "Kiribati",
    country_code: "KI",
    region_code: "KI-G",
};


pub const KI_L: Subdivision = Subdivision {
    name: "Line Islands",
    code: "KI-L",
    subdivision_type: "Group of islands",
    country_name: "Kiribati",
    country_code: "KI",
    region_code: "KI-L",
};


pub const KI_P: Subdivision = Subdivision {
    name: "Phoenix Islands",
    code: "KI-P",
    subdivision_type: "Group of islands",
    country_name: "Kiribati",
    country_code: "KI",
    region_code: "KI-P",
};


pub const KP_01: Subdivision = Subdivision {
    name: "P'yǒngyang",
    code: "KP-01",
    subdivision_type: "Capital city",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-01",
};


pub const KP_02: Subdivision = Subdivision {
    name: "P'yǒngan-namdo",
    code: "KP-02",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-02",
};


pub const KP_03: Subdivision = Subdivision {
    name: "P'yǒngan-bukto",
    code: "KP-03",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-03",
};


pub const KP_04: Subdivision = Subdivision {
    name: "Chagang-do",
    code: "KP-04",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-04",
};


pub const KP_05: Subdivision = Subdivision {
    name: "Hwanghae-namdo",
    code: "KP-05",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-05",
};


pub const KP_06: Subdivision = Subdivision {
    name: "Hwanghae-bukto",
    code: "KP-06",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-06",
};


pub const KP_07: Subdivision = Subdivision {
    name: "Kangweonto",
    code: "KP-07",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-07",
};


pub const KP_08: Subdivision = Subdivision {
    name: "Hamgyǒng-namdo",
    code: "KP-08",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-08",
};


pub const KP_09: Subdivision = Subdivision {
    name: "Hamgyǒng-bukto",
    code: "KP-09",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-09",
};


pub const KP_10: Subdivision = Subdivision {
    name: "Ryangkangto",
    code: "KP-10",
    subdivision_type: "Province",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-10",
};


pub const KP_13: Subdivision = Subdivision {
    name: "Nasǒn",
    code: "KP-13",
    subdivision_type: "Special city",
    country_name: "Korea, Democratic People's Republic Of",
    country_code: "KP",
    region_code: "KP-13",
};


pub const KR_11: Subdivision = Subdivision {
    name: "Seoul-T'ǔkpyǒlshi",
    code: "KR-11",
    subdivision_type: "Special city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-11",
};


pub const KR_26: Subdivision = Subdivision {
    name: "Busan Gwang'yeogsi",
    code: "KR-26",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-26",
};


pub const KR_27: Subdivision = Subdivision {
    name: "Daegu Gwang'yeogsi",
    code: "KR-27",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-27",
};


pub const KR_28: Subdivision = Subdivision {
    name: "Inch'ǒn-Kwangyǒkshi",
    code: "KR-28",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-28",
};


pub const KR_29: Subdivision = Subdivision {
    name: "Gwangju Gwang'yeogsi",
    code: "KR-29",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-29",
};


pub const KR_30: Subdivision = Subdivision {
    name: "Daejeon Gwang'yeogsi",
    code: "KR-30",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-30",
};


pub const KR_31: Subdivision = Subdivision {
    name: "Ulsan Gwang'yeogsi",
    code: "KR-31",
    subdivision_type: "Metropolitan city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-31",
};


pub const KR_41: Subdivision = Subdivision {
    name: "Gyeonggido",
    code: "KR-41",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-41",
};


pub const KR_42: Subdivision = Subdivision {
    name: "Gang'weondo",
    code: "KR-42",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-42",
};


pub const KR_43: Subdivision = Subdivision {
    name: "Ch'ungch'ǒngbuk-do",
    code: "KR-43",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-43",
};


pub const KR_44: Subdivision = Subdivision {
    name: "Ch'ungch'ǒngnam-do",
    code: "KR-44",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-44",
};


pub const KR_45: Subdivision = Subdivision {
    name: "Chǒllabuk-do",
    code: "KR-45",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-45",
};


pub const KR_46: Subdivision = Subdivision {
    name: "Chǒllanam-do",
    code: "KR-46",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-46",
};


pub const KR_47: Subdivision = Subdivision {
    name: "Gyeongsangbugdo",
    code: "KR-47",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-47",
};


pub const KR_48: Subdivision = Subdivision {
    name: "Gyeongsangnamdo",
    code: "KR-48",
    subdivision_type: "Province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-48",
};


pub const KR_49: Subdivision = Subdivision {
    name: "Cheju-do",
    code: "KR-49",
    subdivision_type: "Special self-governing province",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-49",
};


pub const KR_50: Subdivision = Subdivision {
    name: "Sejong",
    code: "KR-50",
    subdivision_type: "Special self-governing city",
    country_name: "Korea, Republic of",
    country_code: "KR",
    region_code: "KR-50",
};


pub const KW_AH: Subdivision = Subdivision {
    name: "Al Aḩmadī",
    code: "KW-AH",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-AH",
};


pub const KW_FA: Subdivision = Subdivision {
    name: "Al Farwānīyah",
    code: "KW-FA",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-FA",
};


pub const KW_HA: Subdivision = Subdivision {
    name: "Ḩawallī",
    code: "KW-HA",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-HA",
};


pub const KW_JA: Subdivision = Subdivision {
    name: "Al Jahrā’",
    code: "KW-JA",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-JA",
};


pub const KW_KU: Subdivision = Subdivision {
    name: "Al Kuwayt",
    code: "KW-KU",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-KU",
};


pub const KW_MU: Subdivision = Subdivision {
    name: "Mubārak al Kabīr",
    code: "KW-MU",
    subdivision_type: "Governorate",
    country_name: "Kuwait",
    country_code: "KW",
    region_code: "KW-MU",
};


pub const KG_B: Subdivision = Subdivision {
    name: "Batken",
    code: "KG-B",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-B",
};


pub const KG_C: Subdivision = Subdivision {
    name: "Chü",
    code: "KG-C",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-C",
};


pub const KG_GB: Subdivision = Subdivision {
    name: "Bishkek",
    code: "KG-GB",
    subdivision_type: "City",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-GB",
};


pub const KG_GO: Subdivision = Subdivision {
    name: "Gorod Oš",
    code: "KG-GO",
    subdivision_type: "City",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-GO",
};


pub const KG_J: Subdivision = Subdivision {
    name: "Džalal-Abadskaja oblast'",
    code: "KG-J",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-J",
};


pub const KG_N: Subdivision = Subdivision {
    name: "Naryn",
    code: "KG-N",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-N",
};


pub const KG_O: Subdivision = Subdivision {
    name: "Osh",
    code: "KG-O",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-O",
};


pub const KG_T: Subdivision = Subdivision {
    name: "Talas",
    code: "KG-T",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-T",
};


pub const KG_Y: Subdivision = Subdivision {
    name: "Issyk-Kul'skaja oblast'",
    code: "KG-Y",
    subdivision_type: "Region",
    country_name: "Kyrgyzstan",
    country_code: "KG",
    region_code: "KG-Y",
};


pub const LA_AT: Subdivision = Subdivision {
    name: "Attapu",
    code: "LA-AT",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-AT",
};


pub const LA_BK: Subdivision = Subdivision {
    name: "Bokèo",
    code: "LA-BK",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-BK",
};


pub const LA_BL: Subdivision = Subdivision {
    name: "Bolikhamxai",
    code: "LA-BL",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-BL",
};


pub const LA_CH: Subdivision = Subdivision {
    name: "Champasak",
    code: "LA-CH",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-CH",
};


pub const LA_HO: Subdivision = Subdivision {
    name: "Houaphan",
    code: "LA-HO",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-HO",
};


pub const LA_KH: Subdivision = Subdivision {
    name: "Khammouan",
    code: "LA-KH",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-KH",
};


pub const LA_LM: Subdivision = Subdivision {
    name: "Louang Namtha",
    code: "LA-LM",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-LM",
};


pub const LA_LP: Subdivision = Subdivision {
    name: "Louangphabang",
    code: "LA-LP",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-LP",
};


pub const LA_OU: Subdivision = Subdivision {
    name: "Oudomsai",
    code: "LA-OU",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-OU",
};


pub const LA_PH: Subdivision = Subdivision {
    name: "Phôngsali",
    code: "LA-PH",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-PH",
};


pub const LA_SL: Subdivision = Subdivision {
    name: "Salavan",
    code: "LA-SL",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-SL",
};


pub const LA_SV: Subdivision = Subdivision {
    name: "Savannakhét",
    code: "LA-SV",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-SV",
};


pub const LA_VI: Subdivision = Subdivision {
    name: "Vientiane",
    code: "LA-VI",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-VI",
};


pub const LA_VT: Subdivision = Subdivision {
    name: "Vientiane",
    code: "LA-VT",
    subdivision_type: "Prefecture",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-VT",
};


pub const LA_XA: Subdivision = Subdivision {
    name: "Sayaboury",
    code: "LA-XA",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-XA",
};


pub const LA_XE: Subdivision = Subdivision {
    name: "Sékong",
    code: "LA-XE",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-XE",
};


pub const LA_XI: Subdivision = Subdivision {
    name: "Xiangkhouang",
    code: "LA-XI",
    subdivision_type: "Province",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-XI",
};


pub const LA_XN: Subdivision = Subdivision {
    name: "Xaisômboun",
    code: "LA-XN",
    subdivision_type: "Special zone",
    country_name: "Laos",
    country_code: "LA",
    region_code: "LA-XN",
};


pub const LV_001: Subdivision = Subdivision {
    name: "Aglonas novads",
    code: "LV-001",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-001",
};


pub const LV_002: Subdivision = Subdivision {
    name: "Aizkraukles novads",
    code: "LV-002",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-002",
};


pub const LV_003: Subdivision = Subdivision {
    name: "Aizputes novads",
    code: "LV-003",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-003",
};


pub const LV_004: Subdivision = Subdivision {
    name: "Aknīstes novads",
    code: "LV-004",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-004",
};


pub const LV_005: Subdivision = Subdivision {
    name: "Alojas novads",
    code: "LV-005",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-005",
};


pub const LV_006: Subdivision = Subdivision {
    name: "Alsungas novads",
    code: "LV-006",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-006",
};


pub const LV_007: Subdivision = Subdivision {
    name: "Alūksnes novads",
    code: "LV-007",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-007",
};


pub const LV_008: Subdivision = Subdivision {
    name: "Amatas novads",
    code: "LV-008",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-008",
};


pub const LV_009: Subdivision = Subdivision {
    name: "Apes novads",
    code: "LV-009",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-009",
};


pub const LV_010: Subdivision = Subdivision {
    name: "Auces novads",
    code: "LV-010",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-010",
};


pub const LV_011: Subdivision = Subdivision {
    name: "Ādažu novads",
    code: "LV-011",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-011",
};


pub const LV_012: Subdivision = Subdivision {
    name: "Babītes novads",
    code: "LV-012",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-012",
};


pub const LV_013: Subdivision = Subdivision {
    name: "Baldones novads",
    code: "LV-013",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-013",
};


pub const LV_014: Subdivision = Subdivision {
    name: "Baltinavas novads",
    code: "LV-014",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-014",
};


pub const LV_015: Subdivision = Subdivision {
    name: "Balvu novads",
    code: "LV-015",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-015",
};


pub const LV_016: Subdivision = Subdivision {
    name: "Bauskas novads",
    code: "LV-016",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-016",
};


pub const LV_017: Subdivision = Subdivision {
    name: "Beverīnas novads",
    code: "LV-017",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-017",
};


pub const LV_018: Subdivision = Subdivision {
    name: "Brocēnu novads",
    code: "LV-018",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-018",
};


pub const LV_019: Subdivision = Subdivision {
    name: "Burtnieku novads",
    code: "LV-019",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-019",
};


pub const LV_020: Subdivision = Subdivision {
    name: "Carnikavas novads",
    code: "LV-020",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-020",
};


pub const LV_021: Subdivision = Subdivision {
    name: "Cesvaines novads",
    code: "LV-021",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-021",
};


pub const LV_022: Subdivision = Subdivision {
    name: "Cēsu novads",
    code: "LV-022",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-022",
};


pub const LV_023: Subdivision = Subdivision {
    name: "Ciblas novads",
    code: "LV-023",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-023",
};


pub const LV_024: Subdivision = Subdivision {
    name: "Dagdas novads",
    code: "LV-024",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-024",
};


pub const LV_025: Subdivision = Subdivision {
    name: "Daugavpils novads",
    code: "LV-025",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-025",
};


pub const LV_026: Subdivision = Subdivision {
    name: "Dobeles novads",
    code: "LV-026",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-026",
};


pub const LV_027: Subdivision = Subdivision {
    name: "Dundagas novads",
    code: "LV-027",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-027",
};


pub const LV_028: Subdivision = Subdivision {
    name: "Durbes novads",
    code: "LV-028",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-028",
};


pub const LV_029: Subdivision = Subdivision {
    name: "Engures novads",
    code: "LV-029",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-029",
};


pub const LV_030: Subdivision = Subdivision {
    name: "Ērgļu novads",
    code: "LV-030",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-030",
};


pub const LV_031: Subdivision = Subdivision {
    name: "Garkalnes novads",
    code: "LV-031",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-031",
};


pub const LV_032: Subdivision = Subdivision {
    name: "Grobiņas novads",
    code: "LV-032",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-032",
};


pub const LV_033: Subdivision = Subdivision {
    name: "Gulbenes novads",
    code: "LV-033",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-033",
};


pub const LV_034: Subdivision = Subdivision {
    name: "Iecavas novads",
    code: "LV-034",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-034",
};


pub const LV_035: Subdivision = Subdivision {
    name: "Ikšķiles novads",
    code: "LV-035",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-035",
};


pub const LV_036: Subdivision = Subdivision {
    name: "Ilūkstes novads",
    code: "LV-036",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-036",
};


pub const LV_037: Subdivision = Subdivision {
    name: "Inčukalna novads",
    code: "LV-037",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-037",
};


pub const LV_038: Subdivision = Subdivision {
    name: "Jaunjelgavas novads",
    code: "LV-038",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-038",
};


pub const LV_039: Subdivision = Subdivision {
    name: "Jaunpiebalgas novads",
    code: "LV-039",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-039",
};


pub const LV_040: Subdivision = Subdivision {
    name: "Jaunpils novads",
    code: "LV-040",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-040",
};


pub const LV_041: Subdivision = Subdivision {
    name: "Jelgavas novads",
    code: "LV-041",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-041",
};


pub const LV_042: Subdivision = Subdivision {
    name: "Jēkabpils novads",
    code: "LV-042",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-042",
};


pub const LV_043: Subdivision = Subdivision {
    name: "Kandavas novads",
    code: "LV-043",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-043",
};


pub const LV_044: Subdivision = Subdivision {
    name: "Kārsavas novads",
    code: "LV-044",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-044",
};


pub const LV_045: Subdivision = Subdivision {
    name: "Kocēnu novads",
    code: "LV-045",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-045",
};


pub const LV_046: Subdivision = Subdivision {
    name: "Kokneses novads",
    code: "LV-046",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-046",
};


pub const LV_047: Subdivision = Subdivision {
    name: "Krāslavas novads",
    code: "LV-047",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-047",
};


pub const LV_048: Subdivision = Subdivision {
    name: "Krimuldas novads",
    code: "LV-048",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-048",
};


pub const LV_049: Subdivision = Subdivision {
    name: "Krustpils novads",
    code: "LV-049",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-049",
};


pub const LV_050: Subdivision = Subdivision {
    name: "Kuldīgas novads",
    code: "LV-050",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-050",
};


pub const LV_051: Subdivision = Subdivision {
    name: "Ķeguma novads",
    code: "LV-051",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-051",
};


pub const LV_052: Subdivision = Subdivision {
    name: "Ķekavas novads",
    code: "LV-052",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-052",
};


pub const LV_053: Subdivision = Subdivision {
    name: "Lielvārdes novads",
    code: "LV-053",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-053",
};


pub const LV_054: Subdivision = Subdivision {
    name: "Limbažu novads",
    code: "LV-054",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-054",
};


pub const LV_055: Subdivision = Subdivision {
    name: "Līgatnes novads",
    code: "LV-055",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-055",
};


pub const LV_056: Subdivision = Subdivision {
    name: "Līvānu novads",
    code: "LV-056",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-056",
};


pub const LV_057: Subdivision = Subdivision {
    name: "Lubānas novads",
    code: "LV-057",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-057",
};


pub const LV_058: Subdivision = Subdivision {
    name: "Ludzas novads",
    code: "LV-058",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-058",
};


pub const LV_059: Subdivision = Subdivision {
    name: "Madonas novads",
    code: "LV-059",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-059",
};


pub const LV_060: Subdivision = Subdivision {
    name: "Mazsalacas novads",
    code: "LV-060",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-060",
};


pub const LV_061: Subdivision = Subdivision {
    name: "Mālpils novads",
    code: "LV-061",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-061",
};


pub const LV_062: Subdivision = Subdivision {
    name: "Mārupes novads",
    code: "LV-062",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-062",
};


pub const LV_063: Subdivision = Subdivision {
    name: "Mērsraga novads",
    code: "LV-063",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-063",
};


pub const LV_064: Subdivision = Subdivision {
    name: "Naukšēnu novads",
    code: "LV-064",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-064",
};


pub const LV_065: Subdivision = Subdivision {
    name: "Neretas novads",
    code: "LV-065",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-065",
};


pub const LV_066: Subdivision = Subdivision {
    name: "Nīcas novads",
    code: "LV-066",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-066",
};


pub const LV_067: Subdivision = Subdivision {
    name: "Ogres novads",
    code: "LV-067",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-067",
};


pub const LV_068: Subdivision = Subdivision {
    name: "Olaines novads",
    code: "LV-068",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-068",
};


pub const LV_069: Subdivision = Subdivision {
    name: "Ozolnieku novads",
    code: "LV-069",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-069",
};


pub const LV_070: Subdivision = Subdivision {
    name: "Pārgaujas novads",
    code: "LV-070",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-070",
};


pub const LV_071: Subdivision = Subdivision {
    name: "Pāvilostas novads",
    code: "LV-071",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-071",
};


pub const LV_072: Subdivision = Subdivision {
    name: "Pļaviņu novads",
    code: "LV-072",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-072",
};


pub const LV_073: Subdivision = Subdivision {
    name: "Preiļu novads",
    code: "LV-073",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-073",
};


pub const LV_074: Subdivision = Subdivision {
    name: "Priekules novads",
    code: "LV-074",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-074",
};


pub const LV_075: Subdivision = Subdivision {
    name: "Priekuļu novads",
    code: "LV-075",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-075",
};


pub const LV_076: Subdivision = Subdivision {
    name: "Raunas novads",
    code: "LV-076",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-076",
};


pub const LV_077: Subdivision = Subdivision {
    name: "Rēzeknes novads",
    code: "LV-077",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-077",
};


pub const LV_078: Subdivision = Subdivision {
    name: "Riebiņu novads",
    code: "LV-078",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-078",
};


pub const LV_079: Subdivision = Subdivision {
    name: "Rojas novads",
    code: "LV-079",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-079",
};


pub const LV_080: Subdivision = Subdivision {
    name: "Ropažu novads",
    code: "LV-080",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-080",
};


pub const LV_081: Subdivision = Subdivision {
    name: "Rucavas novads",
    code: "LV-081",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-081",
};


pub const LV_082: Subdivision = Subdivision {
    name: "Rugāju novads",
    code: "LV-082",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-082",
};


pub const LV_083: Subdivision = Subdivision {
    name: "Rundāles novads",
    code: "LV-083",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-083",
};


pub const LV_084: Subdivision = Subdivision {
    name: "Rūjienas novads",
    code: "LV-084",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-084",
};


pub const LV_085: Subdivision = Subdivision {
    name: "Salas novads",
    code: "LV-085",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-085",
};


pub const LV_086: Subdivision = Subdivision {
    name: "Salacgrīvas novads",
    code: "LV-086",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-086",
};


pub const LV_087: Subdivision = Subdivision {
    name: "Salaspils novads",
    code: "LV-087",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-087",
};


pub const LV_088: Subdivision = Subdivision {
    name: "Saldus novads",
    code: "LV-088",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-088",
};


pub const LV_089: Subdivision = Subdivision {
    name: "Saulkrastu novads",
    code: "LV-089",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-089",
};


pub const LV_090: Subdivision = Subdivision {
    name: "Sējas novads",
    code: "LV-090",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-090",
};


pub const LV_091: Subdivision = Subdivision {
    name: "Siguldas novads",
    code: "LV-091",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-091",
};


pub const LV_092: Subdivision = Subdivision {
    name: "Skrīveru novads",
    code: "LV-092",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-092",
};


pub const LV_093: Subdivision = Subdivision {
    name: "Skrundas novads",
    code: "LV-093",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-093",
};


pub const LV_094: Subdivision = Subdivision {
    name: "Smiltenes novads",
    code: "LV-094",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-094",
};


pub const LV_095: Subdivision = Subdivision {
    name: "Stopiņu novads",
    code: "LV-095",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-095",
};


pub const LV_096: Subdivision = Subdivision {
    name: "Strenču novads",
    code: "LV-096",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-096",
};


pub const LV_097: Subdivision = Subdivision {
    name: "Talsu novads",
    code: "LV-097",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-097",
};


pub const LV_098: Subdivision = Subdivision {
    name: "Tērvetes novads",
    code: "LV-098",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-098",
};


pub const LV_099: Subdivision = Subdivision {
    name: "Tukuma novads",
    code: "LV-099",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-099",
};


pub const LV_100: Subdivision = Subdivision {
    name: "Vaiņodes novads",
    code: "LV-100",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-100",
};


pub const LV_101: Subdivision = Subdivision {
    name: "Valkas novads",
    code: "LV-101",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-101",
};


pub const LV_102: Subdivision = Subdivision {
    name: "Varakļānu novads",
    code: "LV-102",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-102",
};


pub const LV_103: Subdivision = Subdivision {
    name: "Vārkavas novads",
    code: "LV-103",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-103",
};


pub const LV_104: Subdivision = Subdivision {
    name: "Vecpiebalgas novads",
    code: "LV-104",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-104",
};


pub const LV_105: Subdivision = Subdivision {
    name: "Vecumnieku novads",
    code: "LV-105",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-105",
};


pub const LV_106: Subdivision = Subdivision {
    name: "Ventspils novads",
    code: "LV-106",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-106",
};


pub const LV_107: Subdivision = Subdivision {
    name: "Viesītes novads",
    code: "LV-107",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-107",
};


pub const LV_108: Subdivision = Subdivision {
    name: "Viļakas novads",
    code: "LV-108",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-108",
};


pub const LV_109: Subdivision = Subdivision {
    name: "Viļānu novads",
    code: "LV-109",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-109",
};


pub const LV_110: Subdivision = Subdivision {
    name: "Zilupes novads",
    code: "LV-110",
    subdivision_type: "Municipality",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-110",
};


pub const LV_DGV: Subdivision = Subdivision {
    name: "Daugavpils",
    code: "LV-DGV",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-DGV",
};


pub const LV_JEL: Subdivision = Subdivision {
    name: "Jelgava",
    code: "LV-JEL",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-JEL",
};


pub const LV_JKB: Subdivision = Subdivision {
    name: "Jēkabpils",
    code: "LV-JKB",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-JKB",
};


pub const LV_JUR: Subdivision = Subdivision {
    name: "Jūrmala",
    code: "LV-JUR",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-JUR",
};


pub const LV_LPX: Subdivision = Subdivision {
    name: "Liepāja",
    code: "LV-LPX",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-LPX",
};


pub const LV_REZ: Subdivision = Subdivision {
    name: "Rēzekne",
    code: "LV-REZ",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-REZ",
};


pub const LV_RIX: Subdivision = Subdivision {
    name: "Rīga",
    code: "LV-RIX",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-RIX",
};


pub const LV_VEN: Subdivision = Subdivision {
    name: "Ventspils",
    code: "LV-VEN",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-VEN",
};


pub const LV_VMR: Subdivision = Subdivision {
    name: "Valmiera",
    code: "LV-VMR",
    subdivision_type: "Republican city",
    country_name: "Latvia",
    country_code: "LV",
    region_code: "LV-VMR",
};


pub const LB_AK: Subdivision = Subdivision {
    name: "Aakkâr",
    code: "LB-AK",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-AK",
};


pub const LB_AS: Subdivision = Subdivision {
    name: "Ash Shimāl",
    code: "LB-AS",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-AS",
};


pub const LB_BA: Subdivision = Subdivision {
    name: "Bayrūt",
    code: "LB-BA",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-BA",
};


pub const LB_BH: Subdivision = Subdivision {
    name: "Baalbek-Hermel",
    code: "LB-BH",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-BH",
};


pub const LB_BI: Subdivision = Subdivision {
    name: "Al Biqā‘",
    code: "LB-BI",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-BI",
};


pub const LB_JA: Subdivision = Subdivision {
    name: "Al Janūb",
    code: "LB-JA",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-JA",
};


pub const LB_JL: Subdivision = Subdivision {
    name: "Jabal Lubnān",
    code: "LB-JL",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-JL",
};


pub const LB_NA: Subdivision = Subdivision {
    name: "An Nabaţīyah",
    code: "LB-NA",
    subdivision_type: "Governorate",
    country_name: "Lebanon",
    country_code: "LB",
    region_code: "LB-NA",
};


pub const LS_A: Subdivision = Subdivision {
    name: "Maseru",
    code: "LS-A",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-A",
};


pub const LS_B: Subdivision = Subdivision {
    name: "Butha-Buthe",
    code: "LS-B",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-B",
};


pub const LS_C: Subdivision = Subdivision {
    name: "Leribe",
    code: "LS-C",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-C",
};


pub const LS_D: Subdivision = Subdivision {
    name: "Berea",
    code: "LS-D",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-D",
};


pub const LS_E: Subdivision = Subdivision {
    name: "Mafeteng",
    code: "LS-E",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-E",
};


pub const LS_F: Subdivision = Subdivision {
    name: "Mohale's Hoek",
    code: "LS-F",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-F",
};


pub const LS_G: Subdivision = Subdivision {
    name: "Quthing",
    code: "LS-G",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-G",
};


pub const LS_H: Subdivision = Subdivision {
    name: "Qacha's Nek",
    code: "LS-H",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-H",
};


pub const LS_J: Subdivision = Subdivision {
    name: "Mokhotlong",
    code: "LS-J",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-J",
};


pub const LS_K: Subdivision = Subdivision {
    name: "Thaba-Tseka",
    code: "LS-K",
    subdivision_type: "District",
    country_name: "Lesotho",
    country_code: "LS",
    region_code: "LS-K",
};


pub const LR_BG: Subdivision = Subdivision {
    name: "Bong",
    code: "LR-BG",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-BG",
};


pub const LR_BM: Subdivision = Subdivision {
    name: "Bomi",
    code: "LR-BM",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-BM",
};


pub const LR_CM: Subdivision = Subdivision {
    name: "Grand Cape Mount",
    code: "LR-CM",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-CM",
};


pub const LR_GB: Subdivision = Subdivision {
    name: "Grand Bassa",
    code: "LR-GB",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-GB",
};


pub const LR_GG: Subdivision = Subdivision {
    name: "Grand Gedeh",
    code: "LR-GG",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-GG",
};


pub const LR_GK: Subdivision = Subdivision {
    name: "Grand Kru",
    code: "LR-GK",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-GK",
};


pub const LR_GP: Subdivision = Subdivision {
    name: "Gbarpolu",
    code: "LR-GP",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-GP",
};


pub const LR_LO: Subdivision = Subdivision {
    name: "Lofa",
    code: "LR-LO",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-LO",
};


pub const LR_MG: Subdivision = Subdivision {
    name: "Margibi",
    code: "LR-MG",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-MG",
};


pub const LR_MO: Subdivision = Subdivision {
    name: "Montserrado",
    code: "LR-MO",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-MO",
};


pub const LR_MY: Subdivision = Subdivision {
    name: "Maryland",
    code: "LR-MY",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-MY",
};


pub const LR_NI: Subdivision = Subdivision {
    name: "Nimba",
    code: "LR-NI",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-NI",
};


pub const LR_RG: Subdivision = Subdivision {
    name: "River Gee",
    code: "LR-RG",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-RG",
};


pub const LR_RI: Subdivision = Subdivision {
    name: "Rivercess",
    code: "LR-RI",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-RI",
};


pub const LR_SI: Subdivision = Subdivision {
    name: "Sinoe",
    code: "LR-SI",
    subdivision_type: "County",
    country_name: "Liberia",
    country_code: "LR",
    region_code: "LR-SI",
};


pub const LY_BA: Subdivision = Subdivision {
    name: "Banghāzī",
    code: "LY-BA",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-BA",
};


pub const LY_BU: Subdivision = Subdivision {
    name: "Al Buţnān",
    code: "LY-BU",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-BU",
};


pub const LY_DR: Subdivision = Subdivision {
    name: "Darnah",
    code: "LY-DR",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-DR",
};


pub const LY_GT: Subdivision = Subdivision {
    name: "Ghāt",
    code: "LY-GT",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-GT",
};


pub const LY_JA: Subdivision = Subdivision {
    name: "Al Jabal al Akhḑar",
    code: "LY-JA",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-JA",
};


pub const LY_JG: Subdivision = Subdivision {
    name: "Al Jabal al Gharbī",
    code: "LY-JG",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-JG",
};


pub const LY_JI: Subdivision = Subdivision {
    name: "Al Jifārah",
    code: "LY-JI",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-JI",
};


pub const LY_JU: Subdivision = Subdivision {
    name: "Al Jufrah",
    code: "LY-JU",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-JU",
};


pub const LY_KF: Subdivision = Subdivision {
    name: "Al Kufrah",
    code: "LY-KF",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-KF",
};


pub const LY_MB: Subdivision = Subdivision {
    name: "Al Marqab",
    code: "LY-MB",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-MB",
};


pub const LY_MI: Subdivision = Subdivision {
    name: "Mişrātah",
    code: "LY-MI",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-MI",
};


pub const LY_MJ: Subdivision = Subdivision {
    name: "Al Marj",
    code: "LY-MJ",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-MJ",
};


pub const LY_MQ: Subdivision = Subdivision {
    name: "Murzuq",
    code: "LY-MQ",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-MQ",
};


pub const LY_NL: Subdivision = Subdivision {
    name: "Nālūt",
    code: "LY-NL",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-NL",
};


pub const LY_NQ: Subdivision = Subdivision {
    name: "An Nuqāţ al Khams",
    code: "LY-NQ",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-NQ",
};


pub const LY_SB: Subdivision = Subdivision {
    name: "Sabhā",
    code: "LY-SB",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-SB",
};


pub const LY_SR: Subdivision = Subdivision {
    name: "Surt",
    code: "LY-SR",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-SR",
};


pub const LY_TB: Subdivision = Subdivision {
    name: "Ţarābulus",
    code: "LY-TB",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-TB",
};


pub const LY_WA: Subdivision = Subdivision {
    name: "Al Wāḩāt",
    code: "LY-WA",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-WA",
};


pub const LY_WD: Subdivision = Subdivision {
    name: "Wādī al Ḩayāt",
    code: "LY-WD",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-WD",
};


pub const LY_WS: Subdivision = Subdivision {
    name: "Wādī ash Shāţi’",
    code: "LY-WS",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-WS",
};


pub const LY_ZA: Subdivision = Subdivision {
    name: "Az Zāwiyah",
    code: "LY-ZA",
    subdivision_type: "Popularate",
    country_name: "Libya",
    country_code: "LY",
    region_code: "LY-ZA",
};


pub const LI_01: Subdivision = Subdivision {
    name: "Balzers",
    code: "LI-01",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-01",
};


pub const LI_02: Subdivision = Subdivision {
    name: "Eschen",
    code: "LI-02",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-02",
};


pub const LI_03: Subdivision = Subdivision {
    name: "Gamprin",
    code: "LI-03",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-03",
};


pub const LI_04: Subdivision = Subdivision {
    name: "Mauren",
    code: "LI-04",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-04",
};


pub const LI_05: Subdivision = Subdivision {
    name: "Planken",
    code: "LI-05",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-05",
};


pub const LI_06: Subdivision = Subdivision {
    name: "Ruggell",
    code: "LI-06",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-06",
};


pub const LI_07: Subdivision = Subdivision {
    name: "Schaan",
    code: "LI-07",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-07",
};


pub const LI_08: Subdivision = Subdivision {
    name: "Schellenberg",
    code: "LI-08",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-08",
};


pub const LI_09: Subdivision = Subdivision {
    name: "Triesen",
    code: "LI-09",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-09",
};


pub const LI_10: Subdivision = Subdivision {
    name: "Triesenberg",
    code: "LI-10",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-10",
};


pub const LI_11: Subdivision = Subdivision {
    name: "Vaduz",
    code: "LI-11",
    subdivision_type: "Commune",
    country_name: "Liechtenstein",
    country_code: "LI",
    region_code: "LI-11",
};


pub const LT_01: Subdivision = Subdivision {
    name: "Akmenė",
    code: "LT-01",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-01",
};


pub const LT_02: Subdivision = Subdivision {
    name: "Alytaus miestas",
    code: "LT-02",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-02",
};


pub const LT_03: Subdivision = Subdivision {
    name: "Alytus",
    code: "LT-03",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-03",
};


pub const LT_04: Subdivision = Subdivision {
    name: "Anykščiai",
    code: "LT-04",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-04",
};


pub const LT_05: Subdivision = Subdivision {
    name: "Birštono",
    code: "LT-05",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-05",
};


pub const LT_06: Subdivision = Subdivision {
    name: "Biržai",
    code: "LT-06",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-06",
};


pub const LT_07: Subdivision = Subdivision {
    name: "Druskininkai",
    code: "LT-07",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-07",
};


pub const LT_08: Subdivision = Subdivision {
    name: "Elektrénai",
    code: "LT-08",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-08",
};


pub const LT_09: Subdivision = Subdivision {
    name: "Ignalina",
    code: "LT-09",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-09",
};


pub const LT_10: Subdivision = Subdivision {
    name: "Jonava",
    code: "LT-10",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-10",
};


pub const LT_11: Subdivision = Subdivision {
    name: "Joniškis",
    code: "LT-11",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-11",
};


pub const LT_12: Subdivision = Subdivision {
    name: "Jurbarkas",
    code: "LT-12",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-12",
};


pub const LT_13: Subdivision = Subdivision {
    name: "Kaišiadorys",
    code: "LT-13",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-13",
};


pub const LT_14: Subdivision = Subdivision {
    name: "Kalvarijos",
    code: "LT-14",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-14",
};


pub const LT_15: Subdivision = Subdivision {
    name: "Kauno miestas",
    code: "LT-15",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-15",
};


pub const LT_16: Subdivision = Subdivision {
    name: "Kaunas",
    code: "LT-16",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-16",
};


pub const LT_17: Subdivision = Subdivision {
    name: "Kazlų Rūdos",
    code: "LT-17",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-17",
};


pub const LT_18: Subdivision = Subdivision {
    name: "Kėdainiai",
    code: "LT-18",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-18",
};


pub const LT_19: Subdivision = Subdivision {
    name: "Kelmė",
    code: "LT-19",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-19",
};


pub const LT_20: Subdivision = Subdivision {
    name: "Klaipėdos miestas",
    code: "LT-20",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-20",
};


pub const LT_21: Subdivision = Subdivision {
    name: "Klaipėda",
    code: "LT-21",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-21",
};


pub const LT_22: Subdivision = Subdivision {
    name: "Kretinga",
    code: "LT-22",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-22",
};


pub const LT_23: Subdivision = Subdivision {
    name: "Kupiškis",
    code: "LT-23",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-23",
};


pub const LT_24: Subdivision = Subdivision {
    name: "Lazdijai",
    code: "LT-24",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-24",
};


pub const LT_25: Subdivision = Subdivision {
    name: "Marijampolė",
    code: "LT-25",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-25",
};


pub const LT_26: Subdivision = Subdivision {
    name: "Mažeikiai",
    code: "LT-26",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-26",
};


pub const LT_27: Subdivision = Subdivision {
    name: "Molėtai",
    code: "LT-27",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-27",
};


pub const LT_28: Subdivision = Subdivision {
    name: "Neringa",
    code: "LT-28",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-28",
};


pub const LT_29: Subdivision = Subdivision {
    name: "Pagégiai",
    code: "LT-29",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-29",
};


pub const LT_30: Subdivision = Subdivision {
    name: "Pakruojis",
    code: "LT-30",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-30",
};


pub const LT_31: Subdivision = Subdivision {
    name: "Palangos miestas",
    code: "LT-31",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-31",
};


pub const LT_32: Subdivision = Subdivision {
    name: "Panevėžio miestas",
    code: "LT-32",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-32",
};


pub const LT_33: Subdivision = Subdivision {
    name: "Panevėžys",
    code: "LT-33",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-33",
};


pub const LT_34: Subdivision = Subdivision {
    name: "Pasvalys",
    code: "LT-34",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-34",
};


pub const LT_35: Subdivision = Subdivision {
    name: "Plungė",
    code: "LT-35",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-35",
};


pub const LT_36: Subdivision = Subdivision {
    name: "Prienai",
    code: "LT-36",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-36",
};


pub const LT_37: Subdivision = Subdivision {
    name: "Radviliškis",
    code: "LT-37",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-37",
};


pub const LT_38: Subdivision = Subdivision {
    name: "Raseiniai",
    code: "LT-38",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-38",
};


pub const LT_39: Subdivision = Subdivision {
    name: "Rietavo",
    code: "LT-39",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-39",
};


pub const LT_40: Subdivision = Subdivision {
    name: "Rokiškis",
    code: "LT-40",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-40",
};


pub const LT_41: Subdivision = Subdivision {
    name: "Šakiai",
    code: "LT-41",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-41",
};


pub const LT_42: Subdivision = Subdivision {
    name: "Šalčininkai",
    code: "LT-42",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-42",
};


pub const LT_43: Subdivision = Subdivision {
    name: "Šiaulių miestas",
    code: "LT-43",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-43",
};


pub const LT_44: Subdivision = Subdivision {
    name: "Šiauliai",
    code: "LT-44",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-44",
};


pub const LT_45: Subdivision = Subdivision {
    name: "Šilalė",
    code: "LT-45",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-45",
};


pub const LT_46: Subdivision = Subdivision {
    name: "Šilutė",
    code: "LT-46",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-46",
};


pub const LT_47: Subdivision = Subdivision {
    name: "Širvintos",
    code: "LT-47",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-47",
};


pub const LT_48: Subdivision = Subdivision {
    name: "Skuodas",
    code: "LT-48",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-48",
};


pub const LT_49: Subdivision = Subdivision {
    name: "Švenčionys",
    code: "LT-49",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-49",
};


pub const LT_50: Subdivision = Subdivision {
    name: "Tauragė",
    code: "LT-50",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-50",
};


pub const LT_51: Subdivision = Subdivision {
    name: "Telšiai",
    code: "LT-51",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-51",
};


pub const LT_52: Subdivision = Subdivision {
    name: "Trakai",
    code: "LT-52",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-52",
};


pub const LT_53: Subdivision = Subdivision {
    name: "Ukmergė",
    code: "LT-53",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-53",
};


pub const LT_54: Subdivision = Subdivision {
    name: "Utena",
    code: "LT-54",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-54",
};


pub const LT_55: Subdivision = Subdivision {
    name: "Varėna",
    code: "LT-55",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-55",
};


pub const LT_56: Subdivision = Subdivision {
    name: "Vilkaviškis",
    code: "LT-56",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-56",
};


pub const LT_57: Subdivision = Subdivision {
    name: "Vilniaus miestas",
    code: "LT-57",
    subdivision_type: "City municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-57",
};


pub const LT_58: Subdivision = Subdivision {
    name: "Vilnius",
    code: "LT-58",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-58",
};


pub const LT_59: Subdivision = Subdivision {
    name: "Visaginas",
    code: "LT-59",
    subdivision_type: "Municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-59",
};


pub const LT_60: Subdivision = Subdivision {
    name: "Zarasai",
    code: "LT-60",
    subdivision_type: "District municipality",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-60",
};


pub const LT_AL: Subdivision = Subdivision {
    name: "Alytaus apskritis",
    code: "LT-AL",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-AL",
};


pub const LT_KL: Subdivision = Subdivision {
    name: "Klaipėdos apskritis",
    code: "LT-KL",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-KL",
};


pub const LT_KU: Subdivision = Subdivision {
    name: "Kauno apskritis",
    code: "LT-KU",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-KU",
};


pub const LT_MR: Subdivision = Subdivision {
    name: "Marijampolės apskritis",
    code: "LT-MR",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-MR",
};


pub const LT_PN: Subdivision = Subdivision {
    name: "Panevėžio apskritis",
    code: "LT-PN",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-PN",
};


pub const LT_SA: Subdivision = Subdivision {
    name: "Šiaulių apskritis",
    code: "LT-SA",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-SA",
};


pub const LT_TA: Subdivision = Subdivision {
    name: "Tauragės apskritis",
    code: "LT-TA",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-TA",
};


pub const LT_TE: Subdivision = Subdivision {
    name: "Telšių apskritis",
    code: "LT-TE",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-TE",
};


pub const LT_UT: Subdivision = Subdivision {
    name: "Utenos apskritis",
    code: "LT-UT",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-UT",
};


pub const LT_VL: Subdivision = Subdivision {
    name: "Vilniaus apskritis",
    code: "LT-VL",
    subdivision_type: "County",
    country_name: "Lithuania",
    country_code: "LT",
    region_code: "LT-VL",
};


pub const LU_D: Subdivision = Subdivision {
    name: "Diekirch",
    code: "LU-D",
    subdivision_type: "District",
    country_name: "Luxembourg",
    country_code: "LU",
    region_code: "LU-D",
};


pub const LU_G: Subdivision = Subdivision {
    name: "Grevenmacher",
    code: "LU-G",
    subdivision_type: "District",
    country_name: "Luxembourg",
    country_code: "LU",
    region_code: "LU-G",
};


pub const LU_L: Subdivision = Subdivision {
    name: "Lëtzebuerg",
    code: "LU-L",
    subdivision_type: "District",
    country_name: "Luxembourg",
    country_code: "LU",
    region_code: "LU-L",
};


pub const MO_MO: Subdivision = Subdivision {
    name: "Macau",
    code: "MO-MO",
    subdivision_type: "Country",
    country_name: "Macau",
    country_code: "MO",
    region_code: "MO-MO",
};


pub const MK_01: Subdivision = Subdivision {
    name: "Aerodrom †",
    code: "MK-01",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-01",
};


pub const MK_02: Subdivision = Subdivision {
    name: "Aračinovo",
    code: "MK-02",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-02",
};


pub const MK_03: Subdivision = Subdivision {
    name: "Berovo",
    code: "MK-03",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-03",
};


pub const MK_04: Subdivision = Subdivision {
    name: "Bitola",
    code: "MK-04",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-04",
};


pub const MK_05: Subdivision = Subdivision {
    name: "Bogdanci",
    code: "MK-05",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-05",
};


pub const MK_06: Subdivision = Subdivision {
    name: "Bogovinje",
    code: "MK-06",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-06",
};


pub const MK_07: Subdivision = Subdivision {
    name: "Bosilovo",
    code: "MK-07",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-07",
};


pub const MK_08: Subdivision = Subdivision {
    name: "Brvenica",
    code: "MK-08",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-08",
};


pub const MK_09: Subdivision = Subdivision {
    name: "Butel †",
    code: "MK-09",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-09",
};


pub const MK_10: Subdivision = Subdivision {
    name: "Valandovo",
    code: "MK-10",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-10",
};


pub const MK_11: Subdivision = Subdivision {
    name: "Vasilevo",
    code: "MK-11",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-11",
};


pub const MK_12: Subdivision = Subdivision {
    name: "Vevčani",
    code: "MK-12",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-12",
};


pub const MK_13: Subdivision = Subdivision {
    name: "Veles",
    code: "MK-13",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-13",
};


pub const MK_14: Subdivision = Subdivision {
    name: "Vinica",
    code: "MK-14",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-14",
};


pub const MK_15: Subdivision = Subdivision {
    name: "Vraneštica",
    code: "MK-15",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-15",
};


pub const MK_16: Subdivision = Subdivision {
    name: "Vrapčište",
    code: "MK-16",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-16",
};


pub const MK_17: Subdivision = Subdivision {
    name: "Gazi Baba †",
    code: "MK-17",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-17",
};


pub const MK_18: Subdivision = Subdivision {
    name: "Gevgelija",
    code: "MK-18",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-18",
};


pub const MK_19: Subdivision = Subdivision {
    name: "Gostivar",
    code: "MK-19",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-19",
};


pub const MK_20: Subdivision = Subdivision {
    name: "Gradsko",
    code: "MK-20",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-20",
};


pub const MK_21: Subdivision = Subdivision {
    name: "Debar",
    code: "MK-21",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-21",
};


pub const MK_22: Subdivision = Subdivision {
    name: "Debarca",
    code: "MK-22",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-22",
};


pub const MK_23: Subdivision = Subdivision {
    name: "Delčevo",
    code: "MK-23",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-23",
};


pub const MK_24: Subdivision = Subdivision {
    name: "Demir Kapija",
    code: "MK-24",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-24",
};


pub const MK_25: Subdivision = Subdivision {
    name: "Demir Hisar",
    code: "MK-25",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-25",
};


pub const MK_26: Subdivision = Subdivision {
    name: "Dojran",
    code: "MK-26",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-26",
};


pub const MK_27: Subdivision = Subdivision {
    name: "Dolneni",
    code: "MK-27",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-27",
};


pub const MK_28: Subdivision = Subdivision {
    name: "Drugovo",
    code: "MK-28",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-28",
};


pub const MK_29: Subdivision = Subdivision {
    name: "Gjorče Petrov †",
    code: "MK-29",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-29",
};


pub const MK_30: Subdivision = Subdivision {
    name: "Želino",
    code: "MK-30",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-30",
};


pub const MK_31: Subdivision = Subdivision {
    name: "Zajas",
    code: "MK-31",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-31",
};


pub const MK_32: Subdivision = Subdivision {
    name: "Zelenikovo",
    code: "MK-32",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-32",
};


pub const MK_33: Subdivision = Subdivision {
    name: "Zrnovci",
    code: "MK-33",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-33",
};


pub const MK_34: Subdivision = Subdivision {
    name: "Ilinden",
    code: "MK-34",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-34",
};


pub const MK_35: Subdivision = Subdivision {
    name: "Jegunovce",
    code: "MK-35",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-35",
};


pub const MK_36: Subdivision = Subdivision {
    name: "Kavadarci",
    code: "MK-36",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-36",
};


pub const MK_37: Subdivision = Subdivision {
    name: "Karbinci",
    code: "MK-37",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-37",
};


pub const MK_38: Subdivision = Subdivision {
    name: "Karpoš †",
    code: "MK-38",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-38",
};


pub const MK_39: Subdivision = Subdivision {
    name: "Kisela Voda †",
    code: "MK-39",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-39",
};


pub const MK_40: Subdivision = Subdivision {
    name: "Kičevo",
    code: "MK-40",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-40",
};


pub const MK_41: Subdivision = Subdivision {
    name: "Konče",
    code: "MK-41",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-41",
};


pub const MK_42: Subdivision = Subdivision {
    name: "Kočani",
    code: "MK-42",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-42",
};


pub const MK_43: Subdivision = Subdivision {
    name: "Kratovo",
    code: "MK-43",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-43",
};


pub const MK_44: Subdivision = Subdivision {
    name: "Kriva Palanka",
    code: "MK-44",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-44",
};


pub const MK_45: Subdivision = Subdivision {
    name: "Krivogaštani",
    code: "MK-45",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-45",
};


pub const MK_46: Subdivision = Subdivision {
    name: "Kruševo",
    code: "MK-46",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-46",
};


pub const MK_47: Subdivision = Subdivision {
    name: "Kumanovo",
    code: "MK-47",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-47",
};


pub const MK_48: Subdivision = Subdivision {
    name: "Lipkovo",
    code: "MK-48",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-48",
};


pub const MK_49: Subdivision = Subdivision {
    name: "Lozovo",
    code: "MK-49",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-49",
};


pub const MK_50: Subdivision = Subdivision {
    name: "Mavrovo i Rostuša",
    code: "MK-50",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-50",
};


pub const MK_51: Subdivision = Subdivision {
    name: "Makedonska Kamenica",
    code: "MK-51",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-51",
};


pub const MK_52: Subdivision = Subdivision {
    name: "Makedonski Brod",
    code: "MK-52",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-52",
};


pub const MK_53: Subdivision = Subdivision {
    name: "Mogila",
    code: "MK-53",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-53",
};


pub const MK_54: Subdivision = Subdivision {
    name: "Negotino",
    code: "MK-54",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-54",
};


pub const MK_55: Subdivision = Subdivision {
    name: "Novaci",
    code: "MK-55",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-55",
};


pub const MK_56: Subdivision = Subdivision {
    name: "Novo Selo",
    code: "MK-56",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-56",
};


pub const MK_57: Subdivision = Subdivision {
    name: "Oslomej",
    code: "MK-57",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-57",
};


pub const MK_58: Subdivision = Subdivision {
    name: "Ohrid",
    code: "MK-58",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-58",
};


pub const MK_59: Subdivision = Subdivision {
    name: "Petrovec",
    code: "MK-59",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-59",
};


pub const MK_60: Subdivision = Subdivision {
    name: "Pehčevo",
    code: "MK-60",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-60",
};


pub const MK_61: Subdivision = Subdivision {
    name: "Plasnica",
    code: "MK-61",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-61",
};


pub const MK_62: Subdivision = Subdivision {
    name: "Prilep",
    code: "MK-62",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-62",
};


pub const MK_63: Subdivision = Subdivision {
    name: "Probištip",
    code: "MK-63",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-63",
};


pub const MK_64: Subdivision = Subdivision {
    name: "Radoviš",
    code: "MK-64",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-64",
};


pub const MK_65: Subdivision = Subdivision {
    name: "Rankovce",
    code: "MK-65",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-65",
};


pub const MK_66: Subdivision = Subdivision {
    name: "Resen",
    code: "MK-66",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-66",
};


pub const MK_67: Subdivision = Subdivision {
    name: "Rosoman",
    code: "MK-67",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-67",
};


pub const MK_68: Subdivision = Subdivision {
    name: "Saraj †",
    code: "MK-68",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-68",
};


pub const MK_69: Subdivision = Subdivision {
    name: "Sveti Nikole",
    code: "MK-69",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-69",
};


pub const MK_70: Subdivision = Subdivision {
    name: "Sopište",
    code: "MK-70",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-70",
};


pub const MK_71: Subdivision = Subdivision {
    name: "Staro Nagoričane",
    code: "MK-71",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-71",
};


pub const MK_72: Subdivision = Subdivision {
    name: "Struga",
    code: "MK-72",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-72",
};


pub const MK_73: Subdivision = Subdivision {
    name: "Strumica",
    code: "MK-73",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-73",
};


pub const MK_74: Subdivision = Subdivision {
    name: "Studeničani",
    code: "MK-74",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-74",
};


pub const MK_75: Subdivision = Subdivision {
    name: "Tearce",
    code: "MK-75",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-75",
};


pub const MK_76: Subdivision = Subdivision {
    name: "Tetovo",
    code: "MK-76",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-76",
};


pub const MK_77: Subdivision = Subdivision {
    name: "Centar †",
    code: "MK-77",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-77",
};


pub const MK_78: Subdivision = Subdivision {
    name: "Centar Župa",
    code: "MK-78",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-78",
};


pub const MK_79: Subdivision = Subdivision {
    name: "Čair †",
    code: "MK-79",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-79",
};


pub const MK_80: Subdivision = Subdivision {
    name: "Čaška",
    code: "MK-80",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-80",
};


pub const MK_81: Subdivision = Subdivision {
    name: "Češinovo-Obleševo",
    code: "MK-81",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-81",
};


pub const MK_82: Subdivision = Subdivision {
    name: "Čučer Sandevo",
    code: "MK-82",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-82",
};


pub const MK_83: Subdivision = Subdivision {
    name: "Štip",
    code: "MK-83",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-83",
};


pub const MK_84: Subdivision = Subdivision {
    name: "Šuto Orizari †",
    code: "MK-84",
    subdivision_type: "Municipality",
    country_name: "Macedonia, the Former Yugoslav Republic Of",
    country_code: "MK",
    region_code: "MK-84",
};


pub const MG_A: Subdivision = Subdivision {
    name: "Toamasina",
    code: "MG-A",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-A",
};


pub const MG_D: Subdivision = Subdivision {
    name: "Antsiranana",
    code: "MG-D",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-D",
};


pub const MG_F: Subdivision = Subdivision {
    name: "Fianarantsoa",
    code: "MG-F",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-F",
};


pub const MG_M: Subdivision = Subdivision {
    name: "Mahajanga",
    code: "MG-M",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-M",
};


pub const MG_T: Subdivision = Subdivision {
    name: "Antananarivo",
    code: "MG-T",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-T",
};


pub const MG_U: Subdivision = Subdivision {
    name: "Toliara",
    code: "MG-U",
    subdivision_type: "Province",
    country_name: "Madagascar",
    country_code: "MG",
    region_code: "MG-U",
};


pub const MW_BA: Subdivision = Subdivision {
    name: "Balaka",
    code: "MW-BA",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-BA",
};


pub const MW_BL: Subdivision = Subdivision {
    name: "Blantyre",
    code: "MW-BL",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-BL",
};


pub const MW_C: Subdivision = Subdivision {
    name: "Central Region",
    code: "MW-C",
    subdivision_type: "Region",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-C",
};


pub const MW_CK: Subdivision = Subdivision {
    name: "Chikwawa",
    code: "MW-CK",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-CK",
};


pub const MW_CR: Subdivision = Subdivision {
    name: "Chiradzulu",
    code: "MW-CR",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-CR",
};


pub const MW_CT: Subdivision = Subdivision {
    name: "Chitipa",
    code: "MW-CT",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-CT",
};


pub const MW_DE: Subdivision = Subdivision {
    name: "Dedza",
    code: "MW-DE",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-DE",
};


pub const MW_DO: Subdivision = Subdivision {
    name: "Dowa",
    code: "MW-DO",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-DO",
};


pub const MW_KR: Subdivision = Subdivision {
    name: "Karonga",
    code: "MW-KR",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-KR",
};


pub const MW_KS: Subdivision = Subdivision {
    name: "Kasungu",
    code: "MW-KS",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-KS",
};


pub const MW_LI: Subdivision = Subdivision {
    name: "Lilongwe",
    code: "MW-LI",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-LI",
};


pub const MW_LK: Subdivision = Subdivision {
    name: "Likoma",
    code: "MW-LK",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-LK",
};


pub const MW_MC: Subdivision = Subdivision {
    name: "Mchinji",
    code: "MW-MC",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MC",
};


pub const MW_MG: Subdivision = Subdivision {
    name: "Mangochi",
    code: "MW-MG",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MG",
};


pub const MW_MH: Subdivision = Subdivision {
    name: "Machinga",
    code: "MW-MH",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MH",
};


pub const MW_MU: Subdivision = Subdivision {
    name: "Mulanje",
    code: "MW-MU",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MU",
};


pub const MW_MW: Subdivision = Subdivision {
    name: "Mwanza",
    code: "MW-MW",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MW",
};


pub const MW_MZ: Subdivision = Subdivision {
    name: "Mzimba",
    code: "MW-MZ",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-MZ",
};


pub const MW_NB: Subdivision = Subdivision {
    name: "Nkhata Bay",
    code: "MW-NB",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NB",
};


pub const MW_N: Subdivision = Subdivision {
    name: "Chakumpoto",
    code: "MW-N",
    subdivision_type: "Region",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-N",
};


pub const MW_NE: Subdivision = Subdivision {
    name: "Neno",
    code: "MW-NE",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NE",
};


pub const MW_NI: Subdivision = Subdivision {
    name: "Ntchisi",
    code: "MW-NI",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NI",
};


pub const MW_NK: Subdivision = Subdivision {
    name: "Nkhotakota",
    code: "MW-NK",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NK",
};


pub const MW_NS: Subdivision = Subdivision {
    name: "Nsanje",
    code: "MW-NS",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NS",
};


pub const MW_NU: Subdivision = Subdivision {
    name: "Ntcheu",
    code: "MW-NU",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-NU",
};


pub const MW_PH: Subdivision = Subdivision {
    name: "Phalombe",
    code: "MW-PH",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-PH",
};


pub const MW_RU: Subdivision = Subdivision {
    name: "Rumphi",
    code: "MW-RU",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-RU",
};


pub const MW_SA: Subdivision = Subdivision {
    name: "Salima",
    code: "MW-SA",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-SA",
};


pub const MW_S: Subdivision = Subdivision {
    name: "Chakumwera",
    code: "MW-S",
    subdivision_type: "Region",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-S",
};


pub const MW_TH: Subdivision = Subdivision {
    name: "Thyolo",
    code: "MW-TH",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-TH",
};


pub const MW_ZO: Subdivision = Subdivision {
    name: "Zomba",
    code: "MW-ZO",
    subdivision_type: "District",
    country_name: "Malawi",
    country_code: "MW",
    region_code: "MW-ZO",
};


pub const MY_01: Subdivision = Subdivision {
    name: "Johor",
    code: "MY-01",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-01",
};


pub const MY_02: Subdivision = Subdivision {
    name: "Kedah",
    code: "MY-02",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-02",
};


pub const MY_03: Subdivision = Subdivision {
    name: "Kelantan",
    code: "MY-03",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-03",
};


pub const MY_04: Subdivision = Subdivision {
    name: "Melaka",
    code: "MY-04",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-04",
};


pub const MY_05: Subdivision = Subdivision {
    name: "Negeri Sembilan",
    code: "MY-05",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-05",
};


pub const MY_06: Subdivision = Subdivision {
    name: "Pahang",
    code: "MY-06",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-06",
};


pub const MY_07: Subdivision = Subdivision {
    name: "Pulau Pinang",
    code: "MY-07",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-07",
};


pub const MY_08: Subdivision = Subdivision {
    name: "Perak",
    code: "MY-08",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-08",
};


pub const MY_09: Subdivision = Subdivision {
    name: "Perlis",
    code: "MY-09",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-09",
};


pub const MY_10: Subdivision = Subdivision {
    name: "Selangor",
    code: "MY-10",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-10",
};


pub const MY_11: Subdivision = Subdivision {
    name: "Terengganu",
    code: "MY-11",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-11",
};


pub const MY_12: Subdivision = Subdivision {
    name: "Sabah",
    code: "MY-12",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-12",
};


pub const MY_13: Subdivision = Subdivision {
    name: "Sarawak",
    code: "MY-13",
    subdivision_type: "State",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-13",
};


pub const MY_14: Subdivision = Subdivision {
    name: "Wilayah Persekutuan Kuala Lumpur",
    code: "MY-14",
    subdivision_type: "Federal territory",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-14",
};


pub const MY_15: Subdivision = Subdivision {
    name: "Wilayah Persekutuan Labuan",
    code: "MY-15",
    subdivision_type: "Federal territory",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-15",
};


pub const MY_16: Subdivision = Subdivision {
    name: "Wilayah Persekutuan Putrajaya",
    code: "MY-16",
    subdivision_type: "Federal territory",
    country_name: "Malaysia",
    country_code: "MY",
    region_code: "MY-16",
};


pub const MV_00: Subdivision = Subdivision {
    name: "Alifu Dhaalu",
    code: "MV-00",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-00",
};


pub const MV_01: Subdivision = Subdivision {
    name: "Addu Atholhu",
    code: "MV-01",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-01",
};


pub const MV_02: Subdivision = Subdivision {
    name: "Alifu Alifu",
    code: "MV-02",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-02",
};


pub const MV_03: Subdivision = Subdivision {
    name: "Faadhippolhu",
    code: "MV-03",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-03",
};


pub const MV_04: Subdivision = Subdivision {
    name: "Felidhu Atholhu",
    code: "MV-04",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-04",
};


pub const MV_05: Subdivision = Subdivision {
    name: "Hadhdhunmathi",
    code: "MV-05",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-05",
};


pub const MV_07: Subdivision = Subdivision {
    name: "Haa Alifu",
    code: "MV-07",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-07",
};


pub const MV_08: Subdivision = Subdivision {
    name: "Kolhumadulu",
    code: "MV-08",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-08",
};


pub const MV_12: Subdivision = Subdivision {
    name: "Meemu",
    code: "MV-12",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-12",
};


pub const MV_13: Subdivision = Subdivision {
    name: "Maalhosmadulu Uthuruburi",
    code: "MV-13",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-13",
};


pub const MV_14: Subdivision = Subdivision {
    name: "Faafu",
    code: "MV-14",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-14",
};


pub const MV_17: Subdivision = Subdivision {
    name: "Dhaalu",
    code: "MV-17",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-17",
};


pub const MV_20: Subdivision = Subdivision {
    name: "Baa",
    code: "MV-20",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-20",
};


pub const MV_23: Subdivision = Subdivision {
    name: "Haa Dhaalu",
    code: "MV-23",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-23",
};


pub const MV_24: Subdivision = Subdivision {
    name: "Miladhunmadulu Uthuruburi",
    code: "MV-24",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-24",
};


pub const MV_25: Subdivision = Subdivision {
    name: "Miladhunmadulu Dhekunuburi",
    code: "MV-25",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-25",
};


pub const MV_26: Subdivision = Subdivision {
    name: "Kaafu",
    code: "MV-26",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-26",
};


pub const MV_27: Subdivision = Subdivision {
    name: "Gaafu Alifu",
    code: "MV-27",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-27",
};


pub const MV_28: Subdivision = Subdivision {
    name: "Gaafu Dhaalu",
    code: "MV-28",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-28",
};


pub const MV_29: Subdivision = Subdivision {
    name: "Fuvammulah",
    code: "MV-29",
    subdivision_type: "Administrative atoll",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-29",
};


pub const MV_CE: Subdivision = Subdivision {
    name: "Central",
    code: "MV-CE",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-CE",
};


pub const MV_MLE: Subdivision = Subdivision {
    name: "Maale",
    code: "MV-MLE",
    subdivision_type: "Capital",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-MLE",
};


pub const MV_NC: Subdivision = Subdivision {
    name: "Medhu-Uthuru",
    code: "MV-NC",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-NC",
};


pub const MV_NO: Subdivision = Subdivision {
    name: "North",
    code: "MV-NO",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-NO",
};


pub const MV_SC: Subdivision = Subdivision {
    name: "Medhu-Dhekunu",
    code: "MV-SC",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-SC",
};


pub const MV_SU: Subdivision = Subdivision {
    name: "Dhekunu",
    code: "MV-SU",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-SU",
};


pub const MV_UN: Subdivision = Subdivision {
    name: "Mathi-Uthuru",
    code: "MV-UN",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-UN",
};


pub const MV_US: Subdivision = Subdivision {
    name: "Mathi-Dhekunu",
    code: "MV-US",
    subdivision_type: "Province",
    country_name: "Maldives",
    country_code: "MV",
    region_code: "MV-US",
};


pub const ML_1: Subdivision = Subdivision {
    name: "Kayes",
    code: "ML-1",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-1",
};


pub const ML_2: Subdivision = Subdivision {
    name: "Koulikoro",
    code: "ML-2",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-2",
};


pub const ML_3: Subdivision = Subdivision {
    name: "Sikasso",
    code: "ML-3",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-3",
};


pub const ML_4: Subdivision = Subdivision {
    name: "Ségou",
    code: "ML-4",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-4",
};


pub const ML_5: Subdivision = Subdivision {
    name: "Mopti",
    code: "ML-5",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-5",
};


pub const ML_6: Subdivision = Subdivision {
    name: "Tombouctou",
    code: "ML-6",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-6",
};


pub const ML_7: Subdivision = Subdivision {
    name: "Gao",
    code: "ML-7",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-7",
};


pub const ML_8: Subdivision = Subdivision {
    name: "Kidal",
    code: "ML-8",
    subdivision_type: "Region",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-8",
};


pub const ML_BKO: Subdivision = Subdivision {
    name: "Bamako",
    code: "ML-BKO",
    subdivision_type: "District",
    country_name: "Mali",
    country_code: "ML",
    region_code: "ML-BKO",
};


pub const MT_01: Subdivision = Subdivision {
    name: "Attard",
    code: "MT-01",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-01",
};


pub const MT_02: Subdivision = Subdivision {
    name: "Balzan",
    code: "MT-02",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-02",
};


pub const MT_03: Subdivision = Subdivision {
    name: "Birgu",
    code: "MT-03",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-03",
};


pub const MT_04: Subdivision = Subdivision {
    name: "Birkirkara",
    code: "MT-04",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-04",
};


pub const MT_05: Subdivision = Subdivision {
    name: "Birżebbuġa",
    code: "MT-05",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-05",
};


pub const MT_06: Subdivision = Subdivision {
    name: "Bormla",
    code: "MT-06",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-06",
};


pub const MT_07: Subdivision = Subdivision {
    name: "Dingli",
    code: "MT-07",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-07",
};


pub const MT_08: Subdivision = Subdivision {
    name: "Fgura",
    code: "MT-08",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-08",
};


pub const MT_09: Subdivision = Subdivision {
    name: "Floriana",
    code: "MT-09",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-09",
};


pub const MT_10: Subdivision = Subdivision {
    name: "Fontana",
    code: "MT-10",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-10",
};


pub const MT_11: Subdivision = Subdivision {
    name: "Gudja",
    code: "MT-11",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-11",
};


pub const MT_12: Subdivision = Subdivision {
    name: "Gżira",
    code: "MT-12",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-12",
};


pub const MT_13: Subdivision = Subdivision {
    name: "Għajnsielem",
    code: "MT-13",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-13",
};


pub const MT_14: Subdivision = Subdivision {
    name: "Għarb",
    code: "MT-14",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-14",
};


pub const MT_15: Subdivision = Subdivision {
    name: "Għargħur",
    code: "MT-15",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-15",
};


pub const MT_16: Subdivision = Subdivision {
    name: "Għasri",
    code: "MT-16",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-16",
};


pub const MT_17: Subdivision = Subdivision {
    name: "Għaxaq",
    code: "MT-17",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-17",
};


pub const MT_18: Subdivision = Subdivision {
    name: "Ħamrun",
    code: "MT-18",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-18",
};


pub const MT_19: Subdivision = Subdivision {
    name: "Iklin",
    code: "MT-19",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-19",
};


pub const MT_20: Subdivision = Subdivision {
    name: "Isla",
    code: "MT-20",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-20",
};


pub const MT_21: Subdivision = Subdivision {
    name: "Kalkara",
    code: "MT-21",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-21",
};


pub const MT_22: Subdivision = Subdivision {
    name: "Kerċem",
    code: "MT-22",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-22",
};


pub const MT_23: Subdivision = Subdivision {
    name: "Kirkop",
    code: "MT-23",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-23",
};


pub const MT_24: Subdivision = Subdivision {
    name: "Lija",
    code: "MT-24",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-24",
};


pub const MT_25: Subdivision = Subdivision {
    name: "Luqa",
    code: "MT-25",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-25",
};


pub const MT_26: Subdivision = Subdivision {
    name: "Marsa",
    code: "MT-26",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-26",
};


pub const MT_27: Subdivision = Subdivision {
    name: "Marsaskala",
    code: "MT-27",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-27",
};


pub const MT_28: Subdivision = Subdivision {
    name: "Marsaxlokk",
    code: "MT-28",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-28",
};


pub const MT_29: Subdivision = Subdivision {
    name: "Mdina",
    code: "MT-29",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-29",
};


pub const MT_30: Subdivision = Subdivision {
    name: "Mellieħa",
    code: "MT-30",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-30",
};


pub const MT_31: Subdivision = Subdivision {
    name: "Mġarr",
    code: "MT-31",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-31",
};


pub const MT_32: Subdivision = Subdivision {
    name: "Mosta",
    code: "MT-32",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-32",
};


pub const MT_33: Subdivision = Subdivision {
    name: "Mqabba",
    code: "MT-33",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-33",
};


pub const MT_34: Subdivision = Subdivision {
    name: "Msida",
    code: "MT-34",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-34",
};


pub const MT_35: Subdivision = Subdivision {
    name: "Mtarfa",
    code: "MT-35",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-35",
};


pub const MT_36: Subdivision = Subdivision {
    name: "Munxar",
    code: "MT-36",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-36",
};


pub const MT_37: Subdivision = Subdivision {
    name: "Nadur",
    code: "MT-37",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-37",
};


pub const MT_38: Subdivision = Subdivision {
    name: "Naxxar",
    code: "MT-38",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-38",
};


pub const MT_39: Subdivision = Subdivision {
    name: "Paola",
    code: "MT-39",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-39",
};


pub const MT_40: Subdivision = Subdivision {
    name: "Pembroke",
    code: "MT-40",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-40",
};


pub const MT_41: Subdivision = Subdivision {
    name: "Pietà",
    code: "MT-41",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-41",
};


pub const MT_42: Subdivision = Subdivision {
    name: "Qala",
    code: "MT-42",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-42",
};


pub const MT_43: Subdivision = Subdivision {
    name: "Qormi",
    code: "MT-43",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-43",
};


pub const MT_44: Subdivision = Subdivision {
    name: "Qrendi",
    code: "MT-44",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-44",
};


pub const MT_45: Subdivision = Subdivision {
    name: "Rabat Gozo",
    code: "MT-45",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-45",
};


pub const MT_46: Subdivision = Subdivision {
    name: "Rabat Malta",
    code: "MT-46",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-46",
};


pub const MT_47: Subdivision = Subdivision {
    name: "Safi",
    code: "MT-47",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-47",
};


pub const MT_48: Subdivision = Subdivision {
    name: "Saint Julian’s",
    code: "MT-48",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-48",
};


pub const MT_49: Subdivision = Subdivision {
    name: "Saint John",
    code: "MT-49",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-49",
};


pub const MT_50: Subdivision = Subdivision {
    name: "Saint Lawrence",
    code: "MT-50",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-50",
};


pub const MT_51: Subdivision = Subdivision {
    name: "Saint Paul’s Bay",
    code: "MT-51",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-51",
};


pub const MT_52: Subdivision = Subdivision {
    name: "Sannat",
    code: "MT-52",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-52",
};


pub const MT_53: Subdivision = Subdivision {
    name: "Saint Lucia’s",
    code: "MT-53",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-53",
};


pub const MT_54: Subdivision = Subdivision {
    name: "Santa Venera",
    code: "MT-54",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-54",
};


pub const MT_55: Subdivision = Subdivision {
    name: "Siġġiewi",
    code: "MT-55",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-55",
};


pub const MT_56: Subdivision = Subdivision {
    name: "Sliema",
    code: "MT-56",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-56",
};


pub const MT_57: Subdivision = Subdivision {
    name: "Swieqi",
    code: "MT-57",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-57",
};


pub const MT_58: Subdivision = Subdivision {
    name: "Ta’ Xbiex",
    code: "MT-58",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-58",
};


pub const MT_59: Subdivision = Subdivision {
    name: "Tarxien",
    code: "MT-59",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-59",
};


pub const MT_60: Subdivision = Subdivision {
    name: "Valletta",
    code: "MT-60",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-60",
};


pub const MT_61: Subdivision = Subdivision {
    name: "Xagħra",
    code: "MT-61",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-61",
};


pub const MT_62: Subdivision = Subdivision {
    name: "Xewkija",
    code: "MT-62",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-62",
};


pub const MT_63: Subdivision = Subdivision {
    name: "Xgħajra",
    code: "MT-63",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-63",
};


pub const MT_64: Subdivision = Subdivision {
    name: "Żabbar",
    code: "MT-64",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-64",
};


pub const MT_65: Subdivision = Subdivision {
    name: "Żebbuġ Gozo",
    code: "MT-65",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-65",
};


pub const MT_66: Subdivision = Subdivision {
    name: "Żebbuġ Malta",
    code: "MT-66",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-66",
};


pub const MT_67: Subdivision = Subdivision {
    name: "Żejtun",
    code: "MT-67",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-67",
};


pub const MT_68: Subdivision = Subdivision {
    name: "Żurrieq",
    code: "MT-68",
    subdivision_type: "Local council",
    country_name: "Malta",
    country_code: "MT",
    region_code: "MT-68",
};


pub const MH_ALK: Subdivision = Subdivision {
    name: "Ailuk",
    code: "MH-ALK",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-ALK",
};


pub const MH_ALL: Subdivision = Subdivision {
    name: "Ailinglaplap",
    code: "MH-ALL",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-ALL",
};


pub const MH_ARN: Subdivision = Subdivision {
    name: "Arno",
    code: "MH-ARN",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-ARN",
};


pub const MH_AUR: Subdivision = Subdivision {
    name: "Aur",
    code: "MH-AUR",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-AUR",
};


pub const MH_EBO: Subdivision = Subdivision {
    name: "Ebon",
    code: "MH-EBO",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-EBO",
};


pub const MH_ENI: Subdivision = Subdivision {
    name: "Enewetak and Ujelang",
    code: "MH-ENI",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-ENI",
};


pub const MH_JAB: Subdivision = Subdivision {
    name: "Jabat",
    code: "MH-JAB",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-JAB",
};


pub const MH_JAL: Subdivision = Subdivision {
    name: "Jaluit",
    code: "MH-JAL",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-JAL",
};


pub const MH_KIL: Subdivision = Subdivision {
    name: "Bikini and Kili",
    code: "MH-KIL",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-KIL",
};


pub const MH_KWA: Subdivision = Subdivision {
    name: "Kwajalein",
    code: "MH-KWA",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-KWA",
};


pub const MH_LAE: Subdivision = Subdivision {
    name: "Lae",
    code: "MH-LAE",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-LAE",
};


pub const MH_LIB: Subdivision = Subdivision {
    name: "Lib",
    code: "MH-LIB",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-LIB",
};


pub const MH_LIK: Subdivision = Subdivision {
    name: "Likiep",
    code: "MH-LIK",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-LIK",
};


pub const MH_L: Subdivision = Subdivision {
    name: "Ralik chain",
    code: "MH-L",
    subdivision_type: "Chain",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-L",
};


pub const MH_MAJ: Subdivision = Subdivision {
    name: "Majuro",
    code: "MH-MAJ",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-MAJ",
};


pub const MH_MAL: Subdivision = Subdivision {
    name: "Maloelap",
    code: "MH-MAL",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-MAL",
};


pub const MH_MEJ: Subdivision = Subdivision {
    name: "Mejit",
    code: "MH-MEJ",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-MEJ",
};


pub const MH_MIL: Subdivision = Subdivision {
    name: "Mili",
    code: "MH-MIL",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-MIL",
};


pub const MH_NMK: Subdivision = Subdivision {
    name: "Namdrik",
    code: "MH-NMK",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-NMK",
};


pub const MH_NMU: Subdivision = Subdivision {
    name: "Namu",
    code: "MH-NMU",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-NMU",
};


pub const MH_RON: Subdivision = Subdivision {
    name: "Rongelap",
    code: "MH-RON",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-RON",
};


pub const MH_T: Subdivision = Subdivision {
    name: "Ratak chain",
    code: "MH-T",
    subdivision_type: "Chain",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-T",
};


pub const MH_UJA: Subdivision = Subdivision {
    name: "Ujae",
    code: "MH-UJA",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-UJA",
};


pub const MH_UTI: Subdivision = Subdivision {
    name: "Utrik",
    code: "MH-UTI",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-UTI",
};


pub const MH_WTH: Subdivision = Subdivision {
    name: "Wotho",
    code: "MH-WTH",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-WTH",
};


pub const MH_WTJ: Subdivision = Subdivision {
    name: "Wotje",
    code: "MH-WTJ",
    subdivision_type: "Municipality",
    country_name: "Marshall Islands",
    country_code: "MH",
    region_code: "MH-WTJ",
};


pub const MQ_MQ: Subdivision = Subdivision {
    name: "Martinique",
    code: "MQ-MQ",
    subdivision_type: "Country",
    country_name: "Martinique",
    country_code: "MQ",
    region_code: "MQ-MQ",
};


pub const MR_01: Subdivision = Subdivision {
    name: "Hodh ech Chargui",
    code: "MR-01",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-01",
};


pub const MR_02: Subdivision = Subdivision {
    name: "Hodh el Gharbi",
    code: "MR-02",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-02",
};


pub const MR_03: Subdivision = Subdivision {
    name: "Assaba",
    code: "MR-03",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-03",
};


pub const MR_04: Subdivision = Subdivision {
    name: "Gorgol",
    code: "MR-04",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-04",
};


pub const MR_05: Subdivision = Subdivision {
    name: "Brakna",
    code: "MR-05",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-05",
};


pub const MR_06: Subdivision = Subdivision {
    name: "Trarza",
    code: "MR-06",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-06",
};


pub const MR_07: Subdivision = Subdivision {
    name: "Adrar",
    code: "MR-07",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-07",
};


pub const MR_08: Subdivision = Subdivision {
    name: "Dakhlet Nouâdhibou",
    code: "MR-08",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-08",
};


pub const MR_09: Subdivision = Subdivision {
    name: "Tagant",
    code: "MR-09",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-09",
};


pub const MR_10: Subdivision = Subdivision {
    name: "Guidimaka",
    code: "MR-10",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-10",
};


pub const MR_11: Subdivision = Subdivision {
    name: "Tiris Zemmour",
    code: "MR-11",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-11",
};


pub const MR_12: Subdivision = Subdivision {
    name: "Inchiri",
    code: "MR-12",
    subdivision_type: "Region",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-12",
};


pub const MR_NKC: Subdivision = Subdivision {
    name: "Nouakchott",
    code: "MR-NKC",
    subdivision_type: "District",
    country_name: "Mauritania",
    country_code: "MR",
    region_code: "MR-NKC",
};


pub const MU_AG: Subdivision = Subdivision {
    name: "Agalega Islands",
    code: "MU-AG",
    subdivision_type: "Dependency",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-AG",
};


pub const MU_BL: Subdivision = Subdivision {
    name: "Black River",
    code: "MU-BL",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-BL",
};


pub const MU_BR: Subdivision = Subdivision {
    name: "Beau Bassin-Rose Hill",
    code: "MU-BR",
    subdivision_type: "City",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-BR",
};


pub const MU_CC: Subdivision = Subdivision {
    name: "Cargados Carajos Shoals",
    code: "MU-CC",
    subdivision_type: "Dependency",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-CC",
};


pub const MU_CU: Subdivision = Subdivision {
    name: "Curepipe",
    code: "MU-CU",
    subdivision_type: "City",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-CU",
};


pub const MU_FL: Subdivision = Subdivision {
    name: "Flacq",
    code: "MU-FL",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-FL",
};


pub const MU_GP: Subdivision = Subdivision {
    name: "Grand Port",
    code: "MU-GP",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-GP",
};


pub const MU_MO: Subdivision = Subdivision {
    name: "Moka",
    code: "MU-MO",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-MO",
};


pub const MU_PA: Subdivision = Subdivision {
    name: "Pamplemousses",
    code: "MU-PA",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-PA",
};


pub const MU_PL: Subdivision = Subdivision {
    name: "Port Louis",
    code: "MU-PL",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-PL",
};


pub const MU_PU: Subdivision = Subdivision {
    name: "Port Louis",
    code: "MU-PU",
    subdivision_type: "City",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-PU",
};


pub const MU_PW: Subdivision = Subdivision {
    name: "Plaines Wilhems",
    code: "MU-PW",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-PW",
};


pub const MU_QB: Subdivision = Subdivision {
    name: "Quatre Bornes",
    code: "MU-QB",
    subdivision_type: "City",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-QB",
};


pub const MU_RO: Subdivision = Subdivision {
    name: "Rodrigues Island",
    code: "MU-RO",
    subdivision_type: "Dependency",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-RO",
};


pub const MU_RR: Subdivision = Subdivision {
    name: "Rivière du Rempart",
    code: "MU-RR",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-RR",
};


pub const MU_SA: Subdivision = Subdivision {
    name: "Savanne",
    code: "MU-SA",
    subdivision_type: "District",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-SA",
};


pub const MU_VP: Subdivision = Subdivision {
    name: "Vacoas-Phoenix",
    code: "MU-VP",
    subdivision_type: "City",
    country_name: "Mauritius",
    country_code: "MU",
    region_code: "MU-VP",
};


pub const YT_YT: Subdivision = Subdivision {
    name: "Mayotte",
    code: "YT-YT",
    subdivision_type: "Country",
    country_name: "Mayotte",
    country_code: "YT",
    region_code: "YT-YT",
};


pub const MX_AGU: Subdivision = Subdivision {
    name: "Aguascalientes",
    code: "MX-AGU",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-AGU",
};


pub const MX_BCN: Subdivision = Subdivision {
    name: "Baja California",
    code: "MX-BCN",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-BCN",
};


pub const MX_BCS: Subdivision = Subdivision {
    name: "Baja California Sur",
    code: "MX-BCS",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-BCS",
};


pub const MX_CAM: Subdivision = Subdivision {
    name: "Campeche",
    code: "MX-CAM",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-CAM",
};


pub const MX_CHH: Subdivision = Subdivision {
    name: "Chihuahua",
    code: "MX-CHH",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-CHH",
};


pub const MX_CHP: Subdivision = Subdivision {
    name: "Chiapas",
    code: "MX-CHP",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-CHP",
};


pub const MX_COA: Subdivision = Subdivision {
    name: "Coahuila de Zaragoza",
    code: "MX-COA",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-COA",
};


pub const MX_COL: Subdivision = Subdivision {
    name: "Colima",
    code: "MX-COL",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-COL",
};


pub const MX_CMX: Subdivision = Subdivision {
    name: "Ciudad de México",
    code: "MX-CMX",
    subdivision_type: "Federal District",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-CMX",
};


pub const MX_DUR: Subdivision = Subdivision {
    name: "Durango",
    code: "MX-DUR",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-DUR",
};


pub const MX_GRO: Subdivision = Subdivision {
    name: "Guerrero",
    code: "MX-GRO",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-GRO",
};


pub const MX_GUA: Subdivision = Subdivision {
    name: "Guanajuato",
    code: "MX-GUA",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-GUA",
};


pub const MX_HID: Subdivision = Subdivision {
    name: "Hidalgo",
    code: "MX-HID",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-HID",
};


pub const MX_JAL: Subdivision = Subdivision {
    name: "Jalisco",
    code: "MX-JAL",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-JAL",
};


pub const MX_MEX: Subdivision = Subdivision {
    name: "México",
    code: "MX-MEX",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-MEX",
};


pub const MX_MIC: Subdivision = Subdivision {
    name: "Michoacán de Ocampo",
    code: "MX-MIC",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-MIC",
};


pub const MX_MOR: Subdivision = Subdivision {
    name: "Morelos",
    code: "MX-MOR",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-MOR",
};


pub const MX_NAY: Subdivision = Subdivision {
    name: "Nayarit",
    code: "MX-NAY",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-NAY",
};


pub const MX_NLE: Subdivision = Subdivision {
    name: "Nuevo León",
    code: "MX-NLE",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-NLE",
};


pub const MX_OAX: Subdivision = Subdivision {
    name: "Oaxaca",
    code: "MX-OAX",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-OAX",
};


pub const MX_PUE: Subdivision = Subdivision {
    name: "Puebla",
    code: "MX-PUE",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-PUE",
};


pub const MX_QUE: Subdivision = Subdivision {
    name: "Querétaro",
    code: "MX-QUE",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-QUE",
};


pub const MX_ROO: Subdivision = Subdivision {
    name: "Quintana Roo",
    code: "MX-ROO",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-ROO",
};


pub const MX_SIN: Subdivision = Subdivision {
    name: "Sinaloa",
    code: "MX-SIN",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-SIN",
};


pub const MX_SLP: Subdivision = Subdivision {
    name: "San Luis Potosí",
    code: "MX-SLP",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-SLP",
};


pub const MX_SON: Subdivision = Subdivision {
    name: "Sonora",
    code: "MX-SON",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-SON",
};


pub const MX_TAB: Subdivision = Subdivision {
    name: "Tabasco",
    code: "MX-TAB",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-TAB",
};


pub const MX_TAM: Subdivision = Subdivision {
    name: "Tamaulipas",
    code: "MX-TAM",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-TAM",
};


pub const MX_TLA: Subdivision = Subdivision {
    name: "Tlaxcala",
    code: "MX-TLA",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-TLA",
};


pub const MX_VER: Subdivision = Subdivision {
    name: "Veracruz de Ignacio de la Llave",
    code: "MX-VER",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-VER",
};


pub const MX_YUC: Subdivision = Subdivision {
    name: "Yucatán",
    code: "MX-YUC",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-YUC",
};


pub const MX_ZAC: Subdivision = Subdivision {
    name: "Zacatecas",
    code: "MX-ZAC",
    subdivision_type: "State",
    country_name: "Mexico",
    country_code: "MX",
    region_code: "MX-ZAC",
};


pub const FM_KSA: Subdivision = Subdivision {
    name: "Kosrae",
    code: "FM-KSA",
    subdivision_type: "State",
    country_name: "Micronesia, Federated States Of",
    country_code: "FM",
    region_code: "FM-KSA",
};


pub const FM_PNI: Subdivision = Subdivision {
    name: "Pohnpei",
    code: "FM-PNI",
    subdivision_type: "State",
    country_name: "Micronesia, Federated States Of",
    country_code: "FM",
    region_code: "FM-PNI",
};


pub const FM_TRK: Subdivision = Subdivision {
    name: "Chuuk",
    code: "FM-TRK",
    subdivision_type: "State",
    country_name: "Micronesia, Federated States Of",
    country_code: "FM",
    region_code: "FM-TRK",
};


pub const FM_YAP: Subdivision = Subdivision {
    name: "Yap",
    code: "FM-YAP",
    subdivision_type: "State",
    country_name: "Micronesia, Federated States Of",
    country_code: "FM",
    region_code: "FM-YAP",
};


pub const MD_AN: Subdivision = Subdivision {
    name: "Anenii Noi",
    code: "MD-AN",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-AN",
};


pub const MD_BA: Subdivision = Subdivision {
    name: "Bălţi",
    code: "MD-BA",
    subdivision_type: "City",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-BA",
};


pub const MD_BD: Subdivision = Subdivision {
    name: "Bender",
    code: "MD-BD",
    subdivision_type: "City",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-BD",
};


pub const MD_BR: Subdivision = Subdivision {
    name: "Briceni",
    code: "MD-BR",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-BR",
};


pub const MD_BS: Subdivision = Subdivision {
    name: "Basarabeasca",
    code: "MD-BS",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-BS",
};


pub const MD_CA: Subdivision = Subdivision {
    name: "Cahul",
    code: "MD-CA",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CA",
};


pub const MD_CL: Subdivision = Subdivision {
    name: "Călăraşi",
    code: "MD-CL",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CL",
};


pub const MD_CM: Subdivision = Subdivision {
    name: "Cimişlia",
    code: "MD-CM",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CM",
};


pub const MD_CR: Subdivision = Subdivision {
    name: "Criuleni",
    code: "MD-CR",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CR",
};


pub const MD_CS: Subdivision = Subdivision {
    name: "Căuşeni",
    code: "MD-CS",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CS",
};


pub const MD_CT: Subdivision = Subdivision {
    name: "Cantemir",
    code: "MD-CT",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CT",
};


pub const MD_CU: Subdivision = Subdivision {
    name: "Chişinău",
    code: "MD-CU",
    subdivision_type: "City",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-CU",
};


pub const MD_DO: Subdivision = Subdivision {
    name: "Donduşeni",
    code: "MD-DO",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-DO",
};


pub const MD_DR: Subdivision = Subdivision {
    name: "Drochia",
    code: "MD-DR",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-DR",
};


pub const MD_DU: Subdivision = Subdivision {
    name: "Dubăsari",
    code: "MD-DU",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-DU",
};


pub const MD_ED: Subdivision = Subdivision {
    name: "Edineţ",
    code: "MD-ED",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-ED",
};


pub const MD_FA: Subdivision = Subdivision {
    name: "Făleşti",
    code: "MD-FA",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-FA",
};


pub const MD_FL: Subdivision = Subdivision {
    name: "Floreşti",
    code: "MD-FL",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-FL",
};


pub const MD_GA: Subdivision = Subdivision {
    name: "Găgăuzia, Unitatea teritorială autonomă",
    code: "MD-GA",
    subdivision_type: "Autonomous territorial unit",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-GA",
};


pub const MD_GL: Subdivision = Subdivision {
    name: "Glodeni",
    code: "MD-GL",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-GL",
};


pub const MD_HI: Subdivision = Subdivision {
    name: "Hînceşti",
    code: "MD-HI",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-HI",
};


pub const MD_IA: Subdivision = Subdivision {
    name: "Ialoveni",
    code: "MD-IA",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-IA",
};


pub const MD_LE: Subdivision = Subdivision {
    name: "Leova",
    code: "MD-LE",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-LE",
};


pub const MD_NI: Subdivision = Subdivision {
    name: "Nisporeni",
    code: "MD-NI",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-NI",
};


pub const MD_OC: Subdivision = Subdivision {
    name: "Ocniţa",
    code: "MD-OC",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-OC",
};


pub const MD_OR: Subdivision = Subdivision {
    name: "Orhei",
    code: "MD-OR",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-OR",
};


pub const MD_RE: Subdivision = Subdivision {
    name: "Rezina",
    code: "MD-RE",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-RE",
};


pub const MD_RI: Subdivision = Subdivision {
    name: "Rîşcani",
    code: "MD-RI",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-RI",
};


pub const MD_SD: Subdivision = Subdivision {
    name: "Şoldăneşti",
    code: "MD-SD",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-SD",
};


pub const MD_SI: Subdivision = Subdivision {
    name: "Sîngerei",
    code: "MD-SI",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-SI",
};


pub const MD_SN: Subdivision = Subdivision {
    name: "Stînga Nistrului, unitatea teritorială din",
    code: "MD-SN",
    subdivision_type: "Territorial unit",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-SN",
};


pub const MD_SO: Subdivision = Subdivision {
    name: "Soroca",
    code: "MD-SO",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-SO",
};


pub const MD_ST: Subdivision = Subdivision {
    name: "Străşeni",
    code: "MD-ST",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-ST",
};


pub const MD_SV: Subdivision = Subdivision {
    name: "Ştefan Vodă",
    code: "MD-SV",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-SV",
};


pub const MD_TA: Subdivision = Subdivision {
    name: "Taraclia",
    code: "MD-TA",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-TA",
};


pub const MD_TE: Subdivision = Subdivision {
    name: "Teleneşti",
    code: "MD-TE",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-TE",
};


pub const MD_UN: Subdivision = Subdivision {
    name: "Ungheni",
    code: "MD-UN",
    subdivision_type: "District",
    country_name: "Moldova",
    country_code: "MD",
    region_code: "MD-UN",
};


pub const MC_CL: Subdivision = Subdivision {
    name: "La Colle",
    code: "MC-CL",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-CL",
};


pub const MC_CO: Subdivision = Subdivision {
    name: "La Condamine",
    code: "MC-CO",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-CO",
};


pub const MC_FO: Subdivision = Subdivision {
    name: "Fontvieille",
    code: "MC-FO",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-FO",
};


pub const MC_GA: Subdivision = Subdivision {
    name: "La Gare",
    code: "MC-GA",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-GA",
};


pub const MC_JE: Subdivision = Subdivision {
    name: "Jardin Exotique",
    code: "MC-JE",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-JE",
};


pub const MC_LA: Subdivision = Subdivision {
    name: "Larvotto",
    code: "MC-LA",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-LA",
};


pub const MC_MA: Subdivision = Subdivision {
    name: "Malbousquet",
    code: "MC-MA",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-MA",
};


pub const MC_MC: Subdivision = Subdivision {
    name: "Monte-Carlo",
    code: "MC-MC",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-MC",
};


pub const MC_MG: Subdivision = Subdivision {
    name: "Moneghetti",
    code: "MC-MG",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-MG",
};


pub const MC_MO: Subdivision = Subdivision {
    name: "Monaco-Ville",
    code: "MC-MO",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-MO",
};


pub const MC_MU: Subdivision = Subdivision {
    name: "Moulins",
    code: "MC-MU",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-MU",
};


pub const MC_PH: Subdivision = Subdivision {
    name: "Port-Hercule",
    code: "MC-PH",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-PH",
};


pub const MC_SD: Subdivision = Subdivision {
    name: "Sainte-Dévote",
    code: "MC-SD",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-SD",
};


pub const MC_SO: Subdivision = Subdivision {
    name: "La Source",
    code: "MC-SO",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-SO",
};


pub const MC_SP: Subdivision = Subdivision {
    name: "Spélugues",
    code: "MC-SP",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-SP",
};


pub const MC_SR: Subdivision = Subdivision {
    name: "Saint-Roman",
    code: "MC-SR",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-SR",
};


pub const MC_VR: Subdivision = Subdivision {
    name: "Vallon de la Rousse",
    code: "MC-VR",
    subdivision_type: "Quarter",
    country_name: "Monaco",
    country_code: "MC",
    region_code: "MC-VR",
};


pub const MN_035: Subdivision = Subdivision {
    name: "Orhon",
    code: "MN-035",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-035",
};


pub const MN_037: Subdivision = Subdivision {
    name: "Darhan uul",
    code: "MN-037",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-037",
};


pub const MN_039: Subdivision = Subdivision {
    name: "Hentiy",
    code: "MN-039",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-039",
};


pub const MN_041: Subdivision = Subdivision {
    name: "Hövsgöl",
    code: "MN-041",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-041",
};


pub const MN_043: Subdivision = Subdivision {
    name: "Hovd",
    code: "MN-043",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-043",
};


pub const MN_046: Subdivision = Subdivision {
    name: "Uvs",
    code: "MN-046",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-046",
};


pub const MN_047: Subdivision = Subdivision {
    name: "Töv",
    code: "MN-047",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-047",
};


pub const MN_049: Subdivision = Subdivision {
    name: "Selenge",
    code: "MN-049",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-049",
};


pub const MN_051: Subdivision = Subdivision {
    name: "Sühbaatar",
    code: "MN-051",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-051",
};


pub const MN_053: Subdivision = Subdivision {
    name: "Ömnögovĭ",
    code: "MN-053",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-053",
};


pub const MN_055: Subdivision = Subdivision {
    name: "Övörhangay",
    code: "MN-055",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-055",
};


pub const MN_057: Subdivision = Subdivision {
    name: "Dzavhan",
    code: "MN-057",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-057",
};


pub const MN_059: Subdivision = Subdivision {
    name: "Dundgovĭ",
    code: "MN-059",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-059",
};


pub const MN_061: Subdivision = Subdivision {
    name: "Dornod",
    code: "MN-061",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-061",
};


pub const MN_063: Subdivision = Subdivision {
    name: "Dornogovĭ",
    code: "MN-063",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-063",
};


pub const MN_064: Subdivision = Subdivision {
    name: "Govĭ-Sümber",
    code: "MN-064",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-064",
};


pub const MN_065: Subdivision = Subdivision {
    name: "Govĭ-Altay",
    code: "MN-065",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-065",
};


pub const MN_067: Subdivision = Subdivision {
    name: "Bulgan",
    code: "MN-067",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-067",
};


pub const MN_069: Subdivision = Subdivision {
    name: "Bayanhongor",
    code: "MN-069",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-069",
};


pub const MN_071: Subdivision = Subdivision {
    name: "Bayan-Ölgiy",
    code: "MN-071",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-071",
};


pub const MN_073: Subdivision = Subdivision {
    name: "Arhangay",
    code: "MN-073",
    subdivision_type: "Province",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-073",
};


pub const MN_1: Subdivision = Subdivision {
    name: "Ulaanbaatar",
    code: "MN-1",
    subdivision_type: "Capital city",
    country_name: "Mongolia",
    country_code: "MN",
    region_code: "MN-1",
};


pub const ME_01: Subdivision = Subdivision {
    name: "Andrijevica",
    code: "ME-01",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-01",
};


pub const ME_02: Subdivision = Subdivision {
    name: "Bar",
    code: "ME-02",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-02",
};


pub const ME_03: Subdivision = Subdivision {
    name: "Berane",
    code: "ME-03",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-03",
};


pub const ME_04: Subdivision = Subdivision {
    name: "Bijelo Polje",
    code: "ME-04",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-04",
};


pub const ME_05: Subdivision = Subdivision {
    name: "Budva",
    code: "ME-05",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-05",
};


pub const ME_06: Subdivision = Subdivision {
    name: "Cetinje",
    code: "ME-06",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-06",
};


pub const ME_07: Subdivision = Subdivision {
    name: "Danilovgrad",
    code: "ME-07",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-07",
};


pub const ME_08: Subdivision = Subdivision {
    name: "Herceg-Novi",
    code: "ME-08",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-08",
};


pub const ME_09: Subdivision = Subdivision {
    name: "Kolašin",
    code: "ME-09",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-09",
};


pub const ME_10: Subdivision = Subdivision {
    name: "Kotor",
    code: "ME-10",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-10",
};


pub const ME_11: Subdivision = Subdivision {
    name: "Mojkovac",
    code: "ME-11",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-11",
};


pub const ME_12: Subdivision = Subdivision {
    name: "Nikšić",
    code: "ME-12",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-12",
};


pub const ME_13: Subdivision = Subdivision {
    name: "Plav",
    code: "ME-13",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-13",
};


pub const ME_14: Subdivision = Subdivision {
    name: "Pljevlja",
    code: "ME-14",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-14",
};


pub const ME_15: Subdivision = Subdivision {
    name: "Plužine",
    code: "ME-15",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-15",
};


pub const ME_16: Subdivision = Subdivision {
    name: "Podgorica",
    code: "ME-16",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-16",
};


pub const ME_17: Subdivision = Subdivision {
    name: "Rožaje",
    code: "ME-17",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-17",
};


pub const ME_18: Subdivision = Subdivision {
    name: "Šavnik",
    code: "ME-18",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-18",
};


pub const ME_19: Subdivision = Subdivision {
    name: "Tivat",
    code: "ME-19",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-19",
};


pub const ME_20: Subdivision = Subdivision {
    name: "Ulcinj",
    code: "ME-20",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-20",
};


pub const ME_21: Subdivision = Subdivision {
    name: "Žabljak",
    code: "ME-21",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-21",
};


pub const ME_22: Subdivision = Subdivision {
    name: "Gusinje",
    code: "ME-22",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-22",
};


pub const ME_23: Subdivision = Subdivision {
    name: "Petnjica",
    code: "ME-23",
    subdivision_type: "Municipality",
    country_name: "Montenegro",
    country_code: "ME",
    region_code: "ME-23",
};


pub const MS_MS: Subdivision = Subdivision {
    name: "Montserrat",
    code: "MS-MS",
    subdivision_type: "Country",
    country_name: "Montserrat",
    country_code: "MS",
    region_code: "MS-MS",
};


pub const MA_01: Subdivision = Subdivision {
    name: "Tanger-Tétouan",
    code: "MA-01",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-01",
};


pub const MA_02: Subdivision = Subdivision {
    name: "Gharb-Chrarda-Beni Hssen",
    code: "MA-02",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-02",
};


pub const MA_03: Subdivision = Subdivision {
    name: "Taza-Al Hoceima-Taounate",
    code: "MA-03",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-03",
};


pub const MA_04: Subdivision = Subdivision {
    name: "L'Oriental",
    code: "MA-04",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-04",
};


pub const MA_05: Subdivision = Subdivision {
    name: "Fès-Boulemane",
    code: "MA-05",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-05",
};


pub const MA_06: Subdivision = Subdivision {
    name: "Meknès-Tafilalet",
    code: "MA-06",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-06",
};


pub const MA_07: Subdivision = Subdivision {
    name: "Rabat-Salé-Zemmour-Zaer",
    code: "MA-07",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-07",
};


pub const MA_08: Subdivision = Subdivision {
    name: "Grand Casablanca",
    code: "MA-08",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-08",
};


pub const MA_09: Subdivision = Subdivision {
    name: "Chaouia-Ouardigha",
    code: "MA-09",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-09",
};


pub const MA_10: Subdivision = Subdivision {
    name: "Doukkala-Abda",
    code: "MA-10",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-10",
};


pub const MA_11: Subdivision = Subdivision {
    name: "Marrakech-Tensift-Al Haouz",
    code: "MA-11",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-11",
};


pub const MA_12: Subdivision = Subdivision {
    name: "Tadla-Azilal",
    code: "MA-12",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-12",
};


pub const MA_13: Subdivision = Subdivision {
    name: "Sous-Massa-Draa",
    code: "MA-13",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-13",
};


pub const MA_14: Subdivision = Subdivision {
    name: "Guelmim-Es Semara",
    code: "MA-14",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-14",
};


pub const MA_15: Subdivision = Subdivision {
    name: "Laâyoune-Boujdour-Sakia el Hamra",
    code: "MA-15",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-15",
};


pub const MA_16: Subdivision = Subdivision {
    name: "Oued ed Dahab-Lagouira",
    code: "MA-16",
    subdivision_type: "Region",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-16",
};


pub const MA_AGD: Subdivision = Subdivision {
    name: "Agadir-Ida-Outanane",
    code: "MA-AGD",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-AGD",
};


pub const MA_AOU: Subdivision = Subdivision {
    name: "Aousserd",
    code: "MA-AOU",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-AOU",
};


pub const MA_ASZ: Subdivision = Subdivision {
    name: "Assa-Zag",
    code: "MA-ASZ",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-ASZ",
};


pub const MA_AZI: Subdivision = Subdivision {
    name: "Azilal",
    code: "MA-AZI",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-AZI",
};


pub const MA_BEM: Subdivision = Subdivision {
    name: "Beni Mellal",
    code: "MA-BEM",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-BEM",
};


pub const MA_BER: Subdivision = Subdivision {
    name: "Berkane",
    code: "MA-BER",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-BER",
};


pub const MA_BES: Subdivision = Subdivision {
    name: "Ben Slimane",
    code: "MA-BES",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-BES",
};


pub const MA_BOD: Subdivision = Subdivision {
    name: "Boujdour",
    code: "MA-BOD",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-BOD",
};


pub const MA_BOM: Subdivision = Subdivision {
    name: "Boulemane",
    code: "MA-BOM",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-BOM",
};


pub const MA_CAS: Subdivision = Subdivision {
    name: "Casablanca",
    code: "MA-CAS",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-CAS",
};


pub const MA_CHE: Subdivision = Subdivision {
    name: "Chefchaouen",
    code: "MA-CHE",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-CHE",
};


pub const MA_CHI: Subdivision = Subdivision {
    name: "Chichaoua",
    code: "MA-CHI",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-CHI",
};


pub const MA_CHT: Subdivision = Subdivision {
    name: "Chtouka-Ait Baha",
    code: "MA-CHT",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-CHT",
};


pub const MA_ERR: Subdivision = Subdivision {
    name: "Errachidia",
    code: "MA-ERR",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-ERR",
};


pub const MA_ESI: Subdivision = Subdivision {
    name: "Essaouira",
    code: "MA-ESI",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-ESI",
};


pub const MA_ESM: Subdivision = Subdivision {
    name: "Es Smara",
    code: "MA-ESM",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-ESM",
};


pub const MA_FAH: Subdivision = Subdivision {
    name: "Fahs-Beni Makada",
    code: "MA-FAH",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-FAH",
};


pub const MA_FES: Subdivision = Subdivision {
    name: "Fès-Dar-Dbibegh",
    code: "MA-FES",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-FES",
};


pub const MA_FIG: Subdivision = Subdivision {
    name: "Figuig",
    code: "MA-FIG",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-FIG",
};


pub const MA_GUE: Subdivision = Subdivision {
    name: "Guelmim",
    code: "MA-GUE",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-GUE",
};


pub const MA_HAJ: Subdivision = Subdivision {
    name: "El Hajeb",
    code: "MA-HAJ",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-HAJ",
};


pub const MA_HAO: Subdivision = Subdivision {
    name: "Al Haouz",
    code: "MA-HAO",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-HAO",
};


pub const MA_HOC: Subdivision = Subdivision {
    name: "Al Hoceïma",
    code: "MA-HOC",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-HOC",
};


pub const MA_IFR: Subdivision = Subdivision {
    name: "Ifrane",
    code: "MA-IFR",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-IFR",
};


pub const MA_INE: Subdivision = Subdivision {
    name: "Inezgane-Ait Melloul",
    code: "MA-INE",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-INE",
};


pub const MA_JDI: Subdivision = Subdivision {
    name: "El Jadida",
    code: "MA-JDI",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-JDI",
};


pub const MA_JRA: Subdivision = Subdivision {
    name: "Jrada",
    code: "MA-JRA",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-JRA",
};


pub const MA_KEN: Subdivision = Subdivision {
    name: "Kénitra",
    code: "MA-KEN",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-KEN",
};


pub const MA_KES: Subdivision = Subdivision {
    name: "Kelaat es Sraghna",
    code: "MA-KES",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-KES",
};


pub const MA_KHE: Subdivision = Subdivision {
    name: "Khemisset",
    code: "MA-KHE",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-KHE",
};


pub const MA_KHN: Subdivision = Subdivision {
    name: "Khenifra",
    code: "MA-KHN",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-KHN",
};


pub const MA_KHO: Subdivision = Subdivision {
    name: "Khouribga",
    code: "MA-KHO",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-KHO",
};


pub const MA_LAA: Subdivision = Subdivision {
    name: "Laâyoune",
    code: "MA-LAA",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-LAA",
};


pub const MA_LAR: Subdivision = Subdivision {
    name: "Larache",
    code: "MA-LAR",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-LAR",
};


pub const MA_MED: Subdivision = Subdivision {
    name: "Médiouna",
    code: "MA-MED",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MED",
};


pub const MA_MEK: Subdivision = Subdivision {
    name: "Meknès",
    code: "MA-MEK",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MEK",
};


pub const MA_MMD: Subdivision = Subdivision {
    name: "Marrakech-Medina",
    code: "MA-MMD",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MMD",
};


pub const MA_MMN: Subdivision = Subdivision {
    name: "Marrakech-Menara",
    code: "MA-MMN",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MMN",
};


pub const MA_MOH: Subdivision = Subdivision {
    name: "Mohammadia",
    code: "MA-MOH",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MOH",
};


pub const MA_MOU: Subdivision = Subdivision {
    name: "Moulay Yacoub",
    code: "MA-MOU",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-MOU",
};


pub const MA_NAD: Subdivision = Subdivision {
    name: "Nador",
    code: "MA-NAD",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-NAD",
};


pub const MA_NOU: Subdivision = Subdivision {
    name: "Nouaceur",
    code: "MA-NOU",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-NOU",
};


pub const MA_OUA: Subdivision = Subdivision {
    name: "Ouarzazate",
    code: "MA-OUA",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-OUA",
};


pub const MA_OUD: Subdivision = Subdivision {
    name: "Oued ed Dahab",
    code: "MA-OUD",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-OUD",
};


pub const MA_OUJ: Subdivision = Subdivision {
    name: "Oujda-Angad",
    code: "MA-OUJ",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-OUJ",
};


pub const MA_RAB: Subdivision = Subdivision {
    name: "Rabat",
    code: "MA-RAB",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-RAB",
};


pub const MA_SAF: Subdivision = Subdivision {
    name: "Safi",
    code: "MA-SAF",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SAF",
};


pub const MA_SAL: Subdivision = Subdivision {
    name: "Salé",
    code: "MA-SAL",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SAL",
};


pub const MA_SEF: Subdivision = Subdivision {
    name: "Sefrou",
    code: "MA-SEF",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SEF",
};


pub const MA_SET: Subdivision = Subdivision {
    name: "Settat",
    code: "MA-SET",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SET",
};


pub const MA_SIK: Subdivision = Subdivision {
    name: "Sidi Kacem",
    code: "MA-SIK",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SIK",
};


pub const MA_SKH: Subdivision = Subdivision {
    name: "Skhirate-Témara",
    code: "MA-SKH",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SKH",
};


pub const MA_SYB: Subdivision = Subdivision {
    name: "Sidi Youssef Ben Ali",
    code: "MA-SYB",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-SYB",
};


pub const MA_TAI: Subdivision = Subdivision {
    name: "Taourirt",
    code: "MA-TAI",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TAI",
};


pub const MA_TAO: Subdivision = Subdivision {
    name: "Taounate",
    code: "MA-TAO",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TAO",
};


pub const MA_TAR: Subdivision = Subdivision {
    name: "Taroudant",
    code: "MA-TAR",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TAR",
};


pub const MA_TAT: Subdivision = Subdivision {
    name: "Tata",
    code: "MA-TAT",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TAT",
};


pub const MA_TAZ: Subdivision = Subdivision {
    name: "Taza",
    code: "MA-TAZ",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TAZ",
};


pub const MA_TET: Subdivision = Subdivision {
    name: "Tétouan",
    code: "MA-TET",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TET",
};


pub const MA_TIZ: Subdivision = Subdivision {
    name: "Tiznit",
    code: "MA-TIZ",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TIZ",
};


pub const MA_TNG: Subdivision = Subdivision {
    name: "Tanger-Assilah",
    code: "MA-TNG",
    subdivision_type: "Prefecture",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TNG",
};


pub const MA_TNT: Subdivision = Subdivision {
    name: "Tan-Tan",
    code: "MA-TNT",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-TNT",
};


pub const MA_ZAG: Subdivision = Subdivision {
    name: "Zagora",
    code: "MA-ZAG",
    subdivision_type: "Province",
    country_name: "Morocco",
    country_code: "MA",
    region_code: "MA-ZAG",
};


pub const MZ_A: Subdivision = Subdivision {
    name: "Niassa",
    code: "MZ-A",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-A",
};


pub const MZ_B: Subdivision = Subdivision {
    name: "Manica",
    code: "MZ-B",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-B",
};


pub const MZ_G: Subdivision = Subdivision {
    name: "Gaza",
    code: "MZ-G",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-G",
};


pub const MZ_I: Subdivision = Subdivision {
    name: "Inhambane",
    code: "MZ-I",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-I",
};


pub const MZ_L: Subdivision = Subdivision {
    name: "Maputo",
    code: "MZ-L",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-L",
};


pub const MZ_MPM: Subdivision = Subdivision {
    name: "Maputo",
    code: "MZ-MPM",
    subdivision_type: "City",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-MPM",
};


pub const MZ_N: Subdivision = Subdivision {
    name: "Nampula",
    code: "MZ-N",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-N",
};


pub const MZ_P: Subdivision = Subdivision {
    name: "Cabo Delgado",
    code: "MZ-P",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-P",
};


pub const MZ_Q: Subdivision = Subdivision {
    name: "Zambézia",
    code: "MZ-Q",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-Q",
};


pub const MZ_S: Subdivision = Subdivision {
    name: "Sofala",
    code: "MZ-S",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-S",
};


pub const MZ_T: Subdivision = Subdivision {
    name: "Tete",
    code: "MZ-T",
    subdivision_type: "Province",
    country_name: "Mozambique",
    country_code: "MZ",
    region_code: "MZ-T",
};


pub const MM_01: Subdivision = Subdivision {
    name: "Sagaing",
    code: "MM-01",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-01",
};


pub const MM_02: Subdivision = Subdivision {
    name: "Bago",
    code: "MM-02",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-02",
};


pub const MM_03: Subdivision = Subdivision {
    name: "Magway",
    code: "MM-03",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-03",
};


pub const MM_04: Subdivision = Subdivision {
    name: "Mandalay",
    code: "MM-04",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-04",
};


pub const MM_05: Subdivision = Subdivision {
    name: "Taninthayi",
    code: "MM-05",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-05",
};


pub const MM_06: Subdivision = Subdivision {
    name: "Yangon",
    code: "MM-06",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-06",
};


pub const MM_07: Subdivision = Subdivision {
    name: "Ayeyawady",
    code: "MM-07",
    subdivision_type: "Region",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-07",
};


pub const MM_11: Subdivision = Subdivision {
    name: "Kachin",
    code: "MM-11",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-11",
};


pub const MM_12: Subdivision = Subdivision {
    name: "Kayah",
    code: "MM-12",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-12",
};


pub const MM_13: Subdivision = Subdivision {
    name: "Kayin",
    code: "MM-13",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-13",
};


pub const MM_14: Subdivision = Subdivision {
    name: "Chin",
    code: "MM-14",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-14",
};


pub const MM_15: Subdivision = Subdivision {
    name: "Mon",
    code: "MM-15",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-15",
};


pub const MM_16: Subdivision = Subdivision {
    name: "Rakhine",
    code: "MM-16",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-16",
};


pub const MM_17: Subdivision = Subdivision {
    name: "Shan",
    code: "MM-17",
    subdivision_type: "State",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-17",
};


pub const MM_18: Subdivision = Subdivision {
    name: "Nay Pyi Taw",
    code: "MM-18",
    subdivision_type: "Union territory",
    country_name: "Myanmar",
    country_code: "MM",
    region_code: "MM-18",
};


pub const NA_CA: Subdivision = Subdivision {
    name: "Zambezi",
    code: "NA-CA",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-CA",
};


pub const NA_ER: Subdivision = Subdivision {
    name: "Erongo",
    code: "NA-ER",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-ER",
};


pub const NA_HA: Subdivision = Subdivision {
    name: "Hardap",
    code: "NA-HA",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-HA",
};


pub const NA_KA: Subdivision = Subdivision {
    name: "Karas",
    code: "NA-KA",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-KA",
};


pub const NA_KE: Subdivision = Subdivision {
    name: "Kavango East",
    code: "NA-KE",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-KE",
};


pub const NA_KH: Subdivision = Subdivision {
    name: "Khomas",
    code: "NA-KH",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-KH",
};


pub const NA_KU: Subdivision = Subdivision {
    name: "Kunene",
    code: "NA-KU",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-KU",
};


pub const NA_KW: Subdivision = Subdivision {
    name: "Kavango West",
    code: "NA-KW",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-KW",
};


pub const NA_OD: Subdivision = Subdivision {
    name: "Otjozondjupa",
    code: "NA-OD",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-OD",
};


pub const NA_OH: Subdivision = Subdivision {
    name: "Omaheke",
    code: "NA-OH",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-OH",
};


pub const NA_ON: Subdivision = Subdivision {
    name: "Oshana",
    code: "NA-ON",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-ON",
};


pub const NA_OS: Subdivision = Subdivision {
    name: "Omusati",
    code: "NA-OS",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-OS",
};


pub const NA_OT: Subdivision = Subdivision {
    name: "Oshikoto",
    code: "NA-OT",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-OT",
};


pub const NA_OW: Subdivision = Subdivision {
    name: "Ohangwena",
    code: "NA-OW",
    subdivision_type: "Region",
    country_name: "Namibia",
    country_code: "NA",
    region_code: "NA-OW",
};


pub const NR_01: Subdivision = Subdivision {
    name: "Aiwo",
    code: "NR-01",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-01",
};


pub const NR_02: Subdivision = Subdivision {
    name: "Anabar",
    code: "NR-02",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-02",
};


pub const NR_03: Subdivision = Subdivision {
    name: "Anetan",
    code: "NR-03",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-03",
};


pub const NR_04: Subdivision = Subdivision {
    name: "Anibare",
    code: "NR-04",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-04",
};


pub const NR_05: Subdivision = Subdivision {
    name: "Baiti",
    code: "NR-05",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-05",
};


pub const NR_06: Subdivision = Subdivision {
    name: "Boe",
    code: "NR-06",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-06",
};


pub const NR_07: Subdivision = Subdivision {
    name: "Buada",
    code: "NR-07",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-07",
};


pub const NR_08: Subdivision = Subdivision {
    name: "Denigomodu",
    code: "NR-08",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-08",
};


pub const NR_09: Subdivision = Subdivision {
    name: "Ewa",
    code: "NR-09",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-09",
};


pub const NR_10: Subdivision = Subdivision {
    name: "Ijuw",
    code: "NR-10",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-10",
};


pub const NR_11: Subdivision = Subdivision {
    name: "Meneng",
    code: "NR-11",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-11",
};


pub const NR_12: Subdivision = Subdivision {
    name: "Nibok",
    code: "NR-12",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-12",
};


pub const NR_13: Subdivision = Subdivision {
    name: "Uaboe",
    code: "NR-13",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-13",
};


pub const NR_14: Subdivision = Subdivision {
    name: "Yaren",
    code: "NR-14",
    subdivision_type: "District",
    country_name: "Nauru",
    country_code: "NR",
    region_code: "NR-14",
};


pub const NP_1: Subdivision = Subdivision {
    name: "Central",
    code: "NP-1",
    subdivision_type: "Development region",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-1",
};


pub const NP_2: Subdivision = Subdivision {
    name: "Madhya Pashchimanchal",
    code: "NP-2",
    subdivision_type: "Development region",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-2",
};


pub const NP_3: Subdivision = Subdivision {
    name: "Pashchimanchal",
    code: "NP-3",
    subdivision_type: "Development region",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-3",
};


pub const NP_4: Subdivision = Subdivision {
    name: "Eastern",
    code: "NP-4",
    subdivision_type: "Development region",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-4",
};


pub const NP_5: Subdivision = Subdivision {
    name: "Far Western",
    code: "NP-5",
    subdivision_type: "Development region",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-5",
};


pub const NP_BA: Subdivision = Subdivision {
    name: "Bagmati",
    code: "NP-BA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-BA",
};


pub const NP_BH: Subdivision = Subdivision {
    name: "Bheri",
    code: "NP-BH",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-BH",
};


pub const NP_DH: Subdivision = Subdivision {
    name: "Dhawalagiri",
    code: "NP-DH",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-DH",
};


pub const NP_GA: Subdivision = Subdivision {
    name: "Gandaki",
    code: "NP-GA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-GA",
};


pub const NP_JA: Subdivision = Subdivision {
    name: "Janakpur",
    code: "NP-JA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-JA",
};


pub const NP_KA: Subdivision = Subdivision {
    name: "Karnali",
    code: "NP-KA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-KA",
};


pub const NP_KO: Subdivision = Subdivision {
    name: "Kosi",
    code: "NP-KO",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-KO",
};


pub const NP_LU: Subdivision = Subdivision {
    name: "Lumbini",
    code: "NP-LU",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-LU",
};


pub const NP_MA: Subdivision = Subdivision {
    name: "Mahakali",
    code: "NP-MA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-MA",
};


pub const NP_ME: Subdivision = Subdivision {
    name: "Mechi",
    code: "NP-ME",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-ME",
};


pub const NP_NA: Subdivision = Subdivision {
    name: "Narayani",
    code: "NP-NA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-NA",
};


pub const NP_RA: Subdivision = Subdivision {
    name: "Rapti",
    code: "NP-RA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-RA",
};


pub const NP_SA: Subdivision = Subdivision {
    name: "Sagarmatha",
    code: "NP-SA",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-SA",
};


pub const NP_SE: Subdivision = Subdivision {
    name: "Seti",
    code: "NP-SE",
    subdivision_type: "Zone",
    country_name: "Nepal",
    country_code: "NP",
    region_code: "NP-SE",
};


pub const NL_AW: Subdivision = Subdivision {
    name: "Aruba",
    code: "NL-AW",
    subdivision_type: "Country",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-AW",
};


pub const NL_BQ1: Subdivision = Subdivision {
    name: "Bonaire",
    code: "NL-BQ1",
    subdivision_type: "Special municipality",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-BQ1",
};


pub const NL_BQ2: Subdivision = Subdivision {
    name: "Saba",
    code: "NL-BQ2",
    subdivision_type: "Special municipality",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-BQ2",
};


pub const NL_BQ3: Subdivision = Subdivision {
    name: "Sint Eustatius",
    code: "NL-BQ3",
    subdivision_type: "Special municipality",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-BQ3",
};


pub const NL_CW: Subdivision = Subdivision {
    name: "Curaçao",
    code: "NL-CW",
    subdivision_type: "Country",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-CW",
};


pub const NL_DR: Subdivision = Subdivision {
    name: "Drenthe",
    code: "NL-DR",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-DR",
};


pub const NL_FL: Subdivision = Subdivision {
    name: "Flevoland",
    code: "NL-FL",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-FL",
};


pub const NL_FR: Subdivision = Subdivision {
    name: "Fryslân",
    code: "NL-FR",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-FR",
};


pub const NL_GE: Subdivision = Subdivision {
    name: "Gelderland",
    code: "NL-GE",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-GE",
};


pub const NL_GR: Subdivision = Subdivision {
    name: "Groningen",
    code: "NL-GR",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-GR",
};


pub const NL_LI: Subdivision = Subdivision {
    name: "Limburg",
    code: "NL-LI",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-LI",
};


pub const NL_NB: Subdivision = Subdivision {
    name: "Noord-Brabant",
    code: "NL-NB",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-NB",
};


pub const NL_NH: Subdivision = Subdivision {
    name: "Noord-Holland",
    code: "NL-NH",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-NH",
};


pub const NL_OV: Subdivision = Subdivision {
    name: "Overijssel",
    code: "NL-OV",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-OV",
};


pub const NL_SX: Subdivision = Subdivision {
    name: "Sint Maarten",
    code: "NL-SX",
    subdivision_type: "Country",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-SX",
};


pub const NL_UT: Subdivision = Subdivision {
    name: "Utrecht",
    code: "NL-UT",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-UT",
};


pub const NL_ZE: Subdivision = Subdivision {
    name: "Zeeland",
    code: "NL-ZE",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-ZE",
};


pub const NL_ZH: Subdivision = Subdivision {
    name: "Zuid-Holland",
    code: "NL-ZH",
    subdivision_type: "Province",
    country_name: "Netherlands",
    country_code: "NL",
    region_code: "NL-ZH",
};


pub const NC_NC: Subdivision = Subdivision {
    name: "New Caledonia",
    code: "NC-NC",
    subdivision_type: "Country",
    country_name: "New Caledonia",
    country_code: "NC",
    region_code: "NC-NC",
};


pub const NZ_AUK: Subdivision = Subdivision {
    name: "Auckland",
    code: "NZ-AUK",
    subdivision_type: "Unitary authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-AUK",
};


pub const NZ_BOP: Subdivision = Subdivision {
    name: "Bay of Plenty",
    code: "NZ-BOP",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-BOP",
};


pub const NZ_CAN: Subdivision = Subdivision {
    name: "Canterbury",
    code: "NZ-CAN",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-CAN",
};


pub const NZ_CIT: Subdivision = Subdivision {
    name: "Chatham Islands Territory",
    code: "NZ-CIT",
    subdivision_type: "Special island authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-CIT",
};


pub const NZ_GIS: Subdivision = Subdivision {
    name: "Gisborne District",
    code: "NZ-GIS",
    subdivision_type: "Unitary authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-GIS",
};


pub const NZ_HKB: Subdivision = Subdivision {
    name: "Hawke's Bay",
    code: "NZ-HKB",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-HKB",
};


pub const NZ_MBH: Subdivision = Subdivision {
    name: "Marlborough District",
    code: "NZ-MBH",
    subdivision_type: "Unitary authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-MBH",
};


pub const NZ_MWT: Subdivision = Subdivision {
    name: "Manawatu-Wanganui",
    code: "NZ-MWT",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-MWT",
};


pub const NZ_N: Subdivision = Subdivision {
    name: "North Island",
    code: "NZ-N",
    subdivision_type: "Island",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-N",
};


pub const NZ_NSN: Subdivision = Subdivision {
    name: "Nelson City",
    code: "NZ-NSN",
    subdivision_type: "Unitary authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-NSN",
};


pub const NZ_NTL: Subdivision = Subdivision {
    name: "Northland",
    code: "NZ-NTL",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-NTL",
};


pub const NZ_OTA: Subdivision = Subdivision {
    name: "Otago",
    code: "NZ-OTA",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-OTA",
};


pub const NZ_S: Subdivision = Subdivision {
    name: "South Island",
    code: "NZ-S",
    subdivision_type: "Island",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-S",
};


pub const NZ_STL: Subdivision = Subdivision {
    name: "Southland",
    code: "NZ-STL",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-STL",
};


pub const NZ_TAS: Subdivision = Subdivision {
    name: "Tasman District",
    code: "NZ-TAS",
    subdivision_type: "Unitary authority",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-TAS",
};


pub const NZ_TKI: Subdivision = Subdivision {
    name: "Taranaki",
    code: "NZ-TKI",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-TKI",
};


pub const NZ_WGN: Subdivision = Subdivision {
    name: "Wellington",
    code: "NZ-WGN",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-WGN",
};


pub const NZ_WKO: Subdivision = Subdivision {
    name: "Waikato",
    code: "NZ-WKO",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-WKO",
};


pub const NZ_WTC: Subdivision = Subdivision {
    name: "West Coast",
    code: "NZ-WTC",
    subdivision_type: "Regional council",
    country_name: "New Zealand",
    country_code: "NZ",
    region_code: "NZ-WTC",
};


pub const NI_AN: Subdivision = Subdivision {
    name: "Atlántico Norte",
    code: "NI-AN",
    subdivision_type: "Autonomous region",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-AN",
};


pub const NI_AS: Subdivision = Subdivision {
    name: "Atlántico Sur",
    code: "NI-AS",
    subdivision_type: "Autonomous region",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-AS",
};


pub const NI_BO: Subdivision = Subdivision {
    name: "Boaco",
    code: "NI-BO",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-BO",
};


pub const NI_CA: Subdivision = Subdivision {
    name: "Carazo",
    code: "NI-CA",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-CA",
};


pub const NI_CI: Subdivision = Subdivision {
    name: "Chinandega",
    code: "NI-CI",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-CI",
};


pub const NI_CO: Subdivision = Subdivision {
    name: "Chontales",
    code: "NI-CO",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-CO",
};


pub const NI_ES: Subdivision = Subdivision {
    name: "Estelí",
    code: "NI-ES",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-ES",
};


pub const NI_GR: Subdivision = Subdivision {
    name: "Granada",
    code: "NI-GR",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-GR",
};


pub const NI_JI: Subdivision = Subdivision {
    name: "Jinotega",
    code: "NI-JI",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-JI",
};


pub const NI_LE: Subdivision = Subdivision {
    name: "León",
    code: "NI-LE",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-LE",
};


pub const NI_MD: Subdivision = Subdivision {
    name: "Madriz",
    code: "NI-MD",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-MD",
};


pub const NI_MN: Subdivision = Subdivision {
    name: "Managua",
    code: "NI-MN",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-MN",
};


pub const NI_MS: Subdivision = Subdivision {
    name: "Masaya",
    code: "NI-MS",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-MS",
};


pub const NI_MT: Subdivision = Subdivision {
    name: "Matagalpa",
    code: "NI-MT",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-MT",
};


pub const NI_NS: Subdivision = Subdivision {
    name: "Nueva Segovia",
    code: "NI-NS",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-NS",
};


pub const NI_RI: Subdivision = Subdivision {
    name: "Rivas",
    code: "NI-RI",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-RI",
};


pub const NI_SJ: Subdivision = Subdivision {
    name: "Río San Juan",
    code: "NI-SJ",
    subdivision_type: "Department",
    country_name: "Nicaragua",
    country_code: "NI",
    region_code: "NI-SJ",
};


pub const NG_AB: Subdivision = Subdivision {
    name: "Abia",
    code: "NG-AB",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-AB",
};


pub const NG_AD: Subdivision = Subdivision {
    name: "Adamawa",
    code: "NG-AD",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-AD",
};


pub const NG_AK: Subdivision = Subdivision {
    name: "Akwa Ibom",
    code: "NG-AK",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-AK",
};


pub const NG_AN: Subdivision = Subdivision {
    name: "Anambra",
    code: "NG-AN",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-AN",
};


pub const NG_BA: Subdivision = Subdivision {
    name: "Bauchi",
    code: "NG-BA",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-BA",
};


pub const NG_BE: Subdivision = Subdivision {
    name: "Benue",
    code: "NG-BE",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-BE",
};


pub const NG_BO: Subdivision = Subdivision {
    name: "Borno",
    code: "NG-BO",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-BO",
};


pub const NG_BY: Subdivision = Subdivision {
    name: "Bayelsa",
    code: "NG-BY",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-BY",
};


pub const NG_CR: Subdivision = Subdivision {
    name: "Cross River",
    code: "NG-CR",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-CR",
};


pub const NG_DE: Subdivision = Subdivision {
    name: "Delta",
    code: "NG-DE",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-DE",
};


pub const NG_EB: Subdivision = Subdivision {
    name: "Ebonyi",
    code: "NG-EB",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-EB",
};


pub const NG_ED: Subdivision = Subdivision {
    name: "Edo",
    code: "NG-ED",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-ED",
};


pub const NG_EK: Subdivision = Subdivision {
    name: "Ekiti",
    code: "NG-EK",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-EK",
};


pub const NG_EN: Subdivision = Subdivision {
    name: "Enugu",
    code: "NG-EN",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-EN",
};


pub const NG_FC: Subdivision = Subdivision {
    name: "Abuja Federal Capital Territory",
    code: "NG-FC",
    subdivision_type: "Capital territory",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-FC",
};


pub const NG_GO: Subdivision = Subdivision {
    name: "Gombe",
    code: "NG-GO",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-GO",
};


pub const NG_IM: Subdivision = Subdivision {
    name: "Imo",
    code: "NG-IM",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-IM",
};


pub const NG_JI: Subdivision = Subdivision {
    name: "Jigawa",
    code: "NG-JI",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-JI",
};


pub const NG_KD: Subdivision = Subdivision {
    name: "Kaduna",
    code: "NG-KD",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KD",
};


pub const NG_KE: Subdivision = Subdivision {
    name: "Kebbi",
    code: "NG-KE",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KE",
};


pub const NG_KN: Subdivision = Subdivision {
    name: "Kano",
    code: "NG-KN",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KN",
};


pub const NG_KO: Subdivision = Subdivision {
    name: "Kogi",
    code: "NG-KO",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KO",
};


pub const NG_KT: Subdivision = Subdivision {
    name: "Katsina",
    code: "NG-KT",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KT",
};


pub const NG_KW: Subdivision = Subdivision {
    name: "Kwara",
    code: "NG-KW",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-KW",
};


pub const NG_LA: Subdivision = Subdivision {
    name: "Lagos",
    code: "NG-LA",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-LA",
};


pub const NG_NA: Subdivision = Subdivision {
    name: "Nasarawa",
    code: "NG-NA",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-NA",
};


pub const NG_NI: Subdivision = Subdivision {
    name: "Niger",
    code: "NG-NI",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-NI",
};


pub const NG_OG: Subdivision = Subdivision {
    name: "Ogun",
    code: "NG-OG",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-OG",
};


pub const NG_ON: Subdivision = Subdivision {
    name: "Ondo",
    code: "NG-ON",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-ON",
};


pub const NG_OS: Subdivision = Subdivision {
    name: "Osun",
    code: "NG-OS",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-OS",
};


pub const NG_OY: Subdivision = Subdivision {
    name: "Oyo",
    code: "NG-OY",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-OY",
};


pub const NG_PL: Subdivision = Subdivision {
    name: "Plateau",
    code: "NG-PL",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-PL",
};


pub const NG_RI: Subdivision = Subdivision {
    name: "Rivers",
    code: "NG-RI",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-RI",
};


pub const NG_SO: Subdivision = Subdivision {
    name: "Sokoto",
    code: "NG-SO",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-SO",
};


pub const NG_TA: Subdivision = Subdivision {
    name: "Taraba",
    code: "NG-TA",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-TA",
};


pub const NG_YO: Subdivision = Subdivision {
    name: "Yobe",
    code: "NG-YO",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-YO",
};


pub const NG_ZA: Subdivision = Subdivision {
    name: "Zamfara",
    code: "NG-ZA",
    subdivision_type: "State",
    country_name: "Nigeria",
    country_code: "NG",
    region_code: "NG-ZA",
};


pub const NE_1: Subdivision = Subdivision {
    name: "Agadez",
    code: "NE-1",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-1",
};


pub const NE_2: Subdivision = Subdivision {
    name: "Diffa",
    code: "NE-2",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-2",
};


pub const NE_3: Subdivision = Subdivision {
    name: "Dosso",
    code: "NE-3",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-3",
};


pub const NE_4: Subdivision = Subdivision {
    name: "Maradi",
    code: "NE-4",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-4",
};


pub const NE_5: Subdivision = Subdivision {
    name: "Tahoua",
    code: "NE-5",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-5",
};


pub const NE_6: Subdivision = Subdivision {
    name: "Tillabéri",
    code: "NE-6",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-6",
};


pub const NE_7: Subdivision = Subdivision {
    name: "Zinder",
    code: "NE-7",
    subdivision_type: "Region",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-7",
};


pub const NE_8: Subdivision = Subdivision {
    name: "Niamey",
    code: "NE-8",
    subdivision_type: "Urban community",
    country_name: "Niger",
    country_code: "NE",
    region_code: "NE-8",
};


pub const NU_NU: Subdivision = Subdivision {
    name: "Niue",
    code: "NU-NU",
    subdivision_type: "Country",
    country_name: "Niue",
    country_code: "NU",
    region_code: "NU-NU",
};


pub const NF_NF: Subdivision = Subdivision {
    name: "Norfolk Island",
    code: "NF-NF",
    subdivision_type: "Country",
    country_name: "Norfolk Island",
    country_code: "NF",
    region_code: "NF-NF",
};


pub const MP_MP: Subdivision = Subdivision {
    name: "Northern Mariana Islands",
    code: "MP-MP",
    subdivision_type: "Country",
    country_name: "Northern Mariana Islands",
    country_code: "MP",
    region_code: "MP-MP",
};


pub const NO_01: Subdivision = Subdivision {
    name: "Østfold",
    code: "NO-01",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-01",
};


pub const NO_02: Subdivision = Subdivision {
    name: "Akershus",
    code: "NO-02",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-02",
};


pub const NO_03: Subdivision = Subdivision {
    name: "Oslo",
    code: "NO-03",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-03",
};


pub const NO_04: Subdivision = Subdivision {
    name: "Hedmark",
    code: "NO-04",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-04",
};


pub const NO_05: Subdivision = Subdivision {
    name: "Oppland",
    code: "NO-05",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-05",
};


pub const NO_06: Subdivision = Subdivision {
    name: "Buskerud",
    code: "NO-06",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-06",
};


pub const NO_07: Subdivision = Subdivision {
    name: "Vestfold",
    code: "NO-07",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-07",
};


pub const NO_08: Subdivision = Subdivision {
    name: "Telemark",
    code: "NO-08",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-08",
};


pub const NO_09: Subdivision = Subdivision {
    name: "Aust-Agder",
    code: "NO-09",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-09",
};


pub const NO_10: Subdivision = Subdivision {
    name: "Vest-Agder",
    code: "NO-10",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-10",
};


pub const NO_11: Subdivision = Subdivision {
    name: "Rogaland",
    code: "NO-11",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-11",
};


pub const NO_12: Subdivision = Subdivision {
    name: "Hordaland",
    code: "NO-12",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-12",
};


pub const NO_14: Subdivision = Subdivision {
    name: "Sogn og Fjordane",
    code: "NO-14",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-14",
};


pub const NO_15: Subdivision = Subdivision {
    name: "Møre og Romsdal",
    code: "NO-15",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-15",
};


pub const NO_16: Subdivision = Subdivision {
    name: "Sør-Trøndelag",
    code: "NO-16",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-16",
};


pub const NO_17: Subdivision = Subdivision {
    name: "Nord-Trøndelag",
    code: "NO-17",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-17",
};


pub const NO_18: Subdivision = Subdivision {
    name: "Nordland",
    code: "NO-18",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-18",
};


pub const NO_19: Subdivision = Subdivision {
    name: "Romsa",
    code: "NO-19",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-19",
};


pub const NO_20: Subdivision = Subdivision {
    name: "Finnmark",
    code: "NO-20",
    subdivision_type: "County",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-20",
};


pub const NO_21: Subdivision = Subdivision {
    name: "Svalbard",
    code: "NO-21",
    subdivision_type: "Arctic region",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-21",
};


pub const NO_22: Subdivision = Subdivision {
    name: "Jan Mayen",
    code: "NO-22",
    subdivision_type: "Arctic region",
    country_name: "Norway",
    country_code: "NO",
    region_code: "NO-22",
};


pub const OM_BA: Subdivision = Subdivision {
    name: "Al Bāţinah",
    code: "OM-BA",
    subdivision_type: "Region",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-BA",
};


pub const OM_BU: Subdivision = Subdivision {
    name: "Al Buraymī",
    code: "OM-BU",
    subdivision_type: "Governorate",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-BU",
};


pub const OM_DA: Subdivision = Subdivision {
    name: "Ad Dākhilīyah",
    code: "OM-DA",
    subdivision_type: "Region",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-DA",
};


pub const OM_MA: Subdivision = Subdivision {
    name: "Masqaţ",
    code: "OM-MA",
    subdivision_type: "Governorate",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-MA",
};


pub const OM_MU: Subdivision = Subdivision {
    name: "Musandam",
    code: "OM-MU",
    subdivision_type: "Governorate",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-MU",
};


pub const OM_SH: Subdivision = Subdivision {
    name: "Ash Sharqīyah",
    code: "OM-SH",
    subdivision_type: "Region",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-SH",
};


pub const OM_WU: Subdivision = Subdivision {
    name: "Al Wusţá",
    code: "OM-WU",
    subdivision_type: "Region",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-WU",
};


pub const OM_ZA: Subdivision = Subdivision {
    name: "Az̧ Z̧āhirah",
    code: "OM-ZA",
    subdivision_type: "Region",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-ZA",
};


pub const OM_ZU: Subdivision = Subdivision {
    name: "Z̧ufār",
    code: "OM-ZU",
    subdivision_type: "Governorate",
    country_name: "Oman",
    country_code: "OM",
    region_code: "OM-ZU",
};


pub const PK_BA: Subdivision = Subdivision {
    name: "Balochistan",
    code: "PK-BA",
    subdivision_type: "Province",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-BA",
};


pub const PK_GB: Subdivision = Subdivision {
    name: "Gilgit-Baltistān",
    code: "PK-GB",
    subdivision_type: "Pakistan administered area",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-GB",
};


pub const PK_IS: Subdivision = Subdivision {
    name: "Islāmābād",
    code: "PK-IS",
    subdivision_type: "Federal capital territory",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-IS",
};


pub const PK_JK: Subdivision = Subdivision {
    name: "Azad Kashmir",
    code: "PK-JK",
    subdivision_type: "Pakistan administered area",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-JK",
};


pub const PK_KP: Subdivision = Subdivision {
    name: "Khaībar Pakhtūnkhwā",
    code: "PK-KP",
    subdivision_type: "Province",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-KP",
};


pub const PK_PB: Subdivision = Subdivision {
    name: "Panjāb",
    code: "PK-PB",
    subdivision_type: "Province",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-PB",
};


pub const PK_SD: Subdivision = Subdivision {
    name: "Sindh",
    code: "PK-SD",
    subdivision_type: "Province",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-SD",
};


pub const PK_TA: Subdivision = Subdivision {
    name: "Federally Administered Tribal Areas",
    code: "PK-TA",
    subdivision_type: "Territory",
    country_name: "Pakistan",
    country_code: "PK",
    region_code: "PK-TA",
};


pub const PW_002: Subdivision = Subdivision {
    name: "Aimeliik",
    code: "PW-002",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-002",
};


pub const PW_004: Subdivision = Subdivision {
    name: "Airai",
    code: "PW-004",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-004",
};


pub const PW_010: Subdivision = Subdivision {
    name: "Angaur",
    code: "PW-010",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-010",
};


pub const PW_050: Subdivision = Subdivision {
    name: "Hatobohei",
    code: "PW-050",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-050",
};


pub const PW_100: Subdivision = Subdivision {
    name: "Kayangel",
    code: "PW-100",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-100",
};


pub const PW_150: Subdivision = Subdivision {
    name: "Koror",
    code: "PW-150",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-150",
};


pub const PW_212: Subdivision = Subdivision {
    name: "Melekeok",
    code: "PW-212",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-212",
};


pub const PW_214: Subdivision = Subdivision {
    name: "Ngaraard",
    code: "PW-214",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-214",
};


pub const PW_218: Subdivision = Subdivision {
    name: "Ngarchelong",
    code: "PW-218",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-218",
};


pub const PW_222: Subdivision = Subdivision {
    name: "Ngardmau",
    code: "PW-222",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-222",
};


pub const PW_224: Subdivision = Subdivision {
    name: "Ngatpang",
    code: "PW-224",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-224",
};


pub const PW_226: Subdivision = Subdivision {
    name: "Ngchesar",
    code: "PW-226",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-226",
};


pub const PW_227: Subdivision = Subdivision {
    name: "Ngeremlengui",
    code: "PW-227",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-227",
};


pub const PW_228: Subdivision = Subdivision {
    name: "Ngiwal",
    code: "PW-228",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-228",
};


pub const PW_350: Subdivision = Subdivision {
    name: "Peleliu",
    code: "PW-350",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-350",
};


pub const PW_370: Subdivision = Subdivision {
    name: "Sonsorol",
    code: "PW-370",
    subdivision_type: "State",
    country_name: "Palau",
    country_code: "PW",
    region_code: "PW-370",
};


pub const PS_BTH: Subdivision = Subdivision {
    name: "Bayt Laḩm",
    code: "PS-BTH",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-BTH",
};


pub const PS_DEB: Subdivision = Subdivision {
    name: "Dayr al Balaḩ",
    code: "PS-DEB",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-DEB",
};


pub const PS_GZA: Subdivision = Subdivision {
    name: "Gaza",
    code: "PS-GZA",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-GZA",
};


pub const PS_HBN: Subdivision = Subdivision {
    name: "Al Khalīl",
    code: "PS-HBN",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-HBN",
};


pub const PS_JEM: Subdivision = Subdivision {
    name: "Al Quds",
    code: "PS-JEM",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-JEM",
};


pub const PS_JEN: Subdivision = Subdivision {
    name: "Janīn",
    code: "PS-JEN",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-JEN",
};


pub const PS_JRH: Subdivision = Subdivision {
    name: "Arīḩā wa al Aghwār",
    code: "PS-JRH",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-JRH",
};


pub const PS_KYS: Subdivision = Subdivision {
    name: "Khan Yunis",
    code: "PS-KYS",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-KYS",
};


pub const PS_NBS: Subdivision = Subdivision {
    name: "Nablus",
    code: "PS-NBS",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-NBS",
};


pub const PS_NGZ: Subdivision = Subdivision {
    name: "North Gaza",
    code: "PS-NGZ",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-NGZ",
};


pub const PS_QQA: Subdivision = Subdivision {
    name: "Qalqilya",
    code: "PS-QQA",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-QQA",
};


pub const PS_RBH: Subdivision = Subdivision {
    name: "Ramallah",
    code: "PS-RBH",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-RBH",
};


pub const PS_RFH: Subdivision = Subdivision {
    name: "Rafah",
    code: "PS-RFH",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-RFH",
};


pub const PS_SLT: Subdivision = Subdivision {
    name: "Salfit",
    code: "PS-SLT",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-SLT",
};


pub const PS_TBS: Subdivision = Subdivision {
    name: "Tubas",
    code: "PS-TBS",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-TBS",
};


pub const PS_TKM: Subdivision = Subdivision {
    name: "Tulkarm",
    code: "PS-TKM",
    subdivision_type: "Governorate",
    country_name: "Palestine",
    country_code: "PS",
    region_code: "PS-TKM",
};


pub const PA_10: Subdivision = Subdivision {
    name: "Panamá Oeste",
    code: "PA-10",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-10",
};


pub const PA_1: Subdivision = Subdivision {
    name: "Bocas del Toro",
    code: "PA-1",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-1",
};


pub const PA_2: Subdivision = Subdivision {
    name: "Coclé",
    code: "PA-2",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-2",
};


pub const PA_3: Subdivision = Subdivision {
    name: "Colón",
    code: "PA-3",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-3",
};


pub const PA_4: Subdivision = Subdivision {
    name: "Chiriquí",
    code: "PA-4",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-4",
};


pub const PA_5: Subdivision = Subdivision {
    name: "Darién",
    code: "PA-5",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-5",
};


pub const PA_6: Subdivision = Subdivision {
    name: "Herrera",
    code: "PA-6",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-6",
};


pub const PA_7: Subdivision = Subdivision {
    name: "Los Santos",
    code: "PA-7",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-7",
};


pub const PA_8: Subdivision = Subdivision {
    name: "Panamá",
    code: "PA-8",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-8",
};


pub const PA_9: Subdivision = Subdivision {
    name: "Veraguas",
    code: "PA-9",
    subdivision_type: "Province",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-9",
};


pub const PA_EM: Subdivision = Subdivision {
    name: "Emberá",
    code: "PA-EM",
    subdivision_type: "Indigenous region",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-EM",
};


pub const PA_KY: Subdivision = Subdivision {
    name: "Kuna Yala",
    code: "PA-KY",
    subdivision_type: "Indigenous region",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-KY",
};


pub const PA_NB: Subdivision = Subdivision {
    name: "Ngöbe-Buglé",
    code: "PA-NB",
    subdivision_type: "Indigenous region",
    country_name: "Panama",
    country_code: "PA",
    region_code: "PA-NB",
};


pub const PG_CPK: Subdivision = Subdivision {
    name: "Chimbu",
    code: "PG-CPK",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-CPK",
};


pub const PG_CPM: Subdivision = Subdivision {
    name: "Central",
    code: "PG-CPM",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-CPM",
};


pub const PG_EBR: Subdivision = Subdivision {
    name: "East New Britain",
    code: "PG-EBR",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-EBR",
};


pub const PG_EHG: Subdivision = Subdivision {
    name: "Eastern Highlands",
    code: "PG-EHG",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-EHG",
};


pub const PG_EPW: Subdivision = Subdivision {
    name: "Enga",
    code: "PG-EPW",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-EPW",
};


pub const PG_ESW: Subdivision = Subdivision {
    name: "East Sepik",
    code: "PG-ESW",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-ESW",
};


pub const PG_GPK: Subdivision = Subdivision {
    name: "Gulf",
    code: "PG-GPK",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-GPK",
};


pub const PG_HLA: Subdivision = Subdivision {
    name: "Hela",
    code: "PG-HLA",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-HLA",
};


pub const PG_JWK: Subdivision = Subdivision {
    name: "Jiwaka",
    code: "PG-JWK",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-JWK",
};


pub const PG_MBA: Subdivision = Subdivision {
    name: "Milne Bay",
    code: "PG-MBA",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-MBA",
};


pub const PG_MPL: Subdivision = Subdivision {
    name: "Morobe",
    code: "PG-MPL",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-MPL",
};


pub const PG_MPM: Subdivision = Subdivision {
    name: "Madang",
    code: "PG-MPM",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-MPM",
};


pub const PG_MRL: Subdivision = Subdivision {
    name: "Manus",
    code: "PG-MRL",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-MRL",
};


pub const PG_NCD: Subdivision = Subdivision {
    name: "National Capital District",
    code: "PG-NCD",
    subdivision_type: "District",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-NCD",
};


pub const PG_NIK: Subdivision = Subdivision {
    name: "New Ireland",
    code: "PG-NIK",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-NIK",
};


pub const PG_NPP: Subdivision = Subdivision {
    name: "Northern",
    code: "PG-NPP",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-NPP",
};


pub const PG_NSB: Subdivision = Subdivision {
    name: "Bougainville",
    code: "PG-NSB",
    subdivision_type: "Autonomous region",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-NSB",
};


pub const PG_SAN: Subdivision = Subdivision {
    name: "West Sepik",
    code: "PG-SAN",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-SAN",
};


pub const PG_SHM: Subdivision = Subdivision {
    name: "Southern Highlands",
    code: "PG-SHM",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-SHM",
};


pub const PG_WBK: Subdivision = Subdivision {
    name: "West New Britain",
    code: "PG-WBK",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-WBK",
};


pub const PG_WHM: Subdivision = Subdivision {
    name: "Western Highlands",
    code: "PG-WHM",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-WHM",
};


pub const PG_WPD: Subdivision = Subdivision {
    name: "Western",
    code: "PG-WPD",
    subdivision_type: "Province",
    country_name: "Papua New Guinea",
    country_code: "PG",
    region_code: "PG-WPD",
};


pub const PY_10: Subdivision = Subdivision {
    name: "Alto Paraná",
    code: "PY-10",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-10",
};


pub const PY_11: Subdivision = Subdivision {
    name: "Central",
    code: "PY-11",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-11",
};


pub const PY_12: Subdivision = Subdivision {
    name: "Ñeembucú",
    code: "PY-12",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-12",
};


pub const PY_13: Subdivision = Subdivision {
    name: "Amambay",
    code: "PY-13",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-13",
};


pub const PY_14: Subdivision = Subdivision {
    name: "Canindeyú",
    code: "PY-14",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-14",
};


pub const PY_15: Subdivision = Subdivision {
    name: "Presidente Hayes",
    code: "PY-15",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-15",
};


pub const PY_16: Subdivision = Subdivision {
    name: "Alto Paraguay",
    code: "PY-16",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-16",
};


pub const PY_19: Subdivision = Subdivision {
    name: "Boquerón",
    code: "PY-19",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-19",
};


pub const PY_1: Subdivision = Subdivision {
    name: "Concepción",
    code: "PY-1",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-1",
};


pub const PY_2: Subdivision = Subdivision {
    name: "San Pedro",
    code: "PY-2",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-2",
};


pub const PY_3: Subdivision = Subdivision {
    name: "Cordillera",
    code: "PY-3",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-3",
};


pub const PY_4: Subdivision = Subdivision {
    name: "Guairá",
    code: "PY-4",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-4",
};


pub const PY_5: Subdivision = Subdivision {
    name: "Caaguazú",
    code: "PY-5",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-5",
};


pub const PY_6: Subdivision = Subdivision {
    name: "Caazapá",
    code: "PY-6",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-6",
};


pub const PY_7: Subdivision = Subdivision {
    name: "Itapúa",
    code: "PY-7",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-7",
};


pub const PY_8: Subdivision = Subdivision {
    name: "Misiones",
    code: "PY-8",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-8",
};


pub const PY_9: Subdivision = Subdivision {
    name: "Paraguarí",
    code: "PY-9",
    subdivision_type: "Department",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-9",
};


pub const PY_ASU: Subdivision = Subdivision {
    name: "Asunción",
    code: "PY-ASU",
    subdivision_type: "Capital",
    country_name: "Paraguay",
    country_code: "PY",
    region_code: "PY-ASU",
};


pub const PE_AMA: Subdivision = Subdivision {
    name: "Amarumayu",
    code: "PE-AMA",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-AMA",
};


pub const PE_ANC: Subdivision = Subdivision {
    name: "Ancash",
    code: "PE-ANC",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-ANC",
};


pub const PE_APU: Subdivision = Subdivision {
    name: "Apurímac",
    code: "PE-APU",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-APU",
};


pub const PE_ARE: Subdivision = Subdivision {
    name: "Arequipa",
    code: "PE-ARE",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-ARE",
};


pub const PE_AYA: Subdivision = Subdivision {
    name: "Ayacucho",
    code: "PE-AYA",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-AYA",
};


pub const PE_CAJ: Subdivision = Subdivision {
    name: "Cajamarca",
    code: "PE-CAJ",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-CAJ",
};


pub const PE_CAL: Subdivision = Subdivision {
    name: "El Callao",
    code: "PE-CAL",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-CAL",
};


pub const PE_CUS: Subdivision = Subdivision {
    name: "Cusco",
    code: "PE-CUS",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-CUS",
};


pub const PE_HUC: Subdivision = Subdivision {
    name: "Huánuco",
    code: "PE-HUC",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-HUC",
};


pub const PE_HUV: Subdivision = Subdivision {
    name: "Huancavelica",
    code: "PE-HUV",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-HUV",
};


pub const PE_ICA: Subdivision = Subdivision {
    name: "Ica",
    code: "PE-ICA",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-ICA",
};


pub const PE_JUN: Subdivision = Subdivision {
    name: "Hunin",
    code: "PE-JUN",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-JUN",
};


pub const PE_LAL: Subdivision = Subdivision {
    name: "La Libertad",
    code: "PE-LAL",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-LAL",
};


pub const PE_LAM: Subdivision = Subdivision {
    name: "Lambayeque",
    code: "PE-LAM",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-LAM",
};


pub const PE_LIM: Subdivision = Subdivision {
    name: "Lima",
    code: "PE-LIM",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-LIM",
};


pub const PE_LMA: Subdivision = Subdivision {
    name: "Lima hatun llaqta",
    code: "PE-LMA",
    subdivision_type: "Municipality",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-LMA",
};


pub const PE_LOR: Subdivision = Subdivision {
    name: "Loreto",
    code: "PE-LOR",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-LOR",
};


pub const PE_MDD: Subdivision = Subdivision {
    name: "Madre de Dios",
    code: "PE-MDD",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-MDD",
};


pub const PE_MOQ: Subdivision = Subdivision {
    name: "Moquegua",
    code: "PE-MOQ",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-MOQ",
};


pub const PE_PAS: Subdivision = Subdivision {
    name: "Pasco",
    code: "PE-PAS",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-PAS",
};


pub const PE_PIU: Subdivision = Subdivision {
    name: "Piura",
    code: "PE-PIU",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-PIU",
};


pub const PE_PUN: Subdivision = Subdivision {
    name: "Puno",
    code: "PE-PUN",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-PUN",
};


pub const PE_SAM: Subdivision = Subdivision {
    name: "San Martin",
    code: "PE-SAM",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-SAM",
};


pub const PE_TAC: Subdivision = Subdivision {
    name: "Tacna",
    code: "PE-TAC",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-TAC",
};


pub const PE_TUM: Subdivision = Subdivision {
    name: "Tumbes",
    code: "PE-TUM",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-TUM",
};


pub const PE_UCA: Subdivision = Subdivision {
    name: "Ucayali",
    code: "PE-UCA",
    subdivision_type: "Region",
    country_name: "Peru",
    country_code: "PE",
    region_code: "PE-UCA",
};


pub const PH_00: Subdivision = Subdivision {
    name: "National Capital Region",
    code: "PH-00",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-00",
};


pub const PH_01: Subdivision = Subdivision {
    name: "Ilocos",
    code: "PH-01",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-01",
};


pub const PH_02: Subdivision = Subdivision {
    name: "Cagayan Valley",
    code: "PH-02",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-02",
};


pub const PH_03: Subdivision = Subdivision {
    name: "Central Luzon",
    code: "PH-03",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-03",
};


pub const PH_05: Subdivision = Subdivision {
    name: "Bicol",
    code: "PH-05",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-05",
};


pub const PH_06: Subdivision = Subdivision {
    name: "Rehiyon ng Kanlurang Bisaya",
    code: "PH-06",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-06",
};


pub const PH_07: Subdivision = Subdivision {
    name: "Central Visayas",
    code: "PH-07",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-07",
};


pub const PH_08: Subdivision = Subdivision {
    name: "Eastern Visayas",
    code: "PH-08",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-08",
};


pub const PH_09: Subdivision = Subdivision {
    name: "Rehiyon ng Tangway ng Sambuwangga",
    code: "PH-09",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-09",
};


pub const PH_10: Subdivision = Subdivision {
    name: "Northern Mindanao",
    code: "PH-10",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-10",
};


pub const PH_11: Subdivision = Subdivision {
    name: "Davao",
    code: "PH-11",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-11",
};


pub const PH_12: Subdivision = Subdivision {
    name: "Rehiyon ng Soccsksargen",
    code: "PH-12",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-12",
};


pub const PH_13: Subdivision = Subdivision {
    name: "Caraga",
    code: "PH-13",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-13",
};


pub const PH_14: Subdivision = Subdivision {
    name: "Autonomous Region in Muslim Mindanao",
    code: "PH-14",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-14",
};


pub const PH_15: Subdivision = Subdivision {
    name: "Cordillera Administrative Region",
    code: "PH-15",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-15",
};


pub const PH_40: Subdivision = Subdivision {
    name: "Calabarzon",
    code: "PH-40",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-40",
};


pub const PH_41: Subdivision = Subdivision {
    name: "Mimaropa",
    code: "PH-41",
    subdivision_type: "Region",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-41",
};


pub const PH_ABR: Subdivision = Subdivision {
    name: "Abra",
    code: "PH-ABR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ABR",
};


pub const PH_AGN: Subdivision = Subdivision {
    name: "Agusan del Norte",
    code: "PH-AGN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-AGN",
};


pub const PH_AGS: Subdivision = Subdivision {
    name: "Agusan del Sur",
    code: "PH-AGS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-AGS",
};


pub const PH_AKL: Subdivision = Subdivision {
    name: "Aklan",
    code: "PH-AKL",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-AKL",
};


pub const PH_ALB: Subdivision = Subdivision {
    name: "Albay",
    code: "PH-ALB",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ALB",
};


pub const PH_ANT: Subdivision = Subdivision {
    name: "Antike",
    code: "PH-ANT",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ANT",
};


pub const PH_APA: Subdivision = Subdivision {
    name: "Apayao",
    code: "PH-APA",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-APA",
};


pub const PH_AUR: Subdivision = Subdivision {
    name: "Aurora",
    code: "PH-AUR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-AUR",
};


pub const PH_BAN: Subdivision = Subdivision {
    name: "Bataan",
    code: "PH-BAN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BAN",
};


pub const PH_BAS: Subdivision = Subdivision {
    name: "Basilan",
    code: "PH-BAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BAS",
};


pub const PH_BEN: Subdivision = Subdivision {
    name: "Benget",
    code: "PH-BEN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BEN",
};


pub const PH_BIL: Subdivision = Subdivision {
    name: "Biliran",
    code: "PH-BIL",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BIL",
};


pub const PH_BOH: Subdivision = Subdivision {
    name: "Bohol",
    code: "PH-BOH",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BOH",
};


pub const PH_BTG: Subdivision = Subdivision {
    name: "Batangas",
    code: "PH-BTG",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BTG",
};


pub const PH_BTN: Subdivision = Subdivision {
    name: "Batanes",
    code: "PH-BTN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BTN",
};


pub const PH_BUK: Subdivision = Subdivision {
    name: "Bukidnon",
    code: "PH-BUK",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BUK",
};


pub const PH_BUL: Subdivision = Subdivision {
    name: "Bulacan",
    code: "PH-BUL",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-BUL",
};


pub const PH_CAG: Subdivision = Subdivision {
    name: "Cagayan",
    code: "PH-CAG",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAG",
};


pub const PH_CAM: Subdivision = Subdivision {
    name: "Camiguin",
    code: "PH-CAM",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAM",
};


pub const PH_CAN: Subdivision = Subdivision {
    name: "Camarines Norte",
    code: "PH-CAN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAN",
};


pub const PH_CAP: Subdivision = Subdivision {
    name: "Capiz",
    code: "PH-CAP",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAP",
};


pub const PH_CAS: Subdivision = Subdivision {
    name: "Camarines Sur",
    code: "PH-CAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAS",
};


pub const PH_CAT: Subdivision = Subdivision {
    name: "Catanduanes",
    code: "PH-CAT",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAT",
};


pub const PH_CAV: Subdivision = Subdivision {
    name: "Cavite",
    code: "PH-CAV",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CAV",
};


pub const PH_CEB: Subdivision = Subdivision {
    name: "Cebu",
    code: "PH-CEB",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-CEB",
};


pub const PH_COM: Subdivision = Subdivision {
    name: "Compostela Valley",
    code: "PH-COM",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-COM",
};


pub const PH_DAO: Subdivision = Subdivision {
    name: "Davao Oriental",
    code: "PH-DAO",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-DAO",
};


pub const PH_DAS: Subdivision = Subdivision {
    name: "Davao del Sur",
    code: "PH-DAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-DAS",
};


pub const PH_DAV: Subdivision = Subdivision {
    name: "Davao del Norte",
    code: "PH-DAV",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-DAV",
};


pub const PH_DIN: Subdivision = Subdivision {
    name: "Dinagat Islands",
    code: "PH-DIN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-DIN",
};


pub const PH_EAS: Subdivision = Subdivision {
    name: "Eastern Samar",
    code: "PH-EAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-EAS",
};


pub const PH_GUI: Subdivision = Subdivision {
    name: "Gimaras",
    code: "PH-GUI",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-GUI",
};


pub const PH_IFU: Subdivision = Subdivision {
    name: "Ifugao",
    code: "PH-IFU",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-IFU",
};


pub const PH_ILI: Subdivision = Subdivision {
    name: "Iloilo",
    code: "PH-ILI",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ILI",
};


pub const PH_ILN: Subdivision = Subdivision {
    name: "Hilagang Iloko",
    code: "PH-ILN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ILN",
};


pub const PH_ILS: Subdivision = Subdivision {
    name: "Ilocos Sur",
    code: "PH-ILS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ILS",
};


pub const PH_ISA: Subdivision = Subdivision {
    name: "Isabela",
    code: "PH-ISA",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ISA",
};


pub const PH_KAL: Subdivision = Subdivision {
    name: "Kalinga",
    code: "PH-KAL",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-KAL",
};


pub const PH_LAG: Subdivision = Subdivision {
    name: "Laguna",
    code: "PH-LAG",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-LAG",
};


pub const PH_LAN: Subdivision = Subdivision {
    name: "Hilagang Lanaw",
    code: "PH-LAN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-LAN",
};


pub const PH_LAS: Subdivision = Subdivision {
    name: "Lanao del Sur",
    code: "PH-LAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-LAS",
};


pub const PH_LEY: Subdivision = Subdivision {
    name: "Leyte",
    code: "PH-LEY",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-LEY",
};


pub const PH_LUN: Subdivision = Subdivision {
    name: "La Union",
    code: "PH-LUN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-LUN",
};


pub const PH_MAD: Subdivision = Subdivision {
    name: "Marinduke",
    code: "PH-MAD",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MAD",
};


pub const PH_MAG: Subdivision = Subdivision {
    name: "Magindanaw",
    code: "PH-MAG",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MAG",
};


pub const PH_MAS: Subdivision = Subdivision {
    name: "Masbate",
    code: "PH-MAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MAS",
};


pub const PH_MDC: Subdivision = Subdivision {
    name: "Kanlurang Mindoro",
    code: "PH-MDC",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MDC",
};


pub const PH_MDR: Subdivision = Subdivision {
    name: "Mindoro Oriental",
    code: "PH-MDR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MDR",
};


pub const PH_MOU: Subdivision = Subdivision {
    name: "Lalawigang Bulubundukin",
    code: "PH-MOU",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MOU",
};


pub const PH_MSC: Subdivision = Subdivision {
    name: "Kanlurang Misamis",
    code: "PH-MSC",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MSC",
};


pub const PH_MSR: Subdivision = Subdivision {
    name: "Misamis Oriental",
    code: "PH-MSR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-MSR",
};


pub const PH_NCO: Subdivision = Subdivision {
    name: "Cotabato",
    code: "PH-NCO",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NCO",
};


pub const PH_NEC: Subdivision = Subdivision {
    name: "Kanlurang Negros",
    code: "PH-NEC",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NEC",
};


pub const PH_NER: Subdivision = Subdivision {
    name: "Negros Oriental",
    code: "PH-NER",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NER",
};


pub const PH_NSA: Subdivision = Subdivision {
    name: "Hilagang Samar",
    code: "PH-NSA",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NSA",
};


pub const PH_NUE: Subdivision = Subdivision {
    name: "Nueva Ecija",
    code: "PH-NUE",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NUE",
};


pub const PH_NUV: Subdivision = Subdivision {
    name: "Nueva Vizcaya",
    code: "PH-NUV",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-NUV",
};


pub const PH_PAM: Subdivision = Subdivision {
    name: "Pampanga",
    code: "PH-PAM",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-PAM",
};


pub const PH_PAN: Subdivision = Subdivision {
    name: "Pangasinan",
    code: "PH-PAN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-PAN",
};


pub const PH_PLW: Subdivision = Subdivision {
    name: "Palawan",
    code: "PH-PLW",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-PLW",
};


pub const PH_QUE: Subdivision = Subdivision {
    name: "Keson",
    code: "PH-QUE",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-QUE",
};


pub const PH_QUI: Subdivision = Subdivision {
    name: "Kirino",
    code: "PH-QUI",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-QUI",
};


pub const PH_RIZ: Subdivision = Subdivision {
    name: "Risal",
    code: "PH-RIZ",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-RIZ",
};


pub const PH_ROM: Subdivision = Subdivision {
    name: "Romblon",
    code: "PH-ROM",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ROM",
};


pub const PH_SAR: Subdivision = Subdivision {
    name: "Sarangani",
    code: "PH-SAR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SAR",
};


pub const PH_SCO: Subdivision = Subdivision {
    name: "South Cotabato",
    code: "PH-SCO",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SCO",
};


pub const PH_SIG: Subdivision = Subdivision {
    name: "Sikihor",
    code: "PH-SIG",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SIG",
};


pub const PH_SLE: Subdivision = Subdivision {
    name: "Katimogang Leyte",
    code: "PH-SLE",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SLE",
};


pub const PH_SLU: Subdivision = Subdivision {
    name: "Sulu",
    code: "PH-SLU",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SLU",
};


pub const PH_SOR: Subdivision = Subdivision {
    name: "Sorsogon",
    code: "PH-SOR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SOR",
};


pub const PH_SUK: Subdivision = Subdivision {
    name: "Sultan Kudarat",
    code: "PH-SUK",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SUK",
};


pub const PH_SUN: Subdivision = Subdivision {
    name: "Hilagang Surigaw",
    code: "PH-SUN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SUN",
};


pub const PH_SUR: Subdivision = Subdivision {
    name: "Surigao del Sur",
    code: "PH-SUR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-SUR",
};


pub const PH_TAR: Subdivision = Subdivision {
    name: "Tarlac",
    code: "PH-TAR",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-TAR",
};


pub const PH_TAW: Subdivision = Subdivision {
    name: "Tawi-Tawi",
    code: "PH-TAW",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-TAW",
};


pub const PH_WSA: Subdivision = Subdivision {
    name: "Samar",
    code: "PH-WSA",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-WSA",
};


pub const PH_ZAN: Subdivision = Subdivision {
    name: "Hilagang Sambuwangga",
    code: "PH-ZAN",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ZAN",
};


pub const PH_ZAS: Subdivision = Subdivision {
    name: "Timog Sambuwangga",
    code: "PH-ZAS",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ZAS",
};


pub const PH_ZMB: Subdivision = Subdivision {
    name: "Sambales",
    code: "PH-ZMB",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ZMB",
};


pub const PH_ZSI: Subdivision = Subdivision {
    name: "Sambuwangga Sibugay",
    code: "PH-ZSI",
    subdivision_type: "Province",
    country_name: "Philippines",
    country_code: "PH",
    region_code: "PH-ZSI",
};


pub const PN_PN: Subdivision = Subdivision {
    name: "Pitcairn",
    code: "PN-PN",
    subdivision_type: "Country",
    country_name: "Pitcairn",
    country_code: "PN",
    region_code: "PN-PN",
};


pub const PL_DS: Subdivision = Subdivision {
    name: "Dolnośląskie",
    code: "PL-DS",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-DS",
};


pub const PL_KP: Subdivision = Subdivision {
    name: "Kujawsko-pomorskie",
    code: "PL-KP",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-KP",
};


pub const PL_LB: Subdivision = Subdivision {
    name: "Lubuskie",
    code: "PL-LB",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-LB",
};


pub const PL_LD: Subdivision = Subdivision {
    name: "Łódzkie",
    code: "PL-LD",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-LD",
};


pub const PL_LU: Subdivision = Subdivision {
    name: "Lubelskie",
    code: "PL-LU",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-LU",
};


pub const PL_MA: Subdivision = Subdivision {
    name: "Małopolskie",
    code: "PL-MA",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-MA",
};


pub const PL_MZ: Subdivision = Subdivision {
    name: "Mazowieckie",
    code: "PL-MZ",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-MZ",
};


pub const PL_OP: Subdivision = Subdivision {
    name: "Opolskie",
    code: "PL-OP",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-OP",
};


pub const PL_PD: Subdivision = Subdivision {
    name: "Podlaskie",
    code: "PL-PD",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-PD",
};


pub const PL_PK: Subdivision = Subdivision {
    name: "Podkarpackie",
    code: "PL-PK",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-PK",
};


pub const PL_PM: Subdivision = Subdivision {
    name: "Pomorskie",
    code: "PL-PM",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-PM",
};


pub const PL_SK: Subdivision = Subdivision {
    name: "Świętokrzyskie",
    code: "PL-SK",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-SK",
};


pub const PL_SL: Subdivision = Subdivision {
    name: "Śląskie",
    code: "PL-SL",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-SL",
};


pub const PL_WN: Subdivision = Subdivision {
    name: "Warmińsko-mazurskie",
    code: "PL-WN",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-WN",
};


pub const PL_WP: Subdivision = Subdivision {
    name: "Wielkopolskie",
    code: "PL-WP",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-WP",
};


pub const PL_ZP: Subdivision = Subdivision {
    name: "Zachodniopomorskie",
    code: "PL-ZP",
    subdivision_type: "Province",
    country_name: "Poland",
    country_code: "PL",
    region_code: "PL-ZP",
};


pub const PT_01: Subdivision = Subdivision {
    name: "Aveiro",
    code: "PT-01",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-01",
};


pub const PT_02: Subdivision = Subdivision {
    name: "Beja",
    code: "PT-02",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-02",
};


pub const PT_03: Subdivision = Subdivision {
    name: "Braga",
    code: "PT-03",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-03",
};


pub const PT_04: Subdivision = Subdivision {
    name: "Bragança",
    code: "PT-04",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-04",
};


pub const PT_05: Subdivision = Subdivision {
    name: "Castelo Branco",
    code: "PT-05",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-05",
};


pub const PT_06: Subdivision = Subdivision {
    name: "Coimbra",
    code: "PT-06",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-06",
};


pub const PT_07: Subdivision = Subdivision {
    name: "Évora",
    code: "PT-07",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-07",
};


pub const PT_08: Subdivision = Subdivision {
    name: "Faro",
    code: "PT-08",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-08",
};


pub const PT_09: Subdivision = Subdivision {
    name: "Guarda",
    code: "PT-09",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-09",
};


pub const PT_10: Subdivision = Subdivision {
    name: "Leiria",
    code: "PT-10",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-10",
};


pub const PT_11: Subdivision = Subdivision {
    name: "Lisboa",
    code: "PT-11",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-11",
};


pub const PT_12: Subdivision = Subdivision {
    name: "Portalegre",
    code: "PT-12",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-12",
};


pub const PT_13: Subdivision = Subdivision {
    name: "Porto",
    code: "PT-13",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-13",
};


pub const PT_14: Subdivision = Subdivision {
    name: "Santarém",
    code: "PT-14",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-14",
};


pub const PT_15: Subdivision = Subdivision {
    name: "Setúbal",
    code: "PT-15",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-15",
};


pub const PT_16: Subdivision = Subdivision {
    name: "Viana do Castelo",
    code: "PT-16",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-16",
};


pub const PT_17: Subdivision = Subdivision {
    name: "Vila Real",
    code: "PT-17",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-17",
};


pub const PT_18: Subdivision = Subdivision {
    name: "Viseu",
    code: "PT-18",
    subdivision_type: "District",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-18",
};


pub const PT_20: Subdivision = Subdivision {
    name: "Região Autónoma dos Açores",
    code: "PT-20",
    subdivision_type: "Autonomous region",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-20",
};


pub const PT_30: Subdivision = Subdivision {
    name: "Região Autónoma da Madeira",
    code: "PT-30",
    subdivision_type: "Autonomous region",
    country_name: "Portugal",
    country_code: "PT",
    region_code: "PT-30",
};


pub const PR_PR: Subdivision = Subdivision {
    name: "Puerto Rico",
    code: "PR-PR",
    subdivision_type: "Country",
    country_name: "Puerto Rico",
    country_code: "PR",
    region_code: "PR-PR",
};


pub const QA_DA: Subdivision = Subdivision {
    name: "Ad Dawḩah",
    code: "QA-DA",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-DA",
};


pub const QA_KH: Subdivision = Subdivision {
    name: "Al Khawr wa adh Dhakhīrah",
    code: "QA-KH",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-KH",
};


pub const QA_MS: Subdivision = Subdivision {
    name: "Ash Shamāl",
    code: "QA-MS",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-MS",
};


pub const QA_RA: Subdivision = Subdivision {
    name: "Ar Rayyān",
    code: "QA-RA",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-RA",
};


pub const QA_US: Subdivision = Subdivision {
    name: "Umm Şalāl",
    code: "QA-US",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-US",
};


pub const QA_WA: Subdivision = Subdivision {
    name: "Al Wakrah",
    code: "QA-WA",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-WA",
};


pub const QA_ZA: Subdivision = Subdivision {
    name: "Az̧ Z̧a‘āyin",
    code: "QA-ZA",
    subdivision_type: "Municipality",
    country_name: "Qatar",
    country_code: "QA",
    region_code: "QA-ZA",
};


pub const RE_RE: Subdivision = Subdivision {
    name: "Reunion",
    code: "RE-RE",
    subdivision_type: "Country",
    country_name: "Reunion",
    country_code: "RE",
    region_code: "RE-RE",
};


pub const RO_AB: Subdivision = Subdivision {
    name: "Alba",
    code: "RO-AB",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-AB",
};


pub const RO_AG: Subdivision = Subdivision {
    name: "Argeş",
    code: "RO-AG",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-AG",
};


pub const RO_AR: Subdivision = Subdivision {
    name: "Arad",
    code: "RO-AR",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-AR",
};


pub const RO_B: Subdivision = Subdivision {
    name: "Bucureşti",
    code: "RO-B",
    subdivision_type: "Municipality",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-B",
};


pub const RO_BC: Subdivision = Subdivision {
    name: "Bacău",
    code: "RO-BC",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BC",
};


pub const RO_BH: Subdivision = Subdivision {
    name: "Bihor",
    code: "RO-BH",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BH",
};


pub const RO_BN: Subdivision = Subdivision {
    name: "Bistriţa-Năsăud",
    code: "RO-BN",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BN",
};


pub const RO_BR: Subdivision = Subdivision {
    name: "Brăila",
    code: "RO-BR",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BR",
};


pub const RO_BT: Subdivision = Subdivision {
    name: "Botoşani",
    code: "RO-BT",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BT",
};


pub const RO_BV: Subdivision = Subdivision {
    name: "Braşov",
    code: "RO-BV",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BV",
};


pub const RO_BZ: Subdivision = Subdivision {
    name: "Buzău",
    code: "RO-BZ",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-BZ",
};


pub const RO_CJ: Subdivision = Subdivision {
    name: "Cluj",
    code: "RO-CJ",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-CJ",
};


pub const RO_CL: Subdivision = Subdivision {
    name: "Călăraşi",
    code: "RO-CL",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-CL",
};


pub const RO_CS: Subdivision = Subdivision {
    name: "Caraş-Severin",
    code: "RO-CS",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-CS",
};


pub const RO_CT: Subdivision = Subdivision {
    name: "Constanţa",
    code: "RO-CT",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-CT",
};


pub const RO_CV: Subdivision = Subdivision {
    name: "Covasna",
    code: "RO-CV",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-CV",
};


pub const RO_DB: Subdivision = Subdivision {
    name: "Dâmboviţa",
    code: "RO-DB",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-DB",
};


pub const RO_DJ: Subdivision = Subdivision {
    name: "Dolj",
    code: "RO-DJ",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-DJ",
};


pub const RO_GJ: Subdivision = Subdivision {
    name: "Gorj",
    code: "RO-GJ",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-GJ",
};


pub const RO_GL: Subdivision = Subdivision {
    name: "Galaţi",
    code: "RO-GL",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-GL",
};


pub const RO_GR: Subdivision = Subdivision {
    name: "Giurgiu",
    code: "RO-GR",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-GR",
};


pub const RO_HD: Subdivision = Subdivision {
    name: "Hunedoara",
    code: "RO-HD",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-HD",
};


pub const RO_HR: Subdivision = Subdivision {
    name: "Harghita",
    code: "RO-HR",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-HR",
};


pub const RO_IF: Subdivision = Subdivision {
    name: "Ilfov",
    code: "RO-IF",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-IF",
};


pub const RO_IL: Subdivision = Subdivision {
    name: "Ialomiţa",
    code: "RO-IL",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-IL",
};


pub const RO_IS: Subdivision = Subdivision {
    name: "Iaşi",
    code: "RO-IS",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-IS",
};


pub const RO_MH: Subdivision = Subdivision {
    name: "Mehedinţi",
    code: "RO-MH",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-MH",
};


pub const RO_MM: Subdivision = Subdivision {
    name: "Maramureş",
    code: "RO-MM",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-MM",
};


pub const RO_MS: Subdivision = Subdivision {
    name: "Mureş",
    code: "RO-MS",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-MS",
};


pub const RO_NT: Subdivision = Subdivision {
    name: "Neamţ",
    code: "RO-NT",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-NT",
};


pub const RO_OT: Subdivision = Subdivision {
    name: "Olt",
    code: "RO-OT",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-OT",
};


pub const RO_PH: Subdivision = Subdivision {
    name: "Prahova",
    code: "RO-PH",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-PH",
};


pub const RO_SB: Subdivision = Subdivision {
    name: "Sibiu",
    code: "RO-SB",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-SB",
};


pub const RO_SJ: Subdivision = Subdivision {
    name: "Sălaj",
    code: "RO-SJ",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-SJ",
};


pub const RO_SM: Subdivision = Subdivision {
    name: "Satu Mare",
    code: "RO-SM",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-SM",
};


pub const RO_SV: Subdivision = Subdivision {
    name: "Suceava",
    code: "RO-SV",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-SV",
};


pub const RO_TL: Subdivision = Subdivision {
    name: "Tulcea",
    code: "RO-TL",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-TL",
};


pub const RO_TM: Subdivision = Subdivision {
    name: "Timiş",
    code: "RO-TM",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-TM",
};


pub const RO_TR: Subdivision = Subdivision {
    name: "Teleorman",
    code: "RO-TR",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-TR",
};


pub const RO_VL: Subdivision = Subdivision {
    name: "Vâlcea",
    code: "RO-VL",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-VL",
};


pub const RO_VN: Subdivision = Subdivision {
    name: "Vrancea",
    code: "RO-VN",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-VN",
};


pub const RO_VS: Subdivision = Subdivision {
    name: "Vaslui",
    code: "RO-VS",
    subdivision_type: "Department",
    country_name: "Romania",
    country_code: "RO",
    region_code: "RO-VS",
};


pub const RU_AD: Subdivision = Subdivision {
    name: "Adygeja, Respublika",
    code: "RU-AD",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-AD",
};


pub const RU_AL: Subdivision = Subdivision {
    name: "Altaj, Respublika",
    code: "RU-AL",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-AL",
};


pub const RU_ALT: Subdivision = Subdivision {
    name: "Altajskij kraj",
    code: "RU-ALT",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ALT",
};


pub const RU_AMU: Subdivision = Subdivision {
    name: "Amurskaja oblast'",
    code: "RU-AMU",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-AMU",
};


pub const RU_ARK: Subdivision = Subdivision {
    name: "Arhangel'skaja oblast'",
    code: "RU-ARK",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ARK",
};


pub const RU_AST: Subdivision = Subdivision {
    name: "Astrahanskaja oblast'",
    code: "RU-AST",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-AST",
};


pub const RU_BA: Subdivision = Subdivision {
    name: "Bashkortostan, Respublika",
    code: "RU-BA",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-BA",
};


pub const RU_BEL: Subdivision = Subdivision {
    name: "Belgorodskaja oblast'",
    code: "RU-BEL",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-BEL",
};


pub const RU_BRY: Subdivision = Subdivision {
    name: "Brjanskaja oblast'",
    code: "RU-BRY",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-BRY",
};


pub const RU_BU: Subdivision = Subdivision {
    name: "Burjatija, Respublika",
    code: "RU-BU",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-BU",
};


pub const RU_CE: Subdivision = Subdivision {
    name: "Čečenskaja Respublika",
    code: "RU-CE",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-CE",
};


pub const RU_CHE: Subdivision = Subdivision {
    name: "Čeljabinskaja oblast'",
    code: "RU-CHE",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-CHE",
};


pub const RU_CHU: Subdivision = Subdivision {
    name: "Chukotskiy avtonomnyy okrug",
    code: "RU-CHU",
    subdivision_type: "Autonomous district",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-CHU",
};


pub const RU_CU: Subdivision = Subdivision {
    name: "Chuvashskaya Respublika",
    code: "RU-CU",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-CU",
};


pub const RU_DA: Subdivision = Subdivision {
    name: "Dagestan, Respublika",
    code: "RU-DA",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-DA",
};


pub const RU_IN: Subdivision = Subdivision {
    name: "Ingušetija, Respublika",
    code: "RU-IN",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-IN",
};


pub const RU_IRK: Subdivision = Subdivision {
    name: "Irkutskaja oblast'",
    code: "RU-IRK",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-IRK",
};


pub const RU_IVA: Subdivision = Subdivision {
    name: "Ivanovskaja oblast'",
    code: "RU-IVA",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-IVA",
};


pub const RU_KAM: Subdivision = Subdivision {
    name: "Kamčatskij kraj",
    code: "RU-KAM",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KAM",
};


pub const RU_KB: Subdivision = Subdivision {
    name: "Kabardino-Balkarskaja Respublika",
    code: "RU-KB",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KB",
};


pub const RU_KC: Subdivision = Subdivision {
    name: "Karačaevo-Čerkesskaja Respublika",
    code: "RU-KC",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KC",
};


pub const RU_KDA: Subdivision = Subdivision {
    name: "Krasnodarskij kraj",
    code: "RU-KDA",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KDA",
};


pub const RU_KEM: Subdivision = Subdivision {
    name: "Kemerovskaja oblast'",
    code: "RU-KEM",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KEM",
};


pub const RU_KGD: Subdivision = Subdivision {
    name: "Kaliningradskaja oblast'",
    code: "RU-KGD",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KGD",
};


pub const RU_KGN: Subdivision = Subdivision {
    name: "Kurganskaja oblast'",
    code: "RU-KGN",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KGN",
};


pub const RU_KHA: Subdivision = Subdivision {
    name: "Habarovskij kraj",
    code: "RU-KHA",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KHA",
};


pub const RU_KHM: Subdivision = Subdivision {
    name: "Hanty-Mansijskij avtonomnyj okrug",
    code: "RU-KHM",
    subdivision_type: "Autonomous district",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KHM",
};


pub const RU_KIR: Subdivision = Subdivision {
    name: "Kirovskaja oblast'",
    code: "RU-KIR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KIR",
};


pub const RU_KK: Subdivision = Subdivision {
    name: "Hakasija, Respublika",
    code: "RU-KK",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KK",
};


pub const RU_KL: Subdivision = Subdivision {
    name: "Kalmykija, Respublika",
    code: "RU-KL",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KL",
};


pub const RU_KLU: Subdivision = Subdivision {
    name: "Kaluzhskaya oblast'",
    code: "RU-KLU",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KLU",
};


pub const RU_KO: Subdivision = Subdivision {
    name: "Komi, Respublika",
    code: "RU-KO",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KO",
};


pub const RU_KOS: Subdivision = Subdivision {
    name: "Kostromskaja oblast'",
    code: "RU-KOS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KOS",
};


pub const RU_KR: Subdivision = Subdivision {
    name: "Karelija, Respublika",
    code: "RU-KR",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KR",
};


pub const RU_KRS: Subdivision = Subdivision {
    name: "Kurskaja oblast'",
    code: "RU-KRS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KRS",
};


pub const RU_KYA: Subdivision = Subdivision {
    name: "Krasnojarskij kraj",
    code: "RU-KYA",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-KYA",
};


pub const RU_LEN: Subdivision = Subdivision {
    name: "Leningradskaja oblast'",
    code: "RU-LEN",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-LEN",
};


pub const RU_LIP: Subdivision = Subdivision {
    name: "Lipeckaja oblast'",
    code: "RU-LIP",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-LIP",
};


pub const RU_MAG: Subdivision = Subdivision {
    name: "Magadanskaja oblast'",
    code: "RU-MAG",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-MAG",
};


pub const RU_ME: Subdivision = Subdivision {
    name: "Marij Èl, Respublika",
    code: "RU-ME",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ME",
};


pub const RU_MO: Subdivision = Subdivision {
    name: "Mordovija, Respublika",
    code: "RU-MO",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-MO",
};


pub const RU_MOS: Subdivision = Subdivision {
    name: "Moskovskaja oblast'",
    code: "RU-MOS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-MOS",
};


pub const RU_MOW: Subdivision = Subdivision {
    name: "Moskva",
    code: "RU-MOW",
    subdivision_type: "Autonomous city",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-MOW",
};


pub const RU_MUR: Subdivision = Subdivision {
    name: "Murmanskaja oblast'",
    code: "RU-MUR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-MUR",
};


pub const RU_NEN: Subdivision = Subdivision {
    name: "Neneckij avtonomnyj okrug",
    code: "RU-NEN",
    subdivision_type: "Autonomous district",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-NEN",
};


pub const RU_NGR: Subdivision = Subdivision {
    name: "Novgorodskaja oblast'",
    code: "RU-NGR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-NGR",
};


pub const RU_NIZ: Subdivision = Subdivision {
    name: "Nižegorodskaja oblast'",
    code: "RU-NIZ",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-NIZ",
};


pub const RU_NVS: Subdivision = Subdivision {
    name: "Novosibirskaja oblast'",
    code: "RU-NVS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-NVS",
};


pub const RU_OMS: Subdivision = Subdivision {
    name: "Omskaja oblast'",
    code: "RU-OMS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-OMS",
};


pub const RU_ORE: Subdivision = Subdivision {
    name: "Orenburgskaja oblast'",
    code: "RU-ORE",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ORE",
};


pub const RU_ORL: Subdivision = Subdivision {
    name: "Orlovskaja oblast'",
    code: "RU-ORL",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ORL",
};


pub const RU_PER: Subdivision = Subdivision {
    name: "Permskij kraj",
    code: "RU-PER",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-PER",
};


pub const RU_PNZ: Subdivision = Subdivision {
    name: "Penzenskaja oblast'",
    code: "RU-PNZ",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-PNZ",
};


pub const RU_PRI: Subdivision = Subdivision {
    name: "Primorskij kraj",
    code: "RU-PRI",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-PRI",
};


pub const RU_PSK: Subdivision = Subdivision {
    name: "Pskovskaja oblast'",
    code: "RU-PSK",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-PSK",
};


pub const RU_ROS: Subdivision = Subdivision {
    name: "Rostovskaja oblast'",
    code: "RU-ROS",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ROS",
};


pub const RU_RYA: Subdivision = Subdivision {
    name: "Rjazanskaja oblast'",
    code: "RU-RYA",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-RYA",
};


pub const RU_SAK: Subdivision = Subdivision {
    name: "Sahalinskaja oblast'",
    code: "RU-SAK",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SAK",
};


pub const RU_SAM: Subdivision = Subdivision {
    name: "Samarskaja oblast'",
    code: "RU-SAM",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SAM",
};


pub const RU_SAR: Subdivision = Subdivision {
    name: "Saratovskaja oblast'",
    code: "RU-SAR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SAR",
};


pub const RU_SA: Subdivision = Subdivision {
    name: "Saha, Respublika",
    code: "RU-SA",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SA",
};


pub const RU_SE: Subdivision = Subdivision {
    name: "Severnaja Osetija, Respublika",
    code: "RU-SE",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SE",
};


pub const RU_SMO: Subdivision = Subdivision {
    name: "Smolenskaja oblast'",
    code: "RU-SMO",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SMO",
};


pub const RU_SPE: Subdivision = Subdivision {
    name: "Sankt-Peterburg",
    code: "RU-SPE",
    subdivision_type: "Autonomous city",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SPE",
};


pub const RU_STA: Subdivision = Subdivision {
    name: "Stavropol'skij kraj",
    code: "RU-STA",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-STA",
};


pub const RU_SVE: Subdivision = Subdivision {
    name: "Sverdlovskaja oblast'",
    code: "RU-SVE",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-SVE",
};


pub const RU_TAM: Subdivision = Subdivision {
    name: "Tambovskaja oblast'",
    code: "RU-TAM",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TAM",
};


pub const RU_TA: Subdivision = Subdivision {
    name: "Tatarstan, Respublika",
    code: "RU-TA",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TA",
};


pub const RU_TOM: Subdivision = Subdivision {
    name: "Tomskaja oblast'",
    code: "RU-TOM",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TOM",
};


pub const RU_TUL: Subdivision = Subdivision {
    name: "Tul'skaja oblast'",
    code: "RU-TUL",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TUL",
};


pub const RU_TVE: Subdivision = Subdivision {
    name: "Tverskaja oblast'",
    code: "RU-TVE",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TVE",
};


pub const RU_TY: Subdivision = Subdivision {
    name: "Tyva, Respublika",
    code: "RU-TY",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TY",
};


pub const RU_TYU: Subdivision = Subdivision {
    name: "Tjumenskaja oblast'",
    code: "RU-TYU",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-TYU",
};


pub const RU_UD: Subdivision = Subdivision {
    name: "Udmurtskaja Respublika",
    code: "RU-UD",
    subdivision_type: "Republic",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-UD",
};


pub const RU_ULY: Subdivision = Subdivision {
    name: "Ul'janovskaja oblast'",
    code: "RU-ULY",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ULY",
};


pub const RU_VGG: Subdivision = Subdivision {
    name: "Volgogradskaja oblast'",
    code: "RU-VGG",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-VGG",
};


pub const RU_VLA: Subdivision = Subdivision {
    name: "Vladimirskaja oblast'",
    code: "RU-VLA",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-VLA",
};


pub const RU_VLG: Subdivision = Subdivision {
    name: "Vologodskaja oblast'",
    code: "RU-VLG",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-VLG",
};


pub const RU_VOR: Subdivision = Subdivision {
    name: "Voronezhskaya oblast'",
    code: "RU-VOR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-VOR",
};


pub const RU_YAN: Subdivision = Subdivision {
    name: "Jamalo-Neneckij avtonomnyj okrug",
    code: "RU-YAN",
    subdivision_type: "Autonomous district",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-YAN",
};


pub const RU_YAR: Subdivision = Subdivision {
    name: "Jaroslavskaja oblast'",
    code: "RU-YAR",
    subdivision_type: "Administrative region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-YAR",
};


pub const RU_YEV: Subdivision = Subdivision {
    name: "Evrejskaja avtonomnaja oblast'",
    code: "RU-YEV",
    subdivision_type: "Autonomous region",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-YEV",
};


pub const RU_ZAB: Subdivision = Subdivision {
    name: "Zabajkal'skij kraj",
    code: "RU-ZAB",
    subdivision_type: "Administrative territory",
    country_name: "Russia",
    country_code: "RU",
    region_code: "RU-ZAB",
};


pub const RW_01: Subdivision = Subdivision {
    name: "Ville de Kigal",
    code: "RW-01",
    subdivision_type: "Town council",
    country_name: "Rwanda",
    country_code: "RW",
    region_code: "RW-01",
};


pub const RW_02: Subdivision = Subdivision {
    name: "Est",
    code: "RW-02",
    subdivision_type: "Province",
    country_name: "Rwanda",
    country_code: "RW",
    region_code: "RW-02",
};


pub const RW_03: Subdivision = Subdivision {
    name: "Nord",
    code: "RW-03",
    subdivision_type: "Province",
    country_name: "Rwanda",
    country_code: "RW",
    region_code: "RW-03",
};


pub const RW_04: Subdivision = Subdivision {
    name: "Ouest",
    code: "RW-04",
    subdivision_type: "Province",
    country_name: "Rwanda",
    country_code: "RW",
    region_code: "RW-04",
};


pub const RW_05: Subdivision = Subdivision {
    name: "Sud",
    code: "RW-05",
    subdivision_type: "Province",
    country_name: "Rwanda",
    country_code: "RW",
    region_code: "RW-05",
};


pub const SH_AC: Subdivision = Subdivision {
    name: "Ascension",
    code: "SH-AC",
    subdivision_type: "Geographical entity",
    country_name: "Saint Helena, Ascension and Tristan Da Cunha",
    country_code: "SH",
    region_code: "SH-AC",
};


pub const SH_HL: Subdivision = Subdivision {
    name: "Saint Helena",
    code: "SH-HL",
    subdivision_type: "Geographical entity",
    country_name: "Saint Helena, Ascension and Tristan Da Cunha",
    country_code: "SH",
    region_code: "SH-HL",
};


pub const SH_TA: Subdivision = Subdivision {
    name: "Tristan da Cunha",
    code: "SH-TA",
    subdivision_type: "Geographical entity",
    country_name: "Saint Helena, Ascension and Tristan Da Cunha",
    country_code: "SH",
    region_code: "SH-TA",
};


pub const KN_01: Subdivision = Subdivision {
    name: "Christ Church Nichola Town",
    code: "KN-01",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-01",
};


pub const KN_02: Subdivision = Subdivision {
    name: "Saint Anne Sandy Point",
    code: "KN-02",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-02",
};


pub const KN_03: Subdivision = Subdivision {
    name: "Saint George Basseterre",
    code: "KN-03",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-03",
};


pub const KN_04: Subdivision = Subdivision {
    name: "Saint George Gingerland",
    code: "KN-04",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-04",
};


pub const KN_05: Subdivision = Subdivision {
    name: "Saint James Windward",
    code: "KN-05",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-05",
};


pub const KN_06: Subdivision = Subdivision {
    name: "Saint John Capisterre",
    code: "KN-06",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-06",
};


pub const KN_07: Subdivision = Subdivision {
    name: "Saint John Figtree",
    code: "KN-07",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-07",
};


pub const KN_08: Subdivision = Subdivision {
    name: "Saint Mary Cayon",
    code: "KN-08",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-08",
};


pub const KN_09: Subdivision = Subdivision {
    name: "Saint Paul Capisterre",
    code: "KN-09",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-09",
};


pub const KN_10: Subdivision = Subdivision {
    name: "Saint Paul Charlestown",
    code: "KN-10",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-10",
};


pub const KN_11: Subdivision = Subdivision {
    name: "Saint Peter Basseterre",
    code: "KN-11",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-11",
};


pub const KN_12: Subdivision = Subdivision {
    name: "Saint Thomas Lowland",
    code: "KN-12",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-12",
};


pub const KN_13: Subdivision = Subdivision {
    name: "Saint Thomas Middle Island",
    code: "KN-13",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-13",
};


pub const KN_15: Subdivision = Subdivision {
    name: "Trinity Palmetto Point",
    code: "KN-15",
    subdivision_type: "Parish",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-15",
};


pub const KN_K: Subdivision = Subdivision {
    name: "Saint Kitts",
    code: "KN-K",
    subdivision_type: "State",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-K",
};


pub const KN_N: Subdivision = Subdivision {
    name: "Nevis",
    code: "KN-N",
    subdivision_type: "State",
    country_name: "Saint Kitts And Nevis",
    country_code: "KN",
    region_code: "KN-N",
};


pub const LC_01: Subdivision = Subdivision {
    name: "Anse la Raye",
    code: "LC-01",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-01",
};


pub const LC_02: Subdivision = Subdivision {
    name: "Castries",
    code: "LC-02",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-02",
};


pub const LC_03: Subdivision = Subdivision {
    name: "Choiseul",
    code: "LC-03",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-03",
};


pub const LC_05: Subdivision = Subdivision {
    name: "Dennery",
    code: "LC-05",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-05",
};


pub const LC_06: Subdivision = Subdivision {
    name: "Gros Islet",
    code: "LC-06",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-06",
};


pub const LC_07: Subdivision = Subdivision {
    name: "Laborie",
    code: "LC-07",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-07",
};


pub const LC_08: Subdivision = Subdivision {
    name: "Micoud",
    code: "LC-08",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-08",
};


pub const LC_10: Subdivision = Subdivision {
    name: "Soufrière",
    code: "LC-10",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-10",
};


pub const LC_11: Subdivision = Subdivision {
    name: "Vieux Fort",
    code: "LC-11",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-11",
};


pub const LC_12: Subdivision = Subdivision {
    name: "Canaries",
    code: "LC-12",
    subdivision_type: "District",
    country_name: "Saint Lucia",
    country_code: "LC",
    region_code: "LC-12",
};


pub const VC_01: Subdivision = Subdivision {
    name: "Charlotte",
    code: "VC-01",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-01",
};


pub const VC_02: Subdivision = Subdivision {
    name: "Saint Andrew",
    code: "VC-02",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-02",
};


pub const VC_03: Subdivision = Subdivision {
    name: "Saint David",
    code: "VC-03",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-03",
};


pub const VC_04: Subdivision = Subdivision {
    name: "Saint George",
    code: "VC-04",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-04",
};


pub const VC_05: Subdivision = Subdivision {
    name: "Saint Patrick",
    code: "VC-05",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-05",
};


pub const VC_06: Subdivision = Subdivision {
    name: "Grenadines",
    code: "VC-06",
    subdivision_type: "Parish",
    country_name: "Saint Vincent And The Grenadines",
    country_code: "VC",
    region_code: "VC-06",
};


pub const WS_AA: Subdivision = Subdivision {
    name: "A'ana",
    code: "WS-AA",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-AA",
};


pub const WS_AL: Subdivision = Subdivision {
    name: "Aiga-i-le-Tai",
    code: "WS-AL",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-AL",
};


pub const WS_AT: Subdivision = Subdivision {
    name: "Atua",
    code: "WS-AT",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-AT",
};


pub const WS_FA: Subdivision = Subdivision {
    name: "Fa'asaleleaga",
    code: "WS-FA",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-FA",
};


pub const WS_GE: Subdivision = Subdivision {
    name: "Gaga'emauga",
    code: "WS-GE",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-GE",
};


pub const WS_GI: Subdivision = Subdivision {
    name: "Gagaifomauga",
    code: "WS-GI",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-GI",
};


pub const WS_PA: Subdivision = Subdivision {
    name: "Palauli",
    code: "WS-PA",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-PA",
};


pub const WS_SA: Subdivision = Subdivision {
    name: "Satupa'itea",
    code: "WS-SA",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-SA",
};


pub const WS_TU: Subdivision = Subdivision {
    name: "Tuamasaga",
    code: "WS-TU",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-TU",
};


pub const WS_VF: Subdivision = Subdivision {
    name: "Va'a-o-Fonoti",
    code: "WS-VF",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-VF",
};


pub const WS_VS: Subdivision = Subdivision {
    name: "Vaisigano",
    code: "WS-VS",
    subdivision_type: "District",
    country_name: "Samoa",
    country_code: "WS",
    region_code: "WS-VS",
};


pub const SM_01: Subdivision = Subdivision {
    name: "Acquaviva",
    code: "SM-01",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-01",
};


pub const SM_02: Subdivision = Subdivision {
    name: "Chiesanuova",
    code: "SM-02",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-02",
};


pub const SM_03: Subdivision = Subdivision {
    name: "Domagnano",
    code: "SM-03",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-03",
};


pub const SM_04: Subdivision = Subdivision {
    name: "Faetano",
    code: "SM-04",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-04",
};


pub const SM_05: Subdivision = Subdivision {
    name: "Fiorentino",
    code: "SM-05",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-05",
};


pub const SM_06: Subdivision = Subdivision {
    name: "Borgo Maggiore",
    code: "SM-06",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-06",
};


pub const SM_07: Subdivision = Subdivision {
    name: "San Marino",
    code: "SM-07",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-07",
};


pub const SM_08: Subdivision = Subdivision {
    name: "Montegiardino",
    code: "SM-08",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-08",
};


pub const SM_09: Subdivision = Subdivision {
    name: "Serravalle",
    code: "SM-09",
    subdivision_type: "Municipality",
    country_name: "San Marino",
    country_code: "SM",
    region_code: "SM-09",
};


pub const ST_P: Subdivision = Subdivision {
    name: "Príncipe",
    code: "ST-P",
    subdivision_type: "Province",
    country_name: "Sao Tome and Principe",
    country_code: "ST",
    region_code: "ST-P",
};


pub const ST_S: Subdivision = Subdivision {
    name: "São Tomé",
    code: "ST-S",
    subdivision_type: "Province",
    country_name: "Sao Tome and Principe",
    country_code: "ST",
    region_code: "ST-S",
};


pub const SA_01: Subdivision = Subdivision {
    name: "Ar Riyāḑ",
    code: "SA-01",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-01",
};


pub const SA_02: Subdivision = Subdivision {
    name: "Makkah al Mukarramah",
    code: "SA-02",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-02",
};


pub const SA_03: Subdivision = Subdivision {
    name: "Al Madīnah al Munawwarah",
    code: "SA-03",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-03",
};


pub const SA_04: Subdivision = Subdivision {
    name: "Ash Sharqīyah",
    code: "SA-04",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-04",
};


pub const SA_05: Subdivision = Subdivision {
    name: "Al Qaşīm",
    code: "SA-05",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-05",
};


pub const SA_06: Subdivision = Subdivision {
    name: "Ḩā'il",
    code: "SA-06",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-06",
};


pub const SA_07: Subdivision = Subdivision {
    name: "Tabūk",
    code: "SA-07",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-07",
};


pub const SA_08: Subdivision = Subdivision {
    name: "Al Ḩudūd ash Shamālīyah",
    code: "SA-08",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-08",
};


pub const SA_09: Subdivision = Subdivision {
    name: "Jāzān",
    code: "SA-09",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-09",
};


pub const SA_10: Subdivision = Subdivision {
    name: "Najrān",
    code: "SA-10",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-10",
};


pub const SA_11: Subdivision = Subdivision {
    name: "Al Bāḩah",
    code: "SA-11",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-11",
};


pub const SA_12: Subdivision = Subdivision {
    name: "Al Jawf",
    code: "SA-12",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-12",
};


pub const SA_14: Subdivision = Subdivision {
    name: "'Asīr",
    code: "SA-14",
    subdivision_type: "Province",
    country_name: "Saudi Arabia",
    country_code: "SA",
    region_code: "SA-14",
};


pub const SN_DB: Subdivision = Subdivision {
    name: "Diourbel",
    code: "SN-DB",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-DB",
};


pub const SN_DK: Subdivision = Subdivision {
    name: "Dakar",
    code: "SN-DK",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-DK",
};


pub const SN_FK: Subdivision = Subdivision {
    name: "Fatick",
    code: "SN-FK",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-FK",
};


pub const SN_KA: Subdivision = Subdivision {
    name: "Kaffrine",
    code: "SN-KA",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-KA",
};


pub const SN_KD: Subdivision = Subdivision {
    name: "Kolda",
    code: "SN-KD",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-KD",
};


pub const SN_KE: Subdivision = Subdivision {
    name: "Kédougou",
    code: "SN-KE",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-KE",
};


pub const SN_KL: Subdivision = Subdivision {
    name: "Kaolack",
    code: "SN-KL",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-KL",
};


pub const SN_LG: Subdivision = Subdivision {
    name: "Louga",
    code: "SN-LG",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-LG",
};


pub const SN_MT: Subdivision = Subdivision {
    name: "Matam",
    code: "SN-MT",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-MT",
};


pub const SN_SE: Subdivision = Subdivision {
    name: "Sédhiou",
    code: "SN-SE",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-SE",
};


pub const SN_SL: Subdivision = Subdivision {
    name: "Saint-Louis",
    code: "SN-SL",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-SL",
};


pub const SN_TC: Subdivision = Subdivision {
    name: "Tambacounda",
    code: "SN-TC",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-TC",
};


pub const SN_TH: Subdivision = Subdivision {
    name: "Thiès",
    code: "SN-TH",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-TH",
};


pub const SN_ZG: Subdivision = Subdivision {
    name: "Ziguinchor",
    code: "SN-ZG",
    subdivision_type: "Region",
    country_name: "Senegal",
    country_code: "SN",
    region_code: "SN-ZG",
};


pub const RS_00: Subdivision = Subdivision {
    name: "Beograd",
    code: "RS-00",
    subdivision_type: "City",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-00",
};


pub const RS_01: Subdivision = Subdivision {
    name: "Severnobački okrug",
    code: "RS-01",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-01",
};


pub const RS_02: Subdivision = Subdivision {
    name: "Srednjebanatski okrug",
    code: "RS-02",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-02",
};


pub const RS_03: Subdivision = Subdivision {
    name: "Severnobanatski okrug",
    code: "RS-03",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-03",
};


pub const RS_04: Subdivision = Subdivision {
    name: "Južnobanatski okrug",
    code: "RS-04",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-04",
};


pub const RS_05: Subdivision = Subdivision {
    name: "Zapadnobački okrug",
    code: "RS-05",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-05",
};


pub const RS_06: Subdivision = Subdivision {
    name: "Južnobački okrug",
    code: "RS-06",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-06",
};


pub const RS_07: Subdivision = Subdivision {
    name: "Sremski okrug",
    code: "RS-07",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-07",
};


pub const RS_08: Subdivision = Subdivision {
    name: "Mačvanski okrug",
    code: "RS-08",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-08",
};


pub const RS_09: Subdivision = Subdivision {
    name: "Kolubarski okrug",
    code: "RS-09",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-09",
};


pub const RS_10: Subdivision = Subdivision {
    name: "Podunavski okrug",
    code: "RS-10",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-10",
};


pub const RS_11: Subdivision = Subdivision {
    name: "Braničevski okrug",
    code: "RS-11",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-11",
};


pub const RS_12: Subdivision = Subdivision {
    name: "Šumadijski okrug",
    code: "RS-12",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-12",
};


pub const RS_13: Subdivision = Subdivision {
    name: "Pomoravski okrug",
    code: "RS-13",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-13",
};


pub const RS_14: Subdivision = Subdivision {
    name: "Borski okrug",
    code: "RS-14",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-14",
};


pub const RS_15: Subdivision = Subdivision {
    name: "Zaječarski okrug",
    code: "RS-15",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-15",
};


pub const RS_16: Subdivision = Subdivision {
    name: "Zlatiborski okrug",
    code: "RS-16",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-16",
};


pub const RS_17: Subdivision = Subdivision {
    name: "Moravički okrug",
    code: "RS-17",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-17",
};


pub const RS_18: Subdivision = Subdivision {
    name: "Raški okrug",
    code: "RS-18",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-18",
};


pub const RS_19: Subdivision = Subdivision {
    name: "Rasinski okrug",
    code: "RS-19",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-19",
};


pub const RS_20: Subdivision = Subdivision {
    name: "Nišavski okrug",
    code: "RS-20",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-20",
};


pub const RS_21: Subdivision = Subdivision {
    name: "Toplički okrug",
    code: "RS-21",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-21",
};


pub const RS_22: Subdivision = Subdivision {
    name: "Pirotski okrug",
    code: "RS-22",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-22",
};


pub const RS_23: Subdivision = Subdivision {
    name: "Jablanički okrug",
    code: "RS-23",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-23",
};


pub const RS_24: Subdivision = Subdivision {
    name: "Pčinjski okrug",
    code: "RS-24",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-24",
};


pub const RS_25: Subdivision = Subdivision {
    name: "Kosovski okrug",
    code: "RS-25",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-25",
};


pub const RS_26: Subdivision = Subdivision {
    name: "Pećki okrug",
    code: "RS-26",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-26",
};


pub const RS_27: Subdivision = Subdivision {
    name: "Prizrenski okrug",
    code: "RS-27",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-27",
};


pub const RS_28: Subdivision = Subdivision {
    name: "Kosovsko-Mitrovački okrug",
    code: "RS-28",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-28",
};


pub const RS_29: Subdivision = Subdivision {
    name: "Kosovsko-Pomoravski okrug",
    code: "RS-29",
    subdivision_type: "District",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-29",
};


pub const RS_KM: Subdivision = Subdivision {
    name: "Kosovo-Metohija",
    code: "RS-KM",
    subdivision_type: "Autonomous province",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-KM",
};


pub const RS_VO: Subdivision = Subdivision {
    name: "Vojvodina",
    code: "RS-VO",
    subdivision_type: "Autonomous province",
    country_name: "Serbia",
    country_code: "RS",
    region_code: "RS-VO",
};


pub const SC_01: Subdivision = Subdivision {
    name: "Anse aux Pins",
    code: "SC-01",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-01",
};


pub const SC_02: Subdivision = Subdivision {
    name: "Ans Bwalo",
    code: "SC-02",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-02",
};


pub const SC_03: Subdivision = Subdivision {
    name: "Anse Etoile",
    code: "SC-03",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-03",
};


pub const SC_04: Subdivision = Subdivision {
    name: "Au Cap",
    code: "SC-04",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-04",
};


pub const SC_05: Subdivision = Subdivision {
    name: "Anse Royale",
    code: "SC-05",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-05",
};


pub const SC_06: Subdivision = Subdivision {
    name: "Baie Lazare",
    code: "SC-06",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-06",
};


pub const SC_07: Subdivision = Subdivision {
    name: "Baie Sainte-Anne",
    code: "SC-07",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-07",
};


pub const SC_08: Subdivision = Subdivision {
    name: "Beau Vallon",
    code: "SC-08",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-08",
};


pub const SC_09: Subdivision = Subdivision {
    name: "Bel Air",
    code: "SC-09",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-09",
};


pub const SC_10: Subdivision = Subdivision {
    name: "Bel Ombre",
    code: "SC-10",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-10",
};


pub const SC_11: Subdivision = Subdivision {
    name: "Cascade",
    code: "SC-11",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-11",
};


pub const SC_12: Subdivision = Subdivision {
    name: "Glacis",
    code: "SC-12",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-12",
};


pub const SC_13: Subdivision = Subdivision {
    name: "Grand'Anse Mahé",
    code: "SC-13",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-13",
};


pub const SC_14: Subdivision = Subdivision {
    name: "Grand'Anse Praslin",
    code: "SC-14",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-14",
};


pub const SC_15: Subdivision = Subdivision {
    name: "Ladig",
    code: "SC-15",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-15",
};


pub const SC_16: Subdivision = Subdivision {
    name: "English River",
    code: "SC-16",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-16",
};


pub const SC_17: Subdivision = Subdivision {
    name: "Mon Bikston",
    code: "SC-17",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-17",
};


pub const SC_18: Subdivision = Subdivision {
    name: "Mon Fleri",
    code: "SC-18",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-18",
};


pub const SC_19: Subdivision = Subdivision {
    name: "Plaisance",
    code: "SC-19",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-19",
};


pub const SC_20: Subdivision = Subdivision {
    name: "Pointe La Rue",
    code: "SC-20",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-20",
};


pub const SC_21: Subdivision = Subdivision {
    name: "Porglo",
    code: "SC-21",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-21",
};


pub const SC_22: Subdivision = Subdivision {
    name: "Saint Louis",
    code: "SC-22",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-22",
};


pub const SC_23: Subdivision = Subdivision {
    name: "Takamaka",
    code: "SC-23",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-23",
};


pub const SC_24: Subdivision = Subdivision {
    name: "Lemamel",
    code: "SC-24",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-24",
};


pub const SC_25: Subdivision = Subdivision {
    name: "Roche Caïman",
    code: "SC-25",
    subdivision_type: "District",
    country_name: "Seychelles",
    country_code: "SC",
    region_code: "SC-25",
};


pub const GS_GS: Subdivision = Subdivision {
    name: "S.Georgia & S.Sandwich Islands",
    code: "GS-GS",
    subdivision_type: "Country",
    country_name: "S.Georgia & S.Sandwich Islands",
    country_code: "GS",
    region_code: "GS-GS",
};


pub const SL_E: Subdivision = Subdivision {
    name: "Eastern",
    code: "SL-E",
    subdivision_type: "Province",
    country_name: "Sierra Leone",
    country_code: "SL",
    region_code: "SL-E",
};


pub const SL_N: Subdivision = Subdivision {
    name: "Northern",
    code: "SL-N",
    subdivision_type: "Province",
    country_name: "Sierra Leone",
    country_code: "SL",
    region_code: "SL-N",
};


pub const SL_S: Subdivision = Subdivision {
    name: "Southern",
    code: "SL-S",
    subdivision_type: "Province",
    country_name: "Sierra Leone",
    country_code: "SL",
    region_code: "SL-S",
};


pub const SL_W: Subdivision = Subdivision {
    name: "Western Area",
    code: "SL-W",
    subdivision_type: "Area",
    country_name: "Sierra Leone",
    country_code: "SL",
    region_code: "SL-W",
};


pub const SG_01: Subdivision = Subdivision {
    name: "Central Singapore",
    code: "SG-01",
    subdivision_type: "District",
    country_name: "Singapore",
    country_code: "SG",
    region_code: "SG-01",
};


pub const SG_02: Subdivision = Subdivision {
    name: "North East",
    code: "SG-02",
    subdivision_type: "District",
    country_name: "Singapore",
    country_code: "SG",
    region_code: "SG-02",
};


pub const SG_03: Subdivision = Subdivision {
    name: "North West",
    code: "SG-03",
    subdivision_type: "District",
    country_name: "Singapore",
    country_code: "SG",
    region_code: "SG-03",
};


pub const SG_04: Subdivision = Subdivision {
    name: "South East",
    code: "SG-04",
    subdivision_type: "District",
    country_name: "Singapore",
    country_code: "SG",
    region_code: "SG-04",
};


pub const SG_05: Subdivision = Subdivision {
    name: "South West",
    code: "SG-05",
    subdivision_type: "District",
    country_name: "Singapore",
    country_code: "SG",
    region_code: "SG-05",
};


pub const SK_BC: Subdivision = Subdivision {
    name: "Banskobystrický kraj",
    code: "SK-BC",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-BC",
};


pub const SK_BL: Subdivision = Subdivision {
    name: "Bratislavský kraj",
    code: "SK-BL",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-BL",
};


pub const SK_KI: Subdivision = Subdivision {
    name: "Košický kraj",
    code: "SK-KI",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-KI",
};


pub const SK_NI: Subdivision = Subdivision {
    name: "Nitriansky kraj",
    code: "SK-NI",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-NI",
};


pub const SK_PV: Subdivision = Subdivision {
    name: "Prešovský kraj",
    code: "SK-PV",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-PV",
};


pub const SK_TA: Subdivision = Subdivision {
    name: "Trnavský kraj",
    code: "SK-TA",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-TA",
};


pub const SK_TC: Subdivision = Subdivision {
    name: "Trenčiansky kraj",
    code: "SK-TC",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-TC",
};


pub const SK_ZI: Subdivision = Subdivision {
    name: "Žilinský kraj",
    code: "SK-ZI",
    subdivision_type: "Region",
    country_name: "Slovakia",
    country_code: "SK",
    region_code: "SK-ZI",
};


pub const SI_001: Subdivision = Subdivision {
    name: "Ajdovščina",
    code: "SI-001",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-001",
};


pub const SI_002: Subdivision = Subdivision {
    name: "Beltinci",
    code: "SI-002",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-002",
};


pub const SI_003: Subdivision = Subdivision {
    name: "Bled",
    code: "SI-003",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-003",
};


pub const SI_004: Subdivision = Subdivision {
    name: "Bohinj",
    code: "SI-004",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-004",
};


pub const SI_005: Subdivision = Subdivision {
    name: "Borovnica",
    code: "SI-005",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-005",
};


pub const SI_006: Subdivision = Subdivision {
    name: "Bovec",
    code: "SI-006",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-006",
};


pub const SI_007: Subdivision = Subdivision {
    name: "Brda",
    code: "SI-007",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-007",
};


pub const SI_008: Subdivision = Subdivision {
    name: "Brezovica",
    code: "SI-008",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-008",
};


pub const SI_009: Subdivision = Subdivision {
    name: "Brežice",
    code: "SI-009",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-009",
};


pub const SI_010: Subdivision = Subdivision {
    name: "Tišina",
    code: "SI-010",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-010",
};


pub const SI_011: Subdivision = Subdivision {
    name: "Celje",
    code: "SI-011",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-011",
};


pub const SI_012: Subdivision = Subdivision {
    name: "Cerklje na Gorenjskem",
    code: "SI-012",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-012",
};


pub const SI_013: Subdivision = Subdivision {
    name: "Cerknica",
    code: "SI-013",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-013",
};


pub const SI_014: Subdivision = Subdivision {
    name: "Cerkno",
    code: "SI-014",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-014",
};


pub const SI_015: Subdivision = Subdivision {
    name: "Črenšovci",
    code: "SI-015",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-015",
};


pub const SI_016: Subdivision = Subdivision {
    name: "Črna na Koroškem",
    code: "SI-016",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-016",
};


pub const SI_017: Subdivision = Subdivision {
    name: "Črnomelj",
    code: "SI-017",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-017",
};


pub const SI_018: Subdivision = Subdivision {
    name: "Destrnik",
    code: "SI-018",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-018",
};


pub const SI_019: Subdivision = Subdivision {
    name: "Divača",
    code: "SI-019",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-019",
};


pub const SI_020: Subdivision = Subdivision {
    name: "Dobrepolje",
    code: "SI-020",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-020",
};


pub const SI_021: Subdivision = Subdivision {
    name: "Dobrova-Polhov Gradec",
    code: "SI-021",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-021",
};


pub const SI_022: Subdivision = Subdivision {
    name: "Dol pri Ljubljani",
    code: "SI-022",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-022",
};


pub const SI_023: Subdivision = Subdivision {
    name: "Domžale",
    code: "SI-023",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-023",
};


pub const SI_024: Subdivision = Subdivision {
    name: "Dornava",
    code: "SI-024",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-024",
};


pub const SI_025: Subdivision = Subdivision {
    name: "Dravograd",
    code: "SI-025",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-025",
};


pub const SI_026: Subdivision = Subdivision {
    name: "Duplek",
    code: "SI-026",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-026",
};


pub const SI_027: Subdivision = Subdivision {
    name: "Gorenja vas-Poljane",
    code: "SI-027",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-027",
};


pub const SI_028: Subdivision = Subdivision {
    name: "Gorišnica",
    code: "SI-028",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-028",
};


pub const SI_029: Subdivision = Subdivision {
    name: "Gornja Radgona",
    code: "SI-029",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-029",
};


pub const SI_030: Subdivision = Subdivision {
    name: "Gornji Grad",
    code: "SI-030",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-030",
};


pub const SI_031: Subdivision = Subdivision {
    name: "Gornji Petrovci",
    code: "SI-031",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-031",
};


pub const SI_032: Subdivision = Subdivision {
    name: "Grosuplje",
    code: "SI-032",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-032",
};


pub const SI_033: Subdivision = Subdivision {
    name: "Šalovci",
    code: "SI-033",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-033",
};


pub const SI_034: Subdivision = Subdivision {
    name: "Hrastnik",
    code: "SI-034",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-034",
};


pub const SI_035: Subdivision = Subdivision {
    name: "Hrpelje-Kozina",
    code: "SI-035",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-035",
};


pub const SI_036: Subdivision = Subdivision {
    name: "Idrija",
    code: "SI-036",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-036",
};


pub const SI_037: Subdivision = Subdivision {
    name: "Ig",
    code: "SI-037",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-037",
};


pub const SI_038: Subdivision = Subdivision {
    name: "Ilirska Bistrica",
    code: "SI-038",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-038",
};


pub const SI_039: Subdivision = Subdivision {
    name: "Ivančna Gorica",
    code: "SI-039",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-039",
};


pub const SI_040: Subdivision = Subdivision {
    name: "Izola",
    code: "SI-040",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-040",
};


pub const SI_041: Subdivision = Subdivision {
    name: "Jesenice",
    code: "SI-041",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-041",
};


pub const SI_042: Subdivision = Subdivision {
    name: "Juršinci",
    code: "SI-042",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-042",
};


pub const SI_043: Subdivision = Subdivision {
    name: "Kamnik",
    code: "SI-043",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-043",
};


pub const SI_044: Subdivision = Subdivision {
    name: "Kanal",
    code: "SI-044",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-044",
};


pub const SI_045: Subdivision = Subdivision {
    name: "Kidričevo",
    code: "SI-045",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-045",
};


pub const SI_046: Subdivision = Subdivision {
    name: "Kobarid",
    code: "SI-046",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-046",
};


pub const SI_047: Subdivision = Subdivision {
    name: "Kobilje",
    code: "SI-047",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-047",
};


pub const SI_048: Subdivision = Subdivision {
    name: "Kočevje",
    code: "SI-048",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-048",
};


pub const SI_049: Subdivision = Subdivision {
    name: "Komen",
    code: "SI-049",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-049",
};


pub const SI_050: Subdivision = Subdivision {
    name: "Koper",
    code: "SI-050",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-050",
};


pub const SI_051: Subdivision = Subdivision {
    name: "Kozje",
    code: "SI-051",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-051",
};


pub const SI_052: Subdivision = Subdivision {
    name: "Kranj",
    code: "SI-052",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-052",
};


pub const SI_053: Subdivision = Subdivision {
    name: "Kranjska Gora",
    code: "SI-053",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-053",
};


pub const SI_054: Subdivision = Subdivision {
    name: "Krško",
    code: "SI-054",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-054",
};


pub const SI_055: Subdivision = Subdivision {
    name: "Kungota",
    code: "SI-055",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-055",
};


pub const SI_056: Subdivision = Subdivision {
    name: "Kuzma",
    code: "SI-056",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-056",
};


pub const SI_057: Subdivision = Subdivision {
    name: "Laško",
    code: "SI-057",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-057",
};


pub const SI_058: Subdivision = Subdivision {
    name: "Lenart",
    code: "SI-058",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-058",
};


pub const SI_059: Subdivision = Subdivision {
    name: "Lendava",
    code: "SI-059",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-059",
};


pub const SI_060: Subdivision = Subdivision {
    name: "Litija",
    code: "SI-060",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-060",
};


pub const SI_061: Subdivision = Subdivision {
    name: "Ljubljana",
    code: "SI-061",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-061",
};


pub const SI_062: Subdivision = Subdivision {
    name: "Ljubno",
    code: "SI-062",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-062",
};


pub const SI_063: Subdivision = Subdivision {
    name: "Ljutomer",
    code: "SI-063",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-063",
};


pub const SI_064: Subdivision = Subdivision {
    name: "Logatec",
    code: "SI-064",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-064",
};


pub const SI_065: Subdivision = Subdivision {
    name: "Loška Dolina",
    code: "SI-065",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-065",
};


pub const SI_066: Subdivision = Subdivision {
    name: "Loški Potok",
    code: "SI-066",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-066",
};


pub const SI_067: Subdivision = Subdivision {
    name: "Luče",
    code: "SI-067",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-067",
};


pub const SI_068: Subdivision = Subdivision {
    name: "Lukovica",
    code: "SI-068",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-068",
};


pub const SI_069: Subdivision = Subdivision {
    name: "Majšperk",
    code: "SI-069",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-069",
};


pub const SI_070: Subdivision = Subdivision {
    name: "Maribor",
    code: "SI-070",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-070",
};


pub const SI_071: Subdivision = Subdivision {
    name: "Medvode",
    code: "SI-071",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-071",
};


pub const SI_072: Subdivision = Subdivision {
    name: "Mengeš",
    code: "SI-072",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-072",
};


pub const SI_073: Subdivision = Subdivision {
    name: "Metlika",
    code: "SI-073",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-073",
};


pub const SI_074: Subdivision = Subdivision {
    name: "Mežica",
    code: "SI-074",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-074",
};


pub const SI_075: Subdivision = Subdivision {
    name: "Miren-Kostanjevica",
    code: "SI-075",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-075",
};


pub const SI_076: Subdivision = Subdivision {
    name: "Mislinja",
    code: "SI-076",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-076",
};


pub const SI_077: Subdivision = Subdivision {
    name: "Moravče",
    code: "SI-077",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-077",
};


pub const SI_078: Subdivision = Subdivision {
    name: "Moravske Toplice",
    code: "SI-078",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-078",
};


pub const SI_079: Subdivision = Subdivision {
    name: "Mozirje",
    code: "SI-079",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-079",
};


pub const SI_080: Subdivision = Subdivision {
    name: "Murska Sobota",
    code: "SI-080",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-080",
};


pub const SI_081: Subdivision = Subdivision {
    name: "Muta",
    code: "SI-081",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-081",
};


pub const SI_082: Subdivision = Subdivision {
    name: "Naklo",
    code: "SI-082",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-082",
};


pub const SI_083: Subdivision = Subdivision {
    name: "Nazarje",
    code: "SI-083",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-083",
};


pub const SI_084: Subdivision = Subdivision {
    name: "Nova Gorica",
    code: "SI-084",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-084",
};


pub const SI_085: Subdivision = Subdivision {
    name: "Novo Mesto",
    code: "SI-085",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-085",
};


pub const SI_086: Subdivision = Subdivision {
    name: "Odranci",
    code: "SI-086",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-086",
};


pub const SI_087: Subdivision = Subdivision {
    name: "Ormož",
    code: "SI-087",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-087",
};


pub const SI_088: Subdivision = Subdivision {
    name: "Osilnica",
    code: "SI-088",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-088",
};


pub const SI_089: Subdivision = Subdivision {
    name: "Pesnica",
    code: "SI-089",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-089",
};


pub const SI_090: Subdivision = Subdivision {
    name: "Piran",
    code: "SI-090",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-090",
};


pub const SI_091: Subdivision = Subdivision {
    name: "Pivka",
    code: "SI-091",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-091",
};


pub const SI_092: Subdivision = Subdivision {
    name: "Podčetrtek",
    code: "SI-092",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-092",
};


pub const SI_093: Subdivision = Subdivision {
    name: "Podvelka",
    code: "SI-093",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-093",
};


pub const SI_094: Subdivision = Subdivision {
    name: "Postojna",
    code: "SI-094",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-094",
};


pub const SI_095: Subdivision = Subdivision {
    name: "Preddvor",
    code: "SI-095",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-095",
};


pub const SI_096: Subdivision = Subdivision {
    name: "Ptuj",
    code: "SI-096",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-096",
};


pub const SI_097: Subdivision = Subdivision {
    name: "Puconci",
    code: "SI-097",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-097",
};


pub const SI_098: Subdivision = Subdivision {
    name: "Rače-Fram",
    code: "SI-098",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-098",
};


pub const SI_099: Subdivision = Subdivision {
    name: "Radeče",
    code: "SI-099",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-099",
};


pub const SI_100: Subdivision = Subdivision {
    name: "Radenci",
    code: "SI-100",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-100",
};


pub const SI_101: Subdivision = Subdivision {
    name: "Radlje ob Dravi",
    code: "SI-101",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-101",
};


pub const SI_102: Subdivision = Subdivision {
    name: "Radovljica",
    code: "SI-102",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-102",
};


pub const SI_103: Subdivision = Subdivision {
    name: "Ravne na Koroškem",
    code: "SI-103",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-103",
};


pub const SI_104: Subdivision = Subdivision {
    name: "Ribnica",
    code: "SI-104",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-104",
};


pub const SI_105: Subdivision = Subdivision {
    name: "Rogašovci",
    code: "SI-105",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-105",
};


pub const SI_106: Subdivision = Subdivision {
    name: "Rogaška Slatina",
    code: "SI-106",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-106",
};


pub const SI_107: Subdivision = Subdivision {
    name: "Rogatec",
    code: "SI-107",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-107",
};


pub const SI_108: Subdivision = Subdivision {
    name: "Ruše",
    code: "SI-108",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-108",
};


pub const SI_109: Subdivision = Subdivision {
    name: "Semič",
    code: "SI-109",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-109",
};


pub const SI_110: Subdivision = Subdivision {
    name: "Sevnica",
    code: "SI-110",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-110",
};


pub const SI_111: Subdivision = Subdivision {
    name: "Sežana",
    code: "SI-111",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-111",
};


pub const SI_112: Subdivision = Subdivision {
    name: "Slovenj Gradec",
    code: "SI-112",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-112",
};


pub const SI_113: Subdivision = Subdivision {
    name: "Slovenska Bistrica",
    code: "SI-113",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-113",
};


pub const SI_114: Subdivision = Subdivision {
    name: "Slovenske Konjice",
    code: "SI-114",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-114",
};


pub const SI_115: Subdivision = Subdivision {
    name: "Starše",
    code: "SI-115",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-115",
};


pub const SI_116: Subdivision = Subdivision {
    name: "Sveti Jurij",
    code: "SI-116",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-116",
};


pub const SI_117: Subdivision = Subdivision {
    name: "Šenčur",
    code: "SI-117",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-117",
};


pub const SI_118: Subdivision = Subdivision {
    name: "Šentilj",
    code: "SI-118",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-118",
};


pub const SI_119: Subdivision = Subdivision {
    name: "Šentjernej",
    code: "SI-119",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-119",
};


pub const SI_120: Subdivision = Subdivision {
    name: "Šentjur",
    code: "SI-120",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-120",
};


pub const SI_121: Subdivision = Subdivision {
    name: "Škocjan",
    code: "SI-121",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-121",
};


pub const SI_122: Subdivision = Subdivision {
    name: "Škofja Loka",
    code: "SI-122",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-122",
};


pub const SI_123: Subdivision = Subdivision {
    name: "Škofljica",
    code: "SI-123",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-123",
};


pub const SI_124: Subdivision = Subdivision {
    name: "Šmarje pri Jelšah",
    code: "SI-124",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-124",
};


pub const SI_125: Subdivision = Subdivision {
    name: "Šmartno ob Paki",
    code: "SI-125",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-125",
};


pub const SI_126: Subdivision = Subdivision {
    name: "Šoštanj",
    code: "SI-126",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-126",
};


pub const SI_127: Subdivision = Subdivision {
    name: "Štore",
    code: "SI-127",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-127",
};


pub const SI_128: Subdivision = Subdivision {
    name: "Tolmin",
    code: "SI-128",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-128",
};


pub const SI_129: Subdivision = Subdivision {
    name: "Trbovlje",
    code: "SI-129",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-129",
};


pub const SI_130: Subdivision = Subdivision {
    name: "Trebnje",
    code: "SI-130",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-130",
};


pub const SI_131: Subdivision = Subdivision {
    name: "Tržič",
    code: "SI-131",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-131",
};


pub const SI_132: Subdivision = Subdivision {
    name: "Turnišče",
    code: "SI-132",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-132",
};


pub const SI_133: Subdivision = Subdivision {
    name: "Velenje",
    code: "SI-133",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-133",
};


pub const SI_134: Subdivision = Subdivision {
    name: "Velike Lašče",
    code: "SI-134",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-134",
};


pub const SI_135: Subdivision = Subdivision {
    name: "Videm",
    code: "SI-135",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-135",
};


pub const SI_136: Subdivision = Subdivision {
    name: "Vipava",
    code: "SI-136",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-136",
};


pub const SI_137: Subdivision = Subdivision {
    name: "Vitanje",
    code: "SI-137",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-137",
};


pub const SI_138: Subdivision = Subdivision {
    name: "Vodice",
    code: "SI-138",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-138",
};


pub const SI_139: Subdivision = Subdivision {
    name: "Vojnik",
    code: "SI-139",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-139",
};


pub const SI_140: Subdivision = Subdivision {
    name: "Vrhnika",
    code: "SI-140",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-140",
};


pub const SI_141: Subdivision = Subdivision {
    name: "Vuzenica",
    code: "SI-141",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-141",
};


pub const SI_142: Subdivision = Subdivision {
    name: "Zagorje ob Savi",
    code: "SI-142",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-142",
};


pub const SI_143: Subdivision = Subdivision {
    name: "Zavrč",
    code: "SI-143",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-143",
};


pub const SI_144: Subdivision = Subdivision {
    name: "Zreče",
    code: "SI-144",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-144",
};


pub const SI_146: Subdivision = Subdivision {
    name: "Železniki",
    code: "SI-146",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-146",
};


pub const SI_147: Subdivision = Subdivision {
    name: "Žiri",
    code: "SI-147",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-147",
};


pub const SI_148: Subdivision = Subdivision {
    name: "Benedikt",
    code: "SI-148",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-148",
};


pub const SI_149: Subdivision = Subdivision {
    name: "Bistrica ob Sotli",
    code: "SI-149",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-149",
};


pub const SI_150: Subdivision = Subdivision {
    name: "Bloke",
    code: "SI-150",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-150",
};


pub const SI_151: Subdivision = Subdivision {
    name: "Braslovče",
    code: "SI-151",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-151",
};


pub const SI_152: Subdivision = Subdivision {
    name: "Cankova",
    code: "SI-152",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-152",
};


pub const SI_153: Subdivision = Subdivision {
    name: "Cerkvenjak",
    code: "SI-153",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-153",
};


pub const SI_154: Subdivision = Subdivision {
    name: "Dobje",
    code: "SI-154",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-154",
};


pub const SI_155: Subdivision = Subdivision {
    name: "Dobrna",
    code: "SI-155",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-155",
};


pub const SI_156: Subdivision = Subdivision {
    name: "Dobrovnik",
    code: "SI-156",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-156",
};


pub const SI_157: Subdivision = Subdivision {
    name: "Dolenjske Toplice",
    code: "SI-157",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-157",
};


pub const SI_158: Subdivision = Subdivision {
    name: "Grad",
    code: "SI-158",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-158",
};


pub const SI_159: Subdivision = Subdivision {
    name: "Hajdina",
    code: "SI-159",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-159",
};


pub const SI_160: Subdivision = Subdivision {
    name: "Hoče-Slivnica",
    code: "SI-160",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-160",
};


pub const SI_161: Subdivision = Subdivision {
    name: "Hodoš",
    code: "SI-161",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-161",
};


pub const SI_162: Subdivision = Subdivision {
    name: "Horjul",
    code: "SI-162",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-162",
};


pub const SI_163: Subdivision = Subdivision {
    name: "Jezersko",
    code: "SI-163",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-163",
};


pub const SI_164: Subdivision = Subdivision {
    name: "Komenda",
    code: "SI-164",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-164",
};


pub const SI_165: Subdivision = Subdivision {
    name: "Kostel",
    code: "SI-165",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-165",
};


pub const SI_166: Subdivision = Subdivision {
    name: "Križevci",
    code: "SI-166",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-166",
};


pub const SI_167: Subdivision = Subdivision {
    name: "Lovrenc na Pohorju",
    code: "SI-167",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-167",
};


pub const SI_168: Subdivision = Subdivision {
    name: "Markovci",
    code: "SI-168",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-168",
};


pub const SI_169: Subdivision = Subdivision {
    name: "Miklavž na Dravskem Polju",
    code: "SI-169",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-169",
};


pub const SI_170: Subdivision = Subdivision {
    name: "Mirna Peč",
    code: "SI-170",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-170",
};


pub const SI_171: Subdivision = Subdivision {
    name: "Oplotnica",
    code: "SI-171",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-171",
};


pub const SI_172: Subdivision = Subdivision {
    name: "Podlehnik",
    code: "SI-172",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-172",
};


pub const SI_173: Subdivision = Subdivision {
    name: "Polzela",
    code: "SI-173",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-173",
};


pub const SI_174: Subdivision = Subdivision {
    name: "Prebold",
    code: "SI-174",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-174",
};


pub const SI_175: Subdivision = Subdivision {
    name: "Prevalje",
    code: "SI-175",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-175",
};


pub const SI_176: Subdivision = Subdivision {
    name: "Razkrižje",
    code: "SI-176",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-176",
};


pub const SI_177: Subdivision = Subdivision {
    name: "Ribnica na Pohorju",
    code: "SI-177",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-177",
};


pub const SI_178: Subdivision = Subdivision {
    name: "Selnica ob Dravi",
    code: "SI-178",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-178",
};


pub const SI_179: Subdivision = Subdivision {
    name: "Sodražica",
    code: "SI-179",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-179",
};


pub const SI_180: Subdivision = Subdivision {
    name: "Solčava",
    code: "SI-180",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-180",
};


pub const SI_181: Subdivision = Subdivision {
    name: "Sveta Ana",
    code: "SI-181",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-181",
};


pub const SI_182: Subdivision = Subdivision {
    name: "Sveti Andraž v Slovenskih Goricah",
    code: "SI-182",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-182",
};


pub const SI_183: Subdivision = Subdivision {
    name: "Šempeter-Vrtojba",
    code: "SI-183",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-183",
};


pub const SI_184: Subdivision = Subdivision {
    name: "Tabor",
    code: "SI-184",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-184",
};


pub const SI_185: Subdivision = Subdivision {
    name: "Trnovska Vas",
    code: "SI-185",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-185",
};


pub const SI_186: Subdivision = Subdivision {
    name: "Trzin",
    code: "SI-186",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-186",
};


pub const SI_187: Subdivision = Subdivision {
    name: "Velika Polana",
    code: "SI-187",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-187",
};


pub const SI_188: Subdivision = Subdivision {
    name: "Veržej",
    code: "SI-188",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-188",
};


pub const SI_189: Subdivision = Subdivision {
    name: "Vransko",
    code: "SI-189",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-189",
};


pub const SI_190: Subdivision = Subdivision {
    name: "Žalec",
    code: "SI-190",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-190",
};


pub const SI_191: Subdivision = Subdivision {
    name: "Žetale",
    code: "SI-191",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-191",
};


pub const SI_192: Subdivision = Subdivision {
    name: "Žirovnica",
    code: "SI-192",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-192",
};


pub const SI_193: Subdivision = Subdivision {
    name: "Žužemberk",
    code: "SI-193",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-193",
};


pub const SI_194: Subdivision = Subdivision {
    name: "Šmartno pri Litiji",
    code: "SI-194",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-194",
};


pub const SI_195: Subdivision = Subdivision {
    name: "Apače",
    code: "SI-195",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-195",
};


pub const SI_196: Subdivision = Subdivision {
    name: "Cirkulane",
    code: "SI-196",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-196",
};


pub const SI_197: Subdivision = Subdivision {
    name: "Kosanjevica na Krki",
    code: "SI-197",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-197",
};


pub const SI_198: Subdivision = Subdivision {
    name: "Makole",
    code: "SI-198",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-198",
};


pub const SI_199: Subdivision = Subdivision {
    name: "Mokronog-Trebelno",
    code: "SI-199",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-199",
};


pub const SI_200: Subdivision = Subdivision {
    name: "Poljčane",
    code: "SI-200",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-200",
};


pub const SI_201: Subdivision = Subdivision {
    name: "Renče-Vogrsko",
    code: "SI-201",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-201",
};


pub const SI_202: Subdivision = Subdivision {
    name: "Središče ob Dravi",
    code: "SI-202",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-202",
};


pub const SI_203: Subdivision = Subdivision {
    name: "Straža",
    code: "SI-203",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-203",
};


pub const SI_204: Subdivision = Subdivision {
    name: "Sveta Trojica v Slovenskih Goricah",
    code: "SI-204",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-204",
};


pub const SI_205: Subdivision = Subdivision {
    name: "Sveti Tomaž",
    code: "SI-205",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-205",
};


pub const SI_206: Subdivision = Subdivision {
    name: "Šmarješke Toplice",
    code: "SI-206",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-206",
};


pub const SI_207: Subdivision = Subdivision {
    name: "Gorje",
    code: "SI-207",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-207",
};


pub const SI_208: Subdivision = Subdivision {
    name: "Log-Dragomer",
    code: "SI-208",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-208",
};


pub const SI_209: Subdivision = Subdivision {
    name: "Rečica ob Savinji",
    code: "SI-209",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-209",
};


pub const SI_210: Subdivision = Subdivision {
    name: "Sveti Jurij v Slovenskih Goricah",
    code: "SI-210",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-210",
};


pub const SI_211: Subdivision = Subdivision {
    name: "Šentrupert",
    code: "SI-211",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-211",
};


pub const SI_212: Subdivision = Subdivision {
    name: "Mirna",
    code: "SI-212",
    subdivision_type: "Commune",
    country_name: "Slovenia",
    country_code: "SI",
    region_code: "SI-212",
};


pub const SB_CE: Subdivision = Subdivision {
    name: "Central",
    code: "SB-CE",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-CE",
};


pub const SB_CH: Subdivision = Subdivision {
    name: "Choiseul",
    code: "SB-CH",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-CH",
};


pub const SB_CT: Subdivision = Subdivision {
    name: "Capital Territory",
    code: "SB-CT",
    subdivision_type: "Capital territory",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-CT",
};


pub const SB_GU: Subdivision = Subdivision {
    name: "Guadalcanal",
    code: "SB-GU",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-GU",
};


pub const SB_IS: Subdivision = Subdivision {
    name: "Isabel",
    code: "SB-IS",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-IS",
};


pub const SB_MK: Subdivision = Subdivision {
    name: "Makira-Ulawa",
    code: "SB-MK",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-MK",
};


pub const SB_ML: Subdivision = Subdivision {
    name: "Malaita",
    code: "SB-ML",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-ML",
};


pub const SB_RB: Subdivision = Subdivision {
    name: "Rennell and Bellona",
    code: "SB-RB",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-RB",
};


pub const SB_TE: Subdivision = Subdivision {
    name: "Temotu",
    code: "SB-TE",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-TE",
};


pub const SB_WE: Subdivision = Subdivision {
    name: "Western",
    code: "SB-WE",
    subdivision_type: "Province",
    country_name: "Solomon Islands",
    country_code: "SB",
    region_code: "SB-WE",
};


pub const SO_AW: Subdivision = Subdivision {
    name: "Awdal",
    code: "SO-AW",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-AW",
};


pub const SO_BK: Subdivision = Subdivision {
    name: "Bakool",
    code: "SO-BK",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-BK",
};


pub const SO_BN: Subdivision = Subdivision {
    name: "Banaadir",
    code: "SO-BN",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-BN",
};


pub const SO_BR: Subdivision = Subdivision {
    name: "Bari",
    code: "SO-BR",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-BR",
};


pub const SO_BY: Subdivision = Subdivision {
    name: "Bay",
    code: "SO-BY",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-BY",
};


pub const SO_GA: Subdivision = Subdivision {
    name: "Galguduud",
    code: "SO-GA",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-GA",
};


pub const SO_GE: Subdivision = Subdivision {
    name: "Gedo",
    code: "SO-GE",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-GE",
};


pub const SO_HI: Subdivision = Subdivision {
    name: "Hiiraan",
    code: "SO-HI",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-HI",
};


pub const SO_JD: Subdivision = Subdivision {
    name: "Jubbada Dhexe",
    code: "SO-JD",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-JD",
};


pub const SO_JH: Subdivision = Subdivision {
    name: "Jubbada Hoose",
    code: "SO-JH",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-JH",
};


pub const SO_MU: Subdivision = Subdivision {
    name: "Mudug",
    code: "SO-MU",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-MU",
};


pub const SO_NU: Subdivision = Subdivision {
    name: "Nugaal",
    code: "SO-NU",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-NU",
};


pub const SO_SA: Subdivision = Subdivision {
    name: "Sanaag",
    code: "SO-SA",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-SA",
};


pub const SO_SD: Subdivision = Subdivision {
    name: "Shabeellaha Dhexe",
    code: "SO-SD",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-SD",
};


pub const SO_SH: Subdivision = Subdivision {
    name: "Shabeellaha Hoose",
    code: "SO-SH",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-SH",
};


pub const SO_SO: Subdivision = Subdivision {
    name: "Sool",
    code: "SO-SO",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-SO",
};


pub const SO_TO: Subdivision = Subdivision {
    name: "Togdheer",
    code: "SO-TO",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-TO",
};


pub const SO_WO: Subdivision = Subdivision {
    name: "Woqooyi Galbeed",
    code: "SO-WO",
    subdivision_type: "Region",
    country_name: "Somalia",
    country_code: "SO",
    region_code: "SO-WO",
};


pub const ZA_EC: Subdivision = Subdivision {
    name: "Eastern Cape",
    code: "ZA-EC",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-EC",
};


pub const ZA_FS: Subdivision = Subdivision {
    name: "Foreisetata",
    code: "ZA-FS",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-FS",
};


pub const ZA_GT: Subdivision = Subdivision {
    name: "Gauteng",
    code: "ZA-GT",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-GT",
};


pub const ZA_LP: Subdivision = Subdivision {
    name: "Limpopo",
    code: "ZA-LP",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-LP",
};


pub const ZA_MP: Subdivision = Subdivision {
    name: "iMpumalanga",
    code: "ZA-MP",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-MP",
};


pub const ZA_NC: Subdivision = Subdivision {
    name: "iTlhagwini-Kapa",
    code: "ZA-NC",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-NC",
};


pub const ZA_NL: Subdivision = Subdivision {
    name: "GaZulu-Natala",
    code: "ZA-NL",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-NL",
};


pub const ZA_NW: Subdivision = Subdivision {
    name: "Bokone Bophirima",
    code: "ZA-NW",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-NW",
};


pub const ZA_WC: Subdivision = Subdivision {
    name: "iTjhingalanga-Kapa",
    code: "ZA-WC",
    subdivision_type: "Province",
    country_name: "South Africa",
    country_code: "ZA",
    region_code: "ZA-WC",
};


pub const SS_BN: Subdivision = Subdivision {
    name: "Northern Bahr el Ghazal",
    code: "SS-BN",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-BN",
};


pub const SS_BW: Subdivision = Subdivision {
    name: "Western Bahr el Ghazal",
    code: "SS-BW",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-BW",
};


pub const SS_EC: Subdivision = Subdivision {
    name: "Central Equatoria",
    code: "SS-EC",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-EC",
};


pub const SS_EE: Subdivision = Subdivision {
    name: "Eastern Equatoria",
    code: "SS-EE",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-EE",
};


pub const SS_EW: Subdivision = Subdivision {
    name: "Western Equatoria",
    code: "SS-EW",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-EW",
};


pub const SS_JG: Subdivision = Subdivision {
    name: "Jonglei",
    code: "SS-JG",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-JG",
};


pub const SS_LK: Subdivision = Subdivision {
    name: "Lakes",
    code: "SS-LK",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-LK",
};


pub const SS_NU: Subdivision = Subdivision {
    name: "Upper Nile",
    code: "SS-NU",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-NU",
};


pub const SS_UY: Subdivision = Subdivision {
    name: "Unity",
    code: "SS-UY",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-UY",
};


pub const SS_WR: Subdivision = Subdivision {
    name: "Warrap",
    code: "SS-WR",
    subdivision_type: "State",
    country_name: "South Sudan",
    country_code: "SS",
    region_code: "SS-WR",
};


pub const ES_A: Subdivision = Subdivision {
    name: "Alacant*",
    code: "ES-A",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-A",
};


pub const ES_AB: Subdivision = Subdivision {
    name: "Albacete",
    code: "ES-AB",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AB",
};


pub const ES_AL: Subdivision = Subdivision {
    name: "Almería",
    code: "ES-AL",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AL",
};


pub const ES_AN: Subdivision = Subdivision {
    name: "Andalucía",
    code: "ES-AN",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AN",
};


pub const ES_AR: Subdivision = Subdivision {
    name: "Aragón",
    code: "ES-AR",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AR",
};


pub const ES_AS: Subdivision = Subdivision {
    name: "Asturias, Principado de",
    code: "ES-AS",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AS",
};


pub const ES_AV: Subdivision = Subdivision {
    name: "Ávila",
    code: "ES-AV",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-AV",
};


pub const ES_BA: Subdivision = Subdivision {
    name: "Badajoz",
    code: "ES-BA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-BA",
};


pub const ES_B: Subdivision = Subdivision {
    name: "Barcelona",
    code: "ES-B",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-B",
};


pub const ES_BI: Subdivision = Subdivision {
    name: "Bizkaia*",
    code: "ES-BI",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-BI",
};


pub const ES_BU: Subdivision = Subdivision {
    name: "Burgos",
    code: "ES-BU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-BU",
};


pub const ES_CA: Subdivision = Subdivision {
    name: "Cádiz",
    code: "ES-CA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CA",
};


pub const ES_C: Subdivision = Subdivision {
    name: "A Coruña",
    code: "ES-C",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-C",
};


pub const ES_CB: Subdivision = Subdivision {
    name: "Cantabria",
    code: "ES-CB",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CB",
};


pub const ES_CC: Subdivision = Subdivision {
    name: "Cáceres",
    code: "ES-CC",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CC",
};


pub const ES_CE: Subdivision = Subdivision {
    name: "Ceuta",
    code: "ES-CE",
    subdivision_type: "Autonomous city in north africa",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CE",
};


pub const ES_CL: Subdivision = Subdivision {
    name: "Castilla y León",
    code: "ES-CL",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CL",
};


pub const ES_CM: Subdivision = Subdivision {
    name: "Castilla-La Mancha",
    code: "ES-CM",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CM",
};


pub const ES_CN: Subdivision = Subdivision {
    name: "Canarias",
    code: "ES-CN",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CN",
};


pub const ES_CO: Subdivision = Subdivision {
    name: "Córdoba",
    code: "ES-CO",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CO",
};


pub const ES_CR: Subdivision = Subdivision {
    name: "Ciudad Real",
    code: "ES-CR",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CR",
};


pub const ES_CS: Subdivision = Subdivision {
    name: "Castelló*",
    code: "ES-CS",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CS",
};


pub const ES_CT: Subdivision = Subdivision {
    name: "Catalunya",
    code: "ES-CT",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CT",
};


pub const ES_CU: Subdivision = Subdivision {
    name: "Cuenca",
    code: "ES-CU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-CU",
};


pub const ES_EX: Subdivision = Subdivision {
    name: "Extremadura",
    code: "ES-EX",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-EX",
};


pub const ES_GA: Subdivision = Subdivision {
    name: "Galicia",
    code: "ES-GA",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-GA",
};


pub const ES_GC: Subdivision = Subdivision {
    name: "Las Palmas",
    code: "ES-GC",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-GC",
};


pub const ES_GI: Subdivision = Subdivision {
    name: "Girona",
    code: "ES-GI",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-GI",
};


pub const ES_GR: Subdivision = Subdivision {
    name: "Granada",
    code: "ES-GR",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-GR",
};


pub const ES_GU: Subdivision = Subdivision {
    name: "Guadalajara",
    code: "ES-GU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-GU",
};


pub const ES_H: Subdivision = Subdivision {
    name: "Huelva",
    code: "ES-H",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-H",
};


pub const ES_HU: Subdivision = Subdivision {
    name: "Huesca",
    code: "ES-HU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-HU",
};


pub const ES_IB: Subdivision = Subdivision {
    name: "Illes Balears",
    code: "ES-IB",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-IB",
};


pub const ES_J: Subdivision = Subdivision {
    name: "Jaén",
    code: "ES-J",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-J",
};


pub const ES_LE: Subdivision = Subdivision {
    name: "León",
    code: "ES-LE",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-LE",
};


pub const ES_L: Subdivision = Subdivision {
    name: "Lleida",
    code: "ES-L",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-L",
};


pub const ES_LO: Subdivision = Subdivision {
    name: "La Rioja",
    code: "ES-LO",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-LO",
};


pub const ES_LU: Subdivision = Subdivision {
    name: "Lugo",
    code: "ES-LU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-LU",
};


pub const ES_MA: Subdivision = Subdivision {
    name: "Málaga",
    code: "ES-MA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-MA",
};


pub const ES_MC: Subdivision = Subdivision {
    name: "Murcia, Región de",
    code: "ES-MC",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-MC",
};


pub const ES_MD: Subdivision = Subdivision {
    name: "Madrid, Comunidad de",
    code: "ES-MD",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-MD",
};


pub const ES_ML: Subdivision = Subdivision {
    name: "Melilla",
    code: "ES-ML",
    subdivision_type: "Autonomous city in north africa",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-ML",
};


pub const ES_M: Subdivision = Subdivision {
    name: "Madrid",
    code: "ES-M",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-M",
};


pub const ES_MU: Subdivision = Subdivision {
    name: "Murcia",
    code: "ES-MU",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-MU",
};


pub const ES_NA: Subdivision = Subdivision {
    name: "Nafarroa*",
    code: "ES-NA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-NA",
};


pub const ES_NC: Subdivision = Subdivision {
    name: "Nafarroako Foru Komunitatea*",
    code: "ES-NC",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-NC",
};


pub const ES_O: Subdivision = Subdivision {
    name: "Asturias",
    code: "ES-O",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-O",
};


pub const ES_OR: Subdivision = Subdivision {
    name: "Ourense",
    code: "ES-OR",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-OR",
};


pub const ES_PM: Subdivision = Subdivision {
    name: "Balears",
    code: "ES-PM",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-PM",
};


pub const ES_PO: Subdivision = Subdivision {
    name: "Pontevedra",
    code: "ES-PO",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-PO",
};


pub const ES_P: Subdivision = Subdivision {
    name: "Palencia",
    code: "ES-P",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-P",
};


pub const ES_PV: Subdivision = Subdivision {
    name: "Euskal Herria",
    code: "ES-PV",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-PV",
};


pub const ES_RI: Subdivision = Subdivision {
    name: "La Rioja",
    code: "ES-RI",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-RI",
};


pub const ES_SA: Subdivision = Subdivision {
    name: "Salamanca",
    code: "ES-SA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-SA",
};


pub const ES_S: Subdivision = Subdivision {
    name: "Cantabria",
    code: "ES-S",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-S",
};


pub const ES_SE: Subdivision = Subdivision {
    name: "Sevilla",
    code: "ES-SE",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-SE",
};


pub const ES_SG: Subdivision = Subdivision {
    name: "Segovia",
    code: "ES-SG",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-SG",
};


pub const ES_SO: Subdivision = Subdivision {
    name: "Soria",
    code: "ES-SO",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-SO",
};


pub const ES_SS: Subdivision = Subdivision {
    name: "Gipuzkoa*",
    code: "ES-SS",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-SS",
};


pub const ES_TE: Subdivision = Subdivision {
    name: "Teruel",
    code: "ES-TE",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-TE",
};


pub const ES_TF: Subdivision = Subdivision {
    name: "Santa Cruz de Tenerife",
    code: "ES-TF",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-TF",
};


pub const ES_TO: Subdivision = Subdivision {
    name: "Toledo",
    code: "ES-TO",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-TO",
};


pub const ES_T: Subdivision = Subdivision {
    name: "Tarragona",
    code: "ES-T",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-T",
};


pub const ES_VA: Subdivision = Subdivision {
    name: "Valladolid",
    code: "ES-VA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-VA",
};


pub const ES_VC: Subdivision = Subdivision {
    name: "Valenciana, Comunidad",
    code: "ES-VC",
    subdivision_type: "Autonomous community",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-VC",
};


pub const ES_VI: Subdivision = Subdivision {
    name: "Álava",
    code: "ES-VI",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-VI",
};


pub const ES_V: Subdivision = Subdivision {
    name: "Valencia",
    code: "ES-V",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-V",
};


pub const ES_ZA: Subdivision = Subdivision {
    name: "Zamora",
    code: "ES-ZA",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-ZA",
};


pub const ES_Z: Subdivision = Subdivision {
    name: "Zaragoza",
    code: "ES-Z",
    subdivision_type: "Province",
    country_name: "Spain",
    country_code: "ES",
    region_code: "ES-Z",
};


pub const LK_11: Subdivision = Subdivision {
    name: "Colombo",
    code: "LK-11",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-11",
};


pub const LK_12: Subdivision = Subdivision {
    name: "Gampaha",
    code: "LK-12",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-12",
};


pub const LK_13: Subdivision = Subdivision {
    name: "Kalutara",
    code: "LK-13",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-13",
};


pub const LK_1: Subdivision = Subdivision {
    name: "Basnāhira paḷāta",
    code: "LK-1",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-1",
};


pub const LK_21: Subdivision = Subdivision {
    name: "Kandy",
    code: "LK-21",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-21",
};


pub const LK_22: Subdivision = Subdivision {
    name: "Matale",
    code: "LK-22",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-22",
};


pub const LK_23: Subdivision = Subdivision {
    name: "Nuvara Ĕliya",
    code: "LK-23",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-23",
};


pub const LK_2: Subdivision = Subdivision {
    name: "Central Province",
    code: "LK-2",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-2",
};


pub const LK_31: Subdivision = Subdivision {
    name: "Gālla",
    code: "LK-31",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-31",
};


pub const LK_32: Subdivision = Subdivision {
    name: "Matara",
    code: "LK-32",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-32",
};


pub const LK_33: Subdivision = Subdivision {
    name: "Ampāntōṭṭai",
    code: "LK-33",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-33",
};


pub const LK_3: Subdivision = Subdivision {
    name: "Dakuṇu paḷāta",
    code: "LK-3",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-3",
};


pub const LK_41: Subdivision = Subdivision {
    name: "Jaffna",
    code: "LK-41",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-41",
};


pub const LK_42: Subdivision = Subdivision {
    name: "Kilinochchi",
    code: "LK-42",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-42",
};


pub const LK_43: Subdivision = Subdivision {
    name: "Mannar",
    code: "LK-43",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-43",
};


pub const LK_44: Subdivision = Subdivision {
    name: "Vavuniya",
    code: "LK-44",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-44",
};


pub const LK_45: Subdivision = Subdivision {
    name: "Mulativ",
    code: "LK-45",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-45",
};


pub const LK_4: Subdivision = Subdivision {
    name: "Northern Province",
    code: "LK-4",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-4",
};


pub const LK_51: Subdivision = Subdivision {
    name: "Batticaloa",
    code: "LK-51",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-51",
};


pub const LK_52: Subdivision = Subdivision {
    name: "Ampara",
    code: "LK-52",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-52",
};


pub const LK_53: Subdivision = Subdivision {
    name: "Tirukŏṇamalai",
    code: "LK-53",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-53",
};


pub const LK_5: Subdivision = Subdivision {
    name: "Eastern Province",
    code: "LK-5",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-5",
};


pub const LK_61: Subdivision = Subdivision {
    name: "Kuruṇægala",
    code: "LK-61",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-61",
};


pub const LK_62: Subdivision = Subdivision {
    name: "Puttalam",
    code: "LK-62",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-62",
};


pub const LK_6: Subdivision = Subdivision {
    name: "North Western Province",
    code: "LK-6",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-6",
};


pub const LK_71: Subdivision = Subdivision {
    name: "Anuradhapura",
    code: "LK-71",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-71",
};


pub const LK_72: Subdivision = Subdivision {
    name: "Pŏlaṉṉaṛuvai",
    code: "LK-72",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-72",
};


pub const LK_7: Subdivision = Subdivision {
    name: "North Central Province",
    code: "LK-7",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-7",
};


pub const LK_81: Subdivision = Subdivision {
    name: "Badulla",
    code: "LK-81",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-81",
};


pub const LK_82: Subdivision = Subdivision {
    name: "Monaragala",
    code: "LK-82",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-82",
};


pub const LK_8: Subdivision = Subdivision {
    name: "Ūvā mākāṇam",
    code: "LK-8",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-8",
};


pub const LK_91: Subdivision = Subdivision {
    name: "Irattiṉapuri",
    code: "LK-91",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-91",
};


pub const LK_92: Subdivision = Subdivision {
    name: "Kægalla",
    code: "LK-92",
    subdivision_type: "District",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-92",
};


pub const LK_9: Subdivision = Subdivision {
    name: "Chappirakamuva mākāṇam",
    code: "LK-9",
    subdivision_type: "Province",
    country_name: "Sri Lanka",
    country_code: "LK",
    region_code: "LK-9",
};


pub const BL_BL: Subdivision = Subdivision {
    name: "St. Barthélemy",
    code: "BL-BL",
    subdivision_type: "Country",
    country_name: "St. Barthélemy",
    country_code: "BL",
    region_code: "BL-BL",
};


pub const SX_SX: Subdivision = Subdivision {
    name: "St. Maarten",
    code: "SX-SX",
    subdivision_type: "Country",
    country_name: "St. Maarten",
    country_code: "SX",
    region_code: "SX-SX",
};


pub const MF_MF: Subdivision = Subdivision {
    name: "St. Martin",
    code: "MF-MF",
    subdivision_type: "Country",
    country_name: "St. Martin",
    country_code: "MF",
    region_code: "MF-MF",
};


pub const PM_PM: Subdivision = Subdivision {
    name: "St. Pierre & Miquelon",
    code: "PM-PM",
    subdivision_type: "Country",
    country_name: "St. Pierre & Miquelon",
    country_code: "PM",
    region_code: "PM-PM",
};


pub const SD_DC: Subdivision = Subdivision {
    name: "Central Darfur",
    code: "SD-DC",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-DC",
};


pub const SD_DE: Subdivision = Subdivision {
    name: "East Darfur",
    code: "SD-DE",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-DE",
};


pub const SD_DN: Subdivision = Subdivision {
    name: "North Darfur",
    code: "SD-DN",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-DN",
};


pub const SD_DS: Subdivision = Subdivision {
    name: "Janūb Dārfūr",
    code: "SD-DS",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-DS",
};


pub const SD_DW: Subdivision = Subdivision {
    name: "Gharb Dārfūr",
    code: "SD-DW",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-DW",
};


pub const SD_GD: Subdivision = Subdivision {
    name: "Al Qaḑārif",
    code: "SD-GD",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-GD",
};


pub const SD_GZ: Subdivision = Subdivision {
    name: "Al Jazīrah",
    code: "SD-GZ",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-GZ",
};


pub const SD_KA: Subdivision = Subdivision {
    name: "Kassala",
    code: "SD-KA",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-KA",
};


pub const SD_KH: Subdivision = Subdivision {
    name: "Al Kharţūm",
    code: "SD-KH",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-KH",
};


pub const SD_KN: Subdivision = Subdivision {
    name: "North Kordofan",
    code: "SD-KN",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-KN",
};


pub const SD_KS: Subdivision = Subdivision {
    name: "Janūb Kurdufān",
    code: "SD-KS",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-KS",
};


pub const SD_NB: Subdivision = Subdivision {
    name: "An Nīl al Azraq",
    code: "SD-NB",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-NB",
};


pub const SD_NO: Subdivision = Subdivision {
    name: "Ash Shamālīyah",
    code: "SD-NO",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-NO",
};


pub const SD_NR: Subdivision = Subdivision {
    name: "An Nīl",
    code: "SD-NR",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-NR",
};


pub const SD_NW: Subdivision = Subdivision {
    name: "An Nīl al Abyaḑ",
    code: "SD-NW",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-NW",
};


pub const SD_RS: Subdivision = Subdivision {
    name: "Al Baḩr al Aḩmar",
    code: "SD-RS",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-RS",
};


pub const SD_SI: Subdivision = Subdivision {
    name: "Sennar",
    code: "SD-SI",
    subdivision_type: "State",
    country_name: "Sudan",
    country_code: "SD",
    region_code: "SD-SI",
};


pub const SR_BR: Subdivision = Subdivision {
    name: "Brokopondo",
    code: "SR-BR",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-BR",
};


pub const SR_CM: Subdivision = Subdivision {
    name: "Commewijne",
    code: "SR-CM",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-CM",
};


pub const SR_CR: Subdivision = Subdivision {
    name: "Coronie",
    code: "SR-CR",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-CR",
};


pub const SR_MA: Subdivision = Subdivision {
    name: "Marowijne",
    code: "SR-MA",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-MA",
};


pub const SR_NI: Subdivision = Subdivision {
    name: "Nickerie",
    code: "SR-NI",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-NI",
};


pub const SR_PM: Subdivision = Subdivision {
    name: "Paramaribo",
    code: "SR-PM",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-PM",
};


pub const SR_PR: Subdivision = Subdivision {
    name: "Para",
    code: "SR-PR",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-PR",
};


pub const SR_SA: Subdivision = Subdivision {
    name: "Saramacca",
    code: "SR-SA",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-SA",
};


pub const SR_SI: Subdivision = Subdivision {
    name: "Sipaliwini",
    code: "SR-SI",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-SI",
};


pub const SR_WA: Subdivision = Subdivision {
    name: "Wanica",
    code: "SR-WA",
    subdivision_type: "District",
    country_name: "Suriname",
    country_code: "SR",
    region_code: "SR-WA",
};


pub const SJ_SJ: Subdivision = Subdivision {
    name: "Svalbard & Jan Mayen Islands",
    code: "SJ-SJ",
    subdivision_type: "Country",
    country_name: "Svalbard & Jan Mayen Islands",
    country_code: "SJ",
    region_code: "SJ-SJ",
};


pub const SZ_HH: Subdivision = Subdivision {
    name: "Hhohho",
    code: "SZ-HH",
    subdivision_type: "District",
    country_name: "Swaziland",
    country_code: "SZ",
    region_code: "SZ-HH",
};


pub const SZ_LU: Subdivision = Subdivision {
    name: "Lubombo",
    code: "SZ-LU",
    subdivision_type: "District",
    country_name: "Swaziland",
    country_code: "SZ",
    region_code: "SZ-LU",
};


pub const SZ_MA: Subdivision = Subdivision {
    name: "Manzini",
    code: "SZ-MA",
    subdivision_type: "District",
    country_name: "Swaziland",
    country_code: "SZ",
    region_code: "SZ-MA",
};


pub const SZ_SH: Subdivision = Subdivision {
    name: "Shiselweni",
    code: "SZ-SH",
    subdivision_type: "District",
    country_name: "Swaziland",
    country_code: "SZ",
    region_code: "SZ-SH",
};


pub const SE_AB: Subdivision = Subdivision {
    name: "Stockholms län",
    code: "SE-AB",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-AB",
};


pub const SE_AC: Subdivision = Subdivision {
    name: "Västerbottens län",
    code: "SE-AC",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-AC",
};


pub const SE_BD: Subdivision = Subdivision {
    name: "Norrbottens län",
    code: "SE-BD",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-BD",
};


pub const SE_C: Subdivision = Subdivision {
    name: "Uppsala län",
    code: "SE-C",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-C",
};


pub const SE_D: Subdivision = Subdivision {
    name: "Södermanlands län",
    code: "SE-D",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-D",
};


pub const SE_E: Subdivision = Subdivision {
    name: "Östergötlands län",
    code: "SE-E",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-E",
};


pub const SE_F: Subdivision = Subdivision {
    name: "Jönköpings län",
    code: "SE-F",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-F",
};


pub const SE_G: Subdivision = Subdivision {
    name: "Kronobergs län",
    code: "SE-G",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-G",
};


pub const SE_H: Subdivision = Subdivision {
    name: "Kalmar län",
    code: "SE-H",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-H",
};


pub const SE_I: Subdivision = Subdivision {
    name: "Gotlands län",
    code: "SE-I",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-I",
};


pub const SE_K: Subdivision = Subdivision {
    name: "Blekinge län",
    code: "SE-K",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-K",
};


pub const SE_M: Subdivision = Subdivision {
    name: "Skåne län",
    code: "SE-M",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-M",
};


pub const SE_N: Subdivision = Subdivision {
    name: "Hallands län",
    code: "SE-N",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-N",
};


pub const SE_O: Subdivision = Subdivision {
    name: "Västra Götalands län",
    code: "SE-O",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-O",
};


pub const SE_S: Subdivision = Subdivision {
    name: "Värmlands län",
    code: "SE-S",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-S",
};


pub const SE_T: Subdivision = Subdivision {
    name: "Örebro län",
    code: "SE-T",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-T",
};


pub const SE_U: Subdivision = Subdivision {
    name: "Västmanlands län",
    code: "SE-U",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-U",
};


pub const SE_W: Subdivision = Subdivision {
    name: "Dalarnas län",
    code: "SE-W",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-W",
};


pub const SE_X: Subdivision = Subdivision {
    name: "Gävleborgs län",
    code: "SE-X",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-X",
};


pub const SE_Y: Subdivision = Subdivision {
    name: "Västernorrlands län",
    code: "SE-Y",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-Y",
};


pub const SE_Z: Subdivision = Subdivision {
    name: "Jämtlands län",
    code: "SE-Z",
    subdivision_type: "County",
    country_name: "Sweden",
    country_code: "SE",
    region_code: "SE-Z",
};


pub const CH_AG: Subdivision = Subdivision {
    name: "Aargau",
    code: "CH-AG",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-AG",
};


pub const CH_AI: Subdivision = Subdivision {
    name: "Appenzell Innerrhoden",
    code: "CH-AI",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-AI",
};


pub const CH_AR: Subdivision = Subdivision {
    name: "Appenzell Ausserrhoden",
    code: "CH-AR",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-AR",
};


pub const CH_BE: Subdivision = Subdivision {
    name: "Bern",
    code: "CH-BE",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-BE",
};


pub const CH_BL: Subdivision = Subdivision {
    name: "Basel-Landschaft",
    code: "CH-BL",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-BL",
};


pub const CH_BS: Subdivision = Subdivision {
    name: "Basel-Stadt",
    code: "CH-BS",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-BS",
};


pub const CH_FR: Subdivision = Subdivision {
    name: "Freiburg",
    code: "CH-FR",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-FR",
};


pub const CH_GE: Subdivision = Subdivision {
    name: "Genève",
    code: "CH-GE",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-GE",
};


pub const CH_GL: Subdivision = Subdivision {
    name: "Glarus",
    code: "CH-GL",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-GL",
};


pub const CH_GR: Subdivision = Subdivision {
    name: "Graubünden",
    code: "CH-GR",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-GR",
};


pub const CH_JU: Subdivision = Subdivision {
    name: "Jura",
    code: "CH-JU",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-JU",
};


pub const CH_LU: Subdivision = Subdivision {
    name: "Luzern",
    code: "CH-LU",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-LU",
};


pub const CH_NE: Subdivision = Subdivision {
    name: "Neuchâtel",
    code: "CH-NE",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-NE",
};


pub const CH_NW: Subdivision = Subdivision {
    name: "Nidwalden",
    code: "CH-NW",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-NW",
};


pub const CH_OW: Subdivision = Subdivision {
    name: "Obwalden",
    code: "CH-OW",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-OW",
};


pub const CH_SG: Subdivision = Subdivision {
    name: "Sankt Gallen",
    code: "CH-SG",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-SG",
};


pub const CH_SH: Subdivision = Subdivision {
    name: "Schaffhausen",
    code: "CH-SH",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-SH",
};


pub const CH_SO: Subdivision = Subdivision {
    name: "Solothurn",
    code: "CH-SO",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-SO",
};


pub const CH_SZ: Subdivision = Subdivision {
    name: "Schwyz",
    code: "CH-SZ",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-SZ",
};


pub const CH_TG: Subdivision = Subdivision {
    name: "Thurgau",
    code: "CH-TG",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-TG",
};


pub const CH_TI: Subdivision = Subdivision {
    name: "Ticino",
    code: "CH-TI",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-TI",
};


pub const CH_UR: Subdivision = Subdivision {
    name: "Uri",
    code: "CH-UR",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-UR",
};


pub const CH_VD: Subdivision = Subdivision {
    name: "Vaud",
    code: "CH-VD",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-VD",
};


pub const CH_VS: Subdivision = Subdivision {
    name: "Valais",
    code: "CH-VS",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-VS",
};


pub const CH_ZG: Subdivision = Subdivision {
    name: "Zug",
    code: "CH-ZG",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-ZG",
};


pub const CH_ZH: Subdivision = Subdivision {
    name: "Zürich",
    code: "CH-ZH",
    subdivision_type: "Canton",
    country_name: "Switzerland",
    country_code: "CH",
    region_code: "CH-ZH",
};


pub const SY_DI: Subdivision = Subdivision {
    name: "Dimashq",
    code: "SY-DI",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-DI",
};


pub const SY_DR: Subdivision = Subdivision {
    name: "Dar'ā",
    code: "SY-DR",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-DR",
};


pub const SY_DY: Subdivision = Subdivision {
    name: "Dayr az Zawr",
    code: "SY-DY",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-DY",
};


pub const SY_HA: Subdivision = Subdivision {
    name: "Al Ḩasakah",
    code: "SY-HA",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-HA",
};


pub const SY_HI: Subdivision = Subdivision {
    name: "Ḩimş",
    code: "SY-HI",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-HI",
};


pub const SY_HL: Subdivision = Subdivision {
    name: "Ḩalab",
    code: "SY-HL",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-HL",
};


pub const SY_HM: Subdivision = Subdivision {
    name: "Ḩamāh",
    code: "SY-HM",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-HM",
};


pub const SY_ID: Subdivision = Subdivision {
    name: "Idlib",
    code: "SY-ID",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-ID",
};


pub const SY_LA: Subdivision = Subdivision {
    name: "Al Lādhiqīyah",
    code: "SY-LA",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-LA",
};


pub const SY_QU: Subdivision = Subdivision {
    name: "Al Qunayţirah",
    code: "SY-QU",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-QU",
};


pub const SY_RA: Subdivision = Subdivision {
    name: "Ar Raqqah",
    code: "SY-RA",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-RA",
};


pub const SY_RD: Subdivision = Subdivision {
    name: "Rīf Dimashq",
    code: "SY-RD",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-RD",
};


pub const SY_SU: Subdivision = Subdivision {
    name: "As Suwaydā'",
    code: "SY-SU",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-SU",
};


pub const SY_TA: Subdivision = Subdivision {
    name: "Ţarţūs",
    code: "SY-TA",
    subdivision_type: "Province",
    country_name: "Syria",
    country_code: "SY",
    region_code: "SY-TA",
};


pub const TW_CHA: Subdivision = Subdivision {
    name: "Changhua",
    code: "TW-CHA",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-CHA",
};


pub const TW_CYI: Subdivision = Subdivision {
    name: "Chiayi",
    code: "TW-CYI",
    subdivision_type: "Municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-CYI",
};


pub const TW_CYQ: Subdivision = Subdivision {
    name: "Chiayi",
    code: "TW-CYQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-CYQ",
};


pub const TW_HSQ: Subdivision = Subdivision {
    name: "Hsinchu",
    code: "TW-HSQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-HSQ",
};


pub const TW_HSZ: Subdivision = Subdivision {
    name: "Hsinchu",
    code: "TW-HSZ",
    subdivision_type: "Municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-HSZ",
};


pub const TW_HUA: Subdivision = Subdivision {
    name: "Hualien",
    code: "TW-HUA",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-HUA",
};


pub const TW_ILA: Subdivision = Subdivision {
    name: "Ilan",
    code: "TW-ILA",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-ILA",
};


pub const TW_KEE: Subdivision = Subdivision {
    name: "Keelung",
    code: "TW-KEE",
    subdivision_type: "Municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-KEE",
};


pub const TW_KHH: Subdivision = Subdivision {
    name: "Kaohsiung",
    code: "TW-KHH",
    subdivision_type: "Special municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-KHH",
};


pub const TW_KHQ: Subdivision = Subdivision {
    name: "Kaohsiung",
    code: "TW-KHQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-KHQ",
};


pub const TW_MIA: Subdivision = Subdivision {
    name: "Miaoli",
    code: "TW-MIA",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-MIA",
};


pub const TW_NAN: Subdivision = Subdivision {
    name: "Nantou",
    code: "TW-NAN",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-NAN",
};


pub const TW_PEN: Subdivision = Subdivision {
    name: "Penghu",
    code: "TW-PEN",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-PEN",
};


pub const TW_PIF: Subdivision = Subdivision {
    name: "Pingtung",
    code: "TW-PIF",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-PIF",
};


pub const TW_TAO: Subdivision = Subdivision {
    name: "Taoyuan",
    code: "TW-TAO",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TAO",
};


pub const TW_TNN: Subdivision = Subdivision {
    name: "Tainan",
    code: "TW-TNN",
    subdivision_type: "Municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TNN",
};


pub const TW_TNQ: Subdivision = Subdivision {
    name: "Tainan",
    code: "TW-TNQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TNQ",
};


pub const TW_TPE: Subdivision = Subdivision {
    name: "Taipei",
    code: "TW-TPE",
    subdivision_type: "Special municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TPE",
};


pub const TW_TPQ: Subdivision = Subdivision {
    name: "Taipei",
    code: "TW-TPQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TPQ",
};


pub const TW_TTT: Subdivision = Subdivision {
    name: "Taitung",
    code: "TW-TTT",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TTT",
};


pub const TW_TXG: Subdivision = Subdivision {
    name: "Taichung",
    code: "TW-TXG",
    subdivision_type: "Municipality",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TXG",
};


pub const TW_TXQ: Subdivision = Subdivision {
    name: "Taichung",
    code: "TW-TXQ",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-TXQ",
};


pub const TW_YUN: Subdivision = Subdivision {
    name: "Yunlin",
    code: "TW-YUN",
    subdivision_type: "District",
    country_name: "Taiwan",
    country_code: "TW",
    region_code: "TW-YUN",
};


pub const TJ_DU: Subdivision = Subdivision {
    name: "Dushanbe",
    code: "TJ-DU",
    subdivision_type: "Capital territory",
    country_name: "Tajikistan",
    country_code: "TJ",
    region_code: "TJ-DU",
};


pub const TJ_GB: Subdivision = Subdivision {
    name: "Kŭhistoni Badakhshon",
    code: "TJ-GB",
    subdivision_type: "Autonomous region",
    country_name: "Tajikistan",
    country_code: "TJ",
    region_code: "TJ-GB",
};


pub const TJ_KT: Subdivision = Subdivision {
    name: "Khatlon",
    code: "TJ-KT",
    subdivision_type: "Region",
    country_name: "Tajikistan",
    country_code: "TJ",
    region_code: "TJ-KT",
};


pub const TJ_SU: Subdivision = Subdivision {
    name: "Sughd",
    code: "TJ-SU",
    subdivision_type: "Region",
    country_name: "Tajikistan",
    country_code: "TJ",
    region_code: "TJ-SU",
};


pub const TZ_01: Subdivision = Subdivision {
    name: "Arusha",
    code: "TZ-01",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-01",
};


pub const TZ_02: Subdivision = Subdivision {
    name: "Dar es Salaam",
    code: "TZ-02",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-02",
};


pub const TZ_03: Subdivision = Subdivision {
    name: "Dodoma",
    code: "TZ-03",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-03",
};


pub const TZ_04: Subdivision = Subdivision {
    name: "Iringa",
    code: "TZ-04",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-04",
};


pub const TZ_05: Subdivision = Subdivision {
    name: "Kagera",
    code: "TZ-05",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-05",
};


pub const TZ_06: Subdivision = Subdivision {
    name: "Kaskazini Pemba",
    code: "TZ-06",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-06",
};


pub const TZ_07: Subdivision = Subdivision {
    name: "Kaskazini Unguja",
    code: "TZ-07",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-07",
};


pub const TZ_08: Subdivision = Subdivision {
    name: "Kigoma",
    code: "TZ-08",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-08",
};


pub const TZ_09: Subdivision = Subdivision {
    name: "Kilimanjaro",
    code: "TZ-09",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-09",
};


pub const TZ_10: Subdivision = Subdivision {
    name: "Kusini Pemba",
    code: "TZ-10",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-10",
};


pub const TZ_11: Subdivision = Subdivision {
    name: "Kusini Unguja",
    code: "TZ-11",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-11",
};


pub const TZ_12: Subdivision = Subdivision {
    name: "Lindi",
    code: "TZ-12",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-12",
};


pub const TZ_13: Subdivision = Subdivision {
    name: "Mara",
    code: "TZ-13",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-13",
};


pub const TZ_14: Subdivision = Subdivision {
    name: "Mbeya",
    code: "TZ-14",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-14",
};


pub const TZ_15: Subdivision = Subdivision {
    name: "Mjini Magharibi",
    code: "TZ-15",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-15",
};


pub const TZ_16: Subdivision = Subdivision {
    name: "Morogoro",
    code: "TZ-16",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-16",
};


pub const TZ_17: Subdivision = Subdivision {
    name: "Mtwara",
    code: "TZ-17",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-17",
};


pub const TZ_18: Subdivision = Subdivision {
    name: "Mwanza",
    code: "TZ-18",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-18",
};


pub const TZ_19: Subdivision = Subdivision {
    name: "Coast",
    code: "TZ-19",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-19",
};


pub const TZ_20: Subdivision = Subdivision {
    name: "Rukwa",
    code: "TZ-20",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-20",
};


pub const TZ_21: Subdivision = Subdivision {
    name: "Ruvuma",
    code: "TZ-21",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-21",
};


pub const TZ_22: Subdivision = Subdivision {
    name: "Shinyanga",
    code: "TZ-22",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-22",
};


pub const TZ_23: Subdivision = Subdivision {
    name: "Singida",
    code: "TZ-23",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-23",
};


pub const TZ_24: Subdivision = Subdivision {
    name: "Tabora",
    code: "TZ-24",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-24",
};


pub const TZ_25: Subdivision = Subdivision {
    name: "Tanga",
    code: "TZ-25",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-25",
};


pub const TZ_26: Subdivision = Subdivision {
    name: "Manyara",
    code: "TZ-26",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-26",
};


pub const TZ_27: Subdivision = Subdivision {
    name: "Geita",
    code: "TZ-27",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-27",
};


pub const TZ_28: Subdivision = Subdivision {
    name: "Katavi",
    code: "TZ-28",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-28",
};


pub const TZ_29: Subdivision = Subdivision {
    name: "Njombe",
    code: "TZ-29",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-29",
};


pub const TZ_30: Subdivision = Subdivision {
    name: "Simiyu",
    code: "TZ-30",
    subdivision_type: "Region",
    country_name: "Tanzania",
    country_code: "TZ",
    region_code: "TZ-30",
};


pub const TH_10: Subdivision = Subdivision {
    name: "Krung Thep Maha Nakhon",
    code: "TH-10",
    subdivision_type: "Metropolitan administration",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-10",
};


pub const TH_11: Subdivision = Subdivision {
    name: "Samut Prakan",
    code: "TH-11",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-11",
};


pub const TH_12: Subdivision = Subdivision {
    name: "Nonthaburi",
    code: "TH-12",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-12",
};


pub const TH_13: Subdivision = Subdivision {
    name: "Pathum Thani",
    code: "TH-13",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-13",
};


pub const TH_14: Subdivision = Subdivision {
    name: "Phra Nakhon Si Ayutthaya",
    code: "TH-14",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-14",
};


pub const TH_15: Subdivision = Subdivision {
    name: "Ang Thong",
    code: "TH-15",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-15",
};


pub const TH_16: Subdivision = Subdivision {
    name: "Lop Buri",
    code: "TH-16",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-16",
};


pub const TH_17: Subdivision = Subdivision {
    name: "Sing Buri",
    code: "TH-17",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-17",
};


pub const TH_18: Subdivision = Subdivision {
    name: "Chai Nat",
    code: "TH-18",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-18",
};


pub const TH_19: Subdivision = Subdivision {
    name: "Saraburi",
    code: "TH-19",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-19",
};


pub const TH_20: Subdivision = Subdivision {
    name: "Chon Buri",
    code: "TH-20",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-20",
};


pub const TH_21: Subdivision = Subdivision {
    name: "Rayong",
    code: "TH-21",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-21",
};


pub const TH_22: Subdivision = Subdivision {
    name: "Chanthaburi",
    code: "TH-22",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-22",
};


pub const TH_23: Subdivision = Subdivision {
    name: "Trat",
    code: "TH-23",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-23",
};


pub const TH_24: Subdivision = Subdivision {
    name: "Chachoengsao",
    code: "TH-24",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-24",
};


pub const TH_25: Subdivision = Subdivision {
    name: "Prachin Buri",
    code: "TH-25",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-25",
};


pub const TH_26: Subdivision = Subdivision {
    name: "Nakhon Nayok",
    code: "TH-26",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-26",
};


pub const TH_27: Subdivision = Subdivision {
    name: "Sa Kaeo",
    code: "TH-27",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-27",
};


pub const TH_30: Subdivision = Subdivision {
    name: "Nakhon Ratchasima",
    code: "TH-30",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-30",
};


pub const TH_31: Subdivision = Subdivision {
    name: "Buri Ram",
    code: "TH-31",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-31",
};


pub const TH_32: Subdivision = Subdivision {
    name: "Surin",
    code: "TH-32",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-32",
};


pub const TH_33: Subdivision = Subdivision {
    name: "Si Sa Ket",
    code: "TH-33",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-33",
};


pub const TH_34: Subdivision = Subdivision {
    name: "Ubon Ratchathani",
    code: "TH-34",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-34",
};


pub const TH_35: Subdivision = Subdivision {
    name: "Yasothon",
    code: "TH-35",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-35",
};


pub const TH_36: Subdivision = Subdivision {
    name: "Chaiyaphum",
    code: "TH-36",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-36",
};


pub const TH_37: Subdivision = Subdivision {
    name: "Amnat Charoen",
    code: "TH-37",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-37",
};


pub const TH_38: Subdivision = Subdivision {
    name: "Bueng Kan",
    code: "TH-38",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-38",
};


pub const TH_39: Subdivision = Subdivision {
    name: "Nong Bua Lam Phu",
    code: "TH-39",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-39",
};


pub const TH_40: Subdivision = Subdivision {
    name: "Khon Kaen",
    code: "TH-40",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-40",
};


pub const TH_41: Subdivision = Subdivision {
    name: "Udon Thani",
    code: "TH-41",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-41",
};


pub const TH_42: Subdivision = Subdivision {
    name: "Loei",
    code: "TH-42",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-42",
};


pub const TH_43: Subdivision = Subdivision {
    name: "Nong Khai",
    code: "TH-43",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-43",
};


pub const TH_44: Subdivision = Subdivision {
    name: "Maha Sarakham",
    code: "TH-44",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-44",
};


pub const TH_45: Subdivision = Subdivision {
    name: "Roi Et",
    code: "TH-45",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-45",
};


pub const TH_46: Subdivision = Subdivision {
    name: "Kalasin",
    code: "TH-46",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-46",
};


pub const TH_47: Subdivision = Subdivision {
    name: "Sakon Nakhon",
    code: "TH-47",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-47",
};


pub const TH_48: Subdivision = Subdivision {
    name: "Nakhon Phanom",
    code: "TH-48",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-48",
};


pub const TH_49: Subdivision = Subdivision {
    name: "Mukdahan",
    code: "TH-49",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-49",
};


pub const TH_50: Subdivision = Subdivision {
    name: "Chiang Mai",
    code: "TH-50",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-50",
};


pub const TH_51: Subdivision = Subdivision {
    name: "Lamphun",
    code: "TH-51",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-51",
};


pub const TH_52: Subdivision = Subdivision {
    name: "Lampang",
    code: "TH-52",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-52",
};


pub const TH_53: Subdivision = Subdivision {
    name: "Uttaradit",
    code: "TH-53",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-53",
};


pub const TH_54: Subdivision = Subdivision {
    name: "Phrae",
    code: "TH-54",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-54",
};


pub const TH_55: Subdivision = Subdivision {
    name: "Nan",
    code: "TH-55",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-55",
};


pub const TH_56: Subdivision = Subdivision {
    name: "Phayao",
    code: "TH-56",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-56",
};


pub const TH_57: Subdivision = Subdivision {
    name: "Chiang Rai",
    code: "TH-57",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-57",
};


pub const TH_58: Subdivision = Subdivision {
    name: "Mae Hong Son",
    code: "TH-58",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-58",
};


pub const TH_60: Subdivision = Subdivision {
    name: "Nakhon Sawan",
    code: "TH-60",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-60",
};


pub const TH_61: Subdivision = Subdivision {
    name: "Uthai Thani",
    code: "TH-61",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-61",
};


pub const TH_62: Subdivision = Subdivision {
    name: "Kamphaeng Phet",
    code: "TH-62",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-62",
};


pub const TH_63: Subdivision = Subdivision {
    name: "Tak",
    code: "TH-63",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-63",
};


pub const TH_64: Subdivision = Subdivision {
    name: "Sukhothai",
    code: "TH-64",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-64",
};


pub const TH_65: Subdivision = Subdivision {
    name: "Phitsanulok",
    code: "TH-65",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-65",
};


pub const TH_66: Subdivision = Subdivision {
    name: "Phichit",
    code: "TH-66",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-66",
};


pub const TH_67: Subdivision = Subdivision {
    name: "Phetchabun",
    code: "TH-67",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-67",
};


pub const TH_70: Subdivision = Subdivision {
    name: "Ratchaburi",
    code: "TH-70",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-70",
};


pub const TH_71: Subdivision = Subdivision {
    name: "Kanchanaburi",
    code: "TH-71",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-71",
};


pub const TH_72: Subdivision = Subdivision {
    name: "Suphan Buri",
    code: "TH-72",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-72",
};


pub const TH_73: Subdivision = Subdivision {
    name: "Nakhon Pathom",
    code: "TH-73",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-73",
};


pub const TH_74: Subdivision = Subdivision {
    name: "Samut Sakhon",
    code: "TH-74",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-74",
};


pub const TH_75: Subdivision = Subdivision {
    name: "Samut Songkhram",
    code: "TH-75",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-75",
};


pub const TH_76: Subdivision = Subdivision {
    name: "Phetchaburi",
    code: "TH-76",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-76",
};


pub const TH_77: Subdivision = Subdivision {
    name: "Prachuap Khiri Khan",
    code: "TH-77",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-77",
};


pub const TH_80: Subdivision = Subdivision {
    name: "Nakhon Si Thammarat",
    code: "TH-80",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-80",
};


pub const TH_81: Subdivision = Subdivision {
    name: "Krabi",
    code: "TH-81",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-81",
};


pub const TH_82: Subdivision = Subdivision {
    name: "Phangnga",
    code: "TH-82",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-82",
};


pub const TH_83: Subdivision = Subdivision {
    name: "Phuket",
    code: "TH-83",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-83",
};


pub const TH_84: Subdivision = Subdivision {
    name: "Surat Thani",
    code: "TH-84",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-84",
};


pub const TH_85: Subdivision = Subdivision {
    name: "Ranong",
    code: "TH-85",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-85",
};


pub const TH_86: Subdivision = Subdivision {
    name: "Chumphon",
    code: "TH-86",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-86",
};


pub const TH_90: Subdivision = Subdivision {
    name: "Songkhla",
    code: "TH-90",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-90",
};


pub const TH_91: Subdivision = Subdivision {
    name: "Satun",
    code: "TH-91",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-91",
};


pub const TH_92: Subdivision = Subdivision {
    name: "Trang",
    code: "TH-92",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-92",
};


pub const TH_93: Subdivision = Subdivision {
    name: "Phatthalung",
    code: "TH-93",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-93",
};


pub const TH_94: Subdivision = Subdivision {
    name: "Pattani",
    code: "TH-94",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-94",
};


pub const TH_95: Subdivision = Subdivision {
    name: "Yala",
    code: "TH-95",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-95",
};


pub const TH_96: Subdivision = Subdivision {
    name: "Narathiwat",
    code: "TH-96",
    subdivision_type: "Province",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-96",
};


pub const TH_S: Subdivision = Subdivision {
    name: "Phatthaya",
    code: "TH-S",
    subdivision_type: "Special administrative city",
    country_name: "Thailand",
    country_code: "TH",
    region_code: "TH-S",
};


pub const TL_AL: Subdivision = Subdivision {
    name: "Aileu",
    code: "TL-AL",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-AL",
};


pub const TL_AN: Subdivision = Subdivision {
    name: "Ainaro",
    code: "TL-AN",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-AN",
};


pub const TL_BA: Subdivision = Subdivision {
    name: "Baucau",
    code: "TL-BA",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-BA",
};


pub const TL_BO: Subdivision = Subdivision {
    name: "Bobonaro",
    code: "TL-BO",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-BO",
};


pub const TL_CO: Subdivision = Subdivision {
    name: "Cova Lima",
    code: "TL-CO",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-CO",
};


pub const TL_DI: Subdivision = Subdivision {
    name: "Díli",
    code: "TL-DI",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-DI",
};


pub const TL_ER: Subdivision = Subdivision {
    name: "Ermera",
    code: "TL-ER",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-ER",
};


pub const TL_LA: Subdivision = Subdivision {
    name: "Lautém",
    code: "TL-LA",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-LA",
};


pub const TL_LI: Subdivision = Subdivision {
    name: "Likisá",
    code: "TL-LI",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-LI",
};


pub const TL_MF: Subdivision = Subdivision {
    name: "Manufahi",
    code: "TL-MF",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-MF",
};


pub const TL_MT: Subdivision = Subdivision {
    name: "Manatuto",
    code: "TL-MT",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-MT",
};


pub const TL_OE: Subdivision = Subdivision {
    name: "Oecussi",
    code: "TL-OE",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-OE",
};


pub const TL_VI: Subdivision = Subdivision {
    name: "Vikeke",
    code: "TL-VI",
    subdivision_type: "District",
    country_name: "East Timor",
    country_code: "TL",
    region_code: "TL-VI",
};


pub const TG_C: Subdivision = Subdivision {
    name: "Centre",
    code: "TG-C",
    subdivision_type: "Region",
    country_name: "Togo",
    country_code: "TG",
    region_code: "TG-C",
};


pub const TG_K: Subdivision = Subdivision {
    name: "Kara",
    code: "TG-K",
    subdivision_type: "Region",
    country_name: "Togo",
    country_code: "TG",
    region_code: "TG-K",
};


pub const TG_M: Subdivision = Subdivision {
    name: "Maritime",
    code: "TG-M",
    subdivision_type: "Region",
    country_name: "Togo",
    country_code: "TG",
    region_code: "TG-M",
};


pub const TG_P: Subdivision = Subdivision {
    name: "Plateaux",
    code: "TG-P",
    subdivision_type: "Region",
    country_name: "Togo",
    country_code: "TG",
    region_code: "TG-P",
};


pub const TG_S: Subdivision = Subdivision {
    name: "Savannes",
    code: "TG-S",
    subdivision_type: "Region",
    country_name: "Togo",
    country_code: "TG",
    region_code: "TG-S",
};


pub const TK_TK: Subdivision = Subdivision {
    name: "Tokelau",
    code: "TK-TK",
    subdivision_type: "Country",
    country_name: "Tokelau",
    country_code: "TK",
    region_code: "TK-TK",
};


pub const TO_01: Subdivision = Subdivision {
    name: "'Eua",
    code: "TO-01",
    subdivision_type: "Division",
    country_name: "Tonga",
    country_code: "TO",
    region_code: "TO-01",
};


pub const TO_02: Subdivision = Subdivision {
    name: "Ha'apai",
    code: "TO-02",
    subdivision_type: "Division",
    country_name: "Tonga",
    country_code: "TO",
    region_code: "TO-02",
};


pub const TO_03: Subdivision = Subdivision {
    name: "Niuas",
    code: "TO-03",
    subdivision_type: "Division",
    country_name: "Tonga",
    country_code: "TO",
    region_code: "TO-03",
};


pub const TO_04: Subdivision = Subdivision {
    name: "Tongatapu",
    code: "TO-04",
    subdivision_type: "Division",
    country_name: "Tonga",
    country_code: "TO",
    region_code: "TO-04",
};


pub const TO_05: Subdivision = Subdivision {
    name: "Vava'u",
    code: "TO-05",
    subdivision_type: "Division",
    country_name: "Tonga",
    country_code: "TO",
    region_code: "TO-05",
};


pub const TT_ARI: Subdivision = Subdivision {
    name: "Arima",
    code: "TT-ARI",
    subdivision_type: "Municipality",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-ARI",
};


pub const TT_CHA: Subdivision = Subdivision {
    name: "Chaguanas",
    code: "TT-CHA",
    subdivision_type: "Municipality",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-CHA",
};


pub const TT_CTT: Subdivision = Subdivision {
    name: "Couva-Tabaquite-Talparo",
    code: "TT-CTT",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-CTT",
};


pub const TT_DMN: Subdivision = Subdivision {
    name: "Diego Martin",
    code: "TT-DMN",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-DMN",
};


pub const TT_ETO: Subdivision = Subdivision {
    name: "Eastern Tobago",
    code: "TT-ETO",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-ETO",
};


pub const TT_PED: Subdivision = Subdivision {
    name: "Penal-Debe",
    code: "TT-PED",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-PED",
};


pub const TT_POS: Subdivision = Subdivision {
    name: "Port of Spain",
    code: "TT-POS",
    subdivision_type: "Municipality",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-POS",
};


pub const TT_PRT: Subdivision = Subdivision {
    name: "Princes Town",
    code: "TT-PRT",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-PRT",
};


pub const TT_PTF: Subdivision = Subdivision {
    name: "Point Fortin",
    code: "TT-PTF",
    subdivision_type: "Municipality",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-PTF",
};


pub const TT_RCM: Subdivision = Subdivision {
    name: "Rio Claro-Mayaro",
    code: "TT-RCM",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-RCM",
};


pub const TT_SFO: Subdivision = Subdivision {
    name: "San Fernando",
    code: "TT-SFO",
    subdivision_type: "Municipality",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-SFO",
};


pub const TT_SGE: Subdivision = Subdivision {
    name: "Sangre Grande",
    code: "TT-SGE",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-SGE",
};


pub const TT_SIP: Subdivision = Subdivision {
    name: "Siparia",
    code: "TT-SIP",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-SIP",
};


pub const TT_SJL: Subdivision = Subdivision {
    name: "San Juan-Laventille",
    code: "TT-SJL",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-SJL",
};


pub const TT_TUP: Subdivision = Subdivision {
    name: "Tunapuna-Piarco",
    code: "TT-TUP",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-TUP",
};


pub const TT_WTO: Subdivision = Subdivision {
    name: "Western Tobago",
    code: "TT-WTO",
    subdivision_type: "Region",
    country_name: "Trinidad and Tobago",
    country_code: "TT",
    region_code: "TT-WTO",
};


pub const TN_11: Subdivision = Subdivision {
    name: "Tunis",
    code: "TN-11",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-11",
};


pub const TN_12: Subdivision = Subdivision {
    name: "Ariana",
    code: "TN-12",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-12",
};


pub const TN_13: Subdivision = Subdivision {
    name: "Ben Arous",
    code: "TN-13",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-13",
};


pub const TN_14: Subdivision = Subdivision {
    name: "La Manouba",
    code: "TN-14",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-14",
};


pub const TN_21: Subdivision = Subdivision {
    name: "Nabeul",
    code: "TN-21",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-21",
};


pub const TN_22: Subdivision = Subdivision {
    name: "Zaghouan",
    code: "TN-22",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-22",
};


pub const TN_23: Subdivision = Subdivision {
    name: "Bizerte",
    code: "TN-23",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-23",
};


pub const TN_31: Subdivision = Subdivision {
    name: "Béja",
    code: "TN-31",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-31",
};


pub const TN_32: Subdivision = Subdivision {
    name: "Jendouba",
    code: "TN-32",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-32",
};


pub const TN_33: Subdivision = Subdivision {
    name: "Le Kef",
    code: "TN-33",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-33",
};


pub const TN_34: Subdivision = Subdivision {
    name: "Siliana",
    code: "TN-34",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-34",
};


pub const TN_41: Subdivision = Subdivision {
    name: "Kairouan",
    code: "TN-41",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-41",
};


pub const TN_42: Subdivision = Subdivision {
    name: "Kasserine",
    code: "TN-42",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-42",
};


pub const TN_43: Subdivision = Subdivision {
    name: "Sidi Bouzid",
    code: "TN-43",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-43",
};


pub const TN_51: Subdivision = Subdivision {
    name: "Sousse",
    code: "TN-51",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-51",
};


pub const TN_52: Subdivision = Subdivision {
    name: "Monastir",
    code: "TN-52",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-52",
};


pub const TN_53: Subdivision = Subdivision {
    name: "Mahdia",
    code: "TN-53",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-53",
};


pub const TN_61: Subdivision = Subdivision {
    name: "Sfax",
    code: "TN-61",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-61",
};


pub const TN_71: Subdivision = Subdivision {
    name: "Gafsa",
    code: "TN-71",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-71",
};


pub const TN_72: Subdivision = Subdivision {
    name: "Tozeur",
    code: "TN-72",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-72",
};


pub const TN_73: Subdivision = Subdivision {
    name: "Kebili",
    code: "TN-73",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-73",
};


pub const TN_81: Subdivision = Subdivision {
    name: "Gabès",
    code: "TN-81",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-81",
};


pub const TN_82: Subdivision = Subdivision {
    name: "Medenine",
    code: "TN-82",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-82",
};


pub const TN_83: Subdivision = Subdivision {
    name: "Tataouine",
    code: "TN-83",
    subdivision_type: "Governorate",
    country_name: "Tunisia",
    country_code: "TN",
    region_code: "TN-83",
};


pub const TR_01: Subdivision = Subdivision {
    name: "Adana",
    code: "TR-01",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-01",
};


pub const TR_02: Subdivision = Subdivision {
    name: "Adıyaman",
    code: "TR-02",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-02",
};


pub const TR_03: Subdivision = Subdivision {
    name: "Afyonkarahisar",
    code: "TR-03",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-03",
};


pub const TR_04: Subdivision = Subdivision {
    name: "Ağrı",
    code: "TR-04",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-04",
};


pub const TR_05: Subdivision = Subdivision {
    name: "Amasya",
    code: "TR-05",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-05",
};


pub const TR_06: Subdivision = Subdivision {
    name: "Ankara",
    code: "TR-06",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-06",
};


pub const TR_07: Subdivision = Subdivision {
    name: "Antalya",
    code: "TR-07",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-07",
};


pub const TR_08: Subdivision = Subdivision {
    name: "Artvin",
    code: "TR-08",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-08",
};


pub const TR_09: Subdivision = Subdivision {
    name: "Aydın",
    code: "TR-09",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-09",
};


pub const TR_10: Subdivision = Subdivision {
    name: "Balıkesir",
    code: "TR-10",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-10",
};


pub const TR_11: Subdivision = Subdivision {
    name: "Bilecik",
    code: "TR-11",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-11",
};


pub const TR_12: Subdivision = Subdivision {
    name: "Bingöl",
    code: "TR-12",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-12",
};


pub const TR_13: Subdivision = Subdivision {
    name: "Bitlis",
    code: "TR-13",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-13",
};


pub const TR_14: Subdivision = Subdivision {
    name: "Bolu",
    code: "TR-14",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-14",
};


pub const TR_15: Subdivision = Subdivision {
    name: "Burdur",
    code: "TR-15",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-15",
};


pub const TR_16: Subdivision = Subdivision {
    name: "Bursa",
    code: "TR-16",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-16",
};


pub const TR_17: Subdivision = Subdivision {
    name: "Çanakkale",
    code: "TR-17",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-17",
};


pub const TR_18: Subdivision = Subdivision {
    name: "Çankırı",
    code: "TR-18",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-18",
};


pub const TR_19: Subdivision = Subdivision {
    name: "Çorum",
    code: "TR-19",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-19",
};


pub const TR_20: Subdivision = Subdivision {
    name: "Denizli",
    code: "TR-20",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-20",
};


pub const TR_21: Subdivision = Subdivision {
    name: "Diyarbakır",
    code: "TR-21",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-21",
};


pub const TR_22: Subdivision = Subdivision {
    name: "Edirne",
    code: "TR-22",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-22",
};


pub const TR_23: Subdivision = Subdivision {
    name: "Elazığ",
    code: "TR-23",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-23",
};


pub const TR_24: Subdivision = Subdivision {
    name: "Erzincan",
    code: "TR-24",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-24",
};


pub const TR_25: Subdivision = Subdivision {
    name: "Erzurum",
    code: "TR-25",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-25",
};


pub const TR_26: Subdivision = Subdivision {
    name: "Eskişehir",
    code: "TR-26",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-26",
};


pub const TR_27: Subdivision = Subdivision {
    name: "Gaziantep",
    code: "TR-27",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-27",
};


pub const TR_28: Subdivision = Subdivision {
    name: "Giresun",
    code: "TR-28",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-28",
};


pub const TR_29: Subdivision = Subdivision {
    name: "Gümüşhane",
    code: "TR-29",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-29",
};


pub const TR_30: Subdivision = Subdivision {
    name: "Hakkâri",
    code: "TR-30",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-30",
};


pub const TR_31: Subdivision = Subdivision {
    name: "Hatay",
    code: "TR-31",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-31",
};


pub const TR_32: Subdivision = Subdivision {
    name: "Isparta",
    code: "TR-32",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-32",
};


pub const TR_33: Subdivision = Subdivision {
    name: "Mersin",
    code: "TR-33",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-33",
};


pub const TR_34: Subdivision = Subdivision {
    name: "İstanbul",
    code: "TR-34",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-34",
};


pub const TR_35: Subdivision = Subdivision {
    name: "İzmir",
    code: "TR-35",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-35",
};


pub const TR_36: Subdivision = Subdivision {
    name: "Kars",
    code: "TR-36",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-36",
};


pub const TR_37: Subdivision = Subdivision {
    name: "Kastamonu",
    code: "TR-37",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-37",
};


pub const TR_38: Subdivision = Subdivision {
    name: "Kayseri",
    code: "TR-38",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-38",
};


pub const TR_39: Subdivision = Subdivision {
    name: "Kırklareli",
    code: "TR-39",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-39",
};


pub const TR_40: Subdivision = Subdivision {
    name: "Kırşehir",
    code: "TR-40",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-40",
};


pub const TR_41: Subdivision = Subdivision {
    name: "Kocaeli",
    code: "TR-41",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-41",
};


pub const TR_42: Subdivision = Subdivision {
    name: "Konya",
    code: "TR-42",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-42",
};


pub const TR_43: Subdivision = Subdivision {
    name: "Kütahya",
    code: "TR-43",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-43",
};


pub const TR_44: Subdivision = Subdivision {
    name: "Malatya",
    code: "TR-44",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-44",
};


pub const TR_45: Subdivision = Subdivision {
    name: "Manisa",
    code: "TR-45",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-45",
};


pub const TR_46: Subdivision = Subdivision {
    name: "Kahramanmaraş",
    code: "TR-46",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-46",
};


pub const TR_47: Subdivision = Subdivision {
    name: "Mardin",
    code: "TR-47",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-47",
};


pub const TR_48: Subdivision = Subdivision {
    name: "Muğla",
    code: "TR-48",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-48",
};


pub const TR_49: Subdivision = Subdivision {
    name: "Muş",
    code: "TR-49",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-49",
};


pub const TR_50: Subdivision = Subdivision {
    name: "Nevşehir",
    code: "TR-50",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-50",
};


pub const TR_51: Subdivision = Subdivision {
    name: "Niğde",
    code: "TR-51",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-51",
};


pub const TR_52: Subdivision = Subdivision {
    name: "Ordu",
    code: "TR-52",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-52",
};


pub const TR_53: Subdivision = Subdivision {
    name: "Rize",
    code: "TR-53",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-53",
};


pub const TR_54: Subdivision = Subdivision {
    name: "Sakarya",
    code: "TR-54",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-54",
};


pub const TR_55: Subdivision = Subdivision {
    name: "Samsun",
    code: "TR-55",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-55",
};


pub const TR_56: Subdivision = Subdivision {
    name: "Siirt",
    code: "TR-56",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-56",
};


pub const TR_57: Subdivision = Subdivision {
    name: "Sinop",
    code: "TR-57",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-57",
};


pub const TR_58: Subdivision = Subdivision {
    name: "Sivas",
    code: "TR-58",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-58",
};


pub const TR_59: Subdivision = Subdivision {
    name: "Tekirdağ",
    code: "TR-59",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-59",
};


pub const TR_60: Subdivision = Subdivision {
    name: "Tokat",
    code: "TR-60",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-60",
};


pub const TR_61: Subdivision = Subdivision {
    name: "Trabzon",
    code: "TR-61",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-61",
};


pub const TR_62: Subdivision = Subdivision {
    name: "Tunceli",
    code: "TR-62",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-62",
};


pub const TR_63: Subdivision = Subdivision {
    name: "Şanlıurfa",
    code: "TR-63",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-63",
};


pub const TR_64: Subdivision = Subdivision {
    name: "Uşak",
    code: "TR-64",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-64",
};


pub const TR_65: Subdivision = Subdivision {
    name: "Van",
    code: "TR-65",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-65",
};


pub const TR_66: Subdivision = Subdivision {
    name: "Yozgat",
    code: "TR-66",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-66",
};


pub const TR_67: Subdivision = Subdivision {
    name: "Zonguldak",
    code: "TR-67",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-67",
};


pub const TR_68: Subdivision = Subdivision {
    name: "Aksaray",
    code: "TR-68",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-68",
};


pub const TR_69: Subdivision = Subdivision {
    name: "Bayburt",
    code: "TR-69",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-69",
};


pub const TR_70: Subdivision = Subdivision {
    name: "Karaman",
    code: "TR-70",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-70",
};


pub const TR_71: Subdivision = Subdivision {
    name: "Kırıkkale",
    code: "TR-71",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-71",
};


pub const TR_72: Subdivision = Subdivision {
    name: "Batman",
    code: "TR-72",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-72",
};


pub const TR_73: Subdivision = Subdivision {
    name: "Şırnak",
    code: "TR-73",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-73",
};


pub const TR_74: Subdivision = Subdivision {
    name: "Bartın",
    code: "TR-74",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-74",
};


pub const TR_75: Subdivision = Subdivision {
    name: "Ardahan",
    code: "TR-75",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-75",
};


pub const TR_76: Subdivision = Subdivision {
    name: "Iğdır",
    code: "TR-76",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-76",
};


pub const TR_77: Subdivision = Subdivision {
    name: "Yalova",
    code: "TR-77",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-77",
};


pub const TR_78: Subdivision = Subdivision {
    name: "Karabük",
    code: "TR-78",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-78",
};


pub const TR_79: Subdivision = Subdivision {
    name: "Kilis",
    code: "TR-79",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-79",
};


pub const TR_80: Subdivision = Subdivision {
    name: "Osmaniye",
    code: "TR-80",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-80",
};


pub const TR_81: Subdivision = Subdivision {
    name: "Düzce",
    code: "TR-81",
    subdivision_type: "Province",
    country_name: "Turkey",
    country_code: "TR",
    region_code: "TR-81",
};


pub const TM_A: Subdivision = Subdivision {
    name: "Ahal",
    code: "TM-A",
    subdivision_type: "Region",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-A",
};


pub const TM_B: Subdivision = Subdivision {
    name: "Balkan",
    code: "TM-B",
    subdivision_type: "Region",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-B",
};


pub const TM_D: Subdivision = Subdivision {
    name: "Daşoguz",
    code: "TM-D",
    subdivision_type: "Region",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-D",
};


pub const TM_L: Subdivision = Subdivision {
    name: "Lebap",
    code: "TM-L",
    subdivision_type: "Region",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-L",
};


pub const TM_M: Subdivision = Subdivision {
    name: "Mary",
    code: "TM-M",
    subdivision_type: "Region",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-M",
};


pub const TM_S: Subdivision = Subdivision {
    name: "Aşgabat",
    code: "TM-S",
    subdivision_type: "City",
    country_name: "Turkmenistan",
    country_code: "TM",
    region_code: "TM-S",
};


pub const TC_TC: Subdivision = Subdivision {
    name: "Turks & Caicos Islands",
    code: "TC-TC",
    subdivision_type: "Country",
    country_name: "Turks & Caicos Islands",
    country_code: "TC",
    region_code: "TC-TC",
};


pub const TV_FUN: Subdivision = Subdivision {
    name: "Funafuti",
    code: "TV-FUN",
    subdivision_type: "Town council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-FUN",
};


pub const TV_NIT: Subdivision = Subdivision {
    name: "Niutao",
    code: "TV-NIT",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NIT",
};


pub const TV_NKF: Subdivision = Subdivision {
    name: "Nukufetau",
    code: "TV-NKF",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NKF",
};


pub const TV_NKL: Subdivision = Subdivision {
    name: "Nukulaelae",
    code: "TV-NKL",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NKL",
};


pub const TV_NMA: Subdivision = Subdivision {
    name: "Nanumea",
    code: "TV-NMA",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NMA",
};


pub const TV_NMG: Subdivision = Subdivision {
    name: "Nanumanga",
    code: "TV-NMG",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NMG",
};


pub const TV_NUI: Subdivision = Subdivision {
    name: "Nui",
    code: "TV-NUI",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-NUI",
};


pub const TV_VAI: Subdivision = Subdivision {
    name: "Vaitupu",
    code: "TV-VAI",
    subdivision_type: "Island council",
    country_name: "Tuvalu",
    country_code: "TV",
    region_code: "TV-VAI",
};


pub const UG_101: Subdivision = Subdivision {
    name: "Kalangala",
    code: "UG-101",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-101",
};


pub const UG_102: Subdivision = Subdivision {
    name: "Kampala",
    code: "UG-102",
    subdivision_type: "City",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-102",
};


pub const UG_103: Subdivision = Subdivision {
    name: "Kiboga",
    code: "UG-103",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-103",
};


pub const UG_104: Subdivision = Subdivision {
    name: "Luwero",
    code: "UG-104",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-104",
};


pub const UG_105: Subdivision = Subdivision {
    name: "Masaka",
    code: "UG-105",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-105",
};


pub const UG_106: Subdivision = Subdivision {
    name: "Mpigi",
    code: "UG-106",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-106",
};


pub const UG_107: Subdivision = Subdivision {
    name: "Mubende",
    code: "UG-107",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-107",
};


pub const UG_108: Subdivision = Subdivision {
    name: "Mukono",
    code: "UG-108",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-108",
};


pub const UG_109: Subdivision = Subdivision {
    name: "Nakasongola",
    code: "UG-109",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-109",
};


pub const UG_110: Subdivision = Subdivision {
    name: "Rakai",
    code: "UG-110",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-110",
};


pub const UG_111: Subdivision = Subdivision {
    name: "Sembabule",
    code: "UG-111",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-111",
};


pub const UG_112: Subdivision = Subdivision {
    name: "Kayunga",
    code: "UG-112",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-112",
};


pub const UG_113: Subdivision = Subdivision {
    name: "Wakiso",
    code: "UG-113",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-113",
};


pub const UG_114: Subdivision = Subdivision {
    name: "Mityana",
    code: "UG-114",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-114",
};


pub const UG_115: Subdivision = Subdivision {
    name: "Nakaseke",
    code: "UG-115",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-115",
};


pub const UG_116: Subdivision = Subdivision {
    name: "Lyantonde",
    code: "UG-116",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-116",
};


pub const UG_117: Subdivision = Subdivision {
    name: "Buikwe",
    code: "UG-117",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-117",
};


pub const UG_118: Subdivision = Subdivision {
    name: "Bukomansibi",
    code: "UG-118",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-118",
};


pub const UG_119: Subdivision = Subdivision {
    name: "Butambala",
    code: "UG-119",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-119",
};


pub const UG_120: Subdivision = Subdivision {
    name: "Buvuma",
    code: "UG-120",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-120",
};


pub const UG_121: Subdivision = Subdivision {
    name: "Gomba",
    code: "UG-121",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-121",
};


pub const UG_122: Subdivision = Subdivision {
    name: "Kalungu",
    code: "UG-122",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-122",
};


pub const UG_123: Subdivision = Subdivision {
    name: "Kyankwanzi",
    code: "UG-123",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-123",
};


pub const UG_124: Subdivision = Subdivision {
    name: "Lwengo",
    code: "UG-124",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-124",
};


pub const UG_201: Subdivision = Subdivision {
    name: "Bugiri",
    code: "UG-201",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-201",
};


pub const UG_202: Subdivision = Subdivision {
    name: "Busia",
    code: "UG-202",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-202",
};


pub const UG_203: Subdivision = Subdivision {
    name: "Iganga",
    code: "UG-203",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-203",
};


pub const UG_204: Subdivision = Subdivision {
    name: "Jinja",
    code: "UG-204",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-204",
};


pub const UG_205: Subdivision = Subdivision {
    name: "Kamuli",
    code: "UG-205",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-205",
};


pub const UG_206: Subdivision = Subdivision {
    name: "Kapchorwa",
    code: "UG-206",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-206",
};


pub const UG_207: Subdivision = Subdivision {
    name: "Katakwi",
    code: "UG-207",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-207",
};


pub const UG_208: Subdivision = Subdivision {
    name: "Kumi",
    code: "UG-208",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-208",
};


pub const UG_209: Subdivision = Subdivision {
    name: "Mbale",
    code: "UG-209",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-209",
};


pub const UG_210: Subdivision = Subdivision {
    name: "Pallisa",
    code: "UG-210",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-210",
};


pub const UG_211: Subdivision = Subdivision {
    name: "Soroti",
    code: "UG-211",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-211",
};


pub const UG_212: Subdivision = Subdivision {
    name: "Tororo",
    code: "UG-212",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-212",
};


pub const UG_213: Subdivision = Subdivision {
    name: "Kaberamaido",
    code: "UG-213",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-213",
};


pub const UG_214: Subdivision = Subdivision {
    name: "Mayuge",
    code: "UG-214",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-214",
};


pub const UG_215: Subdivision = Subdivision {
    name: "Sironko",
    code: "UG-215",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-215",
};


pub const UG_216: Subdivision = Subdivision {
    name: "Amuria",
    code: "UG-216",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-216",
};


pub const UG_217: Subdivision = Subdivision {
    name: "Budaka",
    code: "UG-217",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-217",
};


pub const UG_218: Subdivision = Subdivision {
    name: "Bukwa",
    code: "UG-218",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-218",
};


pub const UG_219: Subdivision = Subdivision {
    name: "Butaleja",
    code: "UG-219",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-219",
};


pub const UG_220: Subdivision = Subdivision {
    name: "Kaliro",
    code: "UG-220",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-220",
};


pub const UG_221: Subdivision = Subdivision {
    name: "Manafwa",
    code: "UG-221",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-221",
};


pub const UG_222: Subdivision = Subdivision {
    name: "Namutumba",
    code: "UG-222",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-222",
};


pub const UG_223: Subdivision = Subdivision {
    name: "Bududa",
    code: "UG-223",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-223",
};


pub const UG_224: Subdivision = Subdivision {
    name: "Bukedea",
    code: "UG-224",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-224",
};


pub const UG_225: Subdivision = Subdivision {
    name: "Bulambuli",
    code: "UG-225",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-225",
};


pub const UG_226: Subdivision = Subdivision {
    name: "Buyende",
    code: "UG-226",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-226",
};


pub const UG_227: Subdivision = Subdivision {
    name: "Kibuku",
    code: "UG-227",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-227",
};


pub const UG_228: Subdivision = Subdivision {
    name: "Kween",
    code: "UG-228",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-228",
};


pub const UG_229: Subdivision = Subdivision {
    name: "Luuka",
    code: "UG-229",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-229",
};


pub const UG_230: Subdivision = Subdivision {
    name: "Namayingo",
    code: "UG-230",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-230",
};


pub const UG_231: Subdivision = Subdivision {
    name: "Ngora",
    code: "UG-231",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-231",
};


pub const UG_232: Subdivision = Subdivision {
    name: "Serere",
    code: "UG-232",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-232",
};


pub const UG_301: Subdivision = Subdivision {
    name: "Adjumani",
    code: "UG-301",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-301",
};


pub const UG_302: Subdivision = Subdivision {
    name: "Apac",
    code: "UG-302",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-302",
};


pub const UG_303: Subdivision = Subdivision {
    name: "Arua",
    code: "UG-303",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-303",
};


pub const UG_304: Subdivision = Subdivision {
    name: "Gulu",
    code: "UG-304",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-304",
};


pub const UG_305: Subdivision = Subdivision {
    name: "Kitgum",
    code: "UG-305",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-305",
};


pub const UG_306: Subdivision = Subdivision {
    name: "Kotido",
    code: "UG-306",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-306",
};


pub const UG_307: Subdivision = Subdivision {
    name: "Lira",
    code: "UG-307",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-307",
};


pub const UG_308: Subdivision = Subdivision {
    name: "Moroto",
    code: "UG-308",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-308",
};


pub const UG_309: Subdivision = Subdivision {
    name: "Moyo",
    code: "UG-309",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-309",
};


pub const UG_310: Subdivision = Subdivision {
    name: "Nebbi",
    code: "UG-310",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-310",
};


pub const UG_311: Subdivision = Subdivision {
    name: "Nakapiripirit",
    code: "UG-311",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-311",
};


pub const UG_312: Subdivision = Subdivision {
    name: "Pader",
    code: "UG-312",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-312",
};


pub const UG_313: Subdivision = Subdivision {
    name: "Yumbe",
    code: "UG-313",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-313",
};


pub const UG_314: Subdivision = Subdivision {
    name: "Amolatar",
    code: "UG-314",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-314",
};


pub const UG_315: Subdivision = Subdivision {
    name: "Kaabong",
    code: "UG-315",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-315",
};


pub const UG_316: Subdivision = Subdivision {
    name: "Koboko",
    code: "UG-316",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-316",
};


pub const UG_317: Subdivision = Subdivision {
    name: "Abim",
    code: "UG-317",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-317",
};


pub const UG_318: Subdivision = Subdivision {
    name: "Dokolo",
    code: "UG-318",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-318",
};


pub const UG_319: Subdivision = Subdivision {
    name: "Amuru",
    code: "UG-319",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-319",
};


pub const UG_320: Subdivision = Subdivision {
    name: "Maracha",
    code: "UG-320",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-320",
};


pub const UG_321: Subdivision = Subdivision {
    name: "Oyam",
    code: "UG-321",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-321",
};


pub const UG_322: Subdivision = Subdivision {
    name: "Agago",
    code: "UG-322",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-322",
};


pub const UG_323: Subdivision = Subdivision {
    name: "Alebtong",
    code: "UG-323",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-323",
};


pub const UG_324: Subdivision = Subdivision {
    name: "Amudat",
    code: "UG-324",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-324",
};


pub const UG_325: Subdivision = Subdivision {
    name: "Buhweju",
    code: "UG-325",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-325",
};


pub const UG_326: Subdivision = Subdivision {
    name: "Kole",
    code: "UG-326",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-326",
};


pub const UG_327: Subdivision = Subdivision {
    name: "Lamwo",
    code: "UG-327",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-327",
};


pub const UG_328: Subdivision = Subdivision {
    name: "Napak",
    code: "UG-328",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-328",
};


pub const UG_329: Subdivision = Subdivision {
    name: "Nwoya",
    code: "UG-329",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-329",
};


pub const UG_330: Subdivision = Subdivision {
    name: "Otuke",
    code: "UG-330",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-330",
};


pub const UG_331: Subdivision = Subdivision {
    name: "Zombo",
    code: "UG-331",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-331",
};


pub const UG_401: Subdivision = Subdivision {
    name: "Bundibugyo",
    code: "UG-401",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-401",
};


pub const UG_402: Subdivision = Subdivision {
    name: "Bushenyi",
    code: "UG-402",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-402",
};


pub const UG_403: Subdivision = Subdivision {
    name: "Hoima",
    code: "UG-403",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-403",
};


pub const UG_404: Subdivision = Subdivision {
    name: "Kabale",
    code: "UG-404",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-404",
};


pub const UG_405: Subdivision = Subdivision {
    name: "Kabarole",
    code: "UG-405",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-405",
};


pub const UG_406: Subdivision = Subdivision {
    name: "Kasese",
    code: "UG-406",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-406",
};


pub const UG_407: Subdivision = Subdivision {
    name: "Kibaale",
    code: "UG-407",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-407",
};


pub const UG_408: Subdivision = Subdivision {
    name: "Kisoro",
    code: "UG-408",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-408",
};


pub const UG_409: Subdivision = Subdivision {
    name: "Masindi",
    code: "UG-409",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-409",
};


pub const UG_410: Subdivision = Subdivision {
    name: "Mbarara",
    code: "UG-410",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-410",
};


pub const UG_411: Subdivision = Subdivision {
    name: "Ntungamo",
    code: "UG-411",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-411",
};


pub const UG_412: Subdivision = Subdivision {
    name: "Rukungiri",
    code: "UG-412",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-412",
};


pub const UG_413: Subdivision = Subdivision {
    name: "Kamwenge",
    code: "UG-413",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-413",
};


pub const UG_414: Subdivision = Subdivision {
    name: "Kanungu",
    code: "UG-414",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-414",
};


pub const UG_415: Subdivision = Subdivision {
    name: "Kyenjojo",
    code: "UG-415",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-415",
};


pub const UG_416: Subdivision = Subdivision {
    name: "Ibanda",
    code: "UG-416",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-416",
};


pub const UG_417: Subdivision = Subdivision {
    name: "Isingiro",
    code: "UG-417",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-417",
};


pub const UG_418: Subdivision = Subdivision {
    name: "Kiruhura",
    code: "UG-418",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-418",
};


pub const UG_419: Subdivision = Subdivision {
    name: "Buliisa",
    code: "UG-419",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-419",
};


pub const UG_420: Subdivision = Subdivision {
    name: "Kiryandongo",
    code: "UG-420",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-420",
};


pub const UG_421: Subdivision = Subdivision {
    name: "Kyegegwa",
    code: "UG-421",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-421",
};


pub const UG_422: Subdivision = Subdivision {
    name: "Mitooma",
    code: "UG-422",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-422",
};


pub const UG_423: Subdivision = Subdivision {
    name: "Ntoroko",
    code: "UG-423",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-423",
};


pub const UG_424: Subdivision = Subdivision {
    name: "Rubirizi",
    code: "UG-424",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-424",
};


pub const UG_425: Subdivision = Subdivision {
    name: "Sheema",
    code: "UG-425",
    subdivision_type: "District",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-425",
};


pub const UG_C: Subdivision = Subdivision {
    name: "Central",
    code: "UG-C",
    subdivision_type: "Geographical region",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-C",
};


pub const UG_E: Subdivision = Subdivision {
    name: "Eastern",
    code: "UG-E",
    subdivision_type: "Geographical region",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-E",
};


pub const UG_N: Subdivision = Subdivision {
    name: "Northern",
    code: "UG-N",
    subdivision_type: "Geographical region",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-N",
};


pub const UG_W: Subdivision = Subdivision {
    name: "Western",
    code: "UG-W",
    subdivision_type: "Geographical region",
    country_name: "Uganda",
    country_code: "UG",
    region_code: "UG-W",
};


pub const UA_05: Subdivision = Subdivision {
    name: "Vinnytska oblast",
    code: "UA-05",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-05",
};


pub const UA_07: Subdivision = Subdivision {
    name: "Volynska oblast",
    code: "UA-07",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-07",
};


pub const UA_09: Subdivision = Subdivision {
    name: "Luhanska oblast",
    code: "UA-09",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-09",
};


pub const UA_12: Subdivision = Subdivision {
    name: "Dnipropetrovska oblast",
    code: "UA-12",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-12",
};


pub const UA_14: Subdivision = Subdivision {
    name: "Donetska oblast",
    code: "UA-14",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-14",
};


pub const UA_18: Subdivision = Subdivision {
    name: "Zhytomyrska oblast",
    code: "UA-18",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-18",
};


pub const UA_21: Subdivision = Subdivision {
    name: "Zakarpatska oblast",
    code: "UA-21",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-21",
};


pub const UA_23: Subdivision = Subdivision {
    name: "Zaporizka oblast",
    code: "UA-23",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-23",
};


pub const UA_26: Subdivision = Subdivision {
    name: "Ivano-Frankivska oblast",
    code: "UA-26",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-26",
};


pub const UA_30: Subdivision = Subdivision {
    name: "Kyiv",
    code: "UA-30",
    subdivision_type: "City",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-30",
};


pub const UA_32: Subdivision = Subdivision {
    name: "Kyivska oblast",
    code: "UA-32",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-32",
};


pub const UA_35: Subdivision = Subdivision {
    name: "Kirovohradska oblast",
    code: "UA-35",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-35",
};


pub const UA_40: Subdivision = Subdivision {
    name: "Sevastopol",
    code: "UA-40",
    subdivision_type: "City",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-40",
};


pub const UA_43: Subdivision = Subdivision {
    name: "Avtonomna Respublika Krym",
    code: "UA-43",
    subdivision_type: "Republic",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-43",
};


pub const UA_46: Subdivision = Subdivision {
    name: "Lvivska oblast",
    code: "UA-46",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-46",
};


pub const UA_48: Subdivision = Subdivision {
    name: "Mykolaivska oblast",
    code: "UA-48",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-48",
};


pub const UA_51: Subdivision = Subdivision {
    name: "Odeska oblast",
    code: "UA-51",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-51",
};


pub const UA_53: Subdivision = Subdivision {
    name: "Poltavska oblast",
    code: "UA-53",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-53",
};


pub const UA_56: Subdivision = Subdivision {
    name: "Rivnenska oblast",
    code: "UA-56",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-56",
};


pub const UA_59: Subdivision = Subdivision {
    name: "Sumska oblast",
    code: "UA-59",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-59",
};


pub const UA_61: Subdivision = Subdivision {
    name: "Ternopilska oblast",
    code: "UA-61",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-61",
};


pub const UA_63: Subdivision = Subdivision {
    name: "Kharkivska oblast",
    code: "UA-63",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-63",
};


pub const UA_65: Subdivision = Subdivision {
    name: "Khersonska oblast",
    code: "UA-65",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-65",
};


pub const UA_68: Subdivision = Subdivision {
    name: "Khmelnytska oblast",
    code: "UA-68",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-68",
};


pub const UA_71: Subdivision = Subdivision {
    name: "Cherkaska oblast",
    code: "UA-71",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-71",
};


pub const UA_74: Subdivision = Subdivision {
    name: "Chernihivska oblast",
    code: "UA-74",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-74",
};


pub const UA_77: Subdivision = Subdivision {
    name: "Chernivetska oblast",
    code: "UA-77",
    subdivision_type: "Region",
    country_name: "Ukraine",
    country_code: "UA",
    region_code: "UA-77",
};


pub const AE_AJ: Subdivision = Subdivision {
    name: "'Ajmān",
    code: "AE-AJ",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-AJ",
};


pub const AE_AZ: Subdivision = Subdivision {
    name: "Abū Z̧aby",
    code: "AE-AZ",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-AZ",
};


pub const AE_DU: Subdivision = Subdivision {
    name: "Dubayy",
    code: "AE-DU",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-DU",
};


pub const AE_FU: Subdivision = Subdivision {
    name: "Al Fujayrah",
    code: "AE-FU",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-FU",
};


pub const AE_RK: Subdivision = Subdivision {
    name: "Ra's al Khaymah",
    code: "AE-RK",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-RK",
};


pub const AE_SH: Subdivision = Subdivision {
    name: "Ash Shāriqah",
    code: "AE-SH",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-SH",
};


pub const AE_UQ: Subdivision = Subdivision {
    name: "Umm al Qaywayn",
    code: "AE-UQ",
    subdivision_type: "Emirate",
    country_name: "United Arab Emirates",
    country_code: "AE",
    region_code: "AE-UQ",
};


pub const GB_ABD: Subdivision = Subdivision {
    name: "Aberdeenshire",
    code: "GB-ABD",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ABD",
};


pub const GB_ABE: Subdivision = Subdivision {
    name: "Aberdeen City",
    code: "GB-ABE",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ABE",
};


pub const GB_AGB: Subdivision = Subdivision {
    name: "Argyll and Bute",
    code: "GB-AGB",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-AGB",
};


pub const GB_AGY: Subdivision = Subdivision {
    name: "Isle of Anglesey",
    code: "GB-AGY",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-AGY",
};


pub const GB_ANS: Subdivision = Subdivision {
    name: "Angus",
    code: "GB-ANS",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ANS",
};


pub const GB_ANT: Subdivision = Subdivision {
    name: "Antrim",
    code: "GB-ANT",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ANT",
};


pub const GB_ARD: Subdivision = Subdivision {
    name: "Ards",
    code: "GB-ARD",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ARD",
};


pub const GB_ARM: Subdivision = Subdivision {
    name: "Armagh",
    code: "GB-ARM",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ARM",
};


pub const GB_BAS: Subdivision = Subdivision {
    name: "Bath and North East Somerset",
    code: "GB-BAS",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BAS",
};


pub const GB_BBD: Subdivision = Subdivision {
    name: "Blackburn with Darwen",
    code: "GB-BBD",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BBD",
};


pub const GB_BDF: Subdivision = Subdivision {
    name: "Bedford",
    code: "GB-BDF",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BDF",
};


pub const GB_BDG: Subdivision = Subdivision {
    name: "Barking and Dagenham",
    code: "GB-BDG",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BDG",
};


pub const GB_BEN: Subdivision = Subdivision {
    name: "Brent",
    code: "GB-BEN",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BEN",
};


pub const GB_BEX: Subdivision = Subdivision {
    name: "Bexley",
    code: "GB-BEX",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BEX",
};


pub const GB_BFS: Subdivision = Subdivision {
    name: "Belfast",
    code: "GB-BFS",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BFS",
};


pub const GB_BGE: Subdivision = Subdivision {
    name: "Bridgend",
    code: "GB-BGE",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BGE",
};


pub const GB_BGW: Subdivision = Subdivision {
    name: "Blaenau Gwent",
    code: "GB-BGW",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BGW",
};


pub const GB_BIR: Subdivision = Subdivision {
    name: "Birmingham",
    code: "GB-BIR",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BIR",
};


pub const GB_BKM: Subdivision = Subdivision {
    name: "Buckinghamshire",
    code: "GB-BKM",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BKM",
};


pub const GB_BLA: Subdivision = Subdivision {
    name: "Ballymena",
    code: "GB-BLA",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BLA",
};


pub const GB_BLY: Subdivision = Subdivision {
    name: "Ballymoney",
    code: "GB-BLY",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BLY",
};


pub const GB_BMH: Subdivision = Subdivision {
    name: "Bournemouth",
    code: "GB-BMH",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BMH",
};


pub const GB_BNB: Subdivision = Subdivision {
    name: "Banbridge",
    code: "GB-BNB",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BNB",
};


pub const GB_BNE: Subdivision = Subdivision {
    name: "Barnet",
    code: "GB-BNE",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BNE",
};


pub const GB_BNH: Subdivision = Subdivision {
    name: "Brighton and Hove",
    code: "GB-BNH",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BNH",
};


pub const GB_BNS: Subdivision = Subdivision {
    name: "Barnsley",
    code: "GB-BNS",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BNS",
};


pub const GB_BOL: Subdivision = Subdivision {
    name: "Bolton",
    code: "GB-BOL",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BOL",
};


pub const GB_BPL: Subdivision = Subdivision {
    name: "Blackpool",
    code: "GB-BPL",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BPL",
};


pub const GB_BRC: Subdivision = Subdivision {
    name: "Bracknell Forest",
    code: "GB-BRC",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BRC",
};


pub const GB_BRD: Subdivision = Subdivision {
    name: "Bradford",
    code: "GB-BRD",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BRD",
};


pub const GB_BRY: Subdivision = Subdivision {
    name: "Bromley",
    code: "GB-BRY",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BRY",
};


pub const GB_BST: Subdivision = Subdivision {
    name: "Bristol, City of",
    code: "GB-BST",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BST",
};


pub const GB_BUR: Subdivision = Subdivision {
    name: "Bury",
    code: "GB-BUR",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-BUR",
};


pub const GB_CAM: Subdivision = Subdivision {
    name: "Cambridgeshire",
    code: "GB-CAM",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CAM",
};


pub const GB_CAY: Subdivision = Subdivision {
    name: "Caerphilly",
    code: "GB-CAY",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CAY",
};


pub const GB_CBF: Subdivision = Subdivision {
    name: "Central Bedfordshire",
    code: "GB-CBF",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CBF",
};


pub const GB_CGN: Subdivision = Subdivision {
    name: "Ceredigion",
    code: "GB-CGN",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CGN",
};


pub const GB_CGV: Subdivision = Subdivision {
    name: "Craigavon",
    code: "GB-CGV",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CGV",
};


pub const GB_CHE: Subdivision = Subdivision {
    name: "Cheshire East",
    code: "GB-CHE",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CHE",
};


pub const GB_CHW: Subdivision = Subdivision {
    name: "Cheshire West and Chester",
    code: "GB-CHW",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CHW",
};


pub const GB_CKF: Subdivision = Subdivision {
    name: "Carrickfergus",
    code: "GB-CKF",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CKF",
};


pub const GB_CKT: Subdivision = Subdivision {
    name: "Cookstown",
    code: "GB-CKT",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CKT",
};


pub const GB_CLD: Subdivision = Subdivision {
    name: "Calderdale",
    code: "GB-CLD",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CLD",
};


pub const GB_CLK: Subdivision = Subdivision {
    name: "Clackmannanshire",
    code: "GB-CLK",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CLK",
};


pub const GB_CLR: Subdivision = Subdivision {
    name: "Coleraine",
    code: "GB-CLR",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CLR",
};


pub const GB_CMA: Subdivision = Subdivision {
    name: "Cumbria",
    code: "GB-CMA",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CMA",
};


pub const GB_CMD: Subdivision = Subdivision {
    name: "Camden",
    code: "GB-CMD",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CMD",
};


pub const GB_CMN: Subdivision = Subdivision {
    name: "Carmarthenshire",
    code: "GB-CMN",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CMN",
};


pub const GB_CON: Subdivision = Subdivision {
    name: "Cornwall",
    code: "GB-CON",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CON",
};


pub const GB_COV: Subdivision = Subdivision {
    name: "Coventry",
    code: "GB-COV",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-COV",
};


pub const GB_CRF: Subdivision = Subdivision {
    name: "Cardiff",
    code: "GB-CRF",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CRF",
};


pub const GB_CRY: Subdivision = Subdivision {
    name: "Croydon",
    code: "GB-CRY",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CRY",
};


pub const GB_CSR: Subdivision = Subdivision {
    name: "Castlereagh",
    code: "GB-CSR",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CSR",
};


pub const GB_CWY: Subdivision = Subdivision {
    name: "Conwy",
    code: "GB-CWY",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-CWY",
};


pub const GB_DAL: Subdivision = Subdivision {
    name: "Darlington",
    code: "GB-DAL",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DAL",
};


pub const GB_DBY: Subdivision = Subdivision {
    name: "Derbyshire",
    code: "GB-DBY",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DBY",
};


pub const GB_DEN: Subdivision = Subdivision {
    name: "Denbighshire",
    code: "GB-DEN",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DEN",
};


pub const GB_DER: Subdivision = Subdivision {
    name: "Derby",
    code: "GB-DER",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DER",
};


pub const GB_DEV: Subdivision = Subdivision {
    name: "Devon",
    code: "GB-DEV",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DEV",
};


pub const GB_DGN: Subdivision = Subdivision {
    name: "Dungannon and South Tyrone",
    code: "GB-DGN",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DGN",
};


pub const GB_DGY: Subdivision = Subdivision {
    name: "Dumfries and Galloway",
    code: "GB-DGY",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DGY",
};


pub const GB_DNC: Subdivision = Subdivision {
    name: "Doncaster",
    code: "GB-DNC",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DNC",
};


pub const GB_DND: Subdivision = Subdivision {
    name: "Dundee City",
    code: "GB-DND",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DND",
};


pub const GB_DOR: Subdivision = Subdivision {
    name: "Dorset",
    code: "GB-DOR",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DOR",
};


pub const GB_DOW: Subdivision = Subdivision {
    name: "Down",
    code: "GB-DOW",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DOW",
};


pub const GB_DRY: Subdivision = Subdivision {
    name: "Derry",
    code: "GB-DRY",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DRY",
};


pub const GB_DUD: Subdivision = Subdivision {
    name: "Dudley",
    code: "GB-DUD",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DUD",
};


pub const GB_DUR: Subdivision = Subdivision {
    name: "Durham County",
    code: "GB-DUR",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-DUR",
};


pub const GB_EAL: Subdivision = Subdivision {
    name: "Ealing",
    code: "GB-EAL",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-EAL",
};


pub const GB_EAW: Subdivision = Subdivision {
    name: "England and Wales",
    code: "GB-EAW",
    subdivision_type: "Nation",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-EAW",
};


pub const GB_EAY: Subdivision = Subdivision {
    name: "East Ayrshire",
    code: "GB-EAY",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-EAY",
};


pub const GB_EDH: Subdivision = Subdivision {
    name: "Edinburgh, City of",
    code: "GB-EDH",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-EDH",
};


pub const GB_EDU: Subdivision = Subdivision {
    name: "East Dunbartonshire",
    code: "GB-EDU",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-EDU",
};


pub const GB_ELN: Subdivision = Subdivision {
    name: "East Lothian",
    code: "GB-ELN",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ELN",
};


pub const GB_ELS: Subdivision = Subdivision {
    name: "Eilean Siar",
    code: "GB-ELS",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ELS",
};


pub const GB_ENF: Subdivision = Subdivision {
    name: "Enfield",
    code: "GB-ENF",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ENF",
};


pub const GB_ENG: Subdivision = Subdivision {
    name: "England",
    code: "GB-ENG",
    subdivision_type: "Country",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ENG",
};


pub const GB_ERW: Subdivision = Subdivision {
    name: "East Renfrewshire",
    code: "GB-ERW",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ERW",
};


pub const GB_ERY: Subdivision = Subdivision {
    name: "East Riding of Yorkshire",
    code: "GB-ERY",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ERY",
};


pub const GB_ESS: Subdivision = Subdivision {
    name: "Essex",
    code: "GB-ESS",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ESS",
};


pub const GB_ESX: Subdivision = Subdivision {
    name: "East Sussex",
    code: "GB-ESX",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ESX",
};


pub const GB_FAL: Subdivision = Subdivision {
    name: "Falkirk",
    code: "GB-FAL",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-FAL",
};


pub const GB_FER: Subdivision = Subdivision {
    name: "Fermanagh",
    code: "GB-FER",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-FER",
};


pub const GB_FIF: Subdivision = Subdivision {
    name: "Fife",
    code: "GB-FIF",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-FIF",
};


pub const GB_FLN: Subdivision = Subdivision {
    name: "Flintshire",
    code: "GB-FLN",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-FLN",
};


pub const GB_GAT: Subdivision = Subdivision {
    name: "Gateshead",
    code: "GB-GAT",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GAT",
};


pub const GB_GBN: Subdivision = Subdivision {
    name: "Great Britain",
    code: "GB-GBN",
    subdivision_type: "Nation",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GBN",
};


pub const GB_GLG: Subdivision = Subdivision {
    name: "Glasgow City",
    code: "GB-GLG",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GLG",
};


pub const GB_GLS: Subdivision = Subdivision {
    name: "Gloucestershire",
    code: "GB-GLS",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GLS",
};


pub const GB_GRE: Subdivision = Subdivision {
    name: "Greenwich",
    code: "GB-GRE",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GRE",
};


pub const GB_GWN: Subdivision = Subdivision {
    name: "Gwynedd",
    code: "GB-GWN",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-GWN",
};


pub const GB_HAL: Subdivision = Subdivision {
    name: "Halton",
    code: "GB-HAL",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HAL",
};


pub const GB_HAM: Subdivision = Subdivision {
    name: "Hampshire",
    code: "GB-HAM",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HAM",
};


pub const GB_HAV: Subdivision = Subdivision {
    name: "Havering",
    code: "GB-HAV",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HAV",
};


pub const GB_HCK: Subdivision = Subdivision {
    name: "Hackney",
    code: "GB-HCK",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HCK",
};


pub const GB_HEF: Subdivision = Subdivision {
    name: "Herefordshire",
    code: "GB-HEF",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HEF",
};


pub const GB_HIL: Subdivision = Subdivision {
    name: "Hillingdon",
    code: "GB-HIL",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HIL",
};


pub const GB_HLD: Subdivision = Subdivision {
    name: "Highland",
    code: "GB-HLD",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HLD",
};


pub const GB_HMF: Subdivision = Subdivision {
    name: "Hammersmith and Fulham",
    code: "GB-HMF",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HMF",
};


pub const GB_HNS: Subdivision = Subdivision {
    name: "Hounslow",
    code: "GB-HNS",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HNS",
};


pub const GB_HPL: Subdivision = Subdivision {
    name: "Hartlepool",
    code: "GB-HPL",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HPL",
};


pub const GB_HRT: Subdivision = Subdivision {
    name: "Hertfordshire",
    code: "GB-HRT",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HRT",
};


pub const GB_HRW: Subdivision = Subdivision {
    name: "Harrow",
    code: "GB-HRW",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HRW",
};


pub const GB_HRY: Subdivision = Subdivision {
    name: "Haringey",
    code: "GB-HRY",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-HRY",
};


pub const GB_IOS: Subdivision = Subdivision {
    name: "Isles of Scilly",
    code: "GB-IOS",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-IOS",
};


pub const GB_IOW: Subdivision = Subdivision {
    name: "Isle of Wight",
    code: "GB-IOW",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-IOW",
};


pub const GB_ISL: Subdivision = Subdivision {
    name: "Islington",
    code: "GB-ISL",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-ISL",
};


pub const GB_IVC: Subdivision = Subdivision {
    name: "Inverclyde",
    code: "GB-IVC",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-IVC",
};


pub const GB_KEC: Subdivision = Subdivision {
    name: "Kensington and Chelsea",
    code: "GB-KEC",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KEC",
};


pub const GB_KEN: Subdivision = Subdivision {
    name: "Kent",
    code: "GB-KEN",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KEN",
};


pub const GB_KHL: Subdivision = Subdivision {
    name: "Kingston upon Hull",
    code: "GB-KHL",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KHL",
};


pub const GB_KIR: Subdivision = Subdivision {
    name: "Kirklees",
    code: "GB-KIR",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KIR",
};


pub const GB_KTT: Subdivision = Subdivision {
    name: "Kingston upon Thames",
    code: "GB-KTT",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KTT",
};


pub const GB_KWL: Subdivision = Subdivision {
    name: "Knowsley",
    code: "GB-KWL",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-KWL",
};


pub const GB_LAN: Subdivision = Subdivision {
    name: "Lancashire",
    code: "GB-LAN",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LAN",
};


pub const GB_LBH: Subdivision = Subdivision {
    name: "Lambeth",
    code: "GB-LBH",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LBH",
};


pub const GB_LCE: Subdivision = Subdivision {
    name: "Leicester",
    code: "GB-LCE",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LCE",
};


pub const GB_LDS: Subdivision = Subdivision {
    name: "Leeds",
    code: "GB-LDS",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LDS",
};


pub const GB_LEC: Subdivision = Subdivision {
    name: "Leicestershire",
    code: "GB-LEC",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LEC",
};


pub const GB_LEW: Subdivision = Subdivision {
    name: "Lewisham",
    code: "GB-LEW",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LEW",
};


pub const GB_LIN: Subdivision = Subdivision {
    name: "Lincolnshire",
    code: "GB-LIN",
    subdivision_type: "Two-tier county",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LIN",
};


pub const GB_LIV: Subdivision = Subdivision {
    name: "Liverpool",
    code: "GB-LIV",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LIV",
};


pub const GB_LMV: Subdivision = Subdivision {
    name: "Limavady",
    code: "GB-LMV",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LMV",
};


pub const GB_LND: Subdivision = Subdivision {
    name: "London, City of",
    code: "GB-LND",
    subdivision_type: "City corporation",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LND",
};


pub const GB_LRN: Subdivision = Subdivision {
    name: "Larne",
    code: "GB-LRN",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LRN",
};


pub const GB_LSB: Subdivision = Subdivision {
    name: "Lisburn",
    code: "GB-LSB",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LSB",
};


pub const GB_LUT: Subdivision = Subdivision {
    name: "Luton",
    code: "GB-LUT",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-LUT",
};


pub const GB_MAN: Subdivision = Subdivision {
    name: "Manchester",
    code: "GB-MAN",
    subdivision_type: "Metropolitan district",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MAN",
};


pub const GB_MDB: Subdivision = Subdivision {
    name: "Middlesbrough",
    code: "GB-MDB",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MDB",
};


pub const GB_MDW: Subdivision = Subdivision {
    name: "Medway",
    code: "GB-MDW",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MDW",
};


pub const GB_MFT: Subdivision = Subdivision {
    name: "Magherafelt",
    code: "GB-MFT",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MFT",
};


pub const GB_MIK: Subdivision = Subdivision {
    name: "Milton Keynes",
    code: "GB-MIK",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MIK",
};


pub const GB_MLN: Subdivision = Subdivision {
    name: "Midlothian",
    code: "GB-MLN",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MLN",
};


pub const GB_MON: Subdivision = Subdivision {
    name: "Monmouthshire",
    code: "GB-MON",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MON",
};


pub const GB_MRT: Subdivision = Subdivision {
    name: "Merton",
    code: "GB-MRT",
    subdivision_type: "London borough",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MRT",
};


pub const GB_MRY: Subdivision = Subdivision {
    name: "Moray",
    code: "GB-MRY",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MRY",
};


pub const GB_MTY: Subdivision = Subdivision {
    name: "Merthyr Tydfil",
    code: "GB-MTY",
    subdivision_type: "Unitary authority",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MTY",
};


pub const GB_MYL: Subdivision = Subdivision {
    name: "Moyle",
    code: "GB-MYL",
    subdivision_type: "District council area",
    country_name: "United Kingdom",
    country_code: "GB",
    region_code: "GB-MYL",
};


pub const GB_NAY: Subdivision = Subdivision {
    name: "North Ayrshire",
    code: "GB-NAY",
    subdivision_type: "Council area",
    country_name: "United Kingdom",
   