//! Miscellaneous convenience functions
//! 
//! These functions make some menial tasks a little easier to do,
//! like getting the executable's name, or loading a file
use std::{env, fmt};
use std::path::Path;
use std::ffi::OsStr;
use std::fs;
use std::fs::DirBuilder;


/// Creates directory (and any parents) and handles any errors that may occur
/// 
/// Returns the following codes:
/// 
/// OK: Successfully created the directory and its parents
/// 
/// ERR: Unable to create the directory
pub fn make_dir<P: fmt::Display>(path: P) -> String {
    let mut dir_build = DirBuilder::new();
    dir_build.recursive(true);

    match dir_build.create(path.to_string()) {
        Ok(_) => "OK".to_string(),
        Err(_) => "ERR".to_string()
    }
}

/// Abstraction around fs::read_to_string() that includes error handling
/// 
/// Returns the file content as a `String` or returns "ERR" if something went wrong
pub fn read_file<P: fmt::Display>(path: P) -> String {
    match fs::read_to_string(path.to_string()) {
        Ok(file) => file,
        Err(_) => "ERR".to_string()
    }
}

/// Abstraction around fs::write() that includes error handling
/// 
/// Returns the following codes:
/// 
/// OK: Save operation successful
/// 
/// ERR: There was a problem writing the file
pub fn save_file<P: fmt::Display>(path: P, contents: &[u8]) -> String {
    match fs::write(path.to_string(), contents) {
        Ok(_) => "OK".to_string(),
        Err(_) => "ERR".to_string()
    }
}

/// Saves a string to a shared value (temp file in /tmp or ~/.cache)
/// 
/// Value must be a string value
pub fn set_shared_val(name: &str, val: &str) {

    //get the .cache folder path
    let cache_dir = format!("{}/.cache/",env::var("HOME").unwrap());

    let code = save_file(format!("/tmp/{}",name), val.as_bytes());
    if code == "ERR" {
        save_file(format!("{}/{}",cache_dir,name), val.as_bytes());
    }
    
}

/// Gets a shared value (temp file in /tmp or .cache)
/// 
/// Value must be a string value
pub fn get_shared_val(name: &str) -> String{
    //get the .cache folder path
    let cache_dir = format!("{}/.cache/",env::var("HOME").unwrap());

    let mut val = read_file(format!("/tmp/{}",name));

    if val == "ERR" {
        val = read_file(format!("{}/{}",cache_dir,name));
    }
    val
}

/// Returns a list of files/directories using path string
pub fn get_dir<D: Into<String>>(dir: D, show_hidden: bool) -> Vec<String> {
    let dir_str = String::from(dir.into());
    let mut dir_name = dir_str.clone();
    if dir_str.chars().last().unwrap()!='/'{
        dir_name = format!("{}/",dir_str);
    }

    let dir = fs::read_dir(&dir_name).unwrap();
    let mut output: Vec<String> = Vec::new();

    for file in dir {
        let name = file.unwrap();
        let sep = "/".to_string();
        let filename = get_filename(name.path().display().to_string());
        let attrs = name.metadata().unwrap();
        if show_hidden || filename.chars().nth(0).unwrap() != '.' {
            if attrs.is_dir() {
                let path_t = format!("{}{}",name.path().display(),sep);
                let path: Vec<&str> = path_t.split('/').collect();
                output.push(format!("{}/",path[path.len()-2].to_string()));
            }
            else {
                output.push(get_filename(name.path().display().to_string()));
            }  
        }
    }
    alphabetize(output)

    
}

/// Alphabetizes a vector of strings
pub fn alphabetize<L: Into<String>>(list: Vec<L>) -> Vec<String> {
    // extract the strings from the generic value
    let mut string_list: Vec<String> = Vec::new();
    for item in list {
        string_list.push(item.into());
    }

    string_list.sort_by(|a, b| a.to_lowercase().cmp(&b.to_lowercase()));
    string_list
}

/// Get the name of the file/directory from a file path
pub fn get_filename<D: Into<String>>(dir: D) -> String {
    let dir_t =dir.into();
    let dir_vec = dir_t.split("/");
    dir_vec.last().unwrap().to_string()
}

/// Gets the parent directory from a file path
pub fn get_parent<D: Into<String>>(dir: D) -> String {
    let dir_t = dir.into();
    let mut dir_vec: Vec<&str> = dir_t.split("/").collect();
    dir_vec.pop();
    dir_vec.pop();
    let mut new_dir= String::from("");

    for (_, folder) in dir_vec.iter().enumerate(){
        for c in folder.chars(){
            new_dir.push(c);
        }
        new_dir.push('/');
    }
    new_dir
}

/// Capitalize the first letter in a string
pub fn capitalize<W: Into<String>>(word: W) -> String {
    let s1 = word.into();
    let mut v: Vec<char> = s1.chars().collect();

    for i in 0..s1.len(){
        if !v[i].is_digit(10){
            v[i] = v[i].to_uppercase().nth(0).unwrap();
            break;
        }
    }
    let s2: String = v.into_iter().collect();
    s2.to_string()
}

/// Returns the location (starting from 0) of the first instance of a character in a string
/// 
/// Skips any leading digits
/// 
/// Useful for parsing arguments
/// 
/// Returns the string's length if the char is not found
pub fn find_char<T: Into<String>>(text: T, sym: char) -> usize {
    let mut counter = 0;
    
    for c in text.into().chars(){
        if c == sym{
            break;
        }
        counter += 1;
    }

    counter
}

/// Get the name of the executable
pub fn get_execname() -> String {
    env::args().next()
        .as_ref()
        .map(Path::new)
        .and_then(Path::file_name)
        .and_then(OsStr::to_str)
        .map(String::from).unwrap()
}