//! Rotating logging API
//! This can be used to create a configurable log
//! that saves to a file and optionally
//! prints to stdout as well.
use std::{
    fmt, fs, env, panic, process,
    thread, sync::RwLock
};
use chrono::{Timelike, Datelike, Local};
use colored::{Colorize, ColoredString};
use lazy_static::lazy_static;

lazy_static! {
    static ref PANIC_LOG_NAME: RwLock<String> = RwLock::new(String::new());
    static ref PANIC_LOG_FOLDER: RwLock<String> = RwLock::new(String::new());
}

/// An automatic rotating log based on the current date
/// 
/// It logs output like the following
/// 
/// `[hh:mm:ss] [LEVEL]: message`
#[derive(Clone)]
pub struct Log {
    name: String,
    path: String,
    folder: String
}

impl Log {
    /// Create a new log handle
    /// 
    /// Panics: You don't seem to have a home folder. Make sure $HOME is set.
    pub fn get(log_name: &str, folder: &str) -> Log {
        let path = format!("{}/.local/share/{}", env::var("HOME").expect("Where the hell is your home folder?!"), folder);
        fs::create_dir_all(&path).unwrap_or(());
        Log {
            name: log_name.to_string(),
            path: path.to_string(),
            folder: folder.to_string()
        }
    }

    /// Print a line to the log
    /// 
    /// This will print any object that implements Display
    pub fn line<T: fmt::Display>(&self, level: LogLevel, text: T, print_stdout: bool) {
        if let LogLevel::Debug(false) = level {
            return;
        }
        let log_path = format!("{}/{}", self.path, self.get_log_name());
        let mut log = fs::read_to_string(&log_path).unwrap_or(String::new());
        let now = Local::now();
        let msg = format!("[{}:{:02}:{:02}] [{}]: {}\n", now.hour(), now.minute(), now.second(), level, text);
        if print_stdout { print!("{}", level.colorize(&msg)); }
    
        log.push_str(&msg);
        fs::write(log_path, log).expect("Unable to write to log file!");
    }
    
    /// Print a line to the log (basic info)
    pub fn line_basic<T: fmt::Display>(&self, text: T, print_stdout: bool) { self.line(LogLevel::Info, text, print_stdout); }

    /// Should this log handle be used to report application panics? This 
    /// creates a panic handler that logs the thread panicked, where the panic occurred in the source
    /// and the backtrace.
    /// 
    /// This could be useful in conjunction with libraries that block stdout/stderr like cursive
    pub fn report_panics(&self, report: bool) {
        if report {
            let mut log_name = PANIC_LOG_NAME.write().unwrap();
            *log_name = self.name.clone();
            let mut log_folder = PANIC_LOG_FOLDER.write().unwrap();
            *log_folder = self.folder.clone();
            panic::set_hook(Box::new(panic_handler))
        }
        else {
            drop(panic::take_hook());
        }
    }

    fn get_log_name(&self) -> String {
        let now = Local::now();
        format!("{}-{}-{}-{}.log", self.name, now.year(), now.month(), now.day())
    }
}

/// Severity level for a log entry
#[derive(Copy, Clone)]
pub enum LogLevel {
    /// Possibly useful information
    Info,

    /// Debug information, can optionally be hidden
    Debug(bool),

    /// This might cause trouble
    Warn,

    /// Oops...
    /// 
    /// Indicates an error has occurred
    Error,

    /// The Application has panicked
    Fatal
}

impl LogLevel {
    fn colorize(&self, input: &str) -> ColoredString {
        match self {
            Self::Debug(_) => input.cyan(),
            Self::Info => input.green(),
            Self::Warn => input.bright_yellow(),
            Self::Error => input.bright_red(),
            Self::Fatal => input.bright_red().on_black()
        }
    }
}

impl fmt::Display for LogLevel {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            LogLevel::Info => write!(f, "INFO"),
            LogLevel::Debug(_) => write!(f, "DEBUG"),
            LogLevel::Warn => write!(f, "WARN"),
            LogLevel::Error => write!(f, "ERROR"),
            LogLevel::Fatal => write!(f, "FATAL")
        }
    }
}

fn panic_handler(info: &panic::PanicInfo) {
    let backtrace = format!("{:?}", backtrace::Backtrace::new());
    let log_name = PANIC_LOG_NAME.read().unwrap_or_else(|_| {
        println!("Internal Error");
        process::exit(101);
    });
    let log_folder = PANIC_LOG_FOLDER.read().unwrap_or_else(|_| {
        println!("Internal Error");
        process::exit(101);
    });
    let panic_log = Log::get(&log_name, &log_folder);

    let cur_thread = thread::current();
    let name = cur_thread.name();
    let id = cur_thread.id();
    let thread_disp = if let Some(n) = name {
        n.to_string()
    }
    else {
        format!("{:?}", id)
    };

    let location = if let Some(loc) = info.location() {
        format!("at {}:{}:{}", loc.file(),loc.line(),loc.column())
    }
    else {
        String::new()
    };

    panic_log.line(LogLevel::Fatal, format!("Thread '{}' panicked {}", thread_disp, location), true);
    let msg = match (info.payload().downcast_ref::<&str>(), info.payload().downcast_ref::<String>()) {
        (Some(s), _) => s.to_string(),
        (_, Some(s)) => s.to_string(),
        (None, None) => String::new(),
    };
    if !msg.is_empty() {
        panic_log.line(LogLevel::Fatal, format!("Error: {}", msg), true);
    }

    panic_log.line(LogLevel::Fatal, "Backtrace:", true);
    for line in backtrace.lines() {
        panic_log.line(LogLevel::Fatal, line, true);
    }
}