#![allow(unused_imports)]
#[cfg(test)]
use crate::{utils, logging::*, config::*};
use std::{
    path::*,
    fs, env,
    thread
};
use serde_derive::{Deserialize, Serialize};
use chrono::{Timelike, Datelike, Local};

// convenience function tests
#[test]
fn make_dir() {
    println!("make_dir() test:");

    // this should succeed
    let code = utils::make_dir("/tmp/test_dir");
    println!("Attempting to create /tmp/test_dir: {}", code);
    
    if code != "OK" {
        println!("Expected return code: OK\nActual return code: {}", code);
        panic!("Test failed!")
    }
    
    // this should fail (permissions)
    let code = utils::make_dir("/etc/test_dir");
    println!("Attempting to create /etc/test_dir: {}", code);
    
    if code != "ERR" {
        println!("Expected return code: ERR\nActual return code: {}", code);
        panic!("Test failed!")
    }

    // this should succeed
    let mut code = utils::make_dir("/tmp/test_parent/test_dir");
    println!("Attempting to create /tmp/test_parent/test_dir: {}", code);

    if code != "OK" {
        println!("Expected return code: OK\nActual return code: {}", code);
        panic!("Test failed!")
    }

    // this should fail (permissions)
    code = utils::make_dir("/etc/test_parent/test_dir");
    println!("Attempting to create /etc/test_parent/test_dir: {}", code);

    if code != "ERR" {
        println!("Expected return code: ERR\nActual return code: {}", code);
        panic!("Test failed!")
    }

    fs::remove_dir("/tmp/test_dir").unwrap_or(());
    fs::remove_dir_all("/tmp/test_parent/").unwrap_or(());
}

#[test]
fn read_file() {
    save_file();
    println!("read_file() test:");
    utils::save_file("/tmp/test", "test".as_bytes());

    // this should succeed
    let mut contents = utils::read_file("/tmp/test");
    println!("Attempting to read /tmp/test: {}", contents);
    if contents != "test" {
        println!("Expected contents: test\nActual contents: {}", contents);
        panic!("The file was not read correctly!")
    }

    // this should fail (file doesn't exist)
    contents = utils::read_file("/etc/test");
    println!("Attempting to read /etc/test: {}\n", contents);
    if contents != "ERR" {
        println!("Expected contents: ERR\nActual contents: {}", contents);
        panic!("The file was not read correctly!")
    }
    fs::remove_file("/tmp/test").unwrap_or(());
}

#[test]
fn save_file() {
    println!("save_file() test:");
    fs::remove_file("/tmp/test").unwrap_or(());

    // this should succeed
    let mut code = utils::save_file("/tmp/test", "test".as_bytes());
    println!("Attempting to write to /tmp/test: {}", code);
    if code != "OK" {
        println!("Expected return code: OK\nActual return code: {}", code);
        panic!("The expected return code was not returned!")
    }

    // this should fail (permissions)
    code = utils::save_file("/etc/test", "test".as_bytes());
    println!("Attempting to write to /etc/test: {}\n", code);
    if code != "ERR" {
        println!("Expected return code: OK\nActual return code: {}", code);
        panic!("The expected return code was not returned!")
    }

    fs::remove_file("/tmp/test").unwrap_or(());
}

#[test]
fn get_execname() {
    println!("get_execname() test:");
    let name = utils::get_execname();
    println!("This test's executable name is {}", name);
}

#[test]
fn find_char() {
    println!("find_char() test:");
    let text = "test!";
    let expected = 4;
    let val = utils::find_char(text, '!');
    println!("Text: {}\nExpected: {}\nActual: {}\n", text, expected, val);
    if val != expected {
        panic!("The correct index for the char was not returned!")
    }
}

#[test]
fn capitalize() {
    println!("capitalize() test:");
    let expected = "Test";
    let text = utils::capitalize(expected);
    println!("Expected value: {}\nActual value: {}",&expected,&text);
    if expected != text {
        panic!("The string was not capitalized correctly!")
    }
}

#[test]
fn get_parent() {
    println!("get_parent() test:");
    let expected = "/home/";
    let home_dir = format!("{}/", env::var("HOME").unwrap());
    println!("{}", home_dir);

    let parent = utils::get_parent(home_dir);

    println!("Expected directory: {}\nActual directory: {}\n", expected, parent);
    if parent != expected {
        panic!("The expected parent directory was not returned!")
    }
}

#[test]
fn get_filename() {
    save_file();
    println!("get_filename() test:");
    let expected = "test.txt";
    let dir = "/tmp/test.txt";
    utils::save_file("/tmp/test.txt", "test".as_bytes());

    let file = utils::get_filename(dir);

    println!("Directory: {}\nFile name: {}\n", dir, file);
    if file != expected {
        panic!("The expected file name was not returned!")
    }

        fs::remove_file("/tmp/test.txt").unwrap();
}

#[test]
fn alphabetize(){
    println!("alphabetize() test:");
    // the alphabet, randomized
    let alpha_upper_scrambled = vec!["J", "B", "R", "L", "C", "P", "Y", "D", "F", "K", "A", "U", "O", "S", "Z", "E", "X", "Q", "I", "W", "T", "V", "H", "M", "N", "G"];
    let alpha_upper_expected = ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"];

    let alpha_lower_scrambled = vec!["e" ,"c", "r", "n", "x", "j", "o", "a", "g", "v", "y", "f", "l", "s", "q", "d", "k", "t", "b", "w", "m", "h", "u", "i", "p", "z"];
    let alpha_lower_expected = ["a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z"];

    let alpha_mixed_scrambled = vec!["t", "L", "G", "x", "f", "H", "R", "c", "S", "Q", "X", "D", "A", "o", "Y", "V", "b", "Z", "i", "E", "M", "U", "k", "N", "j", "p"];
    let alpha_mixed_expected = ["A", "b", "c", "D", "E", "f", "G", "H", "i", "j", "k", "L", "M", "N", "o", "p", "Q", "R", "S", "t", "U", "V", "x", "X", "Y", "Z"];
    
    // uppercase letters
    let mut alpha_vec: Vec<String> = utils::alphabetize(alpha_upper_scrambled);

    // are the letters in the correct order?
    for (count,letter) in alpha_vec.iter().enumerate() {
        if letter != alpha_upper_expected[count]{
            println!("Expected value: {:?}\nActual value: {:?}", alpha_upper_expected, alpha_vec);
            panic!("The vector was not sorted correctly!")
        }
    }
    println!("Uppercase: {:?}\n", alpha_vec);

    // lowercase letters
    alpha_vec = utils::alphabetize(alpha_lower_scrambled);

    // are the letters in the correct order?
    for (count,letter) in alpha_vec.iter().enumerate() {
        if letter != alpha_lower_expected[count]{
            println!("Expected value: {:?}\nActual value: {:?}", alpha_lower_expected, alpha_vec);
            panic!("The vector was not sorted correctly!")
        }
    }
    println!("Lowercase: {:?}\n", alpha_vec);

    // mixed case letters
    alpha_vec = utils::alphabetize(alpha_mixed_scrambled);

    // are the letters in the correct order?
    for (count,letter) in alpha_vec.iter().enumerate() {
        if letter != alpha_mixed_expected[count]{
            println!("Expected value: {:?}\nActual value: {:?}", alpha_mixed_expected, alpha_vec);
            panic!("The vector was not sorted correctly!")
        }
    }
    println!("Mixed case: {:?}\n",alpha_vec);

}

#[test]
fn get_dir() {
    save_file();
    get_filename();
    make_dir();
    println!("get_dir() test:");

    // generate the test directory
    let test_dir = "/tmp/test/test_subdir";
    let expected = [".hidden1", ".hidden2", "file1", "file2", "file3", "test_subdir/"];
    let expected_v = ["file1", "file2", "file3", "test_subdir/"];
    utils::make_dir(test_dir);
    utils::save_file("/tmp/test/.hidden1", "test".as_bytes());
    utils::save_file("/tmp/test/.hidden2", "test".as_bytes());
    utils::save_file("/tmp/test/file1", "test".as_bytes());
    utils::save_file("/tmp/test/file2", "test".as_bytes());
    utils::save_file("/tmp/test/file3", "test".as_bytes());
        
    // attempt to get the listing with hidden files
    let dir = utils::get_dir("/tmp/test/", true);
    println!("Showing hidden files:");
    println!("{:?}\n", dir);

    // does the listing match the expected listing (with hidden)?
    for (count, _) in expected.iter().enumerate() {
        if expected[count] != dir[count] {
            println!("Hidden files are visible");
            panic!("The expected directory listing was not returned!")
        }
    }

    // attempt to get the listing without hidden files
    let dir_v = utils::get_dir("/tmp/test/", false);
    println!("Hiding hidden files:");
    println!("{:?}\n", dir_v);

    // does the listing match the expected listing (without hidden)?
    for (count, _) in expected_v.iter().enumerate() {
        if expected_v[count] != dir_v[count] {
            println!("Hidden files are hidden");
            panic!("The expected directory listing was not returned!")
        }
    }
    fs::remove_dir_all("/tmp/test/").unwrap();
}

#[test]
fn shared_val() {
    save_file();
    read_file();

    // setting a shared value
    // attempt to set a test value
    utils::set_shared_val("test_val", "test");

    // this is where the test value should be stored
    let file_exists = Path::new("/tmp/test_val").exists();

    // does the file exist?
    if file_exists {
        println!("set_shared_val() test:\nThe file /tmp/test_val exists\n")
    }
    else {
        panic!("The file /tmp/test_val was not found")
    }

    // getting a shared value
    let expected = "test";
    
    // manually write a test value
    utils::save_file("/tmp/test_val", "test".as_bytes());

    // attempt to get that value
    let val = utils::get_shared_val("test_val");

    println!("get_shared_val() test:\nExpected value: {:?}\nActual value: {:?}\n", expected,val);
    if val != expected {
        panic!("The actual value does not match the expected value!");
    }

    fs::remove_file("/tmp/test_val").unwrap();
}

// logging API test
#[test]
fn logging_test() {
    let info_regex = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[INFO\]: Yes. Is this useful?").unwrap();
    let info_regex2 = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[INFO\]: Yes.").unwrap();
    let debug_regex = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[DEBUG\]: Debug info").unwrap();
    let debug_regex2 = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[DEBUG\]: This shouldn't show up").unwrap();
    let warn_regex = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[WARN\]: There may be an issue!").unwrap();
    let error_regex = regex::Regex::new(r"\[[0-9]*:[0-9][0-9]:[0-9][0-9]\] \[ERROR\]: Houston, we have a problem!").unwrap();
    let expected_name = {
        let now = Local::now();
        format!("test-{}-{}-{}.log", now.year(), now.month(), now.day())
    };
    let path = format!("{}/.local/share/coof-testing/{}", env::var("HOME").expect("Where the hell is your home folder?!"), expected_name);
    fs::remove_file(&path).unwrap_or(());
    let log = Log::get("test", "coof-testing");
    log.line(LogLevel::Info, "Yes. Is this useful?", true);
    log.line_basic("Yes.", true);
    log.line(LogLevel::Debug(true), "Debug info", true);
    log.line(LogLevel::Debug(false), "This shouldn't show up", true);
    log.line(LogLevel::Warn, "There may be an issue!", true);
    log.line(LogLevel::Error, "Houston, we have a problem!", true);

    let file = fs::read_to_string(&path).unwrap();

    if !info_regex.is_match(&file) {
        panic!("Where is the first info line?")
    }
    else if !info_regex2.is_match(&file) {
        panic!("Where is the second info line?")
    }
    else if !debug_regex.is_match(&file) {
        panic!("Where is the debug line?")
    }
    else if debug_regex2.is_match(&file) {
        panic!("Why is this showing up?")
    }
    else if !warn_regex.is_match(&file) {
        panic!("Where is the warning line?")
    }
    else if !error_regex.is_match(&file) {
        panic!("Where is the error line?")
    }

    let log2 = log.clone();

    let thread = thread::spawn(move || {
        for _ in 0..100 {
            log2.line(LogLevel::Info, "Concurrent logging...", true);
        }
    });

    for _ in 0..100 {
        log.line(LogLevel::Info, "Concurrent logging!!!", true);
    }

    thread.join().expect("thread did not complete successfully!");
}

#[test]
#[should_panic]
fn logging_panic_test() {
    let log = Log::get("test", "coof-testing");
    log.report_panics(true);
    let user = env::var("USER").expect("What is your name again?");
    let msg = format!("{} has tested positive for the coof!", user);
    panic!("{} has tested positive for the coof!", user);
    //panic!()
}
// config api test
#[test]
fn config_test() {
    #[derive(Deserialize, Serialize, PartialEq, Debug)]
    struct TestConfig {
        pub string: String,
        pub number: u32,
        pub boolean: bool
    }

    impl Default for TestConfig {
        fn default() -> TestConfig {
            TestConfig {
                string: "string".to_string(),
                number: 100,
                boolean: true
            }
        }
    }

    impl Config for TestConfig {
        const FILE_NAME: &'static str = "test.toml";
        fn get_save_dir() -> String { format!("{}/.config/coof-testing", env::var("HOME").expect("Where the hell is your home folder?!")) }
    }

    let config_dir = format!("{}/.config/coof-testing", env::var("HOME").expect("Where the hell is your home folder?!"));
    fs::remove_dir_all(config_dir).unwrap_or(());
    let expected = TestConfig::default();
    let mut config = TestConfig::load();

    if expected != config {
        println!("Expected config: {:?}\nActual config: {:?}", expected, config);
        panic!("default config was not set right!")
    }

    config.number = 5;
    config.save().expect("Unable to save config!");

    let expected = TestConfig {
        string: "string".to_string(),
        number: 5,
        boolean: true
    };
    let config = TestConfig::load();

    if expected != config {
        println!("Expected config: {:?}\nActual config: {:?}", expected, config);
        panic!("config was not loaded right!")
    }
}