//! Rotating logging API
//! This can be used to create a configurable log
//! that saves to a file and optionally
//! prints to stdout as well.
use std::{fmt, fs, env};
use chrono::{Timelike, Datelike, Local};

/// An automatic rotating log based on the current date
/// 
/// It logs output like the following
/// 
/// `[hh:mm:ss] [LEVEL]: message`
pub struct Log {
    name: String,
    path: String
}

impl Log {
    /// Create a new log handle
    pub fn get(log_name: &str, folder: &str) -> Log {
        let path = format!("{}/.local/share/{}", env::var("HOME").expect("Where the hell is your home folder?!"), folder);
        fs::create_dir_all(&path).unwrap_or(());
        Log {
            name: log_name.to_string(),
            path: path.to_string()
        }
    }

    /// Print a line to the log
    /// 
    /// This will print any object that implements Display
    pub fn line<T: fmt::Display>(&self, level: LogLevel, text: T, print_stdout: bool) {
        if let LogLevel::Debug(false) = level {
            return;
        }
        let log_path = format!("{}/{}", self.path, self.get_log_name());
        let mut log = fs::read_to_string(&log_path).unwrap_or(String::new());
        let now = Local::now();
        let msg = format!("[{}:{:02}:{:02}] [{}]: {}\n", now.hour(), now.minute(), now.second(), level, text);
        if print_stdout { print!("{}", msg); }
    
        log.push_str(&msg);
        fs::write(log_path, log).expect("Unable to write to log file!");
    }
    
    /// Print a line to the log (basic info)
    pub fn line_basic<T: fmt::Display>(&self, text: T, print_stdout: bool) { self.line(LogLevel::Info, text.to_string(), print_stdout); }
    
    fn get_log_name(&self) -> String {
        let now = Local::now();
        format!("{}-{}-{}-{}.log", self.name, now.year(), now.month(), now.day())
    }
}

/// Severity level for a log entry
pub enum LogLevel {
    /// Possibly useful information
    Info,

    /// Debug information, can optionally be hidden
    Debug(bool),

    /// This might cause trouble
    Warn,

    /// Oops...
    /// 
    /// Indicates an error has occurred
    Error
}

impl fmt::Display for LogLevel {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            LogLevel::Info => write!(f, "INFO"),
            LogLevel::Debug(_) => write!(f, "DEBUG"),
            LogLevel::Warn => write!(f, "WARN"),
            LogLevel::Error => write!(f, "ERROR")
        }
    }
}